-- init_box.vhd: Integration of initialization vector into key
-- Copyright (C) 2012 Brno University of Technology
-- Author(s): Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: init_box.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity init_box is
  generic(
    -- width of key into hash table
    KEY_WIDTH           : integer := 296;
    -- width of seed
    INIT_WIDTH          : integer := 32;
    -- constant init vector
    IV                  : in std_logic_vector(31 downto 0) := (others => '0')
  );
  port(
    KEY                 : in std_logic_vector(KEY_WIDTH-1 downto 0);
    SEED                : in std_logic_vector(INIT_WIDTH-1 downto 0);
    INIT_KEY            : out std_logic_vector(KEY_WIDTH-1 downto 0) 
  );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture sbox of init_box is
  type key_array_t is array(0 to log2(KEY_WIDTH)) of std_logic_vector(KEY_WIDTH-1 downto 0);
  signal key_var  : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal key_line : key_array_t;
  signal used_iv  : std_logic_vector(((KEY_WIDTH/32)+1)*32-1 downto 0);
  signal used_seed: std_logic_vector(((KEY_WIDTH/INIT_WIDTH)+1)*INIT_WIDTH-1 downto 0);
  function addr (i,j : integer) return integer is
    variable ii,a : integer;
  begin
    ii:=0;
    a:=0;
    while ii<i loop
      a:=a+(KEY_WIDTH/(2**(i+1)));
      ii:=ii+1;
    end loop;
    a:=a+j;
    return a;
  end addr; 
begin
  -- initialization based on variable seed
  var_coll_gen : for j in 0 to (KEY_WIDTH/2)-1 generate
    key_var(j)             <= KEY(j) when used_seed(j)='1' else KEY(KEY_WIDTH-j-1);
    key_var(KEY_WIDTH-j-1) <= KEY(j) when used_seed(j)='0' else KEY(KEY_WIDTH-j-1);
  end generate;
  var_coll_align_gen : if KEY_WIDTH-((KEY_WIDTH/2)*2) > ((KEY_WIDTH/2)*2) generate
    key_var(KEY_WIDTH-((KEY_WIDTH/2)*2)-1 downto (KEY_WIDTH/2)*2) <= KEY(KEY_WIDTH-((KEY_WIDTH/2)*2)-1 downto (KEY_WIDTH/2)*2);
  end generate;
  
  cyclic_seed_gen : for i in 0 to (KEY_WIDTH/INIT_WIDTH) generate
    used_seed((i+1)*INIT_WIDTH-1 downto INIT_WIDTH*i) <= SEED; 
  end generate;
  
  -- initialization based on constant IV
  key_line(0) <= key_var xor used_seed(KEY_WIDTH-1 downto 0);
  INIT_KEY    <= key_line(log2(KEY_WIDTH));
  line_gen : for i in 0 to log2(KEY_WIDTH)-1 generate
    coll_gen : for j in 0 to (KEY_WIDTH/(2**(i+1)))-1 generate
      key_line(i+1)((j+1)*(2**i)-1 downto j*(2**i))                       <= key_line(i)((j+1)*(2**i)-1 downto j*(2**i)) when used_iv(addr(i,j))='1' else key_line(i)(KEY_WIDTH-j*(2**i)-1 downto KEY_WIDTH-(j+1)*(2**i));
      key_line(i+1)(KEY_WIDTH-j*(2**i)-1 downto KEY_WIDTH-(j+1)*(2**i)) <= key_line(i)((j+1)*(2**i)-1 downto j*(2**i)) when used_iv(addr(i,j))='0' else key_line(i)(KEY_WIDTH-j*(2**i)-1 downto KEY_WIDTH-(j+1)*(2**i));
    end generate;
    coll_align_gen : if KEY_WIDTH-((KEY_WIDTH/(2**(i+1)))*(2**i)) > ((KEY_WIDTH/(2**(i+1)))*(2**i)) generate
      key_line(i+1)(KEY_WIDTH-((KEY_WIDTH/(2**(i+1)))*(2**i))-1 downto (KEY_WIDTH/(2**(i+1)))*(2**i)) <= key_line(i)(KEY_WIDTH-((KEY_WIDTH/(2**(i+1)))*(2**i))-1 downto (KEY_WIDTH/(2**(i+1)))*(2**i));
    end generate;
  end generate;

  cyclic_iv_gen : for i in 0 to (KEY_WIDTH/32) generate
    used_iv((i+1)*32-1 downto 32*i) <= IV; 
  end generate;
end architecture;
