-- dispatcher.vhd : Generic unit for discarding packets
-- Copyright (c) 2012 Brno University of Technology 
-- Author: Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: dispatcher.vhd 2420 2012-09-20 08:56:43Z xkekel00 $
--

-- ----------------------------------------------------------------------------
--                             Entity declaration
-- ----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all; 
use work.math_pack.all;

entity dispatcher is
   generic(
      DATA_WIDTH         : integer := 64;
      RESULT_WIDTH       : integer := 32
   );
   port (
      -- Clock signal
      CLK            : in std_logic;
      -- Synchronous reset
      RESET          : in std_logic;
      
      RESULT          : in  std_logic_vector(RESULT_WIDTH-1 downto 0);
      DENY            : in  std_logic;
      CONTROL_READY   : in  std_logic;
      CONTROL_NEXT    : out std_logic; 
      
      -- Input FL interface
      RX_DATA        : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      RX_DREM        : in  std_logic_vector(max(log2(DATA_WIDTH/8)-1, 0) downto 0);
      RX_SOF_N       : in  std_logic;
      RX_EOF_N       : in  std_logic;
      RX_SOP_N       : in  std_logic;
      RX_EOP_N       : in  std_logic;
      RX_SRC_RDY_N   : in  std_logic;
      RX_DST_RDY_N   : out std_logic;
      -- Output FL interface
      TX_DATA        : out std_logic_vector(DATA_WIDTH-1 downto 0);
      TX_DREM        : out std_logic_vector(max(log2(DATA_WIDTH/8)-1, 0) downto 0);
      TX_SOF_N       : out std_logic;
      TX_EOF_N       : out std_logic;
      TX_SOP_N       : out std_logic;
      TX_EOP_N       : out std_logic;
      TX_SRC_RDY_N   : out std_logic;
      TX_DST_RDY_N   : in  std_logic
   );
end entity;

-- ----------------------------------------------------------------------------
--                             Architecture declaration
-- ----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all; 
use work.math_pack.all;

architecture full of dispatcher is
   signal hi_data      : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal hi_rem       : std_logic_vector(LOG2(DATA_WIDTH/8)-1 downto 0);
   signal hi_src_rdy_n : std_logic;
   signal hi_dst_rdy_n : std_logic;
   signal hi_sof_n     : std_logic;
   signal hi_eof_n     : std_logic;
   signal hi_sop_n     : std_logic;
   signal hi_eop_n     : std_logic;
   
   signal sig_deny      : std_logic;
   signal dispatch      : std_logic;
   signal dispatch_reg  : std_logic;
   signal header_detect : std_logic;
   
begin
   header_insert_i : entity work.header_insert
      generic map(
         DATA_WIDTH     => DATA_WIDTH,
         RESULT_WIDTH   => RESULT_WIDTH
      )
      port map(
         CLK            => CLK,
         RESET          => RESET,
         
         RESULT_ID      => RESULT,
         DENY           => DENY,
         INSERT_READY   => CONTROL_READY,
         INSERT_NEXT    => CONTROL_NEXT,

         RX_DATA      => RX_DATA,
         RX_REM       => RX_DREM,
         RX_SRC_RDY_N => RX_SRC_RDY_N,
         RX_DST_RDY_N => RX_DST_RDY_N,
         RX_SOF_N     => RX_SOF_N,
         RX_EOF_N     => RX_EOF_N,
         RX_EOP_N     => RX_EOP_N,
         RX_SOP_N     => RX_SOP_N,

         TX_DATA      => hi_data,
         TX_REM       => hi_rem,
         TX_SRC_RDY_N => hi_src_rdy_n,
         TX_DST_RDY_N => hi_dst_rdy_n,
         TX_SOF_N     => hi_sof_n,
         TX_EOF_N     => hi_eof_n,
         TX_EOP_N     => hi_eop_n,
         TX_SOP_N     => hi_sop_n
      );
   
   TX_DATA      <= hi_data;
   TX_DREM      <= hi_rem;
   TX_SOF_N     <= hi_sof_n;
   TX_EOF_N     <= hi_eof_n;
   TX_EOP_N     <= hi_eop_n;
   TX_SOP_N     <= hi_sop_n;
   TX_SRC_RDY_N <= hi_src_rdy_n or sig_deny;
   hi_dst_rdy_n <= TX_DST_RDY_N and not(sig_deny);
   
   dispatch      <= hi_data(36);
   sig_deny      <= dispatch when hi_sof_n='0' else dispatch_reg;
   
   process(CLK)
   begin
      if CLK'event and CLK='1' then
         if RESET='1' then
            dispatch_reg <= '0';
         elsif hi_sof_n='0' and hi_dst_rdy_n='0' and hi_src_rdy_n='0' then
            dispatch_reg <= dispatch;
         end if; 
      end if;
   end process;
      
end architecture full;
