--!
--! bus_handshake.vhd: Bus handshake synchronizer.
--! Copyright (C) 2014 CESNET
--! Author(s): Jakub Cabal <jakubcabal@gmail.com>
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!
--! $Id$
--!
--! TODO:

library IEEE;	
use IEEE.std_logic_1164.all;	
use IEEE.std_logic_unsigned.all;	
use IEEE.std_logic_arith.all;

--! pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
--! pragma translate_on

   --! -------------------------------------------------------------------------
   --!                      Entity declaration
   --! -------------------------------------------------------------------------

entity ASYNC_BUS_HANDSHAKE_TOP is
   Generic (
      DATA_WIDTH : integer := 32  --! Data BUS width
   ); 
   Port ( 
      --! A clock domain
      ACLK     : in  STD_LOGIC;   --! Source clock
      ARST     : in  STD_LOGIC;   --! Source reset
      ADATAIN  : in  STD_LOGIC_VECTOR(DATA_WIDTH-1 downto 0); --! Data in
      ASEND    : in  STD_LOGIC;   --! Data send signal
      AREADY   : out STD_LOGIC;   --! Ready signal
     
      --! B clock domain
      BCLK     : in  STD_LOGIC;   --! Target clock
      BRST     : in  STD_LOGIC;   --! Target reset
      BDATAOUT : out STD_LOGIC_VECTOR(DATA_WIDTH-1 downto 0) := (others => '0'); --! Data out     
      BLOAD    : in  STD_LOGIC;   --! Data load signal
      BVALID   : out STD_LOGIC    --! Data valid signal
   );
end ASYNC_BUS_HANDSHAKE_TOP;

   --! -------------------------------------------------------------------------
   --!                      Architecture declaration
   --! -------------------------------------------------------------------------

architecture FULL of ASYNC_BUS_HANDSHAKE_TOP is
   
   signal anxt_data     : std_logic;
   signal adata         : std_logic_vector(DATA_WIDTH-1 downto 0) := (others => '0');
   signal a_ack         : std_logic;
   signal a_en          : std_logic := '0';
   signal a_en_next     : std_logic;
   signal b_en          : std_logic;
   signal b_ack         : std_logic := '0';
   signal b_ack_next    : std_logic;    
   signal bload_data    : std_logic;
   signal signal_asend  : std_logic;
   signal signal_aready : std_logic;
   signal signal_bvalid : std_logic;
   signal signal_bload  : std_logic;   
   
   --! ------------------------------------------------------------------------- 
begin
   --! -------------------------------------------------------------------------
   
   anxt_data    <= signal_asend AND signal_aready;
   a_en_next    <= a_en XOR anxt_data;
   bload_data   <= signal_bvalid AND signal_bload;
   b_ack_next   <= b_ack XOR bload_data;
   signal_asend <= ASEND;
   signal_bload <= BLOAD;
   AREADY       <= signal_aready;
   BVALID       <= signal_bvalid; 
    
   --! -------------------------------------------------------------------------
   
   --! a_ack PULSE GENRATOR
   a_ack_PULSE_GENRATOR: entity work.SYNC_PGEN 
   port map(
      ADATAIN  => b_ack,
      BCLK     => ACLK,
      ACLK     => BCLK,
      BRST     => ARST,
      ARST     => BRST,
      BEN      => a_ack,
      BDATAOUT => open 
   );
   
   --! -------------------------------------------------------------------------
   
   --! b_en PULSE GENRATOR
   b_en_PULSE_GENRATOR: entity work.SYNC_PGEN 
   port map(
      ADATAIN  => a_en,
      BCLK     => BCLK,
      ACLK     => ACLK,
      BRST     => BRST,
      ARST     => ARST,
      BEN      => b_en,
      BDATAOUT => open 
   );
   
   --! -------------------------------------------------------------------------
   
   --! TRANSMIT FSM
   TRANSMIT_FSM: entity work.BUS_HANDSHAKE_FSM
   generic map(
      TRANSMIT_FSM => true     
   )  
   port map(
      CLK    => ACLK,
      RST    => ARST,
      ACK    => a_ack,
      EVENT  => signal_asend,
      READY  => signal_aready
   );  
   
   --! -------------------------------------------------------------------------
   
   --! RECEIVE FSM 
   RECEIVE_FSM: entity work.BUS_HANDSHAKE_FSM
   generic map(
      TRANSMIT_FSM => false    
   )  
   port map(
      CLK    => BCLK,
      RST    => BRST,      
      ACK    => b_en,
      EVENT  => signal_bload,
      READY  => signal_bvalid
   );             
   
   --! -------------------------------------------------------------------------
   
   --! a_en REGISTER
   process(ACLK)
   begin
      if (rising_edge(ACLK)) then         
         if (ARST = '1') then
            a_en <= '0';
         else
            a_en <= a_en_next;  
         end if;   
      end if;
   end process;
   
   --! -------------------------------------------------------------------------
   
   --! b_ack REGISTER
   process(BCLK)
   begin
      if (rising_edge(BCLK)) then
         if (BRST = '1') then
            b_ack <= '0';
         else
            b_ack <= b_ack_next;  
         end if;           
      end if;
   end process;
   
   --! -------------------------------------------------------------------------
   
   --! TRANSMIT DATA REGISTER
   process(ACLK)
   begin
      if (rising_edge(ACLK)) then
         if (ARST = '1') then
            adata <= (others => '0');
         elsif (anxt_data = '1') then
            adata <= ADATAIN;  
         end if;
      end if;
   end process;
   
   --! -------------------------------------------------------------------------
   
   --! RECEIVE DATA REGISTER
   process(BCLK)
   begin
      if (rising_edge(BCLK)) then
         if (BRST = '1') then
            BDATAOUT <= (others => '0');
         elsif (bload_data = '1') then
            BDATAOUT <= adata;  
         end if;
      end if;
   end process;
   
   --! -------------------------------------------------------------------------

end architecture FULL;
