/*
 * csid: Combo6, Combo6x, Combo-PTM and add-on card identification tool.
 * Copyright (C) 2003-2008 CESNET
 * Author(s): Jachym Holecek <freza@liberouter.org>
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <err.h>
#include <stdio.h>
#include <time.h>

#include <combo.h>
#include <commlbr.h>

/* CVS ID tag for ident(1) program */
#define VERSION "$Id$"

/*
 * Control verbosity.
 */
extern int verbose;

/*
 * Main and stuff.
 */
#define ARGUMENTS                "c:d:wshvV"

#define CSID_BUFFER_SIZE         4096
#define CSID_BUILT_BUFFER_SIZE   128
#define CSID_INSERT_BEFORE       "PCI brver:"
#define CSID_BUILT_TIME_ADDR     0x4C
#define CSID_COMPONENT_BASE_ADDR 0x00000000
#define CSID_TIME_FORMAT         ""

int
csid_status(int xcard, cs_device_t *device)
{
	char *status;
	char path[64];
	char buffer[CSID_BUILT_BUFFER_SIZE + 1] = {0};
	int fd;
	unsigned long built_time;
	cs_space_t *space;

#ifdef __linux__
	char buf[CSID_BUFFER_SIZE];
	ssize_t size;

	sprintf(path, "/proc/driver/combo6/card%i/id", xcard);

	fd = open(path, O_RDONLY);
	if (fd == -1)
		return EXIT_FAILURE;

	size = read(fd, &buf, sizeof(buf));
	if (size > 0) {
		buf[size] = '\0';
		status = buf;
	}
#else
	struct combo6_status cs;

	sprintf(path, "/dev/combosix/%i", xcard);

	fd = open(path, O_RDONLY);
	if (fd == -1)
		return EXIT_FAILURE;

	memset(&cs, 0, sizeof(cs));
	if (ioctl(fd, COMBO6_IOC_STATUS, &cs))
		err(EXIT_FAILURE, "ioctl() failed");
	status = cs.text;
#endif

	/* map space of id component */
	if (cs_space_map(device, &space, CS_SPACE_FPGA, 0x100,
		CSID_COMPONENT_BASE_ADDR, 0)) {
			err(EXIT_FAILURE, "cs_space_map failed to map space at"
				"address 0x%X!", CSID_COMPONENT_BASE_ADDR);
	}

	/* read time from appropriate register */
	built_time = cs_space_read_4(device, space, CSID_BUILT_TIME_ADDR);

	if (built_time) {
		strftime(buffer, CSID_BUILT_BUFFER_SIZE, "%Y/%m/%d %H:%M:%S",
			localtime((time_t *)&built_time));
	} else {
		strcpy(buffer, "n/a");
	}

	/* Print out status */
	printf("Built at : %s\n", buffer);
	printf("%s", status);

	cs_space_unmap(device, &space);
	close(fd);
	return 0;
}

void
usage(char *me)
{
	printf("Usage: %s [-hsvVw] [-c int -d str]\n", me);
	printf("-c int Use card number 'int', default 0\n");
	printf("-d str Use device file 'str', default %s\n", CS_PATH_DEV(0));
	printf("-w     Use cs_hw_id() rather than cs_identify()\n");
	printf("-s     Show status if design is loaded and activated\n");
	printf("-h     Print this help\n");
	printf("-v     Verbose operation\n");
	printf("-V     Show version\n");
}

int
main(int argc, char *argv[])
{
	int c, xcard = 0;
	int use_hw_id = 0;
	int show_status = 0;
	char *board, *card, *chip;
	char path[64];
	char *number;
	cs_device_t *device;

	verbose = 0;
	strcpy(path, CS_PATH_DEV(0));

	while ((c = getopt(argc, argv, ARGUMENTS)) != -1) {
		switch (c) {
		case 'd': /* Device file path */
			strncpy(path, optarg, sizeof(path));
			path[sizeof(path)-1] = '\0';
			xcard = -1;
			break;

		case 'c':
			xcard = atoi(optarg);
			if (xcard < 0 || xcard > 7)
				errx(1, "card number: %d is out of range"
					" (0 - 7)", xcard);
			sprintf(path, CS_PATH_FMTINTDEV, xcard);
			break;

		case 'w':
			use_hw_id = 1;
			break;

		case 's':
			show_status = 1;
			break;

		case 'h': /* Show help */
			usage(argv[0]);
			return 0;

		case 'v': /* Verbose */
			verbose = 1;
			break;

		case 'V': /* Show version */
			printf("%s\n", VERSION);
			return EXIT_SUCCESS;

		default:
			errx(1, "unknown argument -%c", optopt);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc)
		errx(1, "stray arguments");
	
	if((number = strrchr(path, '/')) != NULL) {
		xcard = atoi(number+1);
	}

	if (cs_attach_noex(&device, path))
		err(EXIT_FAILURE, "cs_attach() failed, unable to open"
			" '%s' device file, probably card is not available or"
			" driver not loaded", path);

	if (show_status && xcard >= 0) {
		if (csid_status(xcard, device) == -1)
			printf("Card %i is not available.\n", xcard);
	} else if (!use_hw_id) {
		if (cs_identify(device, &board, &card, &chip))
			err(1, "cs_identify() failed");
		if (board != NULL)
			printf("%s ", board);
		if (card != NULL)
			printf("%s ", card);
		if (card != NULL)
			printf("%s", chip);
		printf("\n");
	} else {
		printf("%s\n", chip = cs_hw_id(device));
		free(chip);
	}

	cs_detach(&device);
	return EXIT_SUCCESS;
}
