/*
 * csboot: Generic Combo6 loader. Won't work w/ phase0 MTX.
 * Copyright (C) 2003-2008 CESNET
 * Author(s): Jachym Holecek <freza@liberouter.org>
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <combosix.h>
#include <commlbr.h>

/* CVS ID tag for ident(1) program */
__RCSID("$Id$");

/*
 * Main and stuff.
 */
#define ARGUMENTS       "aDd:f:F:v:h"

/*
 * Control verbosity.
 */
extern int verbose;

void
usage()
{
	printf("Usage: %s [-ahvDF] [-d s] [-f d s]\n", getprogname());
	printf("-a     Scan for known designs (child drivers)\n");
	printf("-D     Force detach of child drivers\n");
	printf("-d s   Use device file 's', default %s\n", CS_PATH_DEV(0));
	printf("-f d s Load design for fpga'd' from file 's'\n");
	printf("-F d   Firmware switch to number 'd' (COMBOv2 only)\n");
	printf("-h     Print this help\n");
	printf("-v     Select verbosity level(0-2)\n\n");
	printf("Note:  Fpga0 is the main on mother card, fpga8 is the first on\n");
	printf("       an interface card (fpga9 the second and so on).\n");
}

int
main(int argc, char *argv[])
{
	int c, id;
	int detach = 0, attach = 0, fwnum = -1;
	char *path = CS_PATH_DEV(0);
	cs_device_t *d;
	cs_boot_v1_t *boot = NULL;
	FILE *file;
	int ret;

	verbose = 0;
	setprogname(argv[0]);

	while ((c = getopt(argc, argv, ARGUMENTS)) != -1) {
		switch (c) {
		case 'a': /* Attach children */
			attach = 1;
			break;

		case 'D': /* Detach children */
			detach = 1;
			break;

		case 'd': /* Device file path */
			path = optarg;
			break;

		case 'h': /* Show help */
			usage();
			return EXIT_SUCCESS;

		case 'v': /* Verbose operation */
			if (cl_strtoi32(optarg, &verbose) ||
				(verbose > 2) || (verbose < 0))
					errx(EXIT_FAILURE, "invalid verbosity "
						"level number, please select "
						"number in range 0-2\n");
			break;

		case 'f': /* Fpga id + file 	*/
			if (cl_strtoi32(optarg, &id))
				errx(EXIT_FAILURE, "invalid argument %s", optarg);

			if (optind >= argc)
				errx(EXIT_FAILURE, "missing argument");

			if (id < 1024) {
				boot = calloc(1, sizeof(*boot));
				if (boot == NULL)
					errx(EXIT_FAILURE, "malloc for bootme");

				file = fopen(argv[optind], "r");
				if (file == NULL)
					errx(EXIT_FAILURE, "unable to open %s",
						argv[optind]);

				if ((c = cs_mcs_decode(file, &boot->data, &boot->size)) < 0)
					errx(EXIT_FAILURE, "mcs file %s cannot "
						"be decoded: %s\n",
						argv[optind], strerror(-c));

				boot->addr = id;

				fclose(file);
			} else if (id == 1024) {
				if (verbose)
					putenv("LIBCOMBO_ELF_VERBOSE=1");
				if ((c = cs_load_pcippc(argv[optind], &boot)) < 0)
					errx(EXIT_FAILURE, "PPC ELF file %s "
						"cannot be loaded: %s\n",
						argv[optind], strerror(-c));
			} else {
				errx(EXIT_FAILURE, "uknown id %i", id);
			}

			VERBOSE(CL_VERBOSE_ADVANCED,
				"filename: \"%s\", id = %i, size = 0x%x\n",
				argv[optind], boot->addr, boot->size);

			optind++;
			break;

		case 'F':
			if (cl_strtoi32(optarg, &fwnum))
				errx(EXIT_FAILURE, "invalid argument %s", optarg);
			break;

		default:
			errx(EXIT_FAILURE, "unknown argument -%c", optopt);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc)
		errx(EXIT_FAILURE, "stray arguments");

	if (fwnum >= 0) {
		if ((c = cs_firmware_switch (path, fwnum)))
			errx(EXIT_FAILURE, "cs_firmware_switch failed: %s\n",
				strerror(-c));
	} else if ((boot == NULL) && !(attach || detach)) {
		errx(EXIT_SUCCESS, "nothing to do");
	}

	/*
	 * We're ready for actual work.
	 */
	if (fwnum >= 0) /* it may take time if we switched FW */
		ret = cs_attach_wait(&d, path, 0);
	else
		ret = cs_attach(&d, path);

	if (ret)
		errx(EXIT_FAILURE, "cs_attach failed");

	if (detach || (attach && (boot != NULL))) {
		VERBOSE(CL_VERBOSE_ADVANCED, "detaching children\n");
		if (cs_children_detach(d) < 0) {
			if (errno != EBADFD) {
				cs_detach(&d);
				errx(EXIT_FAILURE, "unable to detach driver");
			}
		}
		VERBOSE(CL_VERBOSE_ADVANCED, "detach done\n");
	}

	if (boot != NULL) {
		c = cs_boot_v1(d, CS_BOOT_V1_STANDARD, boot);
		if (c < 0) {
			cs_detach(&d);
			errx(EXIT_FAILURE, "boot failed: %s\n", strerror(-c));
		}
	}

	if (attach) {
		VERBOSE(CL_VERBOSE_ADVANCED, "attaching children\n");
		if (cs_children_attach(d) < 0) {
			cs_detach(&d);
			errx(EXIT_FAILURE, "unable to attach driver");
		}
		VERBOSE(CL_VERBOSE_ADVANCED, "attach done\n");
	}

	cs_detach(&d);
	return EXIT_SUCCESS;
}
