/*!
 * \file xml.c
 * \brief lightweight XML parsing routines
 * \author Jaroslav Kysela <perex@perex.cz>
 * \date 2004
 *
 * Copyright (C) 2004  CESNET
 */
/* LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <sys/mman.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <commlbr.h>
#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

#if 0
/**
 * Debug switch
 */
#define XML_VERBOSE 1
#endif

/**
 * Debug switch
 */
int xml_debug = 0;

/**
 * Runtime settable debugging.
 */
#define COND_VERBOSE		(getenv("LIBCOMBO_XML_VERBOSE") != NULL)

static int
cs_xml_parse(cs_xml_t *xml, cs_xml_tag_t **tag, cs_xml_tag_t *parent, size_t *begin);
static void
cs_xml_tag_free(cs_xml_tag_t **tag);

/**
 * \brief Open an XML file
 * \param xml Returned XML structure
 * \param root Returned root XML tag
 * \param file Full filename
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_open(cs_xml_t **xml, cs_xml_tag_t **root, const char *file)
{
	int fd, result = 0;
	size_t begin;
	off_t size;

	fd = open(file, O_RDONLY);
	if (fd < 0)
		return -errno;
	*xml = (cs_xml_t*)malloc(sizeof(**xml));
	if (*xml == NULL) {
		result = -ENOMEM;
		goto __end;
	}
	(*xml)->fd = fd;
	if ((size = lseek(fd, 0, SEEK_END)) == ((off_t)-1)) {
		result = -errno;
		goto __end;
	}
	(*xml)->size = (size_t)size;
	(*xml)->data = (const char*)mmap(NULL, (*xml)->size, PROT_READ, MAP_SHARED, fd, 0);
	if ((*xml)->data == NULL || (*xml)->data == MAP_FAILED) {
		result = -errno;
		if (COND_VERBOSE)
			fprintf(stderr, "xml mmap failed: [%i] %s\n", errno, strerror(errno));
		goto __end;
	}
	begin = 0;
	result = cs_xml_parse(*xml, &(*xml)->root, NULL, &begin);
      __end:
	if (result < 0) {
		if (*xml)
			cs_xml_close(xml);
		else
			close(fd);
	} else {
		*root = (*xml)->root;
	}
	return result;
}

/**
 * \brief Close an XML file
 * \param xml XML structure
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_close(cs_xml_t **xml)
{
	if ((*xml)->root)
		cs_xml_tag_free(&(*xml)->root);
	if ((*xml)->data)
		munmap((void *)((*xml)->data), (*xml)->size);
	if ((*xml)->fd >= 0)
		close((*xml)->fd);
	free(*xml);
	*xml = NULL;
	return 0;
}

/**
 * Macro used in cs_xml_parse(). Moves to the next symbol in file.
 */
#define BEGIN_INC(begin) \
	if (++(begin) >= xml->size) { if (cur) free(cur); return -EINVAL; }

/**
 * \brief Parse XML tag
 * \param xml XML structure
 * \param tag Returned tag
 * \param parent Parent tag
 * \param xbegin Current position in XML data
 * \return zero on success otherwise a negative error code
 */
static int
cs_xml_parse(cs_xml_t *xml, cs_xml_tag_t **tag, cs_xml_tag_t *parent, size_t *xbegin)
{
	size_t begin = *xbegin, tmp;
	cs_xml_tag_t *cur = NULL, *prev = NULL;
	int xerr;

#ifdef XML_VERBOSE
	printf("NEW LEVEL! parent=%p\n", parent);
#endif
	*tag = NULL;
	while (begin < xml->size) {
#ifdef XML_VERBOSE
		printf("SIMILAR TAG: prev=%p\n", prev);
#endif
		while (xml->data[begin] != '<') {
			if (++(begin) >= xml->size)
				goto __out;
		}
		if (xml->data[begin] != '<') {
			if (xml_debug)
				fprintf(stderr, "XML: Unable to find next tag\n");
			return -EINVAL;
		}
		BEGIN_INC(begin);
		if (xml->data[begin] == '/') {
			while (xml->data[begin] != '>')
				BEGIN_INC(begin);
			BEGIN_INC(begin);
			goto __out;
		}
		cur = (cs_xml_tag_t*)calloc(1, sizeof(*cur));
#ifdef XML_VERBOSE
		printf("ALLOC TAG: %p\n", cur);
#endif
		if (cur == NULL)
			return -ENOMEM;
		if (prev)
			prev->next = cur;
		else
			*tag = cur;
		cur->parent = parent;
		cur->name = xml->data + begin;
		if (xml->data[begin] == '?')
			BEGIN_INC(begin);
		while (isalnum((int)xml->data[begin]))
			BEGIN_INC(begin);
		if (cur->name == xml->data + begin) {
			if (xml_debug)
				fprintf(stderr, "XML: null tag name? (%c)\n", xml->data[begin]);
			return -EINVAL;
		}
		cur->name_size = xml->data + begin - cur->name;

#ifdef XML_VERBOSE
		{
			int i;
			printf("NEW TAG: '");
			for (i = 0; i < cur->name_size; i++)
				putchar(cur->name[i]);
			printf("'\n");
		}
#endif

		while (xml->data[begin] != '>') {
			if (xml->data[begin] == '=') {
				BEGIN_INC(begin);
				if (xml->data[begin] == '"') {
					BEGIN_INC(begin);
					while (xml->data[begin] != '"')
						BEGIN_INC(begin);
				}
			}
			BEGIN_INC(begin);
		}
		cur->attr_size = xml->data + begin - cur->name;
 		if (xml->data[begin-1] == '/' ||
 		    xml->data[begin-1] == '?') {
			cur->attr_size--;
			tmp = begin++;
			goto __skip_parse;
		}
		BEGIN_INC(begin);
		cur->body = xml->data + begin;

#ifdef XML_VERBOSE
		{
			int i;
			printf("ATTR: '");
			for (i = 0; i < cur->attr_size; i++)
				putchar(cur->name[i]);
			printf("'\n");
		}
#endif
		xerr = cs_xml_parse(xml, &cur->child, cur, &begin);
#ifdef XML_VERBOSE
		printf("BACK TO PREVIOUS LEVEL: cur = %p\n", cur);
#endif
		if (xerr < 0)
			return xerr;
		tmp = begin - 1;
		while (tmp > 0 && xml->data[tmp] != '<')
			tmp--;
		if (tmp-- == 0)
			return -EINVAL;
	      __skip_parse:
		cur->body_size = xml->data + tmp -
				 cur->name - cur->attr_size;
		prev = cur;
	}
      __out:
	*xbegin = begin;
	return 0;
}

/**
 * \brief Free given XML tag (including following neighbors and childs)
 * \param tag XML tag
 */
static void
cs_xml_tag_free(cs_xml_tag_t **tag)
{
	cs_xml_tag_t *lookup = *tag, *next;

	while (lookup) {
		if (lookup->child)
			cs_xml_tag_free(&lookup->child);
		next = lookup->next;
		free(lookup);
		lookup = next;
	}
}

/**
 * \brief Return child tag for given parent tag
 * \param tag Parent tag
 * \return Child tag or NULL
 */
cs_xml_tag_t *
cs_xml_tag_child(cs_xml_tag_t *tag)
{
	return tag->child;
}

/**
 * \brief Return next tag at same level for given tag
 * \param tag Previous tag
 * \return Next tag or NULL
 */
cs_xml_tag_t *
cs_xml_tag_next(cs_xml_tag_t *tag)
{
	return tag->next;
}

/**
 * \brief Return parent tag for given child tag
 * \param tag Child tag
 * \return Parent tag or NULL
 */
cs_xml_tag_t *
cs_xml_tag_parent(cs_xml_tag_t *tag)
{
	return tag->parent;
}

/**
 * \brief Look for n-th XML tag at same level in the forward direction
 * \param tag Base XML tag
 * \param rtag Returned XML tag
 * \param name XML tag name
 * \param nth Ignore n tags with same name
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_tag_find(cs_xml_tag_t *tag, cs_xml_tag_t **rtag,
		const char *name, int nth)
{
	size_t len = strlen(name);

	while (tag) {
		if (tag->name_size == len &&
		    !memcmp(tag->name, name, len) &&
		    nth-- == 0) {
			*rtag = tag;
			return 0;
		}
		tag = tag->next;
	}
	return -ENOENT;
}

/**
 * \brief Get tag name
 * \param tag XML tag
 * \param name Returned XML name start
 * \param size Returned XML name size
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_tag_name(cs_xml_tag_t *tag, const char **name, size_t *size)
{
	*name = tag->name;
	*size = tag->name_size;
	return 0;
}

/**
 * \brief Get tag attribute
 * \param tag XML tag
 * \param attr Returned XML attribute start
 * \param size Returned XML attribute size
 * \param name XML attribute name
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_tag_attr(cs_xml_tag_t *tag, const char **attr, size_t *size, const char *name)
{
	const char *ptr, *start, *start1;
	size_t len, nlen, xlen;

	if (tag->name_size - tag->attr_size == 0)
		return -ENOENT;
	ptr = tag->name + tag->name_size;
	len = tag->attr_size - tag->name_size;
	nlen = strlen(name);
	while (len > 0) {
		while (len > 0 && isspace((int)*ptr)) {
			len--;
			ptr++;
		}
		start = ptr;
		while (len > 0 && isalpha((int)*ptr) && *ptr != '=') {
			len--;
			ptr++;
		}
		if (len > 0 && *ptr == '=') {
			xlen = ptr - start;
			len--;
			ptr++;
			if (len > 0 && *ptr == '"') {
				len--;
				start1 = ++ptr;
				while (len > 0 && *ptr != '"') {
					len--;
					ptr++;
				}
			} else {
				start1 = ptr;
				while (len > 0 && isalnum((int)*ptr)) {
					len--;
					ptr++;
				}
			}
			if (xlen == nlen && !memcmp(start, name, nlen)) {
				if (ptr - start1 == 0)
					return -EINVAL;
				*attr = start1;
				*size = ptr - start1;
				return 0;
			}
		}
		ptr++;
	}
	return -ENOENT;
}

/**
 * \brief Get tag body
 * \param tag XML tag
 * \param body Returned XML body start
 * \param size Returned XML body size
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_tag_body(cs_xml_tag_t *tag, const char **body, size_t *size)
{
	*body = tag->body;
	*size = tag->body_size;
	return 0;
}

/**
 * \brief Compare tag name
 * \param tag Tag
 * \param name Tag name to compare
 * \retval 0 Tag name does not match
 * \retval 1 Tag name match
 */
int
cs_xml_name_match(cs_xml_tag_t *tag, const char *name)
{
	const char * start;
	size_t size;
	int xerr;

	xerr = cs_xml_tag_name(tag, &start, &size);
	if (xerr < 0)
		return 0;
	if (size != strlen(name) || memcmp(name, start, size))
		return 0;
	return 1;

}

/**
 * \brief Compare tag attribute
 * \param tag Tag with given attribute
 * \param name Attribute name
 * \param body Attribute body
 * \retval 0 Attribute does not match
 * \retval 1 Attribute match
 */
int
cs_xml_attr_match(cs_xml_tag_t *tag, const char *name, const char *body)
{
	const char * start;
	size_t size;
	int xerr;

	xerr = cs_xml_tag_attr(tag, &start, &size, name);
	if (xerr < 0)
		return 0;
	if (size != strlen(body) || memcmp(body, start, size))
		return 0;
	return 1;

}

/**
 * \brief Verify XML document version
 * \param xml XML document
 * \param version Matched XML version
 * \retval 0 version does not match
 * \retval 1 version match
 */
int
cs_xml_is_version(cs_xml_t *xml, const char *version)
{
	cs_xml_tag_t * tag;
	const char * start;
	size_t size;
	int xerr;

	xerr = cs_xml_tag_find(xml->root, &tag, "?xml", 0);
	if (xerr < 0) {
		if (xml_debug)
			fprintf(stderr, "cs_xml_is_1_0: No ?xml tag\n");
		return 0;
	}
	xerr = cs_xml_tag_attr(tag, &start, &size, "version");
	if (xerr < 0) {
		if (xml_debug)
			fprintf(stderr, "cs_xml_is_1_0: No version attribute in ?xml tag\n");
		return 0;
	}
	if (size != strlen(version) || memcmp(version, start, size))
		return 0;
	return 1;
}

/**
 * \brief Verify XML document encoding
 * \param xml XML document
 * \param encoding Matched XML encoding
 * \retval 0 encoding does not match
 * \retval 1 encoding match
 */
int
cs_xml_is_encoding(cs_xml_t *xml, const char *encoding)
{
	cs_xml_tag_t * tag;
	const char * start;
	size_t size;
	int xerr;

	xerr = cs_xml_tag_find(xml->root, &tag, "?xml", 0);
	if (xerr < 0) {
		if (xml_debug)
			fprintf(stderr, "cs_xml_is_1_0: No ?xml tag\n");
		return 0;
	}
	xerr = cs_xml_tag_attr(tag, &start, &size, "encoding");
	if (xerr < 0) {
		if (xml_debug)
			fprintf(stderr, "cs_xml_is_1_0: No encoding attribute in ?xml tag\n");
		return 0;
	}
	if (size != strlen(encoding) || memcmp(encoding, start, size))
		return 0;
	return 1;
}

/**
 * \brief Get 32-bit unsigned int from passed XML
 * \param val Returned value
 * \param start Start parsing here
 * \param size Size of parsed area
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_get_uint32(u_int32_t *val, const char *start, size_t size)
{
	u_int32_t neg;

	*val = 0;
	if (size > 2 && !memcmp(start, "0x", 2)) {	/* hexa representation */
		size -= 2;
		start += 2;
		while (size > 0) {
			*val <<= 4;
			if (*start >= 'a' && *start <= 'f')
				*val |= *start - 'a' + 10;
			else if (*start >= 'A' && *start <= 'F')
				*val |= *start - 'A' + 10;
			else if (*start >= '0' && *start <= '9')
				*val |= *start - '0';
			else
				return -EINVAL;
			start++;
			size--;
		}
		return 0;
	}
	if (*start == '-') {
		neg = 1;
		start++;
		size--;
	} else {
		neg = 0;
	}
	while (size > 0) {
		if (*start == '.')
			break;
		if (*val * 10 < *val)	/* overflow test */
			return -EINVAL;
		*val *= 10;
		if (*start >= '0' && *start <= '9') {
			*val += *start - '0';
		} else {
			return -EINVAL;
		}
		start++;
		size--;
	}
	if (neg)
		*val = -*val;
	return 0;
}

/**
 * \brief Get string from passed XML
 * \param str Returned string
 * \param start Start parsing here
 * \param size Size of parsed area
 * \return zero on success otherwise a negative error code
 */
int
cs_xml_get_string(char **str, const char *start, size_t size)
{
	*str = (char*)malloc(size + 1);
	if (*str == NULL)
		return -ENOMEM;
	memcpy(*str, start, size);
	(*str)[size] = '\0';
	return 0;
}

/**
 * \brief Dump the parsed XML structure to stdout
 * \param xml XML structure
 * \param tag Starting tag (NULL = whole tree)
 * \param level Tag level (should be 0)
 */
void
cs_xml_dump(cs_xml_t *xml, cs_xml_tag_t *tag, int level)
{
	int tlevel;
	size_t tmp;

	if (tag == NULL) {
		level = 0;
		tag = xml->root;
		printf("XML file mapped at %p (0x%lx)\n", xml->data, (long)xml->size);
		if (tag == NULL)
			return;
	}
	while (tag != NULL) {
		tlevel = level;
		while (tlevel-- > 0)
			printf("  ");
		printf("TAG '");
		for (tmp = 0; tmp < tag->name_size; tmp++)
			putchar(tag->name[tmp]);
		printf("',ATTR '");
		for (tmp = tag->name_size + 1; tmp < tag->attr_size; tmp++)
			putchar(tag->name[tmp]);
		printf("': tag=%p, name=%p,name_size=0x%x,attr_size=0x%x,body=%p,body_size=0x%x\n",
			tag, tag->name, (unsigned int)tag->name_size, (unsigned int)tag->attr_size,
			tag->body, (unsigned int)tag->body_size);
		if (tag->child != NULL)
			cs_xml_dump(xml, tag->child, level + 1);
		tag = tag->next;
	}
}
