/*!
 * \file microwire.c
 * \brief Basic Microwire read/write routines.
 * \author Ivo Cieslar <xcieslar@fi.muni.cz>
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2003,2006,2007
 */

/*
 * Copyright (C) 2003,2006,2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'', AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE COMPANY OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id$
 *
 */

/* stdc include */
#include <err.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

/* liberouter include */
#include <commlbr.h>

/* local include */
#include "microwire.h"

__RCSID("$Id$");

/* Microwire signals */
/*! Chip Select */
#define MICROWIRE_CS_BIT 	0x1
/*! Data out */
#define MICROWIRE_DO_BIT	0x2
/*! Data in */
#define MICROWIRE_DI_BIT 	0x4
/*! Clock */
#define MICROWIRE_SK_BIT	0x8
/*! All cleared */
#define MICROWIRE_NULL		0x8

/* Microwire operation codes */
/*! Write Enable opcode */
#define MICROWIRE_OP_WENABLE	0x00
/*! Write opcode */
#define MICROWIRE_OP_WRITE	0x01
/*! Read opcode */
#define MICROWIRE_OP_READ	0x10

/*! Default dealay time in nanoseconds between writes to Microwire bus 
 *  Maximum SK frequency should be 1MHz -> 1microsecond wait time
 */
#define MICROWIRE_DELAY		1000

/*! 
 * Register address with attached microwire interface.
 * Can be modified using cs_microwire_setreg function
 */
static u_int32_t microwire_reg = 0x0;

/*!
 * \brief Function is used for delaying at the communication process
 * with the jtag device
 *
 * \param j Jtag device
 */
static void
microwire_delay()
{
	/* set de */
	static struct timespec time;
		
	/* wait  */
	time.tv_sec = MICROWIRE_DELAY / 1000000000;
	time.tv_nsec = MICROWIRE_DELAY % 1000000000;
	
	/* nano sleep */
	nanosleep(&time, NULL);	 
}

/*!
 * \brief Write start condition to Microwire bus. 
 *
 * Start condition consists of three steps. First all signals
 * are cleared, the chip select up signal is written and the
 * start bit is issued.
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 *
 */
static void 
microwire_start_cond(cs_device_t *device, cs_space_t *space)
{
	/* all to zero */
	cs_space_write_4(device, space, microwire_reg, MICROWIRE_NULL);
	microwire_delay();

	/* chip select up */
	cs_space_write_4(device, space, microwire_reg, MICROWIRE_CS_BIT);
	microwire_delay();

	/* write start bit */
	cs_space_write_4(device, space, microwire_reg, 
			MICROWIRE_CS_BIT | MICROWIRE_DI_BIT);
	microwire_delay();
	cs_space_write_4(device, space, microwire_reg, 
			MICROWIRE_CS_BIT | MICROWIRE_DI_BIT | MICROWIRE_SK_BIT);
	microwire_delay();

}

/*!
 * \brief Write stop condition to Microwire bus. 
 *
 * Start condition consists of two steps. First the chip select up signal 
 * is written and then all signals are cleared.
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 */
static void 
microwire_stop_cond(cs_device_t *device, cs_space_t *space)
{
	/* chip select up */
	cs_space_write_4(device, space, microwire_reg, MICROWIRE_CS_BIT);
	microwire_delay();

	/* all to zero */
	cs_space_write_4(device, space, microwire_reg, MICROWIRE_NULL);
	microwire_delay();
}

/*!
 * \brief Write operation code to Microwire bus. 
 *
 * Operation must be specified using macros MICROWIRE_WENABLE, 
 * MICROWIRE_WRITE, MICROWIRE_READ. 
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 */
static void 
microwire_opcode(cs_device_t *device, cs_space_t *space, u_int32_t operation)
{
	/* data used for writing */
	u_int32_t data0 = MICROWIRE_CS_BIT;
	u_int32_t data1 = MICROWIRE_CS_BIT;
	
	/* create written data */
	if(operation == MICROWIRE_OP_WRITE) data1 |= MICROWIRE_DI_BIT;
	if(operation == MICROWIRE_OP_READ) data0 |= MICROWIRE_DI_BIT;
	
	/* operation code (00 = Write Enable) */
	/* operation code (01 = Write) */
	/* operation code (10 = Read) */
	cs_space_write_4(device, space, microwire_reg, data0);
	microwire_delay();
	cs_space_write_4(device, space, microwire_reg, data0 | MICROWIRE_SK_BIT);
	microwire_delay();
	cs_space_write_4(device, space, microwire_reg, data1);
	microwire_delay();
	cs_space_write_4(device, space, microwire_reg, data1 | MICROWIRE_SK_BIT);
	microwire_delay();
}

/*!
 * \brief Write \e bits of \e data to Microwire bus. 
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 * \param address Address to write
 */
static void 
microwire_write(cs_device_t *device, cs_space_t *space, u_int32_t data, u_int32_t bits)
{
	u_int32_t regdata; /* Chip Select is up always */
	int i;

	/* send address */
	for (i = bits - 1; i >= 0; i--) {
		regdata = MICROWIRE_CS_BIT;
		/* if bit set -> set Data in bit */
		if (((data >> (i)) & 1) != 0) regdata |=  MICROWIRE_DI_BIT;

		/* write data */
		cs_space_write_4(device, space, microwire_reg, regdata);
		microwire_delay();
		cs_space_write_4(device, space, microwire_reg, 
				regdata | MICROWIRE_SK_BIT);
		microwire_delay();
	}
}

/*!
 * \brief Read \e bits from address to Microwire bus. 
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 * \param address Address to write
 */
static void 
microwire_read(cs_device_t *device, cs_space_t *space, u_int32_t *word, u_int32_t bits)
{
	u_int32_t data = 0;
	u_int32_t readdw; 
	u_int32_t i;
	
	/* last address bit -> first read is dummy bit -> skip */
        cs_space_write_4(device, space, microwire_reg, MICROWIRE_CS_BIT);
	microwire_delay(); 

	/* read specified bits */
	for (i = 0; i < bits; i++) {
		cs_space_write_4(device, space, microwire_reg, MICROWIRE_CS_BIT | MICROWIRE_SK_BIT);
		microwire_delay();
		cs_space_write_4(device, space, microwire_reg, MICROWIRE_CS_BIT);
		microwire_delay();
		/* read one bit */
		readdw = cs_space_read_4(device, space, microwire_reg);
		readdw = (readdw & MICROWIRE_DO_BIT) >> 1;
		microwire_delay();
		/* shift read value (*2) */
		data <<= 1;
		
		/* add read bit */
		data |= readdw;
	} 

	/* set the read value */
	*word = data;
}

/*!
 * \brief Read status after operation from Microwire bus. 
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 *
 * \retval Read status.
 */
static u_int32_t 
microwire_status(cs_device_t *device, cs_space_t *space)
{
	u_int32_t readdw;
	
	/* read status */
	readdw = cs_space_read_4(device, space, microwire_reg);
	readdw &= MICROWIRE_DO_BIT;
	microwire_delay();
	return readdw;
}
	
/*!
 * \brief Send write-enable operation .
 *
 * This function sends Write Enable operation code through
 * Microwire protocol to connected device.
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 *
 * \retval Status read from device after write enable.
 */
int 
cs_microwire_write_enable(cs_device_t *device, cs_space_t *space)
{
	/* write start condition */
	microwire_start_cond(device, space);
	
	/* operation code (00=Write Enable) */
	microwire_opcode(device, space, MICROWIRE_OP_WENABLE);

	/* write enable address must be 11xxxxxx (x = 0 or 1; don't care) */
	microwire_write(device, space, 0xFF, 8);
	
	/* write stop condition */
	microwire_stop_cond(device, space);
	
	/* return status after write enable */
	return microwire_status(device, space);
}



/*!
 * \brief Write \e word to register specified by \e address.
 *
 * This function writes a word to device using the Microwire protocol.
 * The data is passed to the routine as a string.
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 * \param address Write address passed (only lower 8bits used).
 * \param word Written data (only lower 16bits used).
 * \retval Status read from device after write.
 */
int
cs_microwire_write(cs_device_t *device, cs_space_t *space, 
		u_int32_t address, u_int32_t word)
{
	/* write start condition */
	microwire_start_cond(device, space);
	
	/* operation code (01 = write) */
	microwire_opcode(device, space, MICROWIRE_OP_WRITE);

	/* send 8bit address */
	microwire_write(device, space, address, 8);
		
	/* send 16bit data */
	microwire_write(device, space, word, 16);
	
	/* write stop condition */
	microwire_stop_cond(device, space);

	/* return status after write enable */
	return microwire_status(device, space);
}


/*!
 * \brief Read \e word (16 bit value) from device at \e address.
 *
 * This function reads a 16bit value from device using the Microwire serial
 * protocol. The read value is returned in a 16bit unsigned integer.
 *
 * \param device COMBO6(X) device.
 * \param space Mapped device space.
 * \param address Read address (only lower 8bits used).
 * \param word Read data (only lower 16bits used).
 *
 * \retval Status read from device after read.
 */
int
cs_microwire_read(cs_device_t *device, cs_space_t *space, 
		u_int32_t address, u_int32_t *word)
{
	/* write start condition */
	microwire_start_cond(device, space);
	
	/* operation code (10 = read) */
	microwire_opcode(device, space, MICROWIRE_OP_READ);

	/* send 8bit address */
	microwire_write(device, space, address, 8);
	
	/* read 16 bit data */
	microwire_read(device, space, word, 16);
	
    /* write stop condition */
	microwire_stop_cond(device, space);

	/* return status after write enable */
	return microwire_status(device, space);
}

/*!
 * \brief Function sets register with attached microwire signals
 * \param address Address of register.
 */ 
void 
cs_microwire_set_reg(u_int32_t address) 
{
	microwire_reg = address;	
}

