/**
 * \file mdio.c
 * \brief Implementation of basic MDIO read/write procedures.
 * \author Jachym Holecek <freza@liberouter.org>
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2004,2007
 */

/*
 * Copyright (C) 2004, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'', AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE COMPANY OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id$
 */

/* stdc include */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

/* local include */
#include "mdio.h"

/* combo include */
#include <commlbr.h>

/* MDIO registers in design */
/** MDIO frame register in design */
#define MDIO_FRAME 		0x00
/** MDIO data register in design */
#define MDIO_DATA 		0x08
/** MDIO status register in design */
#define MDIO_STAT 		0x0c

/* MDIO operation constants */
/** MDIO operation for passing address (EDF only) */
#define MDIO_OP_ADDR  	0x00
/** MDIO write opeartion (SDF & EDF) */
#define MDIO_OP_WRITE 	0x01
/** MDIO read operation (SDF only) */
#define MDIO_OP_READ 	0x02
/** MDIO read operation (EDF only) */
#define MDIO_OP_EDF_READ 0x03
/** Start of frame for SDF (stadard data format) */
#define MDIO_SDF 	0x01
/** Start of frame for EDF (extended data format) */
#define MDIO_EDF 	0x00

/* mask data for MDIO_FRAME */
#define _M_DATA(x) 		((x) & 0xffff)
#define _M_REG(x) 		((x) & 0x001f)
#define _M_PHY(x) 		_M_REG(x)
#define _M_OP(x) 		((x) & 0x0003)
#define _M_SDF(x) 		_M_OP(x)

/**
 * \def MDIO_SEND Send a MDIO frame
 */
#define MDIO_SEND(d, s, sdf, op, phy, reg, data, mdioval) \
	do { 								\
		(mdioval) = (_M_DATA(data) << 16) | 			\
		    (_M_REG(reg) << 9) | (_M_PHY(phy) << 4) | 		\
		    (_M_OP(op) << 2) | _M_SDF(sdf); 			\
									\
		usleep(100); 						\
		VERBOSE(2,"SENDING: %x",mdioval)                        \
		cs_space_write_4((d), (s), MDIO_FRAME, (mdioval)); 	\
	} while (/*CONSTCOND*/ 0)

/**
 * \def MDIO_WAIT Wait for MDIO operation 
 */
#define MDIO_WAIT(d, s) \
	do { 								\
		usleep(100); 						\
	} while(cs_space_read_4(d, s, MDIO_STAT) & 0x00010000)


/**
 * \brief Function writes data to mdio bus in standard data format (SDF)
 *
 * \param dev Pointer to combo device
 * \param space Pointer to mapped combo space
 * \param phy PHY address
 * \param reg Register address
 * \param data Data for operation (16b effective)
 */
void
cs_mdio_write_sdf(cs_device_t *dev, cs_space_t *space, u_int phy, u_int reg, u_int32_t data)
{
	u_int32_t 		a;

	cs_lock(CS_LOCK_MDIO);

	/* send frame */
	MDIO_SEND(dev, space, MDIO_SDF, MDIO_OP_WRITE, phy, reg, data, a);
	VERBOSE(CL_VERBOSE_LIBRARY, "write %04x \tphy %x reg %x frame %08x", data, phy, reg, a);
	/* read status */
	a = cs_space_read_4(dev, space, MDIO_STAT);

	cs_unlock(CS_LOCK_MDIO);

	VERBOSE(CL_VERBOSE_LIBRARY, "status %04x", a);
}

/**
 * \brief Function reads data from mdio bus with SDF (standard delay format)  
 *
 * \param dev Pointer to combo device
 * \param space Pointer to mapped combo space
 * \param phy PHY address
 * \param reg Register address
 *
 * \return Read data
 */
u_int32_t
cs_mdio_read_sdf(cs_device_t *dev, cs_space_t *space, u_int phy, u_int reg)
{
	u_int32_t 		val, a;

	cs_lock(CS_LOCK_MDIO);

	MDIO_SEND(dev, space, MDIO_SDF, MDIO_OP_READ, phy, reg, 0, a);
	MDIO_WAIT(dev, space);
	
	usleep(1000000);

	val = cs_space_read_4(dev, space, MDIO_DATA);
	VERBOSE(CL_VERBOSE_LIBRARY, "read %x \tphy %x reg %x frame xxx %08x", \
			val , phy, reg, a);
        
	/* read status */
	a = cs_space_read_4(dev, space, MDIO_STAT);

	cs_unlock(CS_LOCK_MDIO);

	VERBOSE(CL_VERBOSE_LIBRARY, "status %04x", a);
	
	return _M_DATA(val);
}

/**
 * \brief Function reads data from mdio bus with SDF (standard delay format)  
 *
 * \param dev Pointer to combo device
 * \param space Pointer to mapped combo space
 * \param phy PHY address
 * \param dev MDIO device type
 * \param reg Register address
 *
 * \return Read data (16b effective)
 */
u_int32_t
cs_mdio_read(cs_device_t *dev, cs_space_t *space, u_int phy, u_int device, u_int reg)
{
	u_int32_t 		val, a, b;

	cs_lock(CS_LOCK_MDIO);

	MDIO_SEND(dev, space, MDIO_EDF, MDIO_OP_ADDR, phy, device, reg, a);
	MDIO_SEND(dev, space, MDIO_EDF, MDIO_OP_EDF_READ, phy, device, 0, b);
	MDIO_WAIT(dev, space);

	val = cs_space_read_4(dev, space, MDIO_DATA);
	VERBOSE(CL_VERBOSE_LIBRARY, "extread %04x \tphy %x extreg %x reg %x frame %08x %08x", \
	    val & 0xffff, phy, device, reg, a, b);

	/* read status */
	a = cs_space_read_4(dev, space, MDIO_STAT);

	cs_unlock(CS_LOCK_MDIO);

	VERBOSE(CL_VERBOSE_LIBRARY, "status %04x", a);
	
	return _M_DATA(val);
}


/**
 * \brief Function reads data from mdio bus with SDF (standard delay format) without standard waiting routine,
 *        usable only for EMAC IP Core   
 *
 * \param dev Pointer to combo device
 * \param space Pointer to mapped combo space
 * \param phy PHY address
 * \param reg Register address
 *
 * \return Read data
 */
u_int32_t
cs_mdio_read_sdf_nowait(cs_device_t *dev, cs_space_t *space, u_int phy, u_int reg)
{
	u_int32_t 		val, a;

	cs_lock(CS_LOCK_MDIO);

	MDIO_SEND(dev, space, MDIO_SDF, MDIO_OP_READ, phy, reg, 0, a);
	
	usleep(1000000);

	val = cs_space_read_4(dev, space, MDIO_DATA);
	VERBOSE(CL_VERBOSE_LIBRARY, "read %x \tphy %x reg %x frame %08x", \
			val , phy, reg, a);
        
	/* read status */
	a = cs_space_read_4(dev, space, MDIO_STAT);

	cs_unlock(CS_LOCK_MDIO);
	
        VERBOSE(CL_VERBOSE_LIBRARY, "status %04x", a);
	
	return _M_DATA(val);
}

/**
 * \brief Function writes data to mdio bus with SDF (standard delay format)  
 *
 * \param dev Pointer to combo device
 * \param space Pointer to mapped combo space
 * \param phy PHY address
 * \param device MDIO device type
 * \param reg Register address
 * \param data Data for operation (16b effective)
 */
void
cs_mdio_write(cs_device_t *dev, cs_space_t *space, u_int phy, u_int device, u_int reg, \
    u_int32_t data)
{
	u_int32_t 		a, b;

	cs_lock(CS_LOCK_MDIO);

	MDIO_SEND(dev, space, MDIO_EDF, MDIO_OP_ADDR, phy, device, reg, a);
	MDIO_SEND(dev, space, MDIO_EDF, MDIO_OP_WRITE, phy, device, data, b);
	VERBOSE(CL_VERBOSE_LIBRARY, "extwrite %04x \tphy %x extreg %x reg %x frame %08x %08x", \
	    data, phy, device, reg, a, b);

	/* read status */
	a = cs_space_read_4(dev, space, MDIO_STAT);

	cs_unlock(CS_LOCK_MDIO);

	VERBOSE(CL_VERBOSE_LIBRARY, "status %04x", a);

}


