/*!
 * \file lup.c
 * \author Juraj Blaho
 *
 * \brief Provides control over the LUP component.
 */

/*
 * Copyright (C) 2007  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *        notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *        notice, this list of conditions and the following disclaimer in
 *        the documentation and/or other materials provided with the
 *        distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *        may be used to endorse or promote products derived from this
 *        software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <unistd.h>

#include <commlbr.h>

#include "../combosix.h"

#include "lup.h"

__RCSID("$Id:");

#define CS_LUP_MAX_TCAM_ROWS_PER_BURST ((CS_TCAM_MAX_BURST_COUNT/4)*4)

/*
 * LUP configuration constants
 */
static u_int32_t
    CS_LUP_CAM_CTRL_MEMORY_VALUES[CS_LUP_CAM_CTRL_MEMORY_ITEM_COUNT] = {
	0x000002BC,
	0x000003C8,
	0x00000034,
	0x00000140,
	0x0000028D,
	0x000003D9,
	0x00000005,
	0x00000151,
	0x0000029E,
	0x000003EA,
	0x00000016,
	0x00000162,
	0x000002AF,
	0x000003FB,
	0x00000027,
	0x00000173
};

static u_int32_t
    CS_LUP_CAM_INDEX_IFC02_VALUES[CS_LUP_CAM_INDEX_IFC02_ITEM_COUNT] = {
	0x00000000,
	0x00000007,
	0x00000000,
	0x00000001,
	0x00000002,
	0x00000001,
	0x00000002,
	0x00000003,
	0x00000004,
	0x00000003,
	0x00000004,
	0x00000005,
	0x00000006,
	0x00000005,
	0x00000006,
	0x00000007
};

static u_int32_t
    CS_LUP_CAM_INDEX_IFC13_VALUES[CS_LUP_CAM_INDEX_IFC13_ITEM_COUNT] = {
	0x00000001,
	0x00000000,
	0x00000001,
	0x00000002,
	0x00000003,
	0x00000002,
	0x00000003,
	0x00000004,
	0x00000005,
	0x00000004,
	0x00000005,
	0x00000006,
	0x00000007,
	0x00000006,
	0x00000007,
	0x00000000
};

/*! @} */

/*function definitions */

/*!
 * \brief Stop the processing unit.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_stop_pu(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "stop PU");

	/*write 1 to bit 0 of CS_LUP_SSRAM_BRIDGE_CTRL_REG */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "write 1 to bit 0 of CS_LUP_SSRAM_BRIDGE_CTRL_REG");
	tmp = cs_space_read_4(dev, space, CS_LUP_SSRAM_BRIDGE_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_SSRAM_BRIDGE_CTRL_REG, tmp | 1);

	/*wait for bit 0 of CS_LUP_SSRAM_BRIDGE_STATE_REG(0) */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "wait for bit 0 of CS_LUP_SSRAM_BRIDGE_STATE_REG(0)");
	while (!(cs_space_read_4(dev, space, CS_LUP_SSRAM_BRIDGE_STATE_REG),
		1))
		usleep(100);
}

/*!
 * \brief Stop the CAM unit.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_stop_cam(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "stop CAM");

	/*write 1 to bit 0 of CS_LUP_TCAM_BRIDGE_CTRL_REG */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "write 1 to bit 0 of CS_LUP_TCAM_BRIDGE_CTRL_REG");
	tmp = cs_space_read_4(dev, space, CS_LUP_TCAM_BRIDGE_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_TCAM_BRIDGE_CTRL_REG, tmp | 1);

	/*wait for bit 0 of CS_LUP_TCAM_BRIDGE_STATE_REG */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "wait for bit 0 of CS_LUP_TCAM_BRIDGE_STATE_REG");
	while (!(cs_space_read_4(dev, space, CS_LUP_TCAM_BRIDGE_STATE_REG), 1))
		usleep(100);
}

/*!
 * \brief Stop input.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_stop_input(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "stop input");

	/*write 1 to bit 0 of CS_LUP_CAM_CTRL_REG */
	tmp = cs_space_read_4(dev, space, CS_LUP_CAM_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_CAM_CTRL_REG, tmp | 1);
}

/*!
 * \brief Stop input and waits while FIFOs and pipeline are not empty.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_empty(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "empty LUP");

	/*stop input of new UH */
	cs_lup_stop_input(dev, space);

	/*wait for CS_LUP_STATE_REG & ALL_EMPTY (FIFO emptying) */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "wait for CS_LUP_STATE_REG & ALL_EMPTY (FIFO emptying)");
	while (!(cs_space_read_4(dev, space,
		    CS_LUP_CAM_STATE_REG) &
		CS_LUP_CAM_STATE_REG_ALL_EMPTY_FLAG))
		usleep(100);

	/*wait for CS_LUP_PU_STATE_REG & PU_EMPTY (pipeline emptying) */
	VERBOSE(CL_VERBOSE_LIBRARY,
	    "wait for CS_LUP_PU_STATE_REG & PU_EMPTY (pipeline emptying)");
	while (!(cs_space_read_4(dev, space,
		    CS_LUP_PU_STATE_REG) & CS_LUP_PU_STATE_REG_PU_EMPTY_FLAG))
		usleep(100);
}

/*!
 * \brief Start the processing unit.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_start_pu(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "start PU");

	/*write 0 to bit 0 of CS_LUP_SSRAM_BRIDGE_CTRL_REG */
	tmp = cs_space_read_4(dev, space, CS_LUP_SSRAM_BRIDGE_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_SSRAM_BRIDGE_CTRL_REG,
	    tmp & 0xFFFFFFFE);
}

/*!
 * \brief Start the CAM unit.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_start_cam(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "start CAM");

	/*write 0 to bit 0 of LUP_TCAM_BRIDGE_CTRL_REG */
	tmp = cs_space_read_4(dev, space, CS_LUP_TCAM_BRIDGE_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_TCAM_BRIDGE_CTRL_REG,
	    tmp & 0xFFFFFFFE);
}

/*!
 * \brief Start input.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_start_input(cs_device_t * dev, cs_space_t * space)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "start input");

	/*write 0 to bit 0 of CS_LUP_CAM_CTRL_REG */
	tmp = cs_space_read_4(dev, space, CS_LUP_CAM_CTRL_REG);
	cs_space_write_4(dev, space, CS_LUP_CAM_CTRL_REG, tmp & 0xFFFFFFFE);
}

/*!
 * \brief Erase the memory used for debugging.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_erase_debug_memory(cs_device_t * dev, cs_space_t * space)
{
	int i;

	VERBOSE(CL_VERBOSE_LIBRARY, "erase debug memory");

	for (i = 0; i < CS_LUP_RESULTS_MEMORY_ITEM_COUNT; i++) {
		cs_space_write_4(dev, space,
		    CS_LUP_RESULTS_MEMORY_OFFSET + i * 8, 0);
		cs_space_write_4(dev, space,
		    CS_LUP_RESULTS_MEMORY_OFFSET + i * 8 + 4, 0);
	}
}

/*!
 * \brief Store configuration to the control memory. The configuration is given by defined constant.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_write_cam_control_memory(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "write CAM control memory");

	cs_space_write_multi_4(dev, space,
	    CS_LUP_CAM_CTRL_MEMORY_OFFSET,
	    CS_LUP_CAM_CTRL_MEMORY_ITEM_COUNT, CS_LUP_CAM_CTRL_MEMORY_VALUES);
}

/*!
 * \brief Store configuration to the ifc02. The configuration is given by defined constant.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_write_cam_index_ifc02(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "write cam index for ifc02");

	cs_space_write_multi_4(dev, space,
	    CS_LUP_CAM_INDEX_IFC02_OFFSET,
	    CS_LUP_CAM_INDEX_IFC02_ITEM_COUNT, CS_LUP_CAM_INDEX_IFC02_VALUES);
}

/*!
 * \brief Store configuration to the ifc13. The configuration is given by defined constant.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 */
void
cs_lup_write_cam_index_ifc13(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "write cam index for ifc13");

	cs_space_write_multi_4(dev, space,
	    CS_LUP_CAM_INDEX_IFC13_OFFSET,
	    CS_LUP_CAM_INDEX_IFC13_ITEM_COUNT, CS_LUP_CAM_INDEX_IFC13_VALUES);
}

/*!
 * \brief Store one context information.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param context Context that is going to be stored.
 */
void
cs_lup_write_cam_context(cs_device_t * dev, cs_space_t * space,
    cs_lup_context_t * context)
{
	int i;

	VERBOSE(CL_VERBOSE_LIBRARY, "write CAM context; context_id = %u",
	    context->id);

	for (i = 0; i < CS_LUP_REGISTERS_PER_CONTEXT; i++) {
		/*store data to HW */
		cs_space_write_4(dev, space,
		    CS_LUP_CAM_REG_ADDR_IFC02_OFFSET +
		    (context->id * CS_LUP_REGISTERS_PER_CONTEXT + i) * 4,
		    context->regs[i] & 0x1F);
		cs_space_write_4(dev, space,
		    CS_LUP_CAM_REG_ADDR_IFC13_OFFSET +
		    (context->id * CS_LUP_REGISTERS_PER_CONTEXT + i) * 4,
		    context->regs[i] & 0x1F);
	}
}

/*!
 * \brief Store address where packet ID is stored.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param address Address of the UH register that contains packet ID.
 */
void
cs_lup_write_packet_id_address(cs_device_t * dev, cs_space_t * space,
    u_int32_t address)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "write packet id address; address = %u",
	    address);

	cs_space_write_4(dev, space, CS_LUP_PACKET_ID_ADDRESS_REG, address);
}

/*!
 * \brief Store the address of the instruction, which is performed first.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param address Address of the first performed instruction. 
 */
void
cs_lup_write_pu_first_address(cs_device_t * dev, cs_space_t * space,
    u_int32_t address)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "write pu first address; address = 0x%X",
	    address);

	cs_space_write_4(dev, space, CS_LUP_PACKET_ID_ADDRESS_REG, address);
}

/*!
 * \brief Read one row of the debug memory. Must be enabled in the design.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param row_id Number of the row that is going to be read.
 * \param row_data Output - the row data are stored here after this function call.
 */
void
cs_lup_read_debug_memory(cs_device_t * dev, cs_space_t * space,
    u_int32_t row_id, cs_lup_debug_row_t * row_data)
{
	u_int32_t tmp;

	VERBOSE(CL_VERBOSE_LIBRARY, "read debug memory");

	tmp =
	    cs_space_read_4(dev, space,
	    CS_LUP_RESULTS_MEMORY_OFFSET + row_id * 8 + 4);
	row_data->edit_parameters = tmp;

	tmp =
	    cs_space_read_4(dev, space,
	    CS_LUP_RESULTS_MEMORY_OFFSET + row_id * 8);
	row_data->interface_number = (tmp >> 16) & 3;
	row_data->packet_id = (tmp & 0xFFFF);
}

/*!
 * \brief Read cam control memory.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param data Array for output data. Must have at least CS_LUP_CAM_CTRL_MEMORY_ITEM_COUNT items.
 */
void
cs_lup_read_cam_control_memory(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[])
{
	VERBOSE(CL_VERBOSE_LIBRARY, "read CAM control memory");

	cs_space_read_multi_4(dev, space,
	    CS_LUP_CAM_CTRL_MEMORY_OFFSET,
	    CS_LUP_CAM_CTRL_MEMORY_ITEM_COUNT, data);
}

/*!
 * \brief Read cam index ifc02 memory.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param data Array for output data. Must have at least CS_LUP_CAM_INDEX_IFC02_ITEM_COUNT items.
 */
void
cs_lup_read_cam_index_ifc02(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[])
{
	VERBOSE(CL_VERBOSE_LIBRARY, "read CAM index for ifc02");

	cs_space_read_multi_4(dev, space,
	    CS_LUP_CAM_INDEX_IFC02_OFFSET,
	    CS_LUP_CAM_INDEX_IFC02_ITEM_COUNT, data);
}

/*!
 * \brief Read cam index ifc13 memory.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param data Array for output data. Must have at least CS_LUP_CAM_INDEX_IFC13_ITEM_COUNT items.
 */
void
cs_lup_read_cam_index_ifc13(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[])
{
	VERBOSE(CL_VERBOSE_LIBRARY, "read CAM index for ifc13");

	cs_space_read_multi_4(dev, space,
	    CS_LUP_CAM_INDEX_IFC13_OFFSET,
	    CS_LUP_CAM_INDEX_IFC13_ITEM_COUNT, data);
}

/*!
 * \brief Read one cam context.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param context_id ID (number) of the context that is going to be read.
 * \param context_data Used to store output.
 */
void
cs_lup_read_cam_context(cs_device_t * dev, cs_space_t * space,
    u_int32_t context_id, cs_lup_context_t * context_data)
{
	int i;

	VERBOSE(CL_VERBOSE_LIBRARY, "read CAM context; context_id = %u",
	    context_id);

	for (i = 0; i < CS_LUP_REGISTERS_PER_CONTEXT; i++) {
		context_data->regs[i] =
		    cs_space_read_4(dev, space,
		    CS_LUP_CAM_REG_ADDR_IFC02_OFFSET +
		    (context_id * CS_LUP_REGISTERS_PER_CONTEXT + i) * 4);
	}
}

/*!
 * \brief Get the UH register address where the packet ID is stored.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \return The address.
 */
u_int32_t
cs_lup_read_packet_id_address(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "read packet id address");

	return cs_space_read_4(dev, space, CS_LUP_PACKET_ID_ADDRESS_REG);
}

/*!
 * \brief Get the address of the instruction, which is performed first.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \return The address.
 */
u_int32_t
cs_lup_read_pu_first_address(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "read first instruction address");

	return cs_space_read_4(dev, space,
	    CS_LUP_FIRST_INSTRUCTION_ADDRESS_REG);
}

/*!
 * \brief Read LUP status.
 *
 * \param dev Device descriptor.
 * \param space Address space descriptor.
 * \param status Output - status is stored here.
 */
void
cs_lup_read_status(cs_device_t * dev, cs_space_t * space,
    cs_lup_status_t * status)
{
	int i;

	VERBOSE(CL_VERBOSE_LIBRARY, "read LUP status");

	/*PU status */
	if (cs_space_read_4(dev, space, CS_LUP_SSRAM_BRIDGE_STATE_REG) & 1)
		status->pu_running = false;
	else
		status->pu_running = true;

	/*CAM block */
	if (cs_space_read_4(dev, space, CS_LUP_TCAM_BRIDGE_STATE_REG) & 1)
		status->cam_block_running = false;
	else
		status->cam_block_running = true;

	/*UH input */
	if (cs_space_read_4(dev, space, CS_LUP_CAM_CTRL_REG) & 1)
		status->uh_input_enabled = false;
	else
		status->uh_input_enabled = true;

	/*FIFO status regs */
	status->input_info =
	    cs_space_read_4(dev, space, CS_LUP_INPUT_INFO_REG);
	status->output_info =
	    cs_space_read_4(dev, space, CS_LUP_OUTPUT_INFO_REG);
	status->new_items_info =
	    cs_space_read_4(dev, space, CS_LUP_NEW_ITEMS_INFO_REG);

	/*pipeline status regs */
	for (i = 0; i < CS_LUP_PIPELINE_STATE_REG_COUNT; i++)
		status->pipeline_status_regs[i] =
		    cs_space_read_4(dev, space,
		    CS_LUP_FIRST_PIPELINE_STATE_REG + i * 4);
}

/*!
 * \brief Store one instruction to SSRAM.
 *
 * \param ssram_dev SSRAM Device descriptor.
 * \param ssram_space SSRAM Address space descriptor.
 * \param instruction Pointer to instruction.
 */
static void
cs_lup_write_ssram_instruction(cs_device_t * ssram_dev,
    cs_space_t * ssram_space, cs_lup_ssram_instruction_t * instruction)
{
	cs_ssramctrl_row_t tmp;

	tmp.low = instruction->code_lo;
	tmp.high = instruction->code_hi;

	cs_ssramctrl_write_row(ssram_dev, ssram_space, tmp,
	    instruction->address);
}

/*!
 * \brief Store program to SSRAM.
 *
 * \param ssram_dev SSRAM Device descriptor.
 * \param ssram_space SSRAM Address space descriptor.
 * \param instructions Array of instructions.
 * \param instruction_count Number of instructions.
 */
void
cs_lup_init_ssram(cs_device_t * ssram_dev, cs_space_t * ssram_space,
    cs_lup_ssram_instruction_t * instructions, u_int32_t instruction_count)
{
	u_int32_t i, segment = 0;

	VERBOSE(CL_VERBOSE_LIBRARY, "Initializing SSRAM");

	/*get ssram control */
	cs_ssramctrl_get_unit_control(ssram_dev, ssram_space);

	/*set starting base address */
	cs_ssramctrl_set_base_addr(ssram_dev, ssram_space, segment);

	/*store instructions */
	for (i = 0; i < instruction_count; i++) {

		/*set base row if needed */
		if ((instructions[i].address & CS_SSRAM_ROW_SEGMENT_MASK) !=
		    segment) {
			segment =
			    instructions[i].
			    address & CS_SSRAM_ROW_SEGMENT_MASK;
			cs_ssramctrl_set_base_addr(ssram_dev, ssram_space,
			    segment);
		}

		cs_lup_write_ssram_instruction(ssram_dev, ssram_space,
		    &(instructions[i]));
	}
}

/*!
 * \brief Store data or mask rows to TCAM.
 *
 * \param tcam_dev TCAM Device descriptor.
 * \param tcam_space TCAM Address space descriptor.
 * \param data Data rows.
 * \param row_count Number of data/mask rows.
 *
 * \return 0 on succes, other on error.
 */
static int
cs_lup_store_rows_to_tcam(cs_device_t * tcam_dev,
    cs_space_t * tcam_space, cs_tcam_ctrl_row_t * rows,
    u_int32_t row_count, cs_tcam_ctrl_operation_t operation)
{
	u_int32_t offset;

	for (offset = 0; offset < row_count;
	    offset += CS_LUP_MAX_TCAM_ROWS_PER_BURST) {
		u_int32_t count;

		/*get lup row count for this burst */
		count = row_count - offset;
		if (count > CS_LUP_MAX_TCAM_ROWS_PER_BURST)
			count = CS_LUP_MAX_TCAM_ROWS_PER_BURST;

		if (!cs_tcam_ctrl_burst_write(tcam_dev, tcam_space, 0,
			rows + offset, count, offset, operation))
			return (-1);
	}

	return 0;
}

/*!
 * \brief Store data and mask to TCAM.
 *
 * \param tcam_dev TCAM Device descriptor.
 * \param tcam_space TCAM Address space descriptor.
 * \param data Data rows.
 * \param mask Mask rows corresponding to the data rows.
 * \param row_count Number of data/mask rows.
 *
 * \return 0 on succes, other on error.
 */
int
cs_lup_init_tcam(cs_device_t * tcam_dev, cs_space_t * tcam_space,
    cs_tcam_ctrl_row_t * data, cs_tcam_ctrl_row_t * mask, u_int32_t row_count)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "Initializing TCAM");

	/*init TCAM */
	VERBOSE(CL_VERBOSE_LIBRARY, "Allocating TCAM CTRL");
	cs_tcam_ctrl_init(tcam_dev, tcam_space);

	/*data */
	VERBOSE(CL_VERBOSE_LIBRARY, "Writing TCAM data");
	if (cs_lup_store_rows_to_tcam(tcam_dev, tcam_space, data, row_count,
		CS_TCAM_CTRL_OPERATION_DATA))
		return -1;

	/*mask */
	VERBOSE(CL_VERBOSE_LIBRARY, "Writing TCAM mask");
	if (cs_lup_store_rows_to_tcam(tcam_dev, tcam_space, mask, row_count,
		CS_TCAM_CTRL_OPERATION_MASK))
		return -1;

	VERBOSE(CL_VERBOSE_LIBRARY, "TCAM initialized");

	return 0;
}

/*!
 * \brief Total LUP initialization.
 *
 * \param dev Device descriptor.
 * \param lup_space LUP Address space descriptor.
 * \param ssram_space SSRAM Address space descriptor.  
 * \param tcam_space TCAM Address space descriptor.
 * \param config LUP configuration.
 *
 * \return 0 on succes, other on error.
 */

int 
cs_lup_init(cs_device_t * dev, cs_space_t * lup_space, cs_space_t * ssram_space,
	cs_space_t * tcam_space, cs_lup_config_t * config)
{
	u_int32_t i;

	cs_lup_init_tcam(dev, tcam_space, config->data_rows, config->mask_rows, config->tcam_row_count);
	cs_lup_init_ssram(dev, ssram_space, config->instructions, config->instruction_count);
	cs_lup_write_pu_first_address(dev, ssram_space, config->pu_first_addr);
	
	for(i=0;i<config->context_count;i++)
		cs_lup_write_cam_context(dev, lup_space, &(config->contexts[i]));
	
	cs_lup_write_packet_id_address(dev, lup_space, config->packet_id_addr);

	cs_lup_write_cam_control_memory(dev, lup_space);
	cs_lup_write_cam_index_ifc02(dev, lup_space);
	cs_lup_write_cam_index_ifc13(dev, lup_space);

	cs_lup_start_cam(dev, lup_space);
	cs_lup_start_pu(dev, lup_space);

	return 0;
}
