/*!
 * \file 	ibuf.c
 * \brief 	Functions for controlling IBUF component.
 * \author 	Andrej Hank <xhanka00@liberouter.org>
 * \author 	Miroslav Vadkerti <thrix@liberouter.org>
 * \date 	2006, 2007
 */

/* Copyright (C) 2006, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <stdbool.h>

#include <commlbr.h>

#include "../combosix.h"
#include "ibuf.h"

__RCSID("$Id$");

/* if not defined in global design - liberouter.h, define own base and width of
 * component */
#ifndef IBUF_GMII_BASE
	/*! Own base of IBUF component */
	#define IBUF_GMII_BASE		0x0
	/*! Own width of IBUF component*/
	#define IBUF_GMII_WIDTH		0x1000
#endif

/* single unit */
/*! Total Received Frames Counter (TRFC) */
#define IBUF_CNT_PACKETS		0x0000
/*! Correct Frames Counter (CFC) */
#define IBUF_CNT_RECV			0x0004
/*! Discarded Frames Counter (DFC) */
#define IBUF_CNT_RECVERR		0x0008
/*! Counter of frames discarded due to buffer overflow */
#define IBUF_CNT_OVERFLOW		0x000C
/*! Total Received Frames Counter (TRFC) */
#define IBUF_CNT_PACKETS_HI		0x0010
/*! Correct Frames Counter (CFC) */
#define IBUF_CNT_RECV_HI		0x0014
/*! Discarded Frames Counter (DFC) */
#define IBUF_CNT_RECVERR_HI		0x0018
/*! Counter of frames discarded due to buffer overflow */
#define IBUF_CNT_OVERFLOW_HI		0x001C
/*! IBUF enable register */
#define IBUF_EN				0x0020
/*! Error mask register */
#define IBUF_ERRMASK			0x0024
/*! IBUF status register */
#define IBUF_STATUS			0x0028
/*! IBUF command register */
#define IBUF_CTRL			0x002C
/*! Minimum frame length allowed */
#define IBUF_FRAME_MIN_LEN		0x0030
/*! Frame MTU */
#define IBUF_FRAME_MTU			0x0034
/*! MAC address check mode */
#define IBUF_MAC_CHECK			0x0038
/*! Frame buffer size */
#define IBUF_FRAME_BUFFER_SIZE		0x003C
/*! Memory of available MAC addresses */
#define IBUF_MEM_MAC			0x0080

/*!
 * \def		IBUFCMD_STROBE_COUNTERS
 * \brief	Constant for strobe counters command
 */
#define IBUFCMD_STROBE_COUNTERS		0x01

/*!
 * \def		IBUFCMD_RESET_COUNTERS
 * \brief	Constant for strobe counters command
 */
#define IBUFCMD_RESET_COUNTERS		0x02

/*!
 * \def		IBUFCMD_SET_SPEED10
 * \brief	Constant for set 10Mb/s command
 */
#define IBUFCMD_SET_SPEED10		0x04

/*!
 * \def		IBUFCMD_SET_SPEED100
 * \brief	Constant for set 100Mb/s command
 */
#define IBUFCMD_SET_SPEED100		0x05

/*!
 * \def		IBUFCMD_SET_SPEED1000
 * \brief	Constant for set 1000Mb/s command
 */
#define IBUFCMD_SET_SPEED1000		0x06

/*!
 * \def	MAX_MTU
 * \brief	Maximal allowed frame length, which has effect only
 * 		when appropriate register is not avaiable.
 */
#define MAX_MTU	4064

/*!
 * \def	ID_COMP_NAME
 * \brief	ID component name
 */
#define ID_COMP_NAME	"ID32"

/*!
 * \def ID_MAX_MTU
 * \brief Offset of ID register with max MTU
 */
#define ID_MAX_MTU	0x68

/*!
 * \brief 	Read IBUF counters
 * \param dev	Device
 * \param space	Space
 * \param xgmii XGMII IBUF flag
 * \return	Read data in cs_ibuf_t structure
 */
cs_ibuf_t
cs_ibuf_read (cs_device_t *dev, cs_space_t *space, bool xgmii)
{
	cs_ibuf_t	data;
	cs_space_t	*id_space;
	cs_component_info_t id_info = {.name = ID_COMP_NAME, .index = 0};
	int	k;

	cs_lock(CS_LOCK_IBUF);

	cs_space_write_4(dev, space, IBUF_CTRL, IBUFCMD_STROBE_COUNTERS);
	for(k=0; k<1000000; k++);

	/* Clear packet counters */
	data.cnt_packets = data.cnt_recv = data.cnt_error = data.cnt_overflow = 0;

	/* Read counters and registers */
	if(xgmii) {
		data.cnt_packets = cs_space_read_4(dev, space, IBUF_CNT_PACKETS_HI);
		data.cnt_recv    = cs_space_read_4(dev, space, IBUF_CNT_RECV_HI);
		data.cnt_error   = cs_space_read_4(dev, space, IBUF_CNT_RECVERR_HI);
		data.cnt_overflow = cs_space_read_4(dev, space, IBUF_CNT_OVERFLOW_HI);
	}
	data.cnt_packets = (data.cnt_packets << 32) +
		cs_space_read_4(dev, space, IBUF_CNT_PACKETS);
	data.cnt_recv = (data.cnt_recv << 32) +
		cs_space_read_4(dev, space, IBUF_CNT_RECV);
	data.cnt_error = (data.cnt_error << 32) +
		cs_space_read_4(dev, space, IBUF_CNT_RECVERR);
	data.cnt_overflow = (data.cnt_overflow << 32) +
		cs_space_read_4(dev, space, IBUF_CNT_OVERFLOW);

	data.reg_en       = cs_space_read_4(dev, space, IBUF_EN);
	data.err_mask     = cs_space_read_4(dev, space, IBUF_ERRMASK);
	data.ibuf_stat    = cs_space_read_4(dev, space, IBUF_STATUS);
	data.ibuf_frame_min_len = cs_space_read_4(dev, space, IBUF_FRAME_MIN_LEN);
	data.ibuf_frame_mtu = cs_space_read_4(dev, space, IBUF_FRAME_MTU);
	data.ibuf_mac_check = cs_space_read_4(dev, space, IBUF_MAC_CHECK);

	cs_unlock(CS_LOCK_IBUF);

	/* Get max MTU from ID component */
	cs_component_find_space(dev, NULL, &id_info, &id_space);
	data.ibuf_frame_buffer_size = cs_space_read_4(dev, id_space, ID_MAX_MTU);
	/* If there's no valid data, use default max MTU */
	if(data.ibuf_frame_buffer_size == 0) data.ibuf_frame_buffer_size = MAX_MTU;

	return data;
}

/*!
 * \brief 	Enable IBUF unit
 * \param dev	Device
 * \param space	Space
 */
void
cs_ibuf_enable (cs_device_t *dev, cs_space_t *space)
{
	/* enable */
	cs_space_write_4(dev, space, IBUF_EN, 1);
}

/*!
 * \brief 	Disable IBUF unit
 * \param dev	Device
 * \param space	Space
 */
void
cs_ibuf_disable (cs_device_t *dev, cs_space_t *space)
{
	/* disable */
	cs_space_write_4(dev, space, IBUF_EN, 0);
}

/*!
 * \brief 	Enable or disable MAC check mode
 * \param dev	Device
 * \param space	Space
 */
void
cs_ibuf_enable_mac_check (cs_device_t *dev, cs_space_t *space)
{
	u_int32_t	value;

	cs_lock(CS_LOCK_IBUF);

	value = cs_space_read_4(dev, space, IBUF_MAC_CHECK);
	/* disable */
	if(value & 1) cs_space_write_4(dev, space, IBUF_MAC_CHECK, 0);
	/* enable */
	else cs_space_write_4(dev, space, IBUF_MAC_CHECK, 1);

	cs_unlock(CS_LOCK_IBUF);
}

/*!
 * \brief 	Set MAC check mode
 * \param dev	Device
 * \param space	Space
 * \param mode	Mac check mode to set
 * \param enabled IBUF enabled flag
 */
void
cs_ibuf_mac_check (cs_device_t *dev, cs_space_t *space, u_int32_t mode, int enabled)
{
	cs_lock(CS_LOCK_IBUF);

	if(enabled) cs_ibuf_disable(dev, space);
	cs_space_write_4(dev, space, IBUF_MAC_CHECK, mode);
	if(enabled) cs_ibuf_enable(dev, space);

	cs_unlock(CS_LOCK_IBUF);
}

/*!
 * \brief 	Get list of filtered MAC addresses
 * \param dev	Device
 * \param space	Space
 * \param list	Allocated list of size #CS_IBUF_MACLIST_SIZE
 *  to store the values
 * \param enabled IBUF enabled flag
 * \param mac_count Number of MAC addresses that can be placed into CAM
 */
void
cs_ibuf_mac_get_list (cs_device_t *dev, cs_space_t *space, cs_ibuf_mac_t *list, int enabled, int mac_count)
{
	int	i, addr;

	cs_lock(CS_LOCK_IBUF);

	if(enabled) cs_ibuf_disable(dev, space);
	addr = IBUF_MEM_MAC;
	list[0].lo = 0xfffff;
	list[0].hi = 0xfffff;
	for (i = 0; i < mac_count; i++, addr += 8) {
		list[i].lo = cs_space_read_4(dev, space, addr);
		list[i].hi = cs_space_read_4(dev, space, addr + 4);
	}
	if(enabled) cs_ibuf_enable(dev, space);

	cs_unlock(CS_LOCK_IBUF);
}

/*!
 * \brief 	Set list of filtered MAC addresses
 * \param dev	Device
 * \param space	Space
 * \param list	Prepared list of size #CS_IBUF_MACLIST_SIZE
 *  to read the addresses from
 * \param enabled IBUF enabled flag
 * \param mac_count Number of MAC addresses that can be placed into CAM
 */
void
cs_ibuf_mac_set_list (cs_device_t *dev, cs_space_t *space, cs_ibuf_mac_t *list, int enabled, int mac_count)
{
	int	i, addr;

	cs_lock(CS_LOCK_IBUF);

	if(enabled) cs_ibuf_disable(dev, space);
	addr = IBUF_MEM_MAC;
	for (i = 0; i < mac_count; i++, addr += 8) {
		cs_space_write_4(dev, space, addr, list[i].lo);
		cs_space_write_4(dev, space, addr + 4, list[i].hi);
	}
	if(enabled) cs_ibuf_enable(dev, space);

	cs_unlock(CS_LOCK_IBUF);
}

/*!
 * \brief 	Set error mask
 * \param dev	Device
 * \param space	Space
 * \param mask	Error mask to set
 * \param enabled IBUF enabled flag
 */
void
cs_ibuf_err_mask (cs_device_t *dev, cs_space_t *space, u_int32_t mask, int enabled)
{
	cs_lock(CS_LOCK_IBUF);

	if(enabled) cs_ibuf_disable(dev, space);
	cs_space_write_4(dev, space, IBUF_ERRMASK, mask);
	if(enabled) cs_ibuf_enable(dev, space);

	cs_unlock(CS_LOCK_IBUF);

}
/*!
 * \brief 	Reset IBUF frame counters
 * \param dev	Device
 * \param space	Space
 */
void
cs_ibuf_reset(cs_device_t *dev, cs_space_t *space)
{
	cs_space_write_4(dev, space, IBUF_CTRL, IBUFCMD_RESET_COUNTERS);
}

/*!
 * \brief 	Set IBUF speed mode
 * \param dev	Device
 * \param space	Space
 * \param speed Speed to set (10|100|1000)
 */
void
cs_ibuf_set_speed(cs_device_t *dev, cs_space_t *space, int speed)
{
	u_int32_t value;

	switch(speed) {
		case 10: value = IBUFCMD_SET_SPEED10; break;
		case 100: value = IBUFCMD_SET_SPEED100; break;
		default:
		case 1000: value = IBUFCMD_SET_SPEED1000; break;
	}
	cs_space_write_4(dev, space, IBUF_CTRL, value);
}

/*!
 * \brief 	Set maximum/minimum frame length
 * \param dev	Device
 * \param space	Space
 * \param length Frame length
 * \param max 	If true maximum frame length is set, if false minimum.
 * \param enabled IBUF enabled flag
 */
void
cs_ibuf_frame_length (cs_device_t *dev, cs_space_t *space, u_int32_t length, int max, int enabled)
{
	cs_lock(CS_LOCK_IBUF);

	if(enabled) cs_ibuf_disable(dev, space);
	if(max) cs_space_write_4(dev, space, IBUF_FRAME_MTU, length);
	else cs_space_write_4(dev, space, IBUF_FRAME_MIN_LEN, length);
	if(enabled) cs_ibuf_enable(dev, space);

	cs_unlock(CS_LOCK_IBUF);
}
