/*!

 * \file i2cms.c
 * \brief Provides basic i2c bus access methods.
 * \author Jaroslav Zilak <xzilak00@stud.fit.vutbr.cz>
 * \date 2010
 */

/*
 * Copyright (C) 2010 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <err.h>
#include <errno.h>
#include <commlbr.h>

#include "i2cms.h"
 
/* CVS ID tag for ident(1) program */
__RCSID("$Id$");

/* I2C master space pointer */
static cs_space_t *i2cms_space = NULL;
/* I2C device pointer */
static cs_device_t *i2cms_device = NULL;
/* device id of I2C master required for 2.level I2C transmission */
static u_int8_t i2cms_dev_add = I2CMS_DEFAULT_DEVICE;
/* remote address of 2.level I2C master */
static u_int32_t i2cms_2_c_address = I2CMS_2_C_ADDRESS;
static u_int32_t i2cms_2_t_address = I2CMS_2_T_ADDRESS;
/* Device address length, may be zero */
static u_int32_t i2cms_add_len = I2CMS_DEFAULT_ADDR_LEN;
static u_int32_t i2cms_2_add_len = I2CMS_DEFAULT_ADDR_LEN;

/* global error flag */
static bool i2cms_error = false;

/*!
 * \brief This function is used as delay between signals sent to i2c bus.
 * Source of delay is pending on TIP bit which represents transfer in process.
 */
static int
i2c_wait_for_TIP_zero()
{
	int i;
	int cnt = 0; /* Maximal wait time (withou any time units) */
	u_int32_t reg_val;
	i2cms_error = false;
	do {
		/* wait cycle */
		for (i = 0; i < I2CMS_DELAY; ++i)
			__asm__ __volatile__ ("");
		++cnt;
		if (cnt > I2CMS_MAX_DELAY) {
			i2cms_error = true;
			return -1;
		}
	/* check if transfer finished */
	} while((reg_val = cs_space_read_4(i2cms_device, i2cms_space, 0)) &
		I2CMS_TIP);
	if (reg_val & (I2CMS_SLAVE_ACK_BIT | I2CMS_ARBIT_LOST_BIT)) {
		i2cms_error = true;
		return -1;
	}
	return 0;
}

/*!
 * \brief i2c module initialization function. 
 * Main purpose of this function is to simplify read/write function
 * calling. Before calling this function the i2c space must be correctly
 * mapped and the appropriate device attached.
 * 
 * \param device COMBO device
 * \param space Mapped COMBO space
 * 
 */
int
cs_i2cms_init(cs_device_t *device, cs_space_t *space)
{
	if ((device == NULL) || (space == NULL))
		return -1;
	i2cms_space = space;
	i2cms_device = device;

	return 0;
}

/*!
 * \brief level 2 i2c module initialization function. 
 * Main purpose of this function is to simplify read/write function
 * calling.
 * 
 * \param device device address of primary I2C master
 * \param address I2C master transfer register address
 * \param c_address level 2 i2c master constrol register address
 * \param t_address level 2 i2c master transfer register address
 * \param speed transmission speed (0 - 65535), where 0 is default
 *        I2CMS_REMOTE_SPEED.
 * 
 */
int
cs_i2cms_2_init(u_int32_t device, u_int32_t c_address, u_int32_t t_address)
{
	i2cms_dev_add = device;
	i2cms_2_c_address = c_address;
	i2cms_2_t_address = t_address;

	return 0;
}

/*!
 * \brief Function provides settings of i2c communication properties like:
 * i2c address length or remote address for secondary i2c master. Main
 * purpose of it is to extend usability and universality of whole
 * i2cms library module.
 * 
 * \param param determines which parameter should be set
 * \param value defines new value of selected parameter
 * 
 */
int
cs_i2cms_set_params(I2CMS_PARAM param, u_int32_t value)
{
	switch (param) {
	case I2CMS_PARAM_ADDR_LEN:
		if (value > I2CMS_MAX_ADDR_LEN) {
			return -1;
		}
		i2cms_add_len = value;
		break;
	case I2CMS_2_PARAM_ADDR_LEN:
		if (value > I2CMS_MAX_ADDR_LEN) {
			return -1;
		}
		i2cms_2_add_len = value;
		break;
	case I2CMS_2_PARAM_C_ADDRESS:
		i2cms_2_c_address = value;
		break;
	case I2CMS_2_PARAM_T_ADDRESS:
		i2cms_2_t_address = value;
		break;
	default:
		return -1;
	}
	return 0;
}

/*!
 * \brief Function writes \e data to specified \e address of i2c device
 *        selected by \e device_addr
 *
 * \param device_addr i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write.
 * \param size Number of bytes to write.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_write(u_int32_t device_addr, u_int16_t address, u_int8_t *data,
	u_int16_t size)
{
	u_int32_t regs = cs_space_read_4(i2cms_device, i2cms_space, 0) &
		0xFFFF0000;
	u_int16_t cnt = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/* start i2c write process & write device address */
	cs_space_write_4(i2cms_device, i2cms_space, 0, regs | I2CMS_START_BIT |
		I2CMS_WRITE_TO_SLAVE_BIT | ((device_addr & 0x7F) << 9));
	if (i2c_wait_for_TIP_zero()) {
		cs_unlock(CS_LOCK_I2C);
		return -1;
	}

	/* write address */
	if (i2cms_add_len) {
		address &= ~(~0UL << i2cms_add_len);
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_WRITE_TO_SLAVE_BIT | ((address & 0xFF) << 8));
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
	}
	if (i2cms_add_len > 8) {
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_WRITE_TO_SLAVE_BIT | (address & 0xFF00));
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
	}

	/* write data excluding last byte */
	while (cnt < size - 1) {
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_WRITE_TO_SLAVE_BIT | (data[cnt] << 8));
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
		++cnt;
	}

	/* write last byte & finish transfer */
	cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
		I2CMS_WRITE_TO_SLAVE_BIT | I2CMS_STOP_BIT | data[cnt] << 8);
	/* 
	 * We can't test return value, it'll allways be error after STOP_BIT.
	 * Purpouse of this is to delay next read/write.
	 */
	i2c_wait_for_TIP_zero();
	cs_unlock(CS_LOCK_I2C);

	return 0;
}

/*!
 * \brief Simplification function, which writes \e 8bits long data to
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(8b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_write_8b(u_int32_t device_addr, u_int16_t address, u_int8_t data)
{
	u_int8_t data_var = data;

	return cs_i2cms_write(device_addr, address, &data_var, 1);
}

/*!
 * \brief Simplification function, which writes \e 16bits long data to
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(16b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_write_16b(u_int32_t device_addr, u_int16_t address, uint16_t data)
{
	u_int16_t data_var = data;

	return cs_i2cms_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Simplification function, which writes \e 32bits long data to
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(32b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_write_32b(u_int32_t device_addr, u_int16_t address, uint32_t data)
{
	u_int32_t data_var = data;

	return cs_i2cms_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Simplification function, which writes \e 64bits long data to
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(64b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_write_64b(u_int32_t device_addr, u_int16_t address, uint64_t data)
{
	u_int64_t data_var = data;

	return cs_i2cms_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Function writes \e data to specified \e address of i2c device
 *        selected by \e device_addr on second i2c level, which means
 *        communication over two i2c buses.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write.
 * \param size Number of bytes to write.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_write(u_int32_t device_addr, u_int16_t address, u_int8_t *data,
	u_int16_t size)
{
	u_int32_t regs = cs_i2cms_read_32b(i2cms_dev_add, i2cms_2_c_address);
	u_int16_t cnt = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/* starting up remote master */
	cs_i2cms_write_32b(i2cms_dev_add, i2cms_2_c_address,
		(regs & 0xFFFF) | (I2CMS_START_BIT << 16));

	/* start i2c write process & write device address */
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
		(device_addr & 0x7F) << 1);
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
		I2CMS_START_BIT | I2CMS_WRITE_TO_SLAVE_BIT);

	/* check if device address was set */
	if (cs_i2cms_read_8b(i2cms_dev_add, i2cms_2_t_address) &
		(I2CMS_SLAVE_ACK_BIT | I2CMS_ARBIT_LOST_BIT)) {
		cs_unlock(CS_LOCK_I2C);
		return -1;
	}

	/* write address */
	if (i2cms_2_add_len) {
		address &= ~(~0UL << i2cms_2_add_len);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
			address & 0xFF);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_WRITE_TO_SLAVE_BIT);
	}
	if (i2cms_2_add_len > 8) {
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
			(address & 0xFF00) >> 8);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_WRITE_TO_SLAVE_BIT);
	}

	/* write data excluding last byte */
	while (cnt < size - 1) {
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1, 
			data[cnt]);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_WRITE_TO_SLAVE_BIT);
		++cnt;
	}

	/* write last byte & finish transfer */
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1, data[cnt]);
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
		I2CMS_WRITE_TO_SLAVE_BIT | I2CMS_STOP_BIT);

	/* stop remote master */
	cs_i2cms_write_32b(i2cms_dev_add, i2cms_2_c_address,
		regs & 0xFFFF);

	cs_unlock(CS_LOCK_I2C);
	return (i2cms_error) ? -1 : 0;
}

/*!
 * \brief Simplification function, which writes \e 8bits long data to
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(8b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_write_8b(u_int32_t device_addr, u_int16_t address, u_int8_t data)
{
	u_int8_t data_var = data;

	return cs_i2cms_2_write(device_addr, address, &data_var, 1);
}

/*!
 * \brief Simplification function, which writes \e 16bits long data to
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(16b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_write_16b(u_int32_t device_addr, u_int16_t address, uint16_t data)
{
	u_int16_t data_var = data;

	return cs_i2cms_2_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Simplification function, which writes \e 32bits long data to
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(32b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_write_32b(u_int32_t device_addr, u_int16_t address, uint32_t data)
{
	u_int32_t data_var = data;

	return cs_i2cms_2_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Simplification function, which writes \e 64bits long data to
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Destination address - where data will be written.
 * \param data Data to write(64b value).
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_write_64b(u_int32_t device_addr, u_int16_t address, uint64_t data)
{
	u_int64_t data_var = data;

	return cs_i2cms_2_write(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data));
}

/*!
 * \brief Function reads \e data from specified \e address of i2c device
 *        selected by \e device_addr
 *
 * \param device_addr i2c device address.
 * \param address Source address - from which data will be read.
 * \param data Buffer for read data.
 * \param size Size of buffer in bytes.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_read(u_int32_t device_addr, u_int16_t address, u_int8_t *data,
	u_int16_t size)
{
	u_int32_t regs = cs_space_read_4(i2cms_device, i2cms_space, 0) &
		0xFFFF0000;
	u_int16_t cnt = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/*
	 * start i2c read process & write device address to
	 * send register address
	 */
	cs_space_write_4(i2cms_device, i2cms_space, 0, regs | I2CMS_START_BIT |
		I2CMS_WRITE_TO_SLAVE_BIT | ((device_addr & 0x7F) << 9));
	if (i2c_wait_for_TIP_zero()) {
		cs_unlock(CS_LOCK_I2C);
		return -1;
	}

	/* write address */
	if (i2cms_add_len) {
		address &= ~(~0UL << i2cms_add_len);
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_WRITE_TO_SLAVE_BIT | ((address & 0xFF) << 8));
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
	}
	if (i2cms_add_len > 8) {
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_WRITE_TO_SLAVE_BIT | (address & 0xFF00));
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
	}

	/* write device address for data reading */
	cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
		I2CMS_START_BIT | I2CMS_WRITE_TO_SLAVE_BIT |
		((device_addr & 0x7F) << 9) | I2CMS_READING_FLAG);
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}

	/* read data excluding last byte */
	while (cnt < size - 1) {
		cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
			I2CMS_READ_FROM_SLAVE_BIT | I2CMS_READING_FLAG);
		if (i2c_wait_for_TIP_zero()) {
			cs_unlock(CS_LOCK_I2C);
			return -1;
		}
		data[cnt] = (cs_space_read_4(i2cms_device, i2cms_space, 0) >> 8) &
			0xFF;
		++cnt;
	}

	/* read last byte & finish transfer */
	cs_space_write_4(i2cms_device, i2cms_space, 0, regs |
		I2CMS_READ_FROM_SLAVE_BIT | I2CMS_STOP_BIT |
		I2CMS_READING_FLAG | I2CMS_ACK_BIT);
	/* 
	 * We can't test return value, it'll allways be error after STOP_BIT.
	 * Purpouse of this is to delay next read/write.
	 */
	i2c_wait_for_TIP_zero();
	data[cnt] = (cs_space_read_4(i2cms_device, i2cms_space, 0) >> 8) & 0xFF;
	cs_unlock(CS_LOCK_I2C);

	return 0;
}

/*!
 * \brief Simplification function, which returns read 8bits long data from
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int8_t
cs_i2cms_read_8b(u_int32_t device_addr, u_int16_t address)
{
	u_int8_t data_var = 0;

	cs_i2cms_read(device_addr, address, &data_var, 1);

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 16bits long data from
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int16_t
cs_i2cms_read_16b(u_int32_t device_addr, u_int16_t address)
{
	u_int16_t data_var;

	cs_i2cms_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 32bits long data from
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int32_t
cs_i2cms_read_32b(u_int32_t device_addr, u_int16_t address)
{
	u_int32_t data_var;

	cs_i2cms_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 64bits long data from
 *        specified \e address of i2c device selected by \e device_addr.
 *        No storage variable required.
 *
 * \param device_addr i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int64_t
cs_i2cms_read_64b(u_int32_t device_addr, u_int16_t address)
{
	u_int64_t data_var;

	cs_i2cms_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}

/*!
 * \brief Function reads \e data from specified \e address of i2c device
 *        selected by \e device_addr on second i2c level, which means
 *        communication over two i2c buses.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Source address - from which data will be read.
 * \param data Buffer for read data.
 * \param size Size of buffer in bytes.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2cms_2_read(u_int32_t device_addr, u_int16_t address, u_int8_t *data,
	u_int16_t size)
{
	u_int32_t regs = cs_i2cms_read_32b(i2cms_dev_add, i2cms_2_c_address);
	u_int16_t cnt = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/* starting up remote master */
	cs_i2cms_write_32b(i2cms_dev_add, i2cms_2_c_address,
		(regs & 0xFFFF) | (I2CMS_START_BIT << 16));

	/*
	 * start i2c read process & write device address to
	 * send register address
	 */
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
		(device_addr & 0x7F) << 1);
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
		I2CMS_START_BIT | I2CMS_WRITE_TO_SLAVE_BIT);

	/* check if device address was set */
	if (cs_i2cms_read_8b(i2cms_dev_add, i2cms_2_t_address) &
			(I2CMS_SLAVE_ACK_BIT | I2CMS_ARBIT_LOST_BIT)) {
		cs_unlock(CS_LOCK_I2C);
		return -1;
	}

	/* write address */
	if (i2cms_2_add_len) {
		address &= ~(~0UL << i2cms_2_add_len);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
			address & 0xFF);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_WRITE_TO_SLAVE_BIT);
	}
	if (i2cms_2_add_len > 8) {
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
			(address & 0xFF00) >> 8);
		cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_WRITE_TO_SLAVE_BIT);
	}

	/* write device address for data reading */
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address + 1,
		((device_addr & 0x7F) << 1) | (I2CMS_READING_FLAG >> 8));
	cs_i2cms_write_8b(i2cms_dev_add, i2cms_2_t_address,
		I2CMS_START_BIT | I2CMS_WRITE_TO_SLAVE_BIT);

	/* read data excluding last byte */
	while (cnt < size - 1) {
		cs_i2cms_write_16b(i2cms_dev_add, i2cms_2_t_address,
			I2CMS_READ_FROM_SLAVE_BIT);
		data[cnt] = cs_i2cms_read_8b(i2cms_dev_add,
			i2cms_2_t_address + 1);
		++cnt;
	}

	/* read last byte & finish transfer */
	cs_i2cms_write_16b(i2cms_dev_add, i2cms_2_t_address,
		I2CMS_READ_FROM_SLAVE_BIT | I2CMS_STOP_BIT |
		I2CMS_ACK_BIT);
	data[cnt] = cs_i2cms_read_8b(i2cms_dev_add, 
		i2cms_2_t_address + 1);

	/* stop remote master */
	cs_i2cms_write_32b(i2cms_dev_add, i2cms_2_c_address,
		regs & 0xFFFF);

	cs_unlock(CS_LOCK_I2C);
	return (i2cms_error) ? -1 : 0;
}

/*!
 * \brief Simplification function, which returns read 8bits long data from
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int8_t
cs_i2cms_2_read_8b(u_int32_t device_addr, u_int16_t address)
{
	u_int8_t data_var;

	cs_i2cms_2_read(device_addr, address, &data_var, 1);

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 16bits long data from
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int16_t
cs_i2cms_2_read_16b(u_int32_t device_addr, u_int16_t address)
{
	u_int16_t data_var;

	cs_i2cms_2_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 32bits long data from
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int32_t
cs_i2cms_2_read_32b(u_int32_t device_addr, u_int16_t address)
{
	u_int32_t data_var;

	cs_i2cms_2_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}

/*!
 * \brief Simplification function, which returns read 64bits long data from
 *        specified \e address of i2c device selected by \e device_addr on
 *        second i2c level. No storage variable required.
 *
 * \param device_addr 2.level i2c device address.
 * \param address Source address - from which data will be read.
 *
 * \retval read data.
 */
u_int64_t
cs_i2cms_2_read_64b(u_int32_t device_addr, u_int16_t address)
{
	u_int64_t data_var;

	cs_i2cms_2_read(device_addr, address, (u_int8_t *)&data_var,
		sizeof(data_var));

	return data_var;
}
