/*!

 * \file i2c.c
 * \brief Implementation of basic communication routines for HW and SW I2C controller.
 * \author Martin Louda <sandin@liberouter.org>
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2007-2009
 */

/*
 * Copyright (C) 2007-2009  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

/* local include */
#include "i2c.h"

/* liberouter include */
#include <commlbr.h>

/* std include */
#include <unistd.h>

__RCSID("$Id$");

/*! READ clock flag */
#define I2C_READ_CLK	0x1
/*! READ data bit */
#define I2C_READ_DATA	0x2
/*! WRITE bit mask */
#define I2C_WRITE 	0x4
/*! WR_CLK_Z bit mask */
#define I2C_WRITE_CLK_Z 0x8
/*! WR_CLK bit mask */
#define I2C_WRITE_CLK 	0x10
/*! WR_DATA bit mask */
#define I2C_WRITE_DATA 	0x40
/*! WR_DATA_Z bit mask */
#define I2C_WRITE_DATA_Z 0x20

/*! Programming delay in cycles */
#define I2C_DELAY 	30000 	
/*! Programming delay in cycles */
#define I2C_HW_DELAY 	500000 	
/*! Clocke stretching error delay */
#define I2C_CLK_STRETCH 100
/*! I2C read operation code. */
#define I2C_READ_OP 	0x1
/*! I2C write operation code. */
#define I2C_WRITE_OP 	0x0
/*! Error return code for functions */
#define I2C_ERR -1

/*! Specifies SW I2C controller type */
#define I2C_CTRL_SW 0

/*! Specifies HW I2C controller type */
#define I2C_CTRL_HW 1

/*!
 * Register address with attached i2c_sw interface.
 * Can be modified using cs_i2c_setreg function
 */
static u_int32_t i2c_reg = 0x0;

/*!
 * I2C address of the EEPROM. Default is 0xAC.
 */
static u_int32_t i2c_addr = 0xAC;

/*!
 * Number of data bytes read and written (default - data is 2 byte long - 16b).
 */
static u_int32_t i2c_bytes = 2;

/*!
 * I2C clk high signal - by default without clock stretching
 */
static u_int32_t i2c_clk_high = I2C_WRITE_CLK;

/*!
 * Specifyies controller type - default SW I2C controller
 */
static u_int32_t i2c_ctrl_type = I2C_CTRL_SW;

/*!
 * \brief Function is used as delay between signals sent to i2c bus.
 */
static void
i2c_delay()
{
	int k;			/* cycle variable */

	/* wait cycle */
	for (k = 0; k < I2C_DELAY; k++)
		__asm__ __volatile__ ("");
}

/*!
 * \brief Function is used as delay between signals sent to i2c bus.
 */
static void
i2c_hw_delay()
{
	int k;			/* cycle variable */

	/* wait cycle */
	for (k = 0; k < I2C_HW_DELAY; k++)
		__asm__ __volatile__ ("");
}

/*!
 * \brief Function writes one bit to i2c bus specified with \e sel number.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus (maximum 4).
 * \param data Written bit value (0 - 1).
 *
 * \return 0 on success, -1 if error
 */
static int
i2c_write_bit(cs_device_t * dev, cs_space_t * phy, u_int32_t sel, u_int32_t data)
{
	/* written data bit */
	u_int32_t data_bit = (data == 0) ? 0 : I2C_WRITE_DATA_Z;
	u_int32_t bus;
	int count = 0;

	/* perform writing to bus */
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE | data_bit) << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE | i2c_clk_high | data_bit) << sel * 8);
	while(1) {
		count++;
		if(count > I2C_CLK_STRETCH) {
			VERBOSE(CL_VERBOSE_LIBRARY, "I2C communication failed: i2c_write_bit clock stretching");
			return I2C_ERR;
		}
		bus = cs_space_read_4(dev, phy, i2c_reg);
		if(((bus >> sel*8) & I2C_READ_CLK) == 1) break;
	}
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE | data_bit) << sel * 8);

	return 0;
}

/*!
 * \brief Function reads one bit from i2c bus specified \e sel.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus - number in interval <0, 3>.
 *
 * \retval Returns the read bit from device.
 */
static int32_t
i2c_read_bit(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	/* read data bit */
	u_int32_t data_bit;
	int count = 0;

	/* perform reading from bus */
	cs_space_write_4(dev, phy, i2c_reg,
	    (I2C_WRITE_DATA_Z | I2C_WRITE) << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE_DATA_Z | i2c_clk_high
		| I2C_WRITE) << sel * 8);
	while(1) {
		count++;
		if(count > I2C_CLK_STRETCH) {
			VERBOSE(CL_VERBOSE_LIBRARY, "I2C communication failed: i2c_read_bit clock stretching");
			return I2C_ERR;
		}
		data_bit = cs_space_read_4(dev, phy, i2c_reg);
		if(((data_bit >> sel*8) & I2C_READ_CLK) == 1) break;
	}
	i2c_delay();
	data_bit = cs_space_read_4(dev, phy, i2c_reg);
	cs_space_write_4(dev, phy, i2c_reg,
	    (I2C_WRITE_DATA_Z | I2C_WRITE) << sel * 8);
	i2c_delay();

	/* return read bit */
	return ((data_bit >> sel * 8) & I2C_READ_DATA) >> 1;
}

/*!
 * \brief Function determines if i2c bus for selected device
 *        is busy.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 *
 * \retval Returns true value if bus busy else returns false value.
 */
static int
i2c_bus_busy(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	/* read value */
	u_int32_t bus;

	/* read 32 bit value */
	i2c_delay();
	bus = cs_space_read_4(dev, phy, i2c_reg);
	i2c_delay();

	/* bus is free when SCL & SDA = 1 */
	return ((bus >> sel * 8) & (I2C_READ_CLK | I2C_READ_DATA))
	    != (I2C_READ_CLK | I2C_READ_DATA);
}

/*!
 * \brief Function writes start bit to i2c bus of selected device
 *        by \e sel.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 */
static void
i2c_start_cond(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE_DATA | i2c_clk_high
		| I2C_WRITE) << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (i2c_clk_high
		| I2C_WRITE) << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, I2C_WRITE << sel * 8);
	i2c_delay();
}

/*!
 * \brief Function writes stop bit to i2c bus of selected device by \e sel.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 *
 * \return I2C_ERR on error, 0 on success
 */
static int
i2c_stop_cond(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	u_int32_t bus;
	u_int32_t count = 0;

	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, I2C_WRITE << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg,
	    (i2c_clk_high | I2C_WRITE) << sel * 8);
	while(1) {
		count++;
		if(count > I2C_CLK_STRETCH) {
			VERBOSE(CL_VERBOSE_LIBRARY, "I2C communication failed: i2c_stop_cond clock stretching");
			return I2C_ERR;
		}
		bus = cs_space_read_4(dev, phy, i2c_reg);
		if(((bus >> sel*8) & I2C_READ_CLK) == 1) break;
		VERBOSE(CL_VERBOSE_LIBRARY, "Clock stretching: i2c_stop_cond");
	}
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE_DATA | i2c_clk_high
		| I2C_WRITE) << sel * 8);
	i2c_delay();
	cs_space_write_4(dev, phy, i2c_reg, (I2C_WRITE_DATA | I2C_WRITE_CLK
		| I2C_WRITE_CLK_Z | I2C_WRITE) << sel * 8);
	i2c_delay();

	return 0;
}

/*!
 * \brief Function writes slave address (7bit value) with r/w bit
 *        to i2c bus.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param addr Selected i2c bus.
 * \param rw Specifies r/w bit (last written bit).
 *
 */
static void
i2c_send_addr(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
    u_int32_t addr, u_int32_t rw)
{
	/* cycle variable */
	int i;

	/* write address bits (MSB first) */
	for (i = 6; i >= 0; i--) {
		i2c_write_bit(dev, phy, sel, (addr >> i) & 0x1);
	}

	/* write r/w bit */
	i2c_write_bit(dev, phy, sel, rw & 0x1);
}

/*!
 * \brief Function reads one byte for i2c bus of selected device by \e sel.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 *
 * \retval Returns the read one byte as a 32bit value.
 */
static int32_t
i2c_read_byte(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	int i;
	u_int32_t data = 0;	/* read data */
	int32_t rval;		/* read bit */

	/* read 8 bits form i2c bus */
	for (i = 0; i < 8; i++) {
		rval = i2c_read_bit(dev, phy, sel);				
		if(rval == I2C_ERR) return I2C_ERR;
		data = (data << 1) | rval;
	}

	/* return read data */
	return data;
}

/*!
 * \brief Function writes one byte to i2c bus of selected device by \e sel.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param data Selected i2c bus.
 *
 * \return 0 on sucess, -1 else
 */
static int
i2c_write_byte(cs_device_t * dev, cs_space_t * phy, u_int32_t sel, u_int32_t data)
{
	/* cycle variable */
	int i;
	int ret;

	/* write 8 bits to i2c bus */
	for (i = 7; i >= 0; i--) {
		ret = i2c_write_bit(dev, phy, sel, (data >> i) & 1);
		if(ret == I2C_ERR) return I2C_ERR;
	}

	return 0;
}

/*!
 * \brief Function returns acknowledge status.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space with device component.
 * \param sel Selected i2c bus.
 *
 * \retval Returns true if device acknowledged else false.
 */
static int
i2c_read_ack(cs_device_t * dev, cs_space_t * phy, u_int32_t sel)
{
	int32_t rval;

	/*
	 * ACK : SDA = 0
	 * !ACK: SDA = 1
	 */
	rval = i2c_read_bit(dev, phy, sel) ;
	if(rval == I2C_ERR) return I2C_ERR;
	return rval == 0x0;
}

/*!
 * \brief Function sends acknowledge or not-acknowledge
 *        to i2c bus.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param acsel If true, send ack, else nack.
 */
static void
i2c_send_ack(cs_device_t * dev, cs_space_t * phy, u_int32_t sel, int acsel)
{
	/*
	 * ACK : SDA = 0
	 * !ACK: SDA = 1
	 */
	if (acsel)
		i2c_write_bit(dev, phy, sel, 0x0);
	else
		i2c_write_bit(dev, phy, sel, 0x1);
}

/*!
 * \brief Function writes data to register \e reg
 *        of i2c bus specified by \e sel. Use with SW I2C controller.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param reg Register address.
 * \param data Efficient data - size i2c_bytes byte long.
 *
 * \retval 0 on success else true value.
 */
int
i2c_sw_write(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
    u_int32_t reg, u_int32_t data)
{
	u_int32_t wdata;	/* written data */
	int k;			/* cycle variable */
	int count = 0;
	int ret;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/* start transfer only when bus is free */
	while (i2c_bus_busy(dev, phy, sel)) {
		count++;
		if(count > I2C_CLK_STRETCH) {
			VERBOSE(2, "I2C communication failed: bus busy");
			cs_unlock(CS_LOCK_I2C);
			return I2C_ERR;
		}
	}

	VERBOSE(0, "Write to register 0x%X data 0x%X", reg, data);
	for(k = i2c_bytes - 1; k >= 0; k--) {
		/* first, send start bit */
		i2c_start_cond(dev, phy, sel);

		/* send device address */
		i2c_send_addr(dev, phy, sel, (i2c_addr) >> 1, I2C_WRITE_OP);

		/* and wait for device to ACK */
		if (i2c_read_ack(dev, phy, sel)) {
			/* create address byte */
			wdata = 0x00FF & reg;

			/* write address byte */
			ret = i2c_write_byte(dev, phy, sel, wdata);
			if(ret == I2C_ERR) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}

			/* read ACK */
			i2c_read_ack(dev, phy, sel);

			/* create upper data byte */
			wdata = (data >> 8*k) & 0xFF;

			/* write upper data byte */
			ret = i2c_write_byte(dev, phy, sel, wdata);
			if(ret == I2C_ERR) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}

			/* read ACK */
			i2c_read_ack(dev, phy, sel);

			/* end transfer */
			ret = i2c_stop_cond(dev, phy, sel);
			if(ret == I2C_ERR) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}
		} else {
			/* end transfer */
			ret = i2c_stop_cond(dev, phy, sel);
			if(ret == I2C_ERR) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}
		}
	}

	/* unlock I2C locker */
	cs_unlock(CS_LOCK_I2C);
	return 0;
}

/*!
 * \brief Function reads data from register \e reg
 *        of if i2c bus specified by \e sel from SW I2C controller.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param reg Read register address.
 * \param data Read data.
 *
 * \retval 0 on success else true value.
 */
int
i2c_sw_read(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
	u_int32_t reg, u_int32_t *data)
{
	u_int32_t	k; /* cycle variable */
	int32_t		rbyte;
	int		ret;

	/* initialize data */
	*data = 0;

	int count = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	/* read i2c_bytes bytes */
	for(k = 0; k < i2c_bytes; k++) {
		/* start transfer only when bus is free */
		while(i2c_bus_busy(dev, phy, sel)) {
			VERBOSE(2, "Bus busy - waiting");
			count++;
			if(count > I2C_CLK_STRETCH) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}
		}

		/* first, send start bit */
		i2c_start_cond(dev, phy, sel);

		/* send device address */
		i2c_send_addr(dev, phy, sel, (i2c_addr) >> 1, I2C_WRITE_OP);

		/* read ack */
		if (i2c_read_ack(dev, phy, sel) == 0) {
			/* end transfer */
			i2c_stop_cond(dev, phy, sel);
			VERBOSE(CL_VERBOSE_LIBRARY, "NOT ACKED 1: register: 0x%X", reg);
			cs_unlock(CS_LOCK_I2C);
			/* no acknowledge - error */
			return I2C_ERR;
		}

		/* send start reg address */
		ret = i2c_write_byte(dev, phy, sel, reg);
		if(ret == I2C_ERR) {
			cs_unlock(CS_LOCK_I2C);
			return I2C_ERR;
		}

		/* read ack */
		if (i2c_read_ack(dev, phy, sel) == 0) {
			/* end transfer */
			i2c_stop_cond(dev, phy, sel);
			VERBOSE(CL_VERBOSE_LIBRARY, "NOT ACKED 2: register: 0x%X", reg);
			cs_unlock(CS_LOCK_I2C);
			/* no acknowledge - error */
			return I2C_ERR;
		}

		/* send start bit again */
		i2c_start_cond(dev, phy, sel);

		/* send device address */
		i2c_send_addr(dev, phy, sel, (i2c_addr) >> 1, I2C_READ_OP);

		/* and wait for device to ACK */
		if (i2c_read_ack(dev, phy, sel)) {
			/* receive data byte*/			
			rbyte = i2c_read_byte(dev, phy, sel);
			if(rbyte == I2C_ERR) {
				cs_unlock(CS_LOCK_I2C);
				return I2C_ERR;
			}
			*data = (*data << 8) | rbyte;
		}
		/* device did not ack :( */
		else {
			/* send !ACK -> won't accept more data */
			i2c_send_ack(dev, phy, sel, 0);

			/* end transfer */
			i2c_stop_cond(dev, phy, sel);

			VERBOSE(CL_VERBOSE_LIBRARY, "NOT ACKED 3: register: 0x%X", reg);
			cs_unlock(CS_LOCK_I2C);

			return I2C_ERR;
		}

		/* send !ACK -> won't accept more data */
		i2c_send_ack(dev, phy, sel, 0);

		/* end transfer */
		ret = i2c_stop_cond(dev, phy, sel);
		if(ret == I2C_ERR) {
			cs_unlock(CS_LOCK_I2C);
			return I2C_ERR;
		}

	}

	/* unlock I2C locker */
	cs_unlock(CS_LOCK_I2C);

	/* return success */
	return 0;
}

/*!
 * \brief Function reads data from register \e reg
 *        from HW I2C controller.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param reg Read register address.
 * \param data Read data.
 *
 * \retval 0 on success else true value.
 */
int
i2c_hw_read(cs_device_t * dev, cs_space_t * i2c, u_int32_t reg, u_int32_t *data)
{
	u_int32_t i;

	/* Init data */
	*data = 0;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	i2c_hw_delay();

	/* Start & device address & write */
	cs_space_write_4(dev, i2c, i2c_reg, i2c_addr << 8 | 0x90);

	i2c_hw_delay();

	/* Write starting adress & stop */
	cs_space_write_4(dev, i2c, i2c_reg, reg << 8 | 0x50);

	i2c_hw_delay();

	/* Start & device address & read */
	cs_space_write_4(dev, i2c, i2c_reg, (i2c_addr | 1) << 8 | 0x90);

	i2c_hw_delay();

	for(i = 0; i < i2c_bytes-1; i++) {
		/* Read & no ACK & stop */
		cs_space_write_4(dev, i2c, i2c_reg, (i2c_addr | 1) << 8 | 0x20);

		i2c_hw_delay();

		/* Read result */
		*data = *data << 8 | ((cs_space_read_4(dev, i2c, i2c_reg) >> 8) & 0xFF);
	}

	/* Read & no ACK & stop */
	cs_space_write_4(dev, i2c, i2c_reg, (i2c_addr | 1) << 8 | 0x68);

	i2c_hw_delay();

	/* Read result */
	*data = *data << 8 | ((cs_space_read_4(dev, i2c, i2c_reg) >> 8) & 0xFF);

	cs_unlock(CS_LOCK_I2C);
	return 0;
}

/*!
 * \brief Function writes data to register \e reg
 *        via HW I2C controller.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param reg Register address to write.
 * \param data Write data.
 *
 * \retval 0 on success else true value.
 */
int
i2c_hw_write(cs_device_t * dev, cs_space_t * i2c, u_int32_t reg, u_int32_t data)
{
	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	i2c_hw_delay();

	/* Start & device address & write */
	cs_space_write_4(dev, i2c, i2c_reg, i2c_addr << 8 | 0x90);

	i2c_hw_delay();

	/* Write starting adress & stop */
	cs_space_write_4(dev, i2c, i2c_reg, reg << 8 | 0x10);

	i2c_hw_delay();

	if(i2c_bytes == 2) {
	
		/* Write data & ack */
		cs_space_write_4(dev, i2c, i2c_reg, (data >> 8) << 8 | 0x10);

		i2c_hw_delay();

		/* Write data & ack & stop */
		cs_space_write_4(dev, i2c, i2c_reg, (data & 0xFF) << 8 | 0x50);

		i2c_hw_delay();


	} else if(i2c_bytes == 1) {

		/* Write data & ack & stop */
		cs_space_write_4(dev, i2c, i2c_reg, data << 8 | 0x50);

		i2c_hw_delay();

	} else {
		cs_unlock(CS_LOCK_I2C);
		return -1;
	}

	cs_unlock(CS_LOCK_I2C);

	return 0;
}

/*!
 * \brief Function writes data to register \e reg
 *        of i2c bus specified by \e sel. Not usable for cryptochips.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus -- used only with SW I2C controller
 * \param reg Register address.
 * \param data Efficient data - size i2c_bytes byte long.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2c_write(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
    u_int32_t reg, u_int32_t data)
{
	VERBOSE(2, "I2C write reg: 0x%X data: 0x%X", reg, data);
	if(i2c_ctrl_type == I2C_CTRL_HW) 
		return i2c_hw_write(dev, phy, reg, data);		
	else return i2c_sw_write(dev, phy, sel, reg, data);
}

/*!
 * \brief Function reads data from register \e reg
 *        of if i2c bus specified by \e sel. Not usable for cryptochips.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus -- used only with SW I2C controller
 * \param reg Read register address.
 * \param data Read data.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2c_read(cs_device_t * dev, cs_space_t * phy, u_int32_t sel,
	u_int32_t reg, u_int32_t *data)
{
	if(i2c_ctrl_type == I2C_CTRL_HW) 
		return i2c_hw_read(dev, phy, reg, data);
	else return i2c_sw_read(dev, phy, sel, reg, data);
}


/*!
 * \brief Function sets register with attached i2c_sw signals.
 * \param address Address of register.
 */
void
cs_i2c_set_reg(u_int32_t address)
{
	i2c_reg = address;
}

/*!
 * \brief Function sets register with attached i2c_sw signals.
 * \param address Address of register.
 */
void
cs_i2c_set_addr(u_int32_t address)
{
	i2c_addr = address;
}

/*!
 * \brief Function changes number of data bytes used (default 2).
 * \param count Number of data i2c bytes used.
 */
void
cs_i2c_set_data_bytes(u_int32_t count)
{
	i2c_bytes = count;
}

/*!
 * \brief Function detects data length of EEPROM.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param sel Selected i2c bus.
 * \param addr I2C device address
 *
 * \return 0 - detected 8bit EEPROM
 * \return 1 - detected 16bit EEPROM
 * \return -1 - error reading
 */
int
cs_i2c_detect_dlength(cs_device_t * dev, cs_space_t * phy, u_int32_t sel, u_int32_t addr)
{

	u_int32_t	regs[10]; /* trying on sample of 10 registers */
	int		flag = 0; /* 16bit flag */
	int 		k;

	/* 2byte long registers */
	cs_i2c_set_data_bytes(2);
	cs_i2c_set_addr(addr);

	for(k = 0; k < 2; k++) {
		if(cs_i2c_read(dev, phy, sel, k, &regs[k]) != 0) return -1;
		if(regs[k] >> 8 != (regs[k] & 0xFF)) {
			flag = 1;
			break;
		}
	}

	return flag;
}

/*!
 * \brief Function enables/disables clock stretching in communication
 *        (_CLK signal keeped low byt transceiver, _CLK_Z drives high CLK).
 * \param enable 1 to enable clock stretching, 0 to disable
 */
void
cs_i2c_enable_clk_stretch(u_int32_t enable)
{
	if(enable) i2c_clk_high = I2C_WRITE_CLK_Z;
	else i2c_clk_high = I2C_WRITE_CLK;
}

/*!
 * \brief Function initializes i2c module. 
 *
 * It detects the propper I2C controller type. This influences 
 * the behavior of the cs_i2c_read/write functions. If this function
 * is not called, the functions assume communication with elder SW I2C 
 * controller because of backward compatibility.
 *
 * Before calling this function the space i2c must be correctly mapped
 * and i2c register address mut be set. 
 *
 * \param dev	COMBO device
 * \param i2c	Mapped COMBO space
 * 
 */
void
cs_i2c_init(cs_device_t * dev, cs_space_t * i2c) 
{
	u_int32_t oldval = 0;
	u_int32_t temp;
	u_int8_t array[2];
	u_int8_t cmd[4] = {0xB6,0x0,0x0,0x1};

	i2c_reg = 0x4;

	/* lock I2C access in libcombo */
	cs_lock(CS_LOCK_I2C);

	/* read current value */
	VERBOSE(CL_VERBOSE_LIBRARY, "I2C init: i2c_reg: 0x%X", i2c_reg);
	oldval = cs_space_read_4(dev, i2c, i2c_reg);

	/* write 0 */
	cs_space_write_4(dev, i2c, i2c_reg, 0x0);

	/* read back value */
	temp = cs_space_read_4(dev, i2c, i2c_reg) & 2;
	VERBOSE(CL_VERBOSE_LIBRARY, "I2C init: read back from controller: 0x%X -> %s I2C controller",
		temp, temp ? "SW" : "HW");

	if(!temp) {
		i2c_ctrl_type = I2C_CTRL_HW;
		/* enable HW I2C controller */
		cs_space_write_4(dev, i2c, 0, 0x8000f9);

		cs_i2c_read_array(dev,i2c,cmd,array);	
	}

	/* write original value */
	cs_space_write_4(dev, i2c, i2c_reg, oldval);

	cs_unlock(CS_LOCK_I2C);
}

/*!
 * \brief Function executes write command using array \array. Usable only for cryptochips.
 *
 * \param dev Pointer to combo device.
 * \param i2c Pointer to mapped combo space device.
 * \param size Size of array.
 * \param array Array with data to write.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2c_write_array(cs_device_t * dev, cs_space_t * i2c, u_int32_t size,
	u_int8_t *array)
{
	u_int32_t k;

	if(size < 4) {
		VERBOSE(2, "cs_i2c_write_array: array must contain at least 5 values");
		return -1;
	}

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	i2c_hw_delay();

	/* Read system zone command, I2C START */
	i2c_hw_delay();
	cs_space_write_4(dev, i2c, i2c_reg, array[0] << 8 | 0x90);

	i2c_hw_delay();

	/* Write address high byte - 0x00 */
	cs_space_write_4(dev, i2c, i2c_reg, array[1] << 8 | 0x10);

	i2c_hw_delay();

	/* Write address low byte = $1 */
	cs_space_write_4(dev, i2c, i2c_reg, array[2] << 8 | 0x10);

	i2c_hw_delay();

	/* Write number of words to write */
	cs_space_write_4(dev, i2c, i2c_reg, array[3] << 8 | 0x10);

	i2c_hw_delay();

	if(size == 4) {
		cs_unlock(CS_LOCK_I2C);
		return 0;
	}

	for(k = 4; k < size-1; k++) {
		/* write & STOP & SEND ACK */
		cs_space_write_4(dev, i2c, i2c_reg, array[k] << 8 | 0x10);
		
		i2c_hw_delay();
	}

	i2c_hw_delay();
	/* write & STOP & SEND ACK */
	cs_space_write_4(dev, i2c, i2c_reg, array[size-1] << 8 | 0x50);

	i2c_hw_delay();
	cs_unlock(CS_LOCK_I2C);
	return 0;
}

/*!
 * \brief Function executes read command \cmd and stores read data to \e data array. Usable only for cryptochips.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped combo space device.
 * \param cmd Command data array (always 4 bytes array).
 * \param data Read data array.
 *
 * \retval 0 on success else true value.
 */
int
cs_i2c_read_array(cs_device_t * dev, cs_space_t * i2c, u_int8_t *cmd, u_int8_t *data)
{
	int k;

	/* lock I2C access in libcombo */
	if (cs_lock(CS_LOCK_I2C) != 0) {
		return -1;
	}

	i2c_hw_delay();

	/* Read system zone command, I2C START */
	cs_space_write_4(dev, i2c, i2c_reg, cmd[0] << 8 | 0x90);

	i2c_hw_delay();

	/* Write address high byte - 0x00 */
	cs_space_write_4(dev, i2c, i2c_reg, cmd[1] << 8 | 0x10);

	i2c_hw_delay();

	/* Write address low byte = $1 */
	cs_space_write_4(dev, i2c, i2c_reg, cmd[2] << 8 | 0x10);

	i2c_hw_delay();

	/* Write number of words to read = 1 */
	cs_space_write_4(dev, i2c, i2c_reg, cmd[3] << 8 | 0x10);

	i2c_hw_delay();

	VERBOSE(2, "TO READ %d items", cmd[3]);
	for(k = 0; k < cmd[3]-1; k++) {
		/* Read & STOP & SEND ACK */
		cs_space_write_4(dev, i2c, i2c_reg, 0x120);
	
		i2c_hw_delay();

		/* Read result */
		data[k] = (cs_space_read_4(dev, i2c, i2c_reg) >> 8) & 0xFF;
		VERBOSE(2, "READ %d: 0x%2X", k, data[k]);

		i2c_hw_delay();
	}

	/* Read & STOP & SEND ACK */
	cs_space_write_4(dev, i2c, i2c_reg, 0x168);

	i2c_hw_delay();

	/* Read result */
	data[k] = (cs_space_read_4(dev, i2c, i2c_reg) >> 8) & 0xFF;

	i2c_hw_delay();
	cs_unlock(CS_LOCK_I2C);
	return 0;
}

