/*!
 * \file hfe2.c
 * \brief Functions for controlling hfe2
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2006, 2007
 */
/*
 * Copyright (C) 2006, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <stdio.h>
#include <unistd.h>
#include <commlbr.h>

#include "hfe2.h"

__RCSID ("$Id$");

/*! 
 * \brief Initialize instruction cache from file
 * 
 * \param dev Device
 * \param space Space
 * \param src_file_name Source file
 * 
 * \return Number of written rows
 */
int cs_hfe2_init_from_file(cs_device_t * dev, cs_space_t * space, char * src_file_name) {
	int count;
	cs_hfe2_row_t *rows;
	
	/* read rows */
	count = cs_hfe2_read_rows_from_file(src_file_name, &rows);
	/* fill */
	cs_hfe2_init(dev, space, rows, count);
	/* free rows */
	free(rows);

	return count;
}

/*! 
 * \brief Initialize instruction cache
 * 
 * \param dev Device
 * \param space Space
 * \param rows Rows to be written
 * \param count Count of rows
 */
void cs_hfe2_init(cs_device_t *dev, cs_space_t *space, cs_hfe2_row_t *rows, int count) {
	int i;
	VERBOSE (CL_VERBOSE_LIBRARY, "Initializing hfe2...");
	for(i = 0; i < count; i++) {
		MSG (CL_VERBOSE_LIBRARY, "%d : 0x%08x -> 0x%08x", i, rows[i], CS_HFE2_INST_MEM + i * 4);
		cs_space_write_4 (dev, space, CS_HFE2_INST_MEM + i * 4, rows[i]);
	}
	VERBOSE (CL_VERBOSE_LIBRARY, "%d rows written successfully", count);
}

/*! 
 * \brief Read rows from file and write into array of cs_hfe2_row_t
 * 
 * \param src_file_name Source file of format 8hex without 0x prefix each on
 * single line
 * \param rows Pointer to array of cs_hfe2_row_t, necessary memory will be
 * allocated, must be freed
 * 
 * \return Number of read lines
 */
int cs_hfe2_read_rows_from_file (char * src_file_name, cs_hfe2_row_t ** rows) {

	const int BUFFER_SIZE = 10;
	FILE * src_file;
	char buffer[BUFFER_SIZE];
	int count = 0;
	int allocated_items;
	u_int32_t value;
	cs_hfe2_row_t *tmp_rows;

	/* Load file. */
	if ((src_file = fopen(src_file_name, "r")) == NULL)
		errx(1, "Error open source file %s.", src_file_name);

	/* allocate memory for returning array, initial size set to 16 */
	allocated_items = 16;
	if(!(*rows = (cs_hfe2_row_t*)malloc(allocated_items * sizeof(cs_hfe2_row_t)))) {
		fclose(src_file);
		errx(1, "Memory allocation failed");
	}

	while (fgets(buffer, BUFFER_SIZE, src_file) != NULL) {
		if (sscanf(buffer, "%08x", &value) != EOF) {
			count++;
			if(allocated_items < count) {
				/* double size */
				allocated_items *= 2;
				if(!(tmp_rows = (cs_hfe2_row_t*)realloc(*rows, allocated_items * sizeof(cs_hfe2_row_t)))) {
					free(*rows);
					fclose(src_file);
					errx(1, "Memory allocation failed");
				}
				*rows = tmp_rows;
			}

			/* write value into array */
			(*rows)[count-1] = value;
			MSG(2, "hfe2 row converted: 0x%08x",value);
		}
	}

	fclose(src_file);

	return count;
}

/*! 
 * \brief Set hfe2 running
 * 
 * \param dev Device
 * \param space Space
 */
void cs_hfe2_set_running(cs_device_t *dev, cs_space_t *space) {
	VERBOSE (CL_VERBOSE_LIBRARY,"Setting core running\n");
	VERBOSE(CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 2, CS_HFE2_CTRL_REG);
	cs_space_write_4 (dev, space, CS_HFE2_CTRL_REG, 2);
}

/*! 
 * \brief Reset hfe2
 * 
 * \param dev Device
 * \param space Space
 */
void cs_hfe2_reset(cs_device_t *dev, cs_space_t *space) {
        VERBOSE (CL_VERBOSE_LIBRARY,"Reset core\n");       
	VERBOSE(CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 1, CS_HFE2_CTRL_REG);
        cs_space_write_4 (dev, space, CS_HFE2_CTRL_REG, 1);
}

/*! 
 * \brief Stop hfe2, immediate or wait to finish packet, waits until hfe2 really
 * stops
 * 
 * \param dev Device
 * \param space Space
 * \param immediate Stop immeiately, do not wait to finish packet
 */
void cs_hfe2_stop(cs_device_t *dev, cs_space_t *space, bool immediate) {
	u_int32_t value;
	int i;
	
	if(immediate) { /* immediate stop */
                VERBOSE (CL_VERBOSE_LIBRARY,"Immediate stop\n");
                value = 1 << 2;
	} else { /* stop after finishing current packet */
		VERBOSE (CL_VERBOSE_LIBRARY,"Waiting for finishing currnet packet\n");                
                value = 1 << 3;
	}

        VERBOSE (CL_VERBOSE_LIBRARY,"Stopping core\n");
	VERBOSE(CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_HFE2_CTRL_REG);
        cs_space_write_4 (dev, space, CS_HFE2_CTRL_REG, value);

        VERBOSE (CL_VERBOSE_LIBRARY,"Waiting for core to stop\n");
	/* waiting for core to stop */
        for (i = 0; i < CS_HFE2_CORE_STOP_CHECK_COUNT; i++) {
		if ((cs_space_read_4 (dev, space, CS_HFE2_STAT_REG) & 1) != 0) {
        		VERBOSE (CL_VERBOSE_LIBRARY, "Waiting for core to stop for %d interval(s)", i);
			usleep(CS_HFE2_CORE_STOP_CHECK_INTERVAL);
		} else {
        		VERBOSE (CL_VERBOSE_LIBRARY, "Core stopped");
			return;
		}
	}
        VERBOSE (CL_VERBOSE_LIBRARY, "Waiting time exceeded limit, continuing...");
}
