/*!
 * \file flwatch.c
 * \brief Functions for controlling FLWATCH component.
 * \author Peter Bartos <teepee@liberouter.org>
 * \author Peter Stranak <stranak@liberouter.org>
 * \date 2010
 */

/*
 * Copyright (C) 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <errno.h>
#include <commlbr.h>
#include "flwatch.h"

/*! Control register address */
#define FLWATCH_CTRL_REG	0x0
/*! Start address of counters */
#define FLWATCH_CNT_REG		0x8

/*! FLWATCH enable bitmask */
#define FLWATCH_BIT_EN		0x00000001
/*! FLWATCH reset bitmask */
#define FLWATCH_BIT_RESET	0x00000002
/*! FLWATCH sampling enable bitmask */
#define FLWATCH_BIT_SAMPLE_EN	0x00000004
/*! FLWATCH sample bitmask */
#define FLWATCH_BIT_SAMPLE	0x00000008
/*! FLWATCH sampling available bitmask */
#define FLWATCH_BIT_CAN_SAMPLE	0x00000010

/* deprecated code, left just for compatibility reasons */

/*!
 * \brief Reads a bit from FLWATCH component register
 * 
 * \param dev	Device
 * \param space	Space
 *
 * \returns 1 unread bit
 */
static int get_a_bit(cs_device_t *dev, cs_space_t *space)
{
	static uint32_t bit_stack = 0;	/* stack of bites */
	static int bit_stack_count = 0;	/* number of bits loaded to stack */
	static int where_read = FLWATCH_CNT_REG;	/* reading register iterator */
	int bit;

	if (bit_stack_count == 0) {	/* if no more bits in stack */
		bit_stack = cs_space_read_4(dev, space, where_read);	/* read another 32 bits */
		bit_stack_count = 32;
		where_read += 4;	/* next read  */
	}

	bit = (bit_stack >> 31) & 1;	/* the first bit */
	bit_stack <<= 1;	/* shift right */
	bit_stack_count--;	/* one bit used */

	return bit;
}

/*!
 * \brief Reads data from FLWATCH component to structure
 *
 * \param dev			Device
 * \param space			Space
 * \param num_interfaces	Number of interfaces to watch 
 * \param cntr_width		CNTR_WIDTH register value
 * \param fl_component		Structure with FLWATCH component data (#cs_flwatch_component_t)
 *
 * \deprecated This function is deprecated and no longer maintained. Please use
 * #cs_flwatch_read_data instead.
 */
void cs_flwatch_read(cs_device_t *dev, cs_space_t *space, int num_interfaces,
	int cntr_width, cs_flwatch_component_t *fl_component)
{
	int i, j;

	/* Allocating space for data */
	for (i = 0; i < CS_FLWATCH_MAX_LEN; i++) {
		fl_component->frames[i] =
		    (uint32_t*)calloc(num_interfaces, sizeof(uint32_t));
		fl_component->invalid_frames[i] =
		    (uint32_t*)calloc(num_interfaces, sizeof(uint32_t));
	}
	fl_component->dst_rdy = (int*)calloc(num_interfaces, sizeof(int));
	fl_component->src_rdy = (int*)calloc(num_interfaces, sizeof(int));

	/* Processing data bit by bit */
	for (i = 0; i < num_interfaces; i++)	/* Frame Counters */
		for (j = 0; j < cntr_width; j++) {	/* Counter Width */
			fl_component->frames[j / 32][i] <<= 1;	/* shift right */
			fl_component->frames[j / 32][i] |= get_a_bit(dev, space);	/* paste another bit */
		}

	for (i = 0; i < num_interfaces; i++)	/* Invalid Frame Counters */
		for (j = 0; j < cntr_width; j++) {	/* Counter Width */
			fl_component->invalid_frames[j / 32][i] <<= 1;	/* shift right */
			fl_component->invalid_frames[j / 32][i] |= get_a_bit(dev, space);	/* paste another bit */
		}

	for (i = 0; i < 32 - (num_interfaces * 2) % 32; i++)	/* have to flush bites due to align */
		get_a_bit(dev, space);

	for (i = 0; i < num_interfaces; i++) {
		fl_component->src_rdy[num_interfaces - i - 1] = get_a_bit(dev, space);	/* SRC_RDY */
		fl_component->dst_rdy[num_interfaces - i - 1] = get_a_bit(dev, space);	/* DST_RDY */
	}

	fl_component->num_interfaces = num_interfaces;
	fl_component->cntr_width = cntr_width;
}

/*!
 * \brief Frees structure of FLWATCH component
 *
 * \param fl_component	Structure with FLWATCH component data
 *
 * \deprecated This function is deprecated and no longer maintained. Please use
 * #cs_flwatch_free_data instead.
 */
void cs_flwatch_free(const cs_flwatch_component_t *fl_component)
{
	int i;
	for (i = 0; i < CS_FLWATCH_MAX_LEN; i++) {
		free(fl_component->frames[i]);
		free(fl_component->invalid_frames[i]);
	}
	free(fl_component->dst_rdy);
	free(fl_component->src_rdy);
}

/* end of deprecated code */


/*!
 * \brief Return current value of the control register
 *
 * \param dev	Device
 * \param space	Space
 *
 * \returns FLWATCH control register
 */
u_int32_t cs_flwatch_read_control(cs_device_t *dev, cs_space_t *space)
{
	return cs_space_read_4(dev, space, FLWATCH_CTRL_REG);
}

/*!
 * \brief Enable/disable FLWATCH
 *
 * \param dev		Device
 * \param space		Space
 * \param enable	Flag (1 - enable, 0 - disable)
 */
void cs_flwatch_enable(cs_device_t *dev, cs_space_t *space, int enable)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLWATCH_CTRL_REG);
	if (enable) {
		control |= FLWATCH_BIT_EN;
	} else {
		control &= ~FLWATCH_BIT_EN;
	}
	cs_space_write_4(dev, space, FLWATCH_CTRL_REG, control);
}

/*!
 * \brief Reset FLWATCH counters
 *
 * \param dev		Device
 * \param space		Space
 */
void cs_flwatch_reset(cs_device_t *dev, cs_space_t *space)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLWATCH_CTRL_REG);
	control |= FLWATCH_BIT_RESET;
	cs_space_write_4(dev, space, FLWATCH_CTRL_REG, control);
}

/*!
 * \brief Enable/disable sampling
 *
 * \param dev		Device
 * \param space		Space
 * \param enable	Flag (1 - enable, 0 - disable)
 */
void cs_flwatch_sampling_enable(cs_device_t *dev, cs_space_t *space, int enable)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLWATCH_CTRL_REG);
	if (enable) {
		control |= FLWATCH_BIT_SAMPLE_EN;
	} else {
		control &= ~FLWATCH_BIT_SAMPLE_EN;
	}
	cs_space_write_4(dev, space, FLWATCH_CTRL_REG, control);
}

/*!
 * \brief Sample current values of counters
 *
 * \param dev		Device
 * \param space		Space
 */
void cs_flwatch_sample(cs_device_t *dev, cs_space_t *space)
{
	u_int32_t control;

	control = cs_space_read_4(dev, space, FLWATCH_CTRL_REG);
	control |= FLWATCH_BIT_SAMPLE;
	cs_space_write_4(dev, space, FLWATCH_CTRL_REG, control);
}

/*!
 * \brief Check if FLWATCH is enabled
 *
 * \param control	FLWATCH control register
 *
 * \retval 1	Enabled
 * \retval 0	Disabled
 */
int cs_flwatch_is_enabled(u_int32_t control)
{
	return (control & FLWATCH_BIT_EN) ? 1 : 0;
}

/*!
 * \brief Check if sampling is enabled
 *
 * \param control	FLWATCH control register
 *
 * \retval 1	Enabled
 * \retval 0	Disabled
 */
int cs_flwatch_is_sampling_enabled(u_int32_t control)
{
	return (control & FLWATCH_BIT_SAMPLE_EN) ? 1 : 0;
}

/*!
 * \brief Check if sampling is available
 *
 * \param control	FLWATCH control register
 *
 * \retval 1	Available
 * \retval 0	Not available
 */
int cs_flwatch_is_sampling_available(u_int32_t control)
{
	return (control & FLWATCH_BIT_CAN_SAMPLE) ? 1 : 0;
}

/*!
 * \brief Fill a structure with data read from FLWATCH

 * \param dev			Device
 * \param space			Space
 * \param num_interfaces	Number of interfaces to watch 
 * \param cntr_width		Bit width of used counters
 * \param fl_comp		Structure with FLWATCH component data (#cs_flwatch_t)
 *
 * \retval 0		OK
 * \retval negative	Error
 */
int cs_flwatch_read_data(cs_device_t *dev, cs_space_t *space,
	int num_interfaces, int cntr_width, cs_flwatch_t *fl_comp)
{
	cs_addr_t offset;
	u_int32_t width;
	u_int32_t word = 0;
	int i, j;

	width = (cntr_width / 32) + ((cntr_width % 32) ? 1 : 0);

	fl_comp->interfaces = num_interfaces;
	fl_comp->cntr_width = cntr_width;

	fl_comp->flwatch_ifc = malloc(num_interfaces * sizeof(struct flwatch_ifc));
	if (fl_comp->flwatch_ifc == NULL) {
		VERBOSE(CL_VERBOSE_LIBRARY, "failed to allocate memory");
		return -ENOMEM;
	}

	for (i = 0; i < num_interfaces; i++) {
		fl_comp->flwatch_ifc[i].frames = NULL;
		fl_comp->flwatch_ifc[i].invalid_frames = NULL;
	}

	/* control register */
	offset = FLWATCH_CTRL_REG;
	fl_comp->control_reg = cs_space_read_4(dev, space, offset);

	/* valid packets */
	offset = FLWATCH_CNT_REG;
	for (i = 0; i < num_interfaces; i++) {
		fl_comp->flwatch_ifc[i].frames = calloc(width, sizeof(u_int32_t));
		if (fl_comp->flwatch_ifc[i].frames == NULL) {
			VERBOSE(CL_VERBOSE_LIBRARY, "failed to allocate memory");
			return -ENOMEM;
		}
		cs_space_read_multi_4(dev, space, offset, width, fl_comp->flwatch_ifc[i].frames);
		offset += width * 4;
	}

	/* invalid packets */
	for (i = 0; i < num_interfaces; i++) {
		fl_comp->flwatch_ifc[i].invalid_frames = calloc(width, sizeof(u_int32_t));
		if (fl_comp->flwatch_ifc[i].invalid_frames == NULL) {
			VERBOSE(CL_VERBOSE_LIBRARY, "failed to allocate memory");
			return -ENOMEM;
		}
		cs_space_read_multi_4(dev, space, offset, width, fl_comp->flwatch_ifc[i].invalid_frames);
		offset += width * 4;
	}

	/* ready signals */
	for (i = 0, j = 32; i < num_interfaces; i++) {
		if (j == 32) {
			word = cs_space_read_4(dev, space, offset);
			offset += 4;
			j = 0;
		}
		fl_comp->flwatch_ifc[i].dst = (word & (1 << j++)) ? 0 : 1;
		fl_comp->flwatch_ifc[i].src = (word & (1 << j++)) ? 0 : 1;
	}

	return 0;
}

/*!
 * \brief Release allocated resources
 *
 * \param fl_comp	Structure with FLWATCH component data (#cs_flwatch_t)
 */
void cs_flwatch_free_data(cs_flwatch_t *fl_comp)
{
	int i;

	if (fl_comp->flwatch_ifc != NULL) {
		for (i = 0; i < fl_comp->interfaces; i++) {
			free(fl_comp->flwatch_ifc[i].frames);
			free(fl_comp->flwatch_ifc[i].invalid_frames);
		}
		free(fl_comp->flwatch_ifc);
	}
}
