/*!
 * \file cam.c
 * \brief Functions for loading cam
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2006
 */
/* Copyright (C) 2006 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include <commlbr.h>

#include "cam.h"

/*! Ident identification */
__RCSID("$Id$");

/*!
 * \brief 	Fill data or mask register in cam from string
 * \param dev	Device
 * \param cam	Cam space
 * \param adr	Address to write (mask reg/data reg)
 * \param str	Value as string to write
 * \return	Number of written bytes
 */
int
cs_cam_fill_reg(cs_device_t * dev, cs_space_t * cam, u_int32_t adr, char *str)
{
	int iterations;
	int i;
	char hexa_buf[9];
	u_int32_t value;
	int length = strlen(str);

	/* convert and write string of 8 hex digits into 32b values, start form end */
	/* number of 32b values */
	if (length % 8 == 0)
		iterations = length / 8;
	else
		iterations = length / 8 + 1;

	for (i = 1; i < iterations; i++) {
		strncpy(hexa_buf, str + (length - i * 8), 9);
		str[length - i * 8] = '\0';
		sscanf(hexa_buf, "%x", &value);
		cs_space_write_4(dev, cam, adr, value);
		VERBOSE(2,"0x%08x -> 0x%08x", value, adr);
		adr += 4;
	}
	/* scan last may be not completed 32b */
	sscanf(str, "%x", &value);
	cs_space_write_4(dev, cam, adr, value);
	VERBOSE(2,"0x%08x -> 0x%08x", value, adr);

	return iterations * 4;	/* bytes written */
}

/*!
 * \brief                 Fill cam with rows
 *
 * \param dev             Device
 * \param cam             Cam space
 * \param rows            Rows to write
 * \param rows_count      Number of rows to write
 *
 * \return                
 * 	- 0 OK
 * 	- else failed
 */
int cs_cam_fill(cs_device_t * dev, cs_space_t * cam, cs_cam_row_t * rows, int rows_count)
{
	int value = 0;
	int loops = 0;
	int i;

	VERBOSE(2, "Sending STOP REQUEST.");

	value = cs_space_read_4(dev, cam, CS_CAM_CMD_REG);
	value = value | 1;
	cs_space_write_4(dev, cam, CS_CAM_CMD_REG, value);	/* write STOP REQUEST */
	VERBOSE(2,"0x%08x -> 0x%08x", value, CS_CAM_CMD_REG);
	VERBOSE(2,"Waiting for CAM to ACK.");

	/* waiting for CAM to ACK */
	while ((cs_space_read_4(dev, cam, CS_CAM_STAT_REG) & (u_int32_t) 1) != 1) {
		usleep(100);
		loops++;
		if (loops > 1000) {
			warnx("There can be problem with cam\n" "- waiting too long\n");
			loops = 0;
		}
	}

	VERBOSE(2,"Setting CAM BUSY.");

	value = cs_space_read_4(dev, cam, CS_CAM_CMD_REG);
	value = value | 2; /* write 1 on 1st bit in CS_CAM_CMD_REG */
	cs_space_write_4(dev, cam, CS_CAM_CMD_REG, value);	/* write BUSY - reserve component */
	VERBOSE(2,"0x%08x -> 0x%08x", value, CS_CAM_CMD_REG);

	VERBOSE (2, "Loading data and masks into CAM:");
	for(i = 0; i < rows_count; i++) {
		cs_cam_write_row(dev, cam, rows[i], i);
	}

	VERBOSE(2,"Clearing command register.");

	value = 0;
	cs_space_write_4(dev, cam, CS_CAM_CMD_REG, value);	/* clear CMD REG */
	VERBOSE(2,"0x%08x -> 0x%08x", value, CS_CAM_CMD_REG);

	return 0;
}

/*! 
 * \brief Convert two strings into cam_row structure
 * 
 * \param data Data
 * \param mask Mask
 * 
 * \return Converted structure
 */
cs_cam_row_t cs_cam_get_row_from_strings(char * data, char * mask) {
	cs_cam_row_t result;

	result.data = data;
	result.mask = mask;

	return result;
}

/*! 
 * \brief Read rows from file and write into array of cs_cam_row_t structure
 * 
 * \param src_file_name Source file in format "line_number data mask", should
 * work generic
 * \param rows Pointer to array of cam_row structures, necessary memory will be
 * allocated, must be freed
 * 
 * \return Number of read lines
 */
int cs_cam_read_rows_from_file (char * src_file_name, cs_cam_row_t ** rows) {

	int result_size = 0;
	FILE * src_file;
	char buffer[101];
	char data_str[30], mask_str[30];
	int count = 0;
	struct stat fstat;

	/* Check if regular file */
	if(stat(src_file_name, &fstat))
		errx(1, "Error while getting information about source file %s.", src_file_name);

	if(!(S_ISREG(fstat.st_mode)))
		errx(1, "Error opening source file %s. Please select regular file, name will be probably cam_data.txt.", src_file_name);

	/* Load file. */
	if ((src_file = fopen(src_file_name, "r")) == NULL)
		errx(1, "Error opening source file %s.", src_file_name);

	while (fgets(buffer, 100, src_file) != NULL) {
		if (sscanf(buffer, "%*d%s%s", data_str, mask_str) != EOF) {
			count++;
			/* allocate memory for returning array */	
			if(result_size == 0) {
				/* initial size set to 16 */
				if(!(*rows = (cs_cam_row_t*)malloc(16 * sizeof(cs_cam_row_t))))
						errx(1, "Memory allocation failed");
				result_size = 16;
			} else if(result_size < count) {
				/* double size */
				if(!(*rows = (cs_cam_row_t*)realloc(*rows, 2 * result_size * sizeof(cs_cam_row_t))))
						errx(1, "Memory allocation failed");
				result_size *= 2;
			}

			/* allocate memory for strings */
			if(!(((*rows)[count-1]).data = (char*)malloc((strlen(data_str)+1)* sizeof(char))))
						errx(1, "Memory allocation failed");
			if(!(((*rows)[count-1]).mask = (char*)malloc((strlen(mask_str)+1) * sizeof(char))))
						errx(1, "Memory allocation failed");

			/* copy strings */
			strcpy(((*rows)[count-1]).data, data_str);
			strcpy(((*rows)[count-1]).mask, mask_str);

			MSG(2, "Cam row converted: %d : %s : %s\n", count, ((*rows)[count-1]).data, ((*rows)[count-1]).mask);
		}
	}

	fclose(src_file);

	return count;
}

/*! 
 * \brief Write one row from cs_cam_row_t structure
 * 
 * \param dev Device
 * \param cam Space
 * \param row Row to write
 * \param cam_count Number of rows written so far to determine address
 */
void cs_cam_write_row(cs_device_t * dev, cs_space_t * cam, cs_cam_row_t row, int cam_count) {

	u_int32_t value;

	VERBOSE(1, "Writing row %d", cam_count);
	/* fill data reg with 32b values */
	VERBOSE(1, "data");
	cs_cam_fill_reg(dev, cam, CS_CAM_DATA_REG, row.data);
	/* fill mask reg with 32b values */
	VERBOSE(1, "mask");
	cs_cam_fill_reg(dev, cam, CS_CAM_MASK_REG, row.mask);

	VERBOSE(1, "finishing row");
	cs_space_write_4(dev, cam, CS_CAM_ADR_REG, cam_count);	/* write address - number of items written so far */
	VERBOSE(2,"0x%08x -> 0x%08x", cam_count, CS_CAM_ADR_REG);
	value = cs_space_read_4(dev, cam, CS_CAM_WRT_CMD);
	value = value | 1;
	cs_space_write_4(dev, cam, CS_CAM_WRT_CMD, value);	/* and write one cam record */
	VERBOSE(2,"0x%08x -> 0x%08x", value, CS_CAM_WRT_CMD);
}

/*! 
 * \brief Free allocated memory from rows
 * 
 * \param rows Array to free
 * \param count Number of allocated rows
 */
void cs_cam_free_rows(cs_cam_row_t *rows, int count) {
	int i;
	for(i = 0; i < count; i++) {
		/* free data and mask */
		free(rows[i].data);
		free(rows[i].mask);
	}
	/* free whole array */
	free(rows);
}

/*! 
 * \brief Fill cam memory from file
 * 
 * \param dev Device
 * \param cam Space
 * \param src_file_name Source file
 * 
 * \return Number of written rows
 */
int cs_cam_fill_from_file(cs_device_t * dev, cs_space_t * cam, char * src_file_name) {
	int cam_count;
	cs_cam_row_t *rows;
	
	/* read rows */
	cam_count = cs_cam_read_rows_from_file(src_file_name, &rows);
	/* fill cam */
	cs_cam_fill(dev, cam, rows, cam_count);
	/* free rows */
	cs_cam_free_rows(rows, cam_count);

	return cam_count;
}
