/*!
 * \file id.c
 * \brief Hardware identification routine
 * \author Jaroslav Kysela <perex@perex.cz>
 * \author Jachym Holecek <freza@liberouter.org>
 * \date 2003,2004
 *
 * Copyright (C) 2003-2004  CESNET
 */
 /* LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <commlbr.h>
#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

#define ID_BUFFER_SIZE	90

/** Array of all supported types of FPGA chips */ 
const char *cs_available_chips[] = {
	CHIP_XCV3000,
	CHIP_XCVP50,
	CHIP_LX110T,
	CHIP_LX155T,
	CHIP_FX100T,
	CHIP_TX240T,
	CHIP_LX240T,
	CHIP_HX365T,
	CHIP_VX690T,
	NULL
};

/**
 * \brief Return the main chip identification
 * \param d Combo6 device
 * \retval Static ASCII string
 *
 * The returned string looks like:
 *  'xcv3000'
 */
const char *
cs_hw_mchip(cs_device_t *d)
{
	if (!strcmp(d->dv_board, "combo6")) {
		return CHIP_XCV3000;
	} else if (!strcmp(d->dv_board, "ptm")) {
		return "xs200";
	} else if (!strcmp(d->dv_board, "combo6x") ||
		!strcmp(d->dv_board, "combo6e")) {
			return CHIP_XCVP50;
	/* combo v2 family cards - new identification concept */
	} else if (!strcmp(d->dv_board, "combo")) {
		if (!strcmp(d->dv_board_subtype, "LXT110"))
			return CHIP_LX110T;
		else if (!strcmp(d->dv_board_subtype, "LXT155"))
			return CHIP_LX155T;
		else if (!strcmp(d->dv_board_subtype, "FXT100"))
			return CHIP_FX100T;
		else if (!strcmp(d->dv_board_subtype, "LXT240"))
			return CHIP_LX240T;
		else if (!strcmp(d->dv_board_subtype, "HXT365"))
			return CHIP_HX365T;
		else
			return "unknown";
		/* FB Cards */
	} else if (!strncmp(d->dv_board, "FB", 2)) {
		if (!strcmp(d->dv_board, "FB8XG"))
			return CHIP_VX690T;
		else if (!strcmp(d->dv_board, "FB1CG"))
			return CHIP_VH580T;
		else
			return "unknown";
		/* NetFPGA 10G4 card */
	} else if (!strcmp(d->dv_board, "NetFPGA 10G4")) {
		if (!strcmp(d->dv_board_subtype, "TXT240"))
			return CHIP_TX240T;
		else
			return "unknown";
	} else {
		return "unknown";
	}
}

/**
 * \brief Return the hardware identification
 * \param d Combo6 device
 * \retval Allocated string or NULL (means error)
 *
 * The returned string looks like:
 *  'main="1.0",xc5vlx110t,combo;addon0="1.0",n/a,10G1;addon1="1.0",n/a,10G1'
 *  (old output looked like: 'main="1.0",xcv3000,combo6;addon="1.0",xcv1000,mtx')
 */
char *
cs_hw_id(cs_device_t *d)
{
	char *res;

	res = (char *)malloc(ID_BUFFER_SIZE);
	if (res == NULL)
		return NULL;

	/*
	 * old (C6 and C6X) output was:
	 * snprintf(res, 80, "main=\"1.0\",%s,%s;addon=\"1.0\",%s,%s", 
	 * 	cs_hw_mchip(d), d->dv_board, d->dv_if0_chip, d->dv_if0_card);
	 */
	/* just modified for cv2 card family identification concept */
	snprintf(res, ID_BUFFER_SIZE, "main=\"1.0\",%s,%s;addon0=\"1.0\",%s,%s;addon1=\"1.0\",%s,%s",
		cs_hw_mchip(d), d->dv_board, d->dv_if0_chip, d->dv_if0_card, 
		d->dv_if1_chip, d->dv_if1_card);
	res[ID_BUFFER_SIZE - 1] = '\0';
	return res;
}
