/*
 * szedataproc.h: Interface to the SZEData core.
 * Copyright (C) 2004-2006 CESNET
 * Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __SZEDATAPROC_H
#define __SZEDATAPROC_H

struct szedata_device;

#include <linux/mutex.h>
#include <linux/poll.h>

/* buffer for information */
struct szedata_info_buffer {
	char *buffer;		/* pointer to begin of buffer */
	char *curr;		/* current position in buffer */
	unsigned long size;	/* current size */
	unsigned long len;	/* total length of buffer */
	int stop;		/* stop flag */
	int error;		/* error code */
};

#define SZEDATA_INFO_CONTENT_TEXT		0

struct szedata_info_entry;

struct szedata_info_entry_text {
	unsigned long read_size;
	void (*read)(struct szedata_info_entry *entry,
			struct szedata_info_buffer *buffer);
};

struct szedata_info_entry {
	const char *name;
	mode_t mode;
	long size;
	unsigned short content;
	union {
		struct szedata_info_entry_text text;
	} c;
	struct szedata_info_entry *parent;
	struct szedata_device *device;
	struct module *module;
	void *private_data;
	void (*private_free)(struct szedata_info_entry *entry);
	struct proc_dir_entry *p;
	struct mutex access;
};

#ifdef CONFIG_PROC_FS

int szedata_iprintf(struct szedata_info_buffer *buffer, char *fmt, ...)
		__attribute__ ((format (printf, 2, 3)));
int szedata_info_init(void);
int szedata_info_done(void);

struct szedata_info_entry *szedata_info_create_module_entry(
		struct module *module, const char *name,
		struct szedata_info_entry *parent);
struct szedata_info_entry *szedata_info_create_device_entry(
		struct szedata_device *device, const char *name,
		struct szedata_info_entry *parent);
void szedata_info_free_entry(struct szedata_info_entry *entry);
int szedata_info_store_text(struct szedata_info_entry *entry);
int szedata_info_restore_text(struct szedata_info_entry *entry);

int szedata_info_device_create(struct szedata_device *device);
int szedata_info_device_free(struct szedata_device *device);
int szedata_info_register(struct szedata_info_entry *entry);
int szedata_info_unregister(struct szedata_info_entry *entry);

/* for device drivers */
int szedata_device_proc_new(struct szedata_device *device, const char *name,
		struct szedata_info_entry **entryp);

static inline void szedata_info_set_text_ops(struct szedata_info_entry *entry,
		void *private_data, long read_size,
		void (*read)(struct szedata_info_entry *,
			struct szedata_info_buffer *))
{
	entry->private_data = private_data;
	entry->c.text.read_size = read_size;
	entry->c.text.read = read;
}


#else

#define szedata_seq_root NULL
#define szedata_oss_root NULL

static inline int szedata_iprintf(struct szedata_info_buffer *buffer,
		char *fmt, ...) { return 0; }
static inline int szedata_info_init(void) { return 0; }
static inline int szedata_info_done(void) { return 0; }

static inline struct szedata_info_entry *szedata_info_create_module_entry(
		struct module *module, const char *name,
		struct szedata_info_entry *parent) { return NULL; }
static inline struct szedata_info_entry *szedata_info_create_device_entry(
		struct szedata_device *device, const char *name,
		struct szedata_info_entry *parent) { return NULL; }
static inline void szedata_info_free_entry(struct szedata_info_entry *entry) { }

static inline int szedata_info_device_create(struct szedata_device *device)
{ return 0; }
static inline int szedata_info_device_free(struct szedata_device *device)
{ return 0; }
static inline int szedata_info_register(struct szedata_info_entry *entry)
{ return 0; }
static inline int szedata_info_unregister(struct szedata_info_entry *entry)
{ return 0; }

static inline int szedata_device_proc_new(struct szedata_device *device,
		const char *name, struct szedata_info_entry **entryp)
{ return 0; }
static inline void szedata_info_set_text_ops(struct szedata_info_entry *entry,
		void *private_data, long read_size,
		void (*read)(struct szedata_info_entry *,
			struct szedata_info_buffer *)) { }

#endif

#endif /* __SZEDATAPROC_H */
