/*
 * Copyright (c) 2007-2008 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#ifndef SZEDATA2K_H_FILE
#define SZEDATA2K_H_FILE

#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/types.h>

#include "szedata2.h"

/* States (HW specific) */
#define SZE2_REGISTERED     1
#define SZE2_MAPINSTANTLY   2
#define SZE2_NOADJUSTTIMER  4
#define SZE2_READPTRONLOCK  8

/**
 * struct szedata2_block - ring buffer page descriptor
 *
 * @virt: virtual address of the page
 * @phys: physical (DMA-ble) address of the page
 */
struct szedata2_block {
	void *virt;
	dma_addr_t phys;
};

/**
 * struct szedata2_area - area descriptor
 *
 * @roffset: offset in ring space
 * @asize: block size * blks (pointers modulo)
 * @timeout: current timeout (for adaptive timeout)
 * @poll_thresh: after how much data wake up applications
 * @start_count: how many times it was started
 */
struct szedata2_area {
	union {
		struct {
			unsigned long head;
			unsigned long tail;
		} rx;
		struct {
			unsigned long head;
			unsigned long tail;
		} tx;
	} u;

	unsigned long roffset;
	size_t asize;

	unsigned long to_reset;
	unsigned long timeout;
	unsigned int irqs;

	unsigned int poll_thresh;
	unsigned int start_count;
};

/**
 * struct szedata2_ring - szedata per ring buffer info
 *
 * @area: descriptor for each area (heads, tails, ...)
 * @areas: areas per space (size of @adesc array)
 * @blk_count: overall count of blocks in this space
 * @blk_size: block size
 * @ring: pointers to ring blocks
 */
struct szedata2_ring {
	struct szedata2_area *area;
	unsigned int areas;
	unsigned int blk_count;
	size_t blk_size;
	struct szedata2_block *ring;
};

/**
 * struct szedata2 - szedata2 information holder
 *
 * The structure holds all needed information to know about szedata2.
 * Including application list, HW-specific hooks, ring buffer
 * descriptors etc.
 *
 * @open, @close, @start and @stop are guaranteed to be called serialized
 * (under a lock).
 *
 * @cdev: /dev/szedataII node information
 * @app_list: list of application being on @cdev
 * @app_list_lock: lock for @app_list
 * @missed_irq: timer fired when no HW interrupts come
 * @ptr_lock: protects pointers in both HW (real) and SW (cached)
 * @tx_locked: which areas are in use for TX dir
 * @to_min: minimum timeout
 * @to_max: maximum timeout
 * @state: status flags, e.g. SZE2_REGISTERED
 * @open_count: how many openers are there
 * @ring: ring buffer descriptors
 * @dev: device itself (for chardev and attributes)
 * @parent: real parent device, e.g. a pci device
 * @owner: who is HW part
 * @eth_ifaces: count of eth interfaces, RX and TX
 * @ndev: network devices associated to areas
 * @open: called when first opener comes
 * @close: called when last opener dissapears
 * @start: called once when some opener calls START ioctl
 * @stop: called once when last opener calls STOP ioctl (or closes @cdev)
 * @get_ptr: to obtain a HW pointer value
 * @set_ptr: to set a HW pointer value
 * @set_intr: to set interrupt values
 * @destroy: called when szedata2 is about to dismiss
 * @private: contains pointer to data of private_size (szedata2_alloc param)
 */
struct szedata2 {
	struct cdev cdev;

	struct list_head app_list;
	rwlock_t app_list_lock;

	wait_queue_head_t poll_wait;
	struct timer_list missed_irq;
	spinlock_t ptr_lock;
	DECLARE_BITMAP(tx_locked, 32);

	unsigned long to_min, to_max, to_step;

	unsigned long state;
	unsigned int open_count;
	int numa_node; 	/* Inherited from PCI device or -1 if unsupported. */

	struct szedata2_ring ring[SZE2_MMIO_MAX];

	struct device *dev;
	struct device *parent;
	struct module *owner;

	unsigned int eth_ifaces[2];
	struct net_device **ndev;

	int (*open)(struct szedata2 *);
	void (*close)(struct szedata2 *);
	int (*start)(struct szedata2 *, unsigned int space, unsigned int area);
	void (*stop)(struct szedata2 *, unsigned int space, unsigned int area);
	unsigned long (*get_ptr)(struct szedata2 *, unsigned int space,
			unsigned int area, unsigned int which);
	void (*set_ptr)(struct szedata2 *, unsigned int space,
			unsigned int area, unsigned long ptr);
	void (*set_intr)(struct szedata2 *, unsigned int space,
			unsigned int area, unsigned long value);
	void (*set_timeout)(struct szedata2 *, unsigned int space,
			unsigned int area, unsigned long value);
	void (*destroy)(struct szedata2 *);

	void *private;
};

extern struct szedata2 *szedata2_alloc(unsigned int private_size,
		struct device *parent);
extern int szedata2_register(struct szedata2 *sd);
extern void szedata2_destroy(struct szedata2 *sd);

extern int szedata2_set_timeout(struct szedata2 *sd, unsigned long min,
		unsigned long max);

extern struct szedata2_block *szedata2_alloc_dma(struct device *dev,
		unsigned int count, size_t size);
extern void szedata2_free_dma(struct device *dev, struct szedata2_block *blks,
		unsigned int count, size_t size);
extern int szedata2_alloc_dmaspace(struct szedata2 *sd, unsigned int space,
		unsigned int areas, unsigned int *count, size_t size,
		unsigned int poll_divisor);
extern void szedata2_free_dmaspace(struct szedata2 *sd, unsigned int space);

extern size_t szedata2_get_addr(struct szedata2 *sd, unsigned int space,
		unsigned int area, void **virt, dma_addr_t *phys,
		unsigned long pos, size_t len);

extern void szedata2_intr(struct szedata2 *sd, unsigned long status,
		unsigned long rx, unsigned long tx);

extern int szedata2_hw_init(void);
extern void szedata2_hw_exit(void);
extern int szedata2_eth_add(struct szedata2 *sd, unsigned int interfaces[2],
		int (*get_mac)(struct szedata2 *sd, unsigned int i, u8 *mac));
extern void szedata2_eth_remove(struct szedata2 *sd);

static inline size_t szedata2_get_virt(struct szedata2 *sd, unsigned int space,
		unsigned int area, void **virt, unsigned long pos, size_t len)
{
	dma_addr_t phys;

	return szedata2_get_addr(sd, space, area, virt, &phys, pos, len);
}

static inline size_t szedata2_get_phys(struct szedata2 *sd, unsigned int space,
		unsigned int area, dma_addr_t *phys, unsigned long pos,
		size_t len)
{
	void *virt;

	return szedata2_get_addr(sd, space, area, &virt, phys, pos, len);
}

static inline __attribute__((const)) size_t
szedata2_tail_head_size(unsigned long head, unsigned long tail, size_t size)
{
	size_t ret;

	ret = size + head - tail;
	ret &= size - 1;

	return ret;
}

static inline __attribute__((const)) size_t
szedata2_head_tail_size(unsigned long head, unsigned long tail, size_t size)
{
	size_t ret;

	if (head == tail)
		return size - 1;

	ret = size + tail - head;
	ret &= size - 1;

	/* disallow meeting of head and tail */
	if (ret)
		ret--;

	return ret;
}

#endif
