/*
 *  szedata_kernel.h: application <-> szedata kernel interface
 *  Copyright (c) 2003-2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */
/*
 *  Detailed implementation description:
 *
 *   The szedata lib will open the /dev/szedata/# device and get the new
 *   application instance (integer). Then it enumerates all available
 *   interfaces (physical network ports on the hardware) and subcribe selected
 *   ones. The ring buffer size for data block contents and block descriptors
 *   is obtained via ioctl. Using these values, the block descriptors and
 *   contents of blocks will be mmaped read-only to the user space (all
 *   applications will see exactly same data).
 *
 *   Data flow control (application side): The application will start / stop
 *   capturing of data using an dedicated ioctl. Note that other applications
 *   might be running after stop, thus blocks can be still overwritten in the
 *   ring buffer even if the application stops the capturing. To determine the
 *   first block in fifo or get a next block from the fifo list mantained for
 *   given application, a lock-next ioctl is used. At the end of communication,
 *   the remaining lock should be unlocked with an unlock ioctl (note that the
 *   lock-next ioctl will unlock the current block automatically, so there is
 *   no need to call always a lock-next -> unlock sequence; the standard use
 *   should be: open, subscribe, lock-next, lock-next, .... unlock, close).
 *   The access to unlocked areas is permitted, but not very useful, because
 *   driver or hardware might overwrite data at any time.
 *
 *   Statistics: Statistics can be obtained via a mmaped area or using an ioctl.
 *   The statistics ioctl will also clear atomically all counters.
 *
 *  Note: The ring buffer is allocated when the driver is loaded. The ring
 *	buffer size will be specified via kernel module parameter or command
 *	line passed to kernel at boot. We can eventually use another external
 *	way to modifythe ring buffer size at runtime (use the proc filesystem
 *	etc.).
 */

#ifndef __SZEDATA_KERNEL_H
#define __SZEDATA_KERNEL_H

#include <linux/types.h>
#include <linux/ioctl.h>

#define SZEDATA_VERSION(major, minor)	((major<<16)|minor)
#define SZEDATA_CUR_VERSION	SZEDATA_VERSION(1, 2)
#define SZEDATA_MAX_IFNAME	128
#define SZEDATA_MAX_INTERFACES	32
#define SZEDATA_MAX_PKTLOCK	1024	/* max count of locked blocks at once */

/**
 * struct szedata_ioctl_get_interface - get_interface ioctl parameter
 *
 * @if_index: interface index starting with 0
 */
struct szedata_ioctl_get_interface {
	u_int32_t	if_index;
	char		if_name[SZEDATA_MAX_IFNAME];
};

/**
 * struct szedata_ioctl_subscribe_interface_old - SI_old ioctl parameter
 *
 * @if_index: array of requested interfaces, 0xff == terminator
 */
struct szedata_ioctl_subscribe_interface_old {
	u_int8_t	if_index[SZEDATA_MAX_INTERFACES];
};

/**
 * struct szedata_ioctl_subscribe_interface - subscribe_interface ioctl param
 *
 * @if_index: array of requested interfaces, 0xff == terminator
 */
struct szedata_ioctl_subscribe_interface {
	u_int8_t	if_index[SZEDATA_MAX_INTERFACES];
	u_int32_t	poll_threshold;
};

/**
 * struct szedata_ioctl_rbuffer - rbuffer ioctl parameter
 *
 * @blocks: count of allocated blocks
 * @block_size: maximum block size (usually 9016 bytes - jumbo)
 * @dma_size: size of mmaped area for block contents (SZEDATA_MMAP_OFFSET_PKT)
 */
struct szedata_ioctl_rbuffer {
	u_int32_t	blocks;
	u_int32_t	block_size;
	u_int32_t	dma_size;
};

/**
 * struct szedata_interface_stats - struct in interface_stats area
 *
 * @count: count of received blocks per interface
 * @bytes: count of received bytes per interface
 * @overwritten: overwritten blocks per interface
 * @dropped: count of dropped blocks by hardware per interface
 * @over_errors: count of frame overflow errors per interface
 * @length_errors: count of length errors per interface
 * @frame_errors: count of frame errors per interface
 * @crc_errors: count of CRC errors per interface
 * @fifo_errors: count of FIFO errors per interface
 */
struct szedata_interface_stats {
	u_int32_t	count;
	u_int64_t	bytes;
	u_int32_t	overwritten;
	u_int32_t	dropped;
	u_int32_t	over_errors;
	u_int32_t	length_errors;
	u_int32_t	frame_errors;
	u_int32_t	crc_errors;
	u_int32_t	fifo_errors;
};

struct szedata_ioctl_stats {
	u_int64_t	timestamp;
	struct szedata_interface_stats s[SZEDATA_MAX_INTERFACES];
};

/* FIXME */
#define SZEDATA_BLOCK_STATUS_OK		(1<<0)
#define SZEDATA_BLOCK_STATUS_CRCERR	(1<<1)
#define SZEDATA_BLOCK_STATUS_RXERR	(1<<2)

/**
 * struct szedata_mmap_block_descriptor - structs in block_descriptor area
 *
 * @if_index: interface index
 * @hh_size: size, hardware dependent header
 * @res: reserved for future use
 * @size: size, including 32-bit CRC
 * @status: block status - see SZEDATA_BLOCK_STATUS_XXXX bits
 * @block: relative offset to block contents in mmaped area
 * @timestamp: timestamp
 */
struct szedata_mmap_block_descriptor {
	u_int8_t	if_index;
	u_int8_t	hh_size;
	u_int8_t	res[2];
	u_int32_t	size;
	u_int32_t	status;
	u_int32_t	block;
	u_int64_t	timestamp;
};

/**
 * struct szedata_mmap_app_lock - structs in app_lock area
 *
 * @count: count of locks
 * @blk: relative offset to block descriptor in mmaped area
 */
struct szedata_mmap_app_lock {
	u_int32_t count;
	u_int32_t blk[SZEDATA_MAX_PKTLOCK];
};

struct szedata_mmap_block_istats {
	struct szedata_interface_stats s[SZEDATA_MAX_INTERFACES];
};

/**
 * struct szedata_mmap_block_status - structs in block_status area
 *
 * @avail: count of available (waiting) blocks
 */
struct szedata_mmap_block_status {
	u_int32_t	avail;
};

/* mmap offset for application lock (szedata_mmap_app_lock) */
#define SZEDATA_MMAP_OFFSET_LOCK	0x00000000
/* mmap offset for block statistics per interface (szedata_mmap_block_istats) */
#define SZEDATA_MMAP_OFFSET_ISTATS	0x00010000
/* mmap offset for block status (szedata_mmap_block_status) */
#define SZEDATA_MMAP_OFFSET_STATUS	0x00011000
/* mmap offset for block descriptors (szedata_mmap_block_descriptor) */
#define SZEDATA_MMAP_OFFSET_PKTDSC	0x00020000
/* mmap offset for block contents */
#define SZEDATA_MMAP_OFFSET_PKT		0x10000000

/*
 *  ioctl definition
 */
#define SZEDATA_IOCTL_VERSION		_IOR('X', 0x00, int)
#define SZEDATA_IOCTL_APP_NUMBER	_IOR('X', 0x01, int)
#define SZEDATA_IOCTL_INTERFACES	_IOR('X', 0x02, int)
#define SZEDATA_IOCTL_GET_INTERFACE	_IOWR('X', 0x02, \
		struct szedata_ioctl_get_interface)
#define SZEDATA_IOCTL_SUBSCRIBE_INTERFACEO _IOW('X', 0x03, \
		struct szedata_ioctl_subscribe_interface_old)
#define SZEDATA_IOCTL_SUBSCRIBE_INTERFACE _IOW('X', 0x03, \
		struct szedata_ioctl_subscribe_interface)
#define SZEDATA_IOCTL_UNSUBSCRIBE_INTERFACE _IOW('X', 0x04, \
		struct szedata_ioctl_subscribe_interface)
#define SZEDATA_IOCTL_RBUFFER		_IOR('X', 0x10, \
		struct szedata_ioctl_rbuffer)
#define SZEDATA_IOCTL_START		_IO('X', 0x20)
#define SZEDATA_IOCTL_STOP		_IO('X', 0x21)
/* lock next block */
#define SZEDATA_IOCTL_LOCK_NEXT		_IO('X', 0x30)
/* lock multiple blocks */
#define SZEDATA_IOCTL_LOCK_MULTI	_IOW('X', 0x31, u_int32_t)
/* unlock and free!! */
#define SZEDATA_IOCTL_UNLOCK		_IO('X', 0x32)
/* get statistics and clear counters */
#define SZEDATA_IOCTL_GET_STATS		_IOR('X', 0x40, \
		struct szedata_ioctl_stats)

#endif /* __SZEDATA_KERNEL_H */
