/**
 *  cv2eth.c: NIC driver module
 *
 *  Copyright (c) 2009 CESNET
 *  Copyright (c) 2009 Petr Kastovsky <kastovsky@liberouter.org>
 *
 * Licensed under GPLv2
 */

/**
 * DOC: Basic description
 * 
 * There should be a simple description of driver operations placed here.
 */

#include <linux/delay.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/jiffies.h>

#include "kocompat.h"
#include "combov2.h"
#include "cv2eth.h"

/* print function name at function entry and exit */
#define PRINT_FNC_ENTRY PDEBUG("%s entry\n", __FUNCTION__)
#define PRINT_FNC_EXIT  PDEBUG("%s exit\n", __FUNCTION__)

#define PDEBUG(fmt, ...)	pr_debug("cv2eth: " fmt, ##__VA_ARGS__)
#define PERR(fmt, ...)	printk(KERN_ERR "cv2eth: " fmt, ##__VA_ARGS__)

// rx descriptor initial flags
#define CV2ETH_RX_FLAGS			0
// descriptor size
#define CV2ETH_DESC_SIZE		16
// size of ethernet control checksum
#define FCS_SIZE				4
// size of hardware header
#define HW_HEADER_SIZE			0
// maximal packet size
#define CV2ETH_PKT_SIZE			(1522 + (FCS_SIZE) + (HW_HEADER_SIZE))
// number of ring blocks
#define CV2ETH_BLOCK_COUNT		16
// maximal number of interfaces (there is no more than 4 ifcs card yet)
#define CV2ETH_MAX_IFCS		4
// should be tunned in future
#define CV2ETH_TX_TIMEOUT	(1*HZ)

#define CV2ETH_NAPI_WEIGHT	32

/* template for network interface name as seen by ifconfig */
static char *netdev_template = "cv2eth";

MODULE_DESCRIPTION("NIC driver for ComboV2");
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Petr Kastovsky <kastovsky@liberouter.org>");

/*----# comboV2 module parameters #-------------------------------------------*/
/* Enable this card */
static int enable[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 };
/* Invalid memory region, fix to 64MB */
static int invalidmemsize[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 };
/* Check design boot */
static int bootprotect[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 };
/* Boot delay in ms before fpga access */
static int bootdelay[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 };
/* Use mode (see README) */
static int usemode[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 };

module_param_array(enable, int, NULL, 0);
MODULE_PARM_DESC(enable, "Enable ComboV2 card.");
module_param_array(invalidmemsize, int, NULL, 0);
MODULE_PARM_DESC(invalidmemsize, "Invalid PCI memory area size - fix to 64MB.");
module_param_array(bootprotect, int, NULL, 0);
MODULE_PARM_DESC(bootprotect, "Enable(default)=1/Disable=0 boot protection checks.");
module_param_array(bootdelay, int, NULL, 0);
MODULE_PARM_DESC(bootdelay, "Boot delay in ms before FPGA access.");
module_param_array(usemode, int, NULL, 0);
MODULE_PARM_DESC(usemode, "Use mode (for debugging only). See README.");

/*----# cv2eth module specific parameters ------------------------------------*/
static unsigned long block_count = CV2ETH_BLOCK_COUNT;

module_param(block_count, ulong, S_IRUGO);
MODULE_PARM_DESC(block_count, "block count [1]");

/*----# cv2eth module specific structures ------------------------------------*/

/**
 * struct cv2eth_ptrs - structure holding pointers to ring buffer
 * @head:		index of first valid item in ring buffer
 * @tail:		index of last valid item in ring buffer
 */
struct cv2eth_ptrs {
	u32 head;
	u32 tail;
};

/**
 * struct cv2eth_desc - descriptor structure
 * @length:		length of the data block
 * @flags:		flags belonging to transfered data
 * @addr:		physical address of the data block in memory
 *
 * Descriptor is basic data structure that is used to share informations between
 * driver and hardware. Each instance shall describe a packet (or fragment of
 * packet).
 */
struct cv2eth_desc {
	u32 length;
	u32 flags;
	u64 addr;
};

/**
 * struct cv2eth_desc - block pointer structure
 * @virt:		virtual address
 * @phys:		physical address
 *
 * Structure is used to store a pair of virtual and physical address of a single
 * memory location.
 */
struct cv2eth_block {
	void *virt;
	dma_addr_t phys;
};

/**
 * struct cv2eth_ring - ring buffer structure
 * @count:		number of blocks constituting the ring buffer
 * @blocks:		array of pointers to the constituting blocks
 *
 * Ring buffer is composed of particular blocks that are bounded using
 * pointers into a ring buffer. Each block is represented by a single
 * memory page. So the count element also presents the number of pages.
 */
struct cv2eth_ring {
	unsigned int count;
	struct cv2eth_block *blocks;
};

/**
 * struct cv2eth_skb - skb holding structure
 * @skb:		pointer to sk_buff structure instance
 * @phys:		physical address of DMA mapping for skb->data
 *
 * Structure is used to store an address of the instance of sk_buff structure
 * as well as address of DMA mapping of corresponding skb->data memory block.
 */
struct cv2eth_skb {
	struct sk_buff* skb;
	dma_addr_t phys;
};

/**
 * struct cv2eth_skb - module private structure
 * @combo6:		pointer to combo6 structure instance
 * @interfaces:	number of interfaces
 * @dev:		array of pointers to net_device structures
 * @rings:		array of pointers to ring buffers
 * @tx_up:		space for tx status upload
 * @tx_up_size:	size of tx status upload space
 * @open_lock:	lock to access hardware
 *
 * Private structure stores information accross specific interfaces. This
 * strucute is stored and accessible as combo6_device->private_data.
 */
struct cv2eth_private {
	struct combo6 *combo6;
	int interfaces;
	struct net_device *dev[CV2ETH_MAX_IFCS];
	struct cv2eth_ring *rings; // 0 = rx, 1 = tx
	struct cv2eth_block tx_up;
	size_t tx_up_size;
	struct mutex open_lock;
};

/**
 * struct cv2eth_netdev - net device (interface) private structure
 * @index:		interface index
 * @dev:		pointer to appropriate net_device structure
 * @stats:		interface statistics
 * @c6device:	pointer to combo6device structure, to access private data
 * @combo6:		pointer to combo6 structure, to access hardware
 * @rx:			head and tail ring buffer pointers
 * @tx:			head and tail ring buffer pointers
 * @rx_skbs:	array of socket buffers and their physical addresses
 * @tx_skbs:	array of socket buffers and their physical addresses
 * @napi:		instance of napi context
 * @max_descs:	maximal number of descriptors
 * @tx_pkts:	number of transmitted packets
 * @tx_sched_pkts:	number of packets scheduled to transmit
 * @rx_pkts:	number of received packets
 *
 * Private structure stores information accross specific interfaces.
 */
struct cv2eth_netdev {
	unsigned int index;
	struct net_device *dev;
	struct net_device_stats stats;
	struct combo6 *combo6;
	struct cv2eth_ptrs rx;
	struct cv2eth_ptrs tx;
	struct cv2eth_skb* rx_skbs;
	struct cv2eth_skb* tx_skbs;
#ifndef CONFIG_OLD_NAPI
	struct napi_struct napi;
#endif
	unsigned int max_descs;
	u32 tx_pkts;
	u32 tx_sched_pkts;
	u32 rx_pkts;
	unsigned long run_flag;
};

/*----# cv2eth helper functions ----------------------------------------------*/

/**
 * cv2eth_print_desc - print descriptor
 * @d:			descriptor to print
 *
 * DEBUG ONLY.
 */
static inline void cv2eth_print_desc(struct cv2eth_desc *d) {

	PDEBUG("descriptor %p addr %08Lu length %08u flags %08u\n", d, d->addr, d->length, d->flags);
}

/**
 * cv2eth_fill_desc - fill descriptor in little endian format
 * @d:			descriptor to fill
 * @addr:		address
 * @len:		length
 * @f:			flags
 *
 * Writes parameters into descriptor in little endian data format.
 */
static inline void cv2eth_fill_desc(struct cv2eth_desc *d, u64 addr, u32 len, u32 f) {
	d->flags = cpu_to_le32(f);
	d->length = cpu_to_le32(len);
	d->addr = cpu_to_le64(addr);
}

/**
 * get_desc_addr - get descriptor`s address
 * @priv:		module private structure
 * @ifc:		interface index
 * @dir:		direction (rx/tx)
 * @index:		descriptor index
 *
 * Returns address of descriptor from appropriate ring. Descriptor is uniquely
 * determined by interface index, direction and descriptor index. Interface
 * and direction determines ring buffer. Index is position in ring buffer.
 */
static inline struct cv2eth_desc* get_desc_addr(struct cv2eth_private* priv, unsigned int ifc,
	   									unsigned int dir, unsigned int index) {
	unsigned int pg_count = priv->rings[dir].count;
	unsigned int pg_offset = pg_count*ifc; 
	unsigned int pg_index = index / (PAGE_SIZE/CV2ETH_DESC_SIZE - 1);
	unsigned int pg = pg_offset + pg_index % pg_count;
	unsigned int idesc = index % (PAGE_SIZE/CV2ETH_DESC_SIZE - 1);
	struct cv2eth_desc* virt = (struct cv2eth_desc*)priv->rings[dir].blocks[pg].virt;
	return (virt + idesc);
}

/**
 * cv2eth_write_phys_addr - write physical address
 * @combo6:		combo6 structure, to access hw
 * @offset:		local address inside card(fpga)
 * @phys:		physical address
 *
 * Writes physical address to appropriate offset in card(fpga).
 */
static inline void cv2eth_write_phys_addr(struct combo6* combo6, u32 offset, dma_addr_t phys) {
	u32* val;
	u32 high;
	val = (u32*)&phys;
	combo6_fpga_writel(combo6, offset, *val);
	high = (sizeof(dma_addr_t) == 4) ? 0x0 : *(val + 1);
	combo6_fpga_writel(combo6, offset + 4, high);
}

/**
 * cv2eth_tx_cnt - get number of transmitted packets
 * @priv:		module private structure
 * @ifc:		interface index
 *
 * Get the number of transmitted packets by reading value of tx status
 * upload memory location. Read only value for specific interface.
 */
static inline u32 cv2eth_tx_cnt(struct cv2eth_private* priv, int ifc) {
	return le32_to_cpu(((u32*)priv->tx_up.virt)[ifc]);
}

/**
 * cv2eth_tx_cnt_clear - clear counter for specific interface
 * @priv:		module private structure
 * @ifc:		interface index
 */
static inline void cv2eth_tx_cnt_clear(struct cv2eth_private* priv, int ifc) {
	((u32*)priv->tx_up.virt)[ifc] = 0;
}

/*----# cv2eth device operations ---------------------------------------------*/
static irqreturn_t cv2eth_interrupt(struct combo6 *combo, unsigned int mask);
static int cv2eth_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces);
static int cv2eth_detach(struct combo6 *combo);
static int cv2eth_netdev_init(struct combo6 *combo, struct net_device **rdev,
		int index);
static void cv2eth_netdev_done(struct net_device *dev);
static int cv2eth_alloc_rxdescs(struct cv2eth_netdev *ndev, unsigned int pos, unsigned int count);

static const struct combo_device_id cv2eth_cids[] = {
	{
		.id_lsw = 0x41c10600, 
		.id_hsw = 0x41c106FF,
		.id_text = "",
	}, {
	}
};

static struct combo_driver cv2eth_ops = {
	.drv = {
		.owner  = THIS_MODULE,
		.name	= "cv2eth",
	},
	.dhw	= DHW_COMBOV2,
	.id_table = cv2eth_cids,
	.attach	= cv2eth_attach,
	.detach	= cv2eth_detach,
	.interrupt = cv2eth_interrupt,
};

/*----# cv2eth irq handling --------------------------------------------------*/
static int cv2eth_rx(struct cv2eth_netdev* ndev, int budget);
#ifdef CONFIG_OLD_NAPI
static int  cv2eth_rx_poll(struct net_device* dev, int *budget);
#else
static int  cv2eth_rx_poll_napi(struct napi_struct* napi, int budget);
#endif
static void cv2eth_tx_irq(struct combo6 *combo);

/**
 * cv2eth_rx_irq - handle rx interrupt request
 * @ndev:		interface specific structure to access napi context
 *
 * Only schedules napi polling method.
 */
static void cv2eth_rx_irq(struct cv2eth_netdev* ndev) {
	if (test_bit(PAC_DIR_RX, &ndev->run_flag))
#ifdef CONFIG_OLD_NAPI
		netif_rx_schedule(ndev->dev);
#else
		napi_schedule(&ndev->napi);
#endif
}

/**
 * cv2eth_interrupt - main interrupt handling routine
 * @device:		interrupting device
 * @mask:		legacy, always 0
 *
 * Reads interrupt status registers located on card. According to read values
 * calls appropriate interrupt service routine.
 */
static irqreturn_t cv2eth_interrupt(struct combo6 *combo, unsigned int mask)
{
	struct cv2eth_private *priv = combo->private_data;
	unsigned int i;
	u32 rx_irq_status = 0, tx_irq_status = 0;
	int status_zero = 0;

//	PRINT_FNC_ENTRY;
	rx_irq_status = combo6_fpga_readl(combo, PAC_HW_RX_IRQ);
	if (rx_irq_status != 0) {
		status_zero = 1;
//		dev_warn(&priv->combo6->pci->dev, "Rx interrupt caught (%X).\n", rx_irq_status);
		for (i = 0; i <  priv->interfaces; i++) {
			if ((rx_irq_status & ( 1 << i)) != 0)
				cv2eth_rx_irq(netdev_priv(priv->dev[i]));
		}
	}
	tx_irq_status = combo6_fpga_readl(combo, PAC_HW_TX_IRQ);
	if (tx_irq_status != 0) {
		status_zero = 1;
		cv2eth_tx_irq(combo);
	}

	if (unlikely(status_zero == 0))
		dev_warn(&combo->pci->dev, "Interrupt caught, but status zero!"
			   " Rx (%08X) Tx (%08X)\n", rx_irq_status, tx_irq_status);

//	PRINT_FNC_EXIT;
	return IRQ_HANDLED;
}

#ifdef CONFIG_OLD_NAPI
static int cv2eth_rx_poll(struct net_device* dev, int* budget) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	int cnt;
	cnt = cv2eth_rx(ndev, *budget);
	dev->quota -= cnt;
	*budget -= cnt;
	return (cnt != 0);
}
#else
static int cv2eth_rx_poll_napi(struct napi_struct* napi, int budget) {
	struct cv2eth_netdev *ndev = netdev_priv(napi->dev);
	return cv2eth_rx(ndev, budget);
}

#endif
/**
 * cv2eth_rx_poll_napi - rx napi polling function
 * @napi:		napi context handle
 * @budget:		maximal number of packets to receive during single invocation
 *
 * Polling function is repeatedly called. Each invocation is allowed to process
 * *budget* packets at maximum. If the budget is not exhausted, polling is
 * shutdown and new timeout value is set into the card.
 *
 * Packet reception consists of unmapping DMA mapping, setting sk_buff structure
 * elements, updating statistics, handing over the packet to the kernel and
 * setting new tail value to the card.
 */
static int cv2eth_rx(struct cv2eth_netdev* ndev, int budget) {
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct combo6 *combo6 = ndev->combo6;
	unsigned int ifc = ndev->index;
	struct cv2eth_desc* desc;
	struct cv2eth_skb* cv2_skb;
	unsigned int cnt = 0;
	unsigned int cdesc = 0;
	u32 length = 0;
//	unsigned long long j_str, j_end, j;

//	PRINT_FNC_ENTRY;
	// we do not suppose scattered packets
//	rdtscll(j_str);
	while (cnt < budget) {
		desc = get_desc_addr(priv, ifc, PAC_DIR_RX, ndev->rx.head);
		if (le64_to_cpu(desc->addr) != 0)
			break;
		cv2_skb = &ndev->rx_skbs[ndev->rx.head % ndev->max_descs];
		pci_unmap_single(combo6->pci, cv2_skb->phys, CV2ETH_PKT_SIZE,
			   	PCI_DMA_FROMDEVICE);

		cv2_skb->skb->dev = ndev->dev;
		length = le32_to_cpu(desc->length);
		skb_put(cv2_skb->skb, length);
#if (HW_HEADER_SIZE > 0)
		skb_pull(cv2_skb->skb, HW_HEADER_SIZE);
#endif
		cv2_skb->skb->protocol = eth_type_trans(cv2_skb->skb, ndev->dev);
		cv2_skb->skb->ip_summed = CHECKSUM_UNNECESSARY;
		netif_receive_skb(cv2_skb->skb);
		cnt++, ndev->rx.head++;
		ndev->stats.rx_packets++;
#if (HW_HEADER_SIZE > 0)
		ndev->stats.rx_bytes += length - HW_HEADER_SIZE;
#else
		ndev->stats.rx_bytes += length;
#endif
		ndev->rx_pkts++;
	}

	if (cnt != 0) {
		cdesc = cv2eth_alloc_rxdescs(ndev, ndev->rx.tail, cnt);
//		if (printk_ratelimit())
//			PDEBUG("Budget %d cnt %d head %d tail %d new tail %d\n", 
//				budget, cnt, ndev->rx.head, ndev->rx.tail, ndev->rx.tail + cdesc);
		ndev->rx.tail += cdesc;
		combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_RX, PAC_HW_REG_TAIL),
					    ndev->rx.tail); /* set new RX tail */
	}

//	if (cnt < budget) {
	if (cnt == 0) {
//		if (printk_ratelimit())
//			PDEBUG("Rx%d NAPI finished polling!\n", ifc);

#ifdef CONFIG_OLD_NAPI
		netif_rx_complete(ndev->dev);
#else
		napi_complete(&ndev->napi);
#endif

		combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_RX, PAC_HW_REG_SWCNT),
					    ndev->rx_pkts); /* set new rx sw cnt */
		combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_RX, PAC_HW_REG_TOUT),
					    PAC_HW_RX_TIMEOUT); // * set RX timeout *
	}
//	rdtscll(j_end);

//	j = j_end - j_str;
//	PDEBUG("Rx%d poll took %Ld ticks (Abs start %Ld end %Ld)!\n", ifc, j, j_str, j_end);
//	PRINT_FNC_EXIT;
	return cnt;
}

/**
 * cv2eth_process_tx - process transmitted packets
 * @dev:		network device handle
 * @pkts:		number of transmitted packets
 *
 * For each transmitted packet statistics are updated and sk_buff instance is
 * freed. Conditionaly new timeout is set.
 */
static void cv2eth_process_tx(struct net_device* dev, unsigned int pkts) {
	struct sk_buff* skb;
	unsigned int nr_frags;
	unsigned int set_timeout = 0;
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	unsigned int ifc = ndev->index;
	unsigned int cnt = 0;
	
//	if (printk_ratelimit())
//	PDEBUG("Tx%d process."
//		   " Tail %d head %d real %d sched %d tx %d\n",  
//			ifc, ndev->tx.tail, ndev->tx.head, pkts, 
//			ndev->tx_sched_pkts, ndev->tx_pkts);
//	PDEBUG("Tx%d pkts (already processed) %d status upload tx %d tail %d head %d\n",  
//			ifc, ndev->tx_pkts, pkts, ndev->tx.tail, ndev->tx.head);
	set_timeout = ndev->tx_pkts != pkts;
	while (ndev->tx_pkts != pkts) {
		if (++cnt > 1000) {
			PDEBUG("Tx%d processing %d packet.\n", ifc, cnt);
			return;
		}
		skb = ndev->tx_skbs[ndev->tx.head % ndev->max_descs].skb;
		nr_frags = skb_shinfo(skb)->nr_frags;
		ndev->tx.head += (nr_frags == 0) ? 1 : nr_frags;
		ndev->tx_pkts++;
		dev_kfree_skb_irq(skb);
		ndev->stats.tx_packets++;
		ndev->stats.tx_bytes += skb->len;
	}
	if (set_timeout) {
		combo6_fpga_writel(ndev->combo6, 
				PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TOUT),
				PAC_HW_TX_TIMEOUT); // * set TX timeout *
	}
/*	else
		PDEBUG("Tx%d nothing done."
		   " Tail %d head %d real %d sched %d tx %d\n",  
			ifc, ndev->tx.tail, ndev->tx.head, pkts, 
			ndev->tx_sched_pkts, ndev->tx_pkts);
			*/
}

/**
 * cv2eth_tx_irq - tx irq handling routine
 * @device:		interrupting device
 *
 * For each interface call packet processing function.
 */
static void cv2eth_tx_irq(struct combo6 *combo)
{
	struct cv2eth_private *priv = combo->private_data;
	struct net_device* dev;
	struct cv2eth_netdev *ndev;
	unsigned int i = 0;

	for (i = 0; i < priv->interfaces; i++) 
	{
		dev = priv->dev[i];
		ndev = netdev_priv(dev);
		if (test_bit(PAC_DIR_TX, &ndev->run_flag))
			cv2eth_process_tx(dev, cv2eth_tx_cnt(priv, i));
	}
}

/**
 * cv2eth_clear_tx - tx packets clearing function
 * @dev:		network device
 *
 * Does the clean up, when there are any unsent packets.
 */
static int cv2eth_clear_tx(struct net_device* dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	u32 pkts, cleared_pkts;
	struct sk_buff* skb;
	unsigned int nr_frags;
	int ret = 0;

	pkts = ndev->tx_sched_pkts;
	cleared_pkts = ndev->tx_pkts;
	ret = pkts - cleared_pkts;
	while (pkts > cleared_pkts) {
		skb = ndev->tx_skbs[ndev->tx.head % ndev->max_descs].skb;
		nr_frags = skb_shinfo(skb)->nr_frags;
		ndev->tx.head += (nr_frags == 0) ? 1 : nr_frags;
		cleared_pkts++;
		dev_kfree_skb(skb);
	}
	return ret;
}

/*----# cv2eth device operations ---------------------------------------------*/
/**
 * cv2eth_alloc_rings - allocate DMAable pages
 * @priv:		module private structure
 * @blocks:		number of pages allocated for each interface and direction
 *
 * Function allocates pages that constitutes ring buffers using consistent DMA
 * mapping.
 */
static int cv2eth_alloc_rings(struct cv2eth_private* priv, unsigned int blocks) {
	int i,j;
	int err_i, err_j;
	int ret = 0;
	void* ptr;
	struct cv2eth_block* pblks;
	
	// check block count has to be >0
	BUG_ON(blocks == 0);

	priv->rings = vmalloc(2*sizeof(struct cv2eth_ring));
	if (priv->rings == NULL) {
		PERR("Unable to allocate ring structure\n");
		return -ENOMEM;
	}
	memset(priv->rings, 0, 2*sizeof(struct cv2eth_ring));

	for (i = 0; i < 2; i++) {
		pblks = vmalloc(priv->interfaces*blocks*sizeof(struct cv2eth_block));
		if (pblks == NULL) {
			PERR("Unable to allocate rings\n");
			ret = -ENOMEM;
			goto err_ring;
		}
		priv->rings[i].blocks = pblks;
		for (j = 0; j < blocks*priv->interfaces; j++) {
			// alloc blocks for all interfaces
			ptr = pci_alloc_consistent(priv->combo6->pci, 
									   PAGE_SIZE, &pblks[j].phys);
			if (ptr == NULL) {
				PERR("Unable to allocate block %u of ring %u\n", j, i);
				ret = -ENOMEM;
				goto err_block;
			}
			memset(ptr, 0, PAGE_SIZE);
			pblks[j].virt = ptr;

		}
		priv->rings[i].count = blocks;

	}
	return 0;

	// free only as much blocks as were allocated
err_block:
	for (err_i = i; err_i >= 0; err_i--)
		for (err_j = (err_i == i) ? i : blocks*priv->interfaces; err_j > 0; err_j--)
			pci_free_consistent(priv->combo6->pci, PAGE_SIZE, 
					priv->rings[err_i].blocks[err_j - 1].virt, 
					priv->rings[err_i].blocks[err_j - 1].phys);

err_ring:
	for (err_i = i; err_i >= 0; err_i--)
		vfree(priv->rings[err_i - 1].blocks);
	vfree(priv->rings);
	return ret;

}

/**
 * cv2eth_bind_ring - bind ring buffer
 * @priv:		module private structure
 * @index:		interface index
 *
 * Function binds single pages into a ring buffer. Binding is done only for a
 * specified interface, but both directions (rx/tx) at once.
 */
static void cv2eth_bind_ring(struct cv2eth_private *priv, int index) {
// link blocks into ring buffer -- rx and tx at once
	struct cv2eth_desc* virt;
	unsigned int first, abs, rel, idesc, i;
	unsigned int blk_count;

	blk_count = priv->rings[0].count;
	idesc = PAGE_SIZE/CV2ETH_DESC_SIZE - 1;
	first = index*blk_count;

	// for each block
	for (abs = first + 1, rel = 1; rel < blk_count; abs++, rel++) {
		// connect the ring block[abs-1].last_desc -> block[abs]
		for (i = 0; i < 2; i++) {
			virt = (struct cv2eth_desc*)priv->rings[i].blocks[abs - 1].virt;
			cv2eth_fill_desc(virt+idesc, priv->rings[i].blocks[abs].phys, 0,
				   	CV2ETH_HW_NDF);
		}
	}
	// connect the block[last].last_desc -> block[0] -- make it look round
	for (i = 0; i < 2; i++) { 
		virt = (struct cv2eth_desc*)priv->rings[i].blocks[abs - 1].virt;
		cv2eth_fill_desc(virt+idesc, priv->rings[i].blocks[first].phys, 0, 
				CV2ETH_HW_NDF);
	}
}

/**
 * cv2eth_free_rings - unmap and free DMAable pages
 * @priv:			module private structure
 * 
 * Unmap a free DMAable pages constituting ring buffer. Should be called
 * in time of device detach. 
 */
static void cv2eth_free_rings(struct cv2eth_private *priv) {
	int i, j;

	for (i = 0; i < 2; i++) {
		for (j = 0; j < priv->rings[i].count*priv->interfaces; j++) {
			pci_free_consistent(priv->combo6->pci, PAGE_SIZE, 
					priv->rings[i].blocks[j].virt, 
					priv->rings[i].blocks[j].phys);
		}
		vfree(priv->rings[i].blocks);
	}

	vfree(priv->rings);
}

/**
 * cv2eth_attach - attach module to the card
 * @device:			device to attach
 * @interfaces:		number of interfaces
 * 
 * The ring buffers for rx and tx, tx status upload space and its DMA mapping
 * are created.
 */
static int cv2eth_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	int i, j, ret;
	struct cv2eth_private *priv;

	PRINT_FNC_ENTRY;

	if (COMBOV2_RX_IFACES(interfaces) != COMBOV2_TX_IFACES(interfaces)) {
		PERR("Number of Rx and Tx interfaces have to match!");
		return -EIO;
	}

	if (COMBOV2_RX_IFACES(interfaces) > CV2ETH_MAX_IFCS) {
		PERR("Card has more (%d) than supported (%d) interfaces!", 
				COMBOV2_RX_IFACES(interfaces),
				CV2ETH_MAX_IFCS);
		return -EIO;
	}

	pci_dev_get(combo->pci);
	priv = vmalloc(sizeof(struct cv2eth_private));
	if (priv == NULL)
		return -ENOMEM;
	memset(priv, 0, sizeof (struct cv2eth_private));

	priv->combo6 = combo;
	priv->interfaces = COMBOV2_RX_IFACES(interfaces);
	mutex_init(&priv->open_lock);

	/* allocate rings */
	ret = cv2eth_alloc_rings(priv, block_count);
	if (ret != 0)
		goto err;

	combo->private_data = priv;

	
	/* alloc space for transmitted packets counters upload */
	priv->tx_up_size = priv->interfaces * PAC_TXUP_CNT_SIZE;

	priv->tx_up.virt = pci_alloc_consistent(combo->pci, priv->tx_up_size, 
			&priv->tx_up.phys);

	if (priv->tx_up.virt == NULL) {
		PERR("Unable to allocate memory for tx packet counters!\n");
		ret = -ENOMEM;
		goto err1;
	}

	PDEBUG("Tx upload space %lX\n", (unsigned long)priv->tx_up.phys);
	/* set TX global upload addr */
	cv2eth_write_phys_addr(priv->combo6, 
						PAC_HW_REG(0, PAC_DIR_TX, PAC_HW_REG_TXGL), 
						priv->tx_up.phys);


	for (i = 0; i < priv->interfaces; i++) {
		cv2eth_bind_ring(priv, i);
		ret = cv2eth_netdev_init(combo, &(priv->dev[i]), i);
		if (ret != 0)
			goto err2;
	}

	PRINT_FNC_EXIT;
	return 0;

err2:
	for (j = 0; j < i; j++)
		cv2eth_netdev_done(priv->dev[j]);
	pci_free_consistent(priv->combo6->pci, priv->tx_up_size, 
					priv->tx_up.virt, priv->tx_up.phys);

err1:
	cv2eth_free_rings(priv);
err:
	vfree(priv);
	combo->private_data = NULL;
	pci_dev_put(combo->pci);
	return ret;
}


/**
 * cv2eth_detach - driver/module detach
 * @driver:			detached driver
 * 
 * Detach module from driver. Interfaces are freed and rings are destroyed.
 */
static int cv2eth_detach(struct combo6 *combo)
{
	struct cv2eth_private *priv = combo->private_data;
	int i;

	PRINT_FNC_ENTRY;
	
	for (i = 0; i < priv->interfaces; i++)
		cv2eth_netdev_done(priv->dev[i]);

	// free TX upload space
	pci_free_consistent(combo->pci, priv->tx_up_size, 
					priv->tx_up.virt, priv->tx_up.phys);

	// free rings
	cv2eth_free_rings(priv);
	vfree(priv);
	combo->private_data = NULL;
	pci_dev_put(combo->pci);

	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_alloc_rxdescs - allocate sk_buffs for incoming packets
 * @ndev:		net driver identification
 * @pos:		position of first free descriptor
 * @count:		number of descriptors to be allocated
 *
 * Function tries to allocate *count* instances of sk_buff structure,
 * creates streaming DMA mapping and fills the data into the descriptor.
 */
static int cv2eth_alloc_rxdescs(struct cv2eth_netdev *ndev, unsigned int pos,
	   							unsigned int count) {
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct cv2eth_desc* dvirt;
	struct sk_buff *skb = NULL;
	dma_addr_t phys = 0;
	unsigned int i = pos;

//	PRINT_FNC_ENTRY;
	for (i = pos; i < pos + count; i++) {
		dvirt = get_desc_addr(priv, ndev->index, PAC_DIR_RX, i);
		skb = dev_alloc_skb(CV2ETH_PKT_SIZE);
		if (unlikely(skb == NULL)) {
			PDEBUG("Failed to allocate sk buff %u\n", i);
			return (i - pos);
		}
		phys = pci_map_single(ndev->combo6->pci, skb->data, CV2ETH_PKT_SIZE,
			   PCI_DMA_FROMDEVICE);
		if (unlikely(phys == 0)) {
			dev_kfree_skb(skb);
			PDEBUG("Failed to create DMA mapping for sk buff %u\n", i);
			return (i - pos);
		}
		ndev->rx_skbs[i % ndev->max_descs].skb = skb;
		ndev->rx_skbs[i % ndev->max_descs].phys = phys;
		cv2eth_fill_desc(dvirt, phys, CV2ETH_PKT_SIZE, CV2ETH_RX_FLAGS);

//		(((i + 1) % 255) == 0) ? CV2ETH_RX_FLAGS | CV2ETH_HW_INTF : CV2ETH_RX_FLAGS);
	}
//	PRINT_FNC_EXIT;
	return (i - pos);
//	return 0;
}

/**
 * cv2eth_free_rxdescs - free previously allocated descriptors
 * @ndev:		net device identification
 * @dir:		direction (rx/tx)
 * @pos:		position of first descriptor in ring buffer
 * @count:		number of descriptors to be freed
 *
 * Function frees *count* descriptor from the *dir* ring buffer of *ndev*
 * begining with *pos*.
 */
static int cv2eth_free_rxdescs(struct cv2eth_netdev *ndev, unsigned int pos, 
		unsigned int count) {
// free allocated descriptors
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct cv2eth_desc* dvirt;
	struct sk_buff *skb;
	dma_addr_t phys;
	unsigned int i;

//	PRINT_FNC_ENTRY;
	for (i = pos; i < pos + count; i++) {
		dvirt = get_desc_addr(priv, ndev->index, PAC_DIR_RX, i);
		skb = ndev->rx_skbs[i % ndev->max_descs].skb;
		if (skb == NULL) { 
			if (printk_ratelimit())
				PDEBUG("Trying to unmap and free NULL skb\n");
			continue;
		}
		phys = ndev->rx_skbs[i % ndev->max_descs].phys;
		pci_unmap_single(ndev->combo6->pci, phys, CV2ETH_PKT_SIZE,
			   	PCI_DMA_FROMDEVICE);
		dev_kfree_skb(skb);
	}

//	PRINT_FNC_EXIT;
	return 0;

}

/*----# cv2eth network device operations -------------------------------------*/
static int cv2eth_netdev_open(struct net_device *dev);
static int cv2eth_netdev_close(struct net_device *dev);
static int cv2eth_tx(struct sk_buff *skb, struct net_device *dev);
static int cv2eth_netdev_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int cv2eth_change_mtu(struct net_device *dev, int new_mtu);
static void cv2eth_set_rx_mode(struct net_device *dev);
static void cv2eth_tx_timeout(struct net_device *dev);
static struct net_device_stats *cv2eth_get_stats(struct net_device *dev);

/**
 * cv2eth_netdev_init - initialize interface private structure
 * @device:		card/device identification
 * @rdev:		structure instance to be initialized
 * @index:		net device interface index
 *
 * Function intializes net_device structure of interface with index *index*.
 *
 * After this call net device is registered in kernel and could be used by
 * userspace.
 */
static int cv2eth_netdev_init(struct combo6 *combo, struct net_device **rdev,
		int index)
{
	struct net_device *dev;
	struct cv2eth_netdev *cv2netdev;
	int erret, ret;

	static const struct net_device_ops ops = {
		.ndo_open 			= &cv2eth_netdev_open,
		.ndo_stop 			= &cv2eth_netdev_close,
		.ndo_start_xmit 	= &cv2eth_tx,
		.ndo_get_stats 		= &cv2eth_get_stats,
		.ndo_set_multicast_list = &cv2eth_set_rx_mode,
		.ndo_do_ioctl 		= &cv2eth_netdev_ioctl,
		.ndo_tx_timeout		= &cv2eth_tx_timeout,
		.ndo_change_mtu 	= &cv2eth_change_mtu,
	};

	dev = alloc_etherdev(sizeof(struct cv2eth_netdev));

	if (strstr(netdev_template, "%d"))
		strcpy(dev->name, netdev_template);
	else
		sprintf(dev->name, "%s%d%d", netdev_template,
				combo->index, index);

	cv2netdev = netdev_priv(dev);
	cv2netdev->index = index;
	cv2netdev->dev = dev;
	cv2netdev->combo6 = combo;
	cv2netdev->rx.head = cv2netdev->rx.tail = 0;
	cv2netdev->tx.head = cv2netdev->tx.tail = 0;
	cv2netdev->max_descs = block_count*(PAGE_SIZE/CV2ETH_DESC_SIZE-1);
	cv2netdev->tx_pkts = 0;
	cv2netdev->tx_sched_pkts = 0;
	cv2netdev->rx_pkts = 0;
	cv2netdev->run_flag = 0;

	cv2netdev->rx_skbs = (struct cv2eth_skb*)
			vmalloc(cv2netdev->max_descs*sizeof(struct cv2eth_skb));
	cv2netdev->tx_skbs = (struct cv2eth_skb*)
			vmalloc(cv2netdev->max_descs*sizeof(struct cv2eth_skb));
	if (cv2netdev->rx_skbs == NULL || cv2netdev->tx_skbs == NULL) {
		PERR("Failed to allocate sk_buff pointers!\n");
		ret = -ENOMEM;
		goto err;
	}

	PDEBUG("%s ndev %p rx_skbs addr %p tx_skbs addr %p\n", __FUNCTION__, cv2netdev, cv2netdev->rx_skbs, cv2netdev->tx_skbs);

#ifdef CONFIG_OLD_NAPI
	dev->poll = cv2eth_rx_poll;
	dev->weight = CV2ETH_NAPI_WEIGHT;
#else
	netif_napi_add(dev, &cv2netdev->napi, cv2eth_rx_poll_napi, CV2ETH_NAPI_WEIGHT);
#endif

	dev->destructor = free_netdev;
	dev->dev_addr[0] = 0x00;
	dev->dev_addr[1] = 0x11;
	dev->dev_addr[2] = 0x17;
	dev->dev_addr[3] = 0xff;
	dev->dev_addr[4] = 0xff;
	/* suppose we have 4 ifcs at maximum */
	dev->dev_addr[5] = (combo->index << 2) | index;

#ifdef CONFIG_HAVE_NET_DEVICE_OPS
	dev->netdev_ops = &ops;
#else
	combo6_copy_ndev_ops(dev, &ops);
#endif

	dev->watchdog_timeo 	= CV2ETH_TX_TIMEOUT;

	PDEBUG("Registering device %s at address %p\n", dev->name, dev);
	erret = register_netdev(dev);
	if (erret < 0) {
		PERR("Failed to create network device %s\n", dev->name);
		ret = erret;
		goto err;
	}

	*rdev = dev;
	return 0;

err:
	vfree(cv2netdev->rx_skbs);
	vfree(cv2netdev->tx_skbs);
	free_netdev(dev);
	return ret;
}

/**
 * cv2eth_netdev_done - the interface is done
 * @dev:		net device identification
 *
 * When detaching card unregister net device. 
 */
static void cv2eth_netdev_done(struct net_device *dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	struct cv2eth_skb* rx_skbs = ndev->rx_skbs;
	struct cv2eth_skb* tx_skbs = ndev->tx_skbs;
//	PRINT_FNC_ENTRY;
	BUG_ON(dev == NULL);

	PDEBUG("Unregistering device %s at address %p\n", dev->name, dev);
	unregister_netdev(dev);

	vfree(rx_skbs);
	vfree(tx_skbs);

//	PRINT_FNC_EXIT;
}

/**
 * cv2eth_start_hw - start hardware
 * @ndev:		cv2 net device identification
 * @ifc:		interface index
 * @dir:		direction (Rx=0/Tx=1)
 *
 * Do all of the hardware specific steps.
 */
static void cv2eth_start_hw(struct cv2eth_netdev* ndev, int ifc, int dir) {
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct combo6* c6 = priv->combo6;
	unsigned int blks = priv->rings[0].count;
	u32 tail = 0;
	u32 timeout = 0;
	dma_addr_t p = 0;
	u32 offset = 0;
	u32 val = 0;
	unsigned int counter = 0;

	int c_dir = (dir == PAC_DIR_RX) ? 'R' : 'T';
	PDEBUG("%cx%d starting.\n", c_dir, ifc);

	/* clear head no more needed - done as a part of descriptor initialization */

	/* rx specific steps */
	if (dir == PAC_DIR_RX) {
		/* set rx sw cnt */
		combo6_fpga_writel(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_SWCNT),
						ndev->rx_pkts); 
		PDEBUG("%cx%d initial sw cnt %u\n", c_dir, ifc, ndev->rx_pkts);

		/* clear rx hw cnt */
		val = combo6_fpga_readl(c6, PAC_HW_DESC_CHANNEL(ifc, dir) + 8);
	}

	/* init desc */
	p = priv->rings[dir].blocks[ifc*blks].phys;
	cv2eth_write_phys_addr(c6, PAC_HW_DESC_INIT(ifc, dir), p);
	PDEBUG("%cx%d initial desc address %lu (0x%lX)\n", c_dir, ifc,
		   (unsigned long)p, (unsigned long)p);

	/* set tail */
	tail = (dir == PAC_DIR_RX) ? ndev->rx.tail : ndev->tx.tail;
	combo6_fpga_writel(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_TAIL), tail);
	PDEBUG("%cx%d initial tail %u\n", c_dir, ifc, tail);


	/* enable ibuf/obuf */
	offset = (dir == PAC_DIR_RX) ? CV2ETH_HW_IBUF_EN(ifc) : CV2ETH_HW_OBUF_EN(ifc);
	combo6_fpga_writel(c6, offset, 0x1);

	/* set timeout */
	timeout = (dir == PAC_DIR_RX) ? PAC_HW_RX_TIMEOUT : PAC_HW_TX_TIMEOUT;
	combo6_fpga_writel(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_TOUT), timeout); 
	PDEBUG("%cx%d initial timeout %u\n", c_dir, ifc, timeout);

	/* start channel */
	combo6_fpga_writel(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_CTRL), PAC_HW_CTRL_RUN); 

	while (1) {
		val = combo6_fpga_readl(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_STATUS));

		if (val == PAC_HW_STAT_RUN) {
			set_bit(dir, &ndev->run_flag);
			PDEBUG("%cx%d up and running.\n", c_dir, ifc);
			break;
		}
		if (counter++ > 100) {
			dev_warn(&c6->pci->dev, "channel didn't become ready in 100 us"
				   " (no running after start), stat %cx=%.8x\n", c_dir, val);
			break;
		}
		udelay(1);
	}
}

/**
 * cv2eth_netdev_open - open the interface
 * @dev:		net device identification
 *
 * Called for example when ifconfig *interface_name* up.
 *
 * Function pre-fills rx ring buffer and starts hw.
 */
static int cv2eth_netdev_open(struct net_device *dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	struct cv2eth_private *priv = ndev->combo6->private_data;
	unsigned int max_descs = ndev->max_descs;
	unsigned int cdesc = 0;
	int ifc = ndev->index;

//	PRINT_FNC_ENTRY;

	mutex_lock(&priv->open_lock);
	cdesc = cv2eth_alloc_rxdescs(ndev, 0, max_descs);
	ndev->rx.tail += cdesc;
#ifndef CONFIG_OLD_NAPI
	napi_enable(&ndev->napi);
#endif
	/* start hw */
	cv2eth_start_hw(ndev, ifc, PAC_DIR_RX);
	cv2eth_start_hw(ndev, ifc, PAC_DIR_TX);
	mutex_unlock(&priv->open_lock);

	netif_start_queue(dev);

//	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_stop_hw - stop hardware
 * @ndev:		cv2 net device identification
 * @ifc:		interface index
 * @dir:		direction (Rx=0/Tx=1)
 *
 * Send stop command to the control register of appropriate channel and
 * wait for response.
 */
static void cv2eth_stop_hw(struct cv2eth_netdev* ndev, int ifc, int dir) {
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct combo6* c6 = priv->combo6;
	u32 stat;
//	int c_dir = (dir == PAC_DIR_RX) ? 'R' : 'T';
	unsigned int counter = 0;
	u32 offset = 0;

	PDEBUG("%cx%d stopping.\n", (dir == PAC_DIR_RX) ? 'R' : 'T', ifc);

	/* disable ibuf/obuf */
	offset = (dir == PAC_DIR_RX) ? CV2ETH_HW_IBUF_EN(ifc) : CV2ETH_HW_OBUF_EN(ifc);
	combo6_fpga_writel(c6, offset, 0x0);

	/* stop channel */
	combo6_fpga_writel(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_CTRL), 
							PAC_HW_CTRL_STP); 

	while (1) {
		stat = combo6_fpga_readl(c6, PAC_HW_REG(ifc, dir, PAC_HW_REG_STATUS));
		if (stat == PAC_HW_STAT_STP) {
			clear_bit(dir, &ndev->run_flag);
			PDEBUG("%cx%d stopped.\n", (dir == PAC_DIR_RX) ? 'R' : 'T', ifc);
			break;
		}
		if (counter++ > 100) {
			dev_warn(&c6->pci->dev, "card didn't stop in 100 us (no stopped "
				   "after stop), stat=%.8x!\n", stat);
			break;
		}
		udelay(1);
	}

}

/**
 * cv2eth_netdev_close - close the interface
 * @dev:		net device identification
 *
 * Called for example when ifconfig *interface_name* down.
 *
 * Function stops hw and clears content of ring buffers.
 */
static int cv2eth_netdev_close(struct net_device *dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	struct cv2eth_private *priv = ndev->combo6->private_data;
	int ifc = ndev->index;
	unsigned int irq_status = 0;

//	PRINT_FNC_ENTRY;

	netif_stop_queue(dev);

	mutex_lock(&priv->open_lock);
	cv2eth_stop_hw(ndev, ifc, PAC_DIR_RX);
	cv2eth_stop_hw(ndev, ifc, PAC_DIR_TX);

	cv2eth_process_tx(dev, cv2eth_tx_cnt(priv, ifc));
	if (ndev->tx_pkts != ndev->tx_sched_pkts) {
		PERR("Tx%d not cleared tx packets %d\n", ifc, 
					ndev->tx_sched_pkts - ndev->tx_pkts);
		cv2eth_clear_tx(dev);
	}
	cv2eth_tx_cnt_clear(priv, ifc);

#ifndef CONFIG_OLD_NAPI
	napi_disable(&ndev->napi);
#endif
	irq_status = combo6_fpga_readl(priv->combo6, PAC_HW_RX_IRQ);
	if (irq_status != 0)
		dev_warn(&priv->combo6->pci->dev,
			   "Rx interrupt status not null at close (%x)\n", irq_status);

	cv2eth_free_rxdescs(ndev, ndev->rx.head, ndev->rx.tail - ndev->rx.head);
	ndev->rx.head = 0; ndev->rx.tail = 0;
	ndev->tx.head = 0; ndev->tx.tail = 0;
	ndev->tx_pkts = 0; ndev->tx_sched_pkts = 0;
	ndev->rx_pkts = 0;

	mutex_unlock(&priv->open_lock);

//	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_tx - schedule packet for transmission
 * @skb:		sk_buff instation containing packet
 * @dev:		net device identification
 *
 * Called for each transmitted packet. Fills descriptor for each packet and
 * appropriately modifies hw indexes.
 */
static int cv2eth_tx(struct sk_buff *skb, struct net_device *dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
	struct cv2eth_private *priv = ndev->combo6->private_data;
	struct combo6 *combo6 = ndev->combo6;
	u32 tail = ndev->tx.tail;
	u32 head = ndev->tx.head;
	unsigned int max_descs = ndev->max_descs;
	int space = max_descs - (tail - head);
	dma_addr_t phys;
	struct cv2eth_desc* dvirt;
	unsigned int ifc = ndev->index;
	u32 len = skb->len;
	unsigned int nr_frags = skb_shinfo(skb)->nr_frags;
	skb_frag_t *frag;
	void *frag_addr;
	unsigned int i;
	unsigned int set_intf;
	u32 tx_up;
	unsigned long trans;

//	PRINT_FNC_ENTRY;
	BUG_ON(space < 0);

	if (unlikely(space == 0 || space < nr_frags)) {
		tx_up = cv2eth_tx_cnt(priv, ifc);
		PDEBUG("Tx%d: ring is full, not enough space! Stopping queue.\n", ifc); 
		PDEBUG("Tx%d max %d tail %d head %d real %d sched %d tx %d\n", 
				ifc, max_descs, tail, head, tx_up, 
				ndev->tx_sched_pkts, ndev->tx_pkts);
		netif_stop_queue(dev);
		return NETDEV_TX_BUSY;
	}
	if (nr_frags != 0) 
		len = skb->len - skb->data_len; /* scattered packet */

	if (((unsigned long)skb->data & 0x7) != 0) {
//		if (printk_ratelimit()) {
//			PDEBUG("Tx%d: Unalligned packet at adress %p.\n", ifc, skb->data);
//			PDEBUG("Tx%d: Head %p data %p tail %p end %p\n", ifc, skb->head, skb->data, skb->tail, skb->end);
//		}
		if ((((unsigned long)skb->head & 0x7) == 0)&&(skb->data > skb->head)) {
			trans = (unsigned long)skb->data - (unsigned long)skb->head;
//			if (printk_ratelimit())
//				PDEBUG("Tx%d: Alligning to head. Distance %d.\n", ifc, trans);
			memcpy(skb->head, skb->data, len);
			skb->data -= trans;
			skb->tail -= trans;
		}
		else {
//			if (printk_ratelimit())
//				PDEBUG("Tx%d: Unalligned packet at adress %p. Skipping.\n", ifc, skb->data);
			dev_kfree_skb(skb);
			return 0;
		}
	}

	/* set interrupt flag for each 32th scheduled tx packet */
//	set_intf = ((++ndev->tx_sched_pkts & 0x1F) == 0) ? CV2ETH_HW_INTF : 0;
	set_intf = ((++ndev->tx_sched_pkts % 0xFF) == 0) ? CV2ETH_HW_INTF : 0;

	tail %= max_descs;
	phys = pci_map_single(combo6->pci, skb->data, len, PCI_DMA_TODEVICE);
	ndev->tx_skbs[tail].skb = skb;
	ndev->tx_skbs[tail].phys = phys;
	dvirt = get_desc_addr(priv, ifc, PAC_DIR_TX, ndev->tx.tail++);
	cv2eth_fill_desc(dvirt, phys, len,  
					(nr_frags == 0) ? (CV2ETH_HW_LFF | set_intf) : 0);

	for (i = 1; i < nr_frags; i++) {
		frag = &skb_shinfo(skb)->frags[i-1];
		frag_addr = (void *) page_address(frag->page + frag->page_offset);

		tail = ndev->tx.tail++ % max_descs; /* sw structure acces modulo size */
		phys = pci_map_single(combo6->pci, frag_addr, frag->size, PCI_DMA_TODEVICE);
		ndev->tx_skbs[tail].skb = NULL;
		ndev->tx_skbs[tail].phys = phys;
		dvirt = get_desc_addr(priv, ifc, PAC_DIR_TX, tail);
		cv2eth_fill_desc(dvirt, phys, frag->size, 
						(i == nr_frags - 1) ? (CV2ETH_HW_LFF | set_intf) : 0);
	}

//	PDEBUG("Tx%d ndev %p addr virt %p phys %u tail %d\n", ifc, ndev, dvirt, phys, ndev->tx.tail);
	dev->trans_start = jiffies;
	/* how to move tail only for a block of packets? */
	combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TAIL),
					ndev->tx.tail); /* set TX tail */
//	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_netdev_ioctl - net device io control
 * @dev:		net device identification
 * @rq:			interface request structure instance
 * @cmd:		command
 *
 * Not used yet.
 */
static int cv2eth_netdev_ioctl(struct net_device *dev, struct ifreq *rq, int cmd) {
	PRINT_FNC_ENTRY;
	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_change_mtu - change MTU
 * @dev:		net device identification
 * @cmd:		new size of maximal transmision unit
 *
 * Not used yet.
 */
static int cv2eth_change_mtu(struct net_device *dev, int new_mtu) {
	PRINT_FNC_ENTRY;
	PRINT_FNC_EXIT;
	return 0;
}

/**
 * cv2eth_set_rx_mode
 * @dev:		net device identification
 *
 * Not used yet.
 */
static void cv2eth_set_rx_mode(struct net_device *dev) {
//	PRINT_FNC_ENTRY;
	/* will probably remain empty */
	/* as we do not perform any packet filtration */
	/* hence no modifications when entering promiscuous mode */
//	PRINT_FNC_EXIT;
}

/**
 * cv2eth_tx_timeout - transmission timeout
 * @dev:		net device identification
 *
 * Not used yet.
 */
static void cv2eth_tx_timeout(struct net_device *dev) {
	struct cv2eth_netdev *ndev = netdev_priv(dev);
//	struct cv2eth_private *priv = ndev->c6device->private_data;
	int ifc = ndev->index;
	u32 tail = ndev->tx.tail;
	u32 head = ndev->tx.head;
	unsigned int max_descs = ndev->max_descs;
	int space = max_descs - (tail - head);

//	PRINT_FNC_ENTRY;

	BUG_ON(space < 0);

	if (space == 0) {
		printk(KERN_INFO "Tx%d we should wait a little longer! Or something serious happened.\n", ifc);
//		cv2eth_process_tx(dev, cv2eth_tx_cnt(priv, ifc));
	}

//	printk(KERN_INFO "Tx%d wakeing up the queue!\n", ifc);

	if (netif_queue_stopped(dev))
		netif_wake_queue(dev);

	
//	PRINT_FNC_EXIT;
}

/**
 * cv2eth_get_stats - get interface statistics
 * @dev:		net device identification
 */
static struct net_device_stats *cv2eth_get_stats(struct net_device *dev){

	struct cv2eth_netdev *ndev = netdev_priv(dev);
//	PRINT_FNC_ENTRY;
//	PRINT_FNC_EXIT;
	return &ndev->stats;
}

/*----# driver loading part #-------------------------------------------------*/
/**
 * cv2eth_probe - pci device probe
 * @pci:		pci device handle
 * @id:			pci device id
 */
static int __devinit cv2eth_probe(struct pci_dev *pci,
					const struct pci_device_id *id)
{
	static int dev;
	struct combov2_init init = {};

//	PRINT_FNC_ENTRY;
	if (!enable[dev]) {
		dev++;
		return -ENOENT;
	}
	init.invalidmemsize = invalidmemsize[dev];
	init.bootprotect = bootprotect[dev];
	init.bootdelay = bootdelay[dev];
	init.usemode = usemode[dev];

	return combov2_probe(pci, id, &init, THIS_MODULE);
}

/**
 * cv2eth_remove - pci device remove
 * @pci:		pci device handle
 */
static void __devexit cv2eth_remove(struct pci_dev *pci)
{
//	PRINT_FNC_ENTRY;
	combov2_remove(pci);
//	PRINT_FNC_EXIT;
}

static struct pci_device_id cv2eth_ids[] = {
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0x6d05,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0100 }, /* ML555 */
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xc032,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0100 }, /* ComboV2 lxt110*/
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xc132,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0100 }, /* ComboV2 lxt155*/
	{ 0, }

};
MODULE_DEVICE_TABLE(pci, cv2eth_ids);

static struct pci_driver driver = {
	.name = "ComboV2 Eth",
	.id_table = cv2eth_ids,
	.probe 	  = cv2eth_probe,
	.remove   = __devexit_p(cv2eth_remove),
};

/*----# module initialization and exit function ------------------------------*/
/**
 * cv2eth_init - module initialization
 */
static int __init cv2eth_init(void) {
	int ret;

	ret = combo_register_driver(&cv2eth_ops);
	if (ret < 0) {
		printk(KERN_ERR "cv2eth: can't register combo driver\n");
		return ret;
	}

	ret = pci_register_driver(&driver);
	if (ret) {
		printk(KERN_ERR "cv2eth: can't register pci driver\n");
		combo_unregister_driver(&cv2eth_ops);
	}
	return ret;

}

/**
 * cv2eth_exit - module exit
 */
static void __exit cv2eth_exit(void) {
	pci_unregister_driver(&driver);
	combo_unregister_driver(&cv2eth_ops);
}

module_init(cv2eth_init)
module_exit(cv2eth_exit)

#ifndef MODULE

/* format is: combov2eth=enable */

static int __init cv2eth_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	PRINT_FNC_ENTRY;
	if (nr_dev >= COMBO6_CARDS)
		return 0;
	(void)(get_option(&str,&enable[nr_dev]) == 2);
	nr_dev++;
	PRINT_FNC_EXIT;
	return 1;
}

__setup("combov2eth=", cv2eth_setup);

#endif
