/*
 * szedata 2 module
 *
 * Copyright (c) 2008-2009 CESNET
 * Copyright (c) 2007-2009 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/cdev.h>
#include <linux/compat.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/ioctl.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/types.h>

#include "szedata2.h"
#include "szedata2k.h"
#include "sze2.h"

struct sze2_subscribe_area_old {
	__u32 areas[2];
	__u32 poll_thresh[2];
};

#define SZE2_IOC_SUBSCRIBE_AREA_OLD _IOW(SZE2_IOC_BASE, 0, \
		struct sze2_subscribe_area_old)

static int szedata2_major;

/*
 * FOPS
 */

static int szedata_open(struct inode *inode, struct file *filp)
{
	struct szedata2 *sd = container_of(inode->i_cdev,
			struct szedata2, cdev);
	struct szedata2_app *app;

	app = szedata2_open(sd, iminor(inode));
	if (IS_ERR(app))
		return PTR_ERR(app);

	filp->private_data = app;

	return 0;
}

static int szedata_release(struct inode *inode, struct file *filp)
{
	struct szedata2_app *app = filp->private_data;

	szedata2_close(app);

	return 0;
}

static long szedata_ioctl(struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	struct szedata2_app *app = filp->private_data;
	void __user *argp = (void __user *)arg;
	int ret = 0;

	switch (cmd) {
	case SZE2_IOC_SUBSCRIBE_AREA_OLD: {
		struct sze2_subscribe_area_old subo;
		struct sze2_subscribe_area sub;

		if (printk_ratelimit())
			printk(KERN_INFO "szedata2: '%s' uses deprecated "
					"SUBSCRIBE_AREA ioctl\n",
					current->comm);

		if (copy_from_user(&subo, argp, sizeof(subo)))
			return -EFAULT;

		BUILD_BUG_ON(sizeof(sub.areas) != sizeof(subo.areas));
		memcpy(&sub.areas, &subo.areas, sizeof(sub.areas));

		ret = szedata2_subscribe_area(app, &sub);
		break;
	}
	case SZE2_IOC_SUBSCRIBE_AREA: {
		struct sze2_subscribe_area sub;

		if (copy_from_user(&sub, argp, sizeof(sub)))
			return -EFAULT;

		ret = szedata2_subscribe_area(app, &sub);
		break;
	}
	case SZE2_IOC_START:
		ret = szedata2_start_devices(app);
		break;
	case SZE2_IOC_RXLOCKDATA: {
		u32 areas;
		ret = get_user(areas, (u32 __user *)argp);
		if (ret)
			break;
		ret = szedata2_rxlock_data(app, areas);
		if (ret < 0)
			break;
		ret = put_user(ret, (u32 __user *)argp);
		break;
	} case SZE2_IOC_RXUNLOCKDATA:
		szedata2_rxunlock_data(app);
		break;
	case SZE2_IOC_TXLOCKDATA: {
		struct sze2_tx_lock tx;

		if (copy_from_user(&tx, argp, sizeof(tx)))
			return -EFAULT;
		ret = szedata2_txlock_data(app, &tx);
		break;
	} case SZE2_IOC_TXUNLOCKDATA: {
		struct sze2_tx_unlock tx;
		if (copy_from_user(&tx, argp, sizeof(tx)))
			return -EFAULT;
		ret = szedata2_txunlock_data(app, &tx);
		break;
	} case SZE2_IOC_TXLOCKUNLOCKDATA: {
		struct sze2_tx_unlock tx_unlock;
		struct sze2_tx_lock tx_lock;

		if (copy_from_user(&tx_lock, argp, sizeof(tx_lock)))
			return -EFAULT;

		if (app->write_size[tx_lock.area]){
			tx_unlock.area = tx_lock.area;
			tx_unlock.size = app->write_size[tx_lock.area];
			ret = szedata2_txunlock_data(app, &tx_unlock);
			if (ret)
				return ret;
		}
		ret = szedata2_txlock_data(app, &tx_lock);
		break;
	} case SZE2_IOC_STOP:
		szedata2_stop_devices(app);
		break;
	case SZE2_IOC_AREASIZE: {
		struct szedata2_ring *ring;
		struct sze2_areasize size;

		if (copy_from_user(&size, argp, sizeof(size)))
			return -EFAULT;
		if (size.space >= SZE2_MMIO_MAX)
			return -EINVAL;
		ring = &app->sd->ring[size.space];
		if (size.area >= ring->areas)
			return -EINVAL;
		size.size = ring->area[size.area].asize;
		if (copy_to_user(argp, &size, sizeof(size)))
			return -EFAULT;
		break;
	} case SZE2_IOC_GETNUMANODE: {
		int node_id;

#ifdef CONFIG_NUMA
		node_id = app->sd->dev->numa_node;
#else
		node_id = -1;
#endif

		ret = put_user(node_id, (int __user *)argp);
		break;
	} default:
		return -ENXIO;
	}

	return ret;
}

#ifdef CONFIG_COMPAT
static long szedata_compat_ioctl(struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	return szedata_ioctl(filp, cmd, (unsigned long)compat_ptr(arg));
}
#else
#define szedata_compat_ioctl NULL
#endif

static unsigned int szedata_poll(struct file *filp,
		struct poll_table_struct *wait)
{
	struct szedata2_app *app = filp->private_data;

	poll_wait(filp, &app->poll_wait, wait);
	return szedata2_poll(app);
}

/*
 * at least one space is expected
 */
static inline unsigned int szedata2_find_space(struct szedata2_app *app,
		unsigned long offset)
{
	unsigned int space;

	for (space = SZE2_MMIO_MAX - 1; space > 0; space--)
		if (app->status_page->sizes[space] &&
				offset >= app->status_page->offsets[space])
			break;

	return space;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18)

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 23)
static int szedata2_ring_mmap_fault(struct vm_area_struct *vma,
		struct vm_fault *vmf)
{
	struct szedata2_app *app = vma->vm_private_data;
	struct szedata2 *sd = app->sd;
	unsigned long offset = vmf->pgoff << PAGE_SHIFT;
	unsigned int space = szedata2_find_space(app, offset);
	struct page *page;
	void *virt;

	offset -= app->status_page->offsets[space];
	if (offset >= app->status_page->sizes[space])
		return VM_FAULT_SIGBUS;

	szedata2_get_virt(sd, space, 0, &virt, offset, PAGE_SIZE);
	page = virt_to_page(virt);
	get_page(page);
	vmf->page = page;

	return 0;
}

static int szedata2_ring_mmap(struct szedata2_app *app,
		struct vm_area_struct *vma)
{
	static struct vm_operations_struct ops = {
		.fault = szedata2_ring_mmap_fault,
	};
	unsigned long i;
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned int space = szedata2_find_space(app, offset);
	unsigned long block_size = app->sd->ring[space].blk_size;
	dma_addr_t phys;

	if (space == 0 && !app->sd->ring[0].blk_count)
		return -ENODEV;

	if (space != SZE2_MMIO_TX &&
			(vma->vm_flags & (VM_WRITE | VM_READ)) != VM_READ)
		return -EINVAL;

	/* the test below may overflow */
	if (size > app->status_page->sizes[space])
		return -EINVAL;
	if (offset - app->status_page->offsets[space] >
			app->status_page->sizes[space] - size)
		return -EINVAL;

	if(app->sd->state & SZE2_MAPINSTANTLY)
	{
		offset -= app->status_page->offsets[space];

		if(offset % block_size)
			return -EINVAL;
		if(size % block_size)
			return -EINVAL;

		for(i = offset; i < offset+size; i+= block_size)
		{
			szedata2_get_phys(app->sd, space, 0, &phys, i, block_size);
			remap_pfn_range(vma, vma->vm_start+i, phys >> PAGE_SHIFT, block_size, vma->vm_page_prot);
		}
	}
	else
		vma->vm_ops = &ops;
	vma->vm_private_data = app;

	return 0;
}
#endif /* >= 2.6.23 */

static int szedata2_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct szedata2_app *app = filp->private_data;

	if (vma->vm_pgoff >= ((PAGE_ALIGN(sizeof(*app->status_page)) +
			PAGE_SIZE) >> PAGE_SHIFT))
		return szedata2_ring_mmap(app, vma);

	if (vma->vm_pgoff >=
			(PAGE_ALIGN(sizeof(*app->status_page)) >> PAGE_SHIFT))
		return remap_vmalloc_range(vma, app->write_size,
				vma->vm_pgoff - (PAGE_SIZE >> PAGE_SHIFT));

	if ((vma->vm_flags & (VM_WRITE | VM_READ)) != VM_READ)
		return -EINVAL;

	return remap_vmalloc_range(vma, app->status_page, vma->vm_pgoff);
}
#endif /* >= 2.6.18 */

static struct file_operations szedata2_fops = {
	.owner = THIS_MODULE,
	.open = szedata_open,
	.release = szedata_release,
	.mmap = szedata2_mmap,
	.poll = szedata_poll,
	.unlocked_ioctl = szedata_ioctl,
	.compat_ioctl = szedata_compat_ioctl,
};

int szedata2_char_add(struct szedata2 *sd, int minor)
{
	int ret;

	cdev_init(&sd->cdev, &szedata2_fops);
	ret = cdev_add(&sd->cdev, MKDEV(szedata2_major, minor), 1);
	if (ret)
		printk(KERN_ERR "szedata2: can't register cdev\n");

	return ret;
}

void szedata2_char_remove(struct szedata2 *sd)
{
	cdev_del(&sd->cdev);
}

__init int szedata2_char_init(void)
{
	dev_t devt;
	int ret;

	ret = alloc_chrdev_region(&devt, 0, SZEDATA2_MAX_MINORS, "szedata2");
	if (ret) {
		printk(KERN_ERR "szedata2: can't alloc chardev\n");
		goto err;
	}
	szedata2_major = MAJOR(devt);

	return 0;

err:
	return ret;
}

__exit void szedata2_char_exit(void)
{
	unregister_chrdev_region(MKDEV(szedata2_major, 0), SZEDATA2_MAX_MINORS);
}
