/*
 *  sc6pcr.c: szedata driver for combo6x, universal DMA driver for PCR bridge
 *  Copyright (c) 2006,2007 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/uaccess.h>

#include "combo6.h"
#include "combo6k.h"
#include "combo6x.h"
#include "combo6pcr.h"
#include "combo6pcrdma.h"

#include "szedata.h"
#include "szedatak.h"

#if 0
#define SC6PCR_DEBUG
#endif

#define SC6PCR_RX_BUFFERS	512
#define SC6PCR_RX_PKTS		(SC6PCR_RX_BUFFERS * PCRDMA_NSEGS)

#define SC6PCR_DMA_RX_FLAGS0	0
#define SC6PCR_DMA_RX_FLAGS1	(SC6PCR_DMA_RX_FLAGS0|DMA_FLAGS_INTR_MASK)

#define PCRDMA_LIMIT_BLOCKS	0x1

/*
 *
 */

struct sc6pcr_tmppage {
	char *virt;
	dma_addr_t phys;
};

struct sc6pcr_pktdma {
	struct szedata_block *pkt;
	struct pcrdma_seg *seg;
	struct pcrdma_dma *dma;
	dma_addr_t dma_addr;
};

struct sc6pcr_priv {
	struct combo6 *combo6;
	int interfaces;
	struct combo6_info_entry *proc_regs;
	char *sgdesc_virt;
	dma_addr_t sgdesc_phys;
	unsigned long sgdesc_size;
	struct pcrdma_cseg *cseg_virt;
	dma_addr_t cseg_phys;
	struct pcrdma_config *config_virt;
	dma_addr_t config_phys;
	struct sc6pcr_pktdma rx_pktdma[SC6PCR_RX_PKTS];
	struct sc6pcr_tmppage *tmp_pages;
	unsigned int tmp_page_count;
	unsigned int tmp_page_incount;
	unsigned int rx_pos;			/* to rx_pktdma */
	unsigned int hh_size;			/* read from PPC */
	unsigned int pkt_skip;			/* read from PPC */
	unsigned int tx_hw_packets;		/* read from PPC - unused */
	unsigned int pcrver;
};

/*
 *
 */

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("Combo6 IPV6 SZEDATA Linux Driver - Universal DMA Interface "
		"for PCR based code");
MODULE_LICENSE("GPL");

static int intr_step = 8;
static int ppc_intr_threshold = 100000;
static int alloc_tpages;
static int blocks[COMBO6_CARDS];
static int block_size[COMBO6_CARDS];

module_param(intr_step, int, 0644);
module_param(ppc_intr_threshold, int, 0644);
module_param(alloc_tpages, int, 0644);
module_param_array(blocks, int, NULL, 0);
module_param_array(block_size, int, NULL, 0);
MODULE_PARM_DESC(intr_step, "Interrupt step for SG list (default 8, use 64+ to "
		"reduce interrupts)");
MODULE_PARM_DESC(ppc_intr_threshold, "PPC interrupt threshold "
		"(default 100000)");
MODULE_PARM_DESC(alloc_tpages, "1 = Allocate temporary pages "
		"(default 0 - disable)");
MODULE_PARM_DESC(blocks, "SZEDATA block count");
MODULE_PARM_DESC(block_size, "SZEDATA block size");

static void check_rx(struct combo6 *combo);

static int sdev_open(struct szedata_device *dev);
static int sdev_close(struct szedata_device *dev);
static int sdev_start(struct szedata_device *sdev);
static int sdev_stop(struct szedata_device *sdev);
static int sdev_port_info(struct szedata_device *sdev,
		struct szedata_ioctl_get_interface *info);

/*
 *  restart
 */

static void restart(struct combo6 *combo)
{
	struct szedata_device *sdev = combo->private_data;
	struct sc6pcr_priv *priv = sdev->private_data;

	priv->rx_pos = 0;
	combo6_br_writel(combo, PCRDMA_RXHEAD, priv->rx_pktdma[0].dma_addr);
	combo6_br_writel(combo, PCRDMA_SET_CTRL, CTRL_ENABLE_RX |
			CTRL_ENABLE_MASK);
}

/*
 * interrupt handler
 */
static irqreturn_t sc6pcr_interrupt(struct combo6 *combo, unsigned int xmask)
{
	if (xmask & PCRDMA_INTR_RX_MASK)
		check_rx(combo);
	if (xmask & PCRDMA_INTR_TX_MASK)
		printk(KERN_ERR "combo6 pcidma TX IRQ for szedata?\n");
	if (xmask & PCRDMA_INTR_ERR_MASK) {
		printk(KERN_ERR "combo6 pcidma error IRQ, restarting "
				"(%lu)!\n", (long)jiffies);
		check_rx(combo);
		restart(combo);
	}
	return IRQ_HANDLED;
}

/*
 * packet I/O part
 */

static void my_alloc_pkt(struct szedata_device *sdev,
			 struct combo6 *combo6,
			 struct sc6pcr_pktdma *pkt,
			 unsigned int pos)
{
	struct sc6pcr_priv *priv = sdev->private_data;

	/* note port=0 is not ideal, but we do not know */
	/* the right port at this moment */
	pkt->pkt = szedata_device_get_block(sdev, ~0, 0);
	if (pkt->pkt == NULL) {
		pkt->seg->seg_stlen = 0;
		return;
	}
	if (priv->tmp_pages) {
		pkt->seg->seg_addr = cpu_to_le32(
			priv->tmp_pages[pos / priv->tmp_page_incount].phys +
			((pos % priv->tmp_page_incount) *
			 sdev->alloc_block_size));
	} else {
		pkt->seg->seg_addr = cpu_to_le32(pkt->pkt->phys);
	}
	pkt->seg->seg_stlen = cpu_to_le32(sdev->alloc_block_size);
}

static void check_rx(struct combo6 *combo6)
{
	struct szedata_device *sdev = combo6->private_data;
	struct sc6pcr_priv *priv = sdev->private_data;
	struct sc6pcr_pktdma *pkt = NULL;
	struct pcrdma_dma *dma = NULL;
	dma_addr_t dma_addr = 0;
	u32 seg_stlen, seg_len;

	while (1) {
		pkt = &priv->rx_pktdma[priv->rx_pos];
		if (pkt->dma != dma) {
			if (dma != NULL) {
				/* clear DONE flag */
				dma->dma_flags = cpu_to_le32(((priv->rx_pos/PCRDMA_NSEGS-1) % intr_step) == 0 ? SC6PCR_DMA_RX_FLAGS1 : SC6PCR_DMA_RX_FLAGS0);
				pci_dma_sync_single_for_device(combo6->pci,
						dma_addr,
						sizeof(struct pcrdma_dma),
						PCI_DMA_TODEVICE);
			}
			dma = pkt->dma;
			dma_addr = pkt->dma_addr;
			pci_dma_sync_single_for_cpu(combo6->pci,
						    dma_addr,
						    sizeof(struct pcrdma_dma),
						    PCI_DMA_FROMDEVICE);
		}
		if ((dma->dma_flags & cpu_to_le32(DMA_FLAGS_DONE_MASK)) == 0)
			break;
#ifdef SC6PCR_DEBUG
		printk("RX DMA (%i) finished, flags = 0x%x, seg_stlen = 0x%x\n",
				priv->rx_pos, pkt->dma->dma_flags,
				pkt->seg->seg_stlen);
#endif
		seg_stlen = le32_to_cpu(pkt->seg->seg_stlen);
		seg_len = SEG_STLEN_LENGTH(seg_stlen);
		if ((seg_stlen & SEG_STLEN_FOP_MASK) == 0 ||
		    (seg_stlen & SEG_STLEN_EOP_MASK) == 0 ||
		    (seg_stlen & SEG_STLEN_PORT_MASK) == 0 ||
		    seg_len == 0) {
			pkt->seg->seg_stlen =
				cpu_to_le32(sdev->alloc_block_size);
			goto __rx_next;
		}
		pkt->pkt->desc->timestamp = 0;
		pkt->pkt->desc->hh_size = priv->hh_size;
		pkt->pkt->desc->size = seg_len;
		if (seg_stlen & (3 << SEG_STLEN_PORT_SHIFT)) {
			pkt->pkt->desc->if_index =
				!!(seg_stlen & (2 << SEG_STLEN_PORT_SHIFT));
		} else {
			pkt->pkt->desc->if_index = 2 +
				!!(seg_stlen & (8 << SEG_STLEN_PORT_SHIFT));
		}
		if (!(seg_stlen & SEG_STLEN_ERR_MASK))
			pkt->pkt->desc->status = SZEDATA_BLOCK_STATUS_OK;
		else
			pkt->pkt->desc->status = SZEDATA_BLOCK_STATUS_RXERR;
		pci_dma_sync_single_for_cpu(combo6->pci,
				le32_to_cpu(pkt->seg->seg_addr),
				seg_len, PCI_DMA_FROMDEVICE);
		if (priv->tmp_pages) {
			memcpy(pkt->pkt->virt,
				priv->tmp_pages[priv->rx_pos / priv->tmp_page_incount].virt +
				((priv->rx_pos % priv->tmp_page_incount) *
				 sdev->alloc_block_size),
				sdev->alloc_block_size);
		}
		szedata_device_lock(sdev);
		szedata_device_put_block(sdev, pkt->pkt, ~0);
		my_alloc_pkt(sdev, combo6, pkt, priv->rx_pos);
		szedata_device_unlock(sdev);
__rx_next:
		priv->rx_pos++;
		priv->rx_pos %= SC6PCR_RX_PKTS;
	}
}

static void sc6pcr_regs_read(struct combo6_info_entry *entry,
		struct combo6_info_buffer *buffer)
{
	struct combo6 *combo = entry->private_data;
	struct szedata_device *sdev = combo->private_data;
	struct sc6pcr_priv *priv = sdev->private_data;
	struct sc6pcr_pktdma *pkt;

	combo6_iprintf(buffer, "PCRDMA_CTRL:\t%08x\n",
			combo6_br_readl(combo, PCRDMA_CTRL));
	combo6_iprintf(buffer, "PCRDMA_GPR1:\t%08x\n",
			combo6_br_readl(combo, PCR_GPR(1)));
	combo6_iprintf(buffer, "PCRDMA_RXHEAD:\t%08x [%08x]\n",
			combo6_br_readl(combo, PCRDMA_RXHEAD),
			(u32)priv->rx_pktdma[0].dma_addr);
	combo6_iprintf(buffer, "PCRDMA_TX0HEAD:\t%08x\n",
			combo6_br_readl(combo, PCRDMA_TX0HEAD));
	combo6_iprintf(buffer, "PCRDMA_TX1HEAD:\t%08x\n",
			combo6_br_readl(combo, PCRDMA_TX1HEAD));
	combo6_iprintf(buffer, "PCRDMA_CTRLHEAD:%08x\n",
			combo6_br_readl(combo, PCRDMA_CTRLHEAD));
	combo6_iprintf(buffer, "PCRDMA_DEBUG:\t%08x\n",
			combo6_br_readl(combo, PCRDMA_DEBUG));
	combo6_iprintf(buffer, "PCRDMA_STATE:\t%08x\n",
			combo6_br_readl(combo, PCRDMA_STATE));
	combo6_iprintf(buffer, "\n");
	combo6_iprintf(buffer, "PCR_INTR_STAT:\t%08x\n",
			combo6_br_readl(combo, PCR_INTR_STAT));
	combo6_iprintf(buffer, "PCR_DMA_ASRC:\t%08x\n",
			combo6_br_readl(combo, PCR_DMA_ADDR_SRC));
	combo6_iprintf(buffer, "PCR_DMA_ADST:\t%08x\n",
			combo6_br_readl(combo, PCR_DMA_ADDR_DST));
	combo6_iprintf(buffer, "PCR_DMA_COUNT:\t%08x\n",
			combo6_br_readl(combo, PCR_DMA_COUNT));
	combo6_iprintf(buffer, "PCR_DMA_CMD:\t%08x\n",
			combo6_br_readl(combo, PCR_DMA_CMD));
	combo6_iprintf(buffer, "\n");
	spin_lock_irq(&combo->reg_lock);
	pkt = &priv->rx_pktdma[priv->rx_pos];
	combo6_iprintf(buffer, "rx_pos:\t\t%i [%08x] [%08x]\n", priv->rx_pos,
			pkt->seg->seg_stlen, pkt->dma->dma_flags);
	spin_unlock_irq(&combo->reg_lock);
	combo6_iprintf(buffer, "hh_size:\t%i\n", priv->hh_size);
	combo6_iprintf(buffer, "pkt_skip:\t%i\n", priv->pkt_skip);
	combo6_iprintf(buffer, "tx_hw_pkts:\t%i\n", priv->tx_hw_packets);
#if 0
	combo6_iprintf(buffer, "\n");
	for (i = 0; i < 4; i++) {
		combo6_iprintf(buffer, "rx_bufstat[%i]:\t%08x [%i]\n", i,
				combo6_fpga_readl(combo,
					0x26088 + i * 0x8000),
				combo6_fpga_readl(combo,
					0x26094 + i * 0x8000));
		combo6_iprintf(buffer, "tx_bufstat[%i]:\t%08x [%i]\n", i,
				combo6_fpga_readl(combo,
					0x2608c + i * 0x8000),
				combo6_fpga_readl(combo,
					0x26098 + i * 0x8000));
	}
#endif
}

/*
 *  initialization and detection part
 */

static void pcr_dma_finish(struct sc6pcr_priv *priv)
{
	u32 ctrl;

	if (priv->pcrver < PCRDMA_VERSION(0, 3)) {
		combo6x_dma_finish(priv->combo6);
		return;
	}

	while (1) {
		ctrl = combo6_br_readl(priv->combo6, PCRDMA_CTRL);
		if (!(ctrl & (CTRL_STATUS_RX_RUNNING|CTRL_STATUS_TX_RUNNING)))
			break;
		udelay(1);
	}
}

static void free_segments(struct szedata_device *sdev, struct sc6pcr_priv *priv)
{
	struct sc6pcr_pktdma *pkt;
	struct pcrdma_dma *dma;
	unsigned pos;

	pcr_dma_finish(priv);

	for (pos = 0, dma = NULL; pos < SC6PCR_RX_PKTS; pos++) {
		pkt = &priv->rx_pktdma[pos];
		if (dma != pkt->dma) {
			dma = pkt->dma;
			dma->dma_flags &= ~cpu_to_le32(DMA_FLAGS_DONE_MASK);
		}
		pkt->seg->seg_stlen = pkt->pkt ?
			cpu_to_le32(sdev->alloc_block_size) : 0;
	}
}

static int do_ppc_config(struct sc6pcr_priv *priv)
{
	struct pcrdma_cseg *cseg = priv->cseg_virt;
	u32 timeout = 50;

	cseg->cseg_addr = priv->config_phys;
	cseg->cseg_lbaddr = cpu_to_le32(0xffffff00); /* configuration */
	cseg->cseg_len = cpu_to_le32(sizeof(struct pcrdma_config));
	cseg->cseg_next = 0;
	cseg->cseg_flags = cpu_to_le32(CSEG_FLAGS_LAST_MASK);

	pci_dma_sync_single_for_device(priv->combo6->pci,
				       priv->cseg_phys,
				       sizeof(struct pcrdma_cseg),
				       PCI_DMA_TODEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, CTRL_ENABLE_CTRL);
	combo6_br_readl(priv->combo6, PCRDMA_CTRL);

	while (1) {
		schedule_timeout_uninterruptible(1);
		pci_dma_sync_single_for_cpu(priv->combo6->pci,
					    priv->cseg_phys,
					    sizeof(struct pcrdma_cseg),
					    PCI_DMA_FROMDEVICE);
		if (cseg->cseg_flags & CSEG_FLAGS_DONE_MASK)
			break;
		if (--timeout == 0)
			break;
	}

	pci_dma_sync_single_for_cpu(priv->combo6->pci,
				    priv->config_phys,
				    sizeof(struct pcrdma_config),
				    PCI_DMA_FROMDEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, 0);

	if (timeout == 0) {
		printk(KERN_ERR "sc6pcr: unable to read DMA PPC "
				"configuration\n");
		return -EIO;
	}

	priv->pcrver = le32_to_cpu(priv->config_virt->version);
	if (priv->pcrver < PCRDMA_VERSION(0, 2) &&
	    priv->pcrver > PCRDMA_VERSION(0, 3)) {
		printk(KERN_ERR "sc6pcr: DMA PPC version does not match "
				"(0x%x)\n",
				le32_to_cpu(priv->config_virt->version));
		return -EIO;
	}

	priv->hh_size = le32_to_cpu(priv->config_virt->hh_size);
	priv->pkt_skip = le32_to_cpu(priv->config_virt->pkt_skip);
	priv->tx_hw_packets = le32_to_cpu(priv->config_virt->tx_hw_packets);
	if (priv->tx_hw_packets > 64) {
		printk(KERN_ERR "sc6pcr: TX HW packet count is wrong (%u)?\n",
				priv->tx_hw_packets);
		return -EIO;
	}

#if 0
	printk(KERN_DEBUG "timeout = 0x%x, phys = 0x%x\n", timeout,
			priv->config_phys);
	printk(KERN_DEBUG "time_threshold = 0x%x, res[0] = 0x%x, "
			"res[3] = 0x%x\n",
			priv->config_virt->time_intr_threshold,
			priv->config_virt->reserved[0],
			priv->config_virt->reserved[3]);
	{
	int i;
	for (i = 0; i < 8; i++)
		printk(KERN_DEBUG "GPR(%i) = 0x%x\n", i,
				combo6_br_readl(priv->combo6, PCR_GPR(i)));
	}
#endif

	if (priv->config_virt->time_intr_threshold >= ppc_intr_threshold)
		return 0;

	priv->config_virt->time_intr_threshold = ppc_intr_threshold;

	cseg->cseg_flags =
		cpu_to_le32(CSEG_FLAGS_LAST_MASK|CSEG_FLAGS_WRITE_MASK);

	combo6_br_writel(priv->combo6, PCRDMA_CTRLHEAD, priv->cseg_phys);

	pci_dma_sync_single_for_device(priv->combo6->pci,
				       priv->cseg_phys,
				       sizeof(struct pcrdma_cseg),
				       PCI_DMA_TODEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, CTRL_ENABLE_CTRL);
	combo6_br_readl(priv->combo6, PCRDMA_CTRL);

	while (1) {
		schedule_timeout_uninterruptible(1);
		pci_dma_sync_single_for_cpu(priv->combo6->pci,
					    priv->cseg_phys,
					    sizeof(struct pcrdma_cseg),
					    PCI_DMA_FROMDEVICE);
		if (cseg->cseg_flags & CSEG_FLAGS_DONE_MASK)
			break;
		if (--timeout == 0)
			break;
	}

	pci_dma_sync_single_for_cpu(priv->combo6->pci,
				    priv->config_phys,
				    sizeof(struct pcrdma_config),
				    PCI_DMA_FROMDEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, 0);

	if (timeout == 0) {
		printk(KERN_ERR "c6pcreth: unable to write DMA PPC "
				"configuration\n");
		return -EIO;
	}

	return 0;
}

static void free_tmp_pages(struct sc6pcr_priv *priv)
{
	struct sc6pcr_tmppage *tmp;
	unsigned int idx;

	if (priv->tmp_pages) {
		for (idx = 0; idx < priv->tmp_page_count; idx++) {
			tmp = &priv->tmp_pages[idx];
			dma_free_coherent(&priv->combo6->pci->dev,
						  PAGE_SIZE,
						  tmp->virt,
						  tmp->phys);
		}
		vfree(priv->tmp_pages);
		priv->tmp_pages = NULL;
		priv->tmp_page_count = 0;
	}
}

static void alloc_tmp_pages(struct sc6pcr_priv *priv)
{
	unsigned int count = (PAGE_SIZE / priv->hh_size / PCRDMA_NSEGS) *
		PCRDMA_NSEGS;
	unsigned int page_count, idx;
	struct sc6pcr_tmppage *pages, *tmp;

	page_count = (SC6PCR_RX_PKTS + count - 1) / count;
	pages = vmalloc(sizeof(struct sc6pcr_tmppage) * page_count);
	if (pages == NULL)
		goto __error;
	memset(pages, 0, sizeof(struct sc6pcr_tmppage) * page_count);
	priv->tmp_pages = pages;
	priv->tmp_page_incount = count;
	priv->tmp_page_count = page_count;
	for (idx = 0; idx < page_count; idx++) {
		tmp = &pages[idx];
		tmp->virt = dma_alloc_coherent(&priv->combo6->pci->dev,
				PAGE_SIZE, &tmp->phys, GFP_KERNEL);
		if (tmp->virt == NULL)
			goto __error;
	}
	return;

__error:
	free_tmp_pages(priv);
	printk(KERN_INFO "unable to allocate temporary PCI pages, driver is "
			"not working optimaly\n");
}


static int sc6pcr_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	struct szedata_device *sdev;
	struct sc6pcr_priv *priv;
	struct pcrdma_dma *dma;
	struct sc6pcr_pktdma *pkt;
	dma_addr_t dma_addr, dma_faddr;
	unsigned int pos, i, j, psize = 0, nblocks = 0;
	struct combo6_info_entry *entry;
	int err;
	u32 brver = combo->pcibr_version;

	if (interfaces < 1 || interfaces > PCRDMA_NPORTS)
		return -EIO;

	if (PCR_ID_MAJOR(brver) < 4 ||
	    (PCR_ID_MAJOR(brver) == 4 &&
	     PCR_ID_MINOR(brver) < 1)) {
		printk(KERN_ERR "c6pcreth: Old version of PCI bridge, "
				"aborting\n");
		return -EIO;
	}

	/* disable interrupts */
	combo6_br_writel(combo, PCR_INTR_MASK, 0);

	/* check if PPC has good firmware (program) */
	if (combo6_br_readl(combo, PCRDMA_DEBUG) != 0x4f6f4b6b) {
		udelay(500);
		if (combo6_br_readl(combo, PCRDMA_DEBUG) != 0x4f6f4b6b) {
			printk(KERN_ERR "c6pcreth: DMA PPC firmware signature "
				"is wrong: 0x%x\n",
				combo6_br_readl(combo, PCRDMA_DEBUG));
			return -EIO;
		}
	}

	if (block_size[combo->index] == 0) {	/* default */
		if (id->driver_data & PCRDMA_LIMIT_BLOCKS)
			psize = 64;
	} else {
		psize = block_size[combo->index];
	}
	if (blocks[combo->index] == 0) {	/* default */
		if (id->driver_data & PCRDMA_LIMIT_BLOCKS)
			nblocks = 65536;
	} else {
		nblocks = blocks[combo->index];
	}
	/* hack - fixme? */
	if (memcmp(combo->u.two.pcippc.i_text,
			"Netflow Network Monitoring (group=16)", 37) == 0)
		psize = 16 * 64;
	if (psize == 0)
		psize = 1516 + 32;
	if (nblocks == 0)
		nblocks = 16384;

	err = szedata_device_alloc(&sdev, interfaces,
				  nblocks,
				  psize,
				  sizeof(struct sc6pcr_priv));
	if (err < 0) {
		printk(KERN_ERR "unable to allocate new szedata driver for "
				"combo%d\n", combo->index);
		return err;
	}
	err = szedata_device_alloc_pci(sdev, combo->pci, nblocks, psize);
	if (err < 0) {
		printk(KERN_ERR "unable to allocate PCI space for driver "
				"combo%d\n", combo->index);
		goto free_dev;
	}
	sdev->open = sdev_open;
	sdev->close = sdev_close;
	sdev->start = sdev_start;
	sdev->stop = sdev_stop;
	sdev->port_info = sdev_port_info;
	priv = sdev->private_data;
	priv->combo6 = combo;
	priv->interfaces = interfaces;
	priv->sgdesc_size = sizeof(struct pcrdma_dma) * SC6PCR_RX_BUFFERS;
	priv->sgdesc_virt = dma_alloc_coherent(&combo->pci->dev,
			priv->sgdesc_size + sizeof(struct pcrdma_cseg) +
			sizeof(struct pcrdma_config), &priv->sgdesc_phys,
			GFP_KERNEL);
	if (priv->sgdesc_virt == NULL) {
		printk(KERN_ERR "unable to allocate memory for combo%d (%li "
				"bytes)\n", combo->index,
				priv->sgdesc_size + sizeof(struct pcrdma_cseg) +
					sizeof(struct pcrdma_config));
		err = -ENOMEM;
		goto free_pci;
	}
	priv->cseg_virt = (struct pcrdma_cseg *)(((char *)priv->sgdesc_virt) +
			priv->sgdesc_size);
	priv->cseg_phys = priv->sgdesc_phys + priv->sgdesc_size;
	priv->config_virt = (struct pcrdma_config *)(((char *)priv->cseg_virt) +
			sizeof(struct pcrdma_cseg));
	priv->config_phys = priv->cseg_phys + sizeof(struct pcrdma_cseg);
	combo6_br_writel(priv->combo6, PCRDMA_CTRLHEAD, priv->cseg_phys);
	err = do_ppc_config(priv);
	if (err >= 0)
		err = szedata_device_register(sdev, THIS_MODULE);
	if (err < 0) {
		printk(KERN_ERR "unable to register new szedata driver for "
				"combo%d\n", combo->index);
		goto free_sgdesc;
	}
	combo->private_data = sdev;

	if (alloc_tpages && psize < 512 && psize == priv->hh_size)
		alloc_tmp_pages(priv);

	dma = (struct pcrdma_dma *)priv->sgdesc_virt;
	dma_addr = dma_faddr = priv->sgdesc_phys;
	pos = 0;
	for (i = 0; i < SC6PCR_RX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			pkt = &priv->rx_pktdma[pos];
			pkt->seg = &dma->dma_segs[j];
			pkt->dma = dma;
			pkt->dma_addr = dma_addr;
			my_alloc_pkt(sdev, priv->combo6, pkt,
					i * PCRDMA_NSEGS + j);
			pos++;
		}
		dma->dma_next = cpu_to_le32(dma_faddr +
					    ((i + 1) % SC6PCR_RX_BUFFERS *
					      sizeof(struct pcrdma_dma)));
		dma->dma_flags = cpu_to_le32((i % intr_step) == 0 ?
				SC6PCR_DMA_RX_FLAGS1 : SC6PCR_DMA_RX_FLAGS0);
		dma++;
		dma_addr += sizeof(struct pcrdma_dma);
	}

	/* enable interrupts */
	combo6_br_writel(priv->combo6, PCR_INTR_SET, PCRDMA_INTR_RX_MASK |
			PCRDMA_INTR_ERR_MASK);

	entry = combo6_info_create_module_entry(THIS_MODULE, "sc6pcr-regs",
			combo->proc_root);
	if (entry) {
		combo6_info_set_text_ops(entry, combo, 4096, sc6pcr_regs_read);
		if (combo6_info_register(entry) < 0) {
			combo6_info_free_entry(entry);
			entry = NULL;
		}
	}
	priv->proc_regs = entry;
	return 0;
free_sgdesc:
	dma_free_coherent(&combo->pci->dev,
			priv->sgdesc_size + sizeof(struct pcrdma_cseg) +
			sizeof(struct pcrdma_config),
			priv->sgdesc_virt, priv->sgdesc_phys);
free_pci:
	szedata_device_free_pci(sdev, combo->pci);
free_dev:
	szedata_device_free(sdev);
	return err;
}

static int sc6pcr_detach(struct combo6 *combo)
{
	struct szedata_device *sdev = combo->private_data;
	struct sc6pcr_priv *priv = sdev->private_data;
	struct sc6pcr_pktdma *pkt;
	unsigned pos;

	pcr_dma_finish(priv);
	combo6x_stop_pcippc(combo);
	for (pos = 0; pos < SC6PCR_RX_PKTS; pos++) {
		pkt = &priv->rx_pktdma[pos];
		if (pkt->pkt) {
			szedata_device_lock_irq(sdev);
			szedata_device_detach_block(sdev, pkt->pkt);
			szedata_device_unlock_irq(sdev);
		}
	}
	free_tmp_pages(priv);
	dma_free_coherent(&combo->pci->dev,
			priv->sgdesc_size + sizeof(struct pcrdma_cseg) +
			sizeof(struct pcrdma_config),
			priv->sgdesc_virt, priv->sgdesc_phys);
	if (priv->proc_regs) {
		combo6_info_unregister(priv->proc_regs);
		priv->proc_regs = NULL;
	}
	szedata_device_free_pci(sdev, combo->pci);
	szedata_device_unregister(sdev);
	combo->private_data = NULL;
	return 0;
}

/*
 *  szedata midlevel layer
 */

static int sdev_open(struct szedata_device *sdev)
{
	return 0;
}

static int sdev_close(struct szedata_device *sdev)
{
	return 0;
}

static int sdev_start(struct szedata_device *sdev)
{
	struct sc6pcr_priv *priv = sdev->private_data;
	struct combo6 *combo = priv->combo6;

	if (combo6_device_get(combo))
		return -EBUSY;

	spin_lock(&combo->reg_lock);
	restart(combo);
	spin_unlock(&combo->reg_lock);

	return 0;
}

static int sdev_stop(struct szedata_device *sdev)
{
	struct sc6pcr_priv *priv = sdev->private_data;
	struct combo6 *combo = priv->combo6;

	spin_lock(&combo->reg_lock);
	combo6_br_writel(priv->combo6, PCRDMA_CLR_CTRL, CTRL_ENABLE_RX |
			CTRL_ENABLE_MASK);
	free_segments(sdev, priv);
	spin_unlock(&combo->reg_lock);

	combo6_device_put(combo);

	return 0;
}

static int sdev_port_info(struct szedata_device *sdev,
		struct szedata_ioctl_get_interface *info)
{
	struct sc6pcr_priv *priv = sdev->private_data;
	struct combo6 *combo = priv->combo6;

	if (info->if_index >= priv->interfaces)
		return -EINVAL;
	sprintf(info->if_name, "combo6x card %d port %d", combo->index,
			info->if_index);
	return 0;
}

/*
 * real initialization
 */

#define pcrdma_attach sc6pcr_attach
#define pcrdma_detach sc6pcr_detach
#define pcrdma_interrupt sc6pcr_interrupt
#include "pcrdma_ops.c"

static int __init sc6pcr_init(void)
{
	combo6x_module_ref();

	return combo_register_driver(&pcrdma_ops);
}

static void __exit sc6pcr_exit(void)
{
	combo_unregister_driver(&pcrdma_ops);
}

module_init(sc6pcr_init)
module_exit(sc6pcr_exit)
