/*
 *  combo6pcrdma.h: Register / structures definitions for embedded PPC on Virtex
 *  Copyright (c) 2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __COMBO6_PCRDMA_H
#define __COMBO6_PCRDMA_H

#include <linux/types.h>

/*
 * PCR bridge registers
 */

#define PCRDMA_VERSION(major, minor) \
				(0xcece0000 | ((major) << 8) | (minor))

/* OCM registers. */
#define PCRDMA_CTRL	     PCR_GPR(0)
#define PCRDMA_SET_CTRL		PCR_SET_GPR(0)
#define PCRDMA_CLR_CTRL	 PCR_CLR_GPR(0)
#define PCRDMA_RXHEAD	   PCR_GPR(2)
#define PCRDMA_TX0HEAD	  PCR_GPR(3)
#define PCRDMA_TX1HEAD	  PCR_GPR(4)
#define PCRDMA_CTRLHEAD	 PCR_GPR(5)
#define PCRDMA_DEBUG	    PCR_GPR(6)
#define PCRDMA_STATE	    PCR_GPR(7)

/* Interrupt bit definitions. */
#define PCRDMA_INTR_RX_MASK	INTR_USER(0)
#define PCRDMA_INTR_TX_MASK	INTR_USER(1)
#define PCRDMA_INTR_CTRL_MASK   INTR_USER(2)
#define PCRDMA_INTR_ERR_MASK	INTR_USER(3)

/* PCRDMA_CTRL bits */
#define CTRL_ENABLE_MASK	0x0000000f
#define CTRL_ENABLE(n)	  (1 << (n))      /* enable specific port */
#define CTRL_ENABLE_RX	  0x00000010
#define CTRL_ENABLE_TX0		0x00000020
#define CTRL_ENABLE_TX1		0x00000040
#define CTRL_ENABLE_TX		0x00000060
#define CTRL_ENABLE_CTRL	0x00000080
#define CTRL_ENABLE_ALL	 0x000000f0
#define CTRL_SET_TX0		0x00000100
#define CTRL_SET_TX1		0x00000200
#define CTRL_STATUS_RX_MASK	0x000f0000	/* from 0.3 */
#define CTRL_STATUS_RX_RUNNING	0x00010000	/* RX engine is running in HW */
#define CTRL_STATUS_TX_MASK	0x00f00000	/* from 0.3 */
#define CTRL_STATUS_TX_RUNNING	0x00100000	/* TX engine is running in HW */

/*
 * DMA stuff
 */

/*
 * Some thoughs about PCRDMA_NSEGS:
 *   we need to support the jumbo frames bytes in future
 *   7 * 1536 (standard space for ethernet packet in kernel) = 10752
 */
#define PCRDMA_NSEGS	    7       /* Arbitrary, limited mem for ppc */
#define PCRDMA_NPORTS	   4       /* Hardware dependent, really */

#define SEG_STLEN_LENGTH(val)   (val&0x000fffff)
#define SEG_STLEN_PORT_MASK     0x00f00000  /* port mask */
#define SEG_STLEN_PORT_SHIFT    20
#define SEG_STLEN_CRCERR_MASK	0x01000000  /* RX: packet has wrong CRC */
#define SEG_STLEN_ERR_MASK      0x02000000  /* RX: there are errors in packet */
#define SEG_STLEN_OVER_MASK     0x04000000  /* RX: packet cannot be placed to segments - no enough space */
#define SEG_STLEN_FOP_MASK	0x40000000  /* First-Of-Packet - this segment is first for packet */
#define SEG_STLEN_EOP_MASK      0x80000000  /* End-Of-Packet - this segment is last for packet */

struct pcrdma_seg {
	u32 seg_addr;		/* 32-bit physical PCI address */
	u32 seg_stlen;		/* status + lenght - see SEG_STLEN_* */
} __attribute__((packed));

#define DMA_FLAGS_INTR_MASK     0x20000000  /* interrupt the host */
#define DMA_FLAGS_LAST_MASK     0x40000000  /* last segment (host) */
#define DMA_FLAGS_DONE_MASK     0x80000000  /* PPC finished work with these DMA segments */

struct pcrdma_dma {
	struct pcrdma_seg dma_segs[PCRDMA_NSEGS];
	u32 dma_next;		/* 32-bit physical PCI address of next DMA */
	u32 dma_flags;		/* dma flags */
} __attribute__((packed));

#define CSEG_FLAGS_WRITE_MASK   0x00000001  /* write data from PCI to local bus, otherwise read */
#define CSEG_FLAGS_INTR_MASK    0x20000000  /* raise interrupt when finished */
#define CSEG_FLAGS_LAST_MASK    0x40000000  /* last segment (host) */
#define CSEG_FLAGS_DONE_MASK    0x80000000  /* PPC finished work with this segment */

struct pcrdma_cseg {
	u32 cseg_addr;		/* 32-bit physical PCI address */
	u32 cseg_lbaddr;	/* 32-bit local bus address */
	u32 cseg_len;		/* 32-bit segment length */
	u32 cseg_next;		/* 32-bit physical PCI address of next CSEG */
	u32 cseg_flags;		/* flags - see CSEG_FLAGS_* */
} __attribute__((packed));

struct pcrdma_config {
	u32	version;
	u32	hh_size;	/* hardware header size in bytes */
	u32	pkt_skip;	/* skip these first bytes from packet */
	u32	tx_hw_packets;	/* TX packets in hardware - use for scheduling */
	u32	time_intr_threshold;
	u32	reserved[27];
	struct {
		u32 packets;
		u32 size;
		u32 overrun;
		u32 crcerr;
		u32 errors;
		u32 res[3];
	} __attribute__((packed)) stat[PCRDMA_NPORTS];
} __attribute__((packed));

#endif /* __COMBO6_PCRDMA_H */
