/*
 *  ccombo6x.c: driver for second combo6 hardware with PCI IP core
 *  Copyright (c) 2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <linux/io.h>
#include <linux/kdev_t.h>
#include <linux/delay.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/types.h>
#include <linux/uaccess.h>

#include "combo6.h"
#include "combo6k.h"
#include "combo6x.h"
#include "combo6cpld.h"
#include "combo6pcr.h"

/*
 *  PCI initialization and detection part
 */

static int enable[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 };      /* Enable this card */
static int invalidmemsize[COMBO6_CARDS]; /* Invalid memory region, fix to 64MB */
static int bootprotect[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 }; /* Check design boot */
static int bootdelay[COMBO6_CARDS]; /* Boot delay in ms before fpga access */
static int usemode[COMBO6_CARDS]; /* Use mode (see README) */

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("Combo6 with PCI IP core Linux Driver");
MODULE_LICENSE("GPL");

module_param_array(enable, int, NULL, 0);
MODULE_PARM_DESC(enable, "Enable Combo6x card.");
module_param_array(invalidmemsize, int, NULL, 0);
MODULE_PARM_DESC(invalidmemsize, "Invalid PCI memory area size - fix to 64MB.");
module_param_array(bootprotect, int, NULL, 0);
MODULE_PARM_DESC(bootprotect, "Enable(default)=1/Disable=0 boot protection "
		"checks.");
module_param_array(bootdelay, int, NULL, 0);
MODULE_PARM_DESC(bootdelay, "Boot delay in ms before FPGA access.");
module_param_array(usemode, int, NULL, 0);
MODULE_PARM_DESC(usemode, "Use mode (for debugging only). See README.");

static irqreturn_t combo6x_interrupt(int irq, void *dev_id);
static int combo6x_ioctl(struct combo6 *combo6,
			 struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg);
static void combo6x_read_id(struct combo6 *combo6);

static int combo6x_boot_brppc(struct combo6 *combo6, u32 *data, u32 count)
{
	u32 base, size;

	if (count != PCR_PPC_ISIZE + PCR_PPC_DSIZE + PCR_PPC_HSIZE)
		return -EINVAL;

	memcpy(&combo6->u.two.pcippc, data, sizeof(struct combo6_eppc_id));
	data += PCR_PPC_HSIZE / 4;

	/* stop PPC CPU */
	combo6_br_writel(combo6, PCR_PPC_CTRL, PPC_CTRL_RESET);
	combo6->u.two.pcippc_run = 0;
	/* load program and data */
	size = PCR_PPC_ISIZE;
	base = PCR_PPC_IBASE;
	while (size > 0) {
		combo6_br_writel(combo6, base, *data++);
		base += 4;
		size -= 4;
	}
	size = PCR_PPC_DSIZE;
	base = PCR_PPC_DBASE;
	while (size > 0) {
		combo6_br_writel(combo6, base, *data++);
		base += 4;
		size -= 4;
	}
	combo6->u.two.pcippc_ok = 1;
	return 0;
}

static int combo6x_boot_fpga(struct combo6 *combo6, u32 unit, u8 *data,
		u32 count)
{
	u32 cfg = 0, i, cpld_conf, cpld_data, tmp;
	long timeout;

	if (unit == 1024)
		return combo6x_boot_brppc(combo6, (u32 *)data, count);

	if (unit != 0 && unit != 8 && unit != 9)
		return -EINVAL;

	cpld_conf = COMBO6_CPLD_BASE(unit) + COMBO6_CPLD_CONF;
	cpld_data = COMBO6_CPLD_BASE(unit) + COMBO6_CPLD_DATA;

	cfg = combo6_cpld_readl(combo6, cpld_conf) | CONF_MSPROG;

	/*
	 * CPLD is on the bus. Go on and generate FSPROG pulse.
	 */
	combo6_cpld_writel(combo6, cpld_conf, cfg | CONF_FSPROG);
	udelay(1);
	combo6_cpld_writel(combo6, cpld_conf, cfg & ~CONF_FSPROG);
	udelay(1);
	combo6_cpld_writel(combo6, cpld_conf, cfg | CONF_FSPROG);
	udelay(5);

	/*
	 * Wait until FPGA is ready for programming.
	 */
	timeout = 100000;
	do {
		udelay(1);
		cfg = combo6_cpld_readl(combo6, cpld_conf);
		if (timeout-- == 0) {
			printk(KERN_ERR "combo6x(%i): fpga fsinit error\n",
					unit);
			return -EIO;
		}
	} while (!(cfg & CONF_FSINIT));

	for (i = tmp = 0; i < count; i++) {
		combo6_cpld_writel(combo6, cpld_data, *data++);
		if (tmp++ > 0x2000) {
			tmp = 0;
			cond_resched();
		}
	}
	udelay(5);

	/*
	 * Wait until programming is complete
	 */
	timeout = 100000;
	do {
		udelay(1);
		cfg = combo6_cpld_readl(combo6, cpld_conf);
		if (timeout-- == 0) {
			printk(KERN_ERR "combo6x(%i): fpga fsdone error\n",
					unit);
			return -EIO;
		}
	} while (!(cfg & CONF_FSDONE));

	if (unit == 0)
		set_bit(COMBO_BOOT_OK, combo6->flags);

	return 0;
}

static void combo6x_go_attach(struct combo6 *combo6)
{
	spin_lock_irq(&combo6->reg_lock);
	/* start PPC core */
	if (combo6->u.two.pcippc_ok) {
		combo6_br_writel(combo6, PCR_PPC_CTRL, PPC_CTRL_CLOCK |
				PPC_CTRL_RESET);
		udelay(100);
		combo6_br_writel(combo6, PCR_PPC_CTRL, PPC_CTRL_CLOCK);
		combo6->u.two.pcippc_run = 1;
	}
	spin_unlock_irq(&combo6->reg_lock);
}

static void combo6x_go_detach(struct combo6 *combo6)
{
	spin_lock_irq(&combo6->reg_lock);
	combo6_br_writel(combo6, PCR_INTR_MASK, 0);
	combo6_br_writel(combo6, PCR_INTR_USER_ACK, INTR_USER_MASK);
	spin_unlock_irq(&combo6->reg_lock);
}

static void combo6x_detached(struct combo6 *combo6)
{
	spin_lock_irq(&combo6->reg_lock);
	/* stop PPC core */
	combo6x_stop_pcippc(combo6);
	spin_unlock_irq(&combo6->reg_lock);
}

static const char *combo6x_get_info(struct combo6 *combo6, char *buf,
		enum combo6_info_type info, unsigned long data)
{
	static const char *addon_cards[] = { "unknown", "sfp", "mtx2", "xfp",
		"sfpro", "xfpro", "xfp2.0", "xfp2", "xfp2.2" };
	static const char *addon_chips[] = { "unknown", "xcv1000", "xcv2000",
		"xc2vp20", "xc2vpx20", "xc2vp30" };

	if (data && (info == COMBO6_INFO_ADDON_CARD ||
				info == COMBO6_INFO_ADDON_CHIP))
		return NULL;

	switch (info) {
	case COMBO6_INFO_TYPE:
		return combo6->is_6e ? "combo6e" : "combo6x";
	case COMBO6_INFO_ADDON_CARD:
		return addon_cards[combo6->addon_card[0]];
	case COMBO6_INFO_ADDON_CHIP:
		return addon_chips[combo6->addon_chip[0]];
	case COMBO6_INFO_PORTS:
		sprintf(buf, "%d", combo6->addon_interfaces);
		return buf;
	default:
		return NULL;
	}
}

static void combo6x_proc_info(struct combo6 *combo6,
		struct combo6_info_buffer *buffer)
{
	const char *str = "";
	u32 ver, date;

	ver = combo6->pcibr_version;
	date = combo6->pcibr_date;
	if (PCR_ID_IDENT(ver) == 0xc610 && PCR_ID_MAJOR(ver) >= 5)
		str = " NetCOPE Bridge";
	combo6_iprintf(buffer, "PCI brver: %04x.%02x.%02x "
			"(%04i/%02i/%02i %02i:%02i)%s\n",
			PCR_ID_IDENT(ver),
			PCR_ID_MAJOR(ver),
			PCR_ID_MINOR(ver),
			BIRTH_YEAR(date) + 2000,
			BIRTH_MONTH(date),
			BIRTH_DAY(date),
			BIRTH_HOUR(date),
			BIRTH_MINUTE(date),
			str);
	combo6_iprintf(buffer, "PCI PPC  : [%s] 0x%x -%s- -%s-\n",
			combo6->u.two.pcippc.i_text,
			combo6->u.two.pcippc.i_version,
			combo6->u.two.pcippc_ok ? "booted" : "not loaded",
			combo6->u.two.pcippc_run ? "running" : "inactive");
}

static int __devinit combo6x_probe(struct pci_dev *pci,
			const struct pci_device_id *id)
{
	static const struct combo6_ops ops = {
		.ioctl = combo6x_ioctl,
		.go_attach = combo6x_go_attach,
		.go_detach = combo6x_go_detach,
		.detached = combo6x_detached,
		.get_info = combo6x_get_info,
		.proc_info = combo6x_proc_info,
	};
	static int dev;
	struct combo6 *combo6;
	u32 conf, tmp32;
	int ret;

	if (!enable[dev]) {
		dev++;
		return -ENOENT;
	}
	ret = pci_enable_device(pci);
	if (ret) {
		dev_err(&pci->dev, "can't enable pci device\n");
		goto err;
	}
	combo6 = combo6_alloc(0);
	if (combo6 == NULL) {
		dev_err(&pci->dev, "can't allocate combo6\n");
		ret = -ENOMEM;
		goto err_dis;
	}
	combo6->pci = pci;
	combo6->ops = &ops;
	combo6->usemode = usemode[dev];
	if (bootprotect[dev])
		set_bit(COMBO_BOOTPROTECT, combo6->flags);
	combo6->bootdelay = bootdelay[dev];

	combo6->is_6x = 1;
	combo6->is_6e = id->driver_data;

	ret = pci_set_dma_mask(pci, DMA_BIT_MASK(32));
	if (ret)
		goto err_free;
	pci_set_master(pci);

	ret = -ENOMEM;
	combo6->mem_phys = pci_resource_start(pci, 0);
	combo6->mem_len = pci_resource_len(pci, 0);
	if (!request_mem_region(combo6->mem_phys, combo6->mem_len,
			"Combo6x PLX")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->mem_phys,
			(u64)(combo6->mem_phys + combo6->mem_len - 1));
		goto err_free;
	}
	combo6->mem_virt = ioremap_nocache(combo6->mem_phys, combo6->mem_len);
	if (combo6->mem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->mem_phys,
			(u64)(combo6->mem_phys + combo6->mem_len - 1));
		release_mem_region(combo6->mem_phys, combo6->mem_len);
		goto err_free;
	}
	if (combo6->usemode & USEMODE_SKIP_BAR12)
		goto __skip_bar12;
	combo6->lmem_phys = pci_resource_start(pci, 1);
	combo6->lmem_len = pci_resource_len(pci, 1);
	combo6->cmem_phys = pci_resource_start(pci, 2);
	combo6->cmem_len = pci_resource_len(pci, 2);
	if (invalidmemsize[dev]) {
		if (combo6->mem_len > 64 * 1024 * 1024)
			combo6->mem_len = 64 * 1024 * 1024;
		if (combo6->lmem_len > 64 * 1024 * 1024)
			combo6->lmem_len = 64 * 1024 * 1024;
		if (combo6->cmem_len > 64 * 1024 * 1024)
			combo6->cmem_len = 64 * 1024 * 1024;
	}
	if (!request_mem_region(combo6->lmem_phys,
			combo6->lmem_len, "Combo6x LB")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		goto err_free;
	}
	combo6->lmem_virt = ioremap_nocache(combo6->lmem_phys,
			combo6->lmem_len);
	if (combo6->lmem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		release_mem_region(combo6->lmem_phys, combo6->lmem_len);
		goto err_free;
	}
	if (!request_mem_region(combo6->cmem_phys, combo6->cmem_len,
				"Combo6x CPLD")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->cmem_phys,
			(u64)(combo6->cmem_phys + combo6->cmem_len - 1));
		goto err_free;
	}
	combo6->cmem_virt = ioremap_nocache(combo6->cmem_phys,
			combo6->cmem_len);
	if (combo6->cmem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->cmem_phys,
			(u64)(combo6->cmem_phys + combo6->cmem_len - 1));
		release_mem_region(combo6->cmem_phys, combo6->cmem_len);
		goto err_free;
	}
__skip_bar12:
	ret = request_irq(pci->irq, combo6x_interrupt, IRQF_SHARED, "Combo6x",
			combo6);
	if (ret) {
		dev_err(&pci->dev, "unable to allocate interrupt %d\n",
				pci->irq);
		goto err_free;
	}
	combo6->irq = pci->irq;

	pci_set_drvdata(pci, combo6);

	if (combo6->usemode & USEMODE_SKIP_BRVER)
		goto __skip_brver;
	combo6->pcibr_version = combo6_br_readl(combo6, PCR_IDENT);
	if (PCR_ID_IDENT(combo6->pcibr_version) != 0xc610) {
		dev_err(&pci->dev, "bad PCI bridge version: 0x%x\n",
				combo6->pcibr_version);
		ret = -EINVAL;
		goto err_free;
	}
	combo6->pcibr_date = combo6_br_readl(combo6, PCR_BIRTH);
__skip_brver:

	/*
	 * Check for presence of CPLD card detection firmware by trying
	 * negation register at FPGA8.CONF and, on success, read and decode
	 * the identification from FPGA9.CONF.
	 */

	combo6->addon_chip[0] = COMBO6_ADDON_CHIP_UNK;
	combo6->addon_card[0] = COMBO6_ADDON_CARD_UNK;
	combo6->addon_interfaces = -1;
	if (combo6->usemode & (USEMODE_SKIP_BAR12|USEMODE_SKIP_CPLD))
		goto __skip_cpld;
	combo6_cpld_writel(combo6, COMBO6_CPLD_BASE(8) + COMBO6_CPLD_CONF,
			ADDON_PATTERN);
	conf = combo6_cpld_readl(combo6, COMBO6_CPLD_BASE(8) +
			COMBO6_CPLD_CONF);
	if ((conf & ADDON_MASK) == (~ADDON_PATTERN & ADDON_MASK)) {
		conf = combo6_cpld_readl(combo6, COMBO6_CPLD_BASE(9) +
				COMBO6_CPLD_CONF);
		tmp32 = COMBO6_ADDON_CARD(conf);
		combo6->addon_card[0] = COMBO6_ADDON_CARD_GET(tmp32,
				COMBO6_ADDON_CHIP(conf));
		tmp32 = COMBO6_ADDON_CHIP(conf);
		combo6->addon_chip[0] =
			COMBO6_ADDON_CHIP_GET(combo6->addon_card[0], tmp32);
		combo6->addon_interfaces =
			COMBO6_ADDON_INTERFACES_GET(combo6->addon_card[0],
					combo6->addon_chip[0]);
	}
__skip_cpld:
	ret = combo6_add(combo6);
	if (ret) {
		dev_err(&pci->dev, "too many cards in system\n");
		goto err_free;
	}
	return 0;
err_free:
	combo6_free(combo6);
err_dis:
	pci_disable_device(pci);
err:
	return ret;
}

static void __devexit combo6x_remove(struct pci_dev *pci)
{
	struct combo6 *combo6 = pci_get_drvdata(pci);

	set_bit(COMBO_GOING_DOWN, combo6->flags);
	combo6_remove(combo6);
	combo6_ioctl_bus_detach(combo6);
	combo6_free(combo6);
	pci_set_drvdata(pci, NULL);
}

static struct pci_device_id combo6x_ids[] __devinitdata = {
	{ PCI_DEVICE(PCI_VENDOR_ID_CESNET, 0xc045), .driver_data = 1 }, /* Combo6E */
	{ PCI_DEVICE(PCI_VENDOR_ID_CESNET, 0xc058) }, /* Combo6X */
	{ 0, }
};
MODULE_DEVICE_TABLE(pci, combo6x_ids);

static struct pci_driver driver = {
	.name = "Combo6X",
	.id_table = combo6x_ids,
	.probe = combo6x_probe,
	.remove = __devexit_p(combo6x_remove),
};

static int combo6x_ioctl_read(struct combo6 *combo6,
		struct combo6_ioc_arg __user *_r)
{
	struct combo6_ioc_arg r;

	if (copy_from_user(&r, _r, sizeof(r)))
		return -EFAULT;
	switch (r.target) {
	case COMBO6_CPLD_BUS:
		if (r.offset >= combo6->cmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->cmem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	case COMBO6_BRIDGE_REGS:
		if (r.offset >= combo6->mem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->mem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	case COMBO6_FPGA_BUS:
		if (!test_bit(COMBO_BOOT_OK, combo6->flags))
			return -EBADFD;
		if (r.offset >= combo6->lmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->lmem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int combo6x_ioctl_write(struct combo6 *combo6,
		struct combo6_ioc_arg __user *_r)
{
	struct combo6_ioc_arg r;
	u32 val;

	if (copy_from_user(&r, _r, sizeof(r)))
		return -EFAULT;
	switch (r.target) {
	case COMBO6_CPLD_BUS:
		if (r.offset >= combo6->cmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->cmem_virt + r.offset);
		break;
	case COMBO6_BRIDGE_REGS:
		if (r.offset >= combo6->mem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->mem_virt + r.offset);
		break;
	case COMBO6_FPGA_BUS:
		if (r.offset >= combo6->lmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->lmem_virt + r.offset);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/*
 *  combo6 device management
 */

#define ID32_TO_STR(strptr, val32) do { \
	*((strptr)+1) = (val32) & 0xff; \
	*((strptr)+0) = ((val32) >> 8) & 0xff; \
	*((strptr)+3) = ((val32) >> 16) & 0xff; \
	*((strptr)+2) = ((val32) >> 24) & 0xff; \
} while (0)

static void combo6x_read_id(struct combo6 *combo6)
{
	u32 tmp;
	int i;

	if (combo6->usemode & USEMODE_SKIP_LOCALBUS)
		return;
	if (!test_bit(COMBO_BOOT_OK, combo6->flags))
		return;
	if (combo6->bootdelay > 0 && combo6->bootdelay <= 100)
		msleep(combo6->bootdelay);
	combo6->id_sw = combo6_fpga_readl(combo6, 0);
	combo6_fpga_writel(combo6, 0, combo6->id_sw);
	if (combo6_fpga_readl(combo6, 0) == ~combo6->id_sw) {
		combo6->id_sw = combo6_fpga_readl(combo6, 4);
		combo6->id_hw = combo6_fpga_readl(combo6, 8);
		for (i = 0; i < 32; i += 4) {
			tmp = combo6_fpga_readl(combo6, 32 + i);
			ID32_TO_STR(combo6->id_txt + i, tmp);
		}
		combo6->id_txt[32] = '\0';
	} else {
		combo6->id_hw = 0;
		combo6->id_txt[0] = '\0';
	}
	if ((combo6->id_sw == 0 || combo6->id_sw == ~0) &&
			test_bit(COMBO_BOOTPROTECT, combo6->flags))
		clear_bit(COMBO_BOOT_OK, combo6->flags);
}

static int combo6x_ioctl_boot(struct combo6 *combo6,
		struct combo6_boot __user *_boot)
{
	struct combo6_boot boot;
	int i, res = 0;
	u8 *data;
	u32 size;

	if (combo6->usemode & USEMODE_SKIP_BAR12)
		return -ENXIO;
	if (copy_from_user(&boot, _boot, sizeof(boot)))
		return -EFAULT;
	mutex_lock(&combo6->drv_mutex);
	if (combo6->driver) {
		mutex_unlock(&combo6->drv_mutex);
		return -EBUSY;
	}
	set_bit(COMBO_BOOTING, combo6->flags);
	mutex_unlock(&combo6->drv_mutex);
	for (i = 0; i < COMBO6_BOOT_IMAGES; i++) {
		size = boot.d[i].size;
		if (size == 0) {
			if (i == 0)
				res = -EINVAL;
			break;
		}
		data = vmalloc(size);
		if (data == NULL) {
			res = -ENOMEM;
			break;
		}
		if (copy_from_user(data, (void __user __force *)boot.d[i].data,
					size)) {
			vfree(data);
			res = -EFAULT;
			break;
		}
		res = combo6x_boot_fpga(combo6, boot.d[i].unit, data, size);
		vfree(data);
		if (res < 0)
			break;
	}
	for (i = 0; i < COMBO6_BOOT_IMAGES; i++) {
		if (boot.d[i].unit == 0) {
			if (res >= 0)
				combo6x_read_id(combo6);
			else
				clear_bit(COMBO_BOOT_OK, combo6->flags);
			break;
		}
	}
	clear_bit(COMBO_BOOTING, combo6->flags);
	return res;
}

/*
 *  ioctl handler
 */

static int combo6x_ioctl(struct combo6 *combo6,
			 struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;

	pr_debug("combo6x_ioctl: 0x%x\n", cmd);
	switch (cmd) {
	case COMBO6_READ:
		return combo6x_ioctl_read(combo6, argp);
	case COMBO6_WRITE:
		return combo6x_ioctl_write(combo6, argp);
	case COMBO6_IOC_BOOT:
		return combo6x_ioctl_boot(combo6, argp);
	case COMBO6_IOC_BUS_ATTACH:
		if (combo6->usemode & USEMODE_SKIP_BAR12)
			return -ENXIO;
		return combo6_ioctl_bus_attach(combo6);
	case COMBO6_IOC_BUS_DETACH:
		if (combo6->usemode & USEMODE_SKIP_BAR12)
			return -ENXIO;
		return combo6_ioctl_bus_detach(combo6);
	case COMBO6_IOC_INFO:
		return combo6_ioctl_info(combo6, argp);
	default:
		return -ENOTTY;
	}
}

static irqreturn_t combo6x_interrupt(int irq, void *dev_id)
{
	struct combo6 *combo6 = dev_id;
	u_int32_t status;

	if (test_bit(COMBO_GOING_DOWN, combo6->flags))
		return IRQ_NONE;
	status = combo6_br_readl(combo6, PCR_INTR_STAT);
	if (status == 0)	/* fastpath */
		return IRQ_NONE;

	pr_debug("interrupt!!! status = 0x%x\n", status);

	if (status & INTR_LOCAL_MASK) {
		spin_lock(&combo6->reg_lock);
		if (combo6->driver && combo6->driver->interrupt)
			combo6->driver->interrupt(combo6, 1);
		spin_unlock(&combo6->reg_lock);
	}
	if (status & INTR_USER_MASK) {
		combo6_br_writel(combo6, PCR_INTR_USER_ACK,
				status & INTR_USER_MASK);
		spin_lock(&combo6->reg_lock);
		if (combo6->driver && combo6->driver->interrupt)
			combo6->driver->interrupt(combo6,
					status & INTR_USER_MASK);
		spin_unlock(&combo6->reg_lock);
	}
	return IRQ_HANDLED;
}

/*
 * real initialization
 */

static int __init combo6x_card_init(void)
{
	return pci_register_driver(&driver);
}

static void __exit combo6x_card_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(combo6x_card_init)
module_exit(combo6x_card_exit)

#ifndef MODULE

/* format is: combo6=enable */

static int __init combo6x_card_setup(char *str)
{
	static unsigned __initdata nr_dev;

	if (nr_dev >= COMBO6_CARDS)
		return 0;
	(void)(get_option(&str, &enable[nr_dev]) == 2);
	nr_dev++;
	return 1;
}

__setup("combo6x=", combo6x_card_setup);

#endif

/*
 * This part might be removed later.
 */

void combo6x_module_ref(void)
{
}
EXPORT_SYMBOL(combo6x_module_ref);
