/*
 * sze2cv3.c: Driver for the ComboV3 card - SZE2 Interface
 * Copyright (c) 2013-2014 CESNET
 * Author(s): Martin Spinler <spinler@cesnet.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/irq.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/pci.h>
#include <linux/timer.h>

#include "combo6k.h"

#include "cv3.h"
#include "szedata2k.h"
#include "sze2cv3.h"

#include "kocompat.h"

#define SZE2CV3_MAX_AGGR       1024
#define SZE2CV3_BLOCK_SIZE		(PAGE_SIZE*SZE2CV3_MAX_AGGR)
#define SZE2CV3_BLOCK_COUNT	(64*1024*1024/SZE2CV3_BLOCK_SIZE)

static unsigned long block_size = SZE2CV3_BLOCK_SIZE;
static unsigned int block_count = 0;
static unsigned int block_count_rx[32] = {
	[0 ... 31] = SZE2CV3_BLOCK_COUNT,
};
static unsigned int block_count_tx[32] = {
	[0 ... 31] = SZE2CV3_BLOCK_COUNT,
};
static unsigned int *const block_counts[2] = { block_count_rx, block_count_tx };
static int allocate_on_modprobe = 1;
static unsigned int poll_divisor = 8;

static inline unsigned int szedata2cv3_space_to_dir(unsigned int space)
{
	return space == SZE2_MMIO_RX ? SZE2CV3_DIR_RX : SZE2CV3_DIR_TX;
}

static struct szedata2_block *szedata2cv3_get_desc(struct szedata2cv3 *sdhw,
		unsigned int space, unsigned int channel)
{
	struct szedata2_block *desc = sdhw->desc;
	unsigned int a, page = 0, rx = space == SZE2_MMIO_RX;

	for (a = 0; a < (rx ? channel : sdhw->interfaces[0]); a++)
		page += DIV_ROUND_UP(block_count_rx[a], PAGE_SIZE / 8 - 1);
	if (!rx)
		for (a = 0; a < channel; a++)
			page += DIV_ROUND_UP(block_count_tx[a],
					PAGE_SIZE / 8 - 1);

	BUG_ON(page >= sdhw->desc_pgs);

	return desc + page;
}

static int szedata2cv3_alloc_descs(struct szedata2cv3 *sdhw)
{
	unsigned int a, b;
	int ret;

	if (WARN_ON(sdhw->desc_pgs))
		return -EINVAL;

	for (a = 0; a < 2; a++)
		for (b = 0; b < sdhw->interfaces[a]; b++) {
			/*
			 * each entry is 8 bytes, we have PAGE_SIZE bytes minus
			 * one entry (pointer to the next block) per page
			 */
			sdhw->desc_pgs += DIV_ROUND_UP(block_counts[a][b],
					PAGE_SIZE / 8 - 1);
		}

	sdhw->desc = szedata2_alloc_dma(&sdhw->combo->pci->dev,
			sdhw->desc_pgs, PAGE_SIZE);
	if (IS_ERR(sdhw->desc)) {
		printk(KERN_ERR "szedata2cv3: can't alloc pci space for "
				"descriptors\n");
		ret = PTR_ERR(sdhw->desc);
		goto err;
	}

	return 0;
err:
	sdhw->desc_pgs = 0;
	return ret;
}

static void szedata2cv3_free_descs(struct szedata2cv3 *sdhw)
{
	if (!sdhw->desc_pgs)
		return;

	szedata2_free_dma(&sdhw->combo->pci->dev, sdhw->desc,
			sdhw->desc_pgs, PAGE_SIZE);
	sdhw->desc_pgs = 0;
}

static void szedata2cv3_fill_descs(struct szedata2cv3 *sdhw,
		unsigned int space, unsigned int channel)
{
	struct szedata2_block *dsc;
	__le64 *uninitialized_var(v);
	dma_addr_t fst_phys;
	unsigned int a, pos;
	size_t len;

	if (!block_counts[space][channel])
		return;

	/* fill descs */
	dsc = szedata2cv3_get_desc(sdhw, space, channel);
	fst_phys = dsc->phys;
	pos = 0;
	while (1) {
		v = dsc->virt;
		for (a = 0; a < PAGE_SIZE / 8 - 1; a++) {
			dma_addr_t p;
			len = szedata2_get_phys(sdhw->sd, space, channel, &p, pos, block_size);
			if((p & 0xffffffff00000000) != ((p+len-1) & 0xffffffff00000000))
			{
				printk(KERN_ERR "DMA buffer block is crossing 32b boundaries. "
						"That doesn't like DMA Desc Manager. Expect errors!\n");
				len = 0x100000000 - (p & 0xffffffff00000000);
			}
			*v = cpu_to_le64(p + 2*(len/PAGE_SIZE -1));
			v++;
			pos += len;

			if (pos >= block_counts[space][channel]*block_size)
				goto finish;
		}

		dsc++;
		*v = cpu_to_le64(dsc->phys | SZE2CV3_DESCS_PTR);
	}
finish:
	*v = cpu_to_le64(fst_phys | SZE2CV3_DESCS_PTR);
}

static int szedata2cv3_alloc_dma(struct szedata2cv3 *sdhw)
{
	unsigned int channel;
	int ret;

	ret = szedata2_alloc_dmaspace(sdhw->sd, SZE2_MMIO_RX,
			sdhw->interfaces[0], block_count_rx, block_size,
			poll_divisor);
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace RX failed\n");
		goto err;
	}

	ret = szedata2_alloc_dmaspace(sdhw->sd, SZE2_MMIO_TX,
			sdhw->interfaces[1], block_count_tx, block_size,
			poll_divisor);
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace TX failed\n");
		goto err_rx;
	}

	ret = szedata2cv3_alloc_descs(sdhw);
	if (ret)
		goto err_tx;

	for (channel = 0; channel < sdhw->interfaces[0]; channel++)
		szedata2cv3_fill_descs(sdhw, SZE2_MMIO_RX, channel);
	for (channel = 0; channel < sdhw->interfaces[1]; channel++)
		szedata2cv3_fill_descs(sdhw, SZE2_MMIO_TX, channel);

	return 0;
err_tx:
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_TX);
err_rx:
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_RX);
err:
	return ret;
}

static void szedata2cv3_free_dma(struct szedata2cv3 *sdhw)
{
	szedata2cv3_free_descs(sdhw);
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_RX);
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_TX);
}

static unsigned long szedata2cv3_get_ptr(struct szedata2 *sd,
		unsigned int space, unsigned int channel, unsigned int which)
{
	/* which = 0 : head - start - sw
	 * which = 1 : tail - end   - hw*/
	struct szedata2cv3 *sdhw = sd->private;

	space = szedata2cv3_space_to_dir(space);

	if((space == 0 && which == 1) || (space == 1 && which == 0))
		return sdhw->swptr[space][channel];

	rmb();
	sdhw->hwptr[space][channel] = ((u32*)sdhw->statusarea_virt)[4096/4 + 8192/4*space + channel];

	return sdhw->hwptr[space][channel];

	/* nonzero `which' means tail
	 * RX: START is tail changed by us, END is head changed by HW
	 * TX: START is tail changed by HW, END is head changed by us
	 */
}

static void szedata2cv3_set_ptr(struct szedata2 *sd, unsigned int space,
		unsigned int channel, unsigned long ptr)
{
	struct szedata2cv3 *sdhw = sd->private;
	struct combo6 *combo = sdhw->combo;

	space = szedata2cv3_space_to_dir(space);

	/* we alter either RX tail or TX head */
	combo6_fpga_writel(combo, szedata2cv3_dma(sdhw, space, channel,
		space == SZE2CV3_DIR_RX ? SZE2CV3_STARTPTR :
		SZE2CV3_STARTPTR), ptr);

	sdhw->swptr[space][channel] = ptr;
}

static void szedata2cv3_set_intr(struct szedata2 *sd, unsigned int space,
		unsigned int channel, unsigned long value)
{
	struct szedata2cv3 *sdhw = sd->private;
	struct combo6 *combo = sdhw->combo;
	u32 irq = SZE2CV3_IRQ_PTR(value) | SZE2CV3_IRQ_PTRE;

	irq |= SZE2CV3_IRQ_TIMEOUTE;

	space = szedata2cv3_space_to_dir(space);

	combo6_fpga_writel(combo, szedata2cv3_dma(sdhw, space, channel, SZE2CV3_IRQ),
			irq);
}

static int szedata2cv3_open(struct szedata2 *sd)
{
	struct szedata2cv3 *sdhw = sd->private;
	int ret;

	if (!allocate_on_modprobe) {
		ret = szedata2cv3_alloc_dma(sdhw);
		if (ret)
			return ret;
	}

	return 0;
}

static void szedata2cv3_close(struct szedata2 *sd)
{
	if (!allocate_on_modprobe)
		szedata2cv3_free_dma(sd->private);
}

static int szedata2cv3_start(struct szedata2 *sd, unsigned int space,
		unsigned int channel)
{
	struct szedata2cv3 *sdhw = sd->private;
	struct szedata2_block *dsc;
	struct combo6 *combo = sdhw->combo;
	unsigned int trans_length = (space == 0 ? 0x100 : 0x200);

	printk(KERN_DEBUG "starting %cX:iface%u\n", space ? 'T' : 'R', channel);

	space = szedata2cv3_space_to_dir(space);
	dsc = szedata2cv3_get_desc(sdhw, space, channel);

	/* Init descriptor */
	((u64*)sdhw->statusarea_virt)[channel + space*8192/8] = (dsc->phys & 0xffffffffffffffffU) | SZE2CV3_DESCS_PTR;

	/* Set pointers to zero */
	sdhw->swptr[space][channel] = 0;

	sdhw->hwptr[space][channel] = 0;
	((u32*)sdhw->statusarea_virt)[4096/4 + 8192/4*space + channel] = 0;

	/* Set buffer size (mask) */
	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_BUFSIZE),
			block_size * block_counts[space][channel] - 1);

	/* Zero buffer ptr */
	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_STARTPTR), 0);

	/* No interrupt     */
	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_IRQ), 0x0);

	/* Timeout */
	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_TIMEOUT), 40000);

	/* Max size */
	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_MAXSIZE), trans_length);

	/* Start controller */
	combo6_fpga_writel(combo, szedata2cv3_dma(sdhw, space, channel, SZE2CV3_CTL),
			SZE2CV3_CTL_START);

	return 0;
}

static void szedata2cv3_stop(struct szedata2 *sd, unsigned int space,
		unsigned int channel)
{
	struct szedata2cv3 *sdhw = sd->private;
	struct combo6 *combo = sdhw->combo;
	unsigned int counter = 0;

	printk(KERN_DEBUG "stopping %cX:iface%u\n", space ? 'T' : 'R', channel);

	space = szedata2cv3_space_to_dir(space);

	if (space == SZE2CV3_DIR_TX) {
		while (1) {
			u32 s, e;
			s = combo6_fpga_readl(combo, szedata2cv3_dma(sdhw, space,
						channel, SZE2CV3_STARTPTR));
			e = combo6_fpga_readl(combo, szedata2cv3_dma(sdhw, space,
						channel, SZE2CV3_ENDPTR));
			if (s == e)
				break;
			if (combo_fatal_signal_pending(current)) {
				dev_warn(&combo->pci->dev, "ptrs not equal, "
						"killed by user\n");
				break;
			}
			msleep(10);
		}
	}

	combo6_fpga_writel(combo, szedata2cv3_dma(sdhw, space, channel, SZE2CV3_CTL),
			SZE2CV3_CTL_STOP | SZE2CV3_CTL_DISCARD);

	if(space == SZE2CV3_DIR_RX)
	{
		szedata2cv3_set_ptr(sd, 0, channel, szedata2cv3_get_ptr(sd, 0, channel, 0));
	}

	while (1) {
		u32 stat = combo6_fpga_readl(combo, szedata2cv3_dma(sdhw, space,
					channel, SZE2CV3_STAT));
		if (!(stat & 1))
			break;
		if (counter++ > 100) {
			dev_warn(&combo->pci->dev, "card didn't stop in 1 "
					"whole second (no idle after stop), "
					"stat=%.8x!\n", stat);
			break;
		}
		msleep(10);
	}
}

static void szedata2cv3_set_timeout(struct szedata2 *sd, unsigned int space,
		unsigned int channel, unsigned long value)
{
	struct szedata2cv3 *sdhw = sd->private;
	struct combo6 *combo = sdhw->combo;

	space = szedata2cv3_space_to_dir(space);

	combo6_fpga_writel(combo,
			szedata2cv3_dma(sdhw, space, channel, SZE2CV3_TIMEOUT),
			SZE2CV3_TIMEOUT_NS(value));
}

static void szedata2cv3_destroy(struct szedata2 *sd)
{
	struct szedata2cv3 *sdhw = sd->private;

	szedata2cv3_free_descs(sdhw);
}

static int szedata2cv3_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	struct szedata2cv3 *sdhw;
	struct szedata2 *sd;
	unsigned int ifaces[2];
	int ret;


	ifaces[0] = COMBOV2_RX_IFACES(interfaces);
	ifaces[1] = COMBOV2_TX_IFACES(interfaces);

	if (!interfaces || ifaces[0] >= ARRAY_SIZE(block_count_rx)
					|| ifaces[1] >= ARRAY_SIZE(block_count_tx))
		return -EINVAL;

	sd = szedata2_alloc(sizeof(struct szedata2cv3), &combo->pci->dev);
	if (IS_ERR(sd)) {
		printk(KERN_ERR "szedata2_alloc failed\n");
		ret = PTR_ERR(sd);
		goto err_dest;
	}
	sd->state = SZE2_MAPINSTANTLY | SZE2_NOADJUSTTIMER | SZE2_READPTRONLOCK;

	sd->owner = THIS_MODULE;
	sd->open = szedata2cv3_open;
	sd->close = szedata2cv3_close;
	sd->start = szedata2cv3_start;
	sd->stop = szedata2cv3_stop;
	sd->get_ptr = szedata2cv3_get_ptr;
	sd->set_ptr = szedata2cv3_set_ptr;
	sd->set_intr = szedata2cv3_set_intr;
	sd->set_timeout = szedata2cv3_set_timeout;
	sd->destroy = szedata2cv3_destroy;

	sdhw = sd->private;
	sdhw->sd = sd;
	sdhw->interfaces[0] = ifaces[0];
	sdhw->interfaces[1] = ifaces[1];
	sdhw->combo = combo;
	sdhw->base = SZE2CV3_ADDR_BASE;

	ret = szedata2_set_timeout(sd, 10000, 1000000);
	if (ret)
		goto err_dest;

	ret = szedata2cv3_alloc_dma(sdhw);
	if (ret)
		goto err_dest;

	sdhw->statusarea_virt = dma_alloc_coherent(&combo->pci->dev, 16384,
		&sdhw->statusarea_phys, GFP_KERNEL);
	if (sdhw->statusarea_virt == NULL) {
		ret = -ENOMEM;
		goto err_dest;
	}

	dev_info(&combo->pci->dev, "SZE2CV3 DMA 16k space at 0x%llx\n",
		(u64) sdhw->statusarea_phys);
#ifdef CONFIG_NUMA
	dev_info(&combo->pci->dev, "SZE2CV3 located nearby NUMA node %d\n",
	    combo->pci->dev.numa_node);
#else
	dev_info(&combo->pci->dev, "SZE2CV3 unknown NUMA locality\n");
#endif

	/* Write base config address */
	combo6_fpga_writel(combo, sdhw->base + SZE2CV3_CONFIG_PHYS + 0,
		sdhw->statusarea_phys);
	combo6_fpga_writel(combo, sdhw->base + SZE2CV3_CONFIG_PHYS + 4,
		sdhw->statusarea_phys >> 32);

	ret = szedata2_register(sd);
	if (ret) {
		printk(KERN_ERR "szedata2_register failed\n");
		goto err_dest;
	}

	combo->private_data = sd;

	return 0;
err_dest:
	szedata2_destroy(sd);
	return ret;
}

static int szedata2cv3_detach(struct combo6 *combo)
{
	struct szedata2 *sd = combo->private_data;
	struct szedata2cv3 *sdhw = sd->private;

	if(sdhw->statusarea_virt) {
		dma_free_coherent(&combo->pci->dev, 16384, sdhw->statusarea_virt, sdhw->statusarea_phys);
		sdhw->statusarea_virt = 0;
	}

	if (combo->id_sw >> 16 == 0x41c1)
		szedata2_eth_remove(sd);
	szedata2_destroy(sd);

	return 0;
}

static irqreturn_t szedata2cv3_interrupt(struct combo6 *combo,
		unsigned int mask)
{
	struct szedata2 *sd = combo->private_data;
	struct szedata2cv3 *sdhw = sd->private;
	u64 rx = 0, tx = 0;
	unsigned long hwptrx;
	int i;

	if (mask & COMBOV3_IRQS_RX) {
		for (i = 0; i < sdhw->interfaces[0]; i++)
		{
			hwptrx = sdhw->hwptr[0][i];
			if(hwptrx !=  szedata2cv3_get_ptr(sd, 0, i, 0))
				rx |= (1 << (i*2));
		}

		for (i = 0; i < sdhw->interfaces[1]; i++)
		{
			hwptrx = sdhw->hwptr[1][i];
			if(hwptrx != szedata2cv3_get_ptr(sd, 1, i, 0))
				tx |= (1 << (i*2));
		}
	}

	if (rx || tx) {
		szedata2_intr(sd, mask, rx, tx);
		return IRQ_HANDLED;
	}

	return IRQ_NONE;
}

static const struct combo_device_id szedata2x_ids[] = {
	{ /* NIC */
		.id_lsw	= 0x41c10700,
		.id_hsw = 0x41c10800,
	},
	{ /* HANIC */
		.id_lsw	= 0xa41c0100,
		.id_hsw = 0xa41c0400,
	},
	{ /* SDM */
		.id_lsw = 0x5d010000,
		.id_hsw = 0x5d011000
	},
	{
	}
};

static struct combo_driver szedata2x_ops = {
	.drv = {
		.owner  = THIS_MODULE,
		.name	= "szedata2cv3",
	},
	.dhw	= DHW_COMBOV2,
	.id_table = szedata2x_ids,
	.attach	= szedata2cv3_attach,
	.detach	= szedata2cv3_detach,
	.interrupt = szedata2cv3_interrupt,
};

static int __devinit szedata2_cv3_probe(struct pci_dev *pci,
					const struct pci_device_id *id)
{
	return combov3_probe(pci, id);
}

static void __devexit szedata2_cv3_remove(struct pci_dev *pci)
{
	combov3_remove(pci);
}

static struct pci_device_id combov3_ids[] __devinitdata = {
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0700 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0701 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0702 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0703 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0704 },

	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfbc1,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0800 },

	{ 0, }
};
MODULE_DEVICE_TABLE(pci, combov3_ids);

static struct pci_driver driver = {
	.name = "szedata2 ComboV3",
	.id_table = combov3_ids,
	.probe = szedata2_cv3_probe,
	.remove = __devexit_p(szedata2_cv3_remove),
};

static void copy_block_count(unsigned int dir, unsigned int channel)
{
	if (block_count)
		block_counts[dir][channel] = block_count;

	if (block_counts[dir][channel] &&
			!is_power_of_2(block_counts[dir][channel])) {
		block_counts[dir][channel] =
			roundup_pow_of_two(block_counts[dir][channel]);
		printk(KERN_INFO "szedata2: block_count for %cx[%u] is not "
				"power of 2, rounding up to %u\n",
				dir ? 't' : 'r', channel, block_counts[dir][channel]);
	}
}

int szedata2cv3_init(void)
{
	unsigned int a, b, sum_blocks = 0;
	int ret;

	if (block_size != PAGE_ALIGN(block_size)) {
		block_size = PAGE_ALIGN(block_size);
		printk(KERN_INFO "szedata2: block_size is not multiple of "
				"page size, rounding up to %lu\n",
				block_size);
	}

	if (poll_divisor < 1 || poll_divisor > block_size) {
		poll_divisor = clamp_t(unsigned int, poll_divisor,
				1, block_size);
		printk(KERN_INFO "szedata2: poll_divisor is not between 1 and "
				"block_size, clamped to %u\n", poll_divisor);
	}

	BUILD_BUG_ON(ARRAY_SIZE(block_count_rx) != ARRAY_SIZE(block_count_tx));

	for (a = 0; a < 2; a++)
		for (b = 0; b < ARRAY_SIZE(block_count_rx); b++) {
			copy_block_count(a, b);
			sum_blocks += block_counts[a][b];
		}

	if (!sum_blocks) {
		printk(KERN_ERR "%s: overall zero block count?\n", __func__);
		return -EINVAL;
	}

	ret = combo_register_driver(&szedata2x_ops);
	if (ret < 0) {
		printk(KERN_ERR "szedata2: can't register combov3 driver\n");
		return ret;
	}

	ret = pci_register_driver(&driver);
	if (ret) {
		printk(KERN_ERR "szedata2: can't register pci driver\n");
		combo_unregister_driver(&szedata2x_ops);
	}
	return ret;
}

void szedata2cv3_exit(void)
{
	pci_unregister_driver(&driver);
	combo_unregister_driver(&szedata2x_ops);
}

module_init(szedata2cv3_init);
module_exit(szedata2cv3_exit);

MODULE_LICENSE("GPL v2");

module_param(block_size, ulong, S_IRUGO);
MODULE_PARM_DESC(block_size, "block size [PAGE_SIZE]");
module_param(block_count, uint, S_IRUGO);
MODULE_PARM_DESC(block_count, "block count (overrides entries in both rx and "
		"tx arrays)");
module_param_array(block_count_rx, uint, NULL, S_IRUGO);
MODULE_PARM_DESC(block_count_rx, "block count array for rx "
		"[64*1024*1024/PAGE_SIZE]");
module_param_array(block_count_tx, uint, NULL, S_IRUGO);
MODULE_PARM_DESC(block_count_tx, "block count array for tx "
		"[64*1024*1024/PAGE_SIZE]");
module_param(allocate_on_modprobe, bool, 0400);
MODULE_PARM_DESC(allocate_on_modprobe, "allocate ring bufs on modprobe [yes]");
module_param(poll_divisor, uint, 0400);
MODULE_PARM_DESC(poll_divisor, "block_size*block_count/poll_divisor data wake "
		"up applications [8]");

