/*
 * szedata 2 test module
 *
 * Copyright (c) 2007-2008 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/random.h>
#include <linux/sched.h>
#include <linux/timer.h>

#include "szedata2k.h"

#include "kocompat.h"

#define SZET_BLOCK_COUNT	(5*1024*1024/SZET_BLOCK_SIZE)
#define SZET_BLOCK_SIZE		PAGE_SIZE
#define SZET_AREA_SIZE		(SZET_BLOCK_SIZE * SZET_BLOCK_COUNT)

static void szetest_timer(unsigned long unused);
static void szetest_timer1(unsigned long unused);
static void szetest_work(struct work_struct *work);

static DEFINE_TIMER(timer, szetest_timer, 0, 0);
static DEFINE_TIMER(timer1, szetest_timer1, 0, 0);
static DECLARE_DELAYED_WORK(work, szetest_work);
static atomic_t refcnt = ATOMIC_INIT(0), refcnt1 = ATOMIC_INIT(0);
static struct szedata2 *sd, *sd1;
static struct file *filp;
static fl_owner_t fl_owner;
static unsigned long rx_head, rx_tail, tx_tail, tx_head;
static u8 reset_tim;

#define get_free(h) ((SZET_AREA_SIZE + rx_tail - (h) - 1) % SZET_AREA_SIZE)

static void szetest_timer(unsigned long unused)
{
	static unsigned long lrx_head;
	static size_t off, count;
	static struct szedata2_packet pkt;
	static u8 c = 'A';
	void *virt;
	size_t len;
	unsigned int packets = 100;

	if (reset_tim) {
		off = 0;
		reset_tim = 0;
	}

	if (off >= sizeof(pkt))
		goto resume_pkt;
	if (off)
		goto resume_hdr;
next:
	count = 2 * SZET_BLOCK_SIZE + random32() % SZET_BLOCK_SIZE;
	pkt.hw_size = ALIGN(count - 100 + 4, 8) - 4;
	pkt.seg_size = 100;
	count = pkt.hw_size + pkt.seg_size;
	rmb();
	lrx_head = rx_head;
resume_hdr:
	len = szedata2_get_virt(sd, SZE2_MMIO_RX, 1, &virt, lrx_head,
			sizeof(pkt) - off);
	len = min(len, get_free(lrx_head));
	if (!len)
		goto end;

	memcpy(virt, (void *)&pkt + off, len);
	off += len;
	lrx_head = (lrx_head + len) % SZET_AREA_SIZE;
	if (off < sizeof(pkt))
		goto resume_hdr;
	while (count > 0) {
resume_pkt:
		len = szedata2_get_virt(sd, SZE2_MMIO_RX, 1, &virt, lrx_head,
				count);
		len = min(len, get_free(lrx_head));
		if (!len)
			goto end;

		memset(virt, c, len);
		count -= len;
		off += len;
		lrx_head = (lrx_head + len) % SZET_AREA_SIZE;
	}
	memset(virt + len - 1, '*', 1);
	off = 0;
	rx_head = lrx_head;

	if (c == 'Z')
		c = 'A';
	else
		c++;

	if (--packets > 0)
		goto next;
end:
	mod_timer(&timer, jiffies + msecs_to_jiffies(off ? 1000 : 200));
}

static void szetest_timer1(unsigned long unused)
{
	tx_tail = (SZET_AREA_SIZE - 1 + tx_head) % SZET_AREA_SIZE;
	mod_timer(&timer1, jiffies + msecs_to_jiffies(150));
}

static void szetest_work(struct work_struct *work)
{
	struct file *filp1;

	filp1 = filp_open("/dev/szedataII3", O_RDWR, 0);
	if (!IS_ERR(filp1)) {
		printk(KERN_WARNING "szedata2: unregistered open didn't "
				"fail\n");
		filp_close(filp1, current->files);
	}
	printk(KERN_DEBUG "closing\n");
	filp_close(filp, fl_owner);
}

static int szetest_open(struct szedata2 *sd)
{
	return 0;
}

static void szetest_close(struct szedata2 *sd)
{
}

static int szetest_start(struct szedata2 *sd, unsigned int space,
		unsigned int area)
{
	printk(KERN_DEBUG "starting %u:%u\n", space, area);
	if (space == SZE2_MMIO_RX && area == 1) {
		if (atomic_inc_return(&refcnt) != 1)
			return 0;
		rx_tail = rx_head = 0;
		reset_tim = 1;
		mod_timer(&timer, jiffies + HZ);
		printk(KERN_DEBUG "HW started\n");
		return 0;
	} else if (space == SZE2_MMIO_TX && area == 0) {
		if (atomic_inc_return(&refcnt1) != 1)
			return 0;
		tx_tail = SZET_AREA_SIZE;
		tx_head = 0;
		mod_timer(&timer1, jiffies + HZ);
		return 0;
	}
	return -EINVAL;
}

static void szetest_stop(struct szedata2 *sd, unsigned int space,
		unsigned int area)
{
	printk(KERN_DEBUG "stopping %u:%u\n", space, area);
	if (space == SZE2_MMIO_RX && area == 1 &&
			atomic_dec_and_test(&refcnt)) {
		del_timer_sync(&timer);
		printk(KERN_DEBUG "HW stopped\n");
	} else if (space == SZE2_MMIO_TX && area == 0 &&
			atomic_dec_and_test(&refcnt1)) {
		del_timer_sync(&timer1);
	}
}

static unsigned long szetest_get_ptr(struct szedata2 *sd, unsigned int space,
		unsigned int area, unsigned int which)
{
	if ((space != SZE2_MMIO_RX || area != 1) &&
			(space != SZE2_MMIO_TX || area != 0))
		return 0;
	rmb();
	if (space == SZE2_MMIO_TX)
		return which ? tx_tail : tx_head;
	else
		return which ? rx_tail : rx_head;
}

static void szetest_set_ptr(struct szedata2 *sd, unsigned int space,
		unsigned int area, unsigned long ptr)
{
	if (space == SZE2_MMIO_TX)
		tx_head = ptr;
	else
		rx_tail = ptr;
	wmb();
}

/*
 * INIT test
 */

static int szetest_test_init(void)
{
	struct szedata2 *sda[8];
	struct pci_dev *pdev;
	unsigned int a, b;
	int ret;
#ifdef SZETEST_FAULT
	/* some fault injection */
	if (!szedata2_register(NULL) || !szedata2_register(ERR_PTR(-EBUSY)))
		printk(KERN_WARNING "szedata2_register bogus didn't fail\n");
	szedata2_destroy(NULL);
	szedata2_destroy(ERR_PTR(-EBUSY));
#endif
	/* real device with private size 1 */
	/* expects one persistent pci device/bridge */
	pdev = pci_get_device(PCI_ANY_ID, PCI_ANY_ID, NULL);
	sd = alloc_szedata2(1, &pdev->dev);
	pci_dev_put(pdev);
	if (IS_ERR(sd)) {
		printk(KERN_ERR "alloc_szedata2 A failed\n");
		ret = PTR_ERR(sd);
		goto err_dest;
	}
	/* owner not set test */
	if (!szedata2_register(sd))
		printk(KERN_WARNING "szedata2_register A didn't fail\n");
	sd->owner = THIS_MODULE;
	sd->open = szetest_open;
	sd->close = szetest_close;
	sd->start = szetest_start;
	sd->stop = szetest_stop;
	sd->get_ptr = szetest_get_ptr;
	sd->set_ptr = szetest_set_ptr;

	ret = szedata2_alloc_dmaspace(sd, SZE2_MMIO_RX, 4,
			SZET_BLOCK_COUNT, PAGE_ALIGN(SZET_BLOCK_SIZE));
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace RX failed\n");
		goto err_dest;
	}

	ret = szedata2_alloc_dmaspace(sd, SZE2_MMIO_TX, 1,
			SZET_BLOCK_COUNT, PAGE_ALIGN(SZET_BLOCK_SIZE));
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace TX failed\n");
		goto err_dest;
	}

	ret = szedata2_register(sd);
	if (ret) {
		printk(KERN_ERR "szedata2_register A failed\n");
		goto err_dest;
	}

	/* too much devices test */
	for (a = 0; a < ARRAY_SIZE(sda); a++) {
		sda[a] = alloc_szedata2(0, NULL);
		if (IS_ERR(sda[a])) {
			printk(KERN_ERR "alloc_szedata2 B failed\n");
			ret = PTR_ERR(sda[a]);
			goto err_dest;
		}
		sda[a]->owner = THIS_MODULE;
		ret = szedata2_register(sda[a]);
		if (ret && a + 1 < ARRAY_SIZE(sda)) {
			printk(KERN_ERR "szedata2_register B failed\n");
			for (b = a + 1; b > 0; b--)
				szedata2_destroy(sda[b - 1]);
			goto err_dest;
		} else if (!ret && a + 1 == ARRAY_SIZE(sda))
			printk(KERN_WARNING "szedata2_register B didn't "
					"fail\n");
		printk(KERN_DEBUG "%u\n", a);
	}
	sd1 = sda[ARRAY_SIZE(sda) - 2]; /* last successful */
	sda[ARRAY_SIZE(sda) - 2] = NULL; /* don't free this */

	msleep(1000); /* wait for udev */
	filp = filp_open("/dev/szedataII3", O_RDWR, 0);
	if (IS_ERR(filp))
		printk(KERN_ERR "can't open szedata node\n");
	else {
		fl_owner = current->files;
		schedule_delayed_work(&work, HZ); /* close it in a sec */
	}

	for (a = 0; a < ARRAY_SIZE(sda); a++) {
		printk(KERN_DEBUG "destroying: %u\n", a);
		szedata2_destroy(sda[a]);
	}

	/* register twice test */
	if (!szedata2_register(sd1))
		printk(KERN_WARNING "szedata2_register C didn't fail\n");

	if (!sd->private || sd1->private)
		printk(KERN_WARNING "szedata2 privates: %p %p\n", sd->private,
				sd1->private);

	return 0;
err_dest:
	szedata2_destroy(sd);
	szedata2_destroy(sd1);
	return ret;
}

static void szetest_kill_init(void)
{
	szedata2_destroy(sd);
	szedata2_destroy(sd1);
}

static int szetest_init(void)
{
	int ret;

	ret = szetest_test_init(); /* fills sd, sd1 */
	if (ret)
		goto err;

	return 0;
err:
	return ret;
}

static void szetest_exit(void)
{
	szetest_kill_init();
}

module_init(szetest_init);
module_exit(szetest_exit);

MODULE_LICENSE("GPL");
