#include "kocompat.h"

#include <linux/version.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 23)
#include <szedata.h>
#include <szedatak.h>
static int szedata_pkt_mmap(struct szedata_device *, struct vm_area_struct *);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18)
static int szedata_mmap(struct file *, struct vm_area_struct *);
#endif

#include "../../../kernel/drivers/szedata/szecore.c"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 23)
static struct page *szedata_mmap_pkt_nopage(struct vm_area_struct *area,
		unsigned long address, int *type)
{
	struct szedata_device *dev = area->vm_private_data;
	unsigned long offset;
	struct page *page;

	offset = area->vm_pgoff << PAGE_SHIFT;
	offset += address - area->vm_start;
	offset -= SZEDATA_MMAP_OFFSET_PKT;
	if (offset > dev->alloc_mmap_pages - PAGE_SIZE)
		return NOPAGE_SIGBUS;
	page = virt_to_page(dev->mmap_page_table[offset / PAGE_SIZE]);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static struct vm_operations_struct szedata_vm_ops_pkt = {
	.nopage = szedata_mmap_pkt_nopage
};

static int szedata_pkt_mmap(struct szedata_device *device,
		struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes = PAGE_ALIGN(device->alloc_mmap_pages);

	if ((area->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_PKT > mmap_bytes - size)
		return -EINVAL;
	area->vm_ops = &szedata_vm_ops_pkt;
	area->vm_private_data = device;
	area->vm_flags |= VM_RESERVED;
	return 0;
}
#endif /* < 2.6.23 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18)
static struct page *szedata_mmap_app_lock_nopage(struct vm_area_struct *area,
		unsigned long address, int *type)
{
	struct szedata_instance *instance = area->vm_private_data;
	unsigned long offset;
	struct page *page;

	offset = area->vm_pgoff << PAGE_SHIFT;
	offset += address - area->vm_start;
	offset -= SZEDATA_MMAP_OFFSET_LOCK;
	if ((offset != 0 && sizeof(struct szedata_mmap_app_lock) < PAGE_SIZE) ||
	    offset >= sizeof(struct szedata_mmap_app_lock))
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page((char *)instance->app_lock + offset);
	if (!PageReserved(page))
		get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static struct vm_operations_struct szedata_vm_ops_app_lock = {
	.nopage = szedata_mmap_app_lock_nopage
};

static int szedata_app_lock_mmap(struct szedata_instance *instance,
		struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes =
		PAGE_ALIGN(sizeof(struct szedata_mmap_app_lock));

	if ((area->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_LOCK > mmap_bytes - size)
		return -EINVAL;
	area->vm_ops = &szedata_vm_ops_app_lock;
	area->vm_private_data = instance;
	area->vm_flags |= VM_RESERVED;

	return 0;
}

static struct page *szedata_mmap_istats_nopage(struct vm_area_struct *area,
		unsigned long address, int *type)
{
	struct szedata_instance *instance = area->vm_private_data;
	unsigned long offset;
	struct page *page;

	offset = area->vm_pgoff << PAGE_SHIFT;
	offset += address - area->vm_start;
	offset -= SZEDATA_MMAP_OFFSET_ISTATS;
	if (offset != 0)
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page(instance->istats);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static struct vm_operations_struct szedata_vm_ops_istats = {
	.nopage = szedata_mmap_istats_nopage
};

static int szedata_istats_mmap(struct szedata_instance *instance,
		struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes =
		PAGE_ALIGN(sizeof(struct szedata_mmap_block_istats));

	if ((area->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_ISTATS > mmap_bytes - size)
		return -EINVAL;
	area->vm_ops = &szedata_vm_ops_istats;
	area->vm_private_data = instance;
	area->vm_flags |= VM_RESERVED;
	return 0;
}

static struct page *szedata_mmap_status_nopage(struct vm_area_struct *area,
		unsigned long address, int *type)
{
	struct szedata_instance *instance = area->vm_private_data;
	unsigned long offset;
	struct page *page;

	offset = area->vm_pgoff << PAGE_SHIFT;
	offset += address - area->vm_start;
	offset -= SZEDATA_MMAP_OFFSET_STATUS;
	if (offset != 0)
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page(instance->status);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static struct vm_operations_struct szedata_vm_ops_status = {
	.nopage = szedata_mmap_status_nopage
};

static int szedata_status_mmap(struct szedata_instance *instance,
		struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes =
		PAGE_ALIGN(sizeof(struct szedata_mmap_block_status));

	if ((area->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_STATUS > mmap_bytes - size)
		return -EINVAL;
	area->vm_ops = &szedata_vm_ops_status;
	area->vm_private_data = instance;
	area->vm_flags |= VM_RESERVED;
	return 0;
}

static struct page *szedata_mmap_pktdsc_nopage(struct vm_area_struct *area,
		unsigned long address, int *type)
{
	struct szedata_device *dev = area->vm_private_data;
	unsigned long offset;
	struct page *page;

	offset = area->vm_pgoff << PAGE_SHIFT;
	offset += address - area->vm_start;
	offset -= SZEDATA_MMAP_OFFSET_PKTDSC;
	if (offset >= dev->alloc_blocks * sizeof(struct szedata_mmap_block_descriptor))
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page((char *)dev->desc + offset);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static struct vm_operations_struct szedata_vm_ops_pktdsc = {
	.nopage = szedata_mmap_pktdsc_nopage
};

static int szedata_pktdsc_mmap(struct szedata_device *device,
		struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes = PAGE_ALIGN(device->alloc_blocks *
			sizeof(struct szedata_mmap_block_descriptor));

	if ((area->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_PKTDSC > mmap_bytes - size)
		return -EINVAL;
	area->vm_ops = &szedata_vm_ops_pktdsc;
	area->vm_private_data = device;
	area->vm_flags |= VM_RESERVED;
	return 0;
}

static int szedata_mmap(struct file *file, struct vm_area_struct *area)
{
	struct szedata_instance *instance = file->private_data;
	unsigned long offset;

	offset = area->vm_pgoff << PAGE_SHIFT;
	if (offset >= SZEDATA_MMAP_OFFSET_PKT)
		return szedata_pkt_mmap(instance->dev, area);
	else if (offset >= SZEDATA_MMAP_OFFSET_PKTDSC)
		return szedata_pktdsc_mmap(instance->dev, area);
	else if (offset >= SZEDATA_MMAP_OFFSET_STATUS)
		return szedata_status_mmap(instance, area);
	else if (offset >= SZEDATA_MMAP_OFFSET_ISTATS)
		return szedata_istats_mmap(instance, area);
	else
		return szedata_app_lock_mmap(instance, area);
}
#endif /* < 2.6.18 */
