/*
								+----------------------------------+
								|                                  |
								| ***  8bit to unicode tables  *** |
								|                                  |
								|   Copyright  -tHE SWINe- 2011   |
								|                                  |
								|          UniTables.inl           |
								|                                  |
								+----------------------------------+
*/

/**
 *	@file UniTables.inl
 *	@brief 8bit charsets to unicode mapping tables
 *	@date 2011
 *	@author -tHE SWINe-
 */

namespace __8bit_to_unicode_mapping_tables {

/**
 *	@brief iso-8859-1 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-1:1998 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-1:1998 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-1 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-1 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_1_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x00d0, "LATIN CAPITAL LETTER ETH (Icelandic)"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x00de, "LATIN CAPITAL LETTER THORN (Icelandic)"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S (German)"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x00f0, "LATIN SMALL LETTER ETH (Icelandic)"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x00fe, "LATIN SMALL LETTER THORN (Icelandic)"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_iso_8859_1_table_size = sizeof(p_iso_8859_1_table) / sizeof(p_iso_8859_1_table[0]);

/**
 *	@brief iso-8859-10 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-10:1998 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.1
 *	Table format:     Format A
 *	Date:             1999 October 11
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-10:1998 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-10 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-10 order.
 *
 *	Version history
 *	1.0 version new.
 *       1.1 corrected mistake in mapping of 0xA4
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_10_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa2, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0xa3, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0xa4, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xa5, 0x0128, "LATIN CAPITAL LETTER I WITH TILDE"},
	{0xa6, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xa9, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xaa, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xab, 0x0166, "LATIN CAPITAL LETTER T WITH STROKE"},
	{0xac, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xaf, 0x014a, "LATIN CAPITAL LETTER ENG"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xb2, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0xb3, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0xb4, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0xb5, 0x0129, "LATIN SMALL LETTER I WITH TILDE"},
	{0xb6, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xb9, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xba, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xbb, 0x0167, "LATIN SMALL LETTER T WITH STROKE"},
	{0xbc, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xbd, 0x2015, "HORIZONTAL BAR"},
	{0xbe, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xbf, 0x014b, "LATIN SMALL LETTER ENG"},
	{0xc0, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x00d0, "LATIN CAPITAL LETTER ETH (Icelandic)"},
	{0xd1, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xd2, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x0168, "LATIN CAPITAL LETTER U WITH TILDE"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x00de, "LATIN CAPITAL LETTER THORN (Icelandic)"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S (German)"},
	{0xe0, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x00f0, "LATIN SMALL LETTER ETH (Icelandic)"},
	{0xf1, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xf2, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x0169, "LATIN SMALL LETTER U WITH TILDE"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x00fe, "LATIN SMALL LETTER THORN (Icelandic)"},
	{0xff, 0x0138, "LATIN SMALL LETTER KRA"}
};
const size_t n_iso_8859_10_table_size = sizeof(p_iso_8859_10_table) / sizeof(p_iso_8859_10_table[0]);

/**
 *	@brief iso-8859-11 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-11:2001 to Unicode
 *	Unicode version:  3.2
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             2002 October 7
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 2002 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-11:2001 characters map into Unicode.
 *
 *	ISO/IEC 8859-11:2001 is equivalent to TIS 620-2533 (1990) with
 *	the addition of 0xA0 NO-BREAK SPACE.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-11 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-11 order.
 *
 *	Version history:
 *		2002 October 7  Created
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	For any comments or problems, please use the Unicode
 *	web contact form at:
 *		http://www.unicode.org/unicode/reporting.html
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_11_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0e01, "THAI CHARACTER KO KAI"},
	{0xa2, 0x0e02, "THAI CHARACTER KHO KHAI"},
	{0xa3, 0x0e03, "THAI CHARACTER KHO KHUAT"},
	{0xa4, 0x0e04, "THAI CHARACTER KHO KHWAI"},
	{0xa5, 0x0e05, "THAI CHARACTER KHO KHON"},
	{0xa6, 0x0e06, "THAI CHARACTER KHO RAKHANG"},
	{0xa7, 0x0e07, "THAI CHARACTER NGO NGU"},
	{0xa8, 0x0e08, "THAI CHARACTER CHO CHAN"},
	{0xa9, 0x0e09, "THAI CHARACTER CHO CHING"},
	{0xaa, 0x0e0a, "THAI CHARACTER CHO CHANG"},
	{0xab, 0x0e0b, "THAI CHARACTER SO SO"},
	{0xac, 0x0e0c, "THAI CHARACTER CHO CHOE"},
	{0xad, 0x0e0d, "THAI CHARACTER YO YING"},
	{0xae, 0x0e0e, "THAI CHARACTER DO CHADA"},
	{0xaf, 0x0e0f, "THAI CHARACTER TO PATAK"},
	{0xb0, 0x0e10, "THAI CHARACTER THO THAN"},
	{0xb1, 0x0e11, "THAI CHARACTER THO NANGMONTHO"},
	{0xb2, 0x0e12, "THAI CHARACTER THO PHUTHAO"},
	{0xb3, 0x0e13, "THAI CHARACTER NO NEN"},
	{0xb4, 0x0e14, "THAI CHARACTER DO DEK"},
	{0xb5, 0x0e15, "THAI CHARACTER TO TAO"},
	{0xb6, 0x0e16, "THAI CHARACTER THO THUNG"},
	{0xb7, 0x0e17, "THAI CHARACTER THO THAHAN"},
	{0xb8, 0x0e18, "THAI CHARACTER THO THONG"},
	{0xb9, 0x0e19, "THAI CHARACTER NO NU"},
	{0xba, 0x0e1a, "THAI CHARACTER BO BAIMAI"},
	{0xbb, 0x0e1b, "THAI CHARACTER PO PLA"},
	{0xbc, 0x0e1c, "THAI CHARACTER PHO PHUNG"},
	{0xbd, 0x0e1d, "THAI CHARACTER FO FA"},
	{0xbe, 0x0e1e, "THAI CHARACTER PHO PHAN"},
	{0xbf, 0x0e1f, "THAI CHARACTER FO FAN"},
	{0xc0, 0x0e20, "THAI CHARACTER PHO SAMPHAO"},
	{0xc1, 0x0e21, "THAI CHARACTER MO MA"},
	{0xc2, 0x0e22, "THAI CHARACTER YO YAK"},
	{0xc3, 0x0e23, "THAI CHARACTER RO RUA"},
	{0xc4, 0x0e24, "THAI CHARACTER RU"},
	{0xc5, 0x0e25, "THAI CHARACTER LO LING"},
	{0xc6, 0x0e26, "THAI CHARACTER LU"},
	{0xc7, 0x0e27, "THAI CHARACTER WO WAEN"},
	{0xc8, 0x0e28, "THAI CHARACTER SO SALA"},
	{0xc9, 0x0e29, "THAI CHARACTER SO RUSI"},
	{0xca, 0x0e2a, "THAI CHARACTER SO SUA"},
	{0xcb, 0x0e2b, "THAI CHARACTER HO HIP"},
	{0xcc, 0x0e2c, "THAI CHARACTER LO CHULA"},
	{0xcd, 0x0e2d, "THAI CHARACTER O ANG"},
	{0xce, 0x0e2e, "THAI CHARACTER HO NOKHUK"},
	{0xcf, 0x0e2f, "THAI CHARACTER PAIYANNOI"},
	{0xd0, 0x0e30, "THAI CHARACTER SARA A"},
	{0xd1, 0x0e31, "THAI CHARACTER MAI HAN-AKAT"},
	{0xd2, 0x0e32, "THAI CHARACTER SARA AA"},
	{0xd3, 0x0e33, "THAI CHARACTER SARA AM"},
	{0xd4, 0x0e34, "THAI CHARACTER SARA I"},
	{0xd5, 0x0e35, "THAI CHARACTER SARA II"},
	{0xd6, 0x0e36, "THAI CHARACTER SARA UE"},
	{0xd7, 0x0e37, "THAI CHARACTER SARA UEE"},
	{0xd8, 0x0e38, "THAI CHARACTER SARA U"},
	{0xd9, 0x0e39, "THAI CHARACTER SARA UU"},
	{0xda, 0x0e3a, "THAI CHARACTER PHINTHU"},
	{0xdf, 0x0e3f, "THAI CURRENCY SYMBOL BAHT"},
	{0xe0, 0x0e40, "THAI CHARACTER SARA E"},
	{0xe1, 0x0e41, "THAI CHARACTER SARA AE"},
	{0xe2, 0x0e42, "THAI CHARACTER SARA O"},
	{0xe3, 0x0e43, "THAI CHARACTER SARA AI MAIMUAN"},
	{0xe4, 0x0e44, "THAI CHARACTER SARA AI MAIMALAI"},
	{0xe5, 0x0e45, "THAI CHARACTER LAKKHANGYAO"},
	{0xe6, 0x0e46, "THAI CHARACTER MAIYAMOK"},
	{0xe7, 0x0e47, "THAI CHARACTER MAITAIKHU"},
	{0xe8, 0x0e48, "THAI CHARACTER MAI EK"},
	{0xe9, 0x0e49, "THAI CHARACTER MAI THO"},
	{0xea, 0x0e4a, "THAI CHARACTER MAI TRI"},
	{0xeb, 0x0e4b, "THAI CHARACTER MAI CHATTAWA"},
	{0xec, 0x0e4c, "THAI CHARACTER THANTHAKHAT"},
	{0xed, 0x0e4d, "THAI CHARACTER NIKHAHIT"},
	{0xee, 0x0e4e, "THAI CHARACTER YAMAKKAN"},
	{0xef, 0x0e4f, "THAI CHARACTER FONGMAN"},
	{0xf0, 0x0e50, "THAI DIGIT ZERO"},
	{0xf1, 0x0e51, "THAI DIGIT ONE"},
	{0xf2, 0x0e52, "THAI DIGIT TWO"},
	{0xf3, 0x0e53, "THAI DIGIT THREE"},
	{0xf4, 0x0e54, "THAI DIGIT FOUR"},
	{0xf5, 0x0e55, "THAI DIGIT FIVE"},
	{0xf6, 0x0e56, "THAI DIGIT SIX"},
	{0xf7, 0x0e57, "THAI DIGIT SEVEN"},
	{0xf8, 0x0e58, "THAI DIGIT EIGHT"},
	{0xf9, 0x0e59, "THAI DIGIT NINE"},
	{0xfa, 0x0e5a, "THAI CHARACTER ANGKHANKHU"},
	{0xfb, 0x0e5b, "THAI CHARACTER KHOMUT"}
};
const size_t n_iso_8859_11_table_size = sizeof(p_iso_8859_11_table) / sizeof(p_iso_8859_11_table[0]);

/**
 *	@brief iso-8859-13 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-13:1998  to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-13:1998 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-13 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-13 order.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_13_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x0156, "LATIN CAPITAL LETTER R WITH CEDILLA"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x0157, "LATIN SMALL LETTER R WITH CEDILLA"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xc0, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xc1, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xc2, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0xc3, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xc7, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0xcb, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0xcc, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0xcd, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xce, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xcf, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xd0, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xd1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xd2, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xd9, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xda, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0xdb, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xde, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S (German)"},
	{0xe0, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xe1, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xe2, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0xe3, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xe7, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xeb, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0xec, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0xed, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xee, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0xef, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xf0, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xf1, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xf2, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xf9, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xfa, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0xfb, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xfe, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xff, 0x2019, "RIGHT SINGLE QUOTATION MARK"}
};
const size_t n_iso_8859_13_table_size = sizeof(p_iso_8859_13_table) / sizeof(p_iso_8859_13_table[0]);

/**
 *	@brief iso-8859-14 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-14:1998 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Markus Kuhn <http://www.cl.cam.ac.uk/~mgk25/>
 *			  Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-14:1998 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-14 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-14 order.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_14_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x1e02, "LATIN CAPITAL LETTER B WITH DOT ABOVE"},
	{0xa2, 0x1e03, "LATIN SMALL LETTER B WITH DOT ABOVE"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x010a, "LATIN CAPITAL LETTER C WITH DOT ABOVE"},
	{0xa5, 0x010b, "LATIN SMALL LETTER C WITH DOT ABOVE"},
	{0xa6, 0x1e0a, "LATIN CAPITAL LETTER D WITH DOT ABOVE"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x1e80, "LATIN CAPITAL LETTER W WITH GRAVE"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x1e82, "LATIN CAPITAL LETTER W WITH ACUTE"},
	{0xab, 0x1e0b, "LATIN SMALL LETTER D WITH DOT ABOVE"},
	{0xac, 0x1ef2, "LATIN CAPITAL LETTER Y WITH GRAVE"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xb0, 0x1e1e, "LATIN CAPITAL LETTER F WITH DOT ABOVE"},
	{0xb1, 0x1e1f, "LATIN SMALL LETTER F WITH DOT ABOVE"},
	{0xb2, 0x0120, "LATIN CAPITAL LETTER G WITH DOT ABOVE"},
	{0xb3, 0x0121, "LATIN SMALL LETTER G WITH DOT ABOVE"},
	{0xb4, 0x1e40, "LATIN CAPITAL LETTER M WITH DOT ABOVE"},
	{0xb5, 0x1e41, "LATIN SMALL LETTER M WITH DOT ABOVE"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x1e56, "LATIN CAPITAL LETTER P WITH DOT ABOVE"},
	{0xb8, 0x1e81, "LATIN SMALL LETTER W WITH GRAVE"},
	{0xb9, 0x1e57, "LATIN SMALL LETTER P WITH DOT ABOVE"},
	{0xba, 0x1e83, "LATIN SMALL LETTER W WITH ACUTE"},
	{0xbb, 0x1e60, "LATIN CAPITAL LETTER S WITH DOT ABOVE"},
	{0xbc, 0x1ef3, "LATIN SMALL LETTER Y WITH GRAVE"},
	{0xbd, 0x1e84, "LATIN CAPITAL LETTER W WITH DIAERESIS"},
	{0xbe, 0x1e85, "LATIN SMALL LETTER W WITH DIAERESIS"},
	{0xbf, 0x1e61, "LATIN SMALL LETTER S WITH DOT ABOVE"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x0174, "LATIN CAPITAL LETTER W WITH CIRCUMFLEX"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x1e6a, "LATIN CAPITAL LETTER T WITH DOT ABOVE"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x0176, "LATIN CAPITAL LETTER Y WITH CIRCUMFLEX"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x0175, "LATIN SMALL LETTER W WITH CIRCUMFLEX"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x1e6b, "LATIN SMALL LETTER T WITH DOT ABOVE"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x0177, "LATIN SMALL LETTER Y WITH CIRCUMFLEX"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_iso_8859_14_table_size = sizeof(p_iso_8859_14_table) / sizeof(p_iso_8859_14_table[0]);

/**
 *	@brief iso-8859-15 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-15:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Markus Kuhn <http://www.cl.cam.ac.uk/~mgk25/>
 *			  Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-15:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-15 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-15 order.
 *
 *	Version history
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_15_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x20ac, "EURO SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xbd, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xbe, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_iso_8859_15_table_size = sizeof(p_iso_8859_15_table) / sizeof(p_iso_8859_15_table[0]);

/**
 *	@brief iso-8859-16 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-16:2001 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             2001 July 26
 *	Authors:          Markus Kuhn <http://www.cl.cam.ac.uk/~mgk25/>
 *
 *	Copyright (c) 1999-2001 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-16:2001 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-16 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-16 order.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_16_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa2, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xa3, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xa4, 0x20ac, "EURO SIGN"},
	{0xa5, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xa6, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x0218, "LATIN CAPITAL LETTER S WITH COMMA BELOW"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xaf, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xb3, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xb4, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xb5, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xb9, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xba, 0x0219, "LATIN SMALL LETTER S WITH COMMA BELOW"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xbd, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xbe, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xbf, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x0102, "LATIN CAPITAL LETTER A WITH BREVE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x0150, "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0xd8, 0x0170, "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xde, 0x021a, "LATIN CAPITAL LETTER T WITH COMMA BELOW"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x0103, "LATIN SMALL LETTER A WITH BREVE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xf1, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x0151, "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0xf8, 0x0171, "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xfe, 0x021b, "LATIN SMALL LETTER T WITH COMMA BELOW"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_iso_8859_16_table_size = sizeof(p_iso_8859_16_table) / sizeof(p_iso_8859_16_table[0]);

/**
 *	@brief iso-8859-2 to unicode mapping table
 *
 *
 *	Name:             ISO 8859-2:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-2:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-2 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-2 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_2_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa2, 0x02d8, "BREVE"},
	{0xa3, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x013d, "LATIN CAPITAL LETTER L WITH CARON"},
	{0xa6, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xaa, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xab, 0x0164, "LATIN CAPITAL LETTER T WITH CARON"},
	{0xac, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xaf, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xb2, 0x02db, "OGONEK"},
	{0xb3, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x013e, "LATIN SMALL LETTER L WITH CARON"},
	{0xb6, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0xb7, 0x02c7, "CARON"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xba, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xbb, 0x0165, "LATIN SMALL LETTER T WITH CARON"},
	{0xbc, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xbd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xbe, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xbf, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xc0, 0x0154, "LATIN CAPITAL LETTER R WITH ACUTE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x0102, "LATIN CAPITAL LETTER A WITH BREVE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x0139, "LATIN CAPITAL LETTER L WITH ACUTE"},
	{0xc6, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x011a, "LATIN CAPITAL LETTER E WITH CARON"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x010e, "LATIN CAPITAL LETTER D WITH CARON"},
	{0xd0, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xd2, 0x0147, "LATIN CAPITAL LETTER N WITH CARON"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x0150, "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x0158, "LATIN CAPITAL LETTER R WITH CARON"},
	{0xd9, 0x016e, "LATIN CAPITAL LETTER U WITH RING ABOVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x0170, "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x0162, "LATIN CAPITAL LETTER T WITH CEDILLA"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x0155, "LATIN SMALL LETTER R WITH ACUTE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x0103, "LATIN SMALL LETTER A WITH BREVE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x013a, "LATIN SMALL LETTER L WITH ACUTE"},
	{0xe6, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x011b, "LATIN SMALL LETTER E WITH CARON"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x010f, "LATIN SMALL LETTER D WITH CARON"},
	{0xf0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xf1, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xf2, 0x0148, "LATIN SMALL LETTER N WITH CARON"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x0151, "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x0159, "LATIN SMALL LETTER R WITH CARON"},
	{0xf9, 0x016f, "LATIN SMALL LETTER U WITH RING ABOVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x0171, "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x0163, "LATIN SMALL LETTER T WITH CEDILLA"},
	{0xff, 0x02d9, "DOT ABOVE"}
};
const size_t n_iso_8859_2_table_size = sizeof(p_iso_8859_2_table) / sizeof(p_iso_8859_2_table[0]);

/**
 *	@brief iso-8859-3 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-3:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-3:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-3 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-3 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_3_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0126, "LATIN CAPITAL LETTER H WITH STROKE"},
	{0xa2, 0x02d8, "BREVE"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa6, 0x0124, "LATIN CAPITAL LETTER H WITH CIRCUMFLEX"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0xaa, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xab, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0xac, 0x0134, "LATIN CAPITAL LETTER J WITH CIRCUMFLEX"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xaf, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x0127, "LATIN SMALL LETTER H WITH STROKE"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x0125, "LATIN SMALL LETTER H WITH CIRCUMFLEX"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xba, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xbb, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xbc, 0x0135, "LATIN SMALL LETTER J WITH CIRCUMFLEX"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbf, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x010a, "LATIN CAPITAL LETTER C WITH DOT ABOVE"},
	{0xc6, 0x0108, "LATIN CAPITAL LETTER C WITH CIRCUMFLEX"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x0120, "LATIN CAPITAL LETTER G WITH DOT ABOVE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x011c, "LATIN CAPITAL LETTER G WITH CIRCUMFLEX"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x016c, "LATIN CAPITAL LETTER U WITH BREVE"},
	{0xde, 0x015c, "LATIN CAPITAL LETTER S WITH CIRCUMFLEX"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x010b, "LATIN SMALL LETTER C WITH DOT ABOVE"},
	{0xe6, 0x0109, "LATIN SMALL LETTER C WITH CIRCUMFLEX"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x0121, "LATIN SMALL LETTER G WITH DOT ABOVE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x011d, "LATIN SMALL LETTER G WITH CIRCUMFLEX"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x016d, "LATIN SMALL LETTER U WITH BREVE"},
	{0xfe, 0x015d, "LATIN SMALL LETTER S WITH CIRCUMFLEX"},
	{0xff, 0x02d9, "DOT ABOVE"}
};
const size_t n_iso_8859_3_table_size = sizeof(p_iso_8859_3_table) / sizeof(p_iso_8859_3_table[0]);

/**
 *	@brief iso-8859-4 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-4:1998 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-4:1998 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-4 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-4 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_4_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa2, 0x0138, "LATIN SMALL LETTER KRA"},
	{0xa3, 0x0156, "LATIN CAPITAL LETTER R WITH CEDILLA"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x0128, "LATIN CAPITAL LETTER I WITH TILDE"},
	{0xa6, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xaa, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0xab, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0xac, 0x0166, "LATIN CAPITAL LETTER T WITH STROKE"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xb2, 0x02db, "OGONEK"},
	{0xb3, 0x0157, "LATIN SMALL LETTER R WITH CEDILLA"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x0129, "LATIN SMALL LETTER I WITH TILDE"},
	{0xb6, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xb7, 0x02c7, "CARON"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xba, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0xbb, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0xbc, 0x0167, "LATIN SMALL LETTER T WITH STROKE"},
	{0xbd, 0x014a, "LATIN CAPITAL LETTER ENG"},
	{0xbe, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xbf, 0x014b, "LATIN SMALL LETTER ENG"},
	{0xc0, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xd0, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd1, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xd2, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xd3, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x0168, "LATIN CAPITAL LETTER U WITH TILDE"},
	{0xde, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0xf0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xf1, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xf2, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0xf3, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x0169, "LATIN SMALL LETTER U WITH TILDE"},
	{0xfe, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xff, 0x02d9, "DOT ABOVE"}
};
const size_t n_iso_8859_4_table_size = sizeof(p_iso_8859_4_table) / sizeof(p_iso_8859_4_table[0]);

/**
 *	@brief iso-8859-5 to unicode mapping table
 *
 *
 *	Name:             ISO 8859-5:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-5:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-5 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-5 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_5_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xa2, 0x0402, "CYRILLIC CAPITAL LETTER DJE"},
	{0xa3, 0x0403, "CYRILLIC CAPITAL LETTER GJE"},
	{0xa4, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0xa5, 0x0405, "CYRILLIC CAPITAL LETTER DZE"},
	{0xa6, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xa7, 0x0407, "CYRILLIC CAPITAL LETTER YI"},
	{0xa8, 0x0408, "CYRILLIC CAPITAL LETTER JE"},
	{0xa9, 0x0409, "CYRILLIC CAPITAL LETTER LJE"},
	{0xaa, 0x040a, "CYRILLIC CAPITAL LETTER NJE"},
	{0xab, 0x040b, "CYRILLIC CAPITAL LETTER TSHE"},
	{0xac, 0x040c, "CYRILLIC CAPITAL LETTER KJE"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x040e, "CYRILLIC CAPITAL LETTER SHORT U"},
	{0xaf, 0x040f, "CYRILLIC CAPITAL LETTER DZHE"},
	{0xb0, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xb1, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xb2, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xb3, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xb4, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xb5, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xb6, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xb7, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xb8, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xb9, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xba, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xbb, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xbc, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xbd, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xbe, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xbf, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xc0, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xc1, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xc2, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xc3, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xc4, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xc5, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xc6, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xc7, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xc8, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xc9, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xca, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0xcb, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xcc, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xcd, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xce, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0xcf, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xd0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xd1, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xd2, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xd3, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xd4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xd5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xd6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xd7, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xd8, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xd9, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xda, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xdb, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xdc, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xdd, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xde, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xdf, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xe0, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xe1, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xe2, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xe3, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xe4, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xe5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xe6, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xe7, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xe8, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xe9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xea, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xeb, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xec, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xed, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xee, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xef, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xf0, 0x2116, "NUMERO SIGN"},
	{0xf1, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xf2, 0x0452, "CYRILLIC SMALL LETTER DJE"},
	{0xf3, 0x0453, "CYRILLIC SMALL LETTER GJE"},
	{0xf4, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0xf5, 0x0455, "CYRILLIC SMALL LETTER DZE"},
	{0xf6, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xf7, 0x0457, "CYRILLIC SMALL LETTER YI"},
	{0xf8, 0x0458, "CYRILLIC SMALL LETTER JE"},
	{0xf9, 0x0459, "CYRILLIC SMALL LETTER LJE"},
	{0xfa, 0x045a, "CYRILLIC SMALL LETTER NJE"},
	{0xfb, 0x045b, "CYRILLIC SMALL LETTER TSHE"},
	{0xfc, 0x045c, "CYRILLIC SMALL LETTER KJE"},
	{0xfd, 0x00a7, "SECTION SIGN"},
	{0xfe, 0x045e, "CYRILLIC SMALL LETTER SHORT U"},
	{0xff, 0x045f, "CYRILLIC SMALL LETTER DZHE"}
};
const size_t n_iso_8859_5_table_size = sizeof(p_iso_8859_5_table) / sizeof(p_iso_8859_5_table[0]);

/**
 *	@brief iso-8859-6 to unicode mapping table
 *
 *
 *	Name:             ISO 8859-6:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-6:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-6 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-6 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *	0x30..0x39 remapped to the ASCII digits (U+0030..U+0039) instead
 *	of the Arabic digits (U+0660..U+0669).
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_6_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xac, 0x060c, "ARABIC COMMA"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xbb, 0x061b, "ARABIC SEMICOLON"},
	{0xbf, 0x061f, "ARABIC QUESTION MARK"},
	{0xc1, 0x0621, "ARABIC LETTER HAMZA"},
	{0xc2, 0x0622, "ARABIC LETTER ALEF WITH MADDA ABOVE"},
	{0xc3, 0x0623, "ARABIC LETTER ALEF WITH HAMZA ABOVE"},
	{0xc4, 0x0624, "ARABIC LETTER WAW WITH HAMZA ABOVE"},
	{0xc5, 0x0625, "ARABIC LETTER ALEF WITH HAMZA BELOW"},
	{0xc6, 0x0626, "ARABIC LETTER YEH WITH HAMZA ABOVE"},
	{0xc7, 0x0627, "ARABIC LETTER ALEF"},
	{0xc8, 0x0628, "ARABIC LETTER BEH"},
	{0xc9, 0x0629, "ARABIC LETTER TEH MARBUTA"},
	{0xca, 0x062a, "ARABIC LETTER TEH"},
	{0xcb, 0x062b, "ARABIC LETTER THEH"},
	{0xcc, 0x062c, "ARABIC LETTER JEEM"},
	{0xcd, 0x062d, "ARABIC LETTER HAH"},
	{0xce, 0x062e, "ARABIC LETTER KHAH"},
	{0xcf, 0x062f, "ARABIC LETTER DAL"},
	{0xd0, 0x0630, "ARABIC LETTER THAL"},
	{0xd1, 0x0631, "ARABIC LETTER REH"},
	{0xd2, 0x0632, "ARABIC LETTER ZAIN"},
	{0xd3, 0x0633, "ARABIC LETTER SEEN"},
	{0xd4, 0x0634, "ARABIC LETTER SHEEN"},
	{0xd5, 0x0635, "ARABIC LETTER SAD"},
	{0xd6, 0x0636, "ARABIC LETTER DAD"},
	{0xd7, 0x0637, "ARABIC LETTER TAH"},
	{0xd8, 0x0638, "ARABIC LETTER ZAH"},
	{0xd9, 0x0639, "ARABIC LETTER AIN"},
	{0xda, 0x063a, "ARABIC LETTER GHAIN"},
	{0xe0, 0x0640, "ARABIC TATWEEL"},
	{0xe1, 0x0641, "ARABIC LETTER FEH"},
	{0xe2, 0x0642, "ARABIC LETTER QAF"},
	{0xe3, 0x0643, "ARABIC LETTER KAF"},
	{0xe4, 0x0644, "ARABIC LETTER LAM"},
	{0xe5, 0x0645, "ARABIC LETTER MEEM"},
	{0xe6, 0x0646, "ARABIC LETTER NOON"},
	{0xe7, 0x0647, "ARABIC LETTER HEH"},
	{0xe8, 0x0648, "ARABIC LETTER WAW"},
	{0xe9, 0x0649, "ARABIC LETTER ALEF MAKSURA"},
	{0xea, 0x064a, "ARABIC LETTER YEH"},
	{0xeb, 0x064b, "ARABIC FATHATAN"},
	{0xec, 0x064c, "ARABIC DAMMATAN"},
	{0xed, 0x064d, "ARABIC KASRATAN"},
	{0xee, 0x064e, "ARABIC FATHA"},
	{0xef, 0x064f, "ARABIC DAMMA"},
	{0xf0, 0x0650, "ARABIC KASRA"},
	{0xf1, 0x0651, "ARABIC SHADDA"},
	{0xf2, 0x0652, "ARABIC SUKUN"}
};
const size_t n_iso_8859_6_table_size = sizeof(p_iso_8859_6_table) / sizeof(p_iso_8859_6_table[0]);

/**
 *	@brief iso-8859-7 to unicode mapping table
 *
 *
 *	Name:             ISO 8859-7:2003 to Unicode
 *	Unicode version:  4.0
 *	Table version:    2.0
 *	Table format:     Format A
 *	Date:             2003-Nov-12
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-2003 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO 8859-7:2003 characters map into Unicode.
 *
 *	ISO 8859-7:1987 is equivalent to ISO-IR-126, ELOT 928,
 *	and ECMA 118. ISO 8859-7:2003 adds two currency signs 
 *	and one other character not in the earlier standard.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO 8859-7 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO 8859-7 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *	Remap 0xA1 to U+2018 (instead of 0x02BD) to match text of 8859-7
 *	Remap 0xA2 to U+2019 (instead of 0x02BC) to match text of 8859-7
 *
 *	2.0 version updates 1.0 version by adding mappings for the
 *	three newly added characters 0xA4, 0xA5, 0xAA.
 *
 *	Updated versions of this file may be found in:
 *		<http://www.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact the Unicode Consortium at:
 *	        <http://www.unicode.org/reporting.html>
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_7_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xa2, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x20ac, "EURO SIGN"},
	{0xa5, 0x20af, "DRACHMA SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x037a, "GREEK YPOGEGRAMMENI"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xaf, 0x2015, "HORIZONTAL BAR"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x0384, "GREEK TONOS"},
	{0xb5, 0x0385, "GREEK DIALYTIKA TONOS"},
	{0xb6, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0xb9, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0xba, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0xbf, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0xc0, 0x0390, "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
	{0xc1, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0xc2, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0xc3, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xc4, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0xc5, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0xc6, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0xc7, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0xc8, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xc9, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0xca, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0xcb, 0x039b, "GREEK CAPITAL LETTER LAMDA"},
	{0xcc, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0xcd, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0xce, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0xcf, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0xd0, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0xd1, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0xd3, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xd4, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0xd5, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0xd6, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xd7, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0xd8, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0xd9, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xda, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0xdb, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0xdc, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0xdd, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0xde, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0xdf, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xe0, 0x03b0, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"},
	{0xe1, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe2, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0xe3, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0xe4, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xe5, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xe6, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0xe7, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0xe8, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0xe9, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0xea, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0xeb, 0x03bb, "GREEK SMALL LETTER LAMDA"},
	{0xec, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xed, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xee, 0x03be, "GREEK SMALL LETTER XI"},
	{0xef, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xf0, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xf1, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xf2, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xf3, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xf4, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xf5, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xf6, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xf7, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xf8, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xf9, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xfa, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xfb, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xfc, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xfd, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xfe, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"}
};
const size_t n_iso_8859_7_table_size = sizeof(p_iso_8859_7_table) / sizeof(p_iso_8859_7_table[0]);

/**
 *	@brief iso-8859-8 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-8:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.1
 *	Table format:     Format A
 *	Date:             2000-Jan-03
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-8:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-8 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-8 order.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *       1.1 version updates to the published 8859-8:1999, correcting
 *          the mapping of 0xAF and adding mappings for LRM and RLM.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_8_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00d7, "MULTIPLICATION SIGN"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00f7, "DIVISION SIGN"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xdf, 0x2017, "DOUBLE LOW LINE"},
	{0xe0, 0x05d0, "HEBREW LETTER ALEF"},
	{0xe1, 0x05d1, "HEBREW LETTER BET"},
	{0xe2, 0x05d2, "HEBREW LETTER GIMEL"},
	{0xe3, 0x05d3, "HEBREW LETTER DALET"},
	{0xe4, 0x05d4, "HEBREW LETTER HE"},
	{0xe5, 0x05d5, "HEBREW LETTER VAV"},
	{0xe6, 0x05d6, "HEBREW LETTER ZAYIN"},
	{0xe7, 0x05d7, "HEBREW LETTER HET"},
	{0xe8, 0x05d8, "HEBREW LETTER TET"},
	{0xe9, 0x05d9, "HEBREW LETTER YOD"},
	{0xea, 0x05da, "HEBREW LETTER FINAL KAF"},
	{0xeb, 0x05db, "HEBREW LETTER KAF"},
	{0xec, 0x05dc, "HEBREW LETTER LAMED"},
	{0xed, 0x05dd, "HEBREW LETTER FINAL MEM"},
	{0xee, 0x05de, "HEBREW LETTER MEM"},
	{0xef, 0x05df, "HEBREW LETTER FINAL NUN"},
	{0xf0, 0x05e0, "HEBREW LETTER NUN"},
	{0xf1, 0x05e1, "HEBREW LETTER SAMEKH"},
	{0xf2, 0x05e2, "HEBREW LETTER AYIN"},
	{0xf3, 0x05e3, "HEBREW LETTER FINAL PE"},
	{0xf4, 0x05e4, "HEBREW LETTER PE"},
	{0xf5, 0x05e5, "HEBREW LETTER FINAL TSADI"},
	{0xf6, 0x05e6, "HEBREW LETTER TSADI"},
	{0xf7, 0x05e7, "HEBREW LETTER QOF"},
	{0xf8, 0x05e8, "HEBREW LETTER RESH"},
	{0xf9, 0x05e9, "HEBREW LETTER SHIN"},
	{0xfa, 0x05ea, "HEBREW LETTER TAV"},
	{0xfd, 0x200e, "LEFT-TO-RIGHT MARK"},
	{0xfe, 0x200f, "RIGHT-TO-LEFT MARK"}
};
const size_t n_iso_8859_8_table_size = sizeof(p_iso_8859_8_table) / sizeof(p_iso_8859_8_table[0]);

/**
 *	@brief iso-8859-9 to unicode mapping table
 *
 *
 *	Name:             ISO/IEC 8859-9:1999 to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on magnetic media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       ISO/IEC 8859-9:1999 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the ISO/IEC 8859-9 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in ISO/IEC 8859-9 order.
 *
 *	ISO/IEC 8859-9 is also equivalent to ISO-IR-148.
 *
 *	Version history
 *	1.0 version updates 0.1 version by adding mappings for all
 *	control characters.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_iso_8859_9_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0xde, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xfe, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_iso_8859_9_table_size = sizeof(p_iso_8859_9_table) / sizeof(p_iso_8859_9_table[0]);

/**
 *	@brief atarist to unicode mapping table
 *
 * File encoding:    UTF-8
 * Name:             AtariST to Unicode
 * Unicode version:  4.0
 * Table version:    1.2
 * Table format:     Format A
 * Date:             2011 October 27
 * Authors:          Philippe Verdy <verdyp AT gmail.com>
 *                   Bruno Haible <bruno AT clisp.org>
 *
 * Copyright (c) 1998 - 2011 Unicode, Inc.  All Rights reserved.
 *
 * This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 * No claims are made as to fitness for any particular purpose.  No
 * warranties of any kind are expressed or implied.  The recipient
 * agrees to determine applicability of information provided.  If this
 * file has been provided on optical media by Unicode, Inc., the sole
 * remedy for any claim will be exchange of defective media within 90
 * days of receipt.
 *
 * Unicode, Inc. hereby grants the right to freely use the information
 * supplied in this file in the creation of products supporting the
 * Unicode Standard, and to make copies of this file in any form for
 * internal or external distribution as long as this notice remains
 * attached.
 *
 * General notes:
 *
 * This table contains the data the Unicode Consortium has on how
 * Atari ST/TT (TOS) characters map into Unicode.
 *
 * Format:  Three tab-separated columns
 *   Column 1 is the Atari ST code (in hex as 0xXX)
 *   Column 2 is the Unicode (in hex as 0xXXXX)
 *   Column 3 is a comment field containing:
 *            - the Unicode name (follows a comment sign, '\#'),
 *            - the actual character or <symbol> (follows a comment sign, '\#'),
 *            - some backward compatible character set mappings.
 *
 * The entries are in Atari ST order.
 *
 * Version history
 * 1.2 Thanks to Ulf Zibis for his report (no mapping changes).
 *     - Fix the correct Unicode character names for
 *       code points 0x00C6, 0x00E6, 0x00D6, 0x00DC, 0x00A8, 0x00B4.
 *     - Fix editorial issues from from the previous HTML reporting form.
 *     - Document that the file is UTF-8 encoded.
 * 1.1 Thanks to Bruno Haible.
 *     - swap the positions of GREEK SMALL LETTER BETA and LATIN SHARP S
 *       on code positions 0x9E and 0xE1. So AtariST mapped the LATIN SHARP S
 *       differently than PC codepages 437 and 850 (was it true on the TT in 1992?).
 *     - Fix the correct code point 0x00B3 instead of 0x22B3 for SUPERSCRIPT THREE,
 *       on code position 0xFE.
 *     - Fix the spelling of a \"GREEP\" letter name to \"GREEK\" (no code changed)
 *     - Change the mapping of code 0xEE from GREEK SMALL LETTER EPSILON to the
 *       mathematical ELEMENT OF SIGN, on code position 0xEE.
 * 1.0 version new, by Philippe Verdy.
 *
 * Updated versions of this file may be found in:
 *   <ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 * Any comments or problems, contact http://www.unicode.org/reporting.html
 *
 */
const CUnicodeMapping::TCharacterName p_atarist_table[] = {
	{0x00, 0x0000, "NULL                                      	# <NUL>	ISO646"},
	{0x01, 0x0001, "START OF HEADING                          	# <SOH>	ISO646"},
	{0x02, 0x0002, "START OF TEXT                             	# <STX>	ISO646"},
	{0x03, 0x0003, "END OF TEXT                               	# <ETX>	ISO646"},
	{0x04, 0x0004, "END OF TRANSMISSION                       	# <EOT>	ISO646"},
	{0x05, 0x0005, "ENQUIRY                                   	# <ENQ>	ISO646"},
	{0x06, 0x0006, "ACKNOWLEDGE                               	# <ACK>	ISO646"},
	{0x07, 0x0007, "BELL                                      	# <BEL>	ISO646"},
	{0x08, 0x0008, "BACKSPACE                                 	# <BS>	ISO646"},
	{0x09, 0x0009, "HORIZONTAL TABULATION                     	# <TAB>	ISO646"},
	{0x0a, 0x000a, "LINE FEED                                 	# <LF>	ISO646"},
	{0x0b, 0x000b, "VERTICAL TABULATION                       	# <VT>	ISO646"},
	{0x0c, 0x000c, "FORM FEED                                 	# <FF>	ISO646"},
	{0x0d, 0x000d, "CARRIAGE RETURN                           	# <CR>	ISO646"},
	{0x0e, 0x000e, "SHIFT OUT                                 	# <SO>	ISO646"},
	{0x0f, 0x000f, "SHIFT IN                                  	# <SI>	ISO646"},
	{0x10, 0x0010, "DATA LINK ESCAPE                          	# <DLE>	ISO646"},
	{0x11, 0x0011, "DEVICE CONTROL ONE                        	# <DC1>	ISO646"},
	{0x12, 0x0012, "DEVICE CONTROL TWO                        	# <DC2>	ISO646"},
	{0x13, 0x0013, "DEVICE CONTROL THREE                      	# <DC3>	ISO646"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR                       	# <DC4>	ISO646"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE                      	# <NAK>	ISO646"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE                          	# <SYN>	ISO646"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK                 	# <ETB>	ISO646"},
	{0x18, 0x0018, "CANCEL                                    	# <CAN>	ISO646"},
	{0x19, 0x0019, "END OF MEDIUM                             	# <EM>	ISO646"},
	{0x1a, 0x001a, "SUBSTITUTE                                	# <SUB>	ISO646"},
	{0x1b, 0x001b, "ESCAPE                                    	# <ESC>	ISO646"},
	{0x1c, 0x001c, "FILE SEPARATOR                            	# <FS>	ISO646"},
	{0x1d, 0x001d, "GROUP SEPARATOR                           	# <GS>	ISO646"},
	{0x1e, 0x001e, "RECORD SEPARATOR                          	# <RS>	ISO646"},
	{0x1f, 0x001f, "UNIT SEPARATOR                            	# <US>	ISO646"},
	{0x20, 0x0020, "SPACE                                     	# <SP>	ISO646"},
	{0x21, 0x0021, "EXCLAMATION MARK                          	# !	ASCII"},
	{0x22, 0x0022, "QUOTATION MARK                            	# \"	ASCII"},
	{0x23, 0x0023, "NUMBER SIGN                               	# #	ASCII"},
	{0x24, 0x0024, "DOLLAR SIGN                               	# $	ASCII"},
	{0x25, 0x0025, "PERCENT SIGN                              	# %	ASCII"},
	{0x26, 0x0026, "AMPERSAND                                 	# &	ASCII"},
	{0x27, 0x0027, "APOSTROPHE                                	# '	ASCII"},
	{0x28, 0x0028, "LEFT PARENTHESIS                          	# (	ASCII"},
	{0x29, 0x0029, "RIGHT PARENTHESIS                         	# )	ASCII"},
	{0x2a, 0x002a, "ASTERISK                                  	# *	ASCII"},
	{0x2b, 0x002b, "PLUS SIGN                                 	# +	ASCII"},
	{0x2c, 0x002c, "COMMA                                     	# ,	ASCII"},
	{0x2d, 0x002d, "HYPHEN-MINUS                              	# -	ASCII"},
	{0x2e, 0x002e, "FULL STOP                                 	# .	ASCII"},
	{0x2f, 0x002f, "SOLIDUS                                   	# /	ASCII"},
	{0x30, 0x0030, "DIGIT ZERO                                	# 0	ASCII"},
	{0x31, 0x0031, "DIGIT ONE                                 	# 1	ASCII"},
	{0x32, 0x0032, "DIGIT TWO                                 	# 2	ASCII"},
	{0x33, 0x0033, "DIGIT THREE                               	# 3	ASCII"},
	{0x34, 0x0034, "DIGIT FOUR                                	# 4	ASCII"},
	{0x35, 0x0035, "DIGIT FIVE                                	# 5	ASCII"},
	{0x36, 0x0036, "DIGIT SIX                                 	# 6	ASCII"},
	{0x37, 0x0037, "DIGIT SEVEN                               	# 7	ASCII"},
	{0x38, 0x0038, "DIGIT EIGHT                               	# 8	ASCII"},
	{0x39, 0x0039, "DIGIT NINE                                	# 9	ASCII"},
	{0x3a, 0x003a, "COLON                                     	# :	ASCII"},
	{0x3b, 0x003b, "SEMICOLON                                 	# ;	ASCII"},
	{0x3c, 0x003c, "LESS-THAN SIGN                            	# <	ASCII"},
	{0x3d, 0x003d, "EQUALS SIGN                               	# =	ASCII"},
	{0x3e, 0x003e, "GREATER-THAN SIGN                         	# >	ASCII"},
	{0x3f, 0x003f, "QUESTION MARK                             	# ?	ASCII"},
	{0x40, 0x0040, "COMMERCIAL AT                             	# @	ASCII"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A                    	# A	ASCII"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B                    	# B	ASCII"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C                    	# C	ASCII"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D                    	# D	ASCII"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E                    	# E	ASCII"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F                    	# F	ASCII"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G                    	# G	ASCII"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H                    	# H	ASCII"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I                    	# I	ASCII"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J                    	# J	ASCII"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K                    	# K	ASCII"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L                    	# L	ASCII"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M                    	# M	ASCII"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N                    	# N	ASCII"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O                    	# O	ASCII"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P                    	# P	ASCII"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q                    	# Q	ASCII"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R                    	# R	ASCII"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S                    	# S	ASCII"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T                    	# T	ASCII"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U                    	# U	ASCII"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V                    	# V	ASCII"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W                    	# W	ASCII"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X                    	# X	ASCII"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y                    	# Y	ASCII"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z                    	# Z	ASCII"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET                       	# [	ASCII"},
	{0x5c, 0x005c, "REVERSE SOLIDUS                           	# \\	ASCII"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET                      	# ]	ASCII"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT                         	# ^	ASCII"},
	{0x5f, 0x005f, "LOW LINE                                  	# _	ASCII"},
	{0x60, 0x0060, "GRAVE ACCENT                              	# `	ASCII"},
	{0x61, 0x0061, "LATIN SMALL LETTER A                      	# a	ASCII"},
	{0x62, 0x0062, "LATIN SMALL LETTER B                      	# b	ASCII"},
	{0x63, 0x0063, "LATIN SMALL LETTER C                      	# c	ASCII"},
	{0x64, 0x0064, "LATIN SMALL LETTER D                      	# d	ASCII"},
	{0x65, 0x0065, "LATIN SMALL LETTER E                      	# e	ASCII"},
	{0x66, 0x0066, "LATIN SMALL LETTER F                      	# f	ASCII"},
	{0x67, 0x0067, "LATIN SMALL LETTER G                      	# g	ASCII"},
	{0x68, 0x0068, "LATIN SMALL LETTER H                      	# h	ASCII"},
	{0x69, 0x0069, "LATIN SMALL LETTER I                      	# i	ASCII"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J                      	# j	ASCII"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K                      	# k	ASCII"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L                      	# l	ASCII"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M                      	# m	ASCII"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N                      	# n	ASCII"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O                      	# o	ASCII"},
	{0x70, 0x0070, "LATIN SMALL LETTER P                      	# p	ASCII"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q                      	# q	ASCII"},
	{0x72, 0x0072, "LATIN SMALL LETTER R                      	# r	ASCII"},
	{0x73, 0x0073, "LATIN SMALL LETTER S                      	# s	ASCII"},
	{0x74, 0x0074, "LATIN SMALL LETTER T                      	# t	ASCII"},
	{0x75, 0x0075, "LATIN SMALL LETTER U                      	# u	ASCII"},
	{0x76, 0x0076, "LATIN SMALL LETTER V                      	# v	ASCII"},
	{0x77, 0x0077, "LATIN SMALL LETTER W                      	# w	ASCII"},
	{0x78, 0x0078, "LATIN SMALL LETTER X                      	# x	ASCII"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y                      	# y	ASCII"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z                      	# z	ASCII"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET                        	# {	ASCII"},
	{0x7c, 0x007c, "VERTICAL LINE                             	# |	ASCII"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET                       	# }	ASCII"},
	{0x7e, 0x007e, "TILDE                                     	# ~	ASCII"},
	{0x7f, 0x007f, "DELETE                                    	# <DEL>	ISO646"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA       	# Ç	CP437,CP850"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS       	# ü	CP437,CP850"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE           	# é	CP437,CP850"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX      	# â	CP437,CP850"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS       	# ä	CP437,CP850"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE           	# à	CP437,CP850"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE      	# å	CP437,CP850"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA         	# ç	CP437,CP850"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX      	# ê	CP437,CP850"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS       	# ë	CP437,CP850"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE           	# è	CP437,CP850"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS       	# ï	CP437,CP850"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX      	# î	CP437,CP850"},
	{0x8d, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE           	# ì	CP437,CP850"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS     	# Ä	CP437,CP850"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE    	# Å	CP437,CP850"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE         	# É	CP437,CP850"},
	{0x91, 0x00e6, "LATIN SMALL LETTER AE                     	# æ	CP437,CP850"},
	{0x92, 0x00c6, "LATIN CAPITAL LETTER AE                   	# Æ	CP437,CP850"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX      	# ô	CP437,CP850"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS       	# ö	CP437,CP850"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE           	# ò	CP437,CP850"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX      	# û	CP437,CP850"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE           	# ù	CP437,CP850"},
	{0x98, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS       	# ÿ	CP437,CP850"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS     	# Ö	CP437,CP850"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS     	# Ü	CP437,CP850"},
	{0x9b, 0x00a2, "CENT SIGN                                 	# ¢	CP437"},
	{0x9c, 0x00a3, "POUND SIGN                                	# £	CP437,CP850"},
	{0x9d, 0x00a5, "YEN SIGN                                  	# ¥	CP437"},
	{0x9e, 0x00df, "LATIN SMALL LETTER SHARP S                	# ß	AtariST"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK            	# ƒ	CP437,CP850"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE           	# á	CP437,CP850"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE           	# í	CP437,CP850"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE           	# ó	CP437,CP850"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE           	# ú	CP437,CP850"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE           	# ñ	CP437,CP850"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE         	# Ñ	CP437,CP850"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR                	# ª	CP437,CP850"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR               	# º	CP437,CP850"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK                    	# ¿	CP437,CP850"},
	{0xa9, 0x2310, "REVERSED NOT SIGN                         	# ⌐	CP437"},
	{0xaa, 0x00ac, "NOT SIGN                                  	# ¬	CP437,CP850"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF                  	# ½	CP437,CP850"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER               	# ¼	CP437,CP850"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK                 	# ¡	CP437,CP850"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK 	# «	CP437,CP850"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK	# »	CP437,CP850"},
	{0xb0, 0x00e3, "LATIN SMALL LETTER A WITH TILDE           	# ã	AtariST"},
	{0xb1, 0x00f5, "LATIN SMALL LETTER O WITH TILDE           	# õ	AtariST"},
	{0xb2, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE        	# Ø	AtariST"},
	{0xb3, 0x00f8, "LATIN SMALL LETTER O WITH STROKE          	# ø	AtariST"},
	{0xb4, 0x0153, "LATIN SMALL LIGATURE OE                   	# œ	AtariST"},
	{0xb5, 0x0152, "LATIN CAPITAL LIGATURE OE                 	# Œ	AtariST"},
	{0xb6, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE         	# À	AtariST"},
	{0xb7, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE         	# Ã	AtariST"},
	{0xb8, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE         	# Õ	AtariST"},
	{0xb9, 0x00a8, "DIAERESIS                                 	# ¨	AtariST"},
	{0xba, 0x00b4, "ACUTE ACCENT                              	# ´	AtariST"},
	{0xbb, 0x2020, "DAGGER                                    	# †	AtariST"},
	{0xbc, 0x00b6, "PILCROW SIGN                              	# ¶	AtariST"},
	{0xbd, 0x00a9, "COPYRIGHT SIGN                            	# ©	AtariST"},
	{0xbe, 0x00ae, "REGISTERED SIGN                           	# ®	AtariST"},
	{0xbf, 0x2122, "TRADE MARK SIGN                           	# ™	AtariST"},
	{0xc0, 0x0133, "LATIN SMALL LIGATURE IJ                   	# ĳ	AtariST"},
	{0xc1, 0x0132, "LATIN CAPITAL LIGATURE IJ                 	# Ĳ	AtariST"},
	{0xc2, 0x05d0, "HEBREW LETTER ALEF                        	# א	AtariST"},
	{0xc3, 0x05d1, "HEBREW LETTER BET                         	# ב	AtariST"},
	{0xc4, 0x05d2, "HEBREW LETTER GIMEL                       	# ג	AtariST"},
	{0xc5, 0x05d3, "HEBREW LETTER DALET                       	# ד	AtariST"},
	{0xc6, 0x05d4, "HEBREW LETTER HE                          	# ה	AtariST"},
	{0xc7, 0x05d5, "HEBREW LETTER VAV                         	# ו	AtariST"},
	{0xc8, 0x05d6, "HEBREW LETTER ZAYIN                       	# ז	AtariST"},
	{0xc9, 0x05d7, "HEBREW LETTER HET                         	# ח	AtariST"},
	{0xca, 0x05d8, "HEBREW LETTER TET                         	# ט	AtariST"},
	{0xcb, 0x05d9, "HEBREW LETTER YOD                         	# י	AtariST"},
	{0xcc, 0x05db, "HEBREW LETTER KAF                         	# כ	AtariST"},
	{0xcd, 0x05dc, "HEBREW LETTER LAMED                       	# ל	AtariST"},
	{0xce, 0x05de, "HEBREW LETTER MEM                         	# מ	AtariST"},
	{0xcf, 0x05e0, "HEBREW LETTER NUN                         	# נ	AtariST"},
	{0xd0, 0x05e1, "HEBREW LETTER SAMEKH                      	# ס	AtariST"},
	{0xd1, 0x05e2, "HEBREW LETTER AYIN                        	# ע	AtariST"},
	{0xd2, 0x05e4, "HEBREW LETTER PE                          	# פ	AtariST"},
	{0xd3, 0x05e6, "HEBREW LETTER TSADI                       	# צ	AtariST"},
	{0xd4, 0x05e7, "HEBREW LETTER QOF                         	# ק	AtariST"},
	{0xd5, 0x05e8, "HEBREW LETTER RESH                        	# ר	AtariST"},
	{0xd6, 0x05e9, "HEBREW LETTER SHIN                        	# ש	AtariST"},
	{0xd7, 0x05ea, "HEBREW LETTER TAV                         	# ת	AtariST"},
	{0xd8, 0x05df, "HEBREW LETTER FINAL NUN                   	# ן	AtariST"},
	{0xd9, 0x05da, "HEBREW LETTER FINAL KAF                   	# ך	AtariST"},
	{0xda, 0x05dd, "HEBREW LETTER FINAL MEM                   	# ם	AtariST"},
	{0xdb, 0x05e3, "HEBREW LETTER FINAL PE                    	# ף	AtariST"},
	{0xdc, 0x05e5, "HEBREW LETTER FINAL TSADI                 	# ץ	AtariST"},
	{0xdd, 0x00a7, "SECTION SIGN                              	# §	AtariST"},
	{0xde, 0x2227, "LOGICAL AND                               	# ∧	AtariST"},
	{0xdf, 0x221e, "INFINITY                                  	# ∞	AtariST"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA                  	# α	CP437,CP850"},
	{0xe1, 0x03b2, "GREEK SMALL LETTER BETA                   	# ß	AtariST"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA                	# Γ	CP437,CP850"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI                     	# π	CP437,CP850"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA                	# Σ	CP437,CP850"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA                  	# σ	CP437,CP850"},
	{0xe6, 0x00b5, "MICRO SIGN                                	# µ	CP437,CP850"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU                    	# τ	CP437,CP850"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI                  	# Φ	CP437,CP850"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA                	# Θ	CP437,CP850"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA                	# Ω	CP437,CP850"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA                  	# δ	CP437,CP850"},
	{0xec, 0x222e, "CONTOUR INTEGRAL                          	# ∮	AtariST"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI                    	# φ	CP437,CP850"},
	{0xee, 0x2208, "ELEMENT OF SIGN                           	# ∈	AtariST"},
	{0xef, 0x2229, "INTERSECTION                              	# ∩	CP437,CP850"},
	{0xf0, 0x2261, "IDENTICAL TO                              	# ≡	CP437,CP850"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN                           	# ±	CP437,CP850"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO                  	# ≥	CP437,CP850"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO                     	# ≤	CP437,CP850"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL                         	# ⌠	CP437,CP850"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL                      	# ⌡	CP437,CP850"},
	{0xf6, 0x00f7, "DIVISION SIGN                             	# ÷	CP437,CP850"},
	{0xf7, 0x2248, "ALMOST EQUAL TO                           	# ≈	CP437,CP850"},
	{0xf8, 0x00b0, "DEGREE SIGN                               	# °	CP437,CP850"},
	{0xf9, 0x2219, "BULLET OPERATOR                           	# ∙	CP437,CP850"},
	{0xfa, 0x00b7, "MIDDLE DOT                                	# ·	CP437,CP850"},
	{0xfb, 0x221a, "SQUARE ROOT                               	# √	CP437,CP850"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N          	# ⁿ	CP437,CP850"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO                           	# ²	CP437,CP850"},
	{0xfe, 0x00b3, "SUPERSCRIPT THREE                         	# ³	AtariST"},
	{0xff, 0x00af, "MACRON                                    	# ¯	AtariST"}
};
const size_t n_atarist_table_size = sizeof(p_atarist_table) / sizeof(p_atarist_table[0]);

/**
 *	@brief cp-037 to unicode mapping table
 *
 *
 *    Name:     cp037_IBMUSCanada to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *    
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp037_IBMUSCanada code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp037_IBMUSCanada order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_037_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x009c, "CONTROL"},
	{0x05, 0x0009, "HORIZONTAL TABULATION"},
	{0x06, 0x0086, "CONTROL"},
	{0x07, 0x007f, "DELETE"},
	{0x08, 0x0097, "CONTROL"},
	{0x09, 0x008d, "CONTROL"},
	{0x0a, 0x008e, "CONTROL"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x009d, "CONTROL"},
	{0x15, 0x0085, "CONTROL"},
	{0x16, 0x0008, "BACKSPACE"},
	{0x17, 0x0087, "CONTROL"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x0092, "CONTROL"},
	{0x1b, 0x008f, "CONTROL"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0080, "CONTROL"},
	{0x21, 0x0081, "CONTROL"},
	{0x22, 0x0082, "CONTROL"},
	{0x23, 0x0083, "CONTROL"},
	{0x24, 0x0084, "CONTROL"},
	{0x25, 0x000a, "LINE FEED"},
	{0x26, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x27, 0x001b, "ESCAPE"},
	{0x28, 0x0088, "CONTROL"},
	{0x29, 0x0089, "CONTROL"},
	{0x2a, 0x008a, "CONTROL"},
	{0x2b, 0x008b, "CONTROL"},
	{0x2c, 0x008c, "CONTROL"},
	{0x2d, 0x0005, "ENQUIRY"},
	{0x2e, 0x0006, "ACKNOWLEDGE"},
	{0x2f, 0x0007, "BELL"},
	{0x30, 0x0090, "CONTROL"},
	{0x31, 0x0091, "CONTROL"},
	{0x32, 0x0016, "SYNCHRONOUS IDLE"},
	{0x33, 0x0093, "CONTROL"},
	{0x34, 0x0094, "CONTROL"},
	{0x35, 0x0095, "CONTROL"},
	{0x36, 0x0096, "CONTROL"},
	{0x37, 0x0004, "END OF TRANSMISSION"},
	{0x38, 0x0098, "CONTROL"},
	{0x39, 0x0099, "CONTROL"},
	{0x3a, 0x009a, "CONTROL"},
	{0x3b, 0x009b, "CONTROL"},
	{0x3c, 0x0014, "DEVICE CONTROL FOUR"},
	{0x3d, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x3e, 0x009e, "CONTROL"},
	{0x3f, 0x001a, "SUBSTITUTE"},
	{0x40, 0x0020, "SPACE"},
	{0x41, 0x00a0, "NO-BREAK SPACE"},
	{0x42, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x43, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x44, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x45, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x46, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x47, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x48, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x49, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x4a, 0x00a2, "CENT SIGN"},
	{0x4b, 0x002e, "FULL STOP"},
	{0x4c, 0x003c, "LESS-THAN SIGN"},
	{0x4d, 0x0028, "LEFT PARENTHESIS"},
	{0x4e, 0x002b, "PLUS SIGN"},
	{0x4f, 0x007c, "VERTICAL LINE"},
	{0x50, 0x0026, "AMPERSAND"},
	{0x51, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x52, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x53, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x54, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x55, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x56, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x57, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x58, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x59, 0x00df, "LATIN SMALL LETTER SHARP S (GERMAN)"},
	{0x5a, 0x0021, "EXCLAMATION MARK"},
	{0x5b, 0x0024, "DOLLAR SIGN"},
	{0x5c, 0x002a, "ASTERISK"},
	{0x5d, 0x0029, "RIGHT PARENTHESIS"},
	{0x5e, 0x003b, "SEMICOLON"},
	{0x5f, 0x00ac, "NOT SIGN"},
	{0x60, 0x002d, "HYPHEN-MINUS"},
	{0x61, 0x002f, "SOLIDUS"},
	{0x62, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x63, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x64, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x65, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0x66, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0x67, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x68, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x69, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x6a, 0x00a6, "BROKEN BAR"},
	{0x6b, 0x002c, "COMMA"},
	{0x6c, 0x0025, "PERCENT SIGN"},
	{0x6d, 0x005f, "LOW LINE"},
	{0x6e, 0x003e, "GREATER-THAN SIGN"},
	{0x6f, 0x003f, "QUESTION MARK"},
	{0x70, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x71, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x72, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x73, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0x74, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x75, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0x76, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0x77, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0x78, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0x79, 0x0060, "GRAVE ACCENT"},
	{0x7a, 0x003a, "COLON"},
	{0x7b, 0x0023, "NUMBER SIGN"},
	{0x7c, 0x0040, "COMMERCIAL AT"},
	{0x7d, 0x0027, "APOSTROPHE"},
	{0x7e, 0x003d, "EQUALS SIGN"},
	{0x7f, 0x0022, "QUOTATION MARK"},
	{0x80, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x81, 0x0061, "LATIN SMALL LETTER A"},
	{0x82, 0x0062, "LATIN SMALL LETTER B"},
	{0x83, 0x0063, "LATIN SMALL LETTER C"},
	{0x84, 0x0064, "LATIN SMALL LETTER D"},
	{0x85, 0x0065, "LATIN SMALL LETTER E"},
	{0x86, 0x0066, "LATIN SMALL LETTER F"},
	{0x87, 0x0067, "LATIN SMALL LETTER G"},
	{0x88, 0x0068, "LATIN SMALL LETTER H"},
	{0x89, 0x0069, "LATIN SMALL LETTER I"},
	{0x8a, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8b, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8c, 0x00f0, "LATIN SMALL LETTER ETH (ICELANDIC)"},
	{0x8d, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0x8e, 0x00fe, "LATIN SMALL LETTER THORN (ICELANDIC)"},
	{0x8f, 0x00b1, "PLUS-MINUS SIGN"},
	{0x90, 0x00b0, "DEGREE SIGN"},
	{0x91, 0x006a, "LATIN SMALL LETTER J"},
	{0x92, 0x006b, "LATIN SMALL LETTER K"},
	{0x93, 0x006c, "LATIN SMALL LETTER L"},
	{0x94, 0x006d, "LATIN SMALL LETTER M"},
	{0x95, 0x006e, "LATIN SMALL LETTER N"},
	{0x96, 0x006f, "LATIN SMALL LETTER O"},
	{0x97, 0x0070, "LATIN SMALL LETTER P"},
	{0x98, 0x0071, "LATIN SMALL LETTER Q"},
	{0x99, 0x0072, "LATIN SMALL LETTER R"},
	{0x9a, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0x9b, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0x9c, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x9d, 0x00b8, "CEDILLA"},
	{0x9e, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x9f, 0x00a4, "CURRENCY SIGN"},
	{0xa0, 0x00b5, "MICRO SIGN"},
	{0xa1, 0x007e, "TILDE"},
	{0xa2, 0x0073, "LATIN SMALL LETTER S"},
	{0xa3, 0x0074, "LATIN SMALL LETTER T"},
	{0xa4, 0x0075, "LATIN SMALL LETTER U"},
	{0xa5, 0x0076, "LATIN SMALL LETTER V"},
	{0xa6, 0x0077, "LATIN SMALL LETTER W"},
	{0xa7, 0x0078, "LATIN SMALL LETTER X"},
	{0xa8, 0x0079, "LATIN SMALL LETTER Y"},
	{0xa9, 0x007a, "LATIN SMALL LETTER Z"},
	{0xaa, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xab, 0x00bf, "INVERTED QUESTION MARK"},
	{0xac, 0x00d0, "LATIN CAPITAL LETTER ETH (ICELANDIC)"},
	{0xad, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xae, 0x00de, "LATIN CAPITAL LETTER THORN (ICELANDIC)"},
	{0xaf, 0x00ae, "REGISTERED SIGN"},
	{0xb0, 0x005e, "CIRCUMFLEX ACCENT"},
	{0xb1, 0x00a3, "POUND SIGN"},
	{0xb2, 0x00a5, "YEN SIGN"},
	{0xb3, 0x00b7, "MIDDLE DOT"},
	{0xb4, 0x00a9, "COPYRIGHT SIGN"},
	{0xb5, 0x00a7, "SECTION SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xb8, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xb9, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xba, 0x005b, "LEFT SQUARE BRACKET"},
	{0xbb, 0x005d, "RIGHT SQUARE BRACKET"},
	{0xbc, 0x00af, "MACRON"},
	{0xbd, 0x00a8, "DIAERESIS"},
	{0xbe, 0x00b4, "ACUTE ACCENT"},
	{0xbf, 0x00d7, "MULTIPLICATION SIGN"},
	{0xc0, 0x007b, "LEFT CURLY BRACKET"},
	{0xc1, 0x0041, "LATIN CAPITAL LETTER A"},
	{0xc2, 0x0042, "LATIN CAPITAL LETTER B"},
	{0xc3, 0x0043, "LATIN CAPITAL LETTER C"},
	{0xc4, 0x0044, "LATIN CAPITAL LETTER D"},
	{0xc5, 0x0045, "LATIN CAPITAL LETTER E"},
	{0xc6, 0x0046, "LATIN CAPITAL LETTER F"},
	{0xc7, 0x0047, "LATIN CAPITAL LETTER G"},
	{0xc8, 0x0048, "LATIN CAPITAL LETTER H"},
	{0xc9, 0x0049, "LATIN CAPITAL LETTER I"},
	{0xca, 0x00ad, "SOFT HYPHEN"},
	{0xcb, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xcc, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xcd, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xce, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xcf, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xd0, 0x007d, "RIGHT CURLY BRACKET"},
	{0xd1, 0x004a, "LATIN CAPITAL LETTER J"},
	{0xd2, 0x004b, "LATIN CAPITAL LETTER K"},
	{0xd3, 0x004c, "LATIN CAPITAL LETTER L"},
	{0xd4, 0x004d, "LATIN CAPITAL LETTER M"},
	{0xd5, 0x004e, "LATIN CAPITAL LETTER N"},
	{0xd6, 0x004f, "LATIN CAPITAL LETTER O"},
	{0xd7, 0x0050, "LATIN CAPITAL LETTER P"},
	{0xd8, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0xd9, 0x0052, "LATIN CAPITAL LETTER R"},
	{0xda, 0x00b9, "SUPERSCRIPT ONE"},
	{0xdb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xde, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xdf, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xe0, 0x005c, "REVERSE SOLIDUS"},
	{0xe1, 0x00f7, "DIVISION SIGN"},
	{0xe2, 0x0053, "LATIN CAPITAL LETTER S"},
	{0xe3, 0x0054, "LATIN CAPITAL LETTER T"},
	{0xe4, 0x0055, "LATIN CAPITAL LETTER U"},
	{0xe5, 0x0056, "LATIN CAPITAL LETTER V"},
	{0xe6, 0x0057, "LATIN CAPITAL LETTER W"},
	{0xe7, 0x0058, "LATIN CAPITAL LETTER X"},
	{0xe8, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0xe9, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0xea, 0x00b2, "SUPERSCRIPT TWO"},
	{0xeb, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xec, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xed, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xf0, 0x0030, "DIGIT ZERO"},
	{0xf1, 0x0031, "DIGIT ONE"},
	{0xf2, 0x0032, "DIGIT TWO"},
	{0xf3, 0x0033, "DIGIT THREE"},
	{0xf4, 0x0034, "DIGIT FOUR"},
	{0xf5, 0x0035, "DIGIT FIVE"},
	{0xf6, 0x0036, "DIGIT SIX"},
	{0xf7, 0x0037, "DIGIT SEVEN"},
	{0xf8, 0x0038, "DIGIT EIGHT"},
	{0xf9, 0x0039, "DIGIT NINE"},
	{0xfa, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xfe, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xff, 0x009f, "CONTROL"}
};
const size_t n_cp_037_table_size = sizeof(p_cp_037_table) / sizeof(p_cp_037_table[0]);

/**
 *	@brief cp-1006 to unicode mapping table
 *
 *
 *	Name:             IBM CP1006  to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             1999 July 27
 *	Authors:          Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       IBM CP1006 characters map into Unicode.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the CP1006 code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in IBM CP1006 order.
 *
 *	Version history
 *	1.0 version new. 
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1006_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0080, "<control>"},
	{0x81, 0x0081, "<control>"},
	{0x82, 0x0082, "<control>"},
	{0x83, 0x0083, "<control>"},
	{0x84, 0x0084, "<control>"},
	{0x85, 0x0085, "<control>"},
	{0x86, 0x0086, "<control>"},
	{0x87, 0x0087, "<control>"},
	{0x88, 0x0088, "<control>"},
	{0x89, 0x0089, "<control>"},
	{0x8a, 0x008a, "<control>"},
	{0x8b, 0x008b, "<control>"},
	{0x8c, 0x008c, "<control>"},
	{0x8d, 0x008d, "<control>"},
	{0x8e, 0x008e, "<control>"},
	{0x8f, 0x008f, "<control>"},
	{0x90, 0x0090, "<control>"},
	{0x91, 0x0091, "<control>"},
	{0x92, 0x0092, "<control>"},
	{0x93, 0x0093, "<control>"},
	{0x94, 0x0094, "<control>"},
	{0x95, 0x0095, "<control>"},
	{0x96, 0x0096, "<control>"},
	{0x97, 0x0097, "<control>"},
	{0x98, 0x0098, "<control>"},
	{0x99, 0x0099, "<control>"},
	{0x9a, 0x009a, "<control>"},
	{0x9b, 0x009b, "<control>"},
	{0x9c, 0x009c, "<control>"},
	{0x9d, 0x009d, "<control>"},
	{0x9e, 0x009e, "<control>"},
	{0x9f, 0x009f, "<control>"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x06f0, "EXTENDED ARABIC-INDIC DIGIT ZERO"},
	{0xa2, 0x06f1, "EXTENDED ARABIC-INDIC DIGIT ONE"},
	{0xa3, 0x06f2, "EXTENDED ARABIC-INDIC DIGIT TWO"},
	{0xa4, 0x06f3, "EXTENDED ARABIC-INDIC DIGIT THREE"},
	{0xa5, 0x06f4, "EXTENDED ARABIC-INDIC DIGIT FOUR"},
	{0xa6, 0x06f5, "EXTENDED ARABIC-INDIC DIGIT FIVE"},
	{0xa7, 0x06f6, "EXTENDED ARABIC-INDIC DIGIT SIX"},
	{0xa8, 0x06f7, "EXTENDED ARABIC-INDIC DIGIT SEVEN"},
	{0xa9, 0x06f8, "EXTENDED ARABIC-INDIC DIGIT EIGHT"},
	{0xaa, 0x06f9, "EXTENDED ARABIC-INDIC DIGIT NINE"},
	{0xab, 0x060c, "ARABIC COMMA"},
	{0xac, 0x061b, "ARABIC SEMICOLON"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x061f, "ARABIC QUESTION MARK"},
	{0xaf, 0xfe81, "ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM"},
	{0xb0, 0xfe8d, "ARABIC LETTER ALEF ISOLATED FORM"},
	{0xb1, 0xfe8e, "ARABIC LETTER ALEF FINAL FORM"},
	{0xb2, 0xfe8e, "ARABIC LETTER ALEF FINAL FORM"},
	{0xb3, 0xfe8f, "ARABIC LETTER BEH ISOLATED FORM"},
	{0xb4, 0xfe91, "ARABIC LETTER BEH INITIAL FORM"},
	{0xb5, 0xfb56, "ARABIC LETTER PEH ISOLATED FORM"},
	{0xb6, 0xfb58, "ARABIC LETTER PEH INITIAL FORM"},
	{0xb7, 0xfe93, "ARABIC LETTER TEH MARBUTA ISOLATED FORM"},
	{0xb8, 0xfe95, "ARABIC LETTER TEH ISOLATED FORM"},
	{0xb9, 0xfe97, "ARABIC LETTER TEH INITIAL FORM"},
	{0xba, 0xfb66, "ARABIC LETTER TTEH ISOLATED FORM"},
	{0xbb, 0xfb68, "ARABIC LETTER TTEH INITIAL FORM"},
	{0xbc, 0xfe99, "ARABIC LETTER THEH ISOLATED FORM"},
	{0xbd, 0xfe9b, "ARABIC LETTER THEH INITIAL FORM"},
	{0xbe, 0xfe9d, "ARABIC LETTER JEEM ISOLATED FORM"},
	{0xbf, 0xfe9f, "ARABIC LETTER JEEM INITIAL FORM"},
	{0xc0, 0xfb7a, "ARABIC LETTER TCHEH ISOLATED FORM"},
	{0xc1, 0xfb7c, "ARABIC LETTER TCHEH INITIAL FORM"},
	{0xc2, 0xfea1, "ARABIC LETTER HAH ISOLATED FORM"},
	{0xc3, 0xfea3, "ARABIC LETTER HAH INITIAL FORM"},
	{0xc4, 0xfea5, "ARABIC LETTER KHAH ISOLATED FORM"},
	{0xc5, 0xfea7, "ARABIC LETTER KHAH INITIAL FORM"},
	{0xc6, 0xfea9, "ARABIC LETTER DAL ISOLATED FORM"},
	{0xc7, 0xfb84, "ARABIC LETTER DAHAL ISOLATED FORMN"},
	{0xc8, 0xfeab, "ARABIC LETTER THAL ISOLATED FORM"},
	{0xc9, 0xfead, "ARABIC LETTER REH ISOLATED FORM"},
	{0xca, 0xfb8c, "ARABIC LETTER RREH ISOLATED FORM"},
	{0xcb, 0xfeaf, "ARABIC LETTER ZAIN ISOLATED FORM"},
	{0xcc, 0xfb8a, "ARABIC LETTER JEH ISOLATED FORM"},
	{0xcd, 0xfeb1, "ARABIC LETTER SEEN ISOLATED FORM"},
	{0xce, 0xfeb3, "ARABIC LETTER SEEN INITIAL FORM"},
	{0xcf, 0xfeb5, "ARABIC LETTER SHEEN ISOLATED FORM"},
	{0xd0, 0xfeb7, "ARABIC LETTER SHEEN INITIAL FORM"},
	{0xd1, 0xfeb9, "ARABIC LETTER SAD ISOLATED FORM"},
	{0xd2, 0xfebb, "ARABIC LETTER SAD INITIAL FORM"},
	{0xd3, 0xfebd, "ARABIC LETTER DAD ISOLATED FORM"},
	{0xd4, 0xfebf, "ARABIC LETTER DAD INITIAL FORM"},
	{0xd5, 0xfec1, "ARABIC LETTER TAH ISOLATED FORM"},
	{0xd6, 0xfec5, "ARABIC LETTER ZAH ISOLATED FORM"},
	{0xd7, 0xfec9, "ARABIC LETTER AIN ISOLATED FORM"},
	{0xd8, 0xfeca, "ARABIC LETTER AIN FINAL FORM"},
	{0xd9, 0xfecb, "ARABIC LETTER AIN INITIAL FORM"},
	{0xda, 0xfecc, "ARABIC LETTER AIN MEDIAL FORM"},
	{0xdb, 0xfecd, "ARABIC LETTER GHAIN ISOLATED FORM"},
	{0xdc, 0xfece, "ARABIC LETTER GHAIN FINAL FORM"},
	{0xdd, 0xfecf, "ARABIC LETTER GHAIN INITIAL FORM"},
	{0xde, 0xfed0, "ARABIC LETTER GHAIN MEDIAL FORM"},
	{0xdf, 0xfed1, "ARABIC LETTER FEH ISOLATED FORM"},
	{0xe0, 0xfed3, "ARABIC LETTER FEH INITIAL FORM"},
	{0xe1, 0xfed5, "ARABIC LETTER QAF ISOLATED FORM"},
	{0xe2, 0xfed7, "ARABIC LETTER QAF INITIAL FORM"},
	{0xe3, 0xfed9, "ARABIC LETTER KAF ISOLATED FORM"},
	{0xe4, 0xfedb, "ARABIC LETTER KAF INITIAL FORM"},
	{0xe5, 0xfb92, "ARABIC LETTER GAF ISOLATED FORM"},
	{0xe6, 0xfb94, "ARABIC LETTER GAF INITIAL FORM"},
	{0xe7, 0xfedd, "ARABIC LETTER LAM ISOLATED FORM"},
	{0xe8, 0xfedf, "ARABIC LETTER LAM INITIAL FORM"},
	{0xe9, 0xfee0, "ARABIC LETTER LAM MEDIAL FORM"},
	{0xea, 0xfee1, "ARABIC LETTER MEEM ISOLATED FORM"},
	{0xeb, 0xfee3, "ARABIC LETTER MEEM INITIAL FORM"},
	{0xec, 0xfb9e, "ARABIC LETTER NOON GHUNNA ISOLATED FORM"},
	{0xed, 0xfee5, "ARABIC LETTER NOON ISOLATED FORM"},
	{0xee, 0xfee7, "ARABIC LETTER NOON INITIAL FORM"},
	{0xef, 0xfe85, "ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM"},
	{0xf0, 0xfeed, "ARABIC LETTER WAW ISOLATED FORM"},
	{0xf1, 0xfba6, "ARABIC LETTER HEH GOAL ISOLATED FORM"},
	{0xf2, 0xfba8, "ARABIC LETTER HEH GOAL INITIAL FORM"},
	{0xf3, 0xfba9, "ARABIC LETTER HEH GOAL MEDIAL FORM"},
	{0xf4, 0xfbaa, "ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM"},
	{0xf5, 0xfe80, "ARABIC LETTER HAMZA ISOLATED FORM"},
	{0xf6, 0xfe89, "ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM"},
	{0xf7, 0xfe8a, "ARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORM"},
	{0xf8, 0xfe8b, "ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM"},
	{0xf9, 0xfef1, "ARABIC LETTER YEH ISOLATED FORM"},
	{0xfa, 0xfef2, "ARABIC LETTER YEH FINAL FORM"},
	{0xfb, 0xfef3, "ARABIC LETTER YEH INITIAL FORM"},
	{0xfc, 0xfbb0, "ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM"},
	{0xfd, 0xfbae, "ARABIC LETTER YEH BARREE ISOLATED FORM"},
	{0xfe, 0xfe7c, "ARABIC SHADDA ISOLATED FORM"},
	{0xff, 0xfe7d, "ARABIC SHADDA MEDIAL FORM"}
};
const size_t n_cp_1006_table_size = sizeof(p_cp_1006_table) / sizeof(p_cp_1006_table[0]);

/**
 *	@brief cp-1026 to unicode mapping table
 *
 *
 *    Name:     cp1026_IBMLatin5Turkish to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *         
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1026_IBMLatin5Turkish code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1026_IBMLatin5Turkish order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1026_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x009c, "CONTROL"},
	{0x05, 0x0009, "HORIZONTAL TABULATION"},
	{0x06, 0x0086, "CONTROL"},
	{0x07, 0x007f, "DELETE"},
	{0x08, 0x0097, "CONTROL"},
	{0x09, 0x008d, "CONTROL"},
	{0x0a, 0x008e, "CONTROL"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x009d, "CONTROL"},
	{0x15, 0x0085, "CONTROL"},
	{0x16, 0x0008, "BACKSPACE"},
	{0x17, 0x0087, "CONTROL"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x0092, "CONTROL"},
	{0x1b, 0x008f, "CONTROL"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0080, "CONTROL"},
	{0x21, 0x0081, "CONTROL"},
	{0x22, 0x0082, "CONTROL"},
	{0x23, 0x0083, "CONTROL"},
	{0x24, 0x0084, "CONTROL"},
	{0x25, 0x000a, "LINE FEED"},
	{0x26, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x27, 0x001b, "ESCAPE"},
	{0x28, 0x0088, "CONTROL"},
	{0x29, 0x0089, "CONTROL"},
	{0x2a, 0x008a, "CONTROL"},
	{0x2b, 0x008b, "CONTROL"},
	{0x2c, 0x008c, "CONTROL"},
	{0x2d, 0x0005, "ENQUIRY"},
	{0x2e, 0x0006, "ACKNOWLEDGE"},
	{0x2f, 0x0007, "BELL"},
	{0x30, 0x0090, "CONTROL"},
	{0x31, 0x0091, "CONTROL"},
	{0x32, 0x0016, "SYNCHRONOUS IDLE"},
	{0x33, 0x0093, "CONTROL"},
	{0x34, 0x0094, "CONTROL"},
	{0x35, 0x0095, "CONTROL"},
	{0x36, 0x0096, "CONTROL"},
	{0x37, 0x0004, "END OF TRANSMISSION"},
	{0x38, 0x0098, "CONTROL"},
	{0x39, 0x0099, "CONTROL"},
	{0x3a, 0x009a, "CONTROL"},
	{0x3b, 0x009b, "CONTROL"},
	{0x3c, 0x0014, "DEVICE CONTROL FOUR"},
	{0x3d, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x3e, 0x009e, "CONTROL"},
	{0x3f, 0x001a, "SUBSTITUTE"},
	{0x40, 0x0020, "SPACE"},
	{0x41, 0x00a0, "NO-BREAK SPACE"},
	{0x42, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x43, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x44, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x45, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x46, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x47, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x48, 0x007b, "LEFT CURLY BRACKET"},
	{0x49, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x4a, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x4b, 0x002e, "FULL STOP"},
	{0x4c, 0x003c, "LESS-THAN SIGN"},
	{0x4d, 0x0028, "LEFT PARENTHESIS"},
	{0x4e, 0x002b, "PLUS SIGN"},
	{0x4f, 0x0021, "EXCLAMATION MARK"},
	{0x50, 0x0026, "AMPERSAND"},
	{0x51, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x52, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x53, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x54, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x55, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x56, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x57, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x58, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x59, 0x00df, "LATIN SMALL LETTER SHARP S (GERMAN)"},
	{0x5a, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0x5b, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0x5c, 0x002a, "ASTERISK"},
	{0x5d, 0x0029, "RIGHT PARENTHESIS"},
	{0x5e, 0x003b, "SEMICOLON"},
	{0x5f, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x60, 0x002d, "HYPHEN-MINUS"},
	{0x61, 0x002f, "SOLIDUS"},
	{0x62, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x63, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x64, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x65, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0x66, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0x67, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x68, 0x005b, "LEFT SQUARE BRACKET"},
	{0x69, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x6a, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0x6b, 0x002c, "COMMA"},
	{0x6c, 0x0025, "PERCENT SIGN"},
	{0x6d, 0x005f, "LOW LINE"},
	{0x6e, 0x003e, "GREATER-THAN SIGN"},
	{0x6f, 0x003f, "QUESTION MARK"},
	{0x70, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x71, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x72, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x73, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0x74, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x75, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0x76, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0x77, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0x78, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0x79, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0x7a, 0x003a, "COLON"},
	{0x7b, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x7c, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0x7d, 0x0027, "APOSTROPHE"},
	{0x7e, 0x003d, "EQUALS SIGN"},
	{0x7f, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x80, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x81, 0x0061, "LATIN SMALL LETTER A"},
	{0x82, 0x0062, "LATIN SMALL LETTER B"},
	{0x83, 0x0063, "LATIN SMALL LETTER C"},
	{0x84, 0x0064, "LATIN SMALL LETTER D"},
	{0x85, 0x0065, "LATIN SMALL LETTER E"},
	{0x86, 0x0066, "LATIN SMALL LETTER F"},
	{0x87, 0x0067, "LATIN SMALL LETTER G"},
	{0x88, 0x0068, "LATIN SMALL LETTER H"},
	{0x89, 0x0069, "LATIN SMALL LETTER I"},
	{0x8a, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8b, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8c, 0x007d, "RIGHT CURLY BRACKET"},
	{0x8d, 0x0060, "GRAVE ACCENT"},
	{0x8e, 0x00a6, "BROKEN BAR"},
	{0x8f, 0x00b1, "PLUS-MINUS SIGN"},
	{0x90, 0x00b0, "DEGREE SIGN"},
	{0x91, 0x006a, "LATIN SMALL LETTER J"},
	{0x92, 0x006b, "LATIN SMALL LETTER K"},
	{0x93, 0x006c, "LATIN SMALL LETTER L"},
	{0x94, 0x006d, "LATIN SMALL LETTER M"},
	{0x95, 0x006e, "LATIN SMALL LETTER N"},
	{0x96, 0x006f, "LATIN SMALL LETTER O"},
	{0x97, 0x0070, "LATIN SMALL LETTER P"},
	{0x98, 0x0071, "LATIN SMALL LETTER Q"},
	{0x99, 0x0072, "LATIN SMALL LETTER R"},
	{0x9a, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0x9b, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0x9c, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x9d, 0x00b8, "CEDILLA"},
	{0x9e, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x9f, 0x00a4, "CURRENCY SIGN"},
	{0xa0, 0x00b5, "MICRO SIGN"},
	{0xa1, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xa2, 0x0073, "LATIN SMALL LETTER S"},
	{0xa3, 0x0074, "LATIN SMALL LETTER T"},
	{0xa4, 0x0075, "LATIN SMALL LETTER U"},
	{0xa5, 0x0076, "LATIN SMALL LETTER V"},
	{0xa6, 0x0077, "LATIN SMALL LETTER W"},
	{0xa7, 0x0078, "LATIN SMALL LETTER X"},
	{0xa8, 0x0079, "LATIN SMALL LETTER Y"},
	{0xa9, 0x007a, "LATIN SMALL LETTER Z"},
	{0xaa, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xab, 0x00bf, "INVERTED QUESTION MARK"},
	{0xac, 0x005d, "RIGHT SQUARE BRACKET"},
	{0xad, 0x0024, "DOLLAR SIGN"},
	{0xae, 0x0040, "COMMERCIAL AT"},
	{0xaf, 0x00ae, "REGISTERED SIGN"},
	{0xb0, 0x00a2, "CENT SIGN"},
	{0xb1, 0x00a3, "POUND SIGN"},
	{0xb2, 0x00a5, "YEN SIGN"},
	{0xb3, 0x00b7, "MIDDLE DOT"},
	{0xb4, 0x00a9, "COPYRIGHT SIGN"},
	{0xb5, 0x00a7, "SECTION SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xb8, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xb9, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xba, 0x00ac, "NOT SIGN"},
	{0xbb, 0x007c, "VERTICAL LINE"},
	{0xbc, 0x00af, "MACRON"},
	{0xbd, 0x00a8, "DIAERESIS"},
	{0xbe, 0x00b4, "ACUTE ACCENT"},
	{0xbf, 0x00d7, "MULTIPLICATION SIGN"},
	{0xc0, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xc1, 0x0041, "LATIN CAPITAL LETTER A"},
	{0xc2, 0x0042, "LATIN CAPITAL LETTER B"},
	{0xc3, 0x0043, "LATIN CAPITAL LETTER C"},
	{0xc4, 0x0044, "LATIN CAPITAL LETTER D"},
	{0xc5, 0x0045, "LATIN CAPITAL LETTER E"},
	{0xc6, 0x0046, "LATIN CAPITAL LETTER F"},
	{0xc7, 0x0047, "LATIN CAPITAL LETTER G"},
	{0xc8, 0x0048, "LATIN CAPITAL LETTER H"},
	{0xc9, 0x0049, "LATIN CAPITAL LETTER I"},
	{0xca, 0x00ad, "SOFT HYPHEN"},
	{0xcb, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xcc, 0x007e, "TILDE"},
	{0xcd, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xce, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xcf, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xd0, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xd1, 0x004a, "LATIN CAPITAL LETTER J"},
	{0xd2, 0x004b, "LATIN CAPITAL LETTER K"},
	{0xd3, 0x004c, "LATIN CAPITAL LETTER L"},
	{0xd4, 0x004d, "LATIN CAPITAL LETTER M"},
	{0xd5, 0x004e, "LATIN CAPITAL LETTER N"},
	{0xd6, 0x004f, "LATIN CAPITAL LETTER O"},
	{0xd7, 0x0050, "LATIN CAPITAL LETTER P"},
	{0xd8, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0xd9, 0x0052, "LATIN CAPITAL LETTER R"},
	{0xda, 0x00b9, "SUPERSCRIPT ONE"},
	{0xdb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x005c, "REVERSE SOLIDUS"},
	{0xdd, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xde, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xdf, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xe0, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xe1, 0x00f7, "DIVISION SIGN"},
	{0xe2, 0x0053, "LATIN CAPITAL LETTER S"},
	{0xe3, 0x0054, "LATIN CAPITAL LETTER T"},
	{0xe4, 0x0055, "LATIN CAPITAL LETTER U"},
	{0xe5, 0x0056, "LATIN CAPITAL LETTER V"},
	{0xe6, 0x0057, "LATIN CAPITAL LETTER W"},
	{0xe7, 0x0058, "LATIN CAPITAL LETTER X"},
	{0xe8, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0xe9, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0xea, 0x00b2, "SUPERSCRIPT TWO"},
	{0xeb, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xec, 0x0023, "NUMBER SIGN"},
	{0xed, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xf0, 0x0030, "DIGIT ZERO"},
	{0xf1, 0x0031, "DIGIT ONE"},
	{0xf2, 0x0032, "DIGIT TWO"},
	{0xf3, 0x0033, "DIGIT THREE"},
	{0xf4, 0x0034, "DIGIT FOUR"},
	{0xf5, 0x0035, "DIGIT FIVE"},
	{0xf6, 0x0036, "DIGIT SIX"},
	{0xf7, 0x0037, "DIGIT SEVEN"},
	{0xf8, 0x0038, "DIGIT EIGHT"},
	{0xf9, 0x0039, "DIGIT NINE"},
	{0xfa, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x0022, "QUOTATION MARK"},
	{0xfd, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xfe, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xff, 0x009f, "CONTROL"}
};
const size_t n_cp_1026_table_size = sizeof(p_cp_1026_table) / sizeof(p_cp_1026_table[0]);

/**
 *	@brief cp-1250 to unicode mapping table
 *
 *
 *    Name:     cp1250 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1250 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1250 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1250_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0x8d, 0x0164, "LATIN CAPITAL LETTER T WITH CARON"},
	{0x8e, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0x8f, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0x9d, 0x0165, "LATIN SMALL LETTER T WITH CARON"},
	{0x9e, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0x9f, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x02c7, "CARON"},
	{0xa2, 0x02d8, "BREVE"},
	{0xa3, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x02db, "OGONEK"},
	{0xb3, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xba, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x013d, "LATIN CAPITAL LETTER L WITH CARON"},
	{0xbd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xbe, 0x013e, "LATIN SMALL LETTER L WITH CARON"},
	{0xbf, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xc0, 0x0154, "LATIN CAPITAL LETTER R WITH ACUTE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x0102, "LATIN CAPITAL LETTER A WITH BREVE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x0139, "LATIN CAPITAL LETTER L WITH ACUTE"},
	{0xc6, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x011a, "LATIN CAPITAL LETTER E WITH CARON"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x010e, "LATIN CAPITAL LETTER D WITH CARON"},
	{0xd0, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xd2, 0x0147, "LATIN CAPITAL LETTER N WITH CARON"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x0150, "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x0158, "LATIN CAPITAL LETTER R WITH CARON"},
	{0xd9, 0x016e, "LATIN CAPITAL LETTER U WITH RING ABOVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x0170, "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x0162, "LATIN CAPITAL LETTER T WITH CEDILLA"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x0155, "LATIN SMALL LETTER R WITH ACUTE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x0103, "LATIN SMALL LETTER A WITH BREVE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x013a, "LATIN SMALL LETTER L WITH ACUTE"},
	{0xe6, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x011b, "LATIN SMALL LETTER E WITH CARON"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x010f, "LATIN SMALL LETTER D WITH CARON"},
	{0xf0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xf1, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xf2, 0x0148, "LATIN SMALL LETTER N WITH CARON"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x0151, "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x0159, "LATIN SMALL LETTER R WITH CARON"},
	{0xf9, 0x016f, "LATIN SMALL LETTER U WITH RING ABOVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x0171, "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x0163, "LATIN SMALL LETTER T WITH CEDILLA"},
	{0xff, 0x02d9, "DOT ABOVE"}
};
const size_t n_cp_1250_table_size = sizeof(p_cp_1250_table) / sizeof(p_cp_1250_table[0]);

/**
 *	@brief cp-1251 to unicode mapping table
 *
 *
 *    Name:     cp1251 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1251 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1251 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1251_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0402, "CYRILLIC CAPITAL LETTER DJE"},
	{0x81, 0x0403, "CYRILLIC CAPITAL LETTER GJE"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0453, "CYRILLIC SMALL LETTER GJE"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x20ac, "EURO SIGN"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0409, "CYRILLIC CAPITAL LETTER LJE"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x040a, "CYRILLIC CAPITAL LETTER NJE"},
	{0x8d, 0x040c, "CYRILLIC CAPITAL LETTER KJE"},
	{0x8e, 0x040b, "CYRILLIC CAPITAL LETTER TSHE"},
	{0x8f, 0x040f, "CYRILLIC CAPITAL LETTER DZHE"},
	{0x90, 0x0452, "CYRILLIC SMALL LETTER DJE"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0459, "CYRILLIC SMALL LETTER LJE"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x045a, "CYRILLIC SMALL LETTER NJE"},
	{0x9d, 0x045c, "CYRILLIC SMALL LETTER KJE"},
	{0x9e, 0x045b, "CYRILLIC SMALL LETTER TSHE"},
	{0x9f, 0x045f, "CYRILLIC SMALL LETTER DZHE"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x040e, "CYRILLIC CAPITAL LETTER SHORT U"},
	{0xa2, 0x045e, "CYRILLIC SMALL LETTER SHORT U"},
	{0xa3, 0x0408, "CYRILLIC CAPITAL LETTER JE"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x0490, "CYRILLIC CAPITAL LETTER GHE WITH UPTURN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x0407, "CYRILLIC CAPITAL LETTER YI"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb3, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb4, 0x0491, "CYRILLIC SMALL LETTER GHE WITH UPTURN"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xb9, 0x2116, "NUMERO SIGN"},
	{0xba, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x0458, "CYRILLIC SMALL LETTER JE"},
	{0xbd, 0x0405, "CYRILLIC CAPITAL LETTER DZE"},
	{0xbe, 0x0455, "CYRILLIC SMALL LETTER DZE"},
	{0xbf, 0x0457, "CYRILLIC SMALL LETTER YI"},
	{0xc0, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xc1, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xc2, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xc3, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xc4, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xc5, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xc6, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xc7, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xc8, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xc9, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xca, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xcb, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xcc, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xcd, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xce, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xcf, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xd0, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xd1, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xd2, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xd3, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xd4, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xd5, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xd6, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xd7, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xd8, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xd9, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xda, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0xdb, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xdc, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xdd, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xde, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0xdf, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xe0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xe1, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xe2, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xe3, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xe4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xe5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xe6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xe7, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xe8, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xe9, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xea, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xeb, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xec, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xed, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xee, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xef, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xf0, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xf1, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xf2, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xf3, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xf4, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xf5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xf6, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xf7, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xf8, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xf9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xfa, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xfb, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xfc, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xfd, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xfe, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xff, 0x044f, "CYRILLIC SMALL LETTER YA"}
};
const size_t n_cp_1251_table_size = sizeof(p_cp_1251_table) / sizeof(p_cp_1251_table[0]);

/**
 *	@brief cp-1252 to unicode mapping table
 *
 *
 *    Name:     cp1252 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1252 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1252 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1252_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0x8e, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x98, 0x02dc, "SMALL TILDE"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0x9e, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0x9f, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xde, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfe, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_cp_1252_table_size = sizeof(p_cp_1252_table) / sizeof(p_cp_1252_table[0]);

/**
 *	@brief cp-1253 to unicode mapping table
 *
 *
 *    Name:     cp1253 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1253 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1253 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1253_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0385, "GREEK DIALYTIKA TONOS"},
	{0xa2, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x2015, "HORIZONTAL BAR"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x0384, "GREEK TONOS"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0xb9, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0xba, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0xbf, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0xc0, 0x0390, "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
	{0xc1, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0xc2, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0xc3, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xc4, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0xc5, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0xc6, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0xc7, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0xc8, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xc9, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0xca, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0xcb, 0x039b, "GREEK CAPITAL LETTER LAMDA"},
	{0xcc, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0xcd, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0xce, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0xcf, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0xd0, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0xd1, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0xd3, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xd4, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0xd5, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0xd6, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xd7, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0xd8, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0xd9, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xda, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0xdb, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0xdc, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0xdd, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0xde, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0xdf, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xe0, 0x03b0, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"},
	{0xe1, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe2, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0xe3, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0xe4, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xe5, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xe6, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0xe7, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0xe8, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0xe9, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0xea, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0xeb, 0x03bb, "GREEK SMALL LETTER LAMDA"},
	{0xec, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xed, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xee, 0x03be, "GREEK SMALL LETTER XI"},
	{0xef, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xf0, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xf1, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xf2, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xf3, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xf4, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xf5, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xf6, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xf7, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xf8, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xf9, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xfa, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xfb, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xfc, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xfd, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xfe, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"}
};
const size_t n_cp_1253_table_size = sizeof(p_cp_1253_table) / sizeof(p_cp_1253_table[0]);

/**
 *	@brief cp-1254 to unicode mapping table
 *
 *
 *    Name:     cp1254 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1254 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1254 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1254_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x98, 0x02dc, "SMALL TILDE"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0x9f, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0xde, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xfe, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_cp_1254_table_size = sizeof(p_cp_1254_table) / sizeof(p_cp_1254_table[0]);

/**
 *	@brief cp-1255 to unicode mapping table
 *
 *
 *    Name:     cp1255 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          1/7/2000
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1255 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1255 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1255_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x98, 0x02dc, "SMALL TILDE"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x20aa, "NEW SHEQEL SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00d7, "MULTIPLICATION SIGN"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00f7, "DIVISION SIGN"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x05b0, "HEBREW POINT SHEVA"},
	{0xc1, 0x05b1, "HEBREW POINT HATAF SEGOL"},
	{0xc2, 0x05b2, "HEBREW POINT HATAF PATAH"},
	{0xc3, 0x05b3, "HEBREW POINT HATAF QAMATS"},
	{0xc4, 0x05b4, "HEBREW POINT HIRIQ"},
	{0xc5, 0x05b5, "HEBREW POINT TSERE"},
	{0xc6, 0x05b6, "HEBREW POINT SEGOL"},
	{0xc7, 0x05b7, "HEBREW POINT PATAH"},
	{0xc8, 0x05b8, "HEBREW POINT QAMATS"},
	{0xc9, 0x05b9, "HEBREW POINT HOLAM"},
	{0xcb, 0x05bb, "HEBREW POINT QUBUTS"},
	{0xcc, 0x05bc, "HEBREW POINT DAGESH OR MAPIQ"},
	{0xcd, 0x05bd, "HEBREW POINT METEG"},
	{0xce, 0x05be, "HEBREW PUNCTUATION MAQAF"},
	{0xcf, 0x05bf, "HEBREW POINT RAFE"},
	{0xd0, 0x05c0, "HEBREW PUNCTUATION PASEQ"},
	{0xd1, 0x05c1, "HEBREW POINT SHIN DOT"},
	{0xd2, 0x05c2, "HEBREW POINT SIN DOT"},
	{0xd3, 0x05c3, "HEBREW PUNCTUATION SOF PASUQ"},
	{0xd4, 0x05f0, "HEBREW LIGATURE YIDDISH DOUBLE VAV"},
	{0xd5, 0x05f1, "HEBREW LIGATURE YIDDISH VAV YOD"},
	{0xd6, 0x05f2, "HEBREW LIGATURE YIDDISH DOUBLE YOD"},
	{0xd7, 0x05f3, "HEBREW PUNCTUATION GERESH"},
	{0xd8, 0x05f4, "HEBREW PUNCTUATION GERSHAYIM"},
	{0xe0, 0x05d0, "HEBREW LETTER ALEF"},
	{0xe1, 0x05d1, "HEBREW LETTER BET"},
	{0xe2, 0x05d2, "HEBREW LETTER GIMEL"},
	{0xe3, 0x05d3, "HEBREW LETTER DALET"},
	{0xe4, 0x05d4, "HEBREW LETTER HE"},
	{0xe5, 0x05d5, "HEBREW LETTER VAV"},
	{0xe6, 0x05d6, "HEBREW LETTER ZAYIN"},
	{0xe7, 0x05d7, "HEBREW LETTER HET"},
	{0xe8, 0x05d8, "HEBREW LETTER TET"},
	{0xe9, 0x05d9, "HEBREW LETTER YOD"},
	{0xea, 0x05da, "HEBREW LETTER FINAL KAF"},
	{0xeb, 0x05db, "HEBREW LETTER KAF"},
	{0xec, 0x05dc, "HEBREW LETTER LAMED"},
	{0xed, 0x05dd, "HEBREW LETTER FINAL MEM"},
	{0xee, 0x05de, "HEBREW LETTER MEM"},
	{0xef, 0x05df, "HEBREW LETTER FINAL NUN"},
	{0xf0, 0x05e0, "HEBREW LETTER NUN"},
	{0xf1, 0x05e1, "HEBREW LETTER SAMEKH"},
	{0xf2, 0x05e2, "HEBREW LETTER AYIN"},
	{0xf3, 0x05e3, "HEBREW LETTER FINAL PE"},
	{0xf4, 0x05e4, "HEBREW LETTER PE"},
	{0xf5, 0x05e5, "HEBREW LETTER FINAL TSADI"},
	{0xf6, 0x05e6, "HEBREW LETTER TSADI"},
	{0xf7, 0x05e7, "HEBREW LETTER QOF"},
	{0xf8, 0x05e8, "HEBREW LETTER RESH"},
	{0xf9, 0x05e9, "HEBREW LETTER SHIN"},
	{0xfa, 0x05ea, "HEBREW LETTER TAV"},
	{0xfd, 0x200e, "LEFT-TO-RIGHT MARK"},
	{0xfe, 0x200f, "RIGHT-TO-LEFT MARK"}
};
const size_t n_cp_1255_table_size = sizeof(p_cp_1255_table) / sizeof(p_cp_1255_table[0]);

/**
 *	@brief cp-1256 to unicode mapping table
 *
 *
 *    Name:     cp1256 to Unicode table
 *    Unicode version: 2.1
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          01/5/99
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1256 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1256 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1256_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x81, 0x067e, "ARABIC LETTER PEH"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0679, "ARABIC LETTER TTEH"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0x8d, 0x0686, "ARABIC LETTER TCHEH"},
	{0x8e, 0x0698, "ARABIC LETTER JEH"},
	{0x8f, 0x0688, "ARABIC LETTER DDAL"},
	{0x90, 0x06af, "ARABIC LETTER GAF"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x98, 0x06a9, "ARABIC LETTER KEHEH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0691, "ARABIC LETTER RREH"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0x9d, 0x200c, "ZERO WIDTH NON-JOINER"},
	{0x9e, 0x200d, "ZERO WIDTH JOINER"},
	{0x9f, 0x06ba, "ARABIC LETTER NOON GHUNNA"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x060c, "ARABIC COMMA"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x06be, "ARABIC LETTER HEH DOACHASHMEE"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x061b, "ARABIC SEMICOLON"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x061f, "ARABIC QUESTION MARK"},
	{0xc0, 0x06c1, "ARABIC LETTER HEH GOAL"},
	{0xc1, 0x0621, "ARABIC LETTER HAMZA"},
	{0xc2, 0x0622, "ARABIC LETTER ALEF WITH MADDA ABOVE"},
	{0xc3, 0x0623, "ARABIC LETTER ALEF WITH HAMZA ABOVE"},
	{0xc4, 0x0624, "ARABIC LETTER WAW WITH HAMZA ABOVE"},
	{0xc5, 0x0625, "ARABIC LETTER ALEF WITH HAMZA BELOW"},
	{0xc6, 0x0626, "ARABIC LETTER YEH WITH HAMZA ABOVE"},
	{0xc7, 0x0627, "ARABIC LETTER ALEF"},
	{0xc8, 0x0628, "ARABIC LETTER BEH"},
	{0xc9, 0x0629, "ARABIC LETTER TEH MARBUTA"},
	{0xca, 0x062a, "ARABIC LETTER TEH"},
	{0xcb, 0x062b, "ARABIC LETTER THEH"},
	{0xcc, 0x062c, "ARABIC LETTER JEEM"},
	{0xcd, 0x062d, "ARABIC LETTER HAH"},
	{0xce, 0x062e, "ARABIC LETTER KHAH"},
	{0xcf, 0x062f, "ARABIC LETTER DAL"},
	{0xd0, 0x0630, "ARABIC LETTER THAL"},
	{0xd1, 0x0631, "ARABIC LETTER REH"},
	{0xd2, 0x0632, "ARABIC LETTER ZAIN"},
	{0xd3, 0x0633, "ARABIC LETTER SEEN"},
	{0xd4, 0x0634, "ARABIC LETTER SHEEN"},
	{0xd5, 0x0635, "ARABIC LETTER SAD"},
	{0xd6, 0x0636, "ARABIC LETTER DAD"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x0637, "ARABIC LETTER TAH"},
	{0xd9, 0x0638, "ARABIC LETTER ZAH"},
	{0xda, 0x0639, "ARABIC LETTER AIN"},
	{0xdb, 0x063a, "ARABIC LETTER GHAIN"},
	{0xdc, 0x0640, "ARABIC TATWEEL"},
	{0xdd, 0x0641, "ARABIC LETTER FEH"},
	{0xde, 0x0642, "ARABIC LETTER QAF"},
	{0xdf, 0x0643, "ARABIC LETTER KAF"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x0644, "ARABIC LETTER LAM"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x0645, "ARABIC LETTER MEEM"},
	{0xe4, 0x0646, "ARABIC LETTER NOON"},
	{0xe5, 0x0647, "ARABIC LETTER HEH"},
	{0xe6, 0x0648, "ARABIC LETTER WAW"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x0649, "ARABIC LETTER ALEF MAKSURA"},
	{0xed, 0x064a, "ARABIC LETTER YEH"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x064b, "ARABIC FATHATAN"},
	{0xf1, 0x064c, "ARABIC DAMMATAN"},
	{0xf2, 0x064d, "ARABIC KASRATAN"},
	{0xf3, 0x064e, "ARABIC FATHA"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x064f, "ARABIC DAMMA"},
	{0xf6, 0x0650, "ARABIC KASRA"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x0651, "ARABIC SHADDA"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x0652, "ARABIC SUKUN"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x200e, "LEFT-TO-RIGHT MARK"},
	{0xfe, 0x200f, "RIGHT-TO-LEFT MARK"},
	{0xff, 0x06d2, "ARABIC LETTER YEH BARREE"}
};
const size_t n_cp_1256_table_size = sizeof(p_cp_1256_table) / sizeof(p_cp_1256_table[0]);

/**
 *	@brief cp-1257 to unicode mapping table
 *
 *
 *    Name:     cp1257 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1257 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1257 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1257_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8d, 0x00a8, "DIAERESIS"},
	{0x8e, 0x02c7, "CARON"},
	{0x8f, 0x00b8, "CEDILLA"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9d, 0x00af, "MACRON"},
	{0x9e, 0x02db, "OGONEK"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x0156, "LATIN CAPITAL LETTER R WITH CEDILLA"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x0157, "LATIN SMALL LETTER R WITH CEDILLA"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xc0, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xc1, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xc2, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0xc3, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xc7, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0xc8, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0xcb, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0xcc, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0xcd, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xce, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xcf, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xd0, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xd1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xd2, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xd5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xd9, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xda, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0xdb, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xde, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xe1, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xe2, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0xe3, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xe7, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0xe8, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xeb, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0xec, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0xed, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xee, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0xef, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xf0, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xf1, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xf2, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0xf5, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xf9, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xfa, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0xfb, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xfe, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xff, 0x02d9, "DOT ABOVE"}
};
const size_t n_cp_1257_table_size = sizeof(p_cp_1257_table) / sizeof(p_cp_1257_table[0]);

/**
 *	@brief cp-1258 to unicode mapping table
 *
 *
 *    Name:     cp1258 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.01
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact:       Shawn.Steele@microsoft.com
 *
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp1258 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp1258 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_1258_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x98, 0x02dc, "SMALL TILDE"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0x9f, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a8, "DIAERESIS"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x00af, "MACRON"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x00b2, "SUPERSCRIPT TWO"},
	{0xb3, 0x00b3, "SUPERSCRIPT THREE"},
	{0xb4, 0x00b4, "ACUTE ACCENT"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x00b8, "CEDILLA"},
	{0xb9, 0x00b9, "SUPERSCRIPT ONE"},
	{0xba, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xbd, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xbe, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xc1, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xc2, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xc3, 0x0102, "LATIN CAPITAL LETTER A WITH BREVE"},
	{0xc4, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0xc5, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0xc6, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xc7, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0xc8, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xc9, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0xca, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xcb, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xcc, 0x0300, "COMBINING GRAVE ACCENT"},
	{0xcd, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xce, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xcf, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd0, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd1, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xd2, 0x0309, "COMBINING HOOK ABOVE"},
	{0xd3, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xd4, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xd5, 0x01a0, "LATIN CAPITAL LETTER O WITH HORN"},
	{0xd6, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xd7, 0x00d7, "MULTIPLICATION SIGN"},
	{0xd8, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xd9, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xda, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xdb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xdd, 0x01af, "LATIN CAPITAL LETTER U WITH HORN"},
	{0xde, 0x0303, "COMBINING TILDE"},
	{0xdf, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe0, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xe1, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xe2, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xe3, 0x0103, "LATIN SMALL LETTER A WITH BREVE"},
	{0xe4, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xe5, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xe6, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xe7, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xe8, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xe9, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xea, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xeb, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xec, 0x0301, "COMBINING ACUTE ACCENT"},
	{0xed, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xee, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xef, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xf0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xf1, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xf2, 0x0323, "COMBINING DOT BELOW"},
	{0xf3, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xf4, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xf5, 0x01a1, "LATIN SMALL LETTER O WITH HORN"},
	{0xf6, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf7, 0x00f7, "DIVISION SIGN"},
	{0xf8, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xf9, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xfa, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xfb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xfd, 0x01b0, "LATIN SMALL LETTER U WITH HORN"},
	{0xfe, 0x20ab, "DONG SIGN"},
	{0xff, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"}
};
const size_t n_cp_1258_table_size = sizeof(p_cp_1258_table) / sizeof(p_cp_1258_table[0]);

/**
 *	@brief cp-424 to unicode mapping table
 *
 *
 *    Name:     IBM EBCDIC CP424 (Hebrew) to Unicode table
 *    Unicode version: 3.0
 *    Table version: 1.0
 *    Table format:  Format A
 *    Date:          1999 July 27
 *    Authors:       Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       CP424 characters map into Unicode.
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the CP424 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *		(or for EBCDIC controls, the EBCDIC name in C1 space)
 *
 *    The entries are in CP424 order
 *
 *	Version history
 *	1.0 version new.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_cp_424_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x009c, "SELECT"},
	{0x05, 0x0009, "HORIZONTAL TABULATION"},
	{0x06, 0x0086, "REQUIRED NEW LINE"},
	{0x07, 0x007f, "DELETE"},
	{0x08, 0x0097, "GRAPHIC ESCAPE"},
	{0x09, 0x008d, "SUPERSCRIPT"},
	{0x0a, 0x008e, "REPEAT"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x009d, "RESTORE/ENABLE PRESENTATION"},
	{0x15, 0x0085, "NEW LINE"},
	{0x16, 0x0008, "BACKSPACE"},
	{0x17, 0x0087, "PROGRAM OPERATOR COMMUNICATION"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x0092, "UNIT BACK SPACE"},
	{0x1b, 0x008f, "CUSTOMER USE ONE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0080, "DIGIT SELECT"},
	{0x21, 0x0081, "START OF SIGNIFICANCE"},
	{0x22, 0x0082, "FIELD SEPARATOR"},
	{0x23, 0x0083, "WORD UNDERSCORE"},
	{0x24, 0x0084, "BYPASS OR INHIBIT PRESENTATION"},
	{0x25, 0x000a, "LINE FEED"},
	{0x26, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x27, 0x001b, "ESCAPE"},
	{0x28, 0x0088, "SET ATTRIBUTE"},
	{0x29, 0x0089, "START FIELD EXTENDED"},
	{0x2a, 0x008a, "SET MODE OR SWITCH"},
	{0x2b, 0x008b, "CONTROL SEQUENCE PREFIX"},
	{0x2c, 0x008c, "MODIFY FIELD ATTRIBUTE"},
	{0x2d, 0x0005, "ENQUIRY"},
	{0x2e, 0x0006, "ACKNOWLEDGE"},
	{0x2f, 0x0007, "BELL"},
	{0x30, 0x0090, "<reserved>"},
	{0x31, 0x0091, "<reserved>"},
	{0x32, 0x0016, "SYNCHRONOUS IDLE"},
	{0x33, 0x0093, "INDEX RETURN"},
	{0x34, 0x0094, "PRESENTATION POSITION"},
	{0x35, 0x0095, "TRANSPARENT"},
	{0x36, 0x0096, "NUMERIC BACKSPACE"},
	{0x37, 0x0004, "END OF TRANSMISSION"},
	{0x38, 0x0098, "SUBSCRIPT"},
	{0x39, 0x0099, "INDENT TABULATION"},
	{0x3a, 0x009a, "REVERSE FORM FEED"},
	{0x3b, 0x009b, "CUSTOMER USE THREE"},
	{0x3c, 0x0014, "DEVICE CONTROL FOUR"},
	{0x3d, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x3e, 0x009e, "<reserved>"},
	{0x3f, 0x001a, "SUBSTITUTE"},
	{0x40, 0x0020, "SPACE"},
	{0x41, 0x05d0, "HEBREW LETTER ALEF"},
	{0x42, 0x05d1, "HEBREW LETTER BET"},
	{0x43, 0x05d2, "HEBREW LETTER GIMEL"},
	{0x44, 0x05d3, "HEBREW LETTER DALET"},
	{0x45, 0x05d4, "HEBREW LETTER HE"},
	{0x46, 0x05d5, "HEBREW LETTER VAV"},
	{0x47, 0x05d6, "HEBREW LETTER ZAYIN"},
	{0x48, 0x05d7, "HEBREW LETTER HET"},
	{0x49, 0x05d8, "HEBREW LETTER TET"},
	{0x4a, 0x00a2, "CENT SIGN"},
	{0x4b, 0x002e, "FULL STOP"},
	{0x4c, 0x003c, "LESS-THAN SIGN"},
	{0x4d, 0x0028, "LEFT PARENTHESIS"},
	{0x4e, 0x002b, "PLUS SIGN"},
	{0x4f, 0x007c, "VERTICAL LINE"},
	{0x50, 0x0026, "AMPERSAND"},
	{0x51, 0x05d9, "HEBREW LETTER YOD"},
	{0x52, 0x05da, "HEBREW LETTER FINAL KAF"},
	{0x53, 0x05db, "HEBREW LETTER KAF"},
	{0x54, 0x05dc, "HEBREW LETTER LAMED"},
	{0x55, 0x05dd, "HEBREW LETTER FINAL MEM"},
	{0x56, 0x05de, "HEBREW LETTER MEM"},
	{0x57, 0x05df, "HEBREW LETTER FINAL NUN"},
	{0x58, 0x05e0, "HEBREW LETTER NUN"},
	{0x59, 0x05e1, "HEBREW LETTER SAMEKH"},
	{0x5a, 0x0021, "EXCLAMATION MARK"},
	{0x5b, 0x0024, "DOLLAR SIGN"},
	{0x5c, 0x002a, "ASTERISK"},
	{0x5d, 0x0029, "RIGHT PARENTHESIS"},
	{0x5e, 0x003b, "SEMICOLON"},
	{0x5f, 0x00ac, "NOT SIGN"},
	{0x60, 0x002d, "HYPHEN-MINUS"},
	{0x61, 0x002f, "SOLIDUS"},
	{0x62, 0x05e2, "HEBREW LETTER AYIN"},
	{0x63, 0x05e3, "HEBREW LETTER FINAL PE"},
	{0x64, 0x05e4, "HEBREW LETTER PE"},
	{0x65, 0x05e5, "HEBREW LETTER FINAL TSADI"},
	{0x66, 0x05e6, "HEBREW LETTER TSADI"},
	{0x67, 0x05e7, "HEBREW LETTER QOF"},
	{0x68, 0x05e8, "HEBREW LETTER RESH"},
	{0x69, 0x05e9, "HEBREW LETTER SHIN"},
	{0x6a, 0x00a6, "BROKEN BAR"},
	{0x6b, 0x002c, "COMMA"},
	{0x6c, 0x0025, "PERCENT SIGN"},
	{0x6d, 0x005f, "LOW LINE"},
	{0x6e, 0x003e, "GREATER-THAN SIGN"},
	{0x6f, 0x003f, "QUESTION MARK"},
	{0x71, 0x05ea, "HEBREW LETTER TAV"},
	{0x74, 0x00a0, "NO-BREAK SPACE"},
	{0x78, 0x2017, "DOUBLE LOW LINE"},
	{0x79, 0x0060, "GRAVE ACCENT"},
	{0x7a, 0x003a, "COLON"},
	{0x7b, 0x0023, "NUMBER SIGN"},
	{0x7c, 0x0040, "COMMERCIAL AT"},
	{0x7d, 0x0027, "APOSTROPHE"},
	{0x7e, 0x003d, "EQUALS SIGN"},
	{0x7f, 0x0022, "QUOTATION MARK"},
	{0x81, 0x0061, "LATIN SMALL LETTER A"},
	{0x82, 0x0062, "LATIN SMALL LETTER B"},
	{0x83, 0x0063, "LATIN SMALL LETTER C"},
	{0x84, 0x0064, "LATIN SMALL LETTER D"},
	{0x85, 0x0065, "LATIN SMALL LETTER E"},
	{0x86, 0x0066, "LATIN SMALL LETTER F"},
	{0x87, 0x0067, "LATIN SMALL LETTER G"},
	{0x88, 0x0068, "LATIN SMALL LETTER H"},
	{0x89, 0x0069, "LATIN SMALL LETTER I"},
	{0x8a, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8b, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8f, 0x00b1, "PLUS-MINUS SIGN"},
	{0x90, 0x00b0, "DEGREE SIGN"},
	{0x91, 0x006a, "LATIN SMALL LETTER J"},
	{0x92, 0x006b, "LATIN SMALL LETTER K"},
	{0x93, 0x006c, "LATIN SMALL LETTER L"},
	{0x94, 0x006d, "LATIN SMALL LETTER M"},
	{0x95, 0x006e, "LATIN SMALL LETTER N"},
	{0x96, 0x006f, "LATIN SMALL LETTER O"},
	{0x97, 0x0070, "LATIN SMALL LETTER P"},
	{0x98, 0x0071, "LATIN SMALL LETTER Q"},
	{0x99, 0x0072, "LATIN SMALL LETTER R"},
	{0x9d, 0x00b8, "CEDILLA"},
	{0x9f, 0x00a4, "CURRENCY SIGN"},
	{0xa0, 0x00b5, "MICRO SIGN"},
	{0xa1, 0x007e, "TILDE"},
	{0xa2, 0x0073, "LATIN SMALL LETTER S"},
	{0xa3, 0x0074, "LATIN SMALL LETTER T"},
	{0xa4, 0x0075, "LATIN SMALL LETTER U"},
	{0xa5, 0x0076, "LATIN SMALL LETTER V"},
	{0xa6, 0x0077, "LATIN SMALL LETTER W"},
	{0xa7, 0x0078, "LATIN SMALL LETTER X"},
	{0xa8, 0x0079, "LATIN SMALL LETTER Y"},
	{0xa9, 0x007a, "LATIN SMALL LETTER Z"},
	{0xaf, 0x00ae, "REGISTERED SIGN"},
	{0xb0, 0x005e, "CIRCUMFLEX ACCENT"},
	{0xb1, 0x00a3, "POUND SIGN"},
	{0xb2, 0x00a5, "YEN SIGN"},
	{0xb3, 0x00b7, "MIDDLE DOT"},
	{0xb4, 0x00a9, "COPYRIGHT SIGN"},
	{0xb5, 0x00a7, "SECTION SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xb8, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xb9, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xba, 0x005b, "LEFT SQUARE BRACKET"},
	{0xbb, 0x005d, "RIGHT SQUARE BRACKET"},
	{0xbc, 0x00af, "MACRON"},
	{0xbd, 0x00a8, "DIAERESIS"},
	{0xbe, 0x00b4, "ACUTE ACCENT"},
	{0xbf, 0x00d7, "MULTIPLICATION SIGN"},
	{0xc0, 0x007b, "LEFT CURLY BRACKET"},
	{0xc1, 0x0041, "LATIN CAPITAL LETTER A"},
	{0xc2, 0x0042, "LATIN CAPITAL LETTER B"},
	{0xc3, 0x0043, "LATIN CAPITAL LETTER C"},
	{0xc4, 0x0044, "LATIN CAPITAL LETTER D"},
	{0xc5, 0x0045, "LATIN CAPITAL LETTER E"},
	{0xc6, 0x0046, "LATIN CAPITAL LETTER F"},
	{0xc7, 0x0047, "LATIN CAPITAL LETTER G"},
	{0xc8, 0x0048, "LATIN CAPITAL LETTER H"},
	{0xc9, 0x0049, "LATIN CAPITAL LETTER I"},
	{0xca, 0x00ad, "SOFT HYPHEN"},
	{0xd0, 0x007d, "RIGHT CURLY BRACKET"},
	{0xd1, 0x004a, "LATIN CAPITAL LETTER J"},
	{0xd2, 0x004b, "LATIN CAPITAL LETTER K"},
	{0xd3, 0x004c, "LATIN CAPITAL LETTER L"},
	{0xd4, 0x004d, "LATIN CAPITAL LETTER M"},
	{0xd5, 0x004e, "LATIN CAPITAL LETTER N"},
	{0xd6, 0x004f, "LATIN CAPITAL LETTER O"},
	{0xd7, 0x0050, "LATIN CAPITAL LETTER P"},
	{0xd8, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0xd9, 0x0052, "LATIN CAPITAL LETTER R"},
	{0xda, 0x00b9, "SUPERSCRIPT ONE"},
	{0xe0, 0x005c, "REVERSE SOLIDUS"},
	{0xe1, 0x00f7, "DIVISION SIGN"},
	{0xe2, 0x0053, "LATIN CAPITAL LETTER S"},
	{0xe3, 0x0054, "LATIN CAPITAL LETTER T"},
	{0xe4, 0x0055, "LATIN CAPITAL LETTER U"},
	{0xe5, 0x0056, "LATIN CAPITAL LETTER V"},
	{0xe6, 0x0057, "LATIN CAPITAL LETTER W"},
	{0xe7, 0x0058, "LATIN CAPITAL LETTER X"},
	{0xe8, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0xe9, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0xea, 0x00b2, "SUPERSCRIPT TWO"},
	{0xf0, 0x0030, "DIGIT ZERO"},
	{0xf1, 0x0031, "DIGIT ONE"},
	{0xf2, 0x0032, "DIGIT TWO"},
	{0xf3, 0x0033, "DIGIT THREE"},
	{0xf4, 0x0034, "DIGIT FOUR"},
	{0xf5, 0x0035, "DIGIT FIVE"},
	{0xf6, 0x0036, "DIGIT SIX"},
	{0xf7, 0x0037, "DIGIT SEVEN"},
	{0xf8, 0x0038, "DIGIT EIGHT"},
	{0xf9, 0x0039, "DIGIT NINE"},
	{0xfa, 0x00b3, "SUPERSCRIPT THREE"},
	{0xff, 0x009f, "EIGHT ONES"}
};
const size_t n_cp_424_table_size = sizeof(p_cp_424_table) / sizeof(p_cp_424_table[0]);

/**
 *	@brief cp-437 to unicode mapping table
 *
 *
 *    Name:     cp437_DOSLatinUS to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp437_DOSLatinUS code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp437_DOSLatinUS order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_437_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00a2, "CENT SIGN"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00a5, "YEN SIGN"},
	{0x9e, 0x20a7, "PESETA SIGN"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x2310, "REVERSED NOT SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_437_table_size = sizeof(p_cp_437_table) / sizeof(p_cp_437_table[0]);

/**
 *	@brief cp-500 to unicode mapping table
 *
 *
 *    Name:     cp500_IBMInternational to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp500_IBMInternational code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp500_IBMInternational order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_500_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x009c, "CONTROL"},
	{0x05, 0x0009, "HORIZONTAL TABULATION"},
	{0x06, 0x0086, "CONTROL"},
	{0x07, 0x007f, "DELETE"},
	{0x08, 0x0097, "CONTROL"},
	{0x09, 0x008d, "CONTROL"},
	{0x0a, 0x008e, "CONTROL"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x009d, "CONTROL"},
	{0x15, 0x0085, "CONTROL"},
	{0x16, 0x0008, "BACKSPACE"},
	{0x17, 0x0087, "CONTROL"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x0092, "CONTROL"},
	{0x1b, 0x008f, "CONTROL"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0080, "CONTROL"},
	{0x21, 0x0081, "CONTROL"},
	{0x22, 0x0082, "CONTROL"},
	{0x23, 0x0083, "CONTROL"},
	{0x24, 0x0084, "CONTROL"},
	{0x25, 0x000a, "LINE FEED"},
	{0x26, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x27, 0x001b, "ESCAPE"},
	{0x28, 0x0088, "CONTROL"},
	{0x29, 0x0089, "CONTROL"},
	{0x2a, 0x008a, "CONTROL"},
	{0x2b, 0x008b, "CONTROL"},
	{0x2c, 0x008c, "CONTROL"},
	{0x2d, 0x0005, "ENQUIRY"},
	{0x2e, 0x0006, "ACKNOWLEDGE"},
	{0x2f, 0x0007, "BELL"},
	{0x30, 0x0090, "CONTROL"},
	{0x31, 0x0091, "CONTROL"},
	{0x32, 0x0016, "SYNCHRONOUS IDLE"},
	{0x33, 0x0093, "CONTROL"},
	{0x34, 0x0094, "CONTROL"},
	{0x35, 0x0095, "CONTROL"},
	{0x36, 0x0096, "CONTROL"},
	{0x37, 0x0004, "END OF TRANSMISSION"},
	{0x38, 0x0098, "CONTROL"},
	{0x39, 0x0099, "CONTROL"},
	{0x3a, 0x009a, "CONTROL"},
	{0x3b, 0x009b, "CONTROL"},
	{0x3c, 0x0014, "DEVICE CONTROL FOUR"},
	{0x3d, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x3e, 0x009e, "CONTROL"},
	{0x3f, 0x001a, "SUBSTITUTE"},
	{0x40, 0x0020, "SPACE"},
	{0x41, 0x00a0, "NO-BREAK SPACE"},
	{0x42, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x43, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x44, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x45, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x46, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x47, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x48, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x49, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x4a, 0x005b, "LEFT SQUARE BRACKET"},
	{0x4b, 0x002e, "FULL STOP"},
	{0x4c, 0x003c, "LESS-THAN SIGN"},
	{0x4d, 0x0028, "LEFT PARENTHESIS"},
	{0x4e, 0x002b, "PLUS SIGN"},
	{0x4f, 0x0021, "EXCLAMATION MARK"},
	{0x50, 0x0026, "AMPERSAND"},
	{0x51, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x52, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x53, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x54, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x55, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x56, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x57, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x58, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x59, 0x00df, "LATIN SMALL LETTER SHARP S (GERMAN)"},
	{0x5a, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5b, 0x0024, "DOLLAR SIGN"},
	{0x5c, 0x002a, "ASTERISK"},
	{0x5d, 0x0029, "RIGHT PARENTHESIS"},
	{0x5e, 0x003b, "SEMICOLON"},
	{0x5f, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x60, 0x002d, "HYPHEN-MINUS"},
	{0x61, 0x002f, "SOLIDUS"},
	{0x62, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x63, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x64, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x65, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0x66, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0x67, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x68, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x69, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x6a, 0x00a6, "BROKEN BAR"},
	{0x6b, 0x002c, "COMMA"},
	{0x6c, 0x0025, "PERCENT SIGN"},
	{0x6d, 0x005f, "LOW LINE"},
	{0x6e, 0x003e, "GREATER-THAN SIGN"},
	{0x6f, 0x003f, "QUESTION MARK"},
	{0x70, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x71, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x72, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x73, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0x74, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x75, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0x76, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0x77, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0x78, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0x79, 0x0060, "GRAVE ACCENT"},
	{0x7a, 0x003a, "COLON"},
	{0x7b, 0x0023, "NUMBER SIGN"},
	{0x7c, 0x0040, "COMMERCIAL AT"},
	{0x7d, 0x0027, "APOSTROPHE"},
	{0x7e, 0x003d, "EQUALS SIGN"},
	{0x7f, 0x0022, "QUOTATION MARK"},
	{0x80, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x81, 0x0061, "LATIN SMALL LETTER A"},
	{0x82, 0x0062, "LATIN SMALL LETTER B"},
	{0x83, 0x0063, "LATIN SMALL LETTER C"},
	{0x84, 0x0064, "LATIN SMALL LETTER D"},
	{0x85, 0x0065, "LATIN SMALL LETTER E"},
	{0x86, 0x0066, "LATIN SMALL LETTER F"},
	{0x87, 0x0067, "LATIN SMALL LETTER G"},
	{0x88, 0x0068, "LATIN SMALL LETTER H"},
	{0x89, 0x0069, "LATIN SMALL LETTER I"},
	{0x8a, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8b, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0x8c, 0x00f0, "LATIN SMALL LETTER ETH (ICELANDIC)"},
	{0x8d, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0x8e, 0x00fe, "LATIN SMALL LETTER THORN (ICELANDIC)"},
	{0x8f, 0x00b1, "PLUS-MINUS SIGN"},
	{0x90, 0x00b0, "DEGREE SIGN"},
	{0x91, 0x006a, "LATIN SMALL LETTER J"},
	{0x92, 0x006b, "LATIN SMALL LETTER K"},
	{0x93, 0x006c, "LATIN SMALL LETTER L"},
	{0x94, 0x006d, "LATIN SMALL LETTER M"},
	{0x95, 0x006e, "LATIN SMALL LETTER N"},
	{0x96, 0x006f, "LATIN SMALL LETTER O"},
	{0x97, 0x0070, "LATIN SMALL LETTER P"},
	{0x98, 0x0071, "LATIN SMALL LETTER Q"},
	{0x99, 0x0072, "LATIN SMALL LETTER R"},
	{0x9a, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0x9b, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0x9c, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x9d, 0x00b8, "CEDILLA"},
	{0x9e, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x9f, 0x00a4, "CURRENCY SIGN"},
	{0xa0, 0x00b5, "MICRO SIGN"},
	{0xa1, 0x007e, "TILDE"},
	{0xa2, 0x0073, "LATIN SMALL LETTER S"},
	{0xa3, 0x0074, "LATIN SMALL LETTER T"},
	{0xa4, 0x0075, "LATIN SMALL LETTER U"},
	{0xa5, 0x0076, "LATIN SMALL LETTER V"},
	{0xa6, 0x0077, "LATIN SMALL LETTER W"},
	{0xa7, 0x0078, "LATIN SMALL LETTER X"},
	{0xa8, 0x0079, "LATIN SMALL LETTER Y"},
	{0xa9, 0x007a, "LATIN SMALL LETTER Z"},
	{0xaa, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xab, 0x00bf, "INVERTED QUESTION MARK"},
	{0xac, 0x00d0, "LATIN CAPITAL LETTER ETH (ICELANDIC)"},
	{0xad, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xae, 0x00de, "LATIN CAPITAL LETTER THORN (ICELANDIC)"},
	{0xaf, 0x00ae, "REGISTERED SIGN"},
	{0xb0, 0x00a2, "CENT SIGN"},
	{0xb1, 0x00a3, "POUND SIGN"},
	{0xb2, 0x00a5, "YEN SIGN"},
	{0xb3, 0x00b7, "MIDDLE DOT"},
	{0xb4, 0x00a9, "COPYRIGHT SIGN"},
	{0xb5, 0x00a7, "SECTION SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xb8, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xb9, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xba, 0x00ac, "NOT SIGN"},
	{0xbb, 0x007c, "VERTICAL LINE"},
	{0xbc, 0x00af, "MACRON"},
	{0xbd, 0x00a8, "DIAERESIS"},
	{0xbe, 0x00b4, "ACUTE ACCENT"},
	{0xbf, 0x00d7, "MULTIPLICATION SIGN"},
	{0xc0, 0x007b, "LEFT CURLY BRACKET"},
	{0xc1, 0x0041, "LATIN CAPITAL LETTER A"},
	{0xc2, 0x0042, "LATIN CAPITAL LETTER B"},
	{0xc3, 0x0043, "LATIN CAPITAL LETTER C"},
	{0xc4, 0x0044, "LATIN CAPITAL LETTER D"},
	{0xc5, 0x0045, "LATIN CAPITAL LETTER E"},
	{0xc6, 0x0046, "LATIN CAPITAL LETTER F"},
	{0xc7, 0x0047, "LATIN CAPITAL LETTER G"},
	{0xc8, 0x0048, "LATIN CAPITAL LETTER H"},
	{0xc9, 0x0049, "LATIN CAPITAL LETTER I"},
	{0xca, 0x00ad, "SOFT HYPHEN"},
	{0xcb, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xcc, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xcd, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xce, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xcf, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xd0, 0x007d, "RIGHT CURLY BRACKET"},
	{0xd1, 0x004a, "LATIN CAPITAL LETTER J"},
	{0xd2, 0x004b, "LATIN CAPITAL LETTER K"},
	{0xd3, 0x004c, "LATIN CAPITAL LETTER L"},
	{0xd4, 0x004d, "LATIN CAPITAL LETTER M"},
	{0xd5, 0x004e, "LATIN CAPITAL LETTER N"},
	{0xd6, 0x004f, "LATIN CAPITAL LETTER O"},
	{0xd7, 0x0050, "LATIN CAPITAL LETTER P"},
	{0xd8, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0xd9, 0x0052, "LATIN CAPITAL LETTER R"},
	{0xda, 0x00b9, "SUPERSCRIPT ONE"},
	{0xdb, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xdc, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xdd, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xde, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xdf, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xe0, 0x005c, "REVERSE SOLIDUS"},
	{0xe1, 0x00f7, "DIVISION SIGN"},
	{0xe2, 0x0053, "LATIN CAPITAL LETTER S"},
	{0xe3, 0x0054, "LATIN CAPITAL LETTER T"},
	{0xe4, 0x0055, "LATIN CAPITAL LETTER U"},
	{0xe5, 0x0056, "LATIN CAPITAL LETTER V"},
	{0xe6, 0x0057, "LATIN CAPITAL LETTER W"},
	{0xe7, 0x0058, "LATIN CAPITAL LETTER X"},
	{0xe8, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0xe9, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0xea, 0x00b2, "SUPERSCRIPT TWO"},
	{0xeb, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xec, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0xed, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xf0, 0x0030, "DIGIT ZERO"},
	{0xf1, 0x0031, "DIGIT ONE"},
	{0xf2, 0x0032, "DIGIT TWO"},
	{0xf3, 0x0033, "DIGIT THREE"},
	{0xf4, 0x0034, "DIGIT FOUR"},
	{0xf5, 0x0035, "DIGIT FIVE"},
	{0xf6, 0x0036, "DIGIT SIX"},
	{0xf7, 0x0037, "DIGIT SEVEN"},
	{0xf8, 0x0038, "DIGIT EIGHT"},
	{0xf9, 0x0039, "DIGIT NINE"},
	{0xfa, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfb, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xfc, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0xfd, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xfe, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xff, 0x009f, "CONTROL"}
};
const size_t n_cp_500_table_size = sizeof(p_cp_500_table) / sizeof(p_cp_500_table[0]);

/**
 *	@brief cp-737 to unicode mapping table
 *
 *
 *    Name:     cp737_DOSGreek to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp737_DOSGreek code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp737_DOSGreek order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_737_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0x81, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0x82, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0x83, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0x84, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0x85, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0x86, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0x87, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0x88, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0x89, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0x8a, 0x039b, "GREEK CAPITAL LETTER LAMDA"},
	{0x8b, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0x8c, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0x8d, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0x8e, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0x8f, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0x90, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0x91, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0x92, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0x93, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0x94, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0x95, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0x96, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0x97, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0x98, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0x99, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0x9a, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0x9b, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0x9c, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0x9d, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0x9e, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0x9f, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0xa0, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0xa1, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0xa2, 0x03bb, "GREEK SMALL LETTER LAMDA"},
	{0xa3, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xa4, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xa5, 0x03be, "GREEK SMALL LETTER XI"},
	{0xa6, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xa7, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xa8, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xa9, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xaa, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xab, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xac, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xad, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xae, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xaf, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xe1, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0xe2, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0xe3, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0xe4, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xe5, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xe6, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xe7, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xe8, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xe9, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"},
	{0xea, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0xeb, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0xec, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0xed, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0xee, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0xef, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0xf0, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0xf5, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_737_table_size = sizeof(p_cp_737_table) / sizeof(p_cp_737_table[0]);

/**
 *	@brief cp-775 to unicode mapping table
 *
 *
 *    Name:     cp775_DOSBaltRim to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp775_DOSBaltRim code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp775_DOSBaltRim order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_775_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0x88, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0x89, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0x8a, 0x0156, "LATIN CAPITAL LETTER R WITH CEDILLA"},
	{0x8b, 0x0157, "LATIN SMALL LETTER R WITH CEDILLA"},
	{0x8c, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0x8d, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0x96, 0x00a2, "CENT SIGN"},
	{0x97, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0x98, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x9e, 0x00d7, "MULTIPLICATION SIGN"},
	{0x9f, 0x00a4, "CURRENCY SIGN"},
	{0xa0, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0xa1, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xa4, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xa5, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xa6, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xa7, 0x00a6, "BROKEN BAR"},
	{0xa8, 0x00a9, "COPYRIGHT SIGN"},
	{0xa9, 0x00ae, "REGISTERED SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xb6, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xb7, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xb8, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xbe, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xc7, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xd0, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xd1, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xd2, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xd3, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0xd4, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xd5, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xd6, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xd7, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xd8, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S (GERMAN)"},
	{0xe2, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xe3, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xe4, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xe5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xe8, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xe9, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xea, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xeb, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xec, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xed, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0xee, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xef, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xf3, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xf4, 0x00b6, "PILCROW SIGN"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x00b9, "SUPERSCRIPT ONE"},
	{0xfc, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_775_table_size = sizeof(p_cp_775_table) / sizeof(p_cp_775_table[0]);

/**
 *	@brief cp-850 to unicode mapping table
 *
 *
 *    Name:     cp850_DOSLatin1 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp850_DOSLatin1 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp850_DOSLatin1 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_850_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x9e, 0x00d7, "MULTIPLICATION SIGN"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x00ae, "REGISTERED SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xb6, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xb7, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xb8, 0x00a9, "COPYRIGHT SIGN"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x00a2, "CENT SIGN"},
	{0xbe, 0x00a5, "YEN SIGN"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xc7, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x00a4, "CURRENCY SIGN"},
	{0xd0, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0xd1, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0xd2, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xd3, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xd4, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xd5, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xd6, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xd7, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xd8, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x00a6, "BROKEN BAR"},
	{0xde, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xe3, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xe4, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xe5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0xe8, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0xe9, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xea, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xeb, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xec, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xed, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xee, 0x00af, "MACRON"},
	{0xef, 0x00b4, "ACUTE ACCENT"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2017, "DOUBLE LOW LINE"},
	{0xf3, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xf4, 0x00b6, "PILCROW SIGN"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x00b8, "CEDILLA"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x00a8, "DIAERESIS"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x00b9, "SUPERSCRIPT ONE"},
	{0xfc, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_850_table_size = sizeof(p_cp_850_table) / sizeof(p_cp_850_table[0]);

/**
 *	@brief cp-852 to unicode mapping table
 *
 *
 *    Name:     cp852_DOSLatin2 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp852_DOSLatin2 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp852_DOSLatin2 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_852_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x016f, "LATIN SMALL LETTER U WITH RING ABOVE"},
	{0x86, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x0150, "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
	{0x8b, 0x0151, "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x0139, "LATIN CAPITAL LETTER L WITH ACUTE"},
	{0x92, 0x013a, "LATIN SMALL LETTER L WITH ACUTE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x013d, "LATIN CAPITAL LETTER L WITH CARON"},
	{0x96, 0x013e, "LATIN SMALL LETTER L WITH CARON"},
	{0x97, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0x98, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x0164, "LATIN CAPITAL LETTER T WITH CARON"},
	{0x9c, 0x0165, "LATIN SMALL LETTER T WITH CARON"},
	{0x9d, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0x9e, 0x00d7, "MULTIPLICATION SIGN"},
	{0x9f, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0xa5, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0xa6, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xa7, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xa8, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xa9, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0xac, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0xad, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xb6, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xb7, 0x011a, "LATIN CAPITAL LETTER E WITH CARON"},
	{0xb8, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xbe, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x0102, "LATIN CAPITAL LETTER A WITH BREVE"},
	{0xc7, 0x0103, "LATIN SMALL LETTER A WITH BREVE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x00a4, "CURRENCY SIGN"},
	{0xd0, 0x0111, "LATIN SMALL LETTER D WITH STROKE"},
	{0xd1, 0x0110, "LATIN CAPITAL LETTER D WITH STROKE"},
	{0xd2, 0x010e, "LATIN CAPITAL LETTER D WITH CARON"},
	{0xd3, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xd4, 0x010f, "LATIN SMALL LETTER D WITH CARON"},
	{0xd5, 0x0147, "LATIN CAPITAL LETTER N WITH CARON"},
	{0xd6, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xd7, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xd8, 0x011b, "LATIN SMALL LETTER E WITH CARON"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x0162, "LATIN CAPITAL LETTER T WITH CEDILLA"},
	{0xde, 0x016e, "LATIN CAPITAL LETTER U WITH RING ABOVE"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xe3, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xe4, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xe5, 0x0148, "LATIN SMALL LETTER N WITH CARON"},
	{0xe6, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xe7, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xe8, 0x0154, "LATIN CAPITAL LETTER R WITH ACUTE"},
	{0xe9, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xea, 0x0155, "LATIN SMALL LETTER R WITH ACUTE"},
	{0xeb, 0x0170, "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
	{0xec, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xed, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xee, 0x0163, "LATIN SMALL LETTER T WITH CEDILLA"},
	{0xef, 0x00b4, "ACUTE ACCENT"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xf2, 0x02db, "OGONEK"},
	{0xf3, 0x02c7, "CARON"},
	{0xf4, 0x02d8, "BREVE"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x00b8, "CEDILLA"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x00a8, "DIAERESIS"},
	{0xfa, 0x02d9, "DOT ABOVE"},
	{0xfb, 0x0171, "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
	{0xfc, 0x0158, "LATIN CAPITAL LETTER R WITH CARON"},
	{0xfd, 0x0159, "LATIN SMALL LETTER R WITH CARON"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_852_table_size = sizeof(p_cp_852_table) / sizeof(p_cp_852_table[0]);

/**
 *	@brief cp-855 to unicode mapping table
 *
 *
 *    Name:     cp855_DOSCyrillic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp855_DOSCyrillic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp855_DOSCyrillic order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_855_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0452, "CYRILLIC SMALL LETTER DJE"},
	{0x81, 0x0402, "CYRILLIC CAPITAL LETTER DJE"},
	{0x82, 0x0453, "CYRILLIC SMALL LETTER GJE"},
	{0x83, 0x0403, "CYRILLIC CAPITAL LETTER GJE"},
	{0x84, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0x85, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0x86, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0x87, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0x88, 0x0455, "CYRILLIC SMALL LETTER DZE"},
	{0x89, 0x0405, "CYRILLIC CAPITAL LETTER DZE"},
	{0x8a, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0x8b, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0x8c, 0x0457, "CYRILLIC SMALL LETTER YI"},
	{0x8d, 0x0407, "CYRILLIC CAPITAL LETTER YI"},
	{0x8e, 0x0458, "CYRILLIC SMALL LETTER JE"},
	{0x8f, 0x0408, "CYRILLIC CAPITAL LETTER JE"},
	{0x90, 0x0459, "CYRILLIC SMALL LETTER LJE"},
	{0x91, 0x0409, "CYRILLIC CAPITAL LETTER LJE"},
	{0x92, 0x045a, "CYRILLIC SMALL LETTER NJE"},
	{0x93, 0x040a, "CYRILLIC CAPITAL LETTER NJE"},
	{0x94, 0x045b, "CYRILLIC SMALL LETTER TSHE"},
	{0x95, 0x040b, "CYRILLIC CAPITAL LETTER TSHE"},
	{0x96, 0x045c, "CYRILLIC SMALL LETTER KJE"},
	{0x97, 0x040c, "CYRILLIC CAPITAL LETTER KJE"},
	{0x98, 0x045e, "CYRILLIC SMALL LETTER SHORT U"},
	{0x99, 0x040e, "CYRILLIC CAPITAL LETTER SHORT U"},
	{0x9a, 0x045f, "CYRILLIC SMALL LETTER DZHE"},
	{0x9b, 0x040f, "CYRILLIC CAPITAL LETTER DZHE"},
	{0x9c, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0x9d, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0x9e, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0x9f, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0xa0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xa1, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xa2, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xa3, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xa4, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xa5, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xa6, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xa7, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xa8, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xa9, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xaa, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xab, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xac, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xad, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xb6, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xb7, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xb8, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xbe, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xc7, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x00a4, "CURRENCY SIGN"},
	{0xd0, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xd1, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xd2, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xd3, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xd4, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xd5, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xd6, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xd7, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xd8, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xde, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xe1, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xe2, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xe3, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xe4, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xe5, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xe6, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xe7, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xe8, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xe9, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xea, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xeb, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xec, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xed, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xee, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xef, 0x2116, "NUMERO SIGN"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xf2, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xf3, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xf4, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xf5, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xf6, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xf7, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xf8, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xf9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xfa, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xfb, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xfc, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xfd, 0x00a7, "SECTION SIGN"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_855_table_size = sizeof(p_cp_855_table) / sizeof(p_cp_855_table[0]);

/**
 *	@brief cp-856 to unicode mapping table
 *
 *
 *    Name:     cp856_Hebrew_PC to Unicode table
 *    Unicode version: 3.0
 *    Table version: 1.0
 *    Table format:  Format A
 *    Date:          1999 July 27
 *    Authors:       Ken Whistler (kenw@sybase.com)
 *
 *	Copyright (c) 1998 - 1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       CP424 characters map into Unicode.
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp856_Hebrew_PC code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp856_Hebrew_PC order
 *
 *	Version history
 *	1.0 version new.
 *
 *	Updated versions of this file may be found in:
 *		<ftp://ftp.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_cp_856_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x05d0, "HEBREW LETTER ALEF"},
	{0x81, 0x05d1, "HEBREW LETTER BET"},
	{0x82, 0x05d2, "HEBREW LETTER GIMEL"},
	{0x83, 0x05d3, "HEBREW LETTER DALET"},
	{0x84, 0x05d4, "HEBREW LETTER HE"},
	{0x85, 0x05d5, "HEBREW LETTER VAV"},
	{0x86, 0x05d6, "HEBREW LETTER ZAYIN"},
	{0x87, 0x05d7, "HEBREW LETTER HET"},
	{0x88, 0x05d8, "HEBREW LETTER TET"},
	{0x89, 0x05d9, "HEBREW LETTER YOD"},
	{0x8a, 0x05da, "HEBREW LETTER FINAL KAF"},
	{0x8b, 0x05db, "HEBREW LETTER KAF"},
	{0x8c, 0x05dc, "HEBREW LETTER LAMED"},
	{0x8d, 0x05dd, "HEBREW LETTER FINAL MEM"},
	{0x8e, 0x05de, "HEBREW LETTER MEM"},
	{0x8f, 0x05df, "HEBREW LETTER FINAL NUN"},
	{0x90, 0x05e0, "HEBREW LETTER NUN"},
	{0x91, 0x05e1, "HEBREW LETTER SAMEKH"},
	{0x92, 0x05e2, "HEBREW LETTER AYIN"},
	{0x93, 0x05e3, "HEBREW LETTER FINAL PE"},
	{0x94, 0x05e4, "HEBREW LETTER PE"},
	{0x95, 0x05e5, "HEBREW LETTER FINAL TSADI"},
	{0x96, 0x05e6, "HEBREW LETTER TSADI"},
	{0x97, 0x05e7, "HEBREW LETTER QOF"},
	{0x98, 0x05e8, "HEBREW LETTER RESH"},
	{0x99, 0x05e9, "HEBREW LETTER SHIN"},
	{0x9a, 0x05ea, "HEBREW LETTER TAV"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9e, 0x00d7, "MULTIPLICATION SIGN"},
	{0xa9, 0x00ae, "REGISTERED SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb8, 0x00a9, "COPYRIGHT SIGN"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x00a2, "CENT SIGN"},
	{0xbe, 0x00a5, "YEN SIGN"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x00a4, "CURRENCY SIGN"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x00a6, "BROKEN BAR"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xee, 0x00af, "MACRON"},
	{0xef, 0x00b4, "ACUTE ACCENT"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2017, "DOUBLE LOW LINE"},
	{0xf3, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xf4, 0x00b6, "PILCROW SIGN"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x00b8, "CEDILLA"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x00a8, "DIAERESIS"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x00b9, "SUPERSCRIPT ONE"},
	{0xfc, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_856_table_size = sizeof(p_cp_856_table) / sizeof(p_cp_856_table[0]);

/**
 *	@brief cp-857 to unicode mapping table
 *
 *
 *    Name:     cp857_DOSTurkish to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp857_DOSTurkish code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp857_DOSTurkish order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_857_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x9e, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0x9f, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0xa7, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x00ae, "REGISTERED SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xb6, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xb7, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xb8, 0x00a9, "COPYRIGHT SIGN"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x00a2, "CENT SIGN"},
	{0xbe, 0x00a5, "YEN SIGN"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xc7, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x00a4, "CURRENCY SIGN"},
	{0xd0, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xd1, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xd2, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xd3, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xd4, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xd6, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xd7, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xd8, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x00a6, "BROKEN BAR"},
	{0xde, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xe3, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xe4, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xe5, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe8, 0x00d7, "MULTIPLICATION SIGN"},
	{0xe9, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xea, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xeb, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xec, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xed, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xee, 0x00af, "MACRON"},
	{0xef, 0x00b4, "ACUTE ACCENT"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf3, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xf4, 0x00b6, "PILCROW SIGN"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x00b8, "CEDILLA"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x00a8, "DIAERESIS"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x00b9, "SUPERSCRIPT ONE"},
	{0xfc, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_857_table_size = sizeof(p_cp_857_table) / sizeof(p_cp_857_table[0]);

/**
 *	@brief cp-860 to unicode mapping table
 *
 *
 *    Name:     cp860_DOSPortuguese to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp860_DOSPortuguese code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp860_DOSPortuguese order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_860_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0x8c, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0x8d, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x8e, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0x8f, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x92, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x96, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0x99, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00a2, "CENT SIGN"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0x9e, 0x20a7, "PESETA SIGN"},
	{0x9f, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_860_table_size = sizeof(p_cp_860_table) / sizeof(p_cp_860_table[0]);

/**
 *	@brief cp-861 to unicode mapping table
 *
 *
 *    Name:     cp861_DOSIcelandic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp861_DOSIcelandic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp861_DOSIcelandic order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_861_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0x8c, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0x8d, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0x98, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x9e, 0x20a7, "PESETA SIGN"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xa5, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xa6, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xa7, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x2310, "REVERSED NOT SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_861_table_size = sizeof(p_cp_861_table) / sizeof(p_cp_861_table[0]);

/**
 *	@brief cp-862 to unicode mapping table
 *
 *
 *    Name:     cp862_DOSHebrew to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp862_DOSHebrew code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp862_DOSHebrew order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_862_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x05d0, "HEBREW LETTER ALEF"},
	{0x81, 0x05d1, "HEBREW LETTER BET"},
	{0x82, 0x05d2, "HEBREW LETTER GIMEL"},
	{0x83, 0x05d3, "HEBREW LETTER DALET"},
	{0x84, 0x05d4, "HEBREW LETTER HE"},
	{0x85, 0x05d5, "HEBREW LETTER VAV"},
	{0x86, 0x05d6, "HEBREW LETTER ZAYIN"},
	{0x87, 0x05d7, "HEBREW LETTER HET"},
	{0x88, 0x05d8, "HEBREW LETTER TET"},
	{0x89, 0x05d9, "HEBREW LETTER YOD"},
	{0x8a, 0x05da, "HEBREW LETTER FINAL KAF"},
	{0x8b, 0x05db, "HEBREW LETTER KAF"},
	{0x8c, 0x05dc, "HEBREW LETTER LAMED"},
	{0x8d, 0x05dd, "HEBREW LETTER FINAL MEM"},
	{0x8e, 0x05de, "HEBREW LETTER MEM"},
	{0x8f, 0x05df, "HEBREW LETTER FINAL NUN"},
	{0x90, 0x05e0, "HEBREW LETTER NUN"},
	{0x91, 0x05e1, "HEBREW LETTER SAMEKH"},
	{0x92, 0x05e2, "HEBREW LETTER AYIN"},
	{0x93, 0x05e3, "HEBREW LETTER FINAL PE"},
	{0x94, 0x05e4, "HEBREW LETTER PE"},
	{0x95, 0x05e5, "HEBREW LETTER FINAL TSADI"},
	{0x96, 0x05e6, "HEBREW LETTER TSADI"},
	{0x97, 0x05e7, "HEBREW LETTER QOF"},
	{0x98, 0x05e8, "HEBREW LETTER RESH"},
	{0x99, 0x05e9, "HEBREW LETTER SHIN"},
	{0x9a, 0x05ea, "HEBREW LETTER TAV"},
	{0x9b, 0x00a2, "CENT SIGN"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00a5, "YEN SIGN"},
	{0x9e, 0x20a7, "PESETA SIGN"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x2310, "REVERSED NOT SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S (GERMAN)"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_862_table_size = sizeof(p_cp_862_table) / sizeof(p_cp_862_table[0]);

/**
 *	@brief cp-863 to unicode mapping table
 *
 *
 *    Name:     cp863_DOSCanadaF to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp863_DOSCanadaF code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp863_DOSCanadaF order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_863_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00b6, "PILCROW SIGN"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x2017, "DOUBLE LOW LINE"},
	{0x8e, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x8f, 0x00a7, "SECTION SIGN"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x92, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0x95, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x00a4, "CURRENCY SIGN"},
	{0x99, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00a2, "CENT SIGN"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0x9e, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00a6, "BROKEN BAR"},
	{0xa1, 0x00b4, "ACUTE ACCENT"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00a8, "DIAERESIS"},
	{0xa5, 0x00b8, "CEDILLA"},
	{0xa6, 0x00b3, "SUPERSCRIPT THREE"},
	{0xa7, 0x00af, "MACRON"},
	{0xa8, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xa9, 0x2310, "REVERSED NOT SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_863_table_size = sizeof(p_cp_863_table) / sizeof(p_cp_863_table[0]);

/**
 *	@brief cp-864 to unicode mapping table
 *
 *
 *    Name:     cp864_DOSArabic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp864_DOSArabic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp864_DOSArabic order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_864_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x066a, "ARABIC PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00b0, "DEGREE SIGN"},
	{0x81, 0x00b7, "MIDDLE DOT"},
	{0x82, 0x2219, "BULLET OPERATOR"},
	{0x83, 0x221a, "SQUARE ROOT"},
	{0x84, 0x2592, "MEDIUM SHADE"},
	{0x85, 0x2500, "FORMS LIGHT HORIZONTAL"},
	{0x86, 0x2502, "FORMS LIGHT VERTICAL"},
	{0x87, 0x253c, "FORMS LIGHT VERTICAL AND HORIZONTAL"},
	{0x88, 0x2524, "FORMS LIGHT VERTICAL AND LEFT"},
	{0x89, 0x252c, "FORMS LIGHT DOWN AND HORIZONTAL"},
	{0x8a, 0x251c, "FORMS LIGHT VERTICAL AND RIGHT"},
	{0x8b, 0x2534, "FORMS LIGHT UP AND HORIZONTAL"},
	{0x8c, 0x2510, "FORMS LIGHT DOWN AND LEFT"},
	{0x8d, 0x250c, "FORMS LIGHT DOWN AND RIGHT"},
	{0x8e, 0x2514, "FORMS LIGHT UP AND RIGHT"},
	{0x8f, 0x2518, "FORMS LIGHT UP AND LEFT"},
	{0x90, 0x03b2, "GREEK SMALL BETA"},
	{0x91, 0x221e, "INFINITY"},
	{0x92, 0x03c6, "GREEK SMALL PHI"},
	{0x93, 0x00b1, "PLUS-OR-MINUS SIGN"},
	{0x94, 0x00bd, "FRACTION 1/2"},
	{0x95, 0x00bc, "FRACTION 1/4"},
	{0x96, 0x2248, "ALMOST EQUAL TO"},
	{0x97, 0x00ab, "LEFT POINTING GUILLEMET"},
	{0x98, 0x00bb, "RIGHT POINTING GUILLEMET"},
	{0x99, 0xfef7, "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM"},
	{0x9a, 0xfef8, "ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM"},
	{0x9d, 0xfefb, "ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM"},
	{0x9e, 0xfefc, "ARABIC LIGATURE LAM WITH ALEF FINAL FORM"},
	{0xa0, 0x00a0, "NON-BREAKING SPACE"},
	{0xa1, 0x00ad, "SOFT HYPHEN"},
	{0xa2, 0xfe82, "ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0xfe84, "ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM"},
	{0xa8, 0xfe8e, "ARABIC LETTER ALEF FINAL FORM"},
	{0xa9, 0xfe8f, "ARABIC LETTER BEH ISOLATED FORM"},
	{0xaa, 0xfe95, "ARABIC LETTER TEH ISOLATED FORM"},
	{0xab, 0xfe99, "ARABIC LETTER THEH ISOLATED FORM"},
	{0xac, 0x060c, "ARABIC COMMA"},
	{0xad, 0xfe9d, "ARABIC LETTER JEEM ISOLATED FORM"},
	{0xae, 0xfea1, "ARABIC LETTER HAH ISOLATED FORM"},
	{0xaf, 0xfea5, "ARABIC LETTER KHAH ISOLATED FORM"},
	{0xb0, 0x0660, "ARABIC-INDIC DIGIT ZERO"},
	{0xb1, 0x0661, "ARABIC-INDIC DIGIT ONE"},
	{0xb2, 0x0662, "ARABIC-INDIC DIGIT TWO"},
	{0xb3, 0x0663, "ARABIC-INDIC DIGIT THREE"},
	{0xb4, 0x0664, "ARABIC-INDIC DIGIT FOUR"},
	{0xb5, 0x0665, "ARABIC-INDIC DIGIT FIVE"},
	{0xb6, 0x0666, "ARABIC-INDIC DIGIT SIX"},
	{0xb7, 0x0667, "ARABIC-INDIC DIGIT SEVEN"},
	{0xb8, 0x0668, "ARABIC-INDIC DIGIT EIGHT"},
	{0xb9, 0x0669, "ARABIC-INDIC DIGIT NINE"},
	{0xba, 0xfed1, "ARABIC LETTER FEH ISOLATED FORM"},
	{0xbb, 0x061b, "ARABIC SEMICOLON"},
	{0xbc, 0xfeb1, "ARABIC LETTER SEEN ISOLATED FORM"},
	{0xbd, 0xfeb5, "ARABIC LETTER SHEEN ISOLATED FORM"},
	{0xbe, 0xfeb9, "ARABIC LETTER SAD ISOLATED FORM"},
	{0xbf, 0x061f, "ARABIC QUESTION MARK"},
	{0xc0, 0x00a2, "CENT SIGN"},
	{0xc1, 0xfe80, "ARABIC LETTER HAMZA ISOLATED FORM"},
	{0xc2, 0xfe81, "ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM"},
	{0xc3, 0xfe83, "ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM"},
	{0xc4, 0xfe85, "ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM"},
	{0xc5, 0xfeca, "ARABIC LETTER AIN FINAL FORM"},
	{0xc6, 0xfe8b, "ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM"},
	{0xc7, 0xfe8d, "ARABIC LETTER ALEF ISOLATED FORM"},
	{0xc8, 0xfe91, "ARABIC LETTER BEH INITIAL FORM"},
	{0xc9, 0xfe93, "ARABIC LETTER TEH MARBUTA ISOLATED FORM"},
	{0xca, 0xfe97, "ARABIC LETTER TEH INITIAL FORM"},
	{0xcb, 0xfe9b, "ARABIC LETTER THEH INITIAL FORM"},
	{0xcc, 0xfe9f, "ARABIC LETTER JEEM INITIAL FORM"},
	{0xcd, 0xfea3, "ARABIC LETTER HAH INITIAL FORM"},
	{0xce, 0xfea7, "ARABIC LETTER KHAH INITIAL FORM"},
	{0xcf, 0xfea9, "ARABIC LETTER DAL ISOLATED FORM"},
	{0xd0, 0xfeab, "ARABIC LETTER THAL ISOLATED FORM"},
	{0xd1, 0xfead, "ARABIC LETTER REH ISOLATED FORM"},
	{0xd2, 0xfeaf, "ARABIC LETTER ZAIN ISOLATED FORM"},
	{0xd3, 0xfeb3, "ARABIC LETTER SEEN INITIAL FORM"},
	{0xd4, 0xfeb7, "ARABIC LETTER SHEEN INITIAL FORM"},
	{0xd5, 0xfebb, "ARABIC LETTER SAD INITIAL FORM"},
	{0xd6, 0xfebf, "ARABIC LETTER DAD INITIAL FORM"},
	{0xd7, 0xfec1, "ARABIC LETTER TAH ISOLATED FORM"},
	{0xd8, 0xfec5, "ARABIC LETTER ZAH ISOLATED FORM"},
	{0xd9, 0xfecb, "ARABIC LETTER AIN INITIAL FORM"},
	{0xda, 0xfecf, "ARABIC LETTER GHAIN INITIAL FORM"},
	{0xdb, 0x00a6, "BROKEN VERTICAL BAR"},
	{0xdc, 0x00ac, "NOT SIGN"},
	{0xdd, 0x00f7, "DIVISION SIGN"},
	{0xde, 0x00d7, "MULTIPLICATION SIGN"},
	{0xdf, 0xfec9, "ARABIC LETTER AIN ISOLATED FORM"},
	{0xe0, 0x0640, "ARABIC TATWEEL"},
	{0xe1, 0xfed3, "ARABIC LETTER FEH INITIAL FORM"},
	{0xe2, 0xfed7, "ARABIC LETTER QAF INITIAL FORM"},
	{0xe3, 0xfedb, "ARABIC LETTER KAF INITIAL FORM"},
	{0xe4, 0xfedf, "ARABIC LETTER LAM INITIAL FORM"},
	{0xe5, 0xfee3, "ARABIC LETTER MEEM INITIAL FORM"},
	{0xe6, 0xfee7, "ARABIC LETTER NOON INITIAL FORM"},
	{0xe7, 0xfeeb, "ARABIC LETTER HEH INITIAL FORM"},
	{0xe8, 0xfeed, "ARABIC LETTER WAW ISOLATED FORM"},
	{0xe9, 0xfeef, "ARABIC LETTER ALEF MAKSURA ISOLATED FORM"},
	{0xea, 0xfef3, "ARABIC LETTER YEH INITIAL FORM"},
	{0xeb, 0xfebd, "ARABIC LETTER DAD ISOLATED FORM"},
	{0xec, 0xfecc, "ARABIC LETTER AIN MEDIAL FORM"},
	{0xed, 0xfece, "ARABIC LETTER GHAIN FINAL FORM"},
	{0xee, 0xfecd, "ARABIC LETTER GHAIN ISOLATED FORM"},
	{0xef, 0xfee1, "ARABIC LETTER MEEM ISOLATED FORM"},
	{0xf0, 0xfe7d, "ARABIC SHADDA MEDIAL FORM"},
	{0xf1, 0x0651, "ARABIC SHADDAH"},
	{0xf2, 0xfee5, "ARABIC LETTER NOON ISOLATED FORM"},
	{0xf3, 0xfee9, "ARABIC LETTER HEH ISOLATED FORM"},
	{0xf4, 0xfeec, "ARABIC LETTER HEH MEDIAL FORM"},
	{0xf5, 0xfef0, "ARABIC LETTER ALEF MAKSURA FINAL FORM"},
	{0xf6, 0xfef2, "ARABIC LETTER YEH FINAL FORM"},
	{0xf7, 0xfed0, "ARABIC LETTER GHAIN MEDIAL FORM"},
	{0xf8, 0xfed5, "ARABIC LETTER QAF ISOLATED FORM"},
	{0xf9, 0xfef5, "ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM"},
	{0xfa, 0xfef6, "ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM"},
	{0xfb, 0xfedd, "ARABIC LETTER LAM ISOLATED FORM"},
	{0xfc, 0xfed9, "ARABIC LETTER KAF ISOLATED FORM"},
	{0xfd, 0xfef1, "ARABIC LETTER YEH ISOLATED FORM"},
	{0xfe, 0x25a0, "BLACK SQUARE"}
};
const size_t n_cp_864_table_size = sizeof(p_cp_864_table) / sizeof(p_cp_864_table[0]);

/**
 *	@brief cp-865 to unicode mapping table
 *
 *
 *    Name:     cp865_DOSNordic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp865_DOSNordic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp865_DOSNordic order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_865_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x81, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0x82, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x83, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x85, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x86, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x87, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x88, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x89, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x8a, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x8b, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x8c, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x8d, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x8e, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x8f, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x90, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x91, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0x92, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0x93, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x94, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x95, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x96, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x97, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x98, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0x99, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0x9e, 0x20a7, "PESETA SIGN"},
	{0x9f, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa0, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xa1, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xa2, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xa3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xa4, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xa5, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0xa6, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xa7, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xa8, 0x00bf, "INVERTED QUESTION MARK"},
	{0xa9, 0x2310, "REVERSED NOT SIGN"},
	{0xaa, 0x00ac, "NOT SIGN"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xad, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00a4, "CURRENCY SIGN"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe1, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xe2, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xe3, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xe4, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xe5, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xe6, 0x00b5, "MICRO SIGN"},
	{0xe7, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xe8, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xe9, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xea, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xeb, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xec, 0x221e, "INFINITY"},
	{0xed, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xee, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xef, 0x2229, "INTERSECTION"},
	{0xf0, 0x2261, "IDENTICAL TO"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xf3, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xf4, 0x2320, "TOP HALF INTEGRAL"},
	{0xf5, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0xf6, 0x00f7, "DIVISION SIGN"},
	{0xf7, 0x2248, "ALMOST EQUAL TO"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x207f, "SUPERSCRIPT LATIN SMALL LETTER N"},
	{0xfd, 0x00b2, "SUPERSCRIPT TWO"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_865_table_size = sizeof(p_cp_865_table) / sizeof(p_cp_865_table[0]);

/**
 *	@brief cp-866 to unicode mapping table
 *
 *
 *    Name:     cp866_DOSCyrillicRussian to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp866_DOSCyrillicRussian code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp866_DOSCyrillicRussian order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_866_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0x81, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0x82, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0x83, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0x84, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0x85, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0x86, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0x87, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0x88, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0x89, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0x8a, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0x8b, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0x8c, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0x8d, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0x8e, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0x8f, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0x90, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0x91, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0x92, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0x93, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0x94, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0x95, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0x96, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0x97, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0x98, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0x99, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0x9a, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0x9b, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0x9c, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0x9d, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0x9e, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0x9f, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xa0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xa1, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xa2, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xa3, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xa4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xa5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xa6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xa7, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xa8, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xa9, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xaa, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xab, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xac, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xad, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xae, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xaf, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb6, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xb8, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xbe, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xc7, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xd0, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xd1, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xd2, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xd3, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xd4, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xd5, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xd6, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xd7, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xd8, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x258c, "LEFT HALF BLOCK"},
	{0xde, 0x2590, "RIGHT HALF BLOCK"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xe1, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xe2, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xe3, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xe4, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xe5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xe6, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xe7, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xe8, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xe9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xea, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xeb, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xec, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xed, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xee, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xef, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xf0, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xf1, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xf2, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0xf3, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0xf4, 0x0407, "CYRILLIC CAPITAL LETTER YI"},
	{0xf5, 0x0457, "CYRILLIC SMALL LETTER YI"},
	{0xf6, 0x040e, "CYRILLIC CAPITAL LETTER SHORT U"},
	{0xf7, 0x045e, "CYRILLIC SMALL LETTER SHORT U"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x2219, "BULLET OPERATOR"},
	{0xfa, 0x00b7, "MIDDLE DOT"},
	{0xfb, 0x221a, "SQUARE ROOT"},
	{0xfc, 0x2116, "NUMERO SIGN"},
	{0xfd, 0x00a4, "CURRENCY SIGN"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_866_table_size = sizeof(p_cp_866_table) / sizeof(p_cp_866_table[0]);

/**
 *	@brief cp-869 to unicode mapping table
 *
 *
 *    Name:     cp869_DOSGreek2 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp869_DOSGreek2 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp869_DOSGreek2 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_869_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x86, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0x88, 0x00b7, "MIDDLE DOT"},
	{0x89, 0x00ac, "NOT SIGN"},
	{0x8a, 0x00a6, "BROKEN BAR"},
	{0x8b, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x8c, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x8d, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0x8e, 0x2015, "HORIZONTAL BAR"},
	{0x8f, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0x90, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0x91, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0x92, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0x95, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0x96, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0x97, 0x00a9, "COPYRIGHT SIGN"},
	{0x98, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0x99, 0x00b2, "SUPERSCRIPT TWO"},
	{0x9a, 0x00b3, "SUPERSCRIPT THREE"},
	{0x9b, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0x9c, 0x00a3, "POUND SIGN"},
	{0x9d, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0x9e, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0x9f, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xa0, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xa1, 0x0390, "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
	{0xa2, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xa3, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xa4, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0xa5, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0xa6, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xa7, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0xa8, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0xa9, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0xaa, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0xab, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xac, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xad, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0xae, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xaf, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xb0, 0x2591, "LIGHT SHADE"},
	{0xb1, 0x2592, "MEDIUM SHADE"},
	{0xb2, 0x2593, "DARK SHADE"},
	{0xb3, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0xb4, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0xb5, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0xb6, 0x039b, "GREEK CAPITAL LETTER LAMDA"},
	{0xb7, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0xb8, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0xb9, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xba, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xbb, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xbc, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xbd, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0xbe, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0xbf, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0xc0, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0xc1, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0xc2, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0xc3, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0xc4, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0xc5, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0xc6, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0xc7, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0xc8, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xc9, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xca, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xcb, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xcc, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xcd, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xce, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xcf, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xd0, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0xd1, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0xd2, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xd3, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0xd4, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0xd5, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xd6, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xd7, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0xd8, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0xd9, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0xda, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0xdb, 0x2588, "FULL BLOCK"},
	{0xdc, 0x2584, "LOWER HALF BLOCK"},
	{0xdd, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xde, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xdf, 0x2580, "UPPER HALF BLOCK"},
	{0xe0, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0xe1, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0xe2, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0xe3, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0xe4, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0xe5, 0x03bb, "GREEK SMALL LETTER LAMDA"},
	{0xe6, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xe7, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xe8, 0x03be, "GREEK SMALL LETTER XI"},
	{0xe9, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xea, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xeb, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xec, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xed, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xee, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xef, 0x0384, "GREEK TONOS"},
	{0xf0, 0x00ad, "SOFT HYPHEN"},
	{0xf1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xf2, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xf3, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xf4, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xf5, 0x00a7, "SECTION SIGN"},
	{0xf6, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xf7, 0x0385, "GREEK DIALYTIKA TONOS"},
	{0xf8, 0x00b0, "DEGREE SIGN"},
	{0xf9, 0x00a8, "DIAERESIS"},
	{0xfa, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xfb, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xfc, 0x03b0, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"},
	{0xfd, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"},
	{0xfe, 0x25a0, "BLACK SQUARE"},
	{0xff, 0x00a0, "NO-BREAK SPACE"}
};
const size_t n_cp_869_table_size = sizeof(p_cp_869_table) / sizeof(p_cp_869_table[0]);

/**
 *	@brief cp-874 to unicode mapping table
 *
 *
 *    Name:     cp874 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/15/98
 *
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp874 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp874 order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_874_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x20ac, "EURO SIGN"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x0e01, "THAI CHARACTER KO KAI"},
	{0xa2, 0x0e02, "THAI CHARACTER KHO KHAI"},
	{0xa3, 0x0e03, "THAI CHARACTER KHO KHUAT"},
	{0xa4, 0x0e04, "THAI CHARACTER KHO KHWAI"},
	{0xa5, 0x0e05, "THAI CHARACTER KHO KHON"},
	{0xa6, 0x0e06, "THAI CHARACTER KHO RAKHANG"},
	{0xa7, 0x0e07, "THAI CHARACTER NGO NGU"},
	{0xa8, 0x0e08, "THAI CHARACTER CHO CHAN"},
	{0xa9, 0x0e09, "THAI CHARACTER CHO CHING"},
	{0xaa, 0x0e0a, "THAI CHARACTER CHO CHANG"},
	{0xab, 0x0e0b, "THAI CHARACTER SO SO"},
	{0xac, 0x0e0c, "THAI CHARACTER CHO CHOE"},
	{0xad, 0x0e0d, "THAI CHARACTER YO YING"},
	{0xae, 0x0e0e, "THAI CHARACTER DO CHADA"},
	{0xaf, 0x0e0f, "THAI CHARACTER TO PATAK"},
	{0xb0, 0x0e10, "THAI CHARACTER THO THAN"},
	{0xb1, 0x0e11, "THAI CHARACTER THO NANGMONTHO"},
	{0xb2, 0x0e12, "THAI CHARACTER THO PHUTHAO"},
	{0xb3, 0x0e13, "THAI CHARACTER NO NEN"},
	{0xb4, 0x0e14, "THAI CHARACTER DO DEK"},
	{0xb5, 0x0e15, "THAI CHARACTER TO TAO"},
	{0xb6, 0x0e16, "THAI CHARACTER THO THUNG"},
	{0xb7, 0x0e17, "THAI CHARACTER THO THAHAN"},
	{0xb8, 0x0e18, "THAI CHARACTER THO THONG"},
	{0xb9, 0x0e19, "THAI CHARACTER NO NU"},
	{0xba, 0x0e1a, "THAI CHARACTER BO BAIMAI"},
	{0xbb, 0x0e1b, "THAI CHARACTER PO PLA"},
	{0xbc, 0x0e1c, "THAI CHARACTER PHO PHUNG"},
	{0xbd, 0x0e1d, "THAI CHARACTER FO FA"},
	{0xbe, 0x0e1e, "THAI CHARACTER PHO PHAN"},
	{0xbf, 0x0e1f, "THAI CHARACTER FO FAN"},
	{0xc0, 0x0e20, "THAI CHARACTER PHO SAMPHAO"},
	{0xc1, 0x0e21, "THAI CHARACTER MO MA"},
	{0xc2, 0x0e22, "THAI CHARACTER YO YAK"},
	{0xc3, 0x0e23, "THAI CHARACTER RO RUA"},
	{0xc4, 0x0e24, "THAI CHARACTER RU"},
	{0xc5, 0x0e25, "THAI CHARACTER LO LING"},
	{0xc6, 0x0e26, "THAI CHARACTER LU"},
	{0xc7, 0x0e27, "THAI CHARACTER WO WAEN"},
	{0xc8, 0x0e28, "THAI CHARACTER SO SALA"},
	{0xc9, 0x0e29, "THAI CHARACTER SO RUSI"},
	{0xca, 0x0e2a, "THAI CHARACTER SO SUA"},
	{0xcb, 0x0e2b, "THAI CHARACTER HO HIP"},
	{0xcc, 0x0e2c, "THAI CHARACTER LO CHULA"},
	{0xcd, 0x0e2d, "THAI CHARACTER O ANG"},
	{0xce, 0x0e2e, "THAI CHARACTER HO NOKHUK"},
	{0xcf, 0x0e2f, "THAI CHARACTER PAIYANNOI"},
	{0xd0, 0x0e30, "THAI CHARACTER SARA A"},
	{0xd1, 0x0e31, "THAI CHARACTER MAI HAN-AKAT"},
	{0xd2, 0x0e32, "THAI CHARACTER SARA AA"},
	{0xd3, 0x0e33, "THAI CHARACTER SARA AM"},
	{0xd4, 0x0e34, "THAI CHARACTER SARA I"},
	{0xd5, 0x0e35, "THAI CHARACTER SARA II"},
	{0xd6, 0x0e36, "THAI CHARACTER SARA UE"},
	{0xd7, 0x0e37, "THAI CHARACTER SARA UEE"},
	{0xd8, 0x0e38, "THAI CHARACTER SARA U"},
	{0xd9, 0x0e39, "THAI CHARACTER SARA UU"},
	{0xda, 0x0e3a, "THAI CHARACTER PHINTHU"},
	{0xdf, 0x0e3f, "THAI CURRENCY SYMBOL BAHT"},
	{0xe0, 0x0e40, "THAI CHARACTER SARA E"},
	{0xe1, 0x0e41, "THAI CHARACTER SARA AE"},
	{0xe2, 0x0e42, "THAI CHARACTER SARA O"},
	{0xe3, 0x0e43, "THAI CHARACTER SARA AI MAIMUAN"},
	{0xe4, 0x0e44, "THAI CHARACTER SARA AI MAIMALAI"},
	{0xe5, 0x0e45, "THAI CHARACTER LAKKHANGYAO"},
	{0xe6, 0x0e46, "THAI CHARACTER MAIYAMOK"},
	{0xe7, 0x0e47, "THAI CHARACTER MAITAIKHU"},
	{0xe8, 0x0e48, "THAI CHARACTER MAI EK"},
	{0xe9, 0x0e49, "THAI CHARACTER MAI THO"},
	{0xea, 0x0e4a, "THAI CHARACTER MAI TRI"},
	{0xeb, 0x0e4b, "THAI CHARACTER MAI CHATTAWA"},
	{0xec, 0x0e4c, "THAI CHARACTER THANTHAKHAT"},
	{0xed, 0x0e4d, "THAI CHARACTER NIKHAHIT"},
	{0xee, 0x0e4e, "THAI CHARACTER YAMAKKAN"},
	{0xef, 0x0e4f, "THAI CHARACTER FONGMAN"},
	{0xf0, 0x0e50, "THAI DIGIT ZERO"},
	{0xf1, 0x0e51, "THAI DIGIT ONE"},
	{0xf2, 0x0e52, "THAI DIGIT TWO"},
	{0xf3, 0x0e53, "THAI DIGIT THREE"},
	{0xf4, 0x0e54, "THAI DIGIT FOUR"},
	{0xf5, 0x0e55, "THAI DIGIT FIVE"},
	{0xf6, 0x0e56, "THAI DIGIT SIX"},
	{0xf7, 0x0e57, "THAI DIGIT SEVEN"},
	{0xf8, 0x0e58, "THAI DIGIT EIGHT"},
	{0xf9, 0x0e59, "THAI DIGIT NINE"},
	{0xfa, 0x0e5a, "THAI CHARACTER ANGKHANKHU"},
	{0xfb, 0x0e5b, "THAI CHARACTER KHOMUT"}
};
const size_t n_cp_874_table_size = sizeof(p_cp_874_table) / sizeof(p_cp_874_table[0]);

/**
 *	@brief cp-875 to unicode mapping table
 *
 *
 *    Name:     cp875_IBMGreek to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 * 
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp875_IBMGreek code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp875_IBMGreek order
 *
 */
const CUnicodeMapping::TCharacterName p_cp_875_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x009c, "CONTROL"},
	{0x05, 0x0009, "HORIZONTAL TABULATION"},
	{0x06, 0x0086, "CONTROL"},
	{0x07, 0x007f, "DELETE"},
	{0x08, 0x0097, "CONTROL"},
	{0x09, 0x008d, "CONTROL"},
	{0x0a, 0x008e, "CONTROL"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x009d, "CONTROL"},
	{0x15, 0x0085, "CONTROL"},
	{0x16, 0x0008, "BACKSPACE"},
	{0x17, 0x0087, "CONTROL"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x0092, "CONTROL"},
	{0x1b, 0x008f, "CONTROL"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0080, "CONTROL"},
	{0x21, 0x0081, "CONTROL"},
	{0x22, 0x0082, "CONTROL"},
	{0x23, 0x0083, "CONTROL"},
	{0x24, 0x0084, "CONTROL"},
	{0x25, 0x000a, "LINE FEED"},
	{0x26, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x27, 0x001b, "ESCAPE"},
	{0x28, 0x0088, "CONTROL"},
	{0x29, 0x0089, "CONTROL"},
	{0x2a, 0x008a, "CONTROL"},
	{0x2b, 0x008b, "CONTROL"},
	{0x2c, 0x008c, "CONTROL"},
	{0x2d, 0x0005, "ENQUIRY"},
	{0x2e, 0x0006, "ACKNOWLEDGE"},
	{0x2f, 0x0007, "BELL"},
	{0x30, 0x0090, "CONTROL"},
	{0x31, 0x0091, "CONTROL"},
	{0x32, 0x0016, "SYNCHRONOUS IDLE"},
	{0x33, 0x0093, "CONTROL"},
	{0x34, 0x0094, "CONTROL"},
	{0x35, 0x0095, "CONTROL"},
	{0x36, 0x0096, "CONTROL"},
	{0x37, 0x0004, "END OF TRANSMISSION"},
	{0x38, 0x0098, "CONTROL"},
	{0x39, 0x0099, "CONTROL"},
	{0x3a, 0x009a, "CONTROL"},
	{0x3b, 0x009b, "CONTROL"},
	{0x3c, 0x0014, "DEVICE CONTROL FOUR"},
	{0x3d, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x3e, 0x009e, "CONTROL"},
	{0x3f, 0x001a, "SUBSTITUTE"},
	{0x40, 0x0020, "SPACE"},
	{0x41, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0x42, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0x43, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0x44, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0x45, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0x46, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0x47, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0x48, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0x49, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0x4a, 0x005b, "LEFT SQUARE BRACKET"},
	{0x4b, 0x002e, "FULL STOP"},
	{0x4c, 0x003c, "LESS-THAN SIGN"},
	{0x4d, 0x0028, "LEFT PARENTHESIS"},
	{0x4e, 0x002b, "PLUS SIGN"},
	{0x4f, 0x0021, "EXCLAMATION MARK"},
	{0x50, 0x0026, "AMPERSAND"},
	{0x51, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0x52, 0x039b, "GREEK CAPITAL LETTER LAMDA"},
	{0x53, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0x54, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0x55, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0x56, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0x57, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0x58, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0x59, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0x5a, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5b, 0x0024, "DOLLAR SIGN"},
	{0x5c, 0x002a, "ASTERISK"},
	{0x5d, 0x0029, "RIGHT PARENTHESIS"},
	{0x5e, 0x003b, "SEMICOLON"},
	{0x5f, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x60, 0x002d, "HYPHEN-MINUS"},
	{0x61, 0x002f, "SOLIDUS"},
	{0x62, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0x63, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0x64, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0x65, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0x66, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0x67, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0x68, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0x69, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0x6a, 0x007c, "VERTICAL LINE"},
	{0x6b, 0x002c, "COMMA"},
	{0x6c, 0x0025, "PERCENT SIGN"},
	{0x6d, 0x005f, "LOW LINE"},
	{0x6e, 0x003e, "GREATER-THAN SIGN"},
	{0x6f, 0x003f, "QUESTION MARK"},
	{0x70, 0x00a8, "DIAERESIS"},
	{0x71, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0x72, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0x73, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0x74, 0x00a0, "NO-BREAK SPACE"},
	{0x75, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0x76, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0x77, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0x78, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0x79, 0x0060, "GRAVE ACCENT"},
	{0x7a, 0x003a, "COLON"},
	{0x7b, 0x0023, "NUMBER SIGN"},
	{0x7c, 0x0040, "COMMERCIAL AT"},
	{0x7d, 0x0027, "APOSTROPHE"},
	{0x7e, 0x003d, "EQUALS SIGN"},
	{0x7f, 0x0022, "QUOTATION MARK"},
	{0x80, 0x0385, "GREEK DIALYTIKA TONOS"},
	{0x81, 0x0061, "LATIN SMALL LETTER A"},
	{0x82, 0x0062, "LATIN SMALL LETTER B"},
	{0x83, 0x0063, "LATIN SMALL LETTER C"},
	{0x84, 0x0064, "LATIN SMALL LETTER D"},
	{0x85, 0x0065, "LATIN SMALL LETTER E"},
	{0x86, 0x0066, "LATIN SMALL LETTER F"},
	{0x87, 0x0067, "LATIN SMALL LETTER G"},
	{0x88, 0x0068, "LATIN SMALL LETTER H"},
	{0x89, 0x0069, "LATIN SMALL LETTER I"},
	{0x8a, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0x8b, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0x8c, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0x8d, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0x8e, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0x8f, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0x90, 0x00b0, "DEGREE SIGN"},
	{0x91, 0x006a, "LATIN SMALL LETTER J"},
	{0x92, 0x006b, "LATIN SMALL LETTER K"},
	{0x93, 0x006c, "LATIN SMALL LETTER L"},
	{0x94, 0x006d, "LATIN SMALL LETTER M"},
	{0x95, 0x006e, "LATIN SMALL LETTER N"},
	{0x96, 0x006f, "LATIN SMALL LETTER O"},
	{0x97, 0x0070, "LATIN SMALL LETTER P"},
	{0x98, 0x0071, "LATIN SMALL LETTER Q"},
	{0x99, 0x0072, "LATIN SMALL LETTER R"},
	{0x9a, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0x9b, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0x9c, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0x9d, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0x9e, 0x03bb, "GREEK SMALL LETTER LAMDA"},
	{0x9f, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xa0, 0x00b4, "ACUTE ACCENT"},
	{0xa1, 0x007e, "TILDE"},
	{0xa2, 0x0073, "LATIN SMALL LETTER S"},
	{0xa3, 0x0074, "LATIN SMALL LETTER T"},
	{0xa4, 0x0075, "LATIN SMALL LETTER U"},
	{0xa5, 0x0076, "LATIN SMALL LETTER V"},
	{0xa6, 0x0077, "LATIN SMALL LETTER W"},
	{0xa7, 0x0078, "LATIN SMALL LETTER X"},
	{0xa8, 0x0079, "LATIN SMALL LETTER Y"},
	{0xa9, 0x007a, "LATIN SMALL LETTER Z"},
	{0xaa, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xab, 0x03be, "GREEK SMALL LETTER XI"},
	{0xac, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xad, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xae, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xaf, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xb0, 0x00a3, "POUND SIGN"},
	{0xb1, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0xb2, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0xb3, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0xb4, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xb5, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xb6, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xb7, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xb8, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xb9, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"},
	{0xba, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xbb, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xbc, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xbd, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xbe, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xbf, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xc0, 0x007b, "LEFT CURLY BRACKET"},
	{0xc1, 0x0041, "LATIN CAPITAL LETTER A"},
	{0xc2, 0x0042, "LATIN CAPITAL LETTER B"},
	{0xc3, 0x0043, "LATIN CAPITAL LETTER C"},
	{0xc4, 0x0044, "LATIN CAPITAL LETTER D"},
	{0xc5, 0x0045, "LATIN CAPITAL LETTER E"},
	{0xc6, 0x0046, "LATIN CAPITAL LETTER F"},
	{0xc7, 0x0047, "LATIN CAPITAL LETTER G"},
	{0xc8, 0x0048, "LATIN CAPITAL LETTER H"},
	{0xc9, 0x0049, "LATIN CAPITAL LETTER I"},
	{0xca, 0x00ad, "SOFT HYPHEN"},
	{0xcb, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xcc, 0x0390, "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
	{0xcd, 0x03b0, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"},
	{0xce, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xcf, 0x2015, "HORIZONTAL BAR"},
	{0xd0, 0x007d, "RIGHT CURLY BRACKET"},
	{0xd1, 0x004a, "LATIN CAPITAL LETTER J"},
	{0xd2, 0x004b, "LATIN CAPITAL LETTER K"},
	{0xd3, 0x004c, "LATIN CAPITAL LETTER L"},
	{0xd4, 0x004d, "LATIN CAPITAL LETTER M"},
	{0xd5, 0x004e, "LATIN CAPITAL LETTER N"},
	{0xd6, 0x004f, "LATIN CAPITAL LETTER O"},
	{0xd7, 0x0050, "LATIN CAPITAL LETTER P"},
	{0xd8, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0xd9, 0x0052, "LATIN CAPITAL LETTER R"},
	{0xda, 0x00b1, "PLUS-MINUS SIGN"},
	{0xdb, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xdc, 0x001a, "SUBSTITUTE"},
	{0xdd, 0x0387, "GREEK ANO TELEIA"},
	{0xde, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xdf, 0x00a6, "BROKEN BAR"},
	{0xe0, 0x005c, "REVERSE SOLIDUS"},
	{0xe1, 0x001a, "SUBSTITUTE"},
	{0xe2, 0x0053, "LATIN CAPITAL LETTER S"},
	{0xe3, 0x0054, "LATIN CAPITAL LETTER T"},
	{0xe4, 0x0055, "LATIN CAPITAL LETTER U"},
	{0xe5, 0x0056, "LATIN CAPITAL LETTER V"},
	{0xe6, 0x0057, "LATIN CAPITAL LETTER W"},
	{0xe7, 0x0058, "LATIN CAPITAL LETTER X"},
	{0xe8, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0xe9, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0xea, 0x00b2, "SUPERSCRIPT TWO"},
	{0xeb, 0x00a7, "SECTION SIGN"},
	{0xec, 0x001a, "SUBSTITUTE"},
	{0xed, 0x001a, "SUBSTITUTE"},
	{0xee, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xef, 0x00ac, "NOT SIGN"},
	{0xf0, 0x0030, "DIGIT ZERO"},
	{0xf1, 0x0031, "DIGIT ONE"},
	{0xf2, 0x0032, "DIGIT TWO"},
	{0xf3, 0x0033, "DIGIT THREE"},
	{0xf4, 0x0034, "DIGIT FOUR"},
	{0xf5, 0x0035, "DIGIT FIVE"},
	{0xf6, 0x0036, "DIGIT SIX"},
	{0xf7, 0x0037, "DIGIT SEVEN"},
	{0xf8, 0x0038, "DIGIT EIGHT"},
	{0xf9, 0x0039, "DIGIT NINE"},
	{0xfa, 0x00b3, "SUPERSCRIPT THREE"},
	{0xfb, 0x00a9, "COPYRIGHT SIGN"},
	{0xfc, 0x001a, "SUBSTITUTE"},
	{0xfd, 0x001a, "SUBSTITUTE"},
	{0xfe, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xff, 0x009f, "CONTROL"}
};
const size_t n_cp_875_table_size = sizeof(p_cp_875_table) / sizeof(p_cp_875_table[0]);

/**
 *	@brief cyrillic to unicode mapping table
 *
 *
 *    Name:     cp10007_MacCyrillic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10007_MacCyrillic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10007_MacCyrillic order
 *
 */
const CUnicodeMapping::TCharacterName p_cyrillic_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0x81, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0x82, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0x83, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0x84, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0x85, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0x86, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0x87, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0x88, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0x89, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0x8a, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0x8b, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0x8c, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0x8d, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0x8e, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0x8f, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0x90, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0x91, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0x92, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0x93, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0x94, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0x95, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0x96, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0x97, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0x98, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0x99, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0x9a, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0x9b, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0x9c, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0x9d, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0x9e, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0x9f, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xa0, 0x2020, "DAGGER"},
	{0xa1, 0x00b0, "DEGREE SIGN"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a7, "SECTION SIGN"},
	{0xa5, 0x2022, "BULLET"},
	{0xa6, 0x00b6, "PILCROW SIGN"},
	{0xa7, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x2122, "TRADE MARK SIGN"},
	{0xab, 0x0402, "CYRILLIC CAPITAL LETTER DJE"},
	{0xac, 0x0452, "CYRILLIC SMALL LETTER DJE"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x0403, "CYRILLIC CAPITAL LETTER GJE"},
	{0xaf, 0x0453, "CYRILLIC SMALL LETTER GJE"},
	{0xb0, 0x221e, "INFINITY"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x2202, "PARTIAL DIFFERENTIAL"},
	{0xb7, 0x0408, "CYRILLIC CAPITAL LETTER JE"},
	{0xb8, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0xb9, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0xba, 0x0407, "CYRILLIC CAPITAL LETTER YI"},
	{0xbb, 0x0457, "CYRILLIC SMALL LETTER YI"},
	{0xbc, 0x0409, "CYRILLIC CAPITAL LETTER LJE"},
	{0xbd, 0x0459, "CYRILLIC SMALL LETTER LJE"},
	{0xbe, 0x040a, "CYRILLIC CAPITAL LETTER NJE"},
	{0xbf, 0x045a, "CYRILLIC SMALL LETTER NJE"},
	{0xc0, 0x0458, "CYRILLIC SMALL LETTER JE"},
	{0xc1, 0x0405, "CYRILLIC CAPITAL LETTER DZE"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x221a, "SQUARE ROOT"},
	{0xc4, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xc5, 0x2248, "ALMOST EQUAL TO"},
	{0xc6, 0x2206, "INCREMENT"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x040b, "CYRILLIC CAPITAL LETTER TSHE"},
	{0xcc, 0x045b, "CYRILLIC SMALL LETTER TSHE"},
	{0xcd, 0x040c, "CYRILLIC CAPITAL LETTER KJE"},
	{0xce, 0x045c, "CYRILLIC SMALL LETTER KJE"},
	{0xcf, 0x0455, "CYRILLIC SMALL LETTER DZE"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2014, "EM DASH"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xd8, 0x040e, "CYRILLIC CAPITAL LETTER SHORT U"},
	{0xd9, 0x045e, "CYRILLIC SMALL LETTER SHORT U"},
	{0xda, 0x040f, "CYRILLIC CAPITAL LETTER DZHE"},
	{0xdb, 0x045f, "CYRILLIC SMALL LETTER DZHE"},
	{0xdc, 0x2116, "NUMERO SIGN"},
	{0xdd, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xde, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xdf, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xe0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xe1, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xe2, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xe3, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xe4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xe5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xe6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xe7, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xe8, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xe9, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xea, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xeb, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xec, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xed, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xee, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xef, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xf0, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xf1, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xf2, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xf3, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xf4, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xf5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xf6, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xf7, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xf8, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xf9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xfa, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xfb, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xfc, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xfd, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xfe, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xff, 0x00a4, "CURRENCY SIGN"}
};
const size_t n_cyrillic_table_size = sizeof(p_cyrillic_table) / sizeof(p_cyrillic_table[0]);

/**
 *	@brief greek to unicode mapping table
 *
 *
 *    Name:     cp10006_MacGreek to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10006_MacGreek code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10006_MacGreek order
 *
 */
const CUnicodeMapping::TCharacterName p_greek_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x81, 0x00b9, "SUPERSCRIPT ONE"},
	{0x82, 0x00b2, "SUPERSCRIPT TWO"},
	{0x83, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x84, 0x00b3, "SUPERSCRIPT THREE"},
	{0x85, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x86, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x87, 0x0385, "GREEK DIALYTIKA TONOS"},
	{0x88, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x89, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x8a, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x8b, 0x0384, "GREEK TONOS"},
	{0x8c, 0x00a8, "DIAERESIS"},
	{0x8d, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x8e, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x8f, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x90, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x91, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x92, 0x00a3, "POUND SIGN"},
	{0x93, 0x2122, "TRADE MARK SIGN"},
	{0x94, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x95, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x96, 0x2022, "BULLET"},
	{0x97, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0x98, 0x2030, "PER MILLE SIGN"},
	{0x99, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x9b, 0x00a6, "BROKEN BAR"},
	{0x9c, 0x00ad, "SOFT HYPHEN"},
	{0x9d, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x9e, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x9f, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xa0, 0x2020, "DAGGER"},
	{0xa1, 0x0393, "GREEK CAPITAL LETTER GAMMA"},
	{0xa2, 0x0394, "GREEK CAPITAL LETTER DELTA"},
	{0xa3, 0x0398, "GREEK CAPITAL LETTER THETA"},
	{0xa4, 0x039b, "GREEK CAPITAL LETTER LAMBDA"},
	{0xa5, 0x039e, "GREEK CAPITAL LETTER XI"},
	{0xa6, 0x03a0, "GREEK CAPITAL LETTER PI"},
	{0xa7, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x03a3, "GREEK CAPITAL LETTER SIGMA"},
	{0xab, 0x03aa, "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
	{0xac, 0x00a7, "SECTION SIGN"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x00b0, "DEGREE SIGN"},
	{0xaf, 0x0387, "GREEK ANO TELEIA"},
	{0xb0, 0x0391, "GREEK CAPITAL LETTER ALPHA"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x00a5, "YEN SIGN"},
	{0xb5, 0x0392, "GREEK CAPITAL LETTER BETA"},
	{0xb6, 0x0395, "GREEK CAPITAL LETTER EPSILON"},
	{0xb7, 0x0396, "GREEK CAPITAL LETTER ZETA"},
	{0xb8, 0x0397, "GREEK CAPITAL LETTER ETA"},
	{0xb9, 0x0399, "GREEK CAPITAL LETTER IOTA"},
	{0xba, 0x039a, "GREEK CAPITAL LETTER KAPPA"},
	{0xbb, 0x039c, "GREEK CAPITAL LETTER MU"},
	{0xbc, 0x03a6, "GREEK CAPITAL LETTER PHI"},
	{0xbd, 0x03ab, "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
	{0xbe, 0x03a8, "GREEK CAPITAL LETTER PSI"},
	{0xbf, 0x03a9, "GREEK CAPITAL LETTER OMEGA"},
	{0xc0, 0x03ac, "GREEK SMALL LETTER ALPHA WITH TONOS"},
	{0xc1, 0x039d, "GREEK CAPITAL LETTER NU"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x039f, "GREEK CAPITAL LETTER OMICRON"},
	{0xc4, 0x03a1, "GREEK CAPITAL LETTER RHO"},
	{0xc5, 0x2248, "ALMOST EQUAL TO"},
	{0xc6, 0x03a4, "GREEK CAPITAL LETTER TAU"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x03a5, "GREEK CAPITAL LETTER UPSILON"},
	{0xcc, 0x03a7, "GREEK CAPITAL LETTER CHI"},
	{0xcd, 0x0386, "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
	{0xce, 0x0388, "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
	{0xcf, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2015, "HORIZONTAL BAR"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x0389, "GREEK CAPITAL LETTER ETA WITH TONOS"},
	{0xd8, 0x038a, "GREEK CAPITAL LETTER IOTA WITH TONOS"},
	{0xd9, 0x038c, "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
	{0xda, 0x038e, "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
	{0xdb, 0x03ad, "GREEK SMALL LETTER EPSILON WITH TONOS"},
	{0xdc, 0x03ae, "GREEK SMALL LETTER ETA WITH TONOS"},
	{0xdd, 0x03af, "GREEK SMALL LETTER IOTA WITH TONOS"},
	{0xde, 0x03cc, "GREEK SMALL LETTER OMICRON WITH TONOS"},
	{0xdf, 0x038f, "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
	{0xe0, 0x03cd, "GREEK SMALL LETTER UPSILON WITH TONOS"},
	{0xe1, 0x03b1, "GREEK SMALL LETTER ALPHA"},
	{0xe2, 0x03b2, "GREEK SMALL LETTER BETA"},
	{0xe3, 0x03c8, "GREEK SMALL LETTER PSI"},
	{0xe4, 0x03b4, "GREEK SMALL LETTER DELTA"},
	{0xe5, 0x03b5, "GREEK SMALL LETTER EPSILON"},
	{0xe6, 0x03c6, "GREEK SMALL LETTER PHI"},
	{0xe7, 0x03b3, "GREEK SMALL LETTER GAMMA"},
	{0xe8, 0x03b7, "GREEK SMALL LETTER ETA"},
	{0xe9, 0x03b9, "GREEK SMALL LETTER IOTA"},
	{0xea, 0x03be, "GREEK SMALL LETTER XI"},
	{0xeb, 0x03ba, "GREEK SMALL LETTER KAPPA"},
	{0xec, 0x03bb, "GREEK SMALL LETTER LAMBDA"},
	{0xed, 0x03bc, "GREEK SMALL LETTER MU"},
	{0xee, 0x03bd, "GREEK SMALL LETTER NU"},
	{0xef, 0x03bf, "GREEK SMALL LETTER OMICRON"},
	{0xf0, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xf1, 0x03ce, "GREEK SMALL LETTER OMEGA WITH TONOS"},
	{0xf2, 0x03c1, "GREEK SMALL LETTER RHO"},
	{0xf3, 0x03c3, "GREEK SMALL LETTER SIGMA"},
	{0xf4, 0x03c4, "GREEK SMALL LETTER TAU"},
	{0xf5, 0x03b8, "GREEK SMALL LETTER THETA"},
	{0xf6, 0x03c9, "GREEK SMALL LETTER OMEGA"},
	{0xf7, 0x03c2, "GREEK SMALL LETTER FINAL SIGMA"},
	{0xf8, 0x03c7, "GREEK SMALL LETTER CHI"},
	{0xf9, 0x03c5, "GREEK SMALL LETTER UPSILON"},
	{0xfa, 0x03b6, "GREEK SMALL LETTER ZETA"},
	{0xfb, 0x03ca, "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
	{0xfc, 0x03cb, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
	{0xfd, 0x0390, "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
	{0xfe, 0x03b0, "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"}
};
const size_t n_greek_table_size = sizeof(p_greek_table) / sizeof(p_greek_table[0]);

/**
 *	@brief iceland to unicode mapping table
 *
 *
 *    Name:     cp10079_MacIcelandic to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10079_MacIcelandic code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10079_MacIcelandic order
 *
 */
const CUnicodeMapping::TCharacterName p_iceland_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x81, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x82, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x83, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x84, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x85, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x86, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x87, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x88, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x89, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x8a, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x8b, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x8c, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x8d, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x8e, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x8f, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x90, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x91, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x92, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x93, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x94, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x95, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x96, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x97, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0x98, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x99, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x9b, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0x9c, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0x9d, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x9e, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x9f, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xa0, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xa1, 0x00b0, "DEGREE SIGN"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a7, "SECTION SIGN"},
	{0xa5, 0x2022, "BULLET"},
	{0xa6, 0x00b6, "PILCROW SIGN"},
	{0xa7, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x2122, "TRADE MARK SIGN"},
	{0xab, 0x00b4, "ACUTE ACCENT"},
	{0xac, 0x00a8, "DIAERESIS"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0xaf, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xb0, 0x221e, "INFINITY"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x00a5, "YEN SIGN"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x2202, "PARTIAL DIFFERENTIAL"},
	{0xb7, 0x2211, "N-ARY SUMMATION"},
	{0xb8, 0x220f, "N-ARY PRODUCT"},
	{0xb9, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xba, 0x222b, "INTEGRAL"},
	{0xbb, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xbc, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbd, 0x2126, "OHM SIGN"},
	{0xbe, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0xbf, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xc0, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x221a, "SQUARE ROOT"},
	{0xc4, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xc5, 0x2248, "ALMOST EQUAL TO"},
	{0xc6, 0x2206, "INCREMENT"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xcc, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xcd, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xce, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xcf, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2014, "EM DASH"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x25ca, "LOZENGE"},
	{0xd8, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xd9, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xda, 0x2044, "FRACTION SLASH"},
	{0xdb, 0x00a4, "CURRENCY SIGN"},
	{0xdc, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0xdd, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0xde, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0xdf, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0xe0, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xe1, 0x00b7, "MIDDLE DOT"},
	{0xe2, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0xe3, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xe4, 0x2030, "PER MILLE SIGN"},
	{0xe5, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xe6, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xe7, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xe8, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xe9, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xea, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xeb, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xec, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xed, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xf1, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xf2, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xf3, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xf4, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xf5, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xf6, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0xf7, 0x02dc, "SMALL TILDE"},
	{0xf8, 0x00af, "MACRON"},
	{0xf9, 0x02d8, "BREVE"},
	{0xfa, 0x02d9, "DOT ABOVE"},
	{0xfb, 0x02da, "RING ABOVE"},
	{0xfc, 0x00b8, "CEDILLA"},
	{0xfd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xfe, 0x02db, "OGONEK"},
	{0xff, 0x02c7, "CARON"}
};
const size_t n_iceland_table_size = sizeof(p_iceland_table) / sizeof(p_iceland_table[0]);

/**
 *	@brief koi8-r to unicode mapping table
 *
 *
 *	Name:             KOI8-R (RFC1489) to Unicode
 *	Unicode version:  3.0
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             18 August 1999
 *	Authors:          Helmut Richter <richter@lrz.de>
 *
 *	Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       KOI8-R characters map into Unicode. The underlying document is the
 *	mapping described in RFC 1489. No statements are made as to whether
 *	this mapping is the same as the mapping defined as \"Code Page 878\"
 *	with some vendors.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the KOI8-R code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in KOI8-R order.
 *
 *	Version history
 *	1.0 version: created.
 *
 *	Any comments or problems, contact <errata@unicode.org>
 *	Please note that <errata@unicode.org> is an archival address;
 *	notices will be checked, but do not expect an immediate response.
 *
 */
const CUnicodeMapping::TCharacterName p_koi8_r_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0x81, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0x82, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0x83, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0x84, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0x85, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0x86, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0x87, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0x88, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0x89, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0x8a, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0x8b, 0x2580, "UPPER HALF BLOCK"},
	{0x8c, 0x2584, "LOWER HALF BLOCK"},
	{0x8d, 0x2588, "FULL BLOCK"},
	{0x8e, 0x258c, "LEFT HALF BLOCK"},
	{0x8f, 0x2590, "RIGHT HALF BLOCK"},
	{0x90, 0x2591, "LIGHT SHADE"},
	{0x91, 0x2592, "MEDIUM SHADE"},
	{0x92, 0x2593, "DARK SHADE"},
	{0x93, 0x2320, "TOP HALF INTEGRAL"},
	{0x94, 0x25a0, "BLACK SQUARE"},
	{0x95, 0x2219, "BULLET OPERATOR"},
	{0x96, 0x221a, "SQUARE ROOT"},
	{0x97, 0x2248, "ALMOST EQUAL TO"},
	{0x98, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0x99, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0x9a, 0x00a0, "NO-BREAK SPACE"},
	{0x9b, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0x9c, 0x00b0, "DEGREE SIGN"},
	{0x9d, 0x00b2, "SUPERSCRIPT TWO"},
	{0x9e, 0x00b7, "MIDDLE DOT"},
	{0x9f, 0x00f7, "DIVISION SIGN"},
	{0xa0, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xa1, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xa2, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xa3, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xa4, 0x2553, "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"},
	{0xa5, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xa6, 0x2555, "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"},
	{0xa7, 0x2556, "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"},
	{0xa8, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xa9, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xaa, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xab, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xac, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xad, 0x255c, "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"},
	{0xae, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xaf, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xb0, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xb1, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xb2, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb3, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xb4, 0x2562, "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"},
	{0xb5, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xb6, 0x2564, "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"},
	{0xb7, 0x2565, "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"},
	{0xb8, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xb9, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xba, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xbb, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xbc, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xbd, 0x256b, "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"},
	{0xbe, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xbf, 0x00a9, "COPYRIGHT SIGN"},
	{0xc0, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xc1, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xc2, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xc3, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xc4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xc5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xc6, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xc7, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xc8, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xc9, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xca, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xcb, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xcc, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xcd, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xce, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xcf, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xd0, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xd1, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xd2, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xd3, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xd4, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xd5, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xd6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xd7, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xd8, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xd9, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xda, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xdb, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xdc, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xdd, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xde, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xdf, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xe0, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0xe1, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xe2, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xe3, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xe4, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xe5, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xe6, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xe7, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xe8, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xe9, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xea, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xeb, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xec, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xed, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xee, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xef, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xf0, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xf1, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xf2, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xf3, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xf4, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xf5, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xf6, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xf7, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xf8, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xf9, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xfa, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xfb, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xfc, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xfd, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xfe, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xff, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"}
};
const size_t n_koi8_r_table_size = sizeof(p_koi8_r_table) / sizeof(p_koi8_r_table[0]);

/**
 *	@brief koi8-u to unicode mapping table
 *
 *
 *	Name:             KOI8-U (RFC2319) to Unicode
 *	Unicode version:  5.1
 *	Table version:    1.0
 *	Table format:     Format A
 *	Date:             13 October 2008
 *	Authors:          Philippe Verdy <verdy_p AT wanadoo.fr>,
 *	                  Helmut Richter <richter@lrz.de>
 *
 *	Copyright (c) 1991-2008 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *	KOI8-U characters map into Unicode. The underlying document is the
 *	mapping described in RFC 2319. No statements are made as to whether
 *	this mapping is the same as the mapping defined as \"Code Page 878\"
 *	with some vendors.
 *
 *	The KOI8-U (RFC2319) encoding is a variant based on KOI8-R (RFC1489),
 *	where the support for four additional Cyrillic letters was added (both small
 *	and capital letters), replacing 8 box-drawing characters. It is still widely
 *	used to encode texts in Ukrainian, Byelorussian and Bulgarian.
 *
 *	Format:  Three tab-separated columns
 *		 Column \#1 is the KOI8-U code (in hex as 0xXX)
 *		 Column \#2 is the Unicode (in hex as 0xXXXX)
 *		 Column \#3 the Unicode name (follows a comment sign, '\#')
 *
 *	The entries are in KOI8-U order.
 *
 *	Version history
 *	1.0 version: created.
 *
 *	Updated versions of this file may be found in:
 *		<http://www.unicode.org/Public/MAPPINGS/>
 *
 *	Any comments or problems, contact the Unicode Consortium at:
 *	        <http://www.unicode.org/reporting.html>
 *
 */
const CUnicodeMapping::TCharacterName p_koi8_u_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x2500, "BOX DRAWINGS LIGHT HORIZONTAL"},
	{0x81, 0x2502, "BOX DRAWINGS LIGHT VERTICAL"},
	{0x82, 0x250c, "BOX DRAWINGS LIGHT DOWN AND RIGHT"},
	{0x83, 0x2510, "BOX DRAWINGS LIGHT DOWN AND LEFT"},
	{0x84, 0x2514, "BOX DRAWINGS LIGHT UP AND RIGHT"},
	{0x85, 0x2518, "BOX DRAWINGS LIGHT UP AND LEFT"},
	{0x86, 0x251c, "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"},
	{0x87, 0x2524, "BOX DRAWINGS LIGHT VERTICAL AND LEFT"},
	{0x88, 0x252c, "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"},
	{0x89, 0x2534, "BOX DRAWINGS LIGHT UP AND HORIZONTAL"},
	{0x8a, 0x253c, "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"},
	{0x8b, 0x2580, "UPPER HALF BLOCK"},
	{0x8c, 0x2584, "LOWER HALF BLOCK"},
	{0x8d, 0x2588, "FULL BLOCK"},
	{0x8e, 0x258c, "LEFT HALF BLOCK"},
	{0x8f, 0x2590, "RIGHT HALF BLOCK"},
	{0x90, 0x2591, "LIGHT SHADE"},
	{0x91, 0x2592, "MEDIUM SHADE"},
	{0x92, 0x2593, "DARK SHADE"},
	{0x93, 0x2320, "TOP HALF INTEGRAL"},
	{0x94, 0x25a0, "BLACK SQUARE"},
	{0x95, 0x2219, "BULLET OPERATOR"},
	{0x96, 0x221a, "SQUARE ROOT"},
	{0x97, 0x2248, "ALMOST EQUAL TO"},
	{0x98, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0x99, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0x9a, 0x00a0, "NO-BREAK SPACE"},
	{0x9b, 0x2321, "BOTTOM HALF INTEGRAL"},
	{0x9c, 0x00b0, "DEGREE SIGN"},
	{0x9d, 0x00b2, "SUPERSCRIPT TWO"},
	{0x9e, 0x00b7, "MIDDLE DOT"},
	{0x9f, 0x00f7, "DIVISION SIGN"},
	{0xa0, 0x2550, "BOX DRAWINGS DOUBLE HORIZONTAL"},
	{0xa1, 0x2551, "BOX DRAWINGS DOUBLE VERTICAL"},
	{0xa2, 0x2552, "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"},
	{0xa3, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xa4, 0x0454, "CYRILLIC SMALL LETTER UKRAINIAN IE"},
	{0xa5, 0x2554, "BOX DRAWINGS DOUBLE DOWN AND RIGHT"},
	{0xa6, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xa7, 0x0457, "CYRILLIC SMALL LETTER YI (UKRAINIAN)"},
	{0xa8, 0x2557, "BOX DRAWINGS DOUBLE DOWN AND LEFT"},
	{0xa9, 0x2558, "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"},
	{0xaa, 0x2559, "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"},
	{0xab, 0x255a, "BOX DRAWINGS DOUBLE UP AND RIGHT"},
	{0xac, 0x255b, "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"},
	{0xad, 0x0491, "CYRILLIC SMALL LETTER GHE WITH UPTURN"},
	{0xae, 0x255d, "BOX DRAWINGS DOUBLE UP AND LEFT"},
	{0xaf, 0x255e, "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"},
	{0xb0, 0x255f, "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"},
	{0xb1, 0x2560, "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"},
	{0xb2, 0x2561, "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"},
	{0xb3, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xb4, 0x0404, "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
	{0xb5, 0x2563, "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"},
	{0xb6, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb7, 0x0407, "CYRILLIC CAPITAL LETTER YI (UKRAINIAN)"},
	{0xb8, 0x2566, "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"},
	{0xb9, 0x2567, "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"},
	{0xba, 0x2568, "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"},
	{0xbb, 0x2569, "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"},
	{0xbc, 0x256a, "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"},
	{0xbd, 0x0490, "CYRILLIC CAPITAL LETTER GHE WITH UPTURN"},
	{0xbe, 0x256c, "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"},
	{0xbf, 0x00a9, "COPYRIGHT SIGN"},
	{0xc0, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xc1, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xc2, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xc3, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xc4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xc5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xc6, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xc7, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xc8, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xc9, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xca, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xcb, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xcc, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xcd, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xce, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xcf, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xd0, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xd1, 0x044f, "CYRILLIC SMALL LETTER YA"},
	{0xd2, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xd3, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xd4, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xd5, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xd6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xd7, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xd8, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xd9, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xda, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xdb, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xdc, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xdd, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xde, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xdf, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xe0, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0xe1, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xe2, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xe3, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xe4, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xe5, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xe6, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xe7, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xe8, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xe9, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xea, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xeb, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xec, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xed, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xee, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xef, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xf0, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xf1, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xf2, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xf3, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xf4, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xf5, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xf6, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xf7, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xf8, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xf9, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xfa, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xfb, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xfc, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xfd, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xfe, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xff, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"}
};
const size_t n_koi8_u_table_size = sizeof(p_koi8_u_table) / sizeof(p_koi8_u_table[0]);

/**
 *	@brief kz1048 to unicode mapping table
 *
 *
 *	Name:            KZ-1048 to Unicode
 *	Unicode version: 5.0
 *	Table version:   2.0
 *	Table format:    Format A
 *	Date:            10/02/2007
 *	Authors:         Ken Whistler <kenw@sybase.com>
 *
 *	Copyright (c) 2007-2009 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *    General notes:
 *
 *    This table is a modification of Windows cp1251, the
 *    Win Cyrillic code page, for the Kazakhstan national
 *    standard, STRK1048-2002, registered in IANA as
 *    KZ-1048.
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the KZ-1048 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in KZ-1048 order
 *
 */
const CUnicodeMapping::TCharacterName p_kz1048_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x0402, "CYRILLIC CAPITAL LETTER DJE"},
	{0x81, 0x0403, "CYRILLIC CAPITAL LETTER GJE"},
	{0x82, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0x83, 0x0453, "CYRILLIC SMALL LETTER GJE"},
	{0x84, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0x85, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0x86, 0x2020, "DAGGER"},
	{0x87, 0x2021, "DOUBLE DAGGER"},
	{0x88, 0x20ac, "EURO SIGN"},
	{0x89, 0x2030, "PER MILLE SIGN"},
	{0x8a, 0x0409, "CYRILLIC CAPITAL LETTER LJE"},
	{0x8b, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0x8c, 0x040a, "CYRILLIC CAPITAL LETTER NJE"},
	{0x8d, 0x049a, "CYRILLIC CAPITAL LETTER KA WITH DESCENDER"},
	{0x8e, 0x04ba, "CYRILLIC CAPITAL LETTER SHHA"},
	{0x8f, 0x040f, "CYRILLIC CAPITAL LETTER DZHE"},
	{0x90, 0x0452, "CYRILLIC SMALL LETTER DJE"},
	{0x91, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0x92, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0x93, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0x94, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0x95, 0x2022, "BULLET"},
	{0x96, 0x2013, "EN DASH"},
	{0x97, 0x2014, "EM DASH"},
	{0x99, 0x2122, "TRADE MARK SIGN"},
	{0x9a, 0x0459, "CYRILLIC SMALL LETTER LJE"},
	{0x9b, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0x9c, 0x045a, "CYRILLIC SMALL LETTER NJE"},
	{0x9d, 0x049b, "CYRILLIC SMALL LETTER KA WITH DESCENDER"},
	{0x9e, 0x04bb, "CYRILLIC SMALL LETTER SHHA"},
	{0x9f, 0x045f, "CYRILLIC SMALL LETTER DZHE"},
	{0xa0, 0x00a0, "NO-BREAK SPACE"},
	{0xa1, 0x04b0, "CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE"},
	{0xa2, 0x04b1, "CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE"},
	{0xa3, 0x04d8, "CYRILLIC CAPITAL LETTER SCHWA"},
	{0xa4, 0x00a4, "CURRENCY SIGN"},
	{0xa5, 0x04e8, "CYRILLIC CAPITAL LETTER BARRED O"},
	{0xa6, 0x00a6, "BROKEN BAR"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x0401, "CYRILLIC CAPITAL LETTER IO"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x0492, "CYRILLIC CAPITAL LETTER GHE WITH STROKE"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x00ac, "NOT SIGN"},
	{0xad, 0x00ad, "SOFT HYPHEN"},
	{0xae, 0x00ae, "REGISTERED SIGN"},
	{0xaf, 0x04ae, "CYRILLIC CAPITAL LETTER STRAIGHT U"},
	{0xb0, 0x00b0, "DEGREE SIGN"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x0406, "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb3, 0x0456, "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
	{0xb4, 0x04e9, "CYRILLIC SMALL LETTER BARRED O"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x00b7, "MIDDLE DOT"},
	{0xb8, 0x0451, "CYRILLIC SMALL LETTER IO"},
	{0xb9, 0x2116, "NUMERO SIGN"},
	{0xba, 0x0493, "CYRILLIC SMALL LETTER GHE WITH STROKE"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x04d9, "CYRILLIC SMALL LETTER SCHWA"},
	{0xbd, 0x04a2, "CYRILLIC CAPITAL LETTER EN WITH DESCENDER"},
	{0xbe, 0x04a3, "CYRILLIC SMALL LETTER EN WITH DESCENDER"},
	{0xbf, 0x04af, "CYRILLIC SMALL LETTER STRAIGHT U"},
	{0xc0, 0x0410, "CYRILLIC CAPITAL LETTER A"},
	{0xc1, 0x0411, "CYRILLIC CAPITAL LETTER BE"},
	{0xc2, 0x0412, "CYRILLIC CAPITAL LETTER VE"},
	{0xc3, 0x0413, "CYRILLIC CAPITAL LETTER GHE"},
	{0xc4, 0x0414, "CYRILLIC CAPITAL LETTER DE"},
	{0xc5, 0x0415, "CYRILLIC CAPITAL LETTER IE"},
	{0xc6, 0x0416, "CYRILLIC CAPITAL LETTER ZHE"},
	{0xc7, 0x0417, "CYRILLIC CAPITAL LETTER ZE"},
	{0xc8, 0x0418, "CYRILLIC CAPITAL LETTER I"},
	{0xc9, 0x0419, "CYRILLIC CAPITAL LETTER SHORT I"},
	{0xca, 0x041a, "CYRILLIC CAPITAL LETTER KA"},
	{0xcb, 0x041b, "CYRILLIC CAPITAL LETTER EL"},
	{0xcc, 0x041c, "CYRILLIC CAPITAL LETTER EM"},
	{0xcd, 0x041d, "CYRILLIC CAPITAL LETTER EN"},
	{0xce, 0x041e, "CYRILLIC CAPITAL LETTER O"},
	{0xcf, 0x041f, "CYRILLIC CAPITAL LETTER PE"},
	{0xd0, 0x0420, "CYRILLIC CAPITAL LETTER ER"},
	{0xd1, 0x0421, "CYRILLIC CAPITAL LETTER ES"},
	{0xd2, 0x0422, "CYRILLIC CAPITAL LETTER TE"},
	{0xd3, 0x0423, "CYRILLIC CAPITAL LETTER U"},
	{0xd4, 0x0424, "CYRILLIC CAPITAL LETTER EF"},
	{0xd5, 0x0425, "CYRILLIC CAPITAL LETTER HA"},
	{0xd6, 0x0426, "CYRILLIC CAPITAL LETTER TSE"},
	{0xd7, 0x0427, "CYRILLIC CAPITAL LETTER CHE"},
	{0xd8, 0x0428, "CYRILLIC CAPITAL LETTER SHA"},
	{0xd9, 0x0429, "CYRILLIC CAPITAL LETTER SHCHA"},
	{0xda, 0x042a, "CYRILLIC CAPITAL LETTER HARD SIGN"},
	{0xdb, 0x042b, "CYRILLIC CAPITAL LETTER YERU"},
	{0xdc, 0x042c, "CYRILLIC CAPITAL LETTER SOFT SIGN"},
	{0xdd, 0x042d, "CYRILLIC CAPITAL LETTER E"},
	{0xde, 0x042e, "CYRILLIC CAPITAL LETTER YU"},
	{0xdf, 0x042f, "CYRILLIC CAPITAL LETTER YA"},
	{0xe0, 0x0430, "CYRILLIC SMALL LETTER A"},
	{0xe1, 0x0431, "CYRILLIC SMALL LETTER BE"},
	{0xe2, 0x0432, "CYRILLIC SMALL LETTER VE"},
	{0xe3, 0x0433, "CYRILLIC SMALL LETTER GHE"},
	{0xe4, 0x0434, "CYRILLIC SMALL LETTER DE"},
	{0xe5, 0x0435, "CYRILLIC SMALL LETTER IE"},
	{0xe6, 0x0436, "CYRILLIC SMALL LETTER ZHE"},
	{0xe7, 0x0437, "CYRILLIC SMALL LETTER ZE"},
	{0xe8, 0x0438, "CYRILLIC SMALL LETTER I"},
	{0xe9, 0x0439, "CYRILLIC SMALL LETTER SHORT I"},
	{0xea, 0x043a, "CYRILLIC SMALL LETTER KA"},
	{0xeb, 0x043b, "CYRILLIC SMALL LETTER EL"},
	{0xec, 0x043c, "CYRILLIC SMALL LETTER EM"},
	{0xed, 0x043d, "CYRILLIC SMALL LETTER EN"},
	{0xee, 0x043e, "CYRILLIC SMALL LETTER O"},
	{0xef, 0x043f, "CYRILLIC SMALL LETTER PE"},
	{0xf0, 0x0440, "CYRILLIC SMALL LETTER ER"},
	{0xf1, 0x0441, "CYRILLIC SMALL LETTER ES"},
	{0xf2, 0x0442, "CYRILLIC SMALL LETTER TE"},
	{0xf3, 0x0443, "CYRILLIC SMALL LETTER U"},
	{0xf4, 0x0444, "CYRILLIC SMALL LETTER EF"},
	{0xf5, 0x0445, "CYRILLIC SMALL LETTER HA"},
	{0xf6, 0x0446, "CYRILLIC SMALL LETTER TSE"},
	{0xf7, 0x0447, "CYRILLIC SMALL LETTER CHE"},
	{0xf8, 0x0448, "CYRILLIC SMALL LETTER SHA"},
	{0xf9, 0x0449, "CYRILLIC SMALL LETTER SHCHA"},
	{0xfa, 0x044a, "CYRILLIC SMALL LETTER HARD SIGN"},
	{0xfb, 0x044b, "CYRILLIC SMALL LETTER YERU"},
	{0xfc, 0x044c, "CYRILLIC SMALL LETTER SOFT SIGN"},
	{0xfd, 0x044d, "CYRILLIC SMALL LETTER E"},
	{0xfe, 0x044e, "CYRILLIC SMALL LETTER YU"},
	{0xff, 0x044f, "CYRILLIC SMALL LETTER YA"}
};
const size_t n_kz1048_table_size = sizeof(p_kz1048_table) / sizeof(p_kz1048_table[0]);

/**
 *	@brief latin2 to unicode mapping table
 *
 *
 *    Name:     cp10029_MacLatin2 to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10029_MacLatin2 code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10029_MacLatin2 order
 *
 */
const CUnicodeMapping::TCharacterName p_latin2_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x81, 0x0100, "LATIN CAPITAL LETTER A WITH MACRON"},
	{0x82, 0x0101, "LATIN SMALL LETTER A WITH MACRON"},
	{0x83, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x84, 0x0104, "LATIN CAPITAL LETTER A WITH OGONEK"},
	{0x85, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x86, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x87, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x88, 0x0105, "LATIN SMALL LETTER A WITH OGONEK"},
	{0x89, 0x010c, "LATIN CAPITAL LETTER C WITH CARON"},
	{0x8a, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x8b, 0x010d, "LATIN SMALL LETTER C WITH CARON"},
	{0x8c, 0x0106, "LATIN CAPITAL LETTER C WITH ACUTE"},
	{0x8d, 0x0107, "LATIN SMALL LETTER C WITH ACUTE"},
	{0x8e, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x8f, 0x0179, "LATIN CAPITAL LETTER Z WITH ACUTE"},
	{0x90, 0x017a, "LATIN SMALL LETTER Z WITH ACUTE"},
	{0x91, 0x010e, "LATIN CAPITAL LETTER D WITH CARON"},
	{0x92, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x93, 0x010f, "LATIN SMALL LETTER D WITH CARON"},
	{0x94, 0x0112, "LATIN CAPITAL LETTER E WITH MACRON"},
	{0x95, 0x0113, "LATIN SMALL LETTER E WITH MACRON"},
	{0x96, 0x0116, "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
	{0x97, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0x98, 0x0117, "LATIN SMALL LETTER E WITH DOT ABOVE"},
	{0x99, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x9b, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0x9c, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0x9d, 0x011a, "LATIN CAPITAL LETTER E WITH CARON"},
	{0x9e, 0x011b, "LATIN SMALL LETTER E WITH CARON"},
	{0x9f, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xa0, 0x2020, "DAGGER"},
	{0xa1, 0x00b0, "DEGREE SIGN"},
	{0xa2, 0x0118, "LATIN CAPITAL LETTER E WITH OGONEK"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a7, "SECTION SIGN"},
	{0xa5, 0x2022, "BULLET"},
	{0xa6, 0x00b6, "PILCROW SIGN"},
	{0xa7, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x2122, "TRADE MARK SIGN"},
	{0xab, 0x0119, "LATIN SMALL LETTER E WITH OGONEK"},
	{0xac, 0x00a8, "DIAERESIS"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x0123, "LATIN SMALL LETTER G WITH CEDILLA"},
	{0xaf, 0x012e, "LATIN CAPITAL LETTER I WITH OGONEK"},
	{0xb0, 0x012f, "LATIN SMALL LETTER I WITH OGONEK"},
	{0xb1, 0x012a, "LATIN CAPITAL LETTER I WITH MACRON"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x012b, "LATIN SMALL LETTER I WITH MACRON"},
	{0xb5, 0x0136, "LATIN CAPITAL LETTER K WITH CEDILLA"},
	{0xb6, 0x2202, "PARTIAL DIFFERENTIAL"},
	{0xb7, 0x2211, "N-ARY SUMMATION"},
	{0xb8, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xb9, 0x013b, "LATIN CAPITAL LETTER L WITH CEDILLA"},
	{0xba, 0x013c, "LATIN SMALL LETTER L WITH CEDILLA"},
	{0xbb, 0x013d, "LATIN CAPITAL LETTER L WITH CARON"},
	{0xbc, 0x013e, "LATIN SMALL LETTER L WITH CARON"},
	{0xbd, 0x0139, "LATIN CAPITAL LETTER L WITH ACUTE"},
	{0xbe, 0x013a, "LATIN SMALL LETTER L WITH ACUTE"},
	{0xbf, 0x0145, "LATIN CAPITAL LETTER N WITH CEDILLA"},
	{0xc0, 0x0146, "LATIN SMALL LETTER N WITH CEDILLA"},
	{0xc1, 0x0143, "LATIN CAPITAL LETTER N WITH ACUTE"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x221a, "SQUARE ROOT"},
	{0xc4, 0x0144, "LATIN SMALL LETTER N WITH ACUTE"},
	{0xc5, 0x0147, "LATIN CAPITAL LETTER N WITH CARON"},
	{0xc6, 0x2206, "INCREMENT"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x0148, "LATIN SMALL LETTER N WITH CARON"},
	{0xcc, 0x0150, "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
	{0xcd, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xce, 0x0151, "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
	{0xcf, 0x014c, "LATIN CAPITAL LETTER O WITH MACRON"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2014, "EM DASH"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x25ca, "LOZENGE"},
	{0xd8, 0x014d, "LATIN SMALL LETTER O WITH MACRON"},
	{0xd9, 0x0154, "LATIN CAPITAL LETTER R WITH ACUTE"},
	{0xda, 0x0155, "LATIN SMALL LETTER R WITH ACUTE"},
	{0xdb, 0x0158, "LATIN CAPITAL LETTER R WITH CARON"},
	{0xdc, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0xdd, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0xde, 0x0159, "LATIN SMALL LETTER R WITH CARON"},
	{0xdf, 0x0156, "LATIN CAPITAL LETTER R WITH CEDILLA"},
	{0xe0, 0x0157, "LATIN SMALL LETTER R WITH CEDILLA"},
	{0xe1, 0x0160, "LATIN CAPITAL LETTER S WITH CARON"},
	{0xe2, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0xe3, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xe4, 0x0161, "LATIN SMALL LETTER S WITH CARON"},
	{0xe5, 0x015a, "LATIN CAPITAL LETTER S WITH ACUTE"},
	{0xe6, 0x015b, "LATIN SMALL LETTER S WITH ACUTE"},
	{0xe7, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xe8, 0x0164, "LATIN CAPITAL LETTER T WITH CARON"},
	{0xe9, 0x0165, "LATIN SMALL LETTER T WITH CARON"},
	{0xea, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xeb, 0x017d, "LATIN CAPITAL LETTER Z WITH CARON"},
	{0xec, 0x017e, "LATIN SMALL LETTER Z WITH CARON"},
	{0xed, 0x016a, "LATIN CAPITAL LETTER U WITH MACRON"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xf0, 0x016b, "LATIN SMALL LETTER U WITH MACRON"},
	{0xf1, 0x016e, "LATIN CAPITAL LETTER U WITH RING ABOVE"},
	{0xf2, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xf3, 0x016f, "LATIN SMALL LETTER U WITH RING ABOVE"},
	{0xf4, 0x0170, "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
	{0xf5, 0x0171, "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
	{0xf6, 0x0172, "LATIN CAPITAL LETTER U WITH OGONEK"},
	{0xf7, 0x0173, "LATIN SMALL LETTER U WITH OGONEK"},
	{0xf8, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0xf9, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xfa, 0x0137, "LATIN SMALL LETTER K WITH CEDILLA"},
	{0xfb, 0x017b, "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
	{0xfc, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xfd, 0x017c, "LATIN SMALL LETTER Z WITH DOT ABOVE"},
	{0xfe, 0x0122, "LATIN CAPITAL LETTER G WITH CEDILLA"},
	{0xff, 0x02c7, "CARON"}
};
const size_t n_latin2_table_size = sizeof(p_latin2_table) / sizeof(p_latin2_table[0]);

/**
 *	@brief nextstep to unicode mapping table
 *
 *
 *       Name:             NextStep Encoding to Unicode
 *       Unicode version:  1.1
 *       Table version:    0.1
 *       Table format:     Format A
 *       Date:             1999 September 23
 *       Authors:          Rick McGowan
 *
 *       Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 *	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 *	No claims are made as to fitness for any particular purpose.  No
 *	warranties of any kind are expressed or implied.  The recipient
 *	agrees to determine applicability of information provided.  If this
 *	file has been provided on optical media by Unicode, Inc., the sole
 *	remedy for any claim will be exchange of defective media within 90
 *	days of receipt.
 *
 *	Unicode, Inc. hereby grants the right to freely use the information
 *	supplied in this file in the creation of products supporting the
 *	Unicode Standard, and to make copies of this file in any form for
 *	internal or external distribution as long as this notice remains
 *	attached.
 *
 *	General notes:
 *
 *	This table contains the data the Unicode Consortium has on how
 *       NextStep Encoding characters map into Unicode.  Since the first
 *	128 characters (0x0 - 0x7f) are identical to ASCII and Unicode,
 *	this table only maps the NextStep range from 0x80 - 0xFF.
 *
 *	This file is provided for historical reference only and pertains
 *	to NextStep and OpenStep products shipped prior to the aquisition
 *	of NeXT by Apple Computer, Inc.  See http://www.apple.com for
 *	further information.
 *
 *       Format:  Three tab-separated columns
 *                Column \#1 is the NextStep code (in hex as 0xXX)
 *                Column \#2 is the Unicode (in hex as 0xXXXX)
 *                Column \#3 NextStep name, Unicode name (follows a comment sign, '\#')
 *
 *       The entries are in NextStep order
 *
 *       Any comments or problems, contact info@unicode.org
 *
 */
const CUnicodeMapping::TCharacterName p_nextstep_table[] = {
	{0x80, 0x00a0, "NO-BREAK SPACE"},
	{0x81, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0x82, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0x83, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0x84, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0x85, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x86, 0x00c5, "LATIN CAPITAL LETTER A WITH RING"},
	{0x87, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x88, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0x89, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x8a, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0x8b, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0x8c, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0x8d, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0x8e, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0x8f, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0x90, 0x00d0, "LATIN CAPITAL LETTER ETH"},
	{0x91, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x92, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0x93, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0x94, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0x95, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0x96, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x97, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0x98, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0x99, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0x9a, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x9b, 0x00dd, "LATIN CAPITAL LETTER Y WITH ACUTE"},
	{0x9c, 0x00de, "LATIN CAPITAL LETTER THORN"},
	{0x9d, 0x00b5, "MICRO SIGN "},
	{0x9e, 0x00d7, "MULTIPLICATION SIGN"},
	{0x9f, 0x00f7, "DIVISION SIGN"},
	{0xa0, 0x00a9, "COPYRIGHT SIGN"},
	{0xa1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x2044, "FRACTION SLASH"},
	{0xa5, 0x00a5, "YEN SIGN"},
	{0xa6, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xa7, 0x00a7, "SECTION SIGN"},
	{0xa8, 0x00a4, "CURRENCY SIGN"},
	{0xa9, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xaa, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xab, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xac, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0xad, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0xae, 0xfb01, "LATIN SMALL LIGATURE FI"},
	{0xaf, 0xfb02, "LATIN SMALL LIGATURE FL"},
	{0xb0, 0x00ae, "REGISTERED SIGN"},
	{0xb1, 0x2013, "EN DASH"},
	{0xb2, 0x2020, "DAGGER"},
	{0xb3, 0x2021, "DOUBLE DAGGER"},
	{0xb4, 0x00b7, "MIDDLE DOT"},
	{0xb5, 0x00a6, "BROKEN BAR"},
	{0xb6, 0x00b6, "PILCROW SIGN"},
	{0xb7, 0x2022, "BULLET"},
	{0xb8, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0xb9, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xba, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xbb, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xbc, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xbd, 0x2030, "PER MILLE SIGN"},
	{0xbe, 0x00ac, "NOT SIGN"},
	{0xbf, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc0, 0x00b9, "SUPERSCRIPT ONE"},
	{0xc1, 0x02cb, "MODIFIER LETTER GRAVE ACCENT"},
	{0xc2, 0x00b4, "ACUTE ACCENT"},
	{0xc3, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0xc4, 0x02dc, "SMALL TILDE"},
	{0xc5, 0x00af, "MACRON"},
	{0xc6, 0x02d8, "BREVE"},
	{0xc7, 0x02d9, "DOT ABOVE"},
	{0xc8, 0x00a8, "DIAERESIS"},
	{0xc9, 0x00b2, "SUPERSCRIPT TWO"},
	{0xca, 0x02da, "RING ABOVE"},
	{0xcb, 0x00b8, "CEDILLA"},
	{0xcc, 0x00b3, "SUPERSCRIPT THREE"},
	{0xcd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xce, 0x02db, "OGONEK"},
	{0xcf, 0x02c7, "CARON"},
	{0xd0, 0x2014, "EM DASH"},
	{0xd1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xd2, 0x00bc, "VULGAR FRACTION ONE QUARTER"},
	{0xd3, 0x00bd, "VULGAR FRACTION ONE HALF"},
	{0xd4, 0x00be, "VULGAR FRACTION THREE QUARTERS"},
	{0xd5, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0xd6, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0xd7, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0xd8, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0xd9, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0xda, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0xdb, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0xdc, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0xdd, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0xde, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0xdf, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0xe0, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0xe1, 0x00c6, "LATIN CAPITAL LETTER AE"},
	{0xe2, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0xe3, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xe4, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0xe5, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0xe6, 0x00f0, "LATIN SMALL LETTER ETH"},
	{0xe7, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0xe8, 0x0141, "LATIN CAPITAL LETTER L WITH STROKE"},
	{0xe9, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xea, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xeb, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xec, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0xed, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0xee, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0xef, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0xf0, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0xf1, 0x00e6, "LATIN SMALL LETTER AE"},
	{0xf2, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0xf3, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0xf4, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0xf5, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xf6, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xf7, 0x00fd, "LATIN SMALL LETTER Y WITH ACUTE"},
	{0xf8, 0x0142, "LATIN SMALL LETTER L WITH STROKE"},
	{0xf9, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xfa, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xfb, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xfc, 0x00fe, "LATIN SMALL LETTER THORN"},
	{0xfd, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xfe, 0xfffd, ".notdef, REPLACEMENT CHARACTER"},
	{0xff, 0xfffd, ".notdef, REPLACEMENT CHARACTER"}
};
const size_t n_nextstep_table_size = sizeof(p_nextstep_table) / sizeof(p_nextstep_table[0]);

/**
 *	@brief roman to unicode mapping table
 *
 *
 *    Name:     cp10000_MacRoman to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10000_MacRoman code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10000_MacRoman order
 *
 */
const CUnicodeMapping::TCharacterName p_roman_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x81, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x82, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x83, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x84, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x85, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x86, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x87, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x88, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x89, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x8a, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x8b, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x8c, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x8d, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x8e, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x8f, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x90, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x91, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x92, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x93, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x94, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x95, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x96, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x97, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0x98, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x99, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x9b, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0x9c, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0x9d, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x9e, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x9f, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xa0, 0x2020, "DAGGER"},
	{0xa1, 0x00b0, "DEGREE SIGN"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a7, "SECTION SIGN"},
	{0xa5, 0x2022, "BULLET"},
	{0xa6, 0x00b6, "PILCROW SIGN"},
	{0xa7, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x2122, "TRADE MARK SIGN"},
	{0xab, 0x00b4, "ACUTE ACCENT"},
	{0xac, 0x00a8, "DIAERESIS"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0xaf, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xb0, 0x221e, "INFINITY"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x00a5, "YEN SIGN"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x2202, "PARTIAL DIFFERENTIAL"},
	{0xb7, 0x2211, "N-ARY SUMMATION"},
	{0xb8, 0x220f, "N-ARY PRODUCT"},
	{0xb9, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xba, 0x222b, "INTEGRAL"},
	{0xbb, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xbc, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbd, 0x2126, "OHM SIGN"},
	{0xbe, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0xbf, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xc0, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x221a, "SQUARE ROOT"},
	{0xc4, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xc5, 0x2248, "ALMOST EQUAL TO"},
	{0xc6, 0x2206, "INCREMENT"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xcc, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xcd, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xce, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xcf, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2014, "EM DASH"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x25ca, "LOZENGE"},
	{0xd8, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xd9, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xda, 0x2044, "FRACTION SLASH"},
	{0xdb, 0x00a4, "CURRENCY SIGN"},
	{0xdc, 0x2039, "SINGLE LEFT-POINTING ANGLE QUOTATION MARK"},
	{0xdd, 0x203a, "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK"},
	{0xde, 0xfb01, "LATIN SMALL LIGATURE FI"},
	{0xdf, 0xfb02, "LATIN SMALL LIGATURE FL"},
	{0xe0, 0x2021, "DOUBLE DAGGER"},
	{0xe1, 0x00b7, "MIDDLE DOT"},
	{0xe2, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0xe3, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xe4, 0x2030, "PER MILLE SIGN"},
	{0xe5, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xe6, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xe7, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xe8, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xe9, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xea, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xeb, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xec, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xed, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xf1, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xf2, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xf3, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xf4, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xf5, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xf6, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0xf7, 0x02dc, "SMALL TILDE"},
	{0xf8, 0x00af, "MACRON"},
	{0xf9, 0x02d8, "BREVE"},
	{0xfa, 0x02d9, "DOT ABOVE"},
	{0xfb, 0x02da, "RING ABOVE"},
	{0xfc, 0x00b8, "CEDILLA"},
	{0xfd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xfe, 0x02db, "OGONEK"},
	{0xff, 0x02c7, "CARON"}
};
const size_t n_roman_table_size = sizeof(p_roman_table) / sizeof(p_roman_table[0]);

/**
 *	@brief turkish to unicode mapping table
 *
 *
 *    Name:     cp10081_MacTurkish to Unicode table
 *    Unicode version: 2.0
 *    Table version: 2.00
 *    Table format:  Format A
 *    Date:          04/24/96
 *    Contact: Shawn.Steele@microsoft.com
 *                   
 *    General notes: none
 *
 *    Format: Three tab-separated columns
 *        Column \#1 is the cp10081_MacTurkish code (in hex)
 *        Column \#2 is the Unicode (in hex as 0xXXXX)
 *        Column \#3 is the Unicode name (follows a comment sign, '\#')
 *
 *    The entries are in cp10081_MacTurkish order
 *
 */
const CUnicodeMapping::TCharacterName p_turkish_table[] = {
	{0x00, 0x0000, "NULL"},
	{0x01, 0x0001, "START OF HEADING"},
	{0x02, 0x0002, "START OF TEXT"},
	{0x03, 0x0003, "END OF TEXT"},
	{0x04, 0x0004, "END OF TRANSMISSION"},
	{0x05, 0x0005, "ENQUIRY"},
	{0x06, 0x0006, "ACKNOWLEDGE"},
	{0x07, 0x0007, "BELL"},
	{0x08, 0x0008, "BACKSPACE"},
	{0x09, 0x0009, "HORIZONTAL TABULATION"},
	{0x0a, 0x000a, "LINE FEED"},
	{0x0b, 0x000b, "VERTICAL TABULATION"},
	{0x0c, 0x000c, "FORM FEED"},
	{0x0d, 0x000d, "CARRIAGE RETURN"},
	{0x0e, 0x000e, "SHIFT OUT"},
	{0x0f, 0x000f, "SHIFT IN"},
	{0x10, 0x0010, "DATA LINK ESCAPE"},
	{0x11, 0x0011, "DEVICE CONTROL ONE"},
	{0x12, 0x0012, "DEVICE CONTROL TWO"},
	{0x13, 0x0013, "DEVICE CONTROL THREE"},
	{0x14, 0x0014, "DEVICE CONTROL FOUR"},
	{0x15, 0x0015, "NEGATIVE ACKNOWLEDGE"},
	{0x16, 0x0016, "SYNCHRONOUS IDLE"},
	{0x17, 0x0017, "END OF TRANSMISSION BLOCK"},
	{0x18, 0x0018, "CANCEL"},
	{0x19, 0x0019, "END OF MEDIUM"},
	{0x1a, 0x001a, "SUBSTITUTE"},
	{0x1b, 0x001b, "ESCAPE"},
	{0x1c, 0x001c, "FILE SEPARATOR"},
	{0x1d, 0x001d, "GROUP SEPARATOR"},
	{0x1e, 0x001e, "RECORD SEPARATOR"},
	{0x1f, 0x001f, "UNIT SEPARATOR"},
	{0x20, 0x0020, "SPACE"},
	{0x21, 0x0021, "EXCLAMATION MARK"},
	{0x22, 0x0022, "QUOTATION MARK"},
	{0x23, 0x0023, "NUMBER SIGN"},
	{0x24, 0x0024, "DOLLAR SIGN"},
	{0x25, 0x0025, "PERCENT SIGN"},
	{0x26, 0x0026, "AMPERSAND"},
	{0x27, 0x0027, "APOSTROPHE"},
	{0x28, 0x0028, "LEFT PARENTHESIS"},
	{0x29, 0x0029, "RIGHT PARENTHESIS"},
	{0x2a, 0x002a, "ASTERISK"},
	{0x2b, 0x002b, "PLUS SIGN"},
	{0x2c, 0x002c, "COMMA"},
	{0x2d, 0x002d, "HYPHEN-MINUS"},
	{0x2e, 0x002e, "FULL STOP"},
	{0x2f, 0x002f, "SOLIDUS"},
	{0x30, 0x0030, "DIGIT ZERO"},
	{0x31, 0x0031, "DIGIT ONE"},
	{0x32, 0x0032, "DIGIT TWO"},
	{0x33, 0x0033, "DIGIT THREE"},
	{0x34, 0x0034, "DIGIT FOUR"},
	{0x35, 0x0035, "DIGIT FIVE"},
	{0x36, 0x0036, "DIGIT SIX"},
	{0x37, 0x0037, "DIGIT SEVEN"},
	{0x38, 0x0038, "DIGIT EIGHT"},
	{0x39, 0x0039, "DIGIT NINE"},
	{0x3a, 0x003a, "COLON"},
	{0x3b, 0x003b, "SEMICOLON"},
	{0x3c, 0x003c, "LESS-THAN SIGN"},
	{0x3d, 0x003d, "EQUALS SIGN"},
	{0x3e, 0x003e, "GREATER-THAN SIGN"},
	{0x3f, 0x003f, "QUESTION MARK"},
	{0x40, 0x0040, "COMMERCIAL AT"},
	{0x41, 0x0041, "LATIN CAPITAL LETTER A"},
	{0x42, 0x0042, "LATIN CAPITAL LETTER B"},
	{0x43, 0x0043, "LATIN CAPITAL LETTER C"},
	{0x44, 0x0044, "LATIN CAPITAL LETTER D"},
	{0x45, 0x0045, "LATIN CAPITAL LETTER E"},
	{0x46, 0x0046, "LATIN CAPITAL LETTER F"},
	{0x47, 0x0047, "LATIN CAPITAL LETTER G"},
	{0x48, 0x0048, "LATIN CAPITAL LETTER H"},
	{0x49, 0x0049, "LATIN CAPITAL LETTER I"},
	{0x4a, 0x004a, "LATIN CAPITAL LETTER J"},
	{0x4b, 0x004b, "LATIN CAPITAL LETTER K"},
	{0x4c, 0x004c, "LATIN CAPITAL LETTER L"},
	{0x4d, 0x004d, "LATIN CAPITAL LETTER M"},
	{0x4e, 0x004e, "LATIN CAPITAL LETTER N"},
	{0x4f, 0x004f, "LATIN CAPITAL LETTER O"},
	{0x50, 0x0050, "LATIN CAPITAL LETTER P"},
	{0x51, 0x0051, "LATIN CAPITAL LETTER Q"},
	{0x52, 0x0052, "LATIN CAPITAL LETTER R"},
	{0x53, 0x0053, "LATIN CAPITAL LETTER S"},
	{0x54, 0x0054, "LATIN CAPITAL LETTER T"},
	{0x55, 0x0055, "LATIN CAPITAL LETTER U"},
	{0x56, 0x0056, "LATIN CAPITAL LETTER V"},
	{0x57, 0x0057, "LATIN CAPITAL LETTER W"},
	{0x58, 0x0058, "LATIN CAPITAL LETTER X"},
	{0x59, 0x0059, "LATIN CAPITAL LETTER Y"},
	{0x5a, 0x005a, "LATIN CAPITAL LETTER Z"},
	{0x5b, 0x005b, "LEFT SQUARE BRACKET"},
	{0x5c, 0x005c, "REVERSE SOLIDUS"},
	{0x5d, 0x005d, "RIGHT SQUARE BRACKET"},
	{0x5e, 0x005e, "CIRCUMFLEX ACCENT"},
	{0x5f, 0x005f, "LOW LINE"},
	{0x60, 0x0060, "GRAVE ACCENT"},
	{0x61, 0x0061, "LATIN SMALL LETTER A"},
	{0x62, 0x0062, "LATIN SMALL LETTER B"},
	{0x63, 0x0063, "LATIN SMALL LETTER C"},
	{0x64, 0x0064, "LATIN SMALL LETTER D"},
	{0x65, 0x0065, "LATIN SMALL LETTER E"},
	{0x66, 0x0066, "LATIN SMALL LETTER F"},
	{0x67, 0x0067, "LATIN SMALL LETTER G"},
	{0x68, 0x0068, "LATIN SMALL LETTER H"},
	{0x69, 0x0069, "LATIN SMALL LETTER I"},
	{0x6a, 0x006a, "LATIN SMALL LETTER J"},
	{0x6b, 0x006b, "LATIN SMALL LETTER K"},
	{0x6c, 0x006c, "LATIN SMALL LETTER L"},
	{0x6d, 0x006d, "LATIN SMALL LETTER M"},
	{0x6e, 0x006e, "LATIN SMALL LETTER N"},
	{0x6f, 0x006f, "LATIN SMALL LETTER O"},
	{0x70, 0x0070, "LATIN SMALL LETTER P"},
	{0x71, 0x0071, "LATIN SMALL LETTER Q"},
	{0x72, 0x0072, "LATIN SMALL LETTER R"},
	{0x73, 0x0073, "LATIN SMALL LETTER S"},
	{0x74, 0x0074, "LATIN SMALL LETTER T"},
	{0x75, 0x0075, "LATIN SMALL LETTER U"},
	{0x76, 0x0076, "LATIN SMALL LETTER V"},
	{0x77, 0x0077, "LATIN SMALL LETTER W"},
	{0x78, 0x0078, "LATIN SMALL LETTER X"},
	{0x79, 0x0079, "LATIN SMALL LETTER Y"},
	{0x7a, 0x007a, "LATIN SMALL LETTER Z"},
	{0x7b, 0x007b, "LEFT CURLY BRACKET"},
	{0x7c, 0x007c, "VERTICAL LINE"},
	{0x7d, 0x007d, "RIGHT CURLY BRACKET"},
	{0x7e, 0x007e, "TILDE"},
	{0x7f, 0x007f, "DELETE"},
	{0x80, 0x00c4, "LATIN CAPITAL LETTER A WITH DIAERESIS"},
	{0x81, 0x00c5, "LATIN CAPITAL LETTER A WITH RING ABOVE"},
	{0x82, 0x00c7, "LATIN CAPITAL LETTER C WITH CEDILLA"},
	{0x83, 0x00c9, "LATIN CAPITAL LETTER E WITH ACUTE"},
	{0x84, 0x00d1, "LATIN CAPITAL LETTER N WITH TILDE"},
	{0x85, 0x00d6, "LATIN CAPITAL LETTER O WITH DIAERESIS"},
	{0x86, 0x00dc, "LATIN CAPITAL LETTER U WITH DIAERESIS"},
	{0x87, 0x00e1, "LATIN SMALL LETTER A WITH ACUTE"},
	{0x88, 0x00e0, "LATIN SMALL LETTER A WITH GRAVE"},
	{0x89, 0x00e2, "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
	{0x8a, 0x00e4, "LATIN SMALL LETTER A WITH DIAERESIS"},
	{0x8b, 0x00e3, "LATIN SMALL LETTER A WITH TILDE"},
	{0x8c, 0x00e5, "LATIN SMALL LETTER A WITH RING ABOVE"},
	{0x8d, 0x00e7, "LATIN SMALL LETTER C WITH CEDILLA"},
	{0x8e, 0x00e9, "LATIN SMALL LETTER E WITH ACUTE"},
	{0x8f, 0x00e8, "LATIN SMALL LETTER E WITH GRAVE"},
	{0x90, 0x00ea, "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
	{0x91, 0x00eb, "LATIN SMALL LETTER E WITH DIAERESIS"},
	{0x92, 0x00ed, "LATIN SMALL LETTER I WITH ACUTE"},
	{0x93, 0x00ec, "LATIN SMALL LETTER I WITH GRAVE"},
	{0x94, 0x00ee, "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
	{0x95, 0x00ef, "LATIN SMALL LETTER I WITH DIAERESIS"},
	{0x96, 0x00f1, "LATIN SMALL LETTER N WITH TILDE"},
	{0x97, 0x00f3, "LATIN SMALL LETTER O WITH ACUTE"},
	{0x98, 0x00f2, "LATIN SMALL LETTER O WITH GRAVE"},
	{0x99, 0x00f4, "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
	{0x9a, 0x00f6, "LATIN SMALL LETTER O WITH DIAERESIS"},
	{0x9b, 0x00f5, "LATIN SMALL LETTER O WITH TILDE"},
	{0x9c, 0x00fa, "LATIN SMALL LETTER U WITH ACUTE"},
	{0x9d, 0x00f9, "LATIN SMALL LETTER U WITH GRAVE"},
	{0x9e, 0x00fb, "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
	{0x9f, 0x00fc, "LATIN SMALL LETTER U WITH DIAERESIS"},
	{0xa0, 0x2020, "DAGGER"},
	{0xa1, 0x00b0, "DEGREE SIGN"},
	{0xa2, 0x00a2, "CENT SIGN"},
	{0xa3, 0x00a3, "POUND SIGN"},
	{0xa4, 0x00a7, "SECTION SIGN"},
	{0xa5, 0x2022, "BULLET"},
	{0xa6, 0x00b6, "PILCROW SIGN"},
	{0xa7, 0x00df, "LATIN SMALL LETTER SHARP S"},
	{0xa8, 0x00ae, "REGISTERED SIGN"},
	{0xa9, 0x00a9, "COPYRIGHT SIGN"},
	{0xaa, 0x2122, "TRADE MARK SIGN"},
	{0xab, 0x00b4, "ACUTE ACCENT"},
	{0xac, 0x00a8, "DIAERESIS"},
	{0xad, 0x2260, "NOT EQUAL TO"},
	{0xae, 0x00c6, "LATIN CAPITAL LIGATURE AE"},
	{0xaf, 0x00d8, "LATIN CAPITAL LETTER O WITH STROKE"},
	{0xb0, 0x221e, "INFINITY"},
	{0xb1, 0x00b1, "PLUS-MINUS SIGN"},
	{0xb2, 0x2264, "LESS-THAN OR EQUAL TO"},
	{0xb3, 0x2265, "GREATER-THAN OR EQUAL TO"},
	{0xb4, 0x00a5, "YEN SIGN"},
	{0xb5, 0x00b5, "MICRO SIGN"},
	{0xb6, 0x2202, "PARTIAL DIFFERENTIAL"},
	{0xb7, 0x2211, "N-ARY SUMMATION"},
	{0xb8, 0x220f, "N-ARY PRODUCT"},
	{0xb9, 0x03c0, "GREEK SMALL LETTER PI"},
	{0xba, 0x222b, "INTEGRAL"},
	{0xbb, 0x00aa, "FEMININE ORDINAL INDICATOR"},
	{0xbc, 0x00ba, "MASCULINE ORDINAL INDICATOR"},
	{0xbd, 0x2126, "OHM SIGN"},
	{0xbe, 0x00e6, "LATIN SMALL LIGATURE AE"},
	{0xbf, 0x00f8, "LATIN SMALL LETTER O WITH STROKE"},
	{0xc0, 0x00bf, "INVERTED QUESTION MARK"},
	{0xc1, 0x00a1, "INVERTED EXCLAMATION MARK"},
	{0xc2, 0x00ac, "NOT SIGN"},
	{0xc3, 0x221a, "SQUARE ROOT"},
	{0xc4, 0x0192, "LATIN SMALL LETTER F WITH HOOK"},
	{0xc5, 0x2248, "ALMOST EQUAL TO"},
	{0xc6, 0x2206, "INCREMENT"},
	{0xc7, 0x00ab, "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc8, 0x00bb, "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
	{0xc9, 0x2026, "HORIZONTAL ELLIPSIS"},
	{0xca, 0x00a0, "NO-BREAK SPACE"},
	{0xcb, 0x00c0, "LATIN CAPITAL LETTER A WITH GRAVE"},
	{0xcc, 0x00c3, "LATIN CAPITAL LETTER A WITH TILDE"},
	{0xcd, 0x00d5, "LATIN CAPITAL LETTER O WITH TILDE"},
	{0xce, 0x0152, "LATIN CAPITAL LIGATURE OE"},
	{0xcf, 0x0153, "LATIN SMALL LIGATURE OE"},
	{0xd0, 0x2013, "EN DASH"},
	{0xd1, 0x2014, "EM DASH"},
	{0xd2, 0x201c, "LEFT DOUBLE QUOTATION MARK"},
	{0xd3, 0x201d, "RIGHT DOUBLE QUOTATION MARK"},
	{0xd4, 0x2018, "LEFT SINGLE QUOTATION MARK"},
	{0xd5, 0x2019, "RIGHT SINGLE QUOTATION MARK"},
	{0xd6, 0x00f7, "DIVISION SIGN"},
	{0xd7, 0x25ca, "LOZENGE"},
	{0xd8, 0x00ff, "LATIN SMALL LETTER Y WITH DIAERESIS"},
	{0xd9, 0x0178, "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
	{0xda, 0x011e, "LATIN CAPITAL LETTER G WITH BREVE"},
	{0xdb, 0x011f, "LATIN SMALL LETTER G WITH BREVE"},
	{0xdc, 0x0130, "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
	{0xdd, 0x0131, "LATIN SMALL LETTER DOTLESS I"},
	{0xde, 0x015e, "LATIN CAPITAL LETTER S WITH CEDILLA"},
	{0xdf, 0x015f, "LATIN SMALL LETTER S WITH CEDILLA"},
	{0xe0, 0x2021, "DOUBLE DAGGER"},
	{0xe1, 0x00b7, "MIDDLE DOT"},
	{0xe2, 0x201a, "SINGLE LOW-9 QUOTATION MARK"},
	{0xe3, 0x201e, "DOUBLE LOW-9 QUOTATION MARK"},
	{0xe4, 0x2030, "PER MILLE SIGN"},
	{0xe5, 0x00c2, "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
	{0xe6, 0x00ca, "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
	{0xe7, 0x00c1, "LATIN CAPITAL LETTER A WITH ACUTE"},
	{0xe8, 0x00cb, "LATIN CAPITAL LETTER E WITH DIAERESIS"},
	{0xe9, 0x00c8, "LATIN CAPITAL LETTER E WITH GRAVE"},
	{0xea, 0x00cd, "LATIN CAPITAL LETTER I WITH ACUTE"},
	{0xeb, 0x00ce, "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
	{0xec, 0x00cf, "LATIN CAPITAL LETTER I WITH DIAERESIS"},
	{0xed, 0x00cc, "LATIN CAPITAL LETTER I WITH GRAVE"},
	{0xee, 0x00d3, "LATIN CAPITAL LETTER O WITH ACUTE"},
	{0xef, 0x00d4, "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
	{0xf1, 0x00d2, "LATIN CAPITAL LETTER O WITH GRAVE"},
	{0xf2, 0x00da, "LATIN CAPITAL LETTER U WITH ACUTE"},
	{0xf3, 0x00db, "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
	{0xf4, 0x00d9, "LATIN CAPITAL LETTER U WITH GRAVE"},
	{0xf6, 0x02c6, "MODIFIER LETTER CIRCUMFLEX ACCENT"},
	{0xf7, 0x02dc, "SMALL TILDE"},
	{0xf8, 0x00af, "MACRON"},
	{0xf9, 0x02d8, "BREVE"},
	{0xfa, 0x02d9, "DOT ABOVE"},
	{0xfb, 0x02da, "RING ABOVE"},
	{0xfc, 0x00b8, "CEDILLA"},
	{0xfd, 0x02dd, "DOUBLE ACUTE ACCENT"},
	{0xfe, 0x02db, "OGONEK"},
	{0xff, 0x02c7, "CARON"}
};
const size_t n_turkish_table_size = sizeof(p_turkish_table) / sizeof(p_turkish_table[0]);

static const struct {
	const char *p_s_name;
	const CUnicodeMapping::TCharacterName *p_table;
	size_t n_table_size;
} p_mappings_table[] = {
	{"iso-8859-1", p_iso_8859_1_table, n_iso_8859_1_table_size},
	{"iso-8859-10", p_iso_8859_10_table, n_iso_8859_10_table_size},
	{"iso-8859-11", p_iso_8859_11_table, n_iso_8859_11_table_size},
	{"iso-8859-13", p_iso_8859_13_table, n_iso_8859_13_table_size},
	{"iso-8859-14", p_iso_8859_14_table, n_iso_8859_14_table_size},
	{"iso-8859-15", p_iso_8859_15_table, n_iso_8859_15_table_size},
	{"iso-8859-16", p_iso_8859_16_table, n_iso_8859_16_table_size},
	{"iso-8859-2", p_iso_8859_2_table, n_iso_8859_2_table_size},
	{"iso-8859-3", p_iso_8859_3_table, n_iso_8859_3_table_size},
	{"iso-8859-4", p_iso_8859_4_table, n_iso_8859_4_table_size},
	{"iso-8859-5", p_iso_8859_5_table, n_iso_8859_5_table_size},
	{"iso-8859-6", p_iso_8859_6_table, n_iso_8859_6_table_size},
	{"iso-8859-7", p_iso_8859_7_table, n_iso_8859_7_table_size},
	{"iso-8859-8", p_iso_8859_8_table, n_iso_8859_8_table_size},
	{"iso-8859-9", p_iso_8859_9_table, n_iso_8859_9_table_size},
	{"atarist", p_atarist_table, n_atarist_table_size},
	{"cp-037", p_cp_037_table, n_cp_037_table_size},
	{"cp-1006", p_cp_1006_table, n_cp_1006_table_size},
	{"cp-1026", p_cp_1026_table, n_cp_1026_table_size},
	{"cp-1250", p_cp_1250_table, n_cp_1250_table_size},
	{"cp-1251", p_cp_1251_table, n_cp_1251_table_size},
	{"cp-1252", p_cp_1252_table, n_cp_1252_table_size},
	{"cp-1253", p_cp_1253_table, n_cp_1253_table_size},
	{"cp-1254", p_cp_1254_table, n_cp_1254_table_size},
	{"cp-1255", p_cp_1255_table, n_cp_1255_table_size},
	{"cp-1256", p_cp_1256_table, n_cp_1256_table_size},
	{"cp-1257", p_cp_1257_table, n_cp_1257_table_size},
	{"cp-1258", p_cp_1258_table, n_cp_1258_table_size},
	{"cp-424", p_cp_424_table, n_cp_424_table_size},
	{"cp-437", p_cp_437_table, n_cp_437_table_size},
	{"cp-500", p_cp_500_table, n_cp_500_table_size},
	{"cp-737", p_cp_737_table, n_cp_737_table_size},
	{"cp-775", p_cp_775_table, n_cp_775_table_size},
	{"cp-850", p_cp_850_table, n_cp_850_table_size},
	{"cp-852", p_cp_852_table, n_cp_852_table_size},
	{"cp-855", p_cp_855_table, n_cp_855_table_size},
	{"cp-856", p_cp_856_table, n_cp_856_table_size},
	{"cp-857", p_cp_857_table, n_cp_857_table_size},
	{"cp-860", p_cp_860_table, n_cp_860_table_size},
	{"cp-861", p_cp_861_table, n_cp_861_table_size},
	{"cp-862", p_cp_862_table, n_cp_862_table_size},
	{"cp-863", p_cp_863_table, n_cp_863_table_size},
	{"cp-864", p_cp_864_table, n_cp_864_table_size},
	{"cp-865", p_cp_865_table, n_cp_865_table_size},
	{"cp-866", p_cp_866_table, n_cp_866_table_size},
	{"cp-869", p_cp_869_table, n_cp_869_table_size},
	{"cp-874", p_cp_874_table, n_cp_874_table_size},
	{"cp-875", p_cp_875_table, n_cp_875_table_size},
	{"cyrillic", p_cyrillic_table, n_cyrillic_table_size},
	{"greek", p_greek_table, n_greek_table_size},
	{"iceland", p_iceland_table, n_iceland_table_size},
	{"koi8-r", p_koi8_r_table, n_koi8_r_table_size},
	{"koi8-u", p_koi8_u_table, n_koi8_u_table_size},
	{"kz1048", p_kz1048_table, n_kz1048_table_size},
	{"latin2", p_latin2_table, n_latin2_table_size},
	{"nextstep", p_nextstep_table, n_nextstep_table_size},
	{"roman", p_roman_table, n_roman_table_size},
	{"turkish", p_turkish_table, n_turkish_table_size}
};
const size_t n_mapping_num = sizeof(p_mappings_table) / sizeof(p_mappings_table[0]);

}; // ~__8bit_to_unicode_mapping_tables
