/*
								+-----------------------------------+
								|                                   |
								|  ***  Function call wrapper  ***  |
								|                                   |
								|   Copyright   -tHE SWINe- 2014   |
								|                                   |
								|         FunctionCall.inl          |
								|                                   |
								+-----------------------------------+
*/

#pragma once
#ifndef __FUNCTION_CALL_WRAPPER_LIBRARY_INLINES_INCLUDED
#define __FUNCTION_CALL_WRAPPER_LIBRARY_INLINES_INCLUDED

/**
 *	@file FunctionCall.inl
 *	@brief function call wrapper inlines
 *	@author -tHE SWINe-
 *	@date 2014
 *
 *	This was heavily inspired by the interface of the XPU library by Nader Khammassi et. al.
 */

// continuation of function_call namespace from FunctionCall.h
namespace function_call {

/**
 *	@brief intellisense helper class
 *
 *	This is not required or practical to be in a separate class at all, the code was originally
 *	a part of CStaticFunctionUtils, but that broke intellisense in Visual Studio 2008.
 *
 *	@tparam CResultType is data type of function call result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *	@tparam A15 is type of function parameter 15
 *	@tparam CArgType_List is typelist with the original argument list
 *
 *	@note This needs to be extended to allow functions with more than 16 arguments.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6,
	class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14, class A15,
	class CArgType_List>
class CStaticFunTable {
public:
	/**
	 *	@brief intermediates, stored as enum
	 */
	enum {
		n_arg_num = CTypelistLength<CArgType_List>::n_result /**< @brief the number of arguments */
		// repeated calculation but the intellisense will break if this is passed as an arg
	};

	typedef CResultType  (*F0)(); /**< @brief type of pointer to a function with no arguments */
	typedef CResultType  (*F1)(A0); /**< @brief type of pointer to a function with a single argument */
	typedef CResultType  (*F2)(A0, A1); /**< @brief type of pointer to a function with 2 arguments */
	typedef CResultType  (*F3)(A0, A1, A2); /**< @brief type of pointer to a function with 3 arguments */
	typedef CResultType  (*F4)(A0, A1, A2, A3); /**< @brief type of pointer to a function with 4 arguments */
	typedef CResultType  (*F5)(A0, A1, A2, A3, A4); /**< @brief type of pointer to a function with 5 arguments */
	typedef CResultType  (*F6)(A0, A1, A2, A3, A4, A5); /**< @brief type of pointer to a function with 6 arguments */
	typedef CResultType  (*F7)(A0, A1, A2, A3, A4, A5, A6); /**< @brief type of pointer to a function with 7 arguments */
	typedef CResultType  (*F8)(A0, A1, A2, A3, A4, A5, A6, A7); /**< @brief type of pointer to a function with 8 arguments */
	typedef CResultType  (*F9)(A0, A1, A2, A3, A4, A5, A6, A7, A8); /**< @brief type of pointer to a function with 9 arguments */
	typedef CResultType (*F10)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9); /**< @brief type of pointer to a function with 10 arguments */
	typedef CResultType (*F11)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10); /**< @brief type of pointer to a function with 11 arguments */
	typedef CResultType (*F12)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11); /**< @brief type of pointer to a function with 12 arguments */
	typedef CResultType (*F13)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12); /**< @brief type of pointer to a function with 13 arguments */
	typedef CResultType (*F14)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13); /**< @brief type of pointer to a function with 14 arguments */
	typedef CResultType (*F15)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14); /**< @brief type of pointer to a function with 15 arguments */
	typedef CResultType (*F16)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15); /**< @brief type of pointer to a function with 16 arguments */

	typedef MakeTypelist17(F0, F1, F2, F3, F4, F5, F6, F7, F8, F9,
		F10, F11, F12, F13, F14, F15, F16) CAllFuncPtrTypes; /**< @brief list of pointer types to functions with different numbers of arguments */

	typedef typename CTypelistItemAt<CAllFuncPtrTypes, n_arg_num>::CResult _TyFuncType; /**< @brief function pointer type */
};

/**
 *	@brief static function utilities tempalte
 *
 *	@tparam CResultType is data type of function call result
 *	@tparam CArgType_List is typelist containing types of function arguments
 *
 *	@note This needs to be extended to allow functions with more than 16 arguments.
 */
template <class CResultType, class CArgType_List>
class CStaticFunctionUtils {
public:
	typedef CResultType _TyResultType; /**< @brief data type of function call result */
	typedef CArgType_List _TyArgType_List; /**< @brief typelist containing types of function arguments */
	typedef CStaticCallContext<CResultType, CArgType_List> _TyCallContext; /**< @brief call context specialization */

	/**
	 *	@brief function parameters, stored as enum
	 */
	enum {
		n_arg_num = CTypelistLength<CArgType_List>::n_result, /**< @brief the number of arguments */
		b_is_void = CIsSameType<CResultType, void>::b_result /**< @brief <tt>void</tt> function flag (true if the function returns <tt>void</tt>, false otherwise) */
	};

	typedef typename CStaticAssert<(n_arg_num <= 16)>::FUNCTIONS_WITH_MORE_THAN_16_ARGUMENTS_NOT_SUPPORTED CAssert0; /**< @brief argument count assertion */

	//typedef typename CConcatTypelist<CArgType_List, MakeTypelist1(int)>::CResult _TyIntPaddedArgType_List; /**< @brief typelist containing types of function arguments with a single <tt>int</tt> appended */
	typedef CTypelist<int, CArgType_List> _TyIntPrefixedArgType_List; /**< @brief typelist containing types of function arguments with a single <tt>int</tt> prepended */
	// note that the prefixed type is easier to compile than the padded

	/**
	 *	@brief argument type template, default <tt>int</tt> (this is only used to shorten the code)
	 *	@tparam n_index is zero-based argument index
	 */
	template <const int n_index>
	struct CArgType {
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
		typedef typename CTypelistItemAt<_TyIntPrefixedArgType_List,
			((n_index < n_arg_num)? n_index + 1 : 0)>::CResult T; /**< @brief the extracted type */
		// the enum sadly does not evaluate correctly in msvc 6.0
#else // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
		/**
		 *	@brief intermediates, stored as enum
		 */
		enum {
			n_ip_index = ((n_index < n_arg_num)? n_index + 1 : 0) /**< @brief index in the padded list (breaks intellisense if directly in the template argument list) */
		};

		typedef typename CTypelistItemAt<_TyIntPrefixedArgType_List, n_ip_index>::CResult T; /**< @brief the extracted type */
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
	};

	typedef typename CArgType<0>::T A0; /**< @brief type of argument 0 */
	typedef typename CArgType<1>::T A1; /**< @brief type of argument 1 */
	typedef typename CArgType<2>::T A2; /**< @brief type of argument 2 */
	typedef typename CArgType<3>::T A3; /**< @brief type of argument 3 */
	typedef typename CArgType<4>::T A4; /**< @brief type of argument 4 */
	typedef typename CArgType<5>::T A5; /**< @brief type of argument 5 */
	typedef typename CArgType<6>::T A6; /**< @brief type of argument 6 */
	typedef typename CArgType<7>::T A7; /**< @brief type of argument 7 */
	typedef typename CArgType<8>::T A8; /**< @brief type of argument 8 */
	typedef typename CArgType<9>::T A9; /**< @brief type of argument 9 */
	typedef typename CArgType<10>::T A10; /**< @brief type of argument 10 */
	typedef typename CArgType<11>::T A11; /**< @brief type of argument 11 */
	typedef typename CArgType<12>::T A12; /**< @brief type of argument 12 */
	typedef typename CArgType<13>::T A13; /**< @brief type of argument 13 */
	typedef typename CArgType<14>::T A14; /**< @brief type of argument 14 */
	typedef typename CArgType<15>::T A15; /**< @brief type of argument 15 */
	typedef typename CArgType<16>::T A16; /**< @brief type of argument 16 */
	// msvc 6.0 requires default int here

	typedef CStaticFunTable<CResultType, A0, A1, A2, A3, A4, A5, A6, A7,
		A8, A9, A10, A11, A12, A13, A14, A15, CArgType_List> CFunTable; /**< @brief helper function pointer table (solves intellisense issues in visual studio) */
	typedef typename CFunTable::_TyFuncType _TyFuncType; /**< @copydoc CStaticFunTable::_TyFuncType */

	typedef _TyFuncType _TyContextData; /**< @brief static call context data (just the function pointer) */

	/**
	 *	@brief argument value helper (only used to make the code inside t_Call() shorter)
	 *	@tparam n_index is zero-based argument index
	 */
	template <const int n_index>
	struct AV {
		typedef typename CArgType<n_index>::T T; /**< @brief argument type */

		/**
		 *	@brief gets argument value
		 *	@param[in] r_args is reference to arguments storage
		 *	@return Returns the argument value.
		 */
		static inline T V(CArgumentStorage<CArgType_List> &r_args)
		{
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
			return r_args.t_Argument6(fc_msvc6::CArgResoloveTag<T, n_index>());
#else // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
			return r_args.IS_OMMIT(template) t_Argument<T, n_index>();
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
		}
	};

	/**
	 *	@brief performs the function call
	 *
	 *	@param[in] p_func is function pointer
	 *	@param[in] r_args is reference to the values of the arguments
	 *
	 *	@return Returns the result of the function call.
	 *
	 *	@note This may throw exceptions, generated by the original function.
	 *	@note r_args cannot be a const reference, would change types of the arguments inside.
	 */
	static inline _TyResultType t_Call(_TyFuncType p_func, CArgumentStorage<CArgType_List> &r_args)
	{
		CArgumentStorage<CArgType_List> &A = r_args; // rename to shorten the code below

		// without the cast to int, g++ complains "case label exceeded maximum value of a type"
		switch(int(n_arg_num)) { // compile-time constant, should optimize away (but we can generate templates for this, it will just be more verbose)
		default: // need default, otherwise "not all control paths return a value"
			_ASSERTE(!n_arg_num);
			// this case intentionally falls through
		case 0:  return  (*(typename CFunTable::F0)p_func)();
		case 1:  return  (*(typename CFunTable::F1)p_func)(AV<0>::V(A));
		case 2:  return  (*(typename CFunTable::F2)p_func)(AV<0>::V(A), AV<1>::V(A));
		case 3:  return  (*(typename CFunTable::F3)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A));
		case 4:  return  (*(typename CFunTable::F4)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A));
		case 5:  return  (*(typename CFunTable::F5)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A));
		case 6:  return  (*(typename CFunTable::F6)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A));
		case 7:  return  (*(typename CFunTable::F7)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A));
		case 8:  return  (*(typename CFunTable::F8)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A));
		case 9:  return  (*(typename CFunTable::F9)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A));
		case 10: return (*(typename CFunTable::F10)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A));
		case 11: return (*(typename CFunTable::F11)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A));
		case 12: return (*(typename CFunTable::F12)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A));
		case 13: return (*(typename CFunTable::F13)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A));
		case 14: return (*(typename CFunTable::F14)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A));
		case 15: return (*(typename CFunTable::F15)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A), AV<14>::V(A));
		case 16: return (*(typename CFunTable::F16)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A), AV<14>::V(A), AV<15>::V(A));
		};
		// this can't be made with variable number of arguments
	}
};

/**
 *	@brief intellisense helper class
 *
 *	This is not required or practical to be in a separate class at all, the code was originally
 *	a part of CMemberFunctionUtils, but that broke intellisense in Visual Studio 2008.
 *
 *	@tparam CClassType is type of the class the function is member of (without a const qualifier)
 *	@tparam CResultType is data type of function call result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *	@tparam A15 is type of function parameter 15
 *	@tparam b_is_const is const function flag (set if the member function is declared as const)
 *	@tparam CArgType_List is typelist with the original argument list
 *
 *	@note This needs to be extended to allow functions with more than 16 arguments.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5,
	class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14, class A15,
	const bool b_is_const, class CArgType_List>
class CMemberFunTable {
public:
	/**
	 *	@brief intermediates, stored as enum
	 */
	enum {
		n_arg_num = CTypelistLength<CArgType_List>::n_result /**< @brief the number of arguments */
		// repeated calculation but the intellisense will break if this is passed as an arg
	};

	typedef typename CChooseType<CResultType (CClassType::*)() const,
		CResultType (CClassType::*)(), b_is_const>::CResult F0; /**< @brief type of pointer to a function with no arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0) const,
		CResultType (CClassType::*)(A0), b_is_const>::CResult F1; /**< @brief type of pointer to a function with a single argument */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1) const,
		CResultType (CClassType::*)(A0, A1), b_is_const>::CResult F2; /**< @brief type of pointer to a function with 2 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2) const,
		CResultType (CClassType::*)(A0, A1, A2), b_is_const>::CResult F3; /**< @brief type of pointer to a function with 3 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3) const,
		CResultType (CClassType::*)(A0, A1, A2, A3), b_is_const>::CResult F4; /**< @brief type of pointer to a function with 4 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4), b_is_const>::CResult F5; /**< @brief type of pointer to a function with 5 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5), b_is_const>::CResult F6; /**< @brief type of pointer to a function with 6 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6), b_is_const>::CResult F7; /**< @brief type of pointer to a function with 7 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7), b_is_const>::CResult F8; /**< @brief type of pointer to a function with 8 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8), b_is_const>::CResult F9; /**< @brief type of pointer to a function with 9 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9), b_is_const>::CResult F10; /**< @brief type of pointer to a function with 10 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10), b_is_const>::CResult F11; /**< @brief type of pointer to a function with 11 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11), b_is_const>::CResult F12; /**< @brief type of pointer to a function with 12 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12), b_is_const>::CResult F13; /**< @brief type of pointer to a function with 13 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13), b_is_const>::CResult F14; /**< @brief type of pointer to a function with 14 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14), b_is_const>::CResult F15; /**< @brief type of pointer to a function with 15 arguments */
	typedef typename CChooseType<CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15) const,
		CResultType (CClassType::*)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15), b_is_const>::CResult F16; /**< @brief type of pointer to a function with 16 arguments */

	typedef MakeTypelist17(F0, F1, F2, F3, F4, F5, F6, F7, F8, F9,
		F10, F11, F12, F13, F14, F15, F16) CAllFuncPtrTypes; /**< @brief list of pointer types to functions with different numbers of arguments */

	typedef typename CTypelistItemAt<CAllFuncPtrTypes, n_arg_num>::CResult _TyFuncType; /**< @brief function pointer type */
};

/**
 *	@brief member function utilities tempalte
 *
 *	@tparam CClassType is type of the class the function is member of (without a const qualifier)
 *	@tparam CResultType is data type of function call result
 *	@tparam CArgType_List is typelist containing types of function arguments
 *	@tparam b_is_const_function is const function flag (set if the member function is declared as const)
 *
 *	@note This needs to be extended to allow functions with more than 16 arguments.
 */
template <class CClassType, class CResultType, class CArgType_List, const bool b_is_const_function /*= false*/> // otherwise says "redefinition of default parameter : parameter 4"
class CMemberFunctionUtils {
public:
	typedef typename CChooseType<const CClassType, CClassType, b_is_const_function>::CResult _TyClassType; /**< @brief const-qualified type of the class the function is member of */
	typedef CClassType _TyNonConstClassType; /**< @brief type of the class the function is member of (non-const) */
	typedef CResultType _TyResultType; /**< @brief data type of function call result */
	typedef CArgType_List _TyArgType_List; /**< @brief typelist containing types of function arguments */
	typedef CMemberCallContext<_TyClassType, CResultType, CArgType_List, b_is_const_function> _TyCallContext; /**< @brief call context specialization */

	/**
	 *	@brief function parameters, stored as enum
	 */
	enum {
		n_arg_num = CTypelistLength<CArgType_List>::n_result, /**< @brief the number of arguments */
		b_is_void = CIsSameType<CResultType, void>::b_result, /**< @brief <tt>void</tt> function flag (true if the function returns <tt>void</tt>, false otherwise) */
		b_is_const = b_is_const_function /**< @brief <tt>const</tt> function flag (true if the function is declared as <tt>const</tt>, false otherwise) */
	};

	typedef typename CStaticAssert<(n_arg_num <= 16)>::FUNCTIONS_WITH_MORE_THAN_16_ARGUMENTS_NOT_SUPPORTED CAssert0; /**< @brief argument count assertion */

	//typedef typename CConcatTypelist<CArgType_List, MakeTypelist1(int)>::CResult _TyIntPaddedArgType_List; /**< @brief typelist containing types of function arguments with a single <tt>int</tt> appended */
	typedef CTypelist<int, CArgType_List> _TyIntPrefixedArgType_List; /**< @brief typelist containing types of function arguments with a single <tt>int</tt> prepended */
	// note that the prefixed type is easier to compile than the padded

	/**
	 *	@brief argument type template, default <tt>int</tt> (this is only used to shorten the code)
	 *	@tparam n_index is zero-based argument index
	 */
	template <const int n_index>
	struct CArgType {
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
		typedef typename CTypelistItemAt<_TyIntPrefixedArgType_List,
			((n_arg_num <= n_index)? 0 : n_index + 1)>::CResult T; /**< @brief the extracted type */
		// the enum sadly does not evaluate correctly in msvc 6.0
#else // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
		/**
		 *	@brief intermediates, stored as enum
		 */
		enum {
			n_ip_index = ((n_arg_num <= n_index)? 0 : n_index + 1) /**< @brief index in the padded list (breaks intellisense if directly in the template argument list) */
		};

		typedef typename CTypelistItemAt<_TyIntPrefixedArgType_List, n_ip_index>::CResult T; /**< @brief the extracted type */
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
	};

	typedef typename CArgType<0>::T A0; /**< @brief type of argument 0 */
	typedef typename CArgType<1>::T A1; /**< @brief type of argument 1 */
	typedef typename CArgType<2>::T A2; /**< @brief type of argument 2 */
	typedef typename CArgType<3>::T A3; /**< @brief type of argument 3 */
	typedef typename CArgType<4>::T A4; /**< @brief type of argument 4 */
	typedef typename CArgType<5>::T A5; /**< @brief type of argument 5 */
	typedef typename CArgType<6>::T A6; /**< @brief type of argument 6 */
	typedef typename CArgType<7>::T A7; /**< @brief type of argument 7 */
	typedef typename CArgType<8>::T A8; /**< @brief type of argument 8 */
	typedef typename CArgType<9>::T A9; /**< @brief type of argument 9 */
	typedef typename CArgType<10>::T A10; /**< @brief type of argument 10 */
	typedef typename CArgType<11>::T A11; /**< @brief type of argument 11 */
	typedef typename CArgType<12>::T A12; /**< @brief type of argument 12 */
	typedef typename CArgType<13>::T A13; /**< @brief type of argument 13 */
	typedef typename CArgType<14>::T A14; /**< @brief type of argument 14 */
	typedef typename CArgType<15>::T A15; /**< @brief type of argument 15 */
	typedef typename CArgType<16>::T A16; /**< @brief type of argument 16 */
	// msvc 6.0 requires default int here

	typedef CMemberFunTable<CClassType, CResultType, A0, A1, A2, A3, A4, A5, A6, A7,
		A8, A9, A10, A11, A12, A13, A14, A15, b_is_const_function, CArgType_List> CFunTable; /**< @brief helper function pointer table (solves intellisense issues in visual studio) */
	typedef typename CFunTable::_TyFuncType _TyFuncType; /**< @copydoc CMemberFunTable::_TyFuncType */

	typedef std::pair<_TyClassType*, _TyFuncType> _TyContextData; /**< @brief static call context data (just the function pointer) */

	/**
	 *	@brief argument value helper (only used to make the code inside t_Call() shorter)
	 *	@tparam n_index is zero-based argument index
	 */
	template <const int n_index>
	struct AV {
		typedef typename CArgType<n_index>::T T; /**< @brief argument type */

		/**
		 *	@brief gets argument value
		 *	@param[in] r_args is reference to arguments storage
		 *	@return Returns the argument value.
		 */
		static inline T V(CArgumentStorage<CArgType_List> &r_args)
		{
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200
			return r_args.t_Argument6(fc_msvc6::CArgResoloveTag<T, n_index>());
#else // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
			return r_args.IS_OMMIT(template) t_Argument<T, n_index>();
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200
		}
	};

	/**
	 *	@brief performs the function call
	 *
	 *	@param[in] p_func is function pointer
	 *	@param[in] r_args is reference to the values of the arguments
	 *
	 *	@return Returns the result of the function call.
	 *
	 *	@note This may throw exceptions, generated by the original function.
	 *	@note r_args cannot be a const reference, would change types of the arguments inside.
	 */
	static inline _TyResultType t_Call(_TyClassType *p_class, _TyFuncType p_func, CArgumentStorage<CArgType_List> &r_args)
	{
		CArgumentStorage<CArgType_List> &A = r_args; // rename to shorten the code below

		// without the cast to int, g++ complains "case label exceeded maximum value of a type"
		switch(int(n_arg_num)) { // compile-time constant, should optimize away (but we can generate templates for this, it will just be more verbose)
		default: // need default, otherwise "not all control paths return a value"
			_ASSERTE(!n_arg_num);
			// this case intentionally falls through
		case 0:  return  (p_class->*(typename CFunTable::F0)p_func)();
		case 1:  return  (p_class->*(typename CFunTable::F1)p_func)(AV<0>::V(A));
		case 2:  return  (p_class->*(typename CFunTable::F2)p_func)(AV<0>::V(A), AV<1>::V(A));
		case 3:  return  (p_class->*(typename CFunTable::F3)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A));
		case 4:  return  (p_class->*(typename CFunTable::F4)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A));
		case 5:  return  (p_class->*(typename CFunTable::F5)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A));
		case 6:  return  (p_class->*(typename CFunTable::F6)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A));
		case 7:  return  (p_class->*(typename CFunTable::F7)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A));
		case 8:  return  (p_class->*(typename CFunTable::F8)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A));
		case 9:  return  (p_class->*(typename CFunTable::F9)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A));
		case 10: return (p_class->*(typename CFunTable::F10)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A));
		case 11: return (p_class->*(typename CFunTable::F11)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A));
		case 12: return (p_class->*(typename CFunTable::F12)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A));
		case 13: return (p_class->*(typename CFunTable::F13)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A));
		case 14: return (p_class->*(typename CFunTable::F14)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A));
		case 15: return (p_class->*(typename CFunTable::F15)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A), AV<14>::V(A));
		case 16: return (p_class->*(typename CFunTable::F16)p_func)(AV<0>::V(A), AV<1>::V(A), AV<2>::V(A), AV<3>::V(A), AV<4>::V(A), AV<5>::V(A), AV<6>::V(A), AV<7>::V(A), AV<8>::V(A), AV<9>::V(A), AV<10>::V(A), AV<11>::V(A), AV<12>::V(A), AV<13>::V(A), AV<14>::V(A), AV<15>::V(A));
		};
		// this can't be made with variable number of arguments
	}

	// todo - see if we can make msvc 6.0 support void functions
};
/**
 *	@brief base class of the arguments initializer
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerBase {
	typedef CArgumentStorage<CTypelistEnd> CMakeArgs; /**< @brief argument chain starter */
	typedef CArgumentStorage<typename CFunUtil::_TyArgType_List> CArgStorage; /**< @brief argument storage specialization */
	typedef typename CFunUtil::_TyContextData _TyCtxData; /**< @brief call context data type */
	typedef typename CFunUtil::_TyCallContext _TyCallCtx; /**< @brief call context type */
	typedef typename CFunUtil::A0 A0; /**< @copydoc CStaticFunctionUtils::A0 */
	typedef typename CFunUtil::A1 A1; /**< @copydoc CStaticFunctionUtils::A1 */
	typedef typename CFunUtil::A2 A2; /**< @copydoc CStaticFunctionUtils::A2 */
	typedef typename CFunUtil::A3 A3; /**< @copydoc CStaticFunctionUtils::A3 */
	typedef typename CFunUtil::A4 A4; /**< @copydoc CStaticFunctionUtils::A4 */
	typedef typename CFunUtil::A5 A5; /**< @copydoc CStaticFunctionUtils::A5 */
	typedef typename CFunUtil::A6 A6; /**< @copydoc CStaticFunctionUtils::A6 */
	typedef typename CFunUtil::A7 A7; /**< @copydoc CStaticFunctionUtils::A7 */
	typedef typename CFunUtil::A8 A8; /**< @copydoc CStaticFunctionUtils::A8 */
	typedef typename CFunUtil::A9 A9; /**< @copydoc CStaticFunctionUtils::A9 */
	typedef typename CFunUtil::A10 A10; /**< @copydoc CStaticFunctionUtils::A10 */
	typedef typename CFunUtil::A11 A11; /**< @copydoc CStaticFunctionUtils::A11 */
	typedef typename CFunUtil::A12 A12; /**< @copydoc CStaticFunctionUtils::A12 */
	typedef typename CFunUtil::A13 A13; /**< @copydoc CStaticFunctionUtils::A13 */
	typedef typename CFunUtil::A14 A14; /**< @copydoc CStaticFunctionUtils::A14 */
	typedef typename CFunUtil::A15 A15; /**< @copydoc CStaticFunctionUtils::A15 */
	typedef typename CFunUtil::A16 A16; /**< @copydoc CStaticFunctionUtils::A16 */
	// to make the initializer specializations code shorter

	typename CFunUtil::_TyContextData m_t_ctx; /**< @brief call context data */
	// need to remember the class instance pointer / function pointer in order
	// to pass it to the final call context instance
};

#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER <= 1200

/**
 *	@brief argument initializer for msvc 6.0
 *
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 *	@tparam n_arg_num is number of function arguments
 */
template <class CFunUtil, const unsigned int n_arg_num>
class CArgsInitializer {
public:
	/**
	 *	@brief argument initializer implementation
	 *	@tparam n_arg_num is number of function arguments
	 */
	template <const unsigned int n_arg_num>
	struct CInitializerImpl;

/**
 *	@def PUSH
 *	@brief chains arguments to argument storage (only used to make the code shorter)
 *
 *	@param[in] CArgType is argument type
 *	@param[in] t_arg_value is argument value
 */
#define PUSH(CArgType,t_arg_value) .Push6(fc_msvc6::CTypeTag<CArgType>(), (t_arg_value))

	/**
	 *	@brief argument initializer implementation (specialization for 0 arguments)
	 */
	template <>
	struct CInitializerImpl<0> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()() { return _TyCallCtx(m_t_ctx, CMakeArgs()); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 1 arguments)
	 */
	template <>
	struct CInitializerImpl<1> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *	@param[in] t_arg0 is value of argument 0
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 2 arguments)
	 */
	template <>
	struct CInitializerImpl<2> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 3 arguments)
	 */
	template <>
	struct CInitializerImpl<3> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A2, t_arg2) PUSH(A1, t_arg1)
			PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 4 arguments)
	 */
	template <>
	struct CInitializerImpl<4> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A3, t_arg3) PUSH(A2, t_arg2)
			PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 5 arguments)
	 */
	template <>
	struct CInitializerImpl<5> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A4, t_arg4) PUSH(A3, t_arg3)
			PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 6 arguments)
	 */
	template <>
	struct CInitializerImpl<6> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A5, t_arg5) PUSH(A4, t_arg4)
			PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 7 arguments)
	 */
	template <>
	struct CInitializerImpl<7> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A6, t_arg6) PUSH(A5, t_arg5)
			PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1)
			PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 8 arguments)
	 */
	template <>
	struct CInitializerImpl<8> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A7, t_arg7) PUSH(A6, t_arg6)
			PUSH(A5, t_arg5) PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2)
			PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 9 arguments)
	 */
	template <>
	struct CInitializerImpl<9> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A8, t_arg8) PUSH(A7, t_arg7)
			PUSH(A6, t_arg6) PUSH(A5, t_arg5) PUSH(A4, t_arg4) PUSH(A3, t_arg3)
			PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 10 arguments)
	 */
	template <>
	struct CInitializerImpl<10> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A9, t_arg9) PUSH(A8, t_arg8)
			PUSH(A7, t_arg7) PUSH(A6, t_arg6) PUSH(A5, t_arg5) PUSH(A4, t_arg4)
			PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 11 arguments)
	 */
	template <>
	struct CInitializerImpl<11> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A10, t_arg10) PUSH(A9, t_arg9)
			PUSH(A8, t_arg8) PUSH(A7, t_arg7) PUSH(A6, t_arg6) PUSH(A5, t_arg5)
			PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1)
			PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 12 arguments)
	 */
	template <>
	struct CInitializerImpl<12> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *	@param[in] t_arg11 is value of argument 11
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10, A11 t_arg11)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A11, t_arg11) PUSH(A10, t_arg10)
			PUSH(A9, t_arg9) PUSH(A8, t_arg8) PUSH(A7, t_arg7) PUSH(A6, t_arg6)
			PUSH(A5, t_arg5) PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2)
			PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 13 arguments)
	 */
	template <>
	struct CInitializerImpl<13> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *	@param[in] t_arg11 is value of argument 11
		 *	@param[in] t_arg12 is value of argument 12
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10, A11 t_arg11, A12 t_arg12)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A12, t_arg12) PUSH(A11, t_arg11)
			PUSH(A10, t_arg10) PUSH(A9, t_arg9) PUSH(A8, t_arg8) PUSH(A7, t_arg7)
			PUSH(A6, t_arg6) PUSH(A5, t_arg5) PUSH(A4, t_arg4) PUSH(A3, t_arg3)
			PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 14 arguments)
	 */
	template <>
	struct CInitializerImpl<14> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *	@param[in] t_arg11 is value of argument 11
		 *	@param[in] t_arg12 is value of argument 12
		 *	@param[in] t_arg13 is value of argument 13
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10, A11 t_arg11, A12 t_arg12, A13 t_arg13)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A13, t_arg13) PUSH(A12, t_arg12)
			PUSH(A11, t_arg11) PUSH(A10, t_arg10) PUSH(A9, t_arg9) PUSH(A8, t_arg8)
			PUSH(A7, t_arg7) PUSH(A6, t_arg6) PUSH(A5, t_arg5) PUSH(A4, t_arg4)
			PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 15 arguments)
	 */
	template <>
	struct CInitializerImpl<15> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *	@param[in] t_arg11 is value of argument 11
		 *	@param[in] t_arg12 is value of argument 12
		 *	@param[in] t_arg13 is value of argument 13
		 *	@param[in] t_arg14 is value of argument 14
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10, A11 t_arg11, A12 t_arg12, A13 t_arg13, A14 t_arg14)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A14, t_arg14) PUSH(A13, t_arg13)
			PUSH(A12, t_arg12) PUSH(A11, t_arg11) PUSH(A10, t_arg10) PUSH(A9, t_arg9)
			PUSH(A8, t_arg8) PUSH(A7, t_arg7) PUSH(A6, t_arg6) PUSH(A5, t_arg5)
			PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2) PUSH(A1, t_arg1)
			PUSH(A0, t_arg0)); }
	};

	/**
	 *	@brief argument initializer implementation (specialization for 16 arguments)
	 */
	template <>
	struct CInitializerImpl<16> : public CArgsInitializerBase<CFunUtil> {
		/**
		 *	@brief argument initialization operator
		 *
		 *	@param[in] t_arg0 is value of argument 0
		 *	@param[in] t_arg1 is value of argument 1
		 *	@param[in] t_arg2 is value of argument 2
		 *	@param[in] t_arg3 is value of argument 3
		 *	@param[in] t_arg4 is value of argument 4
		 *	@param[in] t_arg5 is value of argument 5
		 *	@param[in] t_arg6 is value of argument 6
		 *	@param[in] t_arg7 is value of argument 7
		 *	@param[in] t_arg8 is value of argument 8
		 *	@param[in] t_arg9 is value of argument 9
		 *	@param[in] t_arg10 is value of argument 10
		 *	@param[in] t_arg11 is value of argument 11
		 *	@param[in] t_arg12 is value of argument 12
		 *	@param[in] t_arg13 is value of argument 13
		 *	@param[in] t_arg14 is value of argument 14
		 *	@param[in] t_arg15 is value of argument 15
		 *
		 *	@return Returns a complete call context with arguments initialized.
		 */
		inline _TyCallCtx operator ()(A0 t_arg0, A1 t_arg1, A2 t_arg2, A3 t_arg3,
			A4 t_arg4, A5 t_arg5, A6 t_arg6, A7 t_arg7, A8 t_arg8, A9 t_arg9,
			A10 t_arg10, A11 t_arg11, A12 t_arg12, A13 t_arg13, A14 t_arg14,
			A15 t_arg15)
		{ return _TyCallCtx(m_t_ctx, CMakeArgs() PUSH(A15, t_arg15) PUSH(A14, t_arg14)
			PUSH(A13, t_arg13) PUSH(A12, t_arg12) PUSH(A11, t_arg11) PUSH(A10, t_arg10)
			PUSH(A9, t_arg9) PUSH(A8, t_arg8) PUSH(A7, t_arg7) PUSH(A6, t_arg6)
			PUSH(A5, t_arg5) PUSH(A4, t_arg4) PUSH(A3, t_arg3) PUSH(A2, t_arg2)
			PUSH(A1, t_arg1) PUSH(A0, t_arg0)); }
	};

#undef PUSH // don't leave it defined

	typedef CInitializerImpl<n_arg_num> CImpl;

protected:
	typename CFunUtil::_TyContextData m_t_ctx; /**< @brief call context data */

public:
	/**
	 *	@brief default constructor
	 *	@param[in] t_ctx is call context data
	 */
	inline CArgsInitializer(typename CFunUtil::_TyContextData t_ctx)
		:m_t_ctx(t_ctx)
	{}

	/**
	 *	@brief gets initializer specialization (works arround the inability
	 *		of msvc 6.0 to explicitly specialize template function calls)
	 *	@return Returns an instance of argument initializer.
	 */
	inline CImpl Get()
	{
		CInitializerImpl<n_arg_num> impl;
		impl.m_t_ctx = m_t_ctx; // this is only a pointer / pair of pointers, it is ok to copy
		return impl;
	}
};

#else // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200

/**
 *	@brief implementation of argument initializer
 *
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 *	@tparam n_arg_num is number of function arguments
 */
template <class CFunUtil, const unsigned int n_arg_num>
struct CArgsInitializerImpl; // forward declaration

/**
 *	@brief argument initializer implementation (specialization for no arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 0> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *	@return Returns a complete call context.
	 */
	inline typename B::_TyCallCtx operator ()()
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs()); }
};

/**
 *	@brief argument initializer implementation (specialization for 1 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 1> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *	@param[in] t_arg0 is value of argument 0
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 2 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 2> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 3 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 3> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 4 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 4> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 5 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 5> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 6 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 6> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 7 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 7> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A6>(
		t_arg6).IS_OMMIT(template) Push<typename B::A5>(t_arg5).IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 8 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 8> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A7>(
		t_arg7).IS_OMMIT(template) Push<typename B::A6>(t_arg6).IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 9 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 9> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A8>(
		t_arg8).IS_OMMIT(template) Push<typename B::A7>(t_arg7).IS_OMMIT(template) Push<typename B::A6>(
		t_arg6).IS_OMMIT(template) Push<typename B::A5>(t_arg5).IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 10 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 10> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A9>(
		t_arg9).IS_OMMIT(template) Push<typename B::A8>(t_arg8).IS_OMMIT(template) Push<typename B::A7>(
		t_arg7).IS_OMMIT(template) Push<typename B::A6>(t_arg6).IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 11 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 11> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A10>(
		t_arg10).IS_OMMIT(template) Push<typename B::A9>(t_arg9).IS_OMMIT(template) Push<typename B::A8>(
		t_arg8).IS_OMMIT(template) Push<typename B::A7>(t_arg7).IS_OMMIT(template) Push<typename B::A6>(
		t_arg6).IS_OMMIT(template) Push<typename B::A5>(t_arg5).IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 12 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 12> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *	@param[in] t_arg11 is value of argument 11
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10, typename B::A11 t_arg11)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A11>(
		t_arg11).IS_OMMIT(template) Push<typename B::A10>(t_arg10).IS_OMMIT(template) Push<typename B::A9>(
		t_arg9).IS_OMMIT(template) Push<typename B::A8>(t_arg8).IS_OMMIT(template) Push<typename B::A7>(
		t_arg7).IS_OMMIT(template) Push<typename B::A6>(t_arg6).IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 13 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 13> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *	@param[in] t_arg11 is value of argument 11
	 *	@param[in] t_arg12 is value of argument 12
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10, typename B::A11 t_arg11, typename B::A12 t_arg12)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A12>(
		t_arg12).IS_OMMIT(template) Push<typename B::A11>(t_arg11).IS_OMMIT(template) Push<typename B::A10>(
		t_arg10).IS_OMMIT(template) Push<typename B::A9>(t_arg9).IS_OMMIT(template) Push<typename B::A8>(
		t_arg8).IS_OMMIT(template) Push<typename B::A7>(t_arg7).IS_OMMIT(template) Push<typename B::A6>(
		t_arg6).IS_OMMIT(template) Push<typename B::A5>(t_arg5).IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 14 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 14> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *	@param[in] t_arg11 is value of argument 11
	 *	@param[in] t_arg12 is value of argument 12
	 *	@param[in] t_arg13 is value of argument 13
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10, typename B::A11 t_arg11, typename B::A12 t_arg12,
		typename B::A13 t_arg13)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A13>(
		t_arg13).IS_OMMIT(template) Push<typename B::A12>(t_arg12).IS_OMMIT(template) Push<typename B::A11>(
		t_arg11).IS_OMMIT(template) Push<typename B::A10>(t_arg10).IS_OMMIT(template) Push<typename B::A9>(
		t_arg9).IS_OMMIT(template) Push<typename B::A8>(t_arg8).IS_OMMIT(template) Push<typename B::A7>(
		t_arg7).IS_OMMIT(template) Push<typename B::A6>(t_arg6).IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 15 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 15> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *	@param[in] t_arg11 is value of argument 11
	 *	@param[in] t_arg12 is value of argument 12
	 *	@param[in] t_arg13 is value of argument 13
	 *	@param[in] t_arg14 is value of argument 14
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10, typename B::A11 t_arg11, typename B::A12 t_arg12,
		typename B::A13 t_arg13, typename B::A14 t_arg14)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A14>(
		t_arg14).IS_OMMIT(template) Push<typename B::A13>(t_arg13).IS_OMMIT(template) Push<typename B::A12>(
		t_arg12).IS_OMMIT(template) Push<typename B::A11>(t_arg11).IS_OMMIT(template) Push<typename B::A10>(
		t_arg10).IS_OMMIT(template) Push<typename B::A9>(t_arg9).IS_OMMIT(template) Push<typename B::A8>(
		t_arg8).IS_OMMIT(template) Push<typename B::A7>(t_arg7).IS_OMMIT(template) Push<typename B::A6>(
		t_arg6).IS_OMMIT(template) Push<typename B::A5>(t_arg5).IS_OMMIT(template) Push<typename B::A4>(
		t_arg4).IS_OMMIT(template) Push<typename B::A3>(t_arg3).IS_OMMIT(template) Push<typename B::A2>(
		t_arg2).IS_OMMIT(template) Push<typename B::A1>(t_arg1).IS_OMMIT(template) Push<typename B::A0>(
		t_arg0)); }
};

/**
 *	@brief argument initializer implementation (specialization for 16 arguments)
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 */
template <class CFunUtil>
struct CArgsInitializerImpl<CFunUtil, 16> : public CArgsInitializerBase<CFunUtil> {
	typedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */

	/**
	 *	@brief argument initialization operator
	 *
	 *	@param[in] t_arg0 is value of argument 0
	 *	@param[in] t_arg1 is value of argument 1
	 *	@param[in] t_arg2 is value of argument 2
	 *	@param[in] t_arg3 is value of argument 3
	 *	@param[in] t_arg4 is value of argument 4
	 *	@param[in] t_arg5 is value of argument 5
	 *	@param[in] t_arg6 is value of argument 6
	 *	@param[in] t_arg7 is value of argument 7
	 *	@param[in] t_arg8 is value of argument 8
	 *	@param[in] t_arg9 is value of argument 9
	 *	@param[in] t_arg10 is value of argument 10
	 *	@param[in] t_arg11 is value of argument 11
	 *	@param[in] t_arg12 is value of argument 12
	 *	@param[in] t_arg13 is value of argument 13
	 *	@param[in] t_arg14 is value of argument 14
	 *	@param[in] t_arg15 is value of argument 15
	 *
	 *	@return Returns a complete call context with arguments initialized.
	 */
	inline typename B::_TyCallCtx operator ()(typename B::A0 t_arg0,
		typename B::A1 t_arg1, typename B::A2 t_arg2, typename B::A3 t_arg3,
		typename B::A4 t_arg4, typename B::A5 t_arg5, typename B::A6 t_arg6,
		typename B::A7 t_arg7, typename B::A8 t_arg8, typename B::A9 t_arg9,
		typename B::A10 t_arg10, typename B::A11 t_arg11, typename B::A12 t_arg12,
		typename B::A13 t_arg13, typename B::A14 t_arg14, typename B::A15 t_arg15)
	{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs().IS_OMMIT(template) Push<typename B::A15>(
		t_arg15).IS_OMMIT(template) Push<typename B::A14>(t_arg14).IS_OMMIT(template) Push<typename B::A13>(
		t_arg13).IS_OMMIT(template) Push<typename B::A12>(t_arg12).IS_OMMIT(template) Push<typename B::A11>(
		t_arg11).IS_OMMIT(template) Push<typename B::A10>(t_arg10).IS_OMMIT(template) Push<typename B::A9>(
		t_arg9).IS_OMMIT(template) Push<typename B::A8>(t_arg8).IS_OMMIT(template) Push<typename B::A7>(
		t_arg7).IS_OMMIT(template) Push<typename B::A6>(t_arg6).IS_OMMIT(template) Push<typename B::A5>(
		t_arg5).IS_OMMIT(template) Push<typename B::A4>(t_arg4).IS_OMMIT(template) Push<typename B::A3>(
		t_arg3).IS_OMMIT(template) Push<typename B::A2>(t_arg2).IS_OMMIT(template) Push<typename B::A1>(
		t_arg1).IS_OMMIT(template) Push<typename B::A0>(t_arg0)); }
};

/**
 *	@brief argument initializer initializer
 *
 *	@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils
 *		or function_call::CMemberFunctionUtils
 *	@tparam n_arg_num is number of function arguments
 */
template <class CFunUtil, const unsigned int n_arg_num>
class CArgsInitializer : public CArgsInitializerImpl<CFunUtil, n_arg_num> {
public:
	/**
	 *	@brief default constructor
	 *	@param[in] t_ctx is call context data
	 */
	inline CArgsInitializer(typename CFunUtil::_TyContextData t_context)
	{
		CArgsInitializerBase<CFunUtil>::m_t_ctx = t_context; // this is only a pointer / pair of pointers, it is ok to copy
		// can't member-initialize as it is member of CArgsInitializerBase and
		// we want to avoid writing constructors for all the specializations
	}
};

#endif // _MSC_VER && !__MWERKS__ && _MSC_VER <= 1200

// t_odo - test in msvc6, linux
// t_odo - doc, write generator for those as well
// t_odo - modify documentation of existing GetCallSignature() to explicitly mention static function

/**
 *	@brief static function call signature extraction helper
 *	@tparam CResultType is type of function result
 *	@param[in] p_func is pointer to the static function
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType>
CArgsInitializer<CStaticFunctionUtils<CResultType, CTypelistEnd>, 0> GetCallSignature(CResultType (*p_func)(void))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, CTypelistEnd> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd>, 0> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(void))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd, true>, 0> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(void) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd, true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist1(A0)>, 1> GetCallSignature(CResultType (*p_func)(A0))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist1(A0)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist1(A0)>, 1> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist1(A0)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist1(A0), true>, 1> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist1(A0), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist2(A0, A1)>, 2> GetCallSignature(CResultType (*p_func)(A0, A1))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist2(A0, A1)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist2(A0, A1)>, 2> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist2(A0, A1)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist2(A0, A1), true>, 2> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist2(A0, A1), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist3(A0, A1, A2)>, 3> GetCallSignature(CResultType (*p_func)(A0, A1, A2))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist3(A0, A1, A2)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist3(A0, A1, A2)>, 3> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist3(A0, A1, A2)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist3(A0, A1, A2), true>, 3> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist3(A0, A1, A2), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist4(A0, A1, A2, A3)>, 4> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist4(A0, A1, A2, A3)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist4(A0, A1, A2, A3)>, 4> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist4(A0, A1, A2, A3)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist4(A0, A1, A2, A3), true>, 4> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist4(A0, A1, A2, A3), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist5(A0, A1, A2, A3, A4)>, 5> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist5(A0, A1, A2, A3, A4)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist5(A0, A1, A2, A3, A4)>, 5> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist5(A0, A1, A2, A3, A4)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist5(A0, A1, A2, A3, A4), true>, 5> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist5(A0, A1, A2, A3, A4), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5)>, 6> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5)>, 6> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5), true>, 6> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist6(A0, A1, A2, A3, A4, A5), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6)>, 7> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6)>, 7> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6), true>, 7> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist7(A0, A1, A2, A3, A4, A5, A6), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7)>, 8> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7)>, 8> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7), true>, 8> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist8(A0, A1, A2, A3, A4, A5, A6, A7), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8)>, 9> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8)>, 9> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8), true>, 9> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist9(A0, A1, A2, A3, A4, A5, A6, A7, A8), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9)>, 10> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9)>, 10> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9), true>, 10> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist10(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)>, 11> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)>, 11> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10), true>, 11> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist11(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11)>, 12> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11)>, 12> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11), true>, 12> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist12(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12)>, 13> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12)>, 13> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12), true>, 13> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist13(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13)>, 14> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13)>, 14> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13), true>, 14> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist14(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14)>, 15> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14)>, 15> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14), true>, 15> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist15(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief static function call signature extraction helper
 *
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *	@tparam A15 is type of function parameter 15
 *
 *	@param[in] p_func is pointer to the static function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14, class A15>
CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15)>, 16> GetCallSignature(CResultType (*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15))
{
	return CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15)> >(p_func).Initializer();
}

/**
 *	@brief member function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *	@tparam A15 is type of function parameter 15
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14, class A15>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15)>, 16> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15))
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15)> >(std::make_pair(p_instance, p_func)).Initializer();
}

/**
 *	@brief member const function call signature extraction helper
 *
 *	@tparam CClassType is type of function result
 *	@tparam CResultType is type of function result
 *	@tparam A0 is type of function parameter 0
 *	@tparam A1 is type of function parameter 1
 *	@tparam A2 is type of function parameter 2
 *	@tparam A3 is type of function parameter 3
 *	@tparam A4 is type of function parameter 4
 *	@tparam A5 is type of function parameter 5
 *	@tparam A6 is type of function parameter 6
 *	@tparam A7 is type of function parameter 7
 *	@tparam A8 is type of function parameter 8
 *	@tparam A9 is type of function parameter 9
 *	@tparam A10 is type of function parameter 10
 *	@tparam A11 is type of function parameter 11
 *	@tparam A12 is type of function parameter 12
 *	@tparam A13 is type of function parameter 13
 *	@tparam A14 is type of function parameter 14
 *	@tparam A15 is type of function parameter 15
 *
 *	@param[in] r_instance is reference to the class instance
 *	@param[in] p_func is pointer to the member const function
 *
 *	@return Returns a specialized instance of CCallSignature.
 */
template <class CClassType, class CResultType, class A0, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class A11, class A12, class A13, class A14, class A15>
CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15), true>, 16> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15) const)
{
	return CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist16(A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15), true> >(std::make_pair(p_instance, p_func)).Initializer();
}

} // ~function_call

#if 0

#if defined(_MSC_VER) && !defined(__MWERKS__) && !defined(for) && _MSC_VER <= 1200
#define for if(0) {} else for
#endif // _MSC_VER && !__MWERKS__ && !for && _MSC_VER <= 1200
// msvc 'for' scoping hack

/**
 *	@brief demo static function, used in the examples in documentation of FunctionCall.h
 *	@param[in] x is the first argument
 *	@param[in] z is the second argument
 *	@Return Returns sum of the arguments.
 */
int myFunction(int x, float y)
{
	printf("called myFunction(%d, %g);\n", x, y);
	return int(x + y);
}

/**
 *	@brief demo class, used in the examples in documentation of FunctionCall.h
 */
struct myClass {
	/**
	 *	@brief demo member function, used in the examples in documentation of FunctionCall.h
	 *	@param[in] x is the first argument
	 *	@param[in] z is the second argument
	 *	@Return Returns product of the arguments.
	 */
	int memberFunction(int x, float y)
	{
		printf("called 0x%08x::memberFunction(%d, %g);\n", (int)(ptrdiff_t)this, x, y);
		return int(x * y);
	}
};

#include <vector>
#include <algorithm> // for_each()
#include <functional> // mem_fun()

/**
 *	@brief demo of the code, used in the examples in documentation of FunctionCall.h
 */
void Demo()
{
	{
		CAutoCallPointer p1 = GetSignature(&myFunction)(1, 2.3f);
		p1->Call();
		// managed pointer to a static function (follows the semantics of std::auto_ptr)
	}

	{
		CAutoCallPointer p1 = GetSignature(&myFunction)(1, 2.3f);
		CCallPointer *p = p1.p_Release(); // p is no longer managed
		// see CAutoCallPointer::p_Get() or CAutoCallPointer::p_Release()

		std::vector<CCallPointer*> calls;
		calls.push_back(p);
		// use CCallPointer* for passing call pointers to functions / storing in containers

#if !defined(_MSC_VER) || defined(__MWERKS__) || _MSC_VER >= 1400
		std::for_each(calls.begin(), calls.end(), std::mem_fun(&CCallPointer::Call));
#else // !_MSC_VER || __MWERKS__ || _MSC_VER >= 1400
		for(std::vector<CCallPointer*>::iterator it = calls.begin(); it != calls.end(); ++ it)
			(*it)->Call();
#endif // !_MSC_VER || __MWERKS__ || _MSC_VER >= 1400
		// call all of the functions

		for(std::vector<CCallPointer*>::iterator it = calls.begin(); it != calls.end(); ++ it)
			delete *it;
		// if we used CAutoCallPointer::p_Release(), we have to delete the calls manually
	}

	{
		CStaticCallContext<int, MakeTypelist2(int, float)> p1 = GetSignature(&myFunction)(1, 2.3f);
		int n_result = p1.t_Call(); // can see result
		// statically typed and linked call context
		// (usually the same speed as direct call of the function)
	}

	{
		/*function_call::CTypedCallPointer<CStaticCallContext<int,
			MakeTypelist2(int, float)> >*/
		CStaticCallContext<int, MakeTypelist2(int, float)>::_TyPointer // this is shorter
			p1 = GetSignature(&myFunction)(1, 2.3f);
		int n_result = p1.t_Call(); // can see result
		// statically typed but dynamically linked call context

		CCallPointer *p = &p1;
		p->Call(); // result hidden
		// can convert to CCallPointer and pass to generic functions
	}

	{
		myClass c;
		CAutoCallPointer p2 = GetMemSignature(&c, &myClass::memberFunction)(4, 5.6f);
		p2->Call();
		// managed pointer to a member function
	}

	{
		myClass c;
		CMemberCallContext<myClass, int, MakeTypelist2(int, float), false>::_TyPointer
			p2 = GetMemSignature(&c, &myClass::memberFunction)(4, 5.6f);
		p2.t_Call();
		// statically typed but dynamically linked pointer to a member function
	}
}

#endif // 0

#if 0

#if defined(_MSC_VER) && !defined(__MWERKS__) && !defined(for) && _MSC_VER <= 1200
#define for if(0) {} else for
#endif // _MSC_VER && !__MWERKS__ && !for && _MSC_VER <= 1200
// msvc 'for' scoping hack

/**
 *	@brief generates source code for call signature wrappers source code (the GetCallSignature() functions)
 *	@param[in] n is maximum number of arguments
 */
void Generate_CallSignature_Wrappers(const int n = 16)
{
	printf(
		"/**\n"
		" *\t@brief static function call signature extraction helper\n"
		" *\t@tparam CResultType is type of function result\n"
		" *\t@param[in] p_func is pointer to the static function\n"
		" *\t@return Returns a specialized instance of CCallSignature.\n"
		" */\n"
		"template <class CResultType>\n"
		"CArgsInitializer<CStaticFunctionUtils<CResultType, CTypelistEnd>, %d> GetCallSignature(CResultType (*p_func)(void))\n"
		"{\n"
		"\treturn CCallSignature<CStaticFunctionUtils<CResultType, CTypelistEnd> >(p_func).Initializer();\n"
		"}\n"
		"\n"
		"/**\n"
		" *\t@brief member function call signature extraction helper\n"
		" *\n"
		" *\t@tparam CClassType is type of function result\n"
		" *\t@tparam CResultType is type of function result\n"
		" *\n"
		" *\t@param[in] r_instance is reference to the class instance\n"
		" *\t@param[in] p_func is pointer to the member function\n"
		" *\n"
		" *\t@return Returns a specialized instance of CCallSignature.\n"
		" */\n"
		"template <class CClassType, class CResultType>\n"
		"CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd>, %d> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(void))\n"
		"{\n"
		"\treturn CCallSignature<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd> >(std::make_pair(p_instance, p_func)).Initializer();\n"
		"}\n"
		"\n"
		"/**\n"
		" *\t@brief member const function call signature extraction helper\n"
		" *\n"
		" *\t@tparam CClassType is type of function result\n"
		" *\t@tparam CResultType is type of function result\n"
		" *\n"
		" *\t@param[in] r_instance is reference to the class instance\n"
		" *\t@param[in] p_func is pointer to the member const function\n"
		" *\n"
		" *\t@return Returns a specialized instance of CCallSignature.\n"
		" */\n"
		"template <class CClassType, class CResultType>\n"
		"CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd, true>, %d> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(void) const)\n"
		"{\n"
		"\treturn CCallSignature<CMemberFunctionUtils<CClassType, CResultType, CTypelistEnd, true> >(std::make_pair(p_instance, p_func)).Initializer();\n"
		"}\n"
		"\n", 0, 0, 0);
	for(int i = 1; i <= n; ++ i) {
		{
			printf(
				"/**\n"
				" *\t@brief static function call signature extraction helper\n"
				" *\n"
				" *\t@tparam CResultType is type of function result\n");
			for(int j = 0; j < i; ++ j)
				printf(" *\t@tparam A%d is type of function parameter %d\n", j, j);
			printf(
				" *\n"
				" *\t@param[in] p_func is pointer to the static function\n"
				" *\n"
				" *\t@return Returns a specialized instance of CCallSignature.\n"
				" */\n"
				"template <class CResultType");
			for(int j = 0; j < i; ++ j)
				printf(", class A%d", j);
			printf(">\n"
				"CArgsInitializer<CStaticFunctionUtils<CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf(")>, %d> GetCallSignature(CResultType (*p_func)(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf("))\n"
				"{\n"
				"\treturn CCallSignature<CStaticFunctionUtils<CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf(")> >(p_func).Initializer();\n"
				"}\n"
				"\n");
		}
		// static

		{
			printf(
				"/**\n"
				" *\t@brief member function call signature extraction helper\n"
				" *\n"
				" *\t@tparam CClassType is type of function result\n"
				" *\t@tparam CResultType is type of function result\n");
			for(int j = 0; j < i; ++ j)
				printf(" *\t@tparam A%d is type of function parameter %d\n", j, j);
			printf(
				" *\n"
				" *\t@param[in] r_instance is reference to the class instance\n"
				" *\t@param[in] p_func is pointer to the member function\n"
				" *\n"
				" *\t@return Returns a specialized instance of CCallSignature.\n"
				" */\n"
				"template <class CClassType, class CResultType");
			for(int j = 0; j < i; ++ j)
				printf(", class A%d", j);
			printf(
				">\n"
				"CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf(")>, %d> GetMemCallSignature(CClassType *p_instance, CResultType (CClassType::*p_func)(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf("))\n"
				"{\n"
				"\treturn CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf(")> >(std::make_pair(p_instance, p_func)).Initializer();\n"
				"}\n"
				"\n");
		}
		// member

		{
			printf(
				"/**\n"
				" *\t@brief member const function call signature extraction helper\n"
				" *\n"
				" *\t@tparam CClassType is type of function result\n"
				" *\t@tparam CResultType is type of function result\n");
			for(int j = 0; j < i; ++ j)
				printf(" *\t@tparam A%d is type of function parameter %d\n", j, j);
			printf(
				" *\n"
				" *\t@param[in] r_instance is reference to the class instance\n"
				" *\t@param[in] p_func is pointer to the member const function\n"
				" *\n"
				" *\t@return Returns a specialized instance of CCallSignature.\n"
				" */\n"
				"template <class CClassType, class CResultType");
			for(int j = 0; j < i; ++ j)
				printf(", class A%d", j);
			printf(
				">\n"
				"CArgsInitializer<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf("), true>, %d> GetMemCallSignature(const CClassType *p_instance, CResultType (CClassType::*p_func)(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf(") const)\n"
				"{\n"
				"\treturn CCallSignature<CMemberFunctionUtils<CClassType, CResultType, MakeTypelist%d(", i);
			for(int j = 0; j < i; ++ j)
				printf(", A%d" + 2 * !j, j);
			printf("), true> >(std::make_pair(p_instance, p_func)).Initializer();\n"
				"}\n"
				"\n");
		}
		// member const
	}
}

/**
 *	@brief generates argument initializer specializations source code
 *
 *	@param[in] n is maximum number of arguments
 *	@param[in] n_max_line_length is maximum line length permitted
 */
void Generate_ArgInitializer_Specializations(const int n = 16, const size_t n_max_line_length = 80)
{
	printf(
		"/**\n"
		" *\t@brief argument initializer implementation\n"
		" *\n"
		" *\t@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils\n"
		" *\t\tor function_call::CMemberFunctionUtils\n"
		" *\t@tparam n_arg_num is number of function arguments (must match CFunUtil::n_arg_num)\n"
		" */\n"
		"template <class CFunUtil, const unsigned int n_arg_num>\n"
		"struct CArgsInitializerImpl; // forward declaration\n"
		"\n"
		"/**\n"
		" *\t@brief argument initializer implementation (specialization for no arguments)\n"
		" *\t@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils\n"
		" *\t\tor function_call::CMemberFunctionUtils\n"
		" */\n"
		"template <class CFunUtil>\n"
		"struct CArgsInitializerImpl<CFunUtil, 0> : public CArgsInitializerBase<CFunUtil> {\n"
		"\ttypedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */\n"
		"\n"
		"\t/**\n"
		"\t *\t@brief argument initialization operator\n"
		"\t *\t@return Returns a complete call context.\n"
		"\t */\n"
	//	"\tCArgsInitializer(_TyCtxData t_ctx) :CArgsInitializerBase(t_ctx) {}\n" // not needed anymore
		"\tinline typename B::_TyCallCtx operator ()()\n"
		"\t{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs()); }\n"
		"};\n"
		"\n"); // newline contracted
	for(int i = 1; i <= n; ++ i) {
		printf(
			"/**\n"
			" *\t@brief argument initializer implementation (specialization for %d arguments)\n"
			" *\t@tparam CFunUtil is specialization of function_call::CStaticFunctionUtils\n"
			" *\t\tor function_call::CMemberFunctionUtils\n"
			" */\n"
			"template <class CFunUtil>\n"
			"struct CArgsInitializerImpl<CFunUtil, %d> : public CArgsInitializerBase<CFunUtil> {\n"
			"\ttypedef CArgsInitializerBase<CFunUtil> B; /**< @brief the base class type */\n"
			"\n"
			/*"\tCArgsInitializer(_TyCtxData t_ctx) :CArgsInitializerBase(t_ctx) {}\n"*/ // not needed anymore
			"\t/**\n"
			"\t *\t@brief argument initialization operator\n", i, i);
		if(i > 1)
			printf("\t *\n");
		for(int j = 0; j < i; ++ j)
			printf("\t *\t@param[in] t_arg%d is value of argument %d\n", j, j);
		if(i > 1)
			printf("\t *\n");
		printf(
			"\t *\t@return Returns a complete call context with arguments initialized.\n"
			"\t */\n");
		size_t n_char_num = printf(
			"\tinline typename B::_TyCallCtx operator ()(");
		for(int j = 0; j < i; ++ j) {
			if(j && n_char_num + strlen("typename B::A%d t_arg%d") > n_max_line_length) {
				// not the first one, assumes there will be a ','
				n_char_num = printf(",\n\t\ttypename B::A%d t_arg%d", j, j) - 4 + 8;
			} else
				n_char_num += printf(", typename B::A%d t_arg%d" + 2 * !j, j, j);
		}
		printf(")\n\t");
		n_char_num = printf("{ return typename B::_TyCallCtx(B::m_t_ctx, typename B::CMakeArgs()") + 4;
		for(int j = 0; j < i; ++ j) {
			if(n_char_num > n_max_line_length - strlen(".IS_OMMIT(template) Push<typename B::A%d>(") && j + 1 != i) {
				printf(".IS_OMMIT(template) Push<typename B::A%d>(\n", i - 1 - j);
				n_char_num = printf("\t\tt_arg%d)", i - 1 - j) + 8;
			} else
				n_char_num += printf(".IS_OMMIT(template) Push<typename B::A%d>(t_arg%d)", i - 1 - j, i - 1 - j);
			if(n_char_num > n_max_line_length - strlen(".IS_OMMIT(template) Push<typename B::A%d>(") && j + 1 != i) { // not the last one
				++ j;
				printf(".IS_OMMIT(template) Push<typename B::A%d>(\n\t\t", i - 1 - j); // break after open parenthesis
				n_char_num = printf("t_arg%d)", i - 1 - j) + 8; // count the tabs as well
			}
		}
		printf("); }\n"
			"};\n"
			"\n");
	}
}

/**
 *	@brief generates msvc 6.0 compatible argument initializer specializations source code
 *
 *	@param[in] n is maximum number of arguments
 *	@param[in] n_max_line_length is maximum line length permitted
 */
void Generate_MSVC60ArgInitializer_Specializations(const int n = 16, const size_t n_max_line_length = 80)
{
	printf(
		"\t/**\n"
		"\t *\t@brief argument initializer implementation (specialization for 0 arguments)\n"
		"\t */\n"
		"\ttemplate <>\n"
		"\tstruct CInitializerImpl<0> : public CArgsInitializerBase<CFunUtil> {\n"
		"\t\t/**\n"
		"\t\t *\t@brief argument initialization operator\n"
		"\t\t *\t@return Returns a complete call context with arguments initialized.\n"
		"\t\t */\n"
		"\t\tinline _TyCallCtx operator ()() { return _TyCallCtx(m_t_ctx, CMakeArgs()); }\n"
		"\t};\n"
		"\n"); // newline contracted
	for(int i = 1; i <= n; ++ i) {
		printf(
			"\t/**\n"
			"\t *\t@brief argument initializer implementation (specialization for %d arguments)\n"
			"\t */\n"
			"\ttemplate <>\n"
			"\tstruct CInitializerImpl<%d> : public CArgsInitializerBase<CFunUtil> {\n"
			"\t\t/**\n"
			"\t\t *\t@brief argument initialization operator\n", i, i);
		if(i > 1)
			printf("\t\t *\n");
		for(int j = 0; j < i; ++ j)
			printf("\t\t *\t@param[in] t_arg%d is value of argument %d\n", j, j);
		if(i > 1)
			printf("\t\t *\n");
		printf(
			"\t\t *\t@return Returns a complete call context with arguments initialized.\n"
			"\t\t */\n");
		size_t n_char_num = printf(
			"\t\tinline _TyCallCtx operator ()(") - 2 + 8;
		for(int j = 0; j < i; ++ j) {
			if(j && n_char_num + strlen("A%d t_arg%d") > n_max_line_length) {
				// not the first one, assumes there will be a ','
				n_char_num = printf(",\n\t\t\tA%d t_arg%d", j, j) - 5 + 12;
			} else
				n_char_num += printf(", A%d t_arg%d" + 2 * !j, j, j);
		}
		printf(")\n\t\t");
		n_char_num = printf("{ return _TyCallCtx(m_t_ctx, CMakeArgs()") + 8;
		for(int j = 0; j < i; ++ j) {
			n_char_num += printf(" PUSH(A%d, t_arg%d)", i - 1 - j, i - 1 - j);
			if(n_char_num > n_max_line_length - strlen(" PUSH(A%d,") && j + 1 != i) { // not the last one
				++ j;
				printf("\n\t\t\t");
				n_char_num = printf("PUSH(A%d, t_arg%d)", i - 1 - j, i - 1 - j) + 12; // save a single space; count the tabs as well
			}
		}
		printf("); }\n"
			"\t};\n"
			"\n");
	}
}

#endif // 0

#endif // !__FUNCTION_CALL_WRAPPER_LIBRARY_INLINES_INCLUDED
