# Class describing intercept table for IRI-CORE
#
# Copyright (C) 2013 Radek Hranicky
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

class InterceptTable(object):
    def __init__(self):
        self.intercepts = dict()

    def addIntercept(self, intercept):
        """ Adds a new interception into the incercept table """
        liid = intercept.getLIID()                  # get LIID
        self.intercepts[liid] = intercept           # add intercept into table
        
    def liidExists(self, liid):
        """ Returns if interception with given LIID exists """
        if liid in self.intercepts:
            return True
        else:
            return False
            
    def getAllLIIDs(self):
        """ Returns LIID of all intercepts """
        LIIDs = set()
        for liid in self.intercepts:
            LIIDs.add(liid)
        return LIIDs

    def remIntercept(self, liid):
        """ Removes an intercept from the intercept table """
        if liid in self.intercepts:
            self.intercepts.pop(liid, None)

    def debug_printIntercepts(self):
        """ Prints the content of the intercept table """
        for icept in self.intercepts:
            print(self.intercepts[icept])
    
    def getNID(self, liid):
        """ Returns NID of an interception with given LIID """
        if liid in self.intercepts:
            return self.intercepts[liid].getNID()
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
            
    def getCIN(self, liid):
        """ Returns CIN of an interception with given LIID """
        if liid in self.intercepts:
            cid = self.intercepts[liid].getCID()
            return int(cid.getCIN())
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
            
    def getLevel(self, liid):
        """ Returns level of an interception with given LIID """
        if liid in self.intercepts:
            return self.intercepts[liid].getLevel()
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
            
    def getInterception(self, liid):
        """ Returns an interception with given LIID """
        if liid in self.intercepts:
            return self.intercepts[liid]
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
            
    def setCIN(self, liid, newCin):
        """ Sets new CIN for an interception with given LIID """
        if liid in self.intercepts:
            cid = self.intercepts[liid].getCID()
            cid.setCIN(str(newCin))
            self.intercepts[liid].setCID(cid)
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
            
    def increaseCIN(self, liid):
        """ Increase CIN of an intercept with given LIID """
        if liid in self.intercepts:
            cid = self.intercepts[liid].getCID()
            cid.increaseCIN()
            self.intercepts[liid].setCID(cid)
        else:
            raise NotImplementedError("An interception with LIID: %s does not exist" % liid)
