#include "corner_detector.h"
#include "../marker.h"
#include "umfdebug.h"
#include "draw.h"
#include <Eigen/Dense>

#ifdef UMF_USE_OPENCV
#include <cv.h>
#endif

namespace umf {

template<class T, int NCHAN>
bool binSearchEdge(Image<T, NCHAN> *image,
                   Eigen::Vector2i &edge,
                   Eigen::Vector2i left,
                   Eigen::Vector2i right,
                   const int searchThreshold)
{
    //test for out of bound coordinates
    if(left[0] < 1 || right[0] < 1 || left[1] < 1 || right[1] < 1 ||
            left[0] >= image->width - 1 || right[0] >= image->width - 1 ||
            left[1] >= image->height - 1 || right[1] >= image->height - 1)
    {
        return false;
    }

    Eigen::Matrix<float, NCHAN, 1> leftVal = image->get2De(left[0], left[1]).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> rightVal = image->get2De(right[0], right[1]).template cast<float>();

#ifdef UMF_DEBUG_COUNT_PIXELS
    int pixelCounter = 2;
#endif

    rightVal.norm();
    if((leftVal - rightVal).norm() < searchThreshold)
    {
        return false;
    }

    Eigen::Matrix<float, NCHAN, 1> thresh = (leftVal + rightVal)*0.5;

    Eigen::Vector2i currMid = (left + right)/2;

    Eigen::Matrix<float, NCHAN, 1> currVal = image->get2De(currMid[0], currMid[1]).template cast<float>();
#ifdef UMF_DEBUG_COUNT_PIXELS
    pixelCounter++;
#endif

    //now binary search for a point, that should be our edge pixel
    while( (currVal - thresh).norm() > 1 && (left - right).squaredNorm() > 2)
    {
        float diff1 = (leftVal - currVal).norm();
        float diff2 = (rightVal - currVal).norm();

        if(diff1 > diff2) //the current value is closer to the right part
        {
            right = currMid;
        }
        else
        {
            left = currMid;
        }


        currMid = (left + right)/2;
        currVal = image->get2De(currMid[0], currMid[1]).template cast<float>();

#ifdef UMF_DEBUG_COUNT_PIXELS
        pixelCounter++;
#endif
    }
#ifdef UMF_DEBUG_COUNT_PIXELS
    UMFDSingleton::Instance()->addPixels(pixelCounter);
#endif

    //now we got the edge point position, save it
    edge = currMid;

    return true;
}


const float CORNER_SEARCH_DOT_THRESHOLD = 0.2;

template<class T, int NCHAN>
Eigen::Vector2f findMarkerCorner22(Image<T,NCHAN> *srcImg, std::vector<Eigen::Vector2f> &points)
{
    Eigen::Vector2i edges[4];
    for(int i = 0; i < 4; i++)
    {
        if(!binSearchEdge(srcImg, edges[i], points[i].template cast<int>(), points[(i+1)%4].template cast<int>()))
        {
            //std::cout << "Error finding 2/2" << std::endl;
            return Eigen::Vector2f(-1, -1);
        }
    }

    Eigen::Vector3f line1 = Eigen::Vector3f(edges[0][0], edges[0][1], 1.0f).cross(Eigen::Vector3f(edges[2][0], edges[2][1], 1.0f));
    Eigen::Vector3f line2 = Eigen::Vector3f(edges[1][0], edges[1][1], 1.0f).cross(Eigen::Vector3f(edges[3][0], edges[3][1], 1.0f));

    Eigen::Vector3f interPoint = line1.cross(line2);

	Eigen::Vector2f line1Norm(line1[0], line1[1]);
	Eigen::Vector2f line2Norm(line2[0], line2[1]);
	line1Norm.normalize();
	line2Norm.normalize();

	Eigen::Vector2f line1Orig = points[2] - points[1];
	Eigen::Vector2f line2Orig = points[1] - points[0];
	line1Orig.normalize();
	line2Orig.normalize();

	if(std::abs(line1Orig.dot(line1Norm)) > CORNER_SEARCH_DOT_THRESHOLD ||
            std::abs(line2Orig.dot(line2Norm)) > CORNER_SEARCH_DOT_THRESHOLD)
    {
        return Eigen::Vector2f(-1, -1);
    }

    return Eigen::Vector2f(interPoint[0]/interPoint[2], interPoint[1]/interPoint[2]);
}


template<class T, int NCHAN>
Eigen::Vector2f findMarkerCorner13(Image<T,NCHAN> *srcImg, std::vector<Eigen::Vector2f> &points, int index)
{
    int baseIndexes[2];

    baseIndexes[0] = index;

    baseIndexes[1] = (index + 3) % 4;

    Eigen::Vector2i edges[4];
    for(int pi = 0; pi < 2; pi++)
    {
        int i = baseIndexes[pi];

        Eigen::Vector2f move1 = 0.25*(points[(i+3)%4] - points[i]);
        Eigen::Vector2f move2 = 0.25*(points[(i+2)%4] - points[(i + 1)%4]);

        if(! binSearchEdge(srcImg, edges[pi*2], (points[i] - move1).template cast<int>(), (points[(i+1)%4] - move2).template cast<int>()))
        {
            //std::cout << "simple not found" << std::endl;
            return Eigen::Vector2f(-1, -1);
        }
        //move our seed points in on one direction
        if(! binSearchEdge(srcImg, edges[pi*2 + 1], (points[i] + move1).template cast<int>(), (points[(i+1)%4] + move2).template cast<int>()))
        {
            //std::cout << "Moved not found" << std::endl;
            return Eigen::Vector2f(-1, -1);
        }
    }

    Eigen::Vector3f line1 = Eigen::Vector3f(edges[0][0], edges[0][1], 1.0f).cross(Eigen::Vector3f(edges[1][0], edges[1][1], 1.0f));
    Eigen::Vector3f line2 = Eigen::Vector3f(edges[2][0], edges[2][1], 1.0f).cross(Eigen::Vector3f(edges[3][0], edges[3][1], 1.0f));

    Eigen::Vector3f interPoint = line1.cross(line2);

	Eigen::Vector3f crossLine1 = Eigen::Vector3f(points[0][0], points[0][1], 1.0f).cross(Eigen::Vector3f(points[2][0], points[2][1], 1.0f));
	Eigen::Vector3f crossLine2 = Eigen::Vector3f(points[1][0], points[1][1], 1.0f).cross(Eigen::Vector3f(points[3][0], points[3][1], 1.0f));

	Eigen::Vector3f crossIntersection = crossLine1.cross(crossLine2);

	if(interPoint[2] == 0 || crossIntersection[2] == 0)
	{
		return Eigen::Vector2f(-1, -1);
	}

	//test if we are not too far from the center position
	interPoint *= 1.0f/interPoint[2];
	crossIntersection *= 1.0f/crossIntersection[2];

	Eigen::Vector2f diff = (interPoint - crossIntersection).block(0,0,2,1);
	float diffsq = diff.dot(diff);
	
	Eigen::Vector2f edge1 = (points[1] - points[0]);
	Eigen::Vector2f edge2 = (points[2] - points[1]);

	float edgesq1 = 0.25*edge1.dot(edge1);
	float edgesq2 = 0.25*edge2.dot(edge2);
	
	if(diffsq > edgesq1 || diffsq > edgesq2)
	{
		return Eigen::Vector2f(-1, -1);
	}
	//check dot threshold

	Eigen::Vector2f line1Norm(line1[0], line1[1]);
	Eigen::Vector2f line2Norm(line2[0], line2[1]);
	line1Norm.normalize();
	line2Norm.normalize();

	int pindex = baseIndexes[0];
	Eigen::Vector2f line1Orig = points[(pindex+3)%4] - points[pindex];
	pindex = baseIndexes[1];
	Eigen::Vector2f line2Orig = points[(pindex+3)%4] - points[pindex];
	line1Orig.normalize();
	line2Orig.normalize();

	if(std::abs(line1Orig.dot(line1Norm)) > CORNER_SEARCH_DOT_THRESHOLD ||
            std::abs(line2Orig.dot(line2Norm)) > CORNER_SEARCH_DOT_THRESHOLD)
    {
        return Eigen::Vector2f(-1, -1);
    }

    //test if the point is inside the quad defined by the positions
    for(int i = 0; i < 4; i++){
        Eigen::Vector3f point1(points[i][0], points[i][1], 1.0f);
        Eigen::Vector3f point2(points[(i + 1)%4][0], points[(i + 1)%4][1], 1.0f);
		Eigen::Vector3f interLine = interPoint - point1;
		Eigen::Vector3f origLine = point2 - point1;

        if(origLine.cross(interLine)[2] <= 0) //meaning the point is outside the quad or on the quad connecting line
        {
            return Eigen::Vector2f(-1, -1);
        }
    }
    if(interPoint[2] == 0)
    {
        return Eigen::Vector2f(-1, -1);
    }

    return Eigen::Vector2f(interPoint[0]/interPoint[2], interPoint[1]/interPoint[2]);
}

template<class T, int NCHAN>
Eigen::Vector2f findMarkerCorner(Image<T, NCHAN> *srcImg, std::vector<Eigen::Vector2f> &points, int cornerType)
{

    if(cornerType == CORNER_TYPE_NONE)
    {
        return Eigen::Vector2f(-1, -1);
    }

    Eigen::Vector2f cornerP;

    if(cornerType == CORNER_TYPE_CROSS)
    {
        cornerP = findMarkerCorner22(srcImg, points);
    } else {
        cornerP = findMarkerCorner13(srcImg, points, cornerType - CORNER_TYPE_LEFT_TOP);
    }

    return cornerP;
}

template<class T, int NCHAN>
int findCornersSubpixel(Image<T, NCHAN> *srcImg, std::vector<Eigen::Vector2f> &points)
{
	int successCount = 0;
#ifdef UMF_USE_OPENCV
	
	CvPoint2D32f *cvpoints = new CvPoint2D32f[points.size()];
	for(unsigned int ind = 0; ind < points.size(); ind++)
	{
		cvpoints[ind] = cvPoint2D32f(points[ind][0], points[ind][1]);
	}

	IplImage *imgCV = cvCreateImageHeader(cvSize(srcImg->width, srcImg->height), IPL_DEPTH_8U, NCHAN);
	imgCV->widthStep = srcImg->widthstep;
	imgCV->imageData = imgCV->imageDataOrigin = srcImg->data;
	
	IplImage *srcGray;
	bool releaseGray = false;
	if(NCHAN == 3)
	{
		srcGray = cvCreateImage(cvSize(srcImg->width, srcImg->height), IPL_DEPTH_8U, 1);
		cvCvtColor(imgCV, srcGray, CV_RGB2GRAY);
		releaseGray = true;
	} else if (NCHAN == 1)
	{
		srcGray = imgCV;
	}
	
	cvFindCornerSubPix(srcGray, cvpoints, points.size(), cvSize(12, 12), cvSize(-1, -1), cvTermCriteria(CV_TERMCRIT_ITER | CV_TERMCRIT_EPS, 20, 0.01));

	if(releaseGray)
	{
		cvReleaseImage(&srcGray);
	}
	cvReleaseImageHeader(&imgCV);

	for(unsigned int ind = 0; ind < points.size(); ind++)
	{
		CvPoint2D32f &p = cvpoints[ind];
		Eigen::Vector2f & cornerP = points[ind];
		float diffx = p.x - cornerP[0];
		float diffy = p.y - cornerP[1];
		float diff = std::sqrt(diffx*diffx + diffy*diffy);
		if(diff == 0)
		{
			cornerP = Eigen::Vector2f(-1, -1);
		} else {
			cornerP[0] = p.x;
			cornerP[1] = p.y;
		}

		successCount += (int) (cornerP[0] > 0);
	}
#endif

	return successCount;
}

//INSTANCING

template Eigen::Vector2f findMarkerCorner(ImageGray *srcImg, std::vector<Eigen::Vector2f> &points, int cornerType);
template Eigen::Vector2f findMarkerCorner(ImageRGB *srcImg, std::vector<Eigen::Vector2f> &points, int cornerType);

template int findCornersSubpixel(ImageGray *srcImg, std::vector<Eigen::Vector2f> &points);
template int findCornersSubpixel(ImageRGB *srcImg, std::vector<Eigen::Vector2f> &points);

template bool binSearchEdge(ImageGray *image, Eigen::Vector2i &edge, Eigen::Vector2i left, Eigen::Vector2i right, const int searchThreshold);
template bool binSearchEdge(ImageRGB *image, Eigen::Vector2i &edge, Eigen::Vector2i left, Eigen::Vector2i right, const int searchThreshold);

}//namespace
