
#include <cv.h>
#include <string>
#include "defines.h"

#include "umf_detector_api.h"
#include "umf.h"
#include "util/image.h"
#include "util/stream_factory.h"
#include "util/opencv_factory.h"
#include "util/firewire_factory.h"
#include "util/umfdebug.h"

using namespace umf;

#ifdef __OPENCV_OLD_CV_H__

	#ifdef CV_MAJOR_VERSION
	//if compiled in releas mode remove the "d" suffix
		#define CV_SHORTVERSION CVAUX_STR(CV_MAJOR_VERSION) CVAUX_STR(CV_MINOR_VERSION) CVAUX_STR(CV_SUBMINOR_VERSION) 
		#pragma comment(lib, "opencv_core" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_highgui" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_imgproc" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_features2d" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_calib3d" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_video" CV_SHORTVERSION ".lib")
	#else
		#pragma comment(lib, "opencv_core220.lib")
		#pragma comment(lib, "opencv_highgui220.lib")
		#pragma comment(lib, "opencv_imgproc220.lib") 
		#pragma comment(lib, "opencv_features2d220.lib") 
	#endif

#else
	#pragma comment(lib, "cxcore210.lib")
	#pragma comment(lib, "cv210.lib")
	#pragma comment(lib, "highgui210.lib") 
#endif

UMF int __stdcall umf_detect(DetectorProperties *detector, float timeout)
{
	DetectorResult *result = (DetectorResult *) detector->currentResult;

	ImageFactory *factory = (ImageFactory *) detector->capture;
	UMFDetector<UMF_DETECTOR_CHANNELS> *umfDet = (UMFDetector<UMF_DETECTOR_CHANNELS> *) detector->detector;
	ImageRGB *img = (ImageRGB *) detector->nextImage;
	ImageRGB *debugImg = (ImageRGB *) detector->nextDebug;


	if(factory != NULL && umfDet != NULL && img != NULL)
	{
	
        if(UMF_API_DEBUG)
		{
			memcpy(debugImg->data, img->data, img->widthstep*img->height);
			UMFDebug *dbg = UMFDSingleton::Instance();
			dbg->setImage(debugImg);
		} else {
            UMFDebug *dbg = UMFDSingleton::Instance();
            dbg->setImage(NULL);
        }

#if UMF_DETECTOR_CHANNELS == 1
		ImageGray *grayImg = new ImageGray(img->width, img->height, true, img->width);
		convertToGrayscale(img, grayImg);
        bool success = false;
        try{
		    success = umfDet->detect(grayImg, timeout);
        } catch(DetectionTimeoutException &e)
        {
            success = false;
        }

		delete grayImg;
#else
        bool success = false;
        try {
		    success = umfDet->detect(img, timeout);
        } catch(DetectionTimeoutException &e)
        {
            success = false;
            return -2;
        }
#endif
		if(success)
		{
			
			double position[3];
			double rotation[4];
			umfDet->model.getCameraPosRot(position, rotation);
			
			result->positionX = position[0];
			result->positionY = position[1];
			result->positionZ = position[2];

			result->quatX = rotation[1];
			result->quatY = rotation[2];
			result->quatZ = rotation[3];
			result->quatW = rotation[0];
		}

		return success ? umfDet->model.getCorrespondences().size() : 0;
	} 

	return -1;
}

UMF int __stdcall umf_grab_frame(DetectorProperties* detector)
{
	ImageFactory *factory = (ImageFactory *) detector->capture;
	UMFDetector<UMF_DETECTOR_CHANNELS> *umfDet = (UMFDetector<UMF_DETECTOR_CHANNELS> *) detector->detector;
	ImageRGB **camImg = (ImageRGB **) detector->camImage;

	if(factory != NULL && umfDet != NULL)
	{
        detector->currentFrame = (detector->currentFrame + 1) % detector->bufferSize;
        if( factory->getImage(camImg[detector->currentFrame], false) != EXIT_SUCCESS)
		{
			return 1;
		}
	}

	return 0;
}

UMF void __stdcall umf_get_frame(DetectorProperties* detector, unsigned char* image, int debug)
{
	ImageRGB **camImg = (ImageRGB **) detector->camImage;
	ImageRGB *debugImg = (ImageRGB *) detector->nextDebug;
	ImageRGB *img = NULL;
	if(debug)
	{
		img = debugImg;
	} else {
        img = camImg[detector->currentFrame];
	}

	if(img)
	{
		for(int row = 0 ; row < detector->textureHeight; row++)
		{
			for(int col = 0; col < detector->textureWidth; col++)
			{
				unsigned char *p = img->get2D(col, row);
				unsigned char alpha = 255;

				image[row*detector->textureWidth*4 + col*4 + 0] = p[0];
				image[row*detector->textureWidth*4 + col*4 + 1] = p[1];
				image[row*detector->textureWidth*4 + col*4 + 2] = p[2];
				image[row*detector->textureWidth*4 + col*4 + 3] = alpha;
			}
		}
	} 
}

UMF void __stdcall umf_set_frame(DetectorProperties* detector)
{
	ImageRGB **camImg = (ImageRGB **) detector->camImage;
    detector->nextImage = camImg[detector->currentFrame];
}

UMF void __stdcall umf_get_result(DetectorProperties* detector, DetectorResult* result)
{
	DetectorResult *cached = (DetectorResult *) detector->currentResult;
	result->positionX = cached->positionX;
	result->positionY = cached->positionY;
	result->positionZ = cached->positionZ;
	
	result->quatX = cached->quatX;
	result->quatY = cached->quatY;
	result->quatZ = cached->quatZ;
	result->quatW = cached->quatW;
}


const char* HUGE_MARKER_STR = ""
"25\n"
"25\n"
"4;0\n"
"12\n"
"2;4;4;1;1;3;0;4;4;4;4;0;4;4;3;4;2;0;1;4;3;1;0;2;2\n"
"1;2;2;4;2;0;1;3;0;3;2;0;4;0;0;2;3;4;2;1;2;0;1;3;3\n"
"0;0;1;2;2;1;3;1;4;4;0;2;4;2;3;2;2;3;0;0;3;3;0;1;1\n"
"4;2;3;1;4;0;0;4;0;1;0;0;0;3;1;3;4;4;0;4;0;3;1;3;4\n"
"0;2;0;2;0;2;0;4;0;2;4;0;0;0;4;4;3;0;0;1;0;1;0;4;2\n"
"4;3;0;1;1;1;0;4;1;3;3;0;1;0;4;4;0;1;1;4;3;3;3;1;4\n"
"2;4;0;4;2;3;1;1;0;0;3;4;3;4;1;3;0;1;2;0;4;1;3;1;3\n"
"0;2;4;1;4;1;4;4;1;4;1;4;0;4;4;1;3;1;4;1;0;2;0;4;3\n"
"2;4;1;0;0;1;0;4;1;2;1;2;0;1;4;3;0;2;0;4;4;0;4;2;4\n"
"1;4;3;1;3;3;1;0;2;1;1;3;3;1;3;0;0;3;1;3;4;3;0;0;1\n"
"4;0;0;4;4;1;4;4;1;4;1;1;2;2;3;4;1;2;4;2;0;2;0;2;3\n"
"3;2;3;3;4;2;4;3;4;4;2;1;1;3;0;2;3;2;2;1;4;3;0;3;3\n"
"2;0;1;2;0;1;4;3;0;0;4;1;0;1;1;2;3;2;3;0;3;0;4;4;0\n"
"1;4;3;0;4;0;1;4;0;1;1;3;1;3;2;3;3;0;2;0;0;3;0;0;4\n"
"0;0;2;2;0;2;1;2;1;4;2;1;2;0;4;3;1;0;4;4;4;1;3;0;2\n"
"0;4;1;3;4;2;4;2;0;3;0;3;2;0;4;0;1;4;4;0;1;0;4;2;2\n"
"4;3;0;0;0;4;1;1;4;0;3;4;3;1;1;2;4;1;2;4;0;3;4;0;4\n"
"3;2;4;0;0;4;4;4;1;0;4;3;1;0;4;3;1;3;4;0;0;2;2;3;2\n"
"2;3;2;2;1;0;3;4;1;2;2;2;1;3;0;4;0;2;0;2;4;2;1;3;1\n"
"3;0;4;4;4;0;4;0;2;3;1;1;1;4;0;0;3;0;3;2;0;4;4;4;1\n"
"4;0;1;1;3;0;2;4;1;3;4;3;0;1;1;3;1;3;0;4;2;4;2;0;1\n"
"0;2;2;0;0;3;1;0;2;4;0;2;4;4;3;4;2;4;1;1;2;3;3;1;4\n"
"3;1;4;0;4;0;3;1;4;2;3;4;3;1;4;3;3;2;2;1;0;1;3;2;0\n"
"4;4;1;4;3;1;3;0;1;1;2;2;0;3;3;2;0;4;2;0;4;3;2;1;2\n"
"4;1;1;0;3;2;2;2;0;4;0;1;2;0;0;4;3;1;0;2;1;2;3;3;4";

const char* SMALL_MARKER_STR = ""
"14\n"
"10\n"
"4;0\n"
"12\n"
"2;4;4;1;1;3;0;4;4;4;4;0;4;4\n"
"1;2;2;4;2;0;1;3;0;3;2;0;4;0\n"
"0;0;1;2;2;1;3;1;4;4;0;2;4;2\n"
"4;2;3;1;4;0;0;4;0;1;0;0;0;3\n"
"0;2;0;2;0;2;0;4;0;2;4;0;0;0\n"
"4;3;0;1;1;1;0;4;1;3;3;0;1;0\n"
"2;4;0;4;2;3;1;1;0;0;3;4;3;4\n"
"0;2;4;1;4;1;4;4;1;4;1;4;0;4\n"
"2;4;1;0;0;1;0;4;1;2;1;2;0;1\n"
"1;4;3;1;3;3;1;0;2;1;1;3;3;1";


UMF int __stdcall umf_create_detector(int width, int height, float near, float far, float fov, DetectorProperties *props)
{
	//create an RGB detector
	UMFDetector<UMF_DETECTOR_CHANNELS> *detector = new UMFDetector<UMF_DETECTOR_CHANNELS>(UMF_FLAG_ITER_REFINE|UMF_FLAG_SUBPIXEL|UMF_FLAG_SUBWINDOWS|UMF_FLAG_TRACK_POS);
	ImageFactory *factory = StreamFactory::GetImageFactory(std::string("OPENCV"));
    if(factory == NULL)
    {
        return 1;
    }

    CVImageInitStruct sCVIni;
	sCVIni.cameraIndex = props->cameraIndex;
	sCVIni.file = false;
	sCVIni.width = width;
	sCVIni.height = height;

	float aspectRatio = width*1.0f/height;

    factory->init((void*) &sCVIni);

	props->capture = factory;
    ImageRGB ** pcamImage = new ImageRGB*[props->bufferSize];
    for(int i = 0; i < props->bufferSize; i++)
    {
        pcamImage[i] = new ImageRGB(factory->getWidth(), factory->getHeight(), true);
    }
    props->camImage = pcamImage;

	int success = factory->getImage(pcamImage[0], false);
	if(success != EXIT_SUCCESS)
	{
		return 2;
	}

	props->nextDebug = new ImageRGB(factory->getWidth(), factory->getHeight(), true, pcamImage[0]->widthstep);
	props->nextImage = NULL; //it's just a copy of the pointer

	props->textureWidth = factory->getWidth();
	props->textureHeight = factory->getHeight();

	props->currentResult = new DetectorResult;

	//create detector class

	//load marker
	if(!detector->loadMarker(SMALL_MARKER_STR))
	{
		return 3;
	}

	Eigen::Matrix3d cameraMatrix;
    float focal = props->textureHeight/(2*tan(fov/2));
    Eigen::Vector2f imgSize(props->textureWidth, props->textureHeight);
    cameraMatrix << focal, 0, imgSize[0]/2,
            0, focal, imgSize[1]/2,
            0, 0, 1;

    Eigen::VectorXd distCoeffs(8);
    distCoeffs << 0, 0, 0, 0, 0, 0, 0, 0;
	
	detector->model.setCameraProperties(cameraMatrix, distCoeffs);

	props->detector = detector;

	return 0;
}

UMF int __stdcall umf_set_marker_str(DetectorProperties *props, char *str)
{
	UMFDetector<UMF_DETECTOR_CHANNELS> *umfDet = (UMFDetector<UMF_DETECTOR_CHANNELS> *) props->detector;

	if(!umfDet)
	{
		return 1;
	}

	if(!umfDet->loadMarker(str))
	{
		return 2;
	}

	return 0;
}

UMF void __stdcall umf_free_detector(DetectorProperties *detector)
{
	((ImageFactory *) detector->capture)->release();

    for(int i = 0; i < detector->bufferSize; i++)
    {
        delete ((ImageRGB **) detector->camImage)[i];
    }
    delete [] ((ImageRGB **) detector->camImage);

	delete ((DetectorResult*) detector->currentResult);
	delete ((ImageFactory *) detector->capture);
	delete ((UMFDetector<UMF_DETECTOR_CHANNELS> *) detector->detector);
	delete ((ImageRGB *) detector->nextDebug);

	detector->currentResult = NULL;
	detector->capture = NULL;
	detector->detector = NULL;
	detector->nextDebug = NULL;
	detector->nextImage = NULL;
	detector->camImage = NULL;

	UMFDSingleton::Release();
}

