#ifndef __UMF_MODEL_H
#define __UMF_MODEL_H

#include <set>
#include <vector>
#include <Eigen/Core>
#include "util/pnp_solver.h"
#include "util/image.h"
#include "util/mask_tracker.h"
#include "marker.h"

namespace umf {



template <int NCHAN>
class Model
{
public:
    Model();
    ~Model();


    template<class T>
    bool matchModel(Image<T, NCHAN> *img, unsigned short subW, unsigned short subH,
                    std::vector< typename Marker<NCHAN>::DirectionType > &edgeDir,
                    std::vector< Eigen::Vector2f > &extractionPoints,
                    Location &loc, bool show = false);

    template<class T>
    bool computeCameraPosition(Image<T, NCHAN> *img, unsigned short subW, unsigned short subH,
                               Location &loc, bool refine = true,
                               bool show = false);
	
    bool computeHomography(bool show = false);


    void projectPoints(std::vector<Eigen::Vector3f> &modelPoints,
                       std::vector<Eigen::Vector2f> &imagePoints,
                       Eigen::Vector2i imageSize,
                       bool distort = true,
                       bool swap = false);

	MaskTracker &getMaskTracker() {return this->maskTracker; }
	void updateMaskTracker(Eigen::Vector2i imgSize, int flags);

    /**
     * @brief wether to find the corners in the image, or just rely on the detected field centers
     * @param useCorner flag to use corner search
     */
    void setUseCornerSearch(bool useCorner) { this->useCornerSearch = useCorner; }
    bool getUseCornerSearch() {return this->useCornerSearch; }

	/**
	 * @brief if camera pose should use subpixel search
	 * @param useSubpix flag to use subpixel search
	 */

	void setUseSubPixel(bool useSubpix) { this->useSubPixel = useSubpix; }
	bool getUseSubPixel() { return this->useSubPixel; }

    /**
     * @brief getCorrespondeces get the correspondences
     * @return
     */
    CorrespondenceSet &getCorrespondences() {return this->correspondences; }
    void setCorrespondences(CorrespondenceSet &p) {this->correspondences = p;}

    void addMarker(Marker<NCHAN> *marker);
    Marker<NCHAN> *getMarker(){ return this->marker; }

	void setCameraProperties(Eigen::Matrix3d &matrix, Eigen::VectorXd &distCoeffs);

	int getCameraPosRot(double cameraPos[3], double rotationQuat[4]);
    int getWorldPosRot(double cameraPos[3], double rotationQuat[4]);

    Eigen::Matrix3d &getHomography() { return this->homography; }
private:

    void showCorners(std::vector<unsigned char> &corners, std::vector< Eigen::Vector2f > &extractionPoints, unsigned short subW, unsigned short subH);
    void showCorrespondences(bool refine = false);
    void showBox();

    template<class T>
    bool matchPoints(Image<T, NCHAN> *img,
                     Location &loc,
                     std::vector< Eigen::Vector2f > &extractionPoints,
                     std::vector<unsigned char> &corners,
                     unsigned short subW, unsigned short subH);

    template<class T>
    int getNewPositions(Image<T, NCHAN> *img,
                        const Eigen::MatrixXi &mask);


    CorrespondenceSet correspondences;
    Marker<NCHAN> *marker;
	MaskTracker maskTracker;
    bool useCornerSearch;
	bool useSubPixel;

    PnPSolver *pnp;
    Eigen::Matrix3d homography;
};

}

#endif // MODEL_H
