#ifndef __UMF_MARKER_H__
#define __UMF_MARKER_H__

#include <vector>
#include <Eigen/Core>
#include "defines.h"
#include "decisiontree.h"
#include <stdint.h>

namespace e {
using namespace Eigen;
}

namespace umf {




static const int EDGE_DIRECTION_ROTATE_MAPPING[EDGE_DIRECTION_COUNT] = {
    EDGE_DIRECTION_EQUAL, //eq
    EDGE_DIRECTION_LEFTUP, //rightdown
    EDGE_DIRECTION_RIGHTDOWN, //leftup
    EDGE_DIRECTION_INVALID, //invalid
};

enum MARKER_CORNER_TYPE {
    CORNER_TYPE_CROSS = 0,
    CORNER_TYPE_LEFT_TOP,
    CORNER_TYPE_RIGHT_TOP,
    CORNER_TYPE_RIGHT_BOTTOM,
    CORNER_TYPE_LEFT_BOTTOM,
    CORNER_TYPE_NONE
};

class MarkerType
{
public:
    bool color;
    bool torus;
    int range; //number of shades/colors
    void decode(int code);
    int encode();

    enum BIT_MASKS {
        TYPE_TORUS_BIT = 1,
        TYPE_COLOR_BIT = 2,
        TYPE_RANGE_BIT_START = 2, //shifting of range bits
        TYPE_RANGE_START = 2, //minimum number of shades
        TYPE_RANGE_BITS = 252 //mask for range bits
    };
};


void changeBackLocation(Location &loc, int width, int height, int w = 1, int h = 1);
void changeBackLocationf(Eigen::Vector2f &pos, int rotation, int width, int height, float w = 1, float h = 1);

/**
 * @brief The Marker class
 *
 * Basic use:
 * Marker mymarker;
 * mymarker.setField(extracted_values);
 * ...
 * //extract edge directions
 * Location l = getLocation(subwidth, subheight, subHoriz, subVert);
 * //do something with it
 *
 *
 */
template <int NCHAN>
class Marker
{
public:


    typedef typename Eigen::Matrix<EdgeType, NCHAN, 1> DirectionType;

    /**
     * @brief getPathIndexes get the indexes into the concatenated array of edge direction (horiz + vert)
     * @param subW the width of the subwindow
     * @param subH the height of the subwindow
     * @param indexes the indexes are stored here
     */
    void getPathIndexes(unsigned short subW, unsigned short subH,
                               std::vector<unsigned int> &indexes) const;

    /**
     * @brief getPath get path into the decision tree from the concated version and using the indexes generated by getPathIndex
     * @param indexes
     * @param horizvert
     * @param path
     * @return if some kind of error happened
     */
    bool getPath(std::vector<unsigned int> &indexes,
                 std::vector< DirectionType > &horizvert,
                 std::vector< DirectionType > &path,
                 unsigned int offset = 0) const;

    /**
     * @brief getPath wrapper for getPathIndex and getPath
     * @param subW
     * @param subH
     * @param horiz
     * @param vert
     * @param path
     * @return
     */
    bool getPath(unsigned short subW, unsigned short subH,
                 std::vector< DirectionType > &horiz, std::vector< DirectionType > &vert,
                 std::vector< DirectionType > &path);

    /**
     * @brief rotatePart rotate to a given rotation our vector
     * @param rotation - the rotation - see LOCATION_ROTATION
     * @param subW - width of the subwindow
     * @param subH - height of the subwindow
     * @param horizvert - concatenated horizontal + vertical
     * @return SUCCESS
     *
     * Be careful of the sizes, if rotation is by 90 or 270 degrees, the width and height are swapped
     */
    bool rotatePart(int rotation, unsigned short subW, unsigned short subH,
                    std::vector< DirectionType > &horizvert) const;


    Marker(unsigned short rows, unsigned short cols, unsigned char _n, std::vector< e::Matrix<unsigned char, NCHAN, 1> > colors):
        w(cols), h(rows), nunique(_n), colors(colors), decisionTree(nullptr)
    {
        this->decisionTreeMinHeight = this->nunique*(this->nunique - 1)*0.25; //at least quarter should be the same
        this->locationCorrectPercentage = 70;
    }

    virtual ~Marker() {
        if(this->decisionTree != nullptr)
        {
            delete this->decisionTree;
        }
    }

    /**
     * @brief setField sets the map and inits field
     * @param map the raw data of size w*h
     * @return if the initialization was successful
     */
    bool setField(std::vector<unsigned short> &map);

    /**
     * @brief init initializes the decision tree etc
     * @return if the initialization was successful
     */
    bool init();

    /**
     * @brief getLocation the position of the marker
     * @param subW width of subsize
     * @param subH height of subsize
     * @param subHoriz the extracted horizontal edges
     * @param subVert the extracted vertical edges
     * @return
     */
    Location getLocation(unsigned short subW, unsigned short subH,
                         std::vector< DirectionType > &edgeDir,
                         std::vector<unsigned char> &cornerMask) const;

    /**
     * @brief getCornerType get the corner type (see MARKER_CORNER_TYPE
     * @param l the location in the map
     * @param corner the model position
     * @param edges the edges for further use clockwise manner
     * @return the corner type
     */
    int getCornerType(Location l, e::Vector2f &corner, DirectionType* edges = NULL) const;

    /**
     * @brief set the decision tree minimum height - the minimum number of edges you want to be checked
     *
     * NOTE: this must be set before initializing, afterwards it has no effect unless the marker is reinitialized
     */
    void setDecisionTreeMinHeight(int height) {this->decisionTreeMinHeight = height; }
    int getDecisionTreeMinHeight() {return this->decisionTreeMinHeight;}


    /**
     * @brief setLoctionMinCorrectPercentage
     */
    void setLocationMinCorrectPercentage(float minPerc) { this->locationCorrectPercentage = minPerc; }
    float getLoctionMinCorrectPercentage(){ return this->locationCorrectPercentage; }



    unsigned short w, h, nunique;
    std::vector< e::Matrix<unsigned char, NCHAN, 1> > colors;
    std::vector< DirectionType > horizVert; // (w-1)*h + w*(h-1) theorietically, practically 2*w*h
    std::vector<unsigned char> cornerType; //(w-1)*(h-1) //precache this
private:

    inline void mapEdgeDirection(const DirectionType *src, DirectionType *dest) const;

    float getCorrectPercentage(std::vector< DirectionType > &edgeDir,
                               unsigned short edgeWidth, unsigned short edgeHeight,
                               int globalOffset, int localOffset,
                               unsigned short width, unsigned short height) const;

    int getCornerMask(std::vector< DirectionType > &edgeDir, unsigned short subW, unsigned short subH, Location loc, std::vector<unsigned char> &cornerMask) const;


    bool rotateCorners(int rotation, unsigned short subW, unsigned short subH,
                       std::vector< unsigned char > &corners) const;

    int decisionTreeMinHeight;
    float locationCorrectPercentage;
    DecisionTree<NCHAN> *decisionTree; //our decision tree
};

template <int NCHAN>
inline void Marker<NCHAN>::mapEdgeDirection(const typename Marker<NCHAN>::DirectionType *src, typename Marker<NCHAN>::DirectionType *dest) const
{
    for(int i = 0; i < NCHAN; i++)
    {
        (*dest)(i) = EDGE_DIRECTION_ROTATE_MAPPING[(*src)(i)];
    }
}


template <>
inline void Marker<1>::mapEdgeDirection(const Marker<1>::DirectionType *src, Marker<1>::DirectionType *dest) const
{
    (*dest)(0) = EDGE_DIRECTION_ROTATE_MAPPING[(*src)(0)];
}


template <>
inline void Marker<3>::mapEdgeDirection(const Marker<3>::DirectionType *src, Marker<3>::DirectionType *dest) const
{
    (*dest)(0) = EDGE_DIRECTION_ROTATE_MAPPING[(*src)(0)];
    (*dest)(1) = EDGE_DIRECTION_ROTATE_MAPPING[(*src)(1)];
    (*dest)(2) = EDGE_DIRECTION_ROTATE_MAPPING[(*src)(2)];
}


}
#endif // MARKER_H
