#include <iostream>
using namespace std;

#include "epnp.h"
#include <Eigen/Core>
#include <Eigen/Eigenvalues>
#include <Eigen/Dense>
#include <Eigen/QR>

epnp::epnp(void)
{
    maximum_number_of_correspondences = 0;
    number_of_correspondences = 0;

    pws = 0;
    us = 0;
    alphas = 0;
    pcs = 0;
}

epnp::~epnp()
{
    delete [] pws;
    delete [] us;
    delete [] alphas;
    delete [] pcs;
}

void epnp::set_internal_parameters(double uc, double vc, double fu, double fv)
{
    this->uc = uc;
    this->vc = vc;
    this->fu = fu;
    this->fv = fv;
}

void epnp::set_maximum_number_of_correspondences(int n)
{
    if (maximum_number_of_correspondences < n) {
        if (pws != 0) delete [] pws;
        if (us != 0) delete [] us;
        if (alphas != 0) delete [] alphas;
        if (pcs != 0) delete [] pcs;

        maximum_number_of_correspondences = n;
        pws = new double[3 * maximum_number_of_correspondences];
        us = new double[2 * maximum_number_of_correspondences];
        alphas = new double[4 * maximum_number_of_correspondences];
        pcs = new double[3 * maximum_number_of_correspondences];
    }
}

void epnp::reset_correspondences(void)
{
    number_of_correspondences = 0;
}

void epnp::add_correspondence(double X, double Y, double Z, double u, double v)
{
    pws[3 * number_of_correspondences    ] = X;
    pws[3 * number_of_correspondences + 1] = Y;
    pws[3 * number_of_correspondences + 2] = Z;

    us[2 * number_of_correspondences    ] = u;
    us[2 * number_of_correspondences + 1] = v;

    number_of_correspondences++;
}

void epnp::choose_control_points(void)
{
    // Take C0 as the reference points centroid:
    cws[0][0] = cws[0][1] = cws[0][2] = 0;
    for(int i = 0; i < number_of_correspondences; i++)
        for(int j = 0; j < 3; j++)
            cws[0][j] += pws[3 * i + j];

    for(int j = 0; j < 3; j++)
        cws[0][j] /= number_of_correspondences;

    // Take C1, C2, and C3 from PCA on the reference points:

    RMatrixX3d PW0(number_of_correspondences, 3);

    for(int i = 0; i < number_of_correspondences; i++)
    {
        for(int j = 0 ; j < 3; j++)
        {
            PW0(i,j) = pws[3*i + j] - cws[0][j];
        }
    }

    RMatrix3d PW0tPW0 = PW0.transpose()*PW0;

    Eigen::JacobiSVD<RMatrix3d> eigSVD(PW0tPW0, Eigen::ComputeFullU);
    RVector3d DC = (eigSVD.singularValues().array() / number_of_correspondences ).sqrt();
    RMatrix3d UCt = eigSVD.matrixU().transpose();

    //descending order
    if(DC[0] > DC[2]*1000)
    {
        //2D case - one direction is too small
        DC[2] = 0.01;
    }

    for(int i = 1; i < 4; i++) {
        int eigIndex = i - 1;
        double k = DC[eigIndex];
        for(int j = 0 ; j < 3; j ++)
        {
            cws[i][j] = cws[0][j] + k * UCt(eigIndex,j);
        }
    }

}

void epnp::compute_barycentric_coordinates(void)
{
    RMatrix3d CC;
    RMatrix3d CC_inv;

    for(int i = 0; i < 3; i++)
    {
        for(int j = 1; j < 4; j++)
        {
            CC(i,j-1) = cws[j][i] - cws[0][i];
        }
    }

    CC_inv = CC.inverse();

    for(int i = 0; i < number_of_correspondences; i++)
    {
        double * pi = pws + 3 * i;
        double * a = alphas + 4 * i;

        for(int j = 0; j < 3; j++)
        {
            a[1 + j] = CC_inv(j,0) * (pi[0] - cws[0][0]) +
                    CC_inv(j, 1) * (pi[1] - cws[0][1]) +
                    CC_inv(j, 2) * (pi[2] - cws[0][2]);
        }
        a[0] = 1.0f - a[1] - a[2] - a[3];
    }
}

void epnp::fill_M(RMatrixXd &M,
                  const int row, const double * as, const double u, const double v)
{
    for(int i = 0; i < 4; i++) {
        //M.block<1,3>(row, 3*i) = RVector3d(as[i] * fu, 0.0, as[i] * (uc - u));
        M(row, 3*i    ) = as[i] * fu;
        M(row, 3*i + 1) = 0.0;
        M(row, 3*i + 2) = as[i] * (uc - u);

        M(row + 1, 3*i    ) = 0.0;
        M(row + 1, 3*i + 1) = as[i] * fv;
        M(row + 1, 3*i + 2) = as[i] * (vc - v);
    }
}

void epnp::compute_ccs(const double * betas, const double * ut)
{
    for(int i = 0; i < 4; i++)
        ccs[i][0] = ccs[i][1] = ccs[i][2] = 0.0f;

    for(int i = 0; i < 4; i++) {
        //only the first few smallest eigenvalues are interesting
        const double * v = ut + 12 * (11 - i);
        for(int j = 0; j < 4; j++)
            for(int k = 0; k < 3; k++)
                ccs[j][k] += betas[i] * v[3 * j + k];
    }
}

void epnp::compute_pcs(void)
{
    for(int i = 0; i < number_of_correspondences; i++) {
        double * a = alphas + 4 * i;
        double * pc = pcs + 3 * i;

        for(int j = 0; j < 3; j++)
            pc[j] = a[0] * ccs[0][j] + a[1] * ccs[1][j] + a[2] * ccs[2][j] + a[3] * ccs[3][j];
    }
}

double epnp::compute_pose(double R[3][3], double t[3])
{
    choose_control_points();
    compute_barycentric_coordinates();

    RMatrixXd M(2 * number_of_correspondences, 12);

    for(int i = 0; i < number_of_correspondences; i++)
    {
        fill_M(M, 2*i, alphas + 4 * i, us[2 * i], us[2 * i + 1]);
    }

    RMatrix12d MtM;
    RVector12d D;
    RMatrix12d Ut;

    MtM = M.transpose()*M;

    Eigen::JacobiSVD<RMatrix12d> eigSVD(MtM, Eigen::ComputeFullU);
    D = eigSVD.singularValues();
    Ut = eigSVD.matrixU().transpose();

    RMatrix6x10d L_6x10(6, 10);
    RVector6d Rho(6, 1);

    compute_L_6x10(Ut.data(), L_6x10.data());
    compute_rho(Rho.data());

    double Betas[4][4], rep_errors[4];
    double Rs[4][3][3], ts[4][3];

    find_betas_approx_1(L_6x10, Rho, Betas[1]);
    gauss_newton(L_6x10, Rho, Betas[1]);
    rep_errors[1] = compute_R_and_t(Ut.data(), Betas[1], Rs[1], ts[1]);

    find_betas_approx_2(L_6x10, Rho, Betas[2]);
    gauss_newton(L_6x10, Rho, Betas[2]);
    rep_errors[2] = compute_R_and_t(Ut.data(), Betas[2], Rs[2], ts[2]);

    find_betas_approx_3(L_6x10, Rho, Betas[3]);
    gauss_newton(L_6x10, Rho, Betas[3]);
    rep_errors[3] = compute_R_and_t(Ut.data(), Betas[3], Rs[3], ts[3]);

    int N = 1;
    if (rep_errors[2] < rep_errors[1]) N = 2;
    if (rep_errors[3] < rep_errors[N]) N = 3;

    copy_R_and_t(Rs[N], ts[N], R, t);

    return rep_errors[N];
}

void epnp::copy_R_and_t(const double R_src[3][3], const double t_src[3],
double R_dst[3][3], double t_dst[3])
{
    for(int i = 0; i < 3; i++) {
        for(int j = 0; j < 3; j++)
            R_dst[i][j] = R_src[i][j];
        t_dst[i] = t_src[i];
    }
}

double epnp::dist2(const double * p1, const double * p2)
{
    return
            (p1[0] - p2[0]) * (p1[0] - p2[0]) +
            (p1[1] - p2[1]) * (p1[1] - p2[1]) +
            (p1[2] - p2[2]) * (p1[2] - p2[2]);
}

double epnp::dot(const double * v1, const double * v2)
{
    return v1[0] * v2[0] + v1[1] * v2[1] + v1[2] * v2[2];
}

double epnp::reprojection_error(const double R[3][3], const double t[3])
{
    double sum2 = 0.0;

    for(int i = 0; i < number_of_correspondences; i++) {
        double * pw = pws + 3 * i;
        double Xc = dot(R[0], pw) + t[0];
        double Yc = dot(R[1], pw) + t[1];
        double inv_Zc = 1.0 / (dot(R[2], pw) + t[2]);
        double ue = uc + fu * Xc * inv_Zc;
        double ve = vc + fv * Yc * inv_Zc;
        double u = us[2 * i], v = us[2 * i + 1];

        sum2 += sqrt( (u - ue) * (u - ue) + (v - ve) * (v - ve) );
    }

    return sum2 / number_of_correspondences;
}

void epnp::estimate_R_and_t(double R[3][3], double t[3])
{
    double pc0[3], pw0[3];

    pc0[0] = pc0[1] = pc0[2] = 0.0;
    pw0[0] = pw0[1] = pw0[2] = 0.0;

    for(int i = 0; i < number_of_correspondences; i++) {
        const double * pc = pcs + 3 * i;
        const double * pw = pws + 3 * i;

        for(int j = 0; j < 3; j++) {
            pc0[j] += pc[j];
            pw0[j] += pw[j];
        }
    }
    for(int j = 0; j < 3; j++) {
        pc0[j] /= number_of_correspondences;
        pw0[j] /= number_of_correspondences;
    }

    RMatrix3d ABt;
    RVector3d ABt_D;
    RMatrix3d ABt_U;
    RMatrix3d ABt_V;

    ABt.setZero();

    for(int i = 0; i < number_of_correspondences; i++) {
        double * pc = pcs + 3 * i;
        double * pw = pws + 3 * i;

        for(int j = 0; j < 3; j++) {
            ABt(j, 0) += (pc[j] - pc0[j]) * (pw[0] - pw0[0]);
            ABt(j, 1) += (pc[j] - pc0[j]) * (pw[1] - pw0[1]);
            ABt(j, 2) += (pc[j] - pc0[j]) * (pw[2] - pw0[2]);
        }
    }

    Eigen::JacobiSVD<RMatrix3d> svd(ABt, Eigen::ComputeFullU | Eigen::ComputeFullV);

    ABt_D = svd.singularValues();
    ABt_U = svd.matrixU();
    ABt_V = svd.matrixV();

    for(int i = 0; i < 3; i++)
    {
        for(int j = 0; j < 3; j++)
        {
            R[i][j] = ABt_U.row(i).dot(ABt_V.row(j));
        }
    }
    const double det =
            R[0][0] * R[1][1] * R[2][2] + R[0][1] * R[1][2] * R[2][0] + R[0][2] * R[1][0] * R[2][1] -
            R[0][2] * R[1][1] * R[2][0] - R[0][1] * R[1][0] * R[2][2] - R[0][0] * R[1][2] * R[2][1];

    if (det < 0) {
        R[0][2] = -R[0][2];
        R[1][2] = -R[1][2];
        R[2][2] = -R[2][2];
    }

    t[0] = pc0[0] - dot(R[0], pw0);
    t[1] = pc0[1] - dot(R[1], pw0);
    t[2] = pc0[2] - dot(R[2], pw0);

}

void epnp::print_pose(const double R[3][3], const double t[3])
{
    cout << R[0][0] << " " << R[0][1] << " " << R[0][2] << " " << t[0] << endl;
    cout << R[1][0] << " " << R[1][1] << " " << R[1][2] << " " << t[1] << endl;
    cout << R[2][0] << " " << R[2][1] << " " << R[2][2] << " " << t[2] << endl;
}

void epnp::solve_for_sign(void)
{
    if (pcs[2] < 0.0) {
        for(int i = 0; i < 4; i++)
            for(int j = 0; j < 3; j++)
                ccs[i][j] = -ccs[i][j];

        for(int i = 0; i < number_of_correspondences; i++) {
            pcs[3 * i    ] = -pcs[3 * i];
            pcs[3 * i + 1] = -pcs[3 * i + 1];
            pcs[3 * i + 2] = -pcs[3 * i + 2];
        }
    }
}

double epnp::compute_R_and_t(const double * ut, const double * betas,
                             double R[3][3], double t[3])
{
    compute_ccs(betas, ut);
    compute_pcs();

    solve_for_sign();

    estimate_R_and_t(R, t);

    return reprojection_error(R, t);
}

// betas10        = [B11 B12 B22 B13 B23 B33 B14 B24 B34 B44]
// betas_approx_1 = [B11 B12     B13         B14]

void epnp::find_betas_approx_1(const RMatrix6x10d &L_6x10, const RVector6d &Rho,
                               double * betas)
{
    RMatrix6x4d L_6x4(6,4);
    RVector4d B4;
    B4.setZero();

    for(int i = 0; i < 6; i++) {
        L_6x4(i,0) = L_6x10(i,0);
        L_6x4(i,1) = L_6x10(i,1);
        L_6x4(i,2) = L_6x10(i,3);
        L_6x4(i,3) = L_6x10(i,6);
    }

    B4 = L_6x4.fullPivLu().solve(Rho);

    if (B4[0] < 0) {
        betas[0] = sqrt(-B4[0]);
        betas[1] = -B4[1] / betas[0];
        betas[2] = -B4[2] / betas[0];
        betas[3] = -B4[3] / betas[0];
    } else {
        betas[0] = sqrt(B4[0]);
        betas[1] = B4[1] / betas[0];
        betas[2] = B4[2] / betas[0];
        betas[3] = B4[3] / betas[0];
    }
}

// betas10        = [B11 B12 B22 B13 B23 B33 B14 B24 B34 B44]
// betas_approx_2 = [B11 B12 B22                            ]

void epnp::find_betas_approx_2(const RMatrix6x10d &L_6x10, const RVector6d &Rho,
                               double * betas)
{
    RMatrix6x3d L_6x3(6,3);
    RVector3d B3;

    for(int i = 0; i < 6; i++)
    {
        L_6x3.row(i) = L_6x10.block<1,3>(i,0);
    }

    B3 = L_6x3.fullPivLu().solve(Rho);

    if (B3[0] < 0) {
        betas[0] = sqrt(-B3[0]);
        betas[1] = (B3[2] < 0) ? sqrt(-B3[2]) : 0.0;
    } else {
        betas[0] = sqrt(B3[0]);
        betas[1] = (B3[2] > 0) ? sqrt(B3[2]) : 0.0;
    }

    if (B3[1] < 0) betas[0] = -betas[0];

    betas[2] = 0.0;
    betas[3] = 0.0;
}

// betas10        = [B11 B12 B22 B13 B23 B33 B14 B24 B34 B44]
// betas_approx_3 = [B11 B12 B22 B13 B23                    ]

void epnp::find_betas_approx_3(const RMatrix6x10d &L_6x10, const RVector6d &Rho,
                               double * betas)
{
    RMatrix6x5d L_6x5 (6,5);
    RVector5d B5(5,1);

    for(int i = 0; i < 6; i++)
    {
        L_6x5.row(i) = L_6x10.block<1,5>(i,0);
    }

    B5 = L_6x5.fullPivLu().solve(Rho);

    if (B5[0] < 0) {
        betas[0] = sqrt(-B5[0]);
        betas[1] = (B5[2] < 0) ? sqrt(-B5[2]) : 0.0;
    } else {
        betas[0] = sqrt(B5[0]);
        betas[1] = (B5[2] > 0) ? sqrt(B5[2]) : 0.0;
    }
    if (B5[1] < 0) betas[0] = -betas[0];
    betas[2] = B5[3] / betas[0];
    betas[3] = 0.0;
}

void epnp::compute_L_6x10(const double * ut, double * l_6x10)
{
    const double * v[4];

    //the four smallest eigen Values
    v[0] = ut + 12 * 11;
    v[1] = ut + 12 * 10;
    v[2] = ut + 12 *  9;
    v[3] = ut + 12 *  8;

    double dv[4][6][3];

    for(int i = 0; i < 4; i++) {
        int a = 0, b = 1;
        for(int j = 0; j < 6; j++) {
            dv[i][j][0] = v[i][3 * a    ] - v[i][3 * b];
            dv[i][j][1] = v[i][3 * a + 1] - v[i][3 * b + 1];
            dv[i][j][2] = v[i][3 * a + 2] - v[i][3 * b + 2];

            b++;
            if (b > 3) {
                a++;
                b = a + 1;
            }
        }
    }

    for(int i = 0; i < 6; i++) {
        double * row = l_6x10 + 10 * i;

        row[0] =        dot(dv[0][i], dv[0][i]);
        row[1] = 2.0f * dot(dv[0][i], dv[1][i]);
        row[2] =        dot(dv[1][i], dv[1][i]);
        row[3] = 2.0f * dot(dv[0][i], dv[2][i]);
        row[4] = 2.0f * dot(dv[1][i], dv[2][i]);
        row[5] =        dot(dv[2][i], dv[2][i]);
        row[6] = 2.0f * dot(dv[0][i], dv[3][i]);
        row[7] = 2.0f * dot(dv[1][i], dv[3][i]);
        row[8] = 2.0f * dot(dv[2][i], dv[3][i]);
        row[9] =        dot(dv[3][i], dv[3][i]);
    }
}

void epnp::compute_rho(double * rho)
{
    rho[0] = dist2(cws[0], cws[1]);
    rho[1] = dist2(cws[0], cws[2]);
    rho[2] = dist2(cws[0], cws[3]);
    rho[3] = dist2(cws[1], cws[2]);
    rho[4] = dist2(cws[1], cws[3]);
    rho[5] = dist2(cws[2], cws[3]);
}

void epnp::compute_A_and_b_gauss_newton(const double * l_6x10, const double * rho,
                                        double betas[4], RMatrix6x4d &A, RVector6d &b)
{
    for(int i = 0; i < 6; i++) {
        const double * rowL = l_6x10 + i * 10;
        double * rowA = A.data() + i * 4;

        rowA[0] = 2 * rowL[0] * betas[0] +     rowL[1] * betas[1] +     rowL[3] * betas[2] +     rowL[6] * betas[3];
        rowA[1] =     rowL[1] * betas[0] + 2 * rowL[2] * betas[1] +     rowL[4] * betas[2] +     rowL[7] * betas[3];
        rowA[2] =     rowL[3] * betas[0] +     rowL[4] * betas[1] + 2 * rowL[5] * betas[2] +     rowL[8] * betas[3];
        rowA[3] =     rowL[6] * betas[0] +     rowL[7] * betas[1] +     rowL[8] * betas[2] + 2 * rowL[9] * betas[3];

        b(i) = rho[i] -
                (
                    rowL[0] * betas[0] * betas[0] +
                rowL[1] * betas[0] * betas[1] +
                rowL[2] * betas[1] * betas[1] +
                rowL[3] * betas[0] * betas[2] +
                rowL[4] * betas[1] * betas[2] +
                rowL[5] * betas[2] * betas[2] +
                rowL[6] * betas[0] * betas[3] +
                rowL[7] * betas[1] * betas[3] +
                rowL[8] * betas[2] * betas[3] +
                rowL[9] * betas[3] * betas[3]
                );
    }
}

void epnp::gauss_newton(const RMatrix6x10d &L_6x10, const RVector6d &Rho,
                        double betas[4])
{
    const int iterations_number = 10;

    RMatrix6x4d A(6,4);
    RVector6d B(6, 1);
    RVector4d X;

    for(int k = 0; k < iterations_number; k++)
    {
        compute_A_and_b_gauss_newton(L_6x10.data(), Rho.data(), betas, A, B);
        //based on the precision needed, we can change this
        //X = A.fullPivLu().solve(B);
        X = A.colPivHouseholderQr().solve(B);
        //qr_solve(A, B, X);

        for(int i = 0; i < 4; i++)
        {
            betas[i] += X[i];
        }
    }
}

void epnp::qr_solve(RMatrix6x4d &A, RVector6d &b, RVector4d &X)
{
    static int max_nr = 0;
    static double * A1, * A2;

    const int nr = A.rows();
    const int nc = A.cols();

    if (max_nr != 0 && max_nr < nr) {
        delete [] A1;
        delete [] A2;
    }
    if (max_nr < nr) {
        max_nr = nr;
        A1 = new double[nr];
        A2 = new double[nr];
    }

    double * pA = A.data(), * ppAkk = pA;
    for(int k = 0; k < nc; k++) {
        double * ppAik = ppAkk, eta = fabs(*ppAik);
        for(int i = k + 1; i < nr; i++) {
            double elt = fabs(*ppAik);
            if (eta < elt) eta = elt;
            ppAik += nc;
        }

        if (eta == 0) {
            A1[k] = A2[k] = 0.0;
            cerr << "God damnit, A is singular, this shouldn't happen." << endl;
            return;
        } else {
            double * ppAik = ppAkk, sum = 0.0, inv_eta = 1. / eta;
            for(int i = k; i < nr; i++) {
                *ppAik *= inv_eta;
                sum += *ppAik * *ppAik;
                ppAik += nc;
            }
            double sigma = sqrt(sum);
            if (*ppAkk < 0)
                sigma = -sigma;
            *ppAkk += sigma;
            A1[k] = sigma * *ppAkk;
            A2[k] = -eta * sigma;
            for(int j = k + 1; j < nc; j++) {
                double * ppAik = ppAkk, sum = 0;
                for(int i = k; i < nr; i++) {
                    sum += *ppAik * ppAik[j - k];
                    ppAik += nc;
                }
                double tau = sum / A1[k];
                ppAik = ppAkk;
                for(int i = k; i < nr; i++) {
                    ppAik[j - k] -= tau * *ppAik;
                    ppAik += nc;
                }
            }
        }
        ppAkk += nc + 1;
    }

    // b <- Qt b
    double * ppAjj = pA, * pb = b.data();
    for(int j = 0; j < nc; j++) {
        double * ppAij = ppAjj, tau = 0;
        for(int i = j; i < nr; i++)	{
            tau += *ppAij * pb[i];
            ppAij += nc;
        }
        tau /= A1[j];
        ppAij = ppAjj;
        for(int i = j; i < nr; i++) {
            pb[i] -= tau * *ppAij;
            ppAij += nc;
        }
        ppAjj += nc + 1;
    }

    // X = R-1 b
    double * pX = X.data();
    pX[nc - 1] = pb[nc - 1] / A2[nc - 1];
    for(int i = nc - 2; i >= 0; i--) {
        double * ppAij = pA + i * nc + (i + 1), sum = 0;

        for(int j = i + 1; j < nc; j++) {
            sum += *ppAij * pX[j];
            ppAij++;
        }
        pX[i] = (pb[i] - sum) / A2[i];
    }
}



void epnp::relative_error(double & rot_err, double & transl_err,
                          const double Rtrue[3][3], const double ttrue[3],
const double Rest[3][3],  const double test[3])
{
    double qtrue[4], qest[4];

    mat_to_quat(Rtrue, qtrue);
    mat_to_quat(Rest, qest);

    double rot_err1 = sqrt((qtrue[0] - qest[0]) * (qtrue[0] - qest[0]) +
            (qtrue[1] - qest[1]) * (qtrue[1] - qest[1]) +
            (qtrue[2] - qest[2]) * (qtrue[2] - qest[2]) +
            (qtrue[3] - qest[3]) * (qtrue[3] - qest[3]) ) /
            sqrt(qtrue[0] * qtrue[0] + qtrue[1] * qtrue[1] + qtrue[2] * qtrue[2] + qtrue[3] * qtrue[3]);

    double rot_err2 = sqrt((qtrue[0] + qest[0]) * (qtrue[0] + qest[0]) +
            (qtrue[1] + qest[1]) * (qtrue[1] + qest[1]) +
            (qtrue[2] + qest[2]) * (qtrue[2] + qest[2]) +
            (qtrue[3] + qest[3]) * (qtrue[3] + qest[3]) ) /
            sqrt(qtrue[0] * qtrue[0] + qtrue[1] * qtrue[1] + qtrue[2] * qtrue[2] + qtrue[3] * qtrue[3]);

    rot_err = min(rot_err1, rot_err2);

    transl_err =
            sqrt((ttrue[0] - test[0]) * (ttrue[0] - test[0]) +
            (ttrue[1] - test[1]) * (ttrue[1] - test[1]) +
            (ttrue[2] - test[2]) * (ttrue[2] - test[2])) /
            sqrt(ttrue[0] * ttrue[0] + ttrue[1] * ttrue[1] + ttrue[2] * ttrue[2]);
}

void epnp::mat_to_quat(const double R[3][3], double q[4])
{
    double tr = R[0][0] + R[1][1] + R[2][2];
    double n4;

    if (tr > 0.0f) {
        q[0] = R[1][2] - R[2][1];
        q[1] = R[2][0] - R[0][2];
        q[2] = R[0][1] - R[1][0];
        q[3] = tr + 1.0f;
        n4 = q[3];
    } else if ( (R[0][0] > R[1][1]) && (R[0][0] > R[2][2]) ) {
        q[0] = 1.0f + R[0][0] - R[1][1] - R[2][2];
        q[1] = R[1][0] + R[0][1];
        q[2] = R[2][0] + R[0][2];
        q[3] = R[1][2] - R[2][1];
        n4 = q[0];
    } else if (R[1][1] > R[2][2]) {
        q[0] = R[1][0] + R[0][1];
        q[1] = 1.0f + R[1][1] - R[0][0] - R[2][2];
        q[2] = R[2][1] + R[1][2];
        q[3] = R[2][0] - R[0][2];
        n4 = q[1];
    } else {
        q[0] = R[2][0] + R[0][2];
        q[1] = R[2][1] + R[1][2];
        q[2] = 1.0f + R[2][2] - R[0][0] - R[1][1];
        q[3] = R[0][1] - R[1][0];
        n4 = q[2];
    }
    double scale = 0.5f / double(sqrt(n4));

    q[0] *= scale;
    q[1] *= scale;
    q[2] *= scale;
    q[3] *= scale;
}
