#include <jni.h>
#include <android/log.h>

#include <Eigen/Core>
#include <math.h>


#include "umf.h"
#include "util/umfdebug.h"

#define  LOG_TAG    "UmfDetector"
#define  LOGI(...)  __android_log_print(ANDROID_LOG_INFO,LOG_TAG,__VA_ARGS__)
#define  LOGE(...)  __android_log_print(ANDROID_LOG_ERROR,LOG_TAG,__VA_ARGS__)

extern "C" {
JNIEXPORT void JNICALL Java_org_fit_graph_umfdetector_NativeDetector_init(JNIEnv * env, jobject obj,  jint width, jint height);
JNIEXPORT void JNICALL Java_org_fit_graph_umfdetector_NativeDetector_free(JNIEnv * env, jobject obj);
JNIEXPORT jint JNICALL Java_org_fit_graph_umfdetector_NativeDetector_detect(JNIEnv * env, jobject thiz, jbyteArray yuvData, jint width, jint height, jfloat angle, jint jviewCenterX, jint jviewCenterY, jint jmarkerCenterX, jint jmarkerCenterY, jobject result);
};


void nf_initDetector(void **detPointer, int width, int height, float fov = M_PI_4);
void nf_freeDetector(void **detPointer);

JNIEXPORT void JNICALL Java_org_fit_graph_umfdetector_NativeDetector_init(JNIEnv * env, jobject obj,  jint width, jint height)
{
    jclass cls = env->FindClass("org/fit/graph/umfdetector/NativeDetector");
    jfieldID fid = env->GetFieldID(cls, "detectorPointer", "J");
    //jlong detectorPointer = env->GetLongField(obj, fid);

    void *detector = NULL;
    nf_initDetector(&detector, width, height);
    LOGI("Set detector pointer: %i\n", (int) detector);

    env->SetLongField(obj, fid, (long) detector);
}

JNIEXPORT void JNICALL Java_org_fit_graph_umfdetector_NativeDetector_free(JNIEnv * env, jobject obj)
{
    jclass cls = env->FindClass("org/fit/graph/umfdetector/NativeDetector");
    jfieldID fid = env->GetFieldID(cls, "detectorPointer", "J");
    jlong detectorPointer = env->GetLongField(obj, fid);

    void *detector = (void *) detectorPointer;

    LOGI("Set detector pointer: %i\n", (int) detectorPointer);

    if(detector != NULL)
    {
        nf_freeDetector(&detector);
        detector = NULL;
    }

    env->SetLongField(obj, fid, (long) detector);
}

JNIEXPORT jint JNICALL Java_org_fit_graph_umfdetector_NativeDetector_detect(JNIEnv * env, jobject thiz, jbyteArray yuvData, jint width, jint height, jfloat angle,
                                                                            jint jviewCenterX, jint jviewCenterY, jint jmarkerCenterX, jint jmarkerCenterY, jobject result)
{
    jclass cls = env->FindClass("org/fit/graph/umfdetector/NativeDetector");
    jfieldID fid = env->GetFieldID(cls, "detectorPointer", "J");
    jlong detectorPointer = env->GetLongField(thiz, fid);

    Eigen::Vector2f viewCenter(jviewCenterX, jviewCenterY);
    Eigen::Vector2f markerCenter(jmarkerCenterX, jmarkerCenterY);

    void *detectorP = (void *) detectorPointer;

    if(detectorP == NULL)
    {
        return 2;
    }
    static int calls = 0;

    bool success = false;

    umf::UMFDetector<1> *detector = (umf::UMFDetector<1> *) detectorP;

    jboolean framecopy = false;
    char * imgdata = (char*) env->GetByteArrayElements(yuvData, &framecopy);


    umf::ImageGray *img = new umf::ImageGray(width, height, false);
    img->data = imgdata;


    //detector->edgelDetect.setScanlineStep(128);

    //Eigen::Vector2f imgPos(img->width/2, img->height/2);
    //Eigen::Vector2f modelPos;
    std::vector<Eigen::Vector2f> imgPos(2); imgPos[0] = viewCenter; imgPos[1] = markerCenter;
    std::vector<Eigen::Vector2f> modelPos(2);
    float directions[2] = {angle + M_PI_4, angle + 3*M_PI_4};
    //LOGI("Filtering directions:  %.2f", directions[0]*180.0/M_PI);
    detector->edgelDetect.getOrientationFilter().update(directions);
    success = detector->detectPosition(img, imgPos, modelPos);


    if(success) {
        jclass clazz = env->FindClass("org/fit/graph/umfdetector/NativeDetector$Result");
        jfieldID viewx = env->GetFieldID(clazz, "viewCenterX", "F");
        jfieldID viewy = env->GetFieldID(clazz, "viewCenterY", "F");
        jfieldID markerx = env->GetFieldID(clazz, "markerCenterX", "F");
        jfieldID markery = env->GetFieldID(clazz, "markerCenterY", "F");

        //double camPos[3];
        //double rotQuat[4];
        //detector->model.getCameraPosRot(camPos, rotQuat);
        env->SetFloatField(result, viewx, (float) modelPos[0][0]/detector->model.getMarker()->w);
        env->SetFloatField(result, viewy, (float) modelPos[0][1]/detector->model.getMarker()->h);
        env->SetFloatField(result, markerx, (float) modelPos[1][0]/detector->model.getMarker()->w);
        env->SetFloatField(result, markery, (float) modelPos[1][1]/detector->model.getMarker()->h);
        //LOGI("Resulting position %.2f, %.2f", modelPos[0], modelPos[1]);
    }

    delete img;

//#ifdef UMF_DEBUG_TIMING
    umf::UMFDebug *dbg = umf::UMFDSingleton::Instance();
    dbg->logEvent((double) detector->edgelDetect.getEdgels().size(), "EC");

    stringstream ss;
    std::vector< std::pair<double, std::string> > timing;
    dbg->getUniqLog(timing);
    for(std::vector< std::pair<double, std::string> >::iterator it = timing.begin(); it != timing.end(); it++)
    {
        ss << it->second << ":" << it->first << " ";
    }
    LOGI("Timing: %s ", ss.str().c_str());

    calls++;
    LOGI("Footpring: %f", dbg->getPixels()*1.0/(width*height*calls));
//#endif
    env->ReleaseByteArrayElements(yuvData, (jbyte*) imgdata, JNI_ABORT);

    return success ? detector->edgelDetect.getEdgels().size() : 0;
}

const char* HUGE_MARKER_STR = ""
        "25\n"
        "25\n"
        "4;0\n"
        "12\n"
        "2;4;4;1;1;3;0;4;4;4;4;0;4;4;3;4;2;0;1;4;3;1;0;2;2\n"
        "1;2;2;4;2;0;1;3;0;3;2;0;4;0;0;2;3;4;2;1;2;0;1;3;3\n"
        "0;0;1;2;2;1;3;1;4;4;0;2;4;2;3;2;2;3;0;0;3;3;0;1;1\n"
        "4;2;3;1;4;0;0;4;0;1;0;0;0;3;1;3;4;4;0;4;0;3;1;3;4\n"
        "0;2;0;2;0;2;0;4;0;2;4;0;0;0;4;4;3;0;0;1;0;1;0;4;2\n"
        "4;3;0;1;1;1;0;4;1;3;3;0;1;0;4;4;0;1;1;4;3;3;3;1;4\n"
        "2;4;0;4;2;3;1;1;0;0;3;4;3;4;1;3;0;1;2;0;4;1;3;1;3\n"
        "0;2;4;1;4;1;4;4;1;4;1;4;0;4;4;1;3;1;4;1;0;2;0;4;3\n"
        "2;4;1;0;0;1;0;4;1;2;1;2;0;1;4;3;0;2;0;4;4;0;4;2;4\n"
        "1;4;3;1;3;3;1;0;2;1;1;3;3;1;3;0;0;3;1;3;4;3;0;0;1\n"
        "4;0;0;4;4;1;4;4;1;4;1;1;2;2;3;4;1;2;4;2;0;2;0;2;3\n"
        "3;2;3;3;4;2;4;3;4;4;2;1;1;3;0;2;3;2;2;1;4;3;0;3;3\n"
        "2;0;1;2;0;1;4;3;0;0;4;1;0;1;1;2;3;2;3;0;3;0;4;4;0\n"
        "1;4;3;0;4;0;1;4;0;1;1;3;1;3;2;3;3;0;2;0;0;3;0;0;4\n"
        "0;0;2;2;0;2;1;2;1;4;2;1;2;0;4;3;1;0;4;4;4;1;3;0;2\n"
        "0;4;1;3;4;2;4;2;0;3;0;3;2;0;4;0;1;4;4;0;1;0;4;2;2\n"
        "4;3;0;0;0;4;1;1;4;0;3;4;3;1;1;2;4;1;2;4;0;3;4;0;4\n"
        "3;2;4;0;0;4;4;4;1;0;4;3;1;0;4;3;1;3;4;0;0;2;2;3;2\n"
        "2;3;2;2;1;0;3;4;1;2;2;2;1;3;0;4;0;2;0;2;4;2;1;3;1\n"
        "3;0;4;4;4;0;4;0;2;3;1;1;1;4;0;0;3;0;3;2;0;4;4;4;1\n"
        "4;0;1;1;3;0;2;4;1;3;4;3;0;1;1;3;1;3;0;4;2;4;2;0;1\n"
        "0;2;2;0;0;3;1;0;2;4;0;2;4;4;3;4;2;4;1;1;2;3;3;1;4\n"
        "3;1;4;0;4;0;3;1;4;2;3;4;3;1;4;3;3;2;2;1;0;1;3;2;0\n"
        "4;4;1;4;3;1;3;0;1;1;2;2;0;3;3;2;0;4;2;0;4;3;2;1;2\n"
        "4;1;1;0;3;2;2;2;0;4;0;1;2;0;0;4;3;1;0;2;1;2;3;3;4";


const char* SMALL_MARKER_STR = ""
        "14\n"
        "10\n"
        "4;0\n"
        "12\n"
        "2;4;4;1;1;3;0;4;4;4;4;0;4;4\n"
        "1;2;2;4;2;0;1;3;0;3;2;0;4;0\n"
        "0;0;1;2;2;1;3;1;4;4;0;2;4;2\n"
        "4;2;3;1;4;0;0;4;0;1;0;0;0;3\n"
        "0;2;0;2;0;2;0;4;0;2;4;0;0;0\n"
        "4;3;0;1;1;1;0;4;1;3;3;0;1;0\n"
        "2;4;0;4;2;3;1;1;0;0;3;4;3;4\n"
        "0;2;4;1;4;1;4;4;1;4;1;4;0;4\n"
        "2;4;1;0;0;1;0;4;1;2;1;2;0;1\n"
        "1;4;3;1;3;3;1;0;2;1;1;3;3;1";


const char* ONSCREEN_MARKER_STR = ""
        "52\n"
        "52\n"
        "4;0\n"
        "0\n"
        "0;0;1;1;0;1;1;0;0;0;0;0;0;1;1;1;0;1;1;0;1;1;0;0;0;1;0;0;0;1;0;0;0;0;0;0;0;1;0;0;1;1;0;1;0;0;0;0;1;0;0;0\n"
        "1;0;0;0;1;1;1;1;1;0;1;0;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;1;0;1;0;1;0;1;0;1;0;1;0;1;0;1;1;0;0;0;0;0;0;1;1;1\n"
        "1;1;1;0;1;1;0;1;0;0;0;0;1;0;1;1;1;1;1;0;1;1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;1;1;1;0;1;1;1;1;1;0;0\n"
        "1;0;0;1;0;1;1;1;0;1;0;1;1;1;0;0;0;1;1;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;0;0;1;0;0;0;0;0;0;1;1;0;1;1;1;1;0;1\n"
        "0;0;1;1;0;0;0;1;0;0;1;1;1;1;0;1;0;1;0;1;0;0;1;0;1;0;0;1;0;1;0;1;0;1;0;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1\n"
        "0;0;1;0;1;0;1;1;1;1;0;1;1;1;0;1;0;0;1;0;1;0;1;0;1;0;0;0;1;0;0;0;1;0;0;1;1;0;0;1;1;0;0;0;1;0;1;1;0;1;1;0\n"
        "1;1;0;1;1;0;1;0;0;1;1;0;0;0;0;1;1;0;0;1;0;0;1;1;0;0;1;1;1;0;1;0;0;1;0;0;0;1;0;0;1;0;0;1;1;0;0;0;0;1;1;0\n"
        "1;1;0;0;1;1;0;0;1;0;0;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;0;0;1;0;1;0;1;0;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;0;1\n"
        "1;0;0;1;0;1;1;1;0;0;0;1;1;0;0;0;0;0;0;0;1;1;0;0;1;0;0;1;1;0;0;0;1;1;1;1;1;0;1;0;0;0;1;1;1;0;0;0;0;0;0;1\n"
        "1;0;1;0;1;0;0;0;1;0;1;0;1;0;1;0;0;0;1;0;0;1;0;0;1;1;1;1;1;0;0;1;0;1;1;1;1;0;0;1;1;1;0;1;0;1;0;1;0;1;1;0\n"
        "0;1;0;0;0;0;0;1;0;0;1;1;1;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;1;1;1;1;1;1;1;1;1;1;0;1;1\n"
        "1;1;1;1;0;1;0;1;1;0;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;0;0;0;1;1;0;0;1;1;0;1;0\n"
        "1;1;0;0;0;0;0;1;1;1;0;1;0;0;1;0;1;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;0;0;0;1;1;1;0;0;0;0;0;0\n"
        "1;0;0;1;1;0;0;0;1;0;0;0;0;0;0;0;0;1;1;1;1;1;0;1;0;0;0;1;0;1;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;0;1;0;1;1;0\n"
        "1;1;0;0;1;0;0;0;1;1;0;1;0;1;1;1;0;1;0;1;1;1;1;0;1;1;0;0;1;0;1;1;0;0;0;0;1;0;0;1;1;0;1;0;1;1;0;0;0;0;0;1\n"
        "1;0;1;0;0;1;0;0;1;1;1;1;1;0;0;0;1;1;1;0;0;1;0;0;0;1;1;0;1;0;1;1;0;0;1;1;1;1;1;1;0;1;0;1;0;1;0;1;1;0;0;1\n"
        "0;0;0;0;0;1;0;0;1;0;1;0;1;0;0;0;1;1;1;1;1;1;0;1;1;0;1;1;1;0;0;1;1;0;1;1;1;0;1;1;0;0;1;1;0;0;1;1;0;1;0;1\n"
        "1;0;1;0;0;0;1;1;1;0;1;0;0;1;0;0;1;1;0;0;1;0;0;1;1;0;1;0;0;0;0;1;0;0;1;0;1;0;0;0;1;0;0;0;1;1;1;1;1;1;0;0\n"
        "1;1;1;0;1;1;1;0;0;0;1;0;1;0;0;0;0;0;0;1;0;1;0;0;0;0;1;1;0;1;1;1;0;0;1;0;1;0;0;0;0;1;0;0;0;0;0;1;1;0;1;0\n"
        "0;0;0;0;0;0;0;1;1;1;0;1;0;1;0;1;0;0;0;0;1;0;0;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;0;1\n"
        "1;0;0;0;1;1;1;0;0;0;1;0;0;0;1;1;1;0;0;0;1;0;1;1;1;0;0;0;0;1;0;0;1;1;0;0;0;1;0;1;1;1;1;1;0;0;0;0;1;1;1;1\n"
        "1;1;0;1;1;0;1;1;0;0;0;0;0;0;0;1;0;0;1;0;1;1;0;0;0;1;1;1;1;1;1;1;0;0;0;1;1;1;1;0;1;1;0;1;1;1;1;0;0;0;1;1\n"
        "1;1;0;0;1;0;0;1;0;0;0;0;0;1;1;1;1;0;1;0;1;1;1;0;0;1;0;0;0;1;1;0;0;0;1;0;0;0;0;1;1;1;0;1;0;0;0;0;0;0;1;1\n"
        "1;0;0;1;0;1;0;1;0;1;1;1;0;0;0;0;0;1;1;0;1;1;1;1;1;1;0;0;0;1;0;1;1;0;0;1;0;0;0;1;1;0;0;0;1;1;0;0;1;1;0;0\n"
        "0;1;0;1;0;1;1;0;0;1;1;0;1;0;0;1;0;1;0;0;0;1;1;0;1;0;1;0;1;0;1;0;0;0;0;1;1;0;0;0;1;0;1;1;0;1;1;1;1;0;0;0\n"
        "0;1;1;1;0;0;0;0;1;1;1;1;1;0;1;0;0;1;0;1;0;0;1;0;0;0;0;0;1;0;0;0;1;1;1;1;0;1;0;1;1;1;0;0;1;0;0;1;0;0;0;0\n"
        "0;1;1;1;1;1;0;1;1;0;0;1;1;0;0;0;1;0;1;1;0;0;0;0;0;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;1;0;1;1;0;0;1;1\n"
        "1;1;0;0;1;1;1;0;0;1;1;1;0;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;1;0;1;0;1;0;1;1;1;1;0;1;0;0;1;0\n"
        "1;1;1;0;0;1;1;0;0;1;0;0;1;1;0;1;1;1;1;0;0;0;0;1;0;0;1;0;1;1;1;0;1;1;0;1;0;1;0;0;1;1;1;0;1;0;1;1;1;0;1;1\n"
        "1;0;0;1;1;1;1;1;0;1;0;0;0;0;0;0;1;1;1;0;0;0;1;1;0;0;1;1;1;0;1;0;0;1;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;0;1\n"
        "0;0;0;1;1;1;1;0;1;1;1;0;0;1;1;1;0;0;0;0;1;0;0;1;0;0;0;1;0;1;0;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;0;1;0;0;0\n"
        "0;1;1;1;1;1;1;0;1;0;0;1;1;1;0;1;1;0;1;1;0;0;1;0;1;1;1;1;1;0;1;0;0;0;1;0;0;1;0;1;1;1;0;1;0;1;1;0;1;1;0;0\n"
        "0;1;1;0;1;0;1;0;0;0;1;1;0;0;0;0;0;1;1;1;1;1;1;0;1;0;1;1;0;0;0;1;1;0;1;0;1;1;0;1;1;0;1;0;1;0;0;0;0;0;1;0\n"
        "0;1;1;0;1;0;0;0;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;0;0;0;1;1;1;0;0;1\n"
        "1;1;0;0;1;1;0;1;0;1;1;0;0;1;1;1;0;1;0;0;0;1;0;0;1;1;0;0;0;1;0;1;0;0;1;0;1;0;0;1;1;1;0;0;1;1;0;1;0;0;1;1\n"
        "0;0;0;1;0;1;1;0;1;0;0;1;0;0;0;1;0;0;0;1;1;1;1;0;1;1;1;0;1;0;1;1;0;1;1;1;1;1;0;1;0;0;0;1;1;0;0;0;0;0;1;0\n"
        "0;0;1;0;1;0;1;1;1;0;0;0;1;1;0;1;0;1;1;0;0;0;1;0;1;0;1;1;1;0;0;1;0;1;0;1;1;0;0;0;0;0;0;0;1;1;0;0;1;1;1;0\n"
        "0;0;0;1;0;1;1;0;0;1;0;0;1;1;0;0;0;0;0;1;0;1;0;1;0;0;1;0;0;0;0;0;0;0;1;1;0;0;1;1;1;0;1;0;0;0;1;1;1;0;1;0\n"
        "1;1;1;0;0;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;0;1;1;0;1;1;0;1;0;1;1;1;0;0;1;1;0;0;0;0;1;0;0\n"
        "1;0;1;1;1;1;1;0;0;1;1;1;0;1;1;0;1;1;0;0;1;0;0;0;0;1;0;0;0;1;0;0;1;1;1;0;1;0;0;0;1;0;1;0;1;0;1;0;0;0;1;0\n"
        "0;1;1;0;1;1;1;1;1;1;0;0;1;1;0;1;1;1;1;0;0;0;0;1;1;0;0;0;1;1;0;0;0;1;1;1;0;0;1;1;0;0;0;1;1;1;0;0;0;0;0;1\n"
        "1;0;0;0;0;0;1;0;1;1;0;0;1;0;1;1;0;1;1;0;0;0;0;1;0;0;1;0;0;1;0;0;1;1;1;0;1;0;0;0;1;0;0;0;1;1;0;0;0;1;1;0\n"
        "0;0;0;1;1;1;0;1;0;1;1;0;0;0;0;1;0;0;1;0;0;1;1;0;0;0;0;0;0;1;1;0;1;0;0;1;1;0;0;0;1;1;0;1;0;1;1;1;1;0;1;1\n"
        "0;1;0;0;0;1;0;1;0;0;1;1;1;1;1;1;0;0;1;0;0;0;1;0;1;0;0;1;1;0;1;0;1;0;0;1;0;1;0;1;0;0;1;1;0;1;1;0;0;0;0;1\n"
        "0;0;0;1;0;0;1;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;0;1;0;1;0;1;1;1;0;0;1;0;0;1;1;1;1;0;1;0;0;0;1;1;1;0;0;1;1;1\n"
        "1;0;0;1;1;1;0;1;0;0;0;1;0;0;1;0;1;1;0;1;0;0;1;0;0;0;0;0;1;1;0;1;1;0;0;0;0;0;1;0;1;1;0;0;1;1;1;0;0;0;1;1\n"
        "1;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;1;1;0;1;0;1;1;1;1;0;1;1;0;0;0;1;0;0;1;1;0;0;1;1;0;1;0\n"
        "1;1;0;1;1;1;1;1;0;0;1;1;0;0;1;1;0;0;1;0;1;0;1;1;0;1;1;0;0;1;0;1;0;1;1;1;1;1;0;1;0;0;0;0;1;1;1;0;0;1;1;1\n"
        "0;0;1;1;1;1;0;1;1;0;1;0;1;0;1;0;0;1;0;0;1;0;1;1;0;0;1;1;0;1;1;1;1;0;1;1;1;1;0;0;0;1;0;1;1;1;1;1;1;1;0;1\n"
        "0;1;1;1;0;0;1;0;1;1;1;1;0;0;1;0;1;1;0;0;1;1;0;1;1;1;0;1;0;1;1;0;0;1;0;1;1;1;0;1;1;1;0;0;1;1;0;1;0;0;0;1\n"
        "1;1;0;1;1;0;0;1;1;0;1;0;0;0;0;1;1;1;1;1;0;0;1;1;0;0;1;0;0;0;1;0;1;0;1;0;1;1;0;0;0;1;0;0;0;0;1;1;1;1;1;0\n"
        "1;0;0;0;1;0;0;1;1;0;0;1;0;1;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;1;0;1\n";

const char* ONSCREEN_SOG_MARKER_STR = ""
        "128\n"
        "128\n"
        "4;0\n"
        "4\n"
        "1;2;2;1;1;0;2;2;2;2;0;0;2;0;1;1;2;2;1;0;2;2;0;0;2;1;1;1;1;0;1;2;0;0;1;2;0;0;2;0;0;0;2;0;1;0;1;0;2;0;1;2;2;2;1;2;0;0;0;2;0;2;2;1;2;1;0;0;2;0;0;2;0;0;0;2;1;2;0;1;2;2;2;2;1;0;2;2;1;0;2;1;2;2;2;0;0;0;0;0;2;1;0;2;0;1;2;1;1;0;0;1;2;2;1;2;0;1;1;1;1;1;1;2;0;1;2;2\n"
        "2;0;0;1;2;1;1;2;0;2;0;2;0;2;0;2;2;2;1;1;0;0;2;0;2;0;2;0;2;1;0;2;1;0;1;0;1;1;0;2;2;1;1;1;1;2;2;1;1;1;2;0;0;1;0;2;0;2;0;2;1;0;2;2;0;0;1;0;0;1;2;0;0;1;2;1;1;2;0;0;2;2;0;0;0;1;2;1;0;1;1;2;1;0;2;0;1;0;0;2;1;0;0;2;1;1;2;0;1;2;1;0;2;2;0;1;0;1;2;0;2;2;0;1;2;2;1;1\n"
        "0;2;0;2;0;2;2;2;1;2;0;2;2;1;1;2;1;2;0;0;2;0;1;0;2;2;0;1;1;2;2;0;2;0;0;0;1;0;2;1;0;2;1;0;1;2;2;0;1;0;0;0;1;2;1;0;1;1;1;1;2;1;1;1;2;1;1;2;1;0;2;2;1;1;2;2;1;2;0;1;2;1;1;1;1;0;1;0;0;0;1;2;2;2;1;2;1;2;0;0;0;1;1;1;2;2;2;0;0;2;1;0;1;0;2;0;1;0;1;2;2;2;1;1;2;0;0;1\n"
        "0;2;1;0;0;1;0;0;2;0;0;2;2;0;0;2;0;2;2;2;1;2;1;0;1;0;1;1;0;1;2;0;0;0;2;1;0;0;1;0;0;0;0;0;1;0;2;0;0;2;1;2;0;0;2;1;1;1;2;2;0;2;1;1;1;0;0;1;1;2;1;2;2;1;2;1;2;2;0;0;1;1;2;0;2;2;1;1;2;2;1;2;2;0;0;0;2;2;1;1;1;0;0;0;2;0;0;2;2;1;2;1;0;2;1;0;1;1;2;0;1;2;1;0;0;2;1;2\n"
        "1;2;1;2;1;0;2;0;1;0;2;1;2;0;1;0;2;2;1;1;0;1;0;1;2;1;2;2;1;1;0;0;1;1;2;2;2;1;0;2;2;0;1;2;0;2;2;0;2;1;0;0;2;1;0;0;2;2;0;0;1;0;1;2;2;1;2;0;2;0;0;1;0;1;2;2;1;1;2;1;1;0;1;0;1;2;1;0;2;0;2;0;0;1;0;1;2;2;1;2;0;1;2;0;2;1;0;1;0;1;0;2;1;0;1;0;2;2;1;2;0;1;2;2;0;2;0;0\n"
        "2;1;1;2;2;1;1;2;2;1;1;0;2;2;0;2;2;2;2;2;2;0;1;0;2;0;2;2;1;2;0;1;0;2;1;0;2;0;0;2;2;1;1;0;2;1;2;1;1;1;0;0;2;2;0;1;1;2;1;2;2;2;0;0;0;2;1;1;0;2;0;0;2;1;0;0;0;0;2;1;2;0;2;1;0;0;2;1;1;2;1;0;2;2;2;1;2;0;2;2;2;0;0;1;2;1;1;0;1;0;2;0;2;2;2;2;0;0;1;1;0;2;0;0;1;2;2;1\n"
        "0;2;1;0;2;1;1;2;1;0;0;2;2;0;0;1;0;0;0;0;1;1;0;2;1;1;2;2;1;1;2;1;0;0;1;1;2;0;0;0;2;0;0;1;0;1;2;1;2;2;2;0;1;0;2;2;1;2;1;2;1;0;1;0;1;0;1;0;0;1;0;0;0;2;1;0;2;1;2;1;0;2;1;2;2;2;1;1;2;1;0;1;1;1;1;0;2;2;0;2;2;1;2;2;0;2;0;2;1;0;2;0;0;1;2;2;1;2;0;0;2;1;2;2;0;1;2;0\n"
        "1;0;2;1;2;0;1;2;0;1;2;1;0;2;1;0;1;2;1;1;0;1;1;1;1;2;1;1;2;2;1;1;2;2;2;1;2;1;2;2;1;0;2;1;2;2;1;2;1;2;2;1;2;1;0;0;0;1;2;2;0;1;2;1;0;1;2;2;1;1;2;2;2;0;0;2;0;2;2;1;0;2;2;2;1;0;0;1;2;2;0;0;1;2;1;2;2;1;2;1;0;2;0;0;2;1;1;2;0;1;1;2;0;1;0;1;0;0;2;0;0;0;1;2;1;0;1;1\n"
        "0;1;2;0;0;1;2;0;1;0;0;1;2;1;0;1;2;0;2;0;2;2;2;2;0;1;1;1;2;0;0;2;2;0;0;2;0;2;1;1;2;1;1;2;0;1;0;0;2;0;2;1;0;2;1;1;2;0;0;2;1;1;2;0;1;0;0;2;2;1;1;2;0;2;2;1;0;1;1;1;2;1;2;0;2;1;2;0;2;2;2;1;1;2;2;1;0;2;1;0;1;2;2;2;0;2;1;1;1;0;2;0;2;0;1;2;1;1;2;1;2;1;0;0;0;2;1;0\n"
        "2;2;0;1;2;0;1;0;2;2;1;0;0;2;1;2;2;0;0;0;2;2;0;1;0;0;2;0;0;1;1;0;2;2;1;2;2;0;2;2;0;0;2;0;2;1;1;1;2;1;0;2;1;1;0;2;2;2;0;2;1;2;0;0;1;2;0;1;2;2;0;2;0;2;1;1;1;2;2;1;1;0;0;2;1;0;1;0;1;0;1;0;2;2;1;2;0;1;2;0;1;2;1;2;2;0;2;0;0;2;1;1;1;2;2;0;0;1;2;2;1;0;2;0;0;0;1;0\n"
        "2;1;0;0;0;2;0;2;1;1;2;1;2;1;0;0;1;0;1;2;1;0;1;1;2;1;2;2;2;2;0;1;0;1;2;2;1;0;2;0;1;2;2;0;1;0;2;2;1;0;1;1;2;0;2;1;1;1;1;0;0;2;2;2;0;2;1;2;2;1;0;2;1;0;2;2;1;2;2;0;0;1;2;2;2;2;0;2;1;2;1;0;1;1;2;0;1;0;0;0;2;1;2;1;1;0;1;0;2;0;1;0;2;0;2;1;2;0;2;1;0;2;1;2;2;1;2;0\n"
        "2;1;2;1;2;2;0;1;0;2;1;1;0;0;1;0;1;2;2;2;0;1;0;0;0;2;0;2;1;0;1;0;2;1;2;1;2;0;0;2;0;2;1;1;0;2;1;2;2;0;1;0;0;2;1;0;2;2;0;1;0;2;1;2;1;2;1;0;2;1;2;2;1;2;1;2;0;0;2;1;2;1;2;1;2;0;2;1;0;1;0;1;1;0;0;0;2;1;0;1;0;0;2;2;1;2;2;1;2;1;0;1;0;2;0;2;0;0;2;1;1;0;2;0;0;2;0;1\n"
        "2;1;1;2;1;1;1;2;0;1;1;2;2;2;1;1;2;2;0;2;2;2;1;2;2;1;0;1;2;0;2;1;1;2;0;0;1;2;2;1;2;1;2;2;0;1;0;0;0;1;0;1;2;0;1;0;1;0;1;0;1;1;2;0;1;2;0;0;0;0;2;2;1;0;1;0;1;2;1;0;0;2;1;1;2;2;1;1;0;2;1;2;2;1;0;2;0;1;2;0;0;1;2;0;2;1;1;2;1;2;2;1;0;1;2;2;1;2;1;1;0;1;1;1;2;1;1;2\n"
        "2;0;0;1;0;2;2;0;0;1;2;2;1;1;2;0;0;2;2;1;2;0;0;0;1;2;1;1;0;1;1;0;2;0;2;0;2;2;1;2;0;0;1;2;0;1;0;1;0;0;0;0;1;1;0;1;1;2;2;2;1;2;0;2;1;2;1;2;1;0;1;2;1;2;0;2;1;1;0;1;2;2;1;0;0;0;2;2;0;0;2;1;1;2;2;1;2;1;0;2;1;0;1;2;0;2;2;2;0;0;1;0;2;1;2;1;2;1;0;2;1;0;0;1;2;0;0;0\n"
        "0;0;2;1;2;1;2;2;1;2;2;2;0;1;0;1;0;2;0;2;1;1;2;1;2;1;0;2;2;0;2;2;0;2;1;0;1;0;1;2;0;1;2;2;0;2;1;0;2;2;0;2;0;0;2;0;2;1;1;0;0;1;0;1;1;2;0;1;0;1;0;2;2;2;1;0;2;0;1;2;1;2;1;2;2;2;1;1;2;0;2;2;1;0;1;0;0;1;2;0;1;0;2;1;0;1;2;1;0;1;0;2;0;2;0;0;0;2;2;0;2;2;1;1;0;0;2;0\n"
        "2;2;0;0;1;2;0;0;2;0;1;1;0;2;1;1;2;2;0;1;1;0;0;1;2;2;0;0;2;2;0;0;2;1;1;1;0;1;1;0;1;2;2;0;2;0;2;2;1;0;1;2;0;2;1;0;1;2;0;1;1;2;1;0;2;1;0;0;2;1;1;2;2;1;0;2;2;0;2;0;0;1;0;1;2;2;2;1;1;0;1;2;2;0;2;0;1;2;0;2;2;0;1;0;1;2;1;1;2;1;2;0;2;1;2;2;0;2;1;2;2;0;2;2;0;0;2;0\n"
        "2;1;2;2;1;1;2;1;0;1;0;2;2;2;1;2;0;1;0;2;0;2;2;0;0;1;2;1;0;0;2;2;0;1;2;0;0;2;2;1;0;2;1;2;1;2;1;0;0;1;2;1;2;1;0;2;0;1;1;0;1;0;1;2;0;1;2;0;0;1;0;0;0;1;0;0;0;1;0;2;1;0;1;0;2;1;0;2;0;1;0;0;1;0;1;2;1;1;1;0;0;0;1;0;1;2;1;2;1;0;1;2;2;0;0;1;2;2;2;0;2;1;1;1;1;0;0;0\n"
        "1;2;0;1;2;2;0;0;2;2;2;0;2;2;1;1;0;2;1;2;1;0;0;2;1;1;0;0;2;1;1;1;0;1;0;2;0;0;0;2;0;1;0;2;0;1;1;0;1;2;0;0;2;2;1;1;0;2;2;1;0;1;2;1;1;2;1;2;1;2;1;2;0;2;2;1;2;0;2;0;0;2;0;0;1;2;0;0;2;0;1;0;2;1;2;1;0;2;0;2;0;1;2;1;2;0;2;1;2;1;2;0;2;2;1;2;0;1;2;2;2;1;0;2;2;2;1;0\n"
        "0;1;2;1;1;2;0;0;2;0;1;2;0;0;2;2;0;1;1;1;0;1;1;0;2;2;0;2;2;1;0;0;2;0;1;0;2;1;1;0;0;2;2;0;2;0;2;1;2;0;2;2;2;1;0;2;1;1;2;0;2;2;1;2;0;0;1;0;2;2;1;0;2;1;0;2;1;1;0;0;0;0;1;2;1;2;1;1;0;2;1;0;0;2;0;0;0;1;2;0;1;0;0;0;2;1;0;0;1;1;2;2;1;1;2;0;0;1;2;1;2;0;1;2;1;1;1;2\n"
        "0;0;2;2;1;0;0;0;1;0;1;2;1;1;1;0;2;0;2;0;1;2;1;2;2;0;1;0;2;0;2;0;1;2;2;2;2;2;0;1;2;1;1;2;1;1;0;1;1;2;0;0;0;1;2;1;2;2;1;2;2;1;2;0;1;2;0;2;2;0;0;2;0;0;2;1;0;0;0;2;2;1;0;1;0;0;2;0;2;1;0;2;0;0;0;2;2;0;1;2;2;2;1;2;2;1;0;2;0;1;0;1;0;2;0;1;2;2;2;2;1;2;1;1;2;1;1;2\n"
        "0;2;0;2;0;0;2;2;2;2;1;1;1;2;1;2;1;1;0;2;1;0;2;0;2;1;2;2;0;1;2;1;0;1;1;1;0;0;2;2;0;0;2;0;0;0;2;2;1;1;2;0;1;0;1;1;2;1;1;2;0;2;2;2;0;1;2;0;2;1;1;0;2;1;0;2;1;1;1;2;1;2;1;0;2;2;1;2;1;0;2;1;0;1;2;1;0;1;0;0;0;1;1;2;0;2;2;0;1;2;0;2;0;1;2;1;2;0;0;0;2;2;0;0;1;2;1;2\n"
        "1;2;2;1;0;1;1;0;0;0;1;2;0;0;2;1;2;0;2;1;1;0;2;1;1;2;1;0;2;0;2;1;0;2;0;1;2;1;2;2;2;1;1;0;1;2;2;1;0;0;2;1;2;1;2;0;1;2;0;1;1;0;2;2;0;2;1;2;1;1;2;1;0;0;2;2;0;2;0;0;2;0;1;2;0;1;2;1;0;1;1;2;1;0;2;0;1;0;2;2;2;1;1;0;2;1;2;0;1;0;2;1;1;0;0;1;2;1;1;0;2;1;0;1;2;2;1;1\n"
        "1;1;2;0;2;0;0;2;0;2;0;1;0;1;2;0;1;2;1;2;0;2;2;0;1;2;1;2;1;1;0;2;2;2;1;0;0;2;0;1;1;0;2;2;1;2;0;2;1;2;1;1;1;0;1;1;0;1;2;2;2;0;0;2;0;1;0;1;2;0;2;2;0;1;0;0;1;0;2;0;2;0;2;2;1;0;0;1;2;0;0;2;0;0;2;1;2;0;2;1;0;1;0;2;0;1;0;0;2;0;0;2;0;2;2;1;1;2;0;1;0;0;1;2;2;1;2;1\n"
        "2;1;0;0;2;2;0;1;0;0;1;2;0;2;2;1;2;1;0;1;2;0;0;1;1;2;0;1;1;2;2;1;0;1;2;1;0;2;2;0;1;2;1;2;0;1;2;0;0;1;2;2;1;2;1;0;2;2;1;0;1;2;2;2;1;2;1;2;1;0;2;2;2;0;1;0;2;0;0;2;1;2;0;0;0;1;2;0;2;1;1;0;1;1;1;2;2;0;1;2;1;2;1;0;2;2;2;2;2;2;2;1;0;2;0;1;2;0;1;2;1;0;2;1;2;1;0;2\n"
        "1;2;1;0;1;2;1;1;2;0;2;0;1;0;2;1;1;0;0;1;0;1;2;2;0;2;1;2;1;2;1;0;2;0;0;0;0;2;0;2;0;0;1;1;2;0;1;2;1;0;2;2;2;2;2;0;2;1;2;1;0;1;2;0;0;0;2;0;0;2;1;1;0;2;2;0;1;2;1;0;1;1;2;1;1;0;1;0;1;0;2;0;0;2;1;0;0;1;0;0;0;0;0;1;2;2;2;0;0;1;0;1;0;1;0;2;1;1;0;0;0;2;1;2;0;0;2;0\n"
        "1;2;1;2;0;0;1;0;0;1;1;1;0;2;0;0;2;1;2;0;2;0;1;1;1;2;2;1;0;1;0;2;1;2;1;2;2;1;0;1;2;2;2;2;0;0;0;1;2;1;2;1;2;0;2;0;1;0;2;1;1;2;1;0;1;0;2;2;1;2;1;2;2;0;1;2;2;0;2;1;2;1;2;1;0;0;1;1;0;1;0;2;1;1;2;0;0;1;2;2;1;0;1;2;2;1;2;1;1;2;0;0;1;0;1;0;2;0;2;2;1;0;2;0;2;1;1;2\n"
        "2;0;2;2;2;1;2;1;2;0;2;0;2;1;0;1;0;2;2;2;0;2;1;2;2;1;2;1;2;0;2;1;2;2;0;1;0;1;2;0;2;1;1;0;2;0;1;0;2;0;2;1;0;2;0;2;1;0;2;1;1;2;2;1;2;0;2;1;1;0;1;2;1;0;2;0;1;1;1;2;1;1;2;2;1;2;2;2;0;2;1;2;1;2;0;0;2;0;2;2;1;2;1;2;2;1;0;0;2;2;2;1;1;1;2;0;2;2;2;2;1;1;2;1;0;2;1;1\n"
        "0;1;0;1;0;2;1;2;0;2;1;2;1;0;1;2;2;0;2;1;2;2;2;1;2;0;2;2;0;2;1;0;0;1;2;0;2;0;1;1;2;2;0;0;0;0;1;2;0;1;1;0;1;2;1;1;0;2;1;0;2;0;2;2;1;1;0;0;2;2;1;2;1;0;2;1;0;2;1;1;0;2;1;2;0;0;0;1;0;1;1;2;1;1;2;1;0;0;2;0;2;0;2;1;1;2;0;1;1;1;0;2;2;2;1;2;1;2;1;1;0;2;2;0;2;1;2;0\n"
        "2;1;0;2;1;0;0;1;0;1;0;2;0;1;1;0;0;1;0;1;2;1;1;1;2;0;2;1;0;1;2;1;2;0;0;2;1;1;2;1;0;0;1;1;2;2;0;2;0;0;2;0;2;0;1;0;1;0;2;0;2;1;1;1;1;2;1;2;1;1;1;2;2;2;0;2;0;1;2;1;0;2;1;0;0;2;1;2;1;0;2;1;1;2;0;0;2;2;0;0;1;1;2;0;2;0;0;2;2;2;0;0;1;2;0;1;2;2;2;2;2;1;1;0;0;1;1;0\n"
        "2;1;1;0;0;2;1;2;0;2;1;2;0;2;2;2;2;0;1;2;2;1;1;2;2;0;0;2;0;0;0;2;0;0;2;1;0;2;0;0;2;1;2;0;2;2;1;2;0;2;2;1;1;1;2;0;2;1;1;2;0;1;2;1;0;1;2;0;2;0;2;2;0;1;0;1;1;2;0;2;1;2;2;0;2;1;1;1;0;0;1;0;2;0;2;2;2;1;1;2;0;2;0;1;2;1;2;1;1;1;2;0;0;0;1;2;2;0;1;2;0;2;0;2;0;2;0;2\n"
        "2;2;0;0;1;0;0;1;1;1;2;1;2;0;0;0;1;2;0;0;1;2;1;2;0;2;0;1;2;1;2;2;1;2;2;1;2;1;0;1;0;2;1;2;0;1;0;2;1;1;1;0;2;2;0;0;2;1;2;1;0;2;1;0;1;1;0;2;1;2;1;1;2;0;1;0;2;0;1;0;2;2;0;1;2;1;2;2;2;2;0;1;1;2;2;2;0;0;0;2;2;0;2;2;0;1;0;2;1;2;0;1;2;1;0;1;0;1;0;1;2;1;2;1;0;1;2;1\n"
        "2;0;1;2;0;2;1;0;2;1;1;2;1;2;0;2;0;0;1;2;2;2;0;2;0;1;2;2;0;1;1;2;2;0;1;2;1;2;1;2;2;0;1;0;2;1;0;1;2;1;0;2;1;1;2;2;2;2;2;1;2;1;0;1;2;0;2;1;0;1;0;2;1;1;0;1;0;1;2;0;0;0;0;2;1;0;0;2;1;2;0;0;2;1;0;0;1;2;2;0;1;2;0;2;1;2;0;2;0;1;1;0;1;2;0;0;0;1;0;1;0;0;0;2;1;0;0;1\n"
        "0;0;2;1;0;1;1;1;0;2;1;0;1;1;2;0;1;2;0;0;1;1;0;1;2;0;1;1;1;2;0;2;2;1;0;2;1;2;1;1;2;0;2;1;2;2;2;0;2;1;2;0;0;2;2;0;0;1;1;1;1;2;0;1;0;1;1;2;1;0;2;1;0;2;2;0;2;2;2;1;1;1;2;2;1;1;2;2;0;2;0;2;1;1;2;0;0;2;1;2;0;0;0;2;0;1;0;2;0;2;0;2;0;0;2;2;1;2;1;0;2;2;0;0;2;1;2;1\n"
        "1;2;0;2;2;2;2;0;0;2;2;0;2;0;1;0;0;1;1;0;1;2;0;2;0;0;2;1;2;0;1;0;0;2;0;0;2;0;2;1;0;1;0;0;2;1;1;0;0;0;1;2;1;2;0;1;0;2;2;2;0;1;1;0;1;2;0;2;0;1;1;0;1;1;2;0;0;1;1;2;2;0;1;0;0;2;1;1;1;1;0;1;0;2;1;1;2;0;0;0;2;2;1;0;1;0;1;0;2;0;2;1;1;0;1;2;0;0;0;2;2;2;1;2;1;0;1;0\n"
        "1;2;1;0;0;2;1;1;2;0;1;2;1;1;2;2;2;2;2;2;2;0;2;1;2;1;1;1;2;2;2;0;2;1;1;2;1;0;1;0;1;1;2;2;2;1;0;2;0;1;0;0;1;0;2;0;2;1;2;1;0;2;2;0;2;1;1;1;1;0;2;0;2;1;0;2;0;1;0;1;1;0;2;1;0;1;2;0;0;2;1;2;0;1;0;1;0;2;2;2;0;2;0;2;0;1;1;1;2;1;0;0;2;2;2;1;0;0;1;2;0;0;1;0;2;2;0;0\n"
        "1;2;0;1;0;0;2;0;1;0;2;0;2;2;1;2;2;0;1;1;1;0;1;2;0;0;2;1;2;2;1;1;0;1;2;0;2;1;0;1;0;1;1;0;0;2;0;0;2;1;2;0;2;1;2;0;1;0;1;2;0;0;0;1;0;0;2;2;2;1;1;1;0;0;2;1;1;0;1;1;2;1;1;2;2;2;2;1;0;1;0;1;2;1;1;2;2;0;2;1;2;1;2;0;2;2;2;1;0;1;2;0;0;1;2;0;1;2;0;2;1;1;0;1;2;1;1;0\n"
        "0;1;1;0;2;2;1;0;2;0;1;0;1;1;0;2;0;1;1;2;0;2;1;0;0;2;2;0;1;0;0;2;0;0;2;0;0;2;0;2;1;2;2;0;1;0;0;2;1;2;1;2;0;0;1;2;2;0;0;1;2;1;2;1;2;0;2;2;1;2;2;0;2;2;1;0;0;2;0;2;2;2;0;0;2;1;0;2;1;2;0;1;0;2;2;0;2;2;0;0;0;0;1;2;0;2;1;0;2;1;1;1;2;1;2;1;0;2;1;2;1;0;0;2;2;1;0;1\n"
        "0;0;2;0;1;1;1;0;1;2;0;2;1;2;2;1;1;2;0;2;2;1;2;1;1;2;1;0;1;1;2;1;2;2;0;2;1;1;2;2;1;0;1;2;0;1;2;2;1;0;2;1;1;2;0;1;2;1;2;0;0;0;0;2;0;1;2;1;0;1;1;1;2;0;2;0;2;2;2;0;2;0;2;2;0;2;0;1;0;0;2;2;0;2;0;0;1;2;2;1;2;1;2;1;2;0;1;1;2;1;2;2;1;1;2;2;2;2;0;1;0;0;1;1;0;1;2;2\n"
        "1;2;0;0;0;1;2;2;1;0;2;1;2;1;1;1;2;0;1;2;0;2;1;1;2;0;0;1;2;2;0;0;2;1;2;0;2;0;1;1;2;0;1;1;2;2;1;2;1;2;0;2;1;2;0;2;2;1;0;2;2;1;2;1;2;1;0;2;0;2;2;2;2;1;1;2;1;2;1;0;2;2;1;2;2;0;1;2;1;2;1;0;2;1;0;1;0;1;2;0;0;2;0;0;1;0;2;2;2;1;2;0;1;2;2;0;0;1;0;2;1;2;0;1;0;1;1;2\n"
        "2;1;2;2;1;0;2;0;0;1;0;1;0;2;1;2;0;0;2;0;2;1;0;1;0;1;1;2;2;0;2;1;2;1;1;1;2;1;2;0;2;2;2;0;1;0;2;1;2;0;1;1;2;1;2;0;2;2;0;1;2;2;0;2;1;2;0;1;0;2;2;0;1;2;0;1;1;0;2;1;0;0;2;0;1;2;1;0;1;1;2;2;1;0;1;0;0;2;1;0;2;1;0;1;2;0;2;0;1;2;0;2;2;0;1;1;2;2;2;1;1;2;1;0;2;2;0;2\n"
        "1;1;0;1;2;0;0;2;1;2;1;0;2;0;0;1;1;1;0;1;1;2;0;0;1;1;2;1;1;2;1;0;0;1;2;0;2;2;1;0;2;2;1;1;2;2;0;1;1;0;2;0;0;0;0;2;2;2;0;0;2;2;0;0;1;2;0;0;2;1;1;2;1;0;2;1;2;1;0;1;2;0;1;2;1;0;1;1;2;2;1;1;2;2;0;2;0;0;0;2;0;0;2;2;0;1;1;1;0;2;2;2;1;2;0;1;2;2;2;1;2;0;2;1;1;1;2;1\n"
        "2;0;1;2;0;1;1;1;0;1;0;0;0;2;2;1;2;0;2;1;0;2;0;2;0;2;1;1;0;1;0;2;2;2;1;0;1;1;2;1;2;2;1;1;0;2;2;2;2;2;1;2;1;2;2;0;1;2;2;1;0;0;1;0;0;2;2;2;1;0;0;1;2;0;0;2;0;2;2;0;1;2;0;1;2;0;2;1;0;1;2;0;2;2;1;0;2;1;0;1;2;1;0;1;2;0;1;0;2;1;0;2;2;1;2;0;2;0;1;2;2;1;0;2;1;2;0;0\n"
        "0;1;2;0;2;2;0;1;2;1;1;2;0;2;0;0;1;1;1;0;2;2;0;2;1;1;1;2;1;2;1;1;2;2;1;1;2;2;0;2;1;2;0;2;2;0;0;0;1;1;0;1;1;2;0;2;0;0;0;1;2;1;0;2;1;0;1;1;2;1;1;1;2;1;2;1;2;1;2;0;1;1;0;2;0;1;0;1;2;0;1;1;2;1;0;1;1;1;2;1;1;1;2;0;2;1;2;1;1;2;0;0;2;0;1;2;1;1;2;0;0;1;2;0;2;2;1;2\n"
        "1;1;0;0;2;0;1;0;2;0;2;1;2;2;1;2;0;2;2;0;1;2;0;1;0;2;1;0;2;0;0;1;2;1;2;2;1;1;2;1;2;2;0;1;0;1;2;0;1;2;0;0;1;1;1;0;1;2;2;1;1;2;0;0;1;2;0;0;1;0;2;1;0;2;0;0;2;2;1;1;2;0;2;1;1;1;1;2;1;1;2;2;1;0;1;2;0;1;2;0;0;2;1;1;2;2;2;2;2;0;0;1;0;1;1;0;2;0;2;0;1;0;1;2;0;2;2;1\n"
        "1;2;1;2;2;0;0;1;0;2;1;2;1;0;0;2;1;1;0;1;0;2;2;2;2;0;0;2;0;2;1;2;2;0;1;2;1;0;0;1;0;0;1;2;0;2;1;2;1;1;0;1;2;0;2;1;1;1;1;0;1;2;1;1;0;1;2;1;1;2;1;1;0;2;1;1;2;2;0;0;1;1;0;0;2;2;2;1;1;0;2;2;1;2;1;1;0;2;0;2;1;2;0;1;2;1;0;1;1;1;2;1;0;2;2;0;2;2;1;1;0;0;1;1;2;0;1;0\n"
        "1;2;2;2;1;2;1;0;0;1;1;2;1;0;1;1;2;0;1;0;2;0;1;0;0;0;1;0;2;0;2;1;2;0;2;0;1;2;1;2;2;2;2;0;1;0;2;2;1;1;2;0;1;2;1;0;2;2;2;0;0;1;2;0;2;2;1;0;0;1;1;2;0;0;1;2;1;0;2;2;2;0;1;1;2;2;1;0;2;0;0;2;0;0;0;2;0;0;1;0;1;2;0;1;2;2;2;1;0;1;0;0;1;1;2;2;1;0;2;0;2;0;2;0;0;1;2;1\n"
        "1;1;2;2;1;0;1;2;2;0;0;0;2;1;0;1;0;2;0;1;1;0;2;1;0;2;1;2;0;0;1;0;0;1;1;0;1;0;2;1;0;2;1;1;0;0;0;1;0;2;0;2;1;2;1;2;1;2;1;0;2;0;1;0;1;0;2;0;1;1;1;2;0;1;0;1;2;0;2;2;1;2;0;0;2;1;0;1;2;2;1;1;0;1;1;2;2;0;2;0;2;2;0;2;0;0;0;0;1;0;1;1;0;2;1;0;0;2;2;0;2;1;2;0;2;1;0;0\n"
        "2;1;1;2;0;2;2;2;0;0;2;2;1;1;0;2;2;1;0;2;1;0;1;1;2;1;2;0;2;2;0;2;1;0;2;1;2;2;0;2;2;1;2;0;1;2;1;2;2;1;0;1;0;1;0;0;1;0;0;1;0;2;0;1;0;1;0;2;2;2;2;2;0;2;0;2;0;1;1;2;0;1;1;1;1;2;0;2;1;1;0;2;1;0;2;1;2;2;1;2;1;1;2;2;2;0;1;2;2;1;0;0;0;1;2;1;1;2;1;0;1;1;0;1;0;2;1;0\n"
        "1;0;0;0;2;1;2;1;0;2;0;0;1;0;1;0;0;1;0;1;1;2;2;2;1;2;2;2;2;1;0;0;2;0;2;1;0;1;2;0;1;0;1;2;2;2;1;2;0;1;2;0;1;2;0;2;1;1;2;0;2;2;1;0;0;0;1;0;0;2;2;1;2;0;2;2;0;0;2;1;2;0;2;0;2;0;0;0;2;0;1;1;2;1;0;0;0;0;1;0;0;0;2;1;1;2;1;1;2;1;2;2;2;2;2;1;2;1;1;0;1;2;2;1;0;0;2;2\n"
        "2;1;2;2;2;2;1;1;2;1;2;2;0;2;2;2;2;2;2;0;2;0;1;2;1;1;1;0;2;1;2;1;0;2;2;0;1;0;1;1;2;0;2;2;0;1;2;1;1;2;1;0;2;0;2;1;2;1;1;2;0;0;0;1;2;2;1;2;1;2;1;0;1;2;1;2;2;1;0;1;0;2;2;0;0;1;2;0;1;1;0;2;2;1;0;2;1;2;2;0;2;2;1;0;2;2;0;1;0;0;2;0;0;1;2;1;0;2;2;1;0;2;0;2;1;2;0;2\n"
        "1;2;0;0;1;0;2;2;0;2;2;0;0;2;1;1;0;1;0;2;1;2;0;1;0;2;0;1;0;1;2;2;1;2;0;0;2;0;0;0;2;0;1;2;2;0;1;2;2;1;1;2;1;1;0;2;1;2;1;0;2;0;2;2;0;0;0;1;2;0;1;0;1;0;2;1;0;2;2;2;2;0;2;1;2;0;1;1;2;0;2;0;0;0;2;1;0;0;0;0;2;2;0;1;2;2;2;0;2;1;0;2;2;0;0;2;1;2;1;0;2;0;1;0;1;2;0;0\n"
        "0;0;0;2;0;1;1;2;2;1;0;2;0;2;1;0;2;2;2;1;0;0;2;0;2;1;2;0;1;2;1;0;2;2;2;2;0;2;0;1;0;1;0;0;0;2;1;1;2;1;2;2;2;0;0;1;2;2;0;1;1;2;1;0;2;2;1;2;1;2;0;1;0;0;0;2;0;2;1;2;1;0;0;0;1;1;2;1;0;1;0;2;1;1;0;2;0;2;2;1;1;2;1;0;0;1;2;0;1;1;0;0;1;1;2;0;2;0;2;2;2;1;2;0;1;2;2;1\n"
        "0;2;0;1;0;0;1;0;0;1;2;2;2;0;0;2;1;2;0;0;2;0;0;2;2;2;1;1;2;1;2;2;0;0;2;0;2;0;2;2;0;1;2;0;1;0;2;0;0;2;0;1;1;0;1;1;0;0;1;2;1;1;0;0;2;2;1;1;0;0;1;0;0;2;1;1;2;1;0;0;1;2;2;0;2;2;0;0;2;2;0;2;1;1;2;1;0;0;0;0;2;1;2;2;1;1;1;1;2;0;2;2;1;2;2;1;1;1;2;1;1;0;0;0;1;2;2;1\n"
        "1;0;2;0;2;1;0;1;0;2;0;0;0;2;0;1;2;2;0;0;1;2;0;0;2;0;1;0;1;0;1;0;2;1;1;1;0;1;2;1;1;2;2;2;2;2;1;2;2;1;1;1;2;1;0;0;2;0;2;1;0;0;1;2;0;2;0;2;1;0;0;2;0;2;1;0;2;2;0;2;0;1;1;2;2;0;1;2;0;0;1;2;0;2;1;1;2;2;2;1;2;2;1;1;2;2;1;2;1;2;2;0;1;1;1;0;2;2;0;0;2;1;2;2;0;1;1;0\n"
        "0;1;1;1;0;2;2;0;2;1;0;2;1;1;0;1;2;2;0;2;1;0;2;0;2;2;0;2;0;2;1;2;1;0;0;0;2;0;1;0;1;2;1;0;1;2;0;1;1;2;0;0;1;0;1;0;2;2;0;1;2;1;0;1;2;1;1;0;1;2;1;0;2;2;2;0;1;2;1;0;2;0;1;0;0;2;0;0;2;1;1;2;0;2;0;1;0;2;1;2;0;0;0;2;0;2;2;2;2;0;1;2;0;2;2;1;0;0;2;0;1;1;2;2;1;2;0;1\n"
        "0;1;2;0;2;2;1;0;2;0;0;2;0;1;1;2;2;0;0;2;1;2;0;0;1;2;1;0;1;0;1;2;1;0;0;1;0;2;2;0;2;2;0;1;2;1;2;1;2;0;1;1;2;1;2;0;1;0;0;1;2;2;0;2;1;1;2;2;1;2;1;1;2;1;2;1;0;2;0;0;1;1;2;2;0;1;2;2;2;0;0;1;2;1;2;1;0;1;1;2;1;2;0;0;0;0;2;1;0;2;2;1;0;1;1;2;0;1;0;0;2;1;2;2;2;0;0;0\n"
        "2;2;0;0;1;1;0;0;2;0;2;0;2;2;2;0;1;2;1;2;0;1;2;2;2;2;2;0;2;1;1;2;0;2;2;0;0;0;1;1;0;1;2;0;1;0;0;0;0;2;0;0;1;2;0;1;2;1;2;0;2;1;2;0;0;2;1;1;0;2;2;2;2;2;2;0;1;1;1;2;0;1;0;2;2;1;1;1;2;0;1;2;2;1;1;2;1;2;0;1;0;2;1;2;0;2;0;2;0;1;1;1;2;0;1;0;0;0;1;0;1;1;0;1;0;1;2;1\n"
        "1;2;0;2;2;2;1;0;1;1;2;2;0;1;1;1;2;0;0;1;0;1;1;2;2;2;1;2;1;1;0;0;1;2;2;1;2;1;2;0;0;2;0;0;1;2;2;2;2;1;0;1;2;2;1;2;1;2;2;2;1;2;0;2;1;2;2;0;2;1;2;0;1;0;2;0;0;2;0;0;1;0;2;0;2;0;0;1;2;1;2;0;2;1;0;1;2;0;2;0;0;0;1;1;0;1;2;1;2;2;2;0;0;1;0;2;1;2;2;1;0;2;1;0;2;1;0;2\n"
        "1;0;0;2;0;2;2;1;0;1;2;0;2;2;0;2;0;2;0;2;1;2;0;2;0;0;2;0;1;2;1;2;0;1;0;1;1;2;0;1;1;0;2;1;0;2;1;1;2;2;2;1;2;2;0;2;0;1;0;2;0;1;2;1;0;1;2;1;1;2;0;1;1;2;2;2;0;1;1;1;0;0;2;2;1;2;2;1;0;0;1;2;1;2;1;0;1;1;0;2;2;0;2;1;1;2;0;2;0;1;2;1;0;2;1;1;2;0;2;2;1;1;2;1;1;0;2;2\n"
        "2;2;1;1;1;2;2;2;1;0;2;0;2;1;1;2;1;0;1;0;0;1;0;0;0;0;1;2;0;2;2;0;1;1;2;2;1;2;2;0;2;1;0;2;0;2;2;2;0;1;0;1;2;0;1;0;1;2;2;0;1;2;0;0;1;2;0;0;2;0;0;0;2;2;0;0;2;0;2;0;1;0;2;1;2;2;2;0;2;2;0;0;2;2;1;2;0;2;1;0;0;2;0;0;2;1;1;1;2;1;2;1;2;1;0;2;1;0;2;1;2;1;1;0;1;2;1;1\n"
        "0;1;2;1;1;0;1;1;2;0;0;1;2;2;0;2;0;2;1;2;1;2;0;1;2;2;1;0;0;1;2;1;0;2;0;1;0;2;2;2;0;0;2;0;2;2;0;2;1;1;0;2;2;0;1;0;2;0;0;1;2;0;2;1;0;0;2;0;0;0;2;1;0;1;2;0;1;2;0;2;1;0;1;1;1;0;1;1;0;0;0;2;1;0;2;1;2;2;1;2;0;0;1;1;2;0;2;1;0;1;1;2;0;2;0;1;0;2;0;0;1;0;1;2;2;0;2;1\n"
        "2;2;1;0;2;0;1;2;1;2;1;1;1;0;1;1;2;1;0;1;2;0;2;0;2;1;0;1;2;0;0;2;2;1;2;0;2;1;0;0;1;2;2;0;2;0;1;0;2;2;1;2;0;1;0;2;0;1;0;2;1;0;2;2;2;2;0;1;0;1;2;2;1;2;0;0;1;0;1;0;2;2;0;2;2;1;1;2;2;2;2;1;0;1;1;0;1;1;1;2;1;1;2;2;2;0;0;1;0;2;2;2;2;2;0;0;1;1;1;1;0;0;2;2;0;2;0;1\n"
        "0;2;1;0;1;1;0;0;2;1;2;2;1;2;2;1;1;2;0;0;2;1;0;1;0;2;0;0;1;0;2;2;0;0;0;1;2;2;2;1;1;2;1;0;1;2;1;0;1;1;0;0;1;1;2;1;1;0;2;2;0;1;1;0;2;0;2;2;2;0;0;0;2;0;0;2;2;2;2;0;2;0;2;1;1;1;2;0;2;2;0;2;0;2;1;2;2;2;1;1;2;0;2;2;0;2;1;0;0;1;1;0;1;1;2;1;0;2;2;0;1;1;0;1;1;0;2;1\n"
        "2;1;2;1;1;2;0;2;1;2;0;2;0;1;0;2;0;2;2;1;1;2;1;2;0;2;1;2;2;2;0;1;1;2;0;2;0;1;2;1;0;2;0;2;1;1;0;1;0;0;2;2;2;0;2;1;0;1;0;0;1;0;2;0;2;2;2;0;1;2;1;2;2;2;2;1;0;2;2;1;0;2;1;0;1;0;1;1;2;1;0;1;1;2;1;1;2;1;2;1;1;1;0;2;1;1;0;2;1;1;2;2;1;2;0;2;0;1;2;2;0;2;0;2;1;0;1;1\n"
        "0;1;2;2;0;1;2;2;0;0;2;1;2;2;0;1;2;0;2;1;2;1;0;2;1;2;2;0;1;2;0;2;0;2;2;1;1;0;1;1;2;1;1;2;0;1;1;2;2;1;0;2;2;2;2;1;2;2;1;0;2;1;0;2;2;0;1;1;2;0;1;2;2;2;1;2;1;0;0;1;1;1;0;1;2;0;2;0;0;2;1;0;1;2;0;1;2;1;0;0;1;2;0;0;2;0;1;0;0;1;2;0;2;0;0;1;1;2;2;1;2;1;1;0;0;1;2;0\n"
        "1;0;2;0;2;2;0;1;2;2;2;1;0;1;1;0;1;0;2;0;2;2;2;2;0;1;0;2;1;0;2;2;0;1;2;0;2;0;2;2;2;1;0;0;0;2;1;2;0;0;2;0;0;0;0;2;2;1;0;1;1;2;1;2;1;1;2;1;1;2;0;2;0;2;2;1;0;0;1;2;2;2;1;2;0;2;1;2;1;0;2;0;1;0;2;1;2;2;0;1;2;0;1;2;2;0;1;2;2;0;2;1;0;1;1;0;1;0;0;1;1;2;1;2;1;1;1;0\n"
        "0;1;2;2;2;0;2;2;2;0;2;1;1;2;2;1;0;0;2;2;1;2;1;1;0;1;0;2;2;2;2;1;0;1;0;0;0;2;0;2;0;2;1;2;2;0;0;1;2;2;0;2;0;1;2;1;0;1;2;0;2;0;1;0;0;2;0;2;0;0;2;1;0;1;1;0;1;2;1;1;0;2;2;0;2;1;2;1;2;0;2;2;0;1;1;2;1;1;1;2;1;2;0;1;2;2;1;2;1;0;0;1;2;0;2;1;0;1;0;0;2;2;0;0;1;2;0;2\n"
        "0;1;2;1;0;1;0;0;2;2;0;1;0;1;0;0;2;0;2;0;0;2;0;2;2;0;2;1;0;2;0;2;1;0;0;2;0;1;0;2;0;1;2;1;2;2;1;1;0;1;2;0;0;2;0;2;0;0;1;2;1;0;0;1;0;1;2;1;2;0;1;0;1;0;2;1;0;1;0;1;2;0;1;1;0;1;1;0;0;1;0;1;1;2;1;1;0;2;0;1;0;1;1;2;0;0;1;1;1;2;0;2;2;1;1;0;1;2;2;1;0;2;0;2;2;1;2;2\n"
        "1;2;0;1;0;1;2;1;0;0;0;2;0;2;1;1;2;2;1;1;0;0;2;1;1;0;1;0;2;1;1;0;1;1;2;2;1;2;1;2;2;1;1;1;2;0;2;2;0;2;1;0;2;1;0;0;2;1;2;1;2;2;2;1;2;0;0;1;2;1;0;0;0;1;1;2;2;1;0;2;1;1;0;0;2;1;0;1;0;0;2;0;2;1;2;2;2;2;0;1;2;0;2;2;0;1;0;2;2;0;1;1;2;2;2;1;0;1;1;2;2;0;2;2;0;1;2;2\n"
        "2;0;2;1;2;2;1;2;1;1;2;0;2;0;2;1;2;0;2;0;2;2;2;1;1;2;1;0;1;0;0;1;2;0;0;1;0;2;1;1;1;0;2;1;0;0;2;0;1;0;2;0;0;1;2;0;1;2;0;1;2;1;0;1;0;1;0;0;1;1;2;0;1;2;1;0;2;2;1;0;2;0;1;1;0;0;1;2;2;2;0;1;1;2;2;1;1;0;1;0;1;1;0;0;1;0;2;1;2;0;0;2;0;1;1;0;0;2;1;2;2;2;0;1;1;0;2;2\n"
        "0;0;1;2;0;1;2;2;1;0;1;1;2;1;1;2;0;0;1;1;0;0;2;0;2;0;2;1;1;2;1;2;1;2;1;0;1;1;2;2;0;0;0;2;2;1;2;0;2;1;1;0;1;2;1;0;2;0;2;2;2;1;1;2;0;0;2;1;2;1;0;2;0;0;1;2;0;1;0;2;0;2;0;1;2;1;0;2;2;2;1;0;2;1;2;2;1;2;2;0;0;2;0;0;1;2;1;0;0;1;1;0;1;2;0;2;1;0;2;1;1;1;1;0;2;1;0;1\n"
        "0;2;2;0;0;2;2;1;2;0;2;0;1;2;1;0;1;1;0;0;2;1;1;1;0;0;1;0;2;2;0;2;2;2;0;2;0;2;1;2;0;2;2;0;1;2;0;0;0;2;1;1;0;0;2;1;1;0;2;2;0;2;2;0;2;2;0;1;0;2;2;1;2;0;1;2;2;2;0;0;2;1;2;0;1;1;1;2;0;1;2;1;1;0;1;1;2;2;1;0;1;1;1;2;1;0;2;1;0;0;0;2;1;2;2;1;1;2;0;1;0;1;2;2;0;0;1;2\n"
        "0;0;0;2;1;0;1;0;1;0;0;1;2;0;1;1;0;0;2;2;2;1;1;0;2;2;2;2;1;1;0;2;1;1;1;2;1;1;1;2;0;2;1;2;0;0;0;2;0;0;2;0;2;0;0;2;0;0;2;2;1;2;1;2;2;0;0;0;2;2;1;2;1;0;2;0;0;0;2;0;1;1;2;0;2;2;2;0;0;1;2;2;2;1;2;2;2;0;0;2;2;1;2;2;1;2;2;2;2;2;0;1;1;0;1;0;2;1;1;0;0;2;0;0;2;0;2;0\n"
        "0;2;0;2;1;2;2;1;1;1;0;2;1;1;0;2;1;2;0;2;0;1;2;2;0;0;2;0;1;2;0;1;0;0;2;1;0;1;0;2;1;2;2;2;1;2;0;1;0;1;0;1;2;0;2;0;1;2;1;2;0;2;2;0;0;2;2;1;1;1;1;2;0;1;0;2;2;1;0;2;2;0;0;1;2;1;0;1;2;1;1;0;1;0;1;1;0;1;0;2;2;2;2;0;2;0;0;0;1;1;2;0;2;0;0;2;2;0;0;2;2;1;2;0;0;1;0;0\n"
        "2;0;2;0;2;2;1;1;2;2;1;1;1;2;1;1;2;1;0;2;2;0;1;1;1;1;2;2;2;2;0;2;0;2;0;2;1;2;2;2;1;1;0;2;0;2;1;0;2;1;2;1;2;1;0;2;0;0;1;0;0;1;0;2;1;0;1;0;0;1;2;2;0;0;2;1;0;1;2;0;2;2;1;2;1;2;1;2;0;2;0;0;2;0;2;0;1;0;1;2;0;1;2;0;1;0;1;2;2;1;2;0;2;2;2;1;1;1;0;2;1;0;0;1;1;2;1;1\n"
        "1;0;1;0;1;0;2;0;1;0;1;2;0;0;2;2;0;2;0;2;0;1;2;2;2;2;0;1;2;1;2;1;1;0;2;0;0;1;2;1;2;0;0;2;1;2;2;2;1;2;2;2;1;2;1;1;0;2;0;2;1;1;1;0;1;1;2;1;2;2;2;0;0;2;0;2;2;0;2;0;0;1;0;2;2;1;1;1;1;2;2;0;0;1;1;2;0;0;0;1;0;2;1;0;2;1;0;1;2;2;2;0;1;2;0;0;1;2;0;0;2;0;2;1;2;1;1;2\n"
        "0;2;1;1;2;2;2;2;2;2;1;1;2;1;2;1;2;0;1;1;2;1;1;1;2;2;0;2;2;1;0;0;2;1;1;2;1;0;1;1;0;2;0;0;1;0;2;2;1;0;1;1;2;1;0;2;0;0;2;0;2;2;0;1;0;2;0;2;0;2;1;1;2;0;2;2;0;1;1;1;1;0;0;2;1;0;2;0;0;1;0;0;0;2;1;2;1;2;1;2;2;0;1;0;2;0;2;2;0;1;1;0;2;1;0;1;2;2;1;2;1;2;0;2;1;0;1;0\n"
        "2;1;0;2;1;1;0;0;0;0;0;2;1;1;2;0;0;2;2;2;2;0;2;1;1;2;2;0;2;0;1;2;1;2;0;0;0;2;1;0;1;1;2;0;0;1;1;0;0;1;2;2;2;2;0;1;0;2;2;2;1;0;0;0;2;1;1;0;2;2;1;2;1;2;2;1;1;2;0;0;0;0;2;0;2;1;2;0;1;0;0;2;2;1;0;2;0;2;2;1;1;2;0;0;2;0;2;1;0;2;1;0;0;2;1;2;0;0;2;1;2;2;2;1;2;0;2;1\n"
        "2;1;2;0;0;0;1;2;1;2;0;1;0;0;2;1;2;0;0;0;2;2;1;2;1;0;2;0;1;0;2;0;1;2;2;0;2;1;0;1;0;2;0;1;2;2;0;2;1;1;2;1;1;1;2;0;0;2;1;1;0;1;2;2;0;2;2;1;0;0;2;1;0;1;0;0;2;1;2;2;0;2;0;2;0;1;2;0;0;1;2;2;2;2;2;1;2;2;2;0;0;2;1;0;0;1;0;2;2;2;1;2;2;0;2;0;1;2;1;0;1;1;0;1;0;2;0;0\n"
        "1;0;0;2;2;1;0;1;2;1;1;2;1;2;2;1;2;2;2;1;2;0;2;1;2;0;2;0;2;0;1;2;0;1;2;1;0;0;2;2;1;0;2;0;0;2;1;1;2;1;0;2;2;0;0;2;0;1;2;0;1;0;0;2;2;2;0;0;2;0;0;2;1;0;2;2;0;1;0;2;1;1;2;0;1;2;2;0;0;2;0;2;0;1;2;0;1;0;0;2;1;0;1;2;1;1;1;1;0;0;0;1;2;0;2;0;0;2;0;1;2;2;1;2;0;1;0;2\n"
        "1;0;2;0;1;2;2;1;2;2;2;1;2;0;2;2;2;1;0;0;1;2;0;0;1;2;0;2;0;0;0;2;1;0;0;1;0;2;1;0;2;2;2;2;1;1;0;1;1;2;0;2;2;0;2;1;1;0;0;2;0;1;0;1;1;0;1;2;0;1;2;1;0;2;0;1;2;0;2;0;0;2;2;2;0;2;0;0;2;1;2;0;1;0;2;2;1;2;0;1;0;2;1;0;0;2;2;1;2;1;1;0;2;2;2;2;2;0;2;2;0;0;0;1;2;0;1;0\n"
        "2;1;0;0;0;2;2;1;1;2;2;2;0;2;2;0;1;0;1;0;0;0;1;0;2;1;1;0;1;2;0;0;2;0;0;0;2;0;2;1;2;1;0;0;1;0;1;2;1;2;0;1;0;0;2;2;2;1;2;0;2;0;1;0;1;1;0;1;2;2;2;1;2;1;2;0;2;1;0;0;0;0;1;1;1;1;0;2;1;0;2;0;2;1;1;1;0;2;2;0;1;0;2;1;1;2;2;0;0;2;0;1;0;2;0;1;1;1;0;2;2;0;2;0;1;2;2;2\n"
        "1;2;2;0;2;1;2;2;2;2;1;0;1;1;2;1;2;2;0;1;2;1;0;1;1;0;1;0;2;0;2;0;1;0;2;0;0;1;2;0;0;1;0;2;0;1;2;0;0;1;0;2;0;1;1;2;1;0;2;1;1;2;2;2;0;1;1;0;0;1;1;1;2;0;1;0;1;2;0;2;2;1;2;0;2;0;1;1;1;2;0;0;2;0;0;2;2;1;0;0;2;2;2;0;2;1;0;0;1;1;0;1;2;2;1;2;1;1;2;0;1;1;0;1;2;1;1;1\n"
        "0;2;0;1;1;0;1;2;1;2;2;2;0;0;0;1;2;2;1;2;1;2;2;2;1;0;1;0;2;0;0;1;2;0;2;2;0;1;0;1;1;0;2;0;0;0;1;0;0;0;0;2;0;0;2;1;0;1;2;1;1;2;2;2;0;2;1;0;2;1;0;2;2;2;2;2;2;1;1;0;1;1;2;1;1;1;0;2;0;1;1;2;1;2;1;2;0;2;2;1;2;2;1;1;2;1;2;2;0;0;0;1;2;2;1;1;1;2;1;2;0;0;2;2;1;2;1;2\n"
        "2;1;0;1;2;2;0;1;2;2;0;2;0;1;1;2;0;0;1;0;1;0;0;1;0;2;1;2;2;0;1;1;2;0;2;1;0;2;0;2;0;1;2;0;2;2;1;1;2;0;2;0;0;2;0;1;0;2;1;2;0;0;1;0;1;1;2;1;1;2;1;1;1;1;0;2;0;2;1;0;2;0;1;1;0;2;0;1;1;2;0;1;0;2;2;0;0;1;2;2;1;2;0;2;0;1;2;1;0;1;1;0;1;0;2;0;2;2;1;1;0;2;0;0;2;0;0;1\n"
        "2;2;0;1;1;1;2;2;0;2;0;0;1;2;2;0;2;1;0;0;2;0;2;2;0;1;2;0;1;1;0;1;0;1;0;2;0;2;1;1;2;2;2;0;0;2;2;1;0;1;0;1;0;2;2;0;0;2;2;2;1;0;0;2;0;0;2;1;0;0;2;1;0;1;1;1;1;1;2;2;1;1;2;2;1;0;0;1;2;0;1;1;1;0;1;1;1;1;2;0;2;1;1;0;1;2;1;2;1;1;2;0;0;0;0;2;1;1;2;1;1;0;0;2;0;2;1;2\n"
        "1;1;2;0;1;0;1;2;0;0;0;2;2;2;0;1;0;1;0;1;2;2;0;1;0;1;1;1;0;2;1;2;1;0;2;2;1;1;2;2;2;2;1;2;0;2;0;0;0;0;2;0;0;0;1;1;1;0;2;2;0;1;1;2;2;1;2;1;0;2;0;0;1;2;0;1;2;1;0;0;1;0;1;2;0;0;1;2;1;2;2;2;0;1;1;0;2;1;1;2;0;2;0;2;0;0;1;2;1;2;0;1;1;2;1;2;2;0;1;0;2;0;0;0;1;2;0;1\n"
        "0;0;1;1;0;2;0;0;0;2;1;2;1;2;1;0;1;0;2;0;0;1;2;0;2;2;0;1;2;1;0;0;2;2;1;1;0;0;2;1;0;0;2;0;2;0;0;1;1;1;0;1;2;2;1;0;2;2;1;2;2;1;0;2;0;0;0;2;0;0;2;1;0;1;1;2;0;0;2;1;1;2;0;1;2;2;1;0;0;1;0;2;2;0;2;1;0;0;1;1;1;0;1;1;0;2;0;0;2;1;1;0;0;2;0;1;0;1;1;0;2;1;1;0;1;2;2;0\n"
        "0;1;0;1;2;2;1;2;0;1;0;0;2;2;1;1;2;1;2;2;0;1;2;1;2;0;1;1;0;1;1;0;2;1;2;1;1;0;2;1;2;2;2;0;2;1;1;2;0;1;1;0;2;1;2;2;2;1;0;0;2;0;2;2;1;2;2;0;0;1;0;2;1;0;0;1;0;1;1;2;0;0;1;2;2;1;1;0;1;0;1;1;0;1;0;2;0;0;2;1;2;2;2;1;2;2;2;1;1;1;0;2;1;2;2;2;1;1;0;2;1;2;0;2;1;2;1;2\n"
        "2;1;1;2;1;1;2;1;0;0;2;2;2;1;2;0;1;2;0;0;0;0;2;0;1;1;0;2;0;2;2;0;1;1;2;2;2;0;1;1;1;2;0;1;0;0;2;1;2;0;0;1;2;2;2;0;0;2;0;1;2;2;2;0;2;2;0;1;1;1;0;0;0;1;2;1;1;2;1;2;2;1;1;1;2;0;2;0;0;2;0;1;0;2;0;2;1;1;1;0;1;2;1;2;1;0;2;0;2;0;2;2;0;0;1;0;1;2;1;2;1;1;0;2;0;0;2;0\n"
        "0;0;1;1;0;2;1;1;2;0;2;2;1;2;2;0;0;1;1;2;2;2;2;1;1;2;0;2;1;2;0;1;1;2;0;1;1;1;0;2;0;0;1;2;2;1;2;0;2;0;1;0;0;0;1;1;0;0;2;2;0;0;0;2;0;1;1;2;2;2;2;2;2;1;2;2;0;1;0;2;2;1;2;1;1;2;2;2;0;2;1;0;1;2;0;1;0;2;1;1;2;0;0;2;2;0;1;0;2;2;2;0;1;0;1;0;0;1;0;0;2;0;1;0;2;1;1;0\n"
        "1;0;2;0;1;2;0;1;0;2;1;1;2;2;2;1;0;1;2;0;2;2;0;2;2;1;2;0;1;2;1;0;1;0;2;2;0;2;1;1;2;0;0;2;2;0;0;2;1;2;0;1;0;1;1;2;1;1;2;2;0;1;0;1;1;2;1;1;0;0;2;2;0;2;1;1;1;2;1;2;0;2;1;0;0;2;0;1;1;0;1;0;1;2;1;0;2;0;1;2;0;0;2;2;1;0;0;1;0;1;0;1;0;2;2;2;2;0;1;1;0;0;1;2;0;2;2;0\n"
        "1;1;0;1;2;0;1;0;1;1;2;1;0;0;1;1;2;0;1;2;1;0;1;2;0;2;2;1;2;1;2;0;0;1;0;2;2;1;0;1;0;0;2;1;1;1;2;1;2;1;0;2;1;2;0;2;2;2;0;1;2;2;0;2;1;0;0;1;1;1;2;1;2;1;0;1;2;1;0;1;0;1;0;2;1;1;1;0;1;2;2;0;1;2;1;0;2;1;0;2;2;2;1;2;0;2;0;2;1;0;1;1;2;2;1;1;1;0;2;2;0;2;1;1;1;2;0;2\n"
        "0;0;1;0;0;0;0;1;0;1;1;1;2;0;0;0;0;2;1;1;2;1;2;2;0;1;2;2;0;0;1;0;1;2;2;1;0;2;0;1;1;2;0;1;0;1;2;0;1;0;1;1;2;0;1;2;1;1;0;2;1;1;1;2;2;0;1;2;0;2;2;2;0;1;1;2;1;0;1;0;1;0;1;1;2;0;1;2;1;1;2;0;0;0;2;2;1;2;2;0;2;0;2;1;0;0;1;1;1;0;2;0;1;0;1;0;2;0;1;1;2;0;2;0;1;2;2;0\n"
        "1;2;2;0;2;0;0;2;0;2;1;0;2;0;2;1;2;2;1;1;2;0;0;2;0;2;2;2;2;2;0;1;1;0;0;2;2;0;2;2;2;1;2;1;2;0;2;1;2;1;0;2;1;1;2;2;1;2;2;1;2;2;0;1;2;0;0;0;0;1;0;0;2;2;1;1;2;1;1;2;2;1;2;1;2;2;0;2;2;2;1;1;2;1;0;2;1;1;0;2;2;0;2;2;2;1;1;2;2;0;2;2;0;2;0;0;1;1;0;0;1;2;2;0;1;0;0;1\n"
        "2;0;0;2;2;1;1;2;0;0;2;0;1;1;0;1;1;1;2;0;0;0;1;2;0;2;0;1;2;0;2;1;2;1;1;2;0;0;1;0;0;0;1;2;0;2;0;2;1;0;0;1;2;2;1;2;1;2;1;1;0;2;0;0;0;1;2;1;1;0;1;1;2;1;2;2;0;2;0;2;1;0;2;0;2;0;2;2;2;0;2;2;1;2;0;1;1;2;2;2;1;1;2;1;2;1;1;0;1;1;0;0;2;2;0;2;2;1;1;2;0;1;2;2;2;2;1;0\n"
        "0;1;0;1;2;2;0;0;1;2;1;2;0;2;0;0;2;2;0;2;1;2;1;2;1;0;1;0;1;2;1;1;0;2;0;0;2;1;1;0;1;0;0;2;0;2;0;0;1;1;2;0;1;1;2;2;1;0;0;0;0;2;2;1;1;1;2;0;0;1;0;1;2;2;0;2;1;1;2;0;0;2;0;1;1;2;2;1;0;2;0;0;1;0;2;2;1;2;0;1;0;2;2;2;1;1;2;1;0;2;1;1;1;0;2;2;0;1;2;1;0;2;2;1;1;2;1;2\n"
        "2;2;2;2;1;2;2;0;1;2;1;2;2;0;0;1;0;0;0;1;2;2;0;0;0;0;0;1;0;1;1;2;1;1;1;2;1;0;0;2;1;2;2;0;0;2;1;1;2;1;0;1;2;0;1;0;2;1;2;2;0;0;2;2;0;1;2;1;2;0;1;1;0;1;1;0;2;1;2;0;1;0;1;2;2;0;1;1;1;0;0;2;1;2;1;1;1;2;0;2;0;0;1;0;1;1;1;0;1;1;1;2;2;2;0;1;0;1;2;2;1;0;2;2;2;2;2;1\n"
        "1;2;1;1;2;0;2;1;2;1;0;0;2;1;2;2;1;0;1;2;1;1;2;0;1;1;0;1;0;0;1;2;2;2;1;0;2;1;1;2;2;1;2;0;2;0;0;2;0;1;2;0;2;1;0;1;0;1;0;2;1;1;2;1;0;2;1;1;0;0;1;2;2;0;0;1;1;0;2;0;2;1;2;2;0;0;1;2;2;2;1;0;1;0;2;2;2;1;0;2;0;1;2;0;0;2;0;1;1;0;1;1;2;1;0;1;0;2;0;2;2;2;0;0;0;1;0;0\n"
        "0;0;2;0;0;2;0;0;0;1;2;1;1;2;2;1;1;0;2;1;1;0;1;2;0;2;1;1;1;2;1;1;0;0;2;0;0;2;0;0;1;1;2;0;0;1;2;1;2;2;0;0;1;1;2;1;1;0;2;2;0;0;2;0;2;1;1;0;1;2;0;0;0;0;2;0;1;0;2;1;0;0;2;2;2;2;0;1;2;0;1;0;2;1;0;0;2;1;0;2;2;1;0;2;1;1;0;0;2;0;2;0;0;2;2;0;1;2;1;1;0;1;1;2;2;2;2;0\n"
        "1;0;0;1;1;0;1;1;1;0;1;0;2;0;1;0;0;0;1;1;0;2;0;0;2;2;0;2;2;0;0;2;1;0;2;0;1;2;1;2;2;0;1;1;1;2;2;2;0;0;1;2;1;0;2;1;2;0;1;1;1;1;2;2;2;1;2;0;2;1;2;1;2;1;0;0;1;0;0;2;1;2;1;0;2;1;1;2;2;2;1;0;1;0;2;0;2;0;0;2;1;0;1;0;2;1;1;1;2;2;1;0;2;0;0;2;1;2;1;1;2;2;2;0;1;0;0;1\n"
        "0;1;0;2;2;1;0;2;2;1;1;2;0;2;1;0;2;2;2;0;2;1;2;1;1;0;0;2;0;0;2;1;0;1;0;2;0;0;2;0;1;0;1;0;2;1;0;2;1;1;0;1;2;1;2;1;1;0;2;1;2;0;2;1;2;0;1;2;0;0;0;2;1;2;0;1;0;1;0;2;2;0;0;2;1;0;1;2;1;0;1;0;1;1;2;1;1;0;0;2;1;2;0;2;1;0;0;1;2;0;2;1;2;2;1;0;1;0;1;0;1;0;0;2;2;0;0;1\n"
        "1;2;2;0;1;2;0;2;0;2;1;0;1;0;1;2;1;1;1;2;2;0;2;0;0;1;0;0;2;1;0;0;1;0;1;2;0;0;2;0;2;1;0;2;2;0;0;1;1;2;1;1;2;1;1;0;0;2;1;2;1;2;1;2;2;0;2;2;1;1;2;0;0;1;1;2;1;2;2;0;1;1;0;1;0;0;1;1;0;0;2;2;2;0;1;2;1;1;2;0;1;0;1;2;2;0;0;0;0;1;1;1;2;0;2;0;0;2;2;1;2;1;1;0;2;0;2;1\n"
        "1;0;2;0;1;2;0;0;2;0;2;0;1;2;2;2;2;0;2;2;0;2;1;1;1;2;2;0;2;0;0;2;0;2;2;1;1;2;1;2;1;2;0;0;1;2;2;1;2;0;0;2;2;2;2;2;2;1;2;2;1;2;1;1;0;1;0;1;0;2;0;2;2;0;0;0;1;2;2;1;0;0;1;0;0;1;0;0;2;0;0;1;1;1;0;1;2;2;0;1;0;1;0;1;2;1;2;0;2;2;0;2;0;1;1;0;2;1;2;1;0;2;0;1;1;1;2;1\n"
        "0;2;0;2;1;1;1;2;0;1;0;2;0;0;2;1;1;1;2;1;2;1;0;2;0;1;1;0;2;1;1;0;1;2;1;0;0;2;2;1;0;0;1;1;0;1;2;0;2;0;1;2;1;2;0;0;1;1;0;1;0;2;0;2;2;1;2;0;1;0;2;2;1;0;1;1;1;2;2;1;2;2;1;1;2;1;1;1;2;2;0;2;1;1;2;1;0;2;2;1;0;1;2;1;2;1;0;2;0;0;1;0;0;1;1;2;0;1;1;2;1;2;0;0;0;1;2;1\n"
        "0;1;2;1;0;2;2;1;0;0;0;1;1;1;1;2;1;0;1;1;2;0;2;1;2;1;2;0;2;1;2;0;0;1;2;1;0;1;2;1;1;0;0;0;2;1;1;0;0;1;1;0;1;2;1;0;2;0;1;2;0;0;1;1;2;2;2;0;0;1;0;1;0;0;2;2;0;2;0;1;0;2;0;2;1;0;2;2;1;2;2;1;2;2;2;1;1;0;0;2;0;0;1;1;0;1;2;0;0;2;1;0;2;0;2;1;2;0;0;1;2;2;1;2;2;0;1;1\n"
        "1;2;0;2;2;1;2;0;0;1;2;1;0;0;2;1;1;0;2;2;1;0;2;1;0;0;0;2;1;2;1;0;1;0;0;0;2;2;1;0;0;1;2;2;2;1;2;1;2;0;0;0;1;2;1;1;0;2;1;2;1;2;0;0;0;0;1;2;0;0;2;1;1;2;1;1;1;2;2;0;1;0;2;2;1;1;2;1;0;0;0;0;1;1;1;1;2;2;2;0;0;2;2;2;2;1;0;1;1;0;2;2;2;1;0;0;0;2;1;0;1;0;1;2;1;2;1;1\n"
        "0;0;2;0;0;2;2;0;1;0;1;0;1;2;0;0;0;1;1;0;0;1;2;1;2;1;1;0;1;1;0;1;1;0;2;2;2;0;0;1;1;0;0;0;0;0;1;0;1;0;1;2;1;0;2;0;2;0;2;1;1;0;0;1;0;1;1;1;1;2;2;2;0;2;0;0;0;1;1;2;0;2;0;1;1;2;0;0;2;0;2;2;0;0;1;0;1;2;1;0;1;2;0;0;2;0;1;1;0;0;0;0;0;0;2;1;0;2;0;1;2;2;0;2;0;2;1;1\n"
        "1;0;2;1;2;2;1;2;0;0;0;0;1;0;1;2;0;0;1;0;1;1;1;0;1;2;1;1;0;0;1;1;0;1;2;0;1;1;0;2;2;1;0;1;0;1;2;0;1;0;2;0;2;0;1;1;1;0;0;0;0;2;1;0;1;1;2;0;0;2;2;0;1;2;2;2;1;0;0;0;0;2;1;0;1;0;2;0;1;2;0;2;0;2;0;1;1;0;2;2;0;1;2;2;1;1;2;0;2;0;1;0;1;0;1;0;1;0;0;1;1;0;0;2;0;1;1;2\n"
        "1;2;1;0;0;1;0;0;0;2;0;2;0;1;2;2;1;1;2;2;2;2;1;1;0;1;0;1;1;0;2;0;1;0;0;2;0;0;2;0;0;1;0;2;2;0;1;2;2;1;1;2;1;0;1;2;2;2;1;1;2;1;2;1;1;2;1;1;0;1;2;1;2;2;0;2;0;0;2;0;2;1;2;0;2;1;0;2;1;0;1;1;0;0;1;1;0;0;0;1;1;0;0;1;0;2;1;1;2;0;2;1;1;2;1;0;1;2;1;0;1;2;1;2;1;0;0;1\n"
        "0;1;0;2;2;0;1;0;2;0;2;2;0;2;0;2;1;0;1;0;2;0;0;0;1;0;1;2;0;1;0;2;1;1;2;0;1;0;1;2;1;0;1;1;2;2;1;2;0;1;2;2;1;2;1;2;2;2;1;0;2;2;2;1;0;1;1;0;2;0;0;2;1;1;2;1;1;2;1;1;0;0;2;2;2;0;2;1;0;1;0;0;2;2;1;0;2;0;0;1;0;1;1;0;0;2;2;2;2;0;1;2;0;0;2;1;1;2;2;2;1;2;0;0;0;2;2;2\n"
        "0;0;1;1;2;1;1;2;2;0;1;2;1;0;2;2;1;2;0;1;2;0;1;1;1;2;0;1;1;2;1;1;2;2;1;1;2;0;1;2;2;1;1;0;2;1;2;2;1;0;0;0;2;1;0;0;0;0;2;0;2;0;2;2;0;0;2;0;1;0;2;0;2;2;0;2;1;1;0;0;2;2;1;1;0;1;2;2;1;2;0;1;0;2;1;2;0;0;2;2;1;0;0;2;0;1;2;1;0;1;1;0;1;0;2;1;2;0;0;2;2;0;2;1;1;2;0;2\n"
        "2;2;1;2;0;2;1;0;0;0;2;1;2;1;2;2;1;2;0;2;2;1;1;0;2;1;1;0;1;2;0;1;2;2;1;0;1;2;1;0;0;0;2;1;2;1;0;1;0;1;1;1;0;2;2;0;2;0;2;1;2;0;2;1;0;2;2;2;1;1;0;2;2;2;1;2;1;0;1;0;2;2;2;0;1;1;0;2;1;0;0;1;2;0;0;2;0;1;2;2;2;0;2;2;2;1;2;2;1;1;2;0;1;1;1;1;2;0;1;1;0;2;0;0;0;1;0;1\n"
        "2;0;0;2;1;0;2;1;2;2;0;1;0;0;2;1;2;0;1;2;0;2;0;2;2;1;0;1;0;1;2;0;2;1;1;0;1;1;1;1;0;1;0;2;0;0;2;0;1;0;2;2;1;1;2;1;0;2;2;1;1;1;0;0;2;1;0;2;2;0;1;0;0;1;1;0;1;0;1;0;2;1;0;0;0;0;2;0;0;1;2;2;0;2;0;1;0;1;1;1;2;2;2;1;2;0;2;0;1;2;1;0;0;2;1;0;0;1;0;1;2;1;1;1;1;1;2;2\n"
        "0;1;1;2;0;0;1;0;1;2;2;0;1;0;0;2;1;2;0;1;1;1;2;1;0;1;0;2;0;2;0;2;0;1;0;1;0;2;2;1;2;1;1;0;2;0;0;0;1;0;0;2;0;2;1;1;1;1;0;2;1;0;0;2;0;0;1;1;1;1;2;0;0;2;2;2;0;2;1;1;1;2;1;2;1;1;0;0;2;0;0;2;2;1;1;2;1;0;1;0;2;0;2;2;0;1;0;1;1;2;2;2;0;2;1;2;0;1;1;1;0;1;0;2;2;0;2;2\n"
        "1;0;2;1;1;1;1;2;0;1;0;2;0;1;2;1;0;1;2;2;0;2;0;0;1;2;2;1;2;1;0;0;2;0;1;1;0;1;1;0;2;2;2;1;0;1;0;2;1;2;1;2;0;0;0;2;2;2;1;1;2;2;2;1;0;2;0;0;0;1;1;0;1;1;1;2;1;0;2;0;0;1;0;1;2;1;1;1;0;0;2;0;2;0;0;0;2;0;1;0;0;1;0;1;1;0;1;1;0;0;1;0;0;2;2;2;0;0;2;2;0;2;0;0;0;0;2;0\n"
        "2;0;0;2;0;2;0;2;0;2;0;1;0;0;1;1;0;2;1;1;2;0;1;1;2;1;2;1;2;1;2;1;1;0;1;2;0;2;0;0;1;1;1;2;2;1;0;0;0;0;2;0;0;1;1;1;1;1;0;1;0;0;0;0;1;1;2;2;0;0;0;0;2;0;2;2;1;1;2;2;1;1;2;2;2;2;2;2;1;1;0;1;1;2;1;2;0;2;2;2;2;1;2;2;1;1;0;2;2;2;1;1;2;1;1;2;0;0;1;1;0;2;0;1;1;2;1;1\n"
        "1;2;0;2;0;0;2;1;0;2;1;0;0;2;2;2;0;1;2;2;1;1;0;2;0;1;1;1;2;2;0;0;2;2;0;2;2;1;0;2;0;1;0;1;1;0;1;0;1;1;2;2;2;0;2;2;0;0;1;2;0;2;0;1;0;1;2;2;0;1;2;0;0;0;0;0;0;1;0;1;2;0;1;1;0;0;1;0;1;0;0;2;0;1;0;2;0;0;1;0;1;2;1;0;1;2;0;0;0;1;2;2;1;1;2;2;2;1;0;2;2;0;1;0;2;2;0;2\n"
        "2;0;2;1;0;1;2;0;2;2;0;2;0;1;1;2;2;0;1;2;2;2;1;1;1;0;2;0;2;0;0;2;0;2;1;2;1;0;2;2;0;1;2;1;2;0;0;2;1;1;2;1;1;1;2;2;1;0;2;2;1;0;2;1;1;0;0;2;1;0;1;1;2;2;2;2;2;2;2;1;1;2;2;2;0;2;2;2;0;0;2;1;0;1;0;2;1;0;0;0;2;1;1;2;1;1;0;1;1;2;0;0;1;2;0;2;2;1;0;0;1;2;2;2;1;0;2;0\n"
        "1;2;1;0;1;2;1;1;2;1;2;0;1;2;1;0;2;0;0;1;1;0;2;0;0;2;2;1;2;2;1;0;2;0;2;2;0;1;0;0;2;2;0;0;2;0;1;1;0;2;0;1;0;1;1;0;1;1;2;0;1;2;1;1;2;1;1;0;2;0;0;2;2;2;1;1;2;0;0;2;0;2;0;1;0;2;1;1;0;2;2;2;1;0;1;1;0;1;2;0;2;2;1;2;1;2;2;1;0;2;0;1;2;1;2;0;1;1;2;2;0;1;2;1;1;2;0;2\n"
        "0;0;0;0;0;1;2;0;1;2;1;2;2;2;0;2;1;0;1;0;1;2;0;0;2;0;2;0;1;0;0;2;0;0;2;1;2;2;1;1;2;2;0;2;1;0;1;2;2;2;2;1;2;0;1;2;0;1;0;2;2;0;1;2;0;1;0;0;2;2;2;1;0;2;1;2;2;0;1;1;0;1;0;0;1;2;2;2;1;0;1;0;0;1;0;0;0;2;2;2;1;1;0;1;1;2;2;0;0;1;0;0;0;2;1;2;1;1;2;0;1;2;0;2;0;0;0;1\n"
        "2;2;1;2;0;2;1;2;2;0;1;0;0;1;1;1;0;2;0;1;1;0;2;2;0;0;1;1;0;1;2;1;2;0;1;2;1;2;2;0;1;0;2;1;2;1;0;2;1;0;1;2;1;0;2;1;0;1;2;1;2;2;1;2;0;0;2;1;2;2;2;1;0;0;0;0;1;0;2;0;2;1;2;0;2;1;0;0;2;0;0;1;0;2;2;1;2;2;0;2;2;1;1;0;1;2;0;1;0;0;1;2;2;1;0;2;0;1;0;1;2;0;2;2;2;1;1;1\n"
        "2;0;1;2;1;0;2;2;2;0;0;2;1;0;0;1;2;1;0;2;0;1;1;2;2;0;0;0;0;2;1;2;2;0;1;1;1;1;1;1;1;2;1;0;0;0;1;2;2;0;0;2;2;1;0;2;2;0;2;2;2;2;2;0;0;1;1;2;2;0;1;2;1;2;2;1;0;1;1;2;0;1;2;0;0;2;2;2;2;2;2;0;1;1;0;1;2;0;1;0;1;1;2;2;2;2;1;0;2;0;0;0;2;2;0;2;2;0;0;2;2;0;0;1;2;2;2;2\n"
        "2;1;0;0;0;2;1;0;0;1;1;0;1;2;1;0;2;1;0;1;1;0;1;1;2;1;2;1;0;0;1;0;2;2;2;0;2;0;2;0;1;1;1;2;0;2;2;2;0;1;1;0;2;0;1;1;2;2;2;0;0;1;1;2;1;0;0;1;1;0;1;2;1;2;1;2;2;0;1;0;2;0;1;1;0;2;2;0;2;2;1;2;0;0;2;2;1;1;1;2;1;0;0;0;0;1;2;2;0;2;0;2;0;1;0;2;2;2;0;2;1;2;0;2;2;2;1;1\n"
        "1;0;1;0;1;1;0;1;0;0;1;2;0;0;2;2;2;0;1;2;2;1;2;1;0;0;1;0;2;0;2;1;1;1;2;2;0;0;1;0;0;0;2;2;0;1;1;1;1;0;1;2;1;0;2;2;1;0;2;1;0;2;0;1;0;1;0;0;0;2;2;1;0;1;0;1;0;1;0;0;2;1;1;1;2;0;0;1;0;1;0;1;1;2;1;2;1;2;0;1;2;2;2;2;0;1;0;1;0;1;1;1;2;2;2;2;0;1;1;1;2;1;0;0;0;2;0;1\n"
        "1;1;0;1;0;0;0;2;2;2;0;0;1;2;1;0;0;1;0;0;2;2;2;2;1;2;0;2;2;1;0;1;0;1;0;1;2;1;1;2;1;2;2;0;2;1;0;0;0;1;0;2;1;0;1;2;0;2;1;0;1;0;2;1;2;1;1;2;2;2;1;2;2;0;1;1;2;0;1;2;2;2;2;0;0;1;0;1;1;0;0;2;0;1;2;0;2;0;2;2;2;1;1;1;2;1;0;2;1;2;2;0;0;2;1;1;0;1;0;2;2;1;2;0;2;1;0;1\n"
        "1;0;1;1;1;2;1;1;1;2;1;2;0;1;0;2;0;1;1;2;1;2;1;0;1;0;1;0;0;2;2;2;0;0;1;2;0;2;1;2;2;0;2;2;2;0;2;2;2;1;2;1;1;2;2;0;0;1;0;1;1;2;0;2;1;2;0;0;2;1;0;2;2;2;1;2;0;2;2;1;1;2;2;2;2;2;2;2;0;1;1;0;1;0;2;0;0;2;0;0;2;0;2;2;0;1;1;1;0;0;2;2;0;1;2;2;1;0;2;2;1;1;0;0;2;0;0;0\n"
        "0;0;0;2;1;0;1;2;1;2;0;0;1;0;2;1;2;2;1;1;0;1;1;1;0;1;0;0;2;0;0;0;2;1;0;0;1;2;2;0;2;0;0;0;1;0;0;1;0;1;2;0;0;1;0;0;1;0;1;0;1;1;1;0;0;2;2;2;2;2;0;1;0;0;0;2;1;0;0;2;1;1;0;1;0;0;2;1;1;1;0;0;0;2;1;1;1;0;0;0;0;0;0;1;2;0;0;0;0;2;0;2;0;0;1;0;1;0;0;0;2;0;1;0;2;0;2;2\n";

void nf_initDetector(void **detPointer, int width, int height, float fov)
{
    //create an RGB detector
    umf::UMFDetector<1> *detector = new umf::UMFDetector<1>(umf::UMF_FLAG_SUBWINDOWS|umf::UMF_FLAG_ORIENTATION);


    float aspectRatio = width*1.0f/height;

    //detector->model.getMarker()->setDecisionTreeMinHeight(-1);
    //load marker
    if(!detector->loadMarker(ONSCREEN_SOG_MARKER_STR))
    {
        return;
    }

    Eigen::Matrix3d cameraMatrix;
    float focal = height/(2*tan(fov/2));

    Eigen::Vector2f imgSize(width, height);

    cameraMatrix << focal, 0, imgSize[0]/2,
            0, focal, imgSize[1]/2,
            0, 0, 1;

    Eigen::VectorXd distCoeffs(8);
    distCoeffs << 0, 0, 0, 0, 0, 0, 0, 0;

    detector->model.setCameraProperties(cameraMatrix, distCoeffs);
    detector->model.getMarker()->setLocationMinCorrectPercentage(75);


    *detPointer = detector;
}

void nf_freeDetector(void **detPointer)
{
    delete (umf::UMFDetector<1> *) (*detPointer);
}
