
#include <cv.h>
#include <opencv2/core/core.hpp>
#include <string>
#include "defines.h"

#include "screen_marker_api.h"
#include "umf.h"
#include "util/image.h"
#include "util/umfdebug.h"

using namespace umf;

#ifdef __OPENCV_OLD_CV_H__

	#ifdef CV_MAJOR_VERSION
	//if compiled in releas mode remove the "d" suffix
		#define CV_SHORTVERSION CVAUX_STR(CV_MAJOR_VERSION) CVAUX_STR(CV_MINOR_VERSION) CVAUX_STR(CV_SUBMINOR_VERSION)
		#pragma comment(lib, "opencv_core" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_highgui" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_imgproc" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_features2d" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_calib3d" CV_SHORTVERSION ".lib")
		#pragma comment(lib, "opencv_video" CV_SHORTVERSION ".lib")
	#else
		#pragma comment(lib, "opencv_core220.lib")
		#pragma comment(lib, "opencv_highgui220.lib")
		#pragma comment(lib, "opencv_imgproc220.lib") 
		#pragma comment(lib, "opencv_features2d220.lib") 
	#endif

#else
	#pragma comment(lib, "cxcore210.lib")
	#pragma comment(lib, "cv210.lib")
	#pragma comment(lib, "highgui210.lib") 
#endif


/******** Detection *********************************************************************************/
UMF int __stdcall umf_detect(DetectorProperties *props, cv::Mat *input, DetectorResult *result, cv::Mat *debug)
{

	if(props == NULL || input == NULL || result == NULL)
	{
		return -1;
	}

	UMFDetector<UMF_DETECTOR_CHANNELS> *umfDet = (UMFDetector<UMF_DETECTOR_CHANNELS> *) props->detector;
	
	if(umfDet == NULL)
	{
		return -2;
	}

#if UMF_DETECTOR_CHANNELS == 1
	ImageGray *inputI; 

	if(input->channels() == 3)
	{
		ImageRGB *p = new ImageRGB(input->cols, input->rows, false);
		p->data = (char*) input->data;
		p->widthstep = input->step;
		inputI = new ImageGray(input->cols, input->rows, true, -1);
		convertToGrayscale(p, inputI);
		delete p;
	} else {
		inputI = new ImageGray(input->cols, input->rows, false);
		inputI->data = (char*) input->data;
		inputI->widthstep = input->step;
	}
#else
	ImageRGB *inputI = new ImageRGB(input->cols, input->rows, false);
	inputI->data = (char*) input->data;
	inputI->widthstep = input->step;
#endif

	ImageRGB *debugI = NULL;
	if(debug != NULL)
	{
		if(debug->channels() == 3)
		{
			debugI = new ImageRGB(debug->rows, debug->cols, false);
			debugI->data = (char*) debug->data;
			debugI->widthstep = debug->step;
		}
	}


	UMFDebug *dbg = UMFDSingleton::Instance();
	dbg->setImage(debugI);


	bool success = umfDet->detect(inputI);

	delete inputI;
	if(debugI != NULL)
	{
		delete debugI;
	}

	if(success)
	{
		Eigen::Matrix3d &h = umfDet->model.getHomography();
        result->H = cv::Mat(3, 3, cv::DataType<double>::type);
        result->H.at<double>(0, 0) = h(0, 0);
        result->H.at<double>(0, 1) = h(0, 1);
        result->H.at<double>(0, 2) = h(0, 2);
        result->H.at<double>(1, 0) = h(1, 0);
        result->H.at<double>(1, 1) = h(1, 1);
        result->H.at<double>(1, 2) = h(1, 2);
        result->H.at<double>(2, 0) = h(2, 0);
        result->H.at<double>(2, 1) = h(2, 1);
        result->H.at<double>(2, 2) = h(2, 2);
	}

	return success ? 1 : 0;

}

/******** Initialization ****************************************************************************/

//Create detector 

const char* SMALL_MARKER_STR = ""
"14\n"
"10\n"
"4;0\n"
"12\n"
"2;4;4;1;1;3;0;4;4;4;4;0;4;4\n"
"1;2;2;4;2;0;1;3;0;3;2;0;4;0\n"
"0;0;1;2;2;1;3;1;4;4;0;2;4;2\n"
"4;2;3;1;4;0;0;4;0;1;0;0;0;3\n"
"0;2;0;2;0;2;0;4;0;2;4;0;0;0\n"
"4;3;0;1;1;1;0;4;1;3;3;0;1;0\n"
"2;4;0;4;2;3;1;1;0;0;3;4;3;4\n"
"0;2;4;1;4;1;4;4;1;4;1;4;0;4\n"
"2;4;1;0;0;1;0;4;1;2;1;2;0;1\n"
"1;4;3;1;3;3;1;0;2;1;1;3;3;1";

UMF int __stdcall umf_create_detector(DetectorProperties *props)
{
	if(props == NULL)
	{
		return -1;
	}
	//create an RGB detector
    UMFDetector<UMF_DETECTOR_CHANNELS> *detector = new UMFDetector<UMF_DETECTOR_CHANNELS>(UMF_FLAG_HOMOGRAPHY|UMF_FLAG_SUBPIXEL|UMF_FLAG_SUBWINDOWS|UMF_FLAG_TRACK_LOCAL_POS);

	//create detector class

	//load marker
	if(!detector->loadMarker(SMALL_MARKER_STR))
	{
		return -2;
	}

	props->detector = detector;

	return 0;
}

UMF int __stdcall umf_set_marker_str(DetectorProperties *props, char *str)
{
	UMFDetector<UMF_DETECTOR_CHANNELS> *umfDet = (UMFDetector<UMF_DETECTOR_CHANNELS> *) props->detector;

	if(!umfDet)
	{
		return -1;
	}

	if(!umfDet->loadMarker(str))
	{
		return -2;
	}

	return 0;
}

UMF void __stdcall umf_free_detector(DetectorProperties *detector)
{

	delete ((UMFDetector<UMF_DETECTOR_CHANNELS> *) detector->detector);

	detector->detector = NULL;

	UMFDSingleton::Release();
}

