#include "grid_detector.h"
#include "defines.h"
#include "util/draw.h"
#include "util/umfdebug.h"
#include "util/grid_util.h"
#include <algorithm>
#include <Eigen/Eigenvalues>

namespace umf {

GridDetector::GridDetector()
{
    this->vanishing[0] = Eigen::Vector3f(0, 0, 0);
    this->vanishing[1] = Eigen::Vector3f(0, 0, 0);
    this->horizon = Eigen::Vector3f(0, 0, 0);
    this->pencils[0].clear();
    this->pencils[1].clear();
    this->groups[0].clear();
    this->groups[1].clear();
    this->histogramSize = 12;
    this->transformScale = 1.0f/240;
    this->transformCenter = Eigen::Vector2i(320, 240);
    this->generateLineCountHalf = 4;

    this->ransacIterations = 100;
    this->ransacDotThreshold = 5e-2;
    this->vanishingSoundnessThreshold = 1.f;
    this->replaceEdgels = true;
    this->estimateClusterThreshold = 50; //if replaced edgels this means 25 lines
    this->invKStepThreshold = 0.05; //this is just a rough estimate
}


bool GridDetector::detect(std::vector<Edgel> &edgels, bool show)
{
    bool success = false;

    const bool showTwoGroups = show && false;
    const bool showFilteredLines = show && false;
    const bool showGrid = show && true;

    ////////////////////////////////////////////////////////////////////////////////
    //GROUP
    success = this->separateTwoGroups(edgels, showTwoGroups);

    if(!success)
    {
        return false;
    }

    ////////////////////////////////////////////////////////////////////////////////
    //VANISH
    //finding vanishing point requires the group to be first transformed and normalized
    this->transformEdgels();
    this->normalizeEdgels();

    success = this->findVanish(this->groups[0], this->vanishing[0], showFilteredLines)
            && this->findVanish(this->groups[1], this->vanishing[1], showFilteredLines);


    if(!success)
    {
        return false;
    }

    //extract the lines from edgels
    this->copyEdgels2Pencils();
    //now we can do anything we want with the groups. they are not needed any more

#ifdef UMF_DEBUG
    if(showFilteredLines)
    {
        this->transformEdgelsBack();
        this->showGroups();
    }
#endif

    ////////////////////////////////////////////////////////////////////////////////
    //GRID
    success = this->detectMesh();
    //something went wrong with the grid detection
    if(!success)
    {
        return false;
    }

#ifdef UMF_DEBUG
    if(showGrid)
    {
        //generate pencils going through the corners
        bool p = this->generatePencils(0.5);
        if(p)
        {
            this->transformPencilsBack();
            this->showPencils();
        }
    }
#endif

    //everything is fine, generate our pencils of lines
    success = this->generatePencils();

    if(success)
    {
        //and finally transform it back to normal positions
        this->transformPencilsBack();
    }

    return success;
}

bool GridDetector::detectIndexed(std::vector<Edgel> &edgels, std::vector<int> &indexed, bool show)
{
    //if this is going to be used most of the time, then this should be rewritten
    //so we do not create a subvector each time
    std::vector<Edgel> subPart(indexed.size());
    for(unsigned int i = 0; i < indexed.size(); i++)
    {
        subPart[i] = edgels[indexed[i]];
    }

    return this->detect(subPart, show);
}


inline int getHistogramIndex(float b, float a, const int HIST_SIZE)
{
    double angle = std::atan2(b, a)/M_PI + 1.01;
    return ((int)std::floor(angle*HIST_SIZE))%HIST_SIZE;
}

void getMainTwoBins(std::vector<int> &histogram, int maxBins[2][3])
{
    const int HIST_SIZE = histogram.size();

    maxBins[0][1] = std::max_element(histogram.begin(), histogram.end()) - histogram.begin();
    //left neighborhood
    maxBins[0][0] = (maxBins[0][1] + (HIST_SIZE-1))%HIST_SIZE;
    //right neighborhood
    maxBins[0][2] = (maxBins[0][1] + 1)%HIST_SIZE;

    //zero bin and neighborhood bins
    histogram[maxBins[0][0]] = 0;
    histogram[maxBins[0][1]] = 0;
    histogram[maxBins[0][2]] = 0;

    //same for second max bin + neighborhood
    maxBins[1][1] = std::max_element(histogram.begin(), histogram.end()) - histogram.begin();
    maxBins[1][0] = (maxBins[1][1] + (HIST_SIZE-1))%HIST_SIZE;
    maxBins[1][2] = (maxBins[1][1] + 1)%HIST_SIZE;
}

/**
 * @brief separateTwoGroups separate the edgels into two groups based on a rough histogram
 * @param edgels all the edgels
 * @param show Optionally show the results in the debug output if debugging is enabled
 * @return the success (one or both directions contain zero elements
 *
 * The image below demonstrates the two groups for the edgels extracted by \link EdgelDetector::findEdgels \endlink
 * \image html 4_groups.png
 *
 * The algorithm takes all the lines and creates a rough histogram with low number of bins (default 12 o 18)
 * (set using \link setHistogramSize \endlink). In this histogram two main groups are detected each of width 3.
 * If the groups are overlapping the function will return false.
 * The corresponding edgels are then stored into two groups. In case one the groups is empty, the function again returns false.
 */
bool GridDetector::separateTwoGroups(std::vector<Edgel> &edgels, bool show)
{

    this->groups[0].clear();
    this->groups[1].clear();

    //create set of lines for the four directions
    std::vector< std::vector< std::vector<Edgel>::iterator > > directions(this->histogramSize);
    std::vector<int> directionCounts(this->histogramSize, 0);

    for(std::vector<Edgel>::iterator edgelIt = edgels.begin(); edgelIt != edgels.end(); edgelIt++)
    {

        int hist_index = getHistogramIndex(edgelIt->line[1], edgelIt->line[0], this->histogramSize);

        directions[hist_index].push_back(edgelIt);
        directionCounts[hist_index]++;

    }

    int maxBins[2][3];
    getMainTwoBins(directionCounts, maxBins);

    //bins are too close to each other
    int diff1 = (this->histogramSize + maxBins[0][1] - maxBins[1][1]) % this->histogramSize;
    int diff2 = (this->histogramSize + maxBins[1][1] - maxBins[0][1]) % this->histogramSize;
    if((std::min)(diff1, diff2) < 3)
    {
        return false;
    }

    //group1
    this->groups[0].clear();
    this->groups[0].reserve(directions[maxBins[0][0]].size() +  directions[maxBins[0][1]].size() + directions[maxBins[0][1]].size());

    for(int i = 0; i < 3; i++)
    {
        for(std::vector< std::vector<Edgel>::iterator >::iterator dit = directions[maxBins[0][i]].begin();
            dit != directions[maxBins[0][i]].end(); dit++)
        {
            this->groups[0].push_back(**dit);
        }
    }

    //group2
    this->groups[1].clear();
    this->groups[1].reserve(directions[maxBins[1][0]].size() +  directions[maxBins[1][1]].size() + directions[maxBins[1][1]].size());
    for(int i = 0; i < 3; i++)
    {
        for(std::vector< std::vector<Edgel>::iterator >::iterator dit = directions[maxBins[1][i]].begin();
            dit != directions[maxBins[1][i]].end(); dit++)
        {
            this->groups[1].push_back(**dit);
        }
    }


#ifdef UMF_DEBUG
    if(show)
    {
        this->showGroups();
    }
#endif

    return !(this->groups[0].empty() || this->groups[1].empty());
}

/**
 * @brief Finds a vanishing point to the group of edgels
 * @param group groups of edgels for which the vanishing point is found. The lines in the edgels should be normalized first (as vec3s, not just the normal).
 * @param vanishing the found vanishing point
 * @param replace if the function should replace the group with the filtered lines
 * @return if the vanishing point could be found
 *
 * This function uses ransac to filter out outliers in the group using \link setRANSACIterations \endlink
 * iterations and the threshold for the dot product between the lines and the vanishing point (the closer to 0 the more precise)
 * is set by \link setRANSACDotThreshold \endlink.
 *
 * The RANSAC is done like this:
 *  -# choose randomly two line
 *  -# get the vanishing point by crossing the two lines and normalize it
 *  -# loop through all lines
 *      -# error = dot(line, vanishing)
 *      -# if ( error < thrshold ) accumulate error, and inliers
 *      .
 *  -# compute the score for the group of lines: score = inliers*(threshold - avg_error)*100
 *  -# if so far best store it
 *
 * After RANSAC the best subgroup is used to compute a covariance matrix with mean at the origin
 * ( we are searching for a hyperplane going through the origin). Eigen decomposition is done
 * using eigen - closed form for our 3x3 covariance matrix. After that the smallest eigenvector is
 * chosen. To filter out bad decompositions the \link setEigenSoundnessThreshold \endlink soundness threshold
 * is used (details see that function).
 */
bool GridDetector::findVanish(std::vector<Edgel> &group, Eigen::Vector3f &vanishing, bool replace)
{

    vanishing = Eigen::Vector3f(0, 0, 0);

    int maxLineCount = group.size();
    typedef std::vector< std::vector<Edgel>::iterator > EdgelItList;
    EdgelItList bestSolution;
    int bestSolutionInliers = -1;
    double bestSolutionScore = -1;

    //too few lines
    if(maxLineCount < 2)
    {
        return false;
    }

    //RANSAC
    srand(1);
    for(int iteration = 0; iteration < this->ransacIterations; iteration++)
    {
        //choose two lines at random

        Eigen::Vector3f &line0 = group[rand()%maxLineCount].line;
        Eigen::Vector3f &line1 = group[rand()%maxLineCount].line;

        //get vanishing point
        Eigen::Vector3f ivanishing = line0.cross(line1);
        ivanishing.normalize();

        int inliers = 0;
        float inlierErrorSum = 0;

        EdgelItList currentLines;

        for(std::vector<Edgel>::iterator edgelIt = group.begin(); edgelIt != group.end(); edgelIt++)
        {
            Eigen::Vector3f &edgel = edgelIt->line;

            float dp = std::abs(edgel.dot(ivanishing));
            if(dp < this->ransacDotThreshold)
            {
                inliers++;
                inlierErrorSum += dp;
                currentLines.push_back(edgelIt);
            }
        }

        //funny scoring somehow balance between the number of outliers
        //and the error these lines represen
        double score = inliers*100.0*(this->ransacDotThreshold - inlierErrorSum/inliers);

        //if better than current
        if(score > bestSolutionScore)
        {
            bestSolutionInliers = inliers;
            bestSolutionScore = score;
            //store them as results
            bestSolution = currentLines;
        }
    }

    if(this->ransacIterations < 0)
    {
        for(std::vector<Edgel>::iterator edgelIt = group.begin(); edgelIt != group.end(); edgelIt++)
        {
            bestSolutionInliers++;
            bestSolution.push_back(edgelIt);
        }
    }

    if(bestSolutionInliers < 2)
    {
        if(replace)
        {
            group.clear();
        }
        return false;
    }

    // compute the covariance matrix - the mean is at zero
    // and we have real number so it's pretty simple
    // possible optimization is possible if we don't sum matrices,
    // but do a loop 3x3 and do a loop over all lines there
    //for now enough this way - less issues
    //scale with the length of the edgel
    Eigen::Matrix3f covMat;
    covMat.setZero();
    for(EdgelItList::iterator it = bestSolution.begin(); it != bestSolution.end(); it++)
    {
        covMat += ((**it).line * (**it).line.adjoint()) * (**it).score;
    }

    // now we just have to pick the eigen vector with smallest eigen value
    Eigen::SelfAdjointEigenSolver<Eigen::Matrix3f> eig(covMat);
    //the eigenvectors are sorted in increasing order
    vanishing = eig.eigenvectors().col(0);
    //the soundness of the eigen decomposition - the smaller the better
    float soundness = eig.eigenvalues().coeff(0)/eig.eigenvalues().coeff(1);

    //if we want to replace the group with the best solution from ransac
    if(replace)
    {
        std::vector<Edgel> result;

        for(EdgelItList::iterator it = bestSolution.begin(); it != bestSolution.end(); it++)
        {
            result.push_back(**it);
        }
        group = result;
    }

    if(soundness > this->vanishingSoundnessThreshold)
    {
        return false;
    }

    return true;
}

/**
 * @brief detectMesh Computes the horizon and cals \link detectPencil \endlink for both group of lines
 * @return the success of detecting the pencils
 *
 * The result for our example after group separation \link separateTwoGroups \endlink,
 * vanishing point detection (and also ransac filtering) by \link findVanish \endlink and replacing lines by optionally connecting
 * with the vanishing point \link copyEdgels2Pencils \endlink the result is shown below:
 *
 *  \image html 5_mesh.png "scanline every 200 pixels"
 */
bool GridDetector::detectMesh()
{
    this->horizon = this->vanishing[0].cross(this->vanishing[1]);
    this->horizon.normalize();

    bool success = this->detectPencil(this->vanishing[0], this->pencils[0], this->line0[0], this->indexOffset[0], this->parameterK[0]);

    if(!success)
    {
        return false;
    }

    //problematic passing array elements, if they are not objects
    success = this->detectPencil(this->vanishing[1], this->pencils[1], this->line0[1], this->indexOffset[1], this->parameterK[1]);

    return success;
}

/**
 * @brief detectPencil detect a pencil of lines
 * @param vanishing the theoretical vanishing for the pencil of lines
 * @param pencil the cluster of lines used to detect the pencil
 * @param[out] line0 The line0, relative to which the parameters are calculated is stored here
 * @param[out] indexOffset The index offset of the grid relative to line0 is set if successful
 * @param[out] parameterK The k parameter is stored here
 * @return whether the pencil was successfully detected
 *
 * The equation again which is used to determine the three parameters:
 * l_i = line_0 * k + (i + indexOffset) * h;
 * where l_i are the lines of the grid.
 *
 * This is where the magic happens.
 *  -# choose the line0 as the center of the coordinate system ( [0;0] this corresponds to the transform center of the lines )
 *  -# for each line calculate the k parameter and store it's inverse (the inverse should be linear)
 *  -# somehow estimate the step (see \link  setEstimateClusterThreshold \endlink )
 *      -# option 1 - use simple differences between and take the median
 *      -# option 2 - use sampling and meanshift of the differences - bit strange, but works well
 *      .
 *  -# the estimating algorithm with the estimated step also returns a starting position
 *  -# use this starting position as seed
 *  -# find all lines with corresponding 1/k with threshold (0.5/estimatedK) from the seed
 *  -# set the seed += estimatedK and repeat the last two steps
 *  -# there is a limit for the clusters (currently MAX_CLUSTER_COUNT = 40) - TODO probably should parametrize this too
 *  -# the previous steps create a mapping between cluster indexes and ks
 *  -# run linear regression on these points - returns the slope and offset of the line
 *  -# the slope of the line is the parameterK we are searching for
 *  -# the offset + round(the average index) is the offset of our middle line for generation
 *  -# add 0.5 to the offset so we have the location for the field centers in the marker
 */
bool GridDetector::detectPencil(Eigen::Vector3f &vanishing, std::vector<Eigen::Vector3f> &pencil,
                                Eigen::Vector3f &line0, float &indexOffset, float &parameterK)
{
    if(pencil.empty())
    {
        return false;
    }

    //consider that everythin is normalized, and the normal center is 0, 0
    Eigen::Vector3f ecenter(0.f, 0.0f, 1.0f);

    //now for each line we can calculate it's k based on on the line connecting the reference line and the center
    line0 = ecenter.cross(vanishing);
    line0.normalize();

    //other calculations using inverse product
    Eigen::Vector3f ln = line0.cross(horizon);
    float nsqrinv = 1.0f/ln.dot(ln);

    Eigen::Vector3f l0q = horizon.cross(ln)*nsqrinv;
    Eigen::Vector3f lhq = ln.cross(line0)*nsqrinv;

    //now create a list of all inverted k-s and store it somewhere
    std::vector<float> invks;
    for(unsigned int i = 0; i != pencil.size(); i++)
    {
        float invk = pencil[i].dot(lhq)/pencil[i].dot(l0q);
        invks.push_back(invk);
    }
    /********************************************************************/
    /********************************************************************/

    //now sort all k inverts-s
    std::sort(invks.begin(), invks.end());

    //////////////////////////////////////////////////////////////////////////////
    //estimate the step between clusters of lines

    float estimatedStep = 0;
    float startOffset = 0;
    bool success = false;
    if(invks.size() > this->estimateClusterThreshold)
    {
        success = estimateKCluster(invks, this->invKStepThreshold, estimatedStep, startOffset);
    } else {
        success = estimateKSimple(invks, this->invKStepThreshold,  estimatedStep, startOffset);
    }

    if(!success)
    {
        pencil.clear();
        return false;
    }


    ///////////////////////////////////////////////////////////////////////////////////
    //create relation mapping ks to indexes

    float clusterDiffThreshold = estimatedStep/2;

    //now create groups and assign them indexes

    const int MAX_CLUSTER_COUNT = 40;
    const int MAX_CLUSTER_COUNT_HALF = MAX_CLUSTER_COUNT/2;

    std::vector<float> G0;
    getKGroupSorted(invks, G0, startOffset, clusterDiffThreshold);
    if(G0.empty())
    {
        //this probably means to large deviation was found, that we are unable to detect
        pencil.clear();
        return false;
    }
    //assign index weights this will be important later when we try to find the
    //mean for the indexes, so we generate lines close to the mean

    std::vector<float> indexWeights(MAX_CLUSTER_COUNT, 0);
    //median
    float g0 = G0[G0.size()/2];

    //we must use pointers, otherwise it's problematic being eigen fixed size arrays
    std::vector<Eigen::Vector2f> indexes;
    addClusterMapping(indexes, G0, 0);

    //now that we have a starting position, just move along to the next and create our
    //function for linear regression

    float nextStart = g0 + estimatedStep;
    float maxKinv = invks.back();
    for(int index = 1; index < MAX_CLUSTER_COUNT_HALF && nextStart < maxKinv; index++)
    {
        std::vector<float> Gi;
        getKGroupSorted(invks, Gi, nextStart, clusterDiffThreshold);
        indexWeights[MAX_CLUSTER_COUNT_HALF +index] = Gi.size();
        if(Gi.empty())
        {
            //we skipped one probably
            nextStart += estimatedStep;
        } else {
            //everything is fine, calculate the next start by getting the mean of the current, plus the diffThreshold
            float gi = Gi[Gi.size()/2];
            addClusterMapping(indexes, Gi, index);
            nextStart = gi + estimatedStep;
        }
    }

    nextStart = g0 - estimatedStep;
    float minKinv = invks.front();
    for(int index = -1; index > -MAX_CLUSTER_COUNT_HALF && nextStart > minKinv; index--)
    {
        std::vector<float> Gi;
        getKGroupSorted(invks, Gi, nextStart, clusterDiffThreshold);
        indexWeights[MAX_CLUSTER_COUNT_HALF +index] = Gi.size();
        if(Gi.empty())
        {
            //we skipped one probably
            nextStart -= estimatedStep;
        } else {
            //everything is fine, calculate the next start by getting the mean of the current, plus the diffThreshold
            float gi = Gi[Gi.size()/2];
            addClusterMapping(indexes, Gi, index);
            nextStart = gi - estimatedStep;
        }
    }
    /////////////////////////////////////////////////////////////////////////////////
    // use linear regression to get  coefficients

    //now we should have a nice mapping for each k to its indexes

    Eigen::Vector3f line = fitLine(indexes);

    Eigen::Vector2f coeffs;
    coeffs[0] = - line[0]/line[1];
    coeffs[1] = - line[2]/line[1];

    //great, we have our line matching best our indexes based on k
    /*****************************************************************************/
    /****************************************************************************/
    //first get k
    parameterK = coeffs[0]; //actually this gives us our k - how beautiful :)

    //now get index offset
    //simple least squares with weighted clusters
    float avgIndexSum = 0;
    float weightSum = 0;
    for(int i = 1; i < MAX_CLUSTER_COUNT; i++)
    {
        avgIndexSum += i*indexWeights[i];
        weightSum += indexWeights[i];
    }
    //round the average
    float avgIndex = floor(avgIndexSum/weightSum - MAX_CLUSTER_COUNT_HALF + 0.5);
    /* - offset of the zero's cluster + offset of the mean + offset of the field centers*/
    indexOffset = - coeffs[1] + avgIndex + 0.5f;

    return true;
}

/**
 * @brief Generate pencils of lines based on the stored parameters
 * @param extraIndexOffset extra offset
 *
 * The is that indexOffset goes through the center of the fields. If somebody want lines going through
 * the corners, the extra offset should be set to 0.5.
 * First the function clears the stored pencils and replaces them with our lines
 *
 * The function also test if the grid is a correct perspectively deformed rectangle by using
 * the dot product between the first and last lines in the pencils (if for some reason
 * the dot product is negative, it means the line direction suddenly got inverted -> wrong
 * vanishing point)
 */
bool GridDetector::generatePencils(float extraIndexOffset)
{
    bool success = true;
    for(int pi = 0; pi < 2; pi++)
    {
        this->pencils[pi].clear();

        for(int i = -this->generateLineCountHalf; i < this->generateLineCountHalf; i++)
        {
            //if(i == 0) continue;
            Eigen::Vector3f newline = this->line0[pi]*this->parameterK[pi] + (i + this->indexOffset[pi] + extraIndexOffset)*this->horizon;
            newline.normalize();
            //std::cout << "Line " << i << ": " << newline.a << "; " << newline.b << "; " << newline.c << std::endl;
            this->pencils[pi].push_back(newline);
        }

        success = success && (pencils[pi].front().block(0,0,2,1).dot(pencils[pi].back().block(0,0,2,1)) > 0);
    }
    return success;
}


void GridDetector::showGroups()
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();

    if(imgDbg == nullptr)
    {
        return;
    }

    //Eigen::Vector3i lineColor1(255, 98, 51);
    //Eigen::Vector3i lineColor2(100, 100, 255);
	
    Eigen::Vector3i lineColor1(255, 197, 38);
    Eigen::Vector3i lineColor2(45, 250, 59);

    int lineWidth = 2;

    for(std::vector<Edgel>::iterator eIt = this->groups[0].begin(); eIt != this->groups[0].end(); eIt++)
    {
        drawLineEq(imgDbg, eIt->line, lineColor1, lineWidth);
    }

    for(std::vector<Edgel>::iterator eIt = this->groups[1].begin(); eIt != this->groups[1].end(); eIt++)
    {
        drawLineEq(imgDbg, eIt->line, lineColor2, lineWidth);
    }

}

void GridDetector::showPencils()
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();

    if(imgDbg == nullptr)
    {
        return;
    }


    Eigen::Vector3i lineColor1(255, 98, 51);
    //Eigen::Vector3i lineColor2(100, 100, 255);
	Eigen::Vector3i lineColor2(45, 250, 59);
    int lineWidth = 3;

    for(unsigned int i = 0; i < this->pencils[0].size(); i++)
    {
        drawLineEq(imgDbg, this->pencils[0][i], lineColor1, lineWidth);
    }

    for(unsigned int i = 0; i < this->pencils[1].size(); i++)
    {
        drawLineEq(imgDbg, this->pencils[1][i], lineColor2, lineWidth);
    }

}

/**
 * @brief separateTwoGroupsIndexed same as \link separateTwoGroups \endlink just with prefiltered indexes
 * @param edgels
 * @param indexed
 * @param show
 * @return success
 */
bool GridDetector::separateTwoGroupsIndexed(std::vector<Edgel> &edgels, std::vector<int> &indexed, bool show)
{
    //if this is going to be used most of the time, then this should be rewritten
    //so we do not create a subvector each time
    std::vector<Edgel> subPart(indexed.size());
    for(unsigned int i = 0; i < indexed.size(); i++)
    {
        subPart[i] = edgels[indexed[i]];
    }
    return this->separateTwoGroups(subPart, show);
}



void GridDetector::transformLine(Eigen::Vector3f &line)
{
    line[2] = (line[2] + (line[0]*this->transformCenter[0] + line[1]*this->transformCenter[1]))*this->transformScale;
}

void GridDetector::transformEdgel(Edgel &edgel)
{
    this->transformLine(edgel.line);
    edgel.endPoints[0][0] = (edgel.endPoints[0][0] - this->transformCenter[0])*this->transformScale;
    edgel.endPoints[0][1] = (edgel.endPoints[0][1] - this->transformCenter[1])*this->transformScale;
    edgel.endPoints[1][0] = (edgel.endPoints[1][0] - this->transformCenter[0])*this->transformScale;
    edgel.endPoints[1][1] = (edgel.endPoints[1][1] - this->transformCenter[1])*this->transformScale;
    edgel.score *= this->transformScale;
}

void GridDetector::transformLineBack(Eigen::Vector3f &line)
{
    float scale = 1.0f/this->transformScale;
    line[2] = line[2]*scale - line[0]*this->transformCenter[0] - line[1]*this->transformCenter[1];
}

void GridDetector::transformEdgelBack(Edgel &edgel)
{
    this->transformLineBack(edgel.line);
    float scale = 1.0f/this->transformScale;
    edgel.endPoints[0][0] = edgel.endPoints[0][0]*scale + this->transformCenter[0];
    edgel.endPoints[0][1] = edgel.endPoints[0][1]*scale + this->transformCenter[1];
    edgel.endPoints[1][0] = edgel.endPoints[1][0]*scale + this->transformCenter[0];
    edgel.endPoints[1][1] = edgel.endPoints[1][1]*scale + this->transformCenter[1];
    edgel.score *= scale;
}

void GridDetector::transformEdgels()
{
    for(std::vector<Edgel>::iterator eIt = this->groups[0].begin(); eIt != this->groups[0].end(); eIt++)
    {
        this->transformEdgel(*eIt);
    }
    for(std::vector<Edgel>::iterator eIt = this->groups[1].begin(); eIt != this->groups[1].end(); eIt++)
    {
        this->transformEdgel(*eIt);
    }
}

void GridDetector::transformEdgelsBack()
{
    for(std::vector<Edgel>::iterator eIt = this->groups[0].begin(); eIt != this->groups[0].end(); eIt++)
    {
        this->transformEdgelBack(*eIt);
    }
    for(std::vector<Edgel>::iterator eIt = this->groups[1].begin(); eIt != this->groups[1].end(); eIt++)
    {
        this->transformEdgelBack(*eIt);
    }
}

void GridDetector::transformPencils()
{
    for(unsigned int i = 0; i < this->pencils[0].size(); i++)
    {
        this->transformLine(this->pencils[0][i]);
    }

    for(unsigned int i = 0; i < this->pencils[1].size(); i++)
    {
        this->transformLine(this->pencils[0][i]);
    }
}


void GridDetector::transformPencilsBack()
{
    for(unsigned int i = 0; i < this->pencils[0].size(); i++)
    {
        this->transformLineBack(this->pencils[0][i]);
    }

    for(unsigned int i = 0; i < this->pencils[1].size(); i++)
    {
        this->transformLineBack(this->pencils[1][i]);
    }
}

void GridDetector::normalizeEdgels()
{
    for(std::vector<Edgel>::iterator eIt = this->groups[0].begin(); eIt != this->groups[0].end(); eIt++)
    {
        eIt->line.normalize();
    }
    for(std::vector<Edgel>::iterator eIt = this->groups[1].begin(); eIt != this->groups[1].end(); eIt++)
    {
        eIt->line.normalize();
    }
}

void getEdgelLines(std::vector<Edgel> &edgels, std::vector<Eigen::Vector3f> &pencil, Eigen::Vector3f &vanishing)
{
    pencil.clear();
    for(std::vector<Edgel>::iterator eIt = edgels.begin(); eIt != edgels.end(); eIt++)
    {
        Eigen::Vector3f p1 = vanishing.cross(Eigen::Vector3f(eIt->endPoints[0][0], eIt->endPoints[0][1], 1));
        Eigen::Vector3f p2 = vanishing.cross(Eigen::Vector3f(eIt->endPoints[1][0], eIt->endPoints[1][1], 1));
        p1.normalize();
        p2.normalize();
        pencil.push_back(p1);
        pencil.push_back(p2);
    }
}


/**
 * @brief copyEdgels2Pencils Extract lines from the edgels and store them in the pencils
 * They are further processed then by \link detectMesh \endlink.
 * If the \link setReplaceEdgels \endlink is set, then for each edgel two lines are added
 * by connecting the vanishnig point with the endpoints of the edgel
 */
void GridDetector::copyEdgels2Pencils()
{
    this->pencils[0].clear();
    this->pencils[1].clear();
    if(this->replaceEdgels)
    {
        getEdgelLines(this->groups[0], this->pencils[0], this->vanishing[0]);
        getEdgelLines(this->groups[1], this->pencils[1], this->vanishing[1]);
    } else {
        for(std::vector<Edgel>::iterator eIt = this->groups[0].begin(); eIt != this->groups[0].end(); eIt++)
        {
            this->pencils[0].push_back(eIt->line);
        }
        for(std::vector<Edgel>::iterator eIt = this->groups[1].begin(); eIt != this->groups[1].end(); eIt++)
        {
            this->pencils[1].push_back(eIt->line);
        }
    }
}

}
