// NetDesign.cpp - implementation for CNetDesign
//

#include "stdafx.h"
#include "CESim.h"
#include "NetDesign.h"

////////////////////////////////////////////////////////////////////////////
// CNetDesign
IMPLEMENT_SERIAL(CNetDesign,CObject,1|VERSIONABLE_SCHEMA)

CNetDesign::CNetDesign()
{
	m_window_size.cx=2048;
	m_window_size.cy=1536;
}

void CNetDesign::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CObject::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_window_size;
	}
	else
	{
		ar >> m_window_size;
	}
	m_element_list.Serialize(ar);
}

void CNetDesign::DeleteContents()
{
	m_window_size.cx=2048;
	m_window_size.cy=1536;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)delete m_element_list.GetNext(pos);
	m_element_list.RemoveAll();
//	m_net_conditions.Reset();
}

void CNetDesign::SetWindowSize(CSize window_size)
{
	CSize min_window_size;
	// compute minimal window size
	CRect window_rect,tmp_rect;
	window_rect.SetRectEmpty();
	tmp_rect.SetRectEmpty();
	CPNElement* pElement;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		window_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=window_rect;
	}
	min_window_size.cx=window_rect.right;
	min_window_size.cy=window_rect.bottom;
	if(min_window_size.cx<100)min_window_size.cx=100;
	if(min_window_size.cy<100)min_window_size.cy=100;
	// set window size
	if(window_size.cx<min_window_size.cx)window_size.cx=min_window_size.cx;
	if(window_size.cy<min_window_size.cy)window_size.cy=min_window_size.cy;
	m_window_size=window_size;
}

void CNetDesign::Draw(CDC* pDC)
{
	CPNElement* pElement;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pElement->Draw(pDC);
	}
}

CRect CNetDesign::PlaceSelectedElements(bool place_back,CPNElementList& selected_elements)
{	// place elements to back or front of all selected elements
	CRect invalid_rect,tmp_rect;
	tmp_rect.SetRectEmpty();
	invalid_rect.SetRectEmpty();
	POSITION sel_pos=selected_elements.GetHeadPosition();
	CPNElement* pElement;
	while(sel_pos!=NULL)
	{
		pElement=selected_elements.GetNext(sel_pos);
		// 
		POSITION el_pos=m_element_list.Find(pElement);
		if(el_pos!=NULL)
		{
			m_element_list.RemoveAt(el_pos);
			if(place_back)
				m_element_list.AddHead(pElement);
			else
				m_element_list.AddTail(pElement);
		}
		invalid_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=invalid_rect;
	}
	return invalid_rect;
}

bool CNetDesign::AddElement(CPNElement* pElement)
{	// add new element
	CPNEdge* pEdge=dynamic_cast<CPNEdge*>(pElement);
	if(pEdge)
	{	// adding new edge
		CPNElement* pPomElement;
		POSITION pos=m_element_list.GetHeadPosition();
		while(pos!=NULL)
		{
			pPomElement=m_element_list.GetNext(pos);
			if(pEdge->Compare(dynamic_cast<CPNEdge*>(pPomElement)))
			{	// same edge already exist
				delete pElement;
				return false;
			}
		}
	}
	m_element_list.AddTail(pElement);
	return true;
}

CRect CNetDesign::RemoveSelectedElements(CPNElementList& selected_elements)
{
	CRect invalid_rect,tmp_rect;
	tmp_rect.SetRectEmpty();
	CPNElement* pElement;
	CPNEvent* pEvent;
	CPNLabel* pLabel;
	CPNLabel* pObj_Label;
	CPNCondition* pCondition;
	POSITION rm_pos;
	POSITION oldpos;
	POSITION pos=selected_elements.GetHeadPosition();
	while(pos!=NULL)
	{
		oldpos=pos;
		pElement=selected_elements.GetNext(pos);
		pLabel=dynamic_cast<CPNLabel*>(pElement);
		if(pLabel)
		{	
			if(pLabel->GetOwner()!=NULL)
			{
				invalid_rect.UnionRect(tmp_rect,pElement->GetRect());
				tmp_rect=invalid_rect;
				continue; // vymazani label od objektu se provede az s objektem
			}
		}
		pObj_Label=NULL;
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)pObj_Label=pEvent->GetLabel();
		pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pCondition)pObj_Label=pCondition->GetLabel();
		// delete element
		selected_elements.RemoveAt(oldpos);
		rm_pos=m_element_list.Find(pElement);
		if(rm_pos!=NULL)m_element_list.RemoveAt(rm_pos);
		if(pObj_Label)
		{	// delete element's label
			rm_pos=selected_elements.Find(pObj_Label);
			if(rm_pos!=NULL)selected_elements.RemoveAt(rm_pos);
			rm_pos=m_element_list.Find(pObj_Label);
			if(rm_pos!=NULL)
			{
				m_element_list.RemoveAt(rm_pos);
				invalid_rect.UnionRect(tmp_rect,pObj_Label->GetRect());
				tmp_rect=invalid_rect;
				delete pObj_Label;
			}
			// delete corresponding edges
			CPNEdge* pEdge;
			CPNElement* pPomElement;
			POSITION pompos;
			pos=m_element_list.GetHeadPosition();
			while(pos!=NULL)
			{
				pompos=pos;
				pPomElement=m_element_list.GetNext(pos);
				pEdge=dynamic_cast<CPNEdge*>(pPomElement);
				if(pEdge)
				{
					if(pEdge->IsCorresponding(pElement))
					{
						rm_pos=selected_elements.Find(pEdge);
						if(rm_pos!=NULL)selected_elements.RemoveAt(rm_pos);
						m_element_list.RemoveAt(pompos);
						invalid_rect.UnionRect(tmp_rect,pEdge->GetRect());
						tmp_rect=invalid_rect;
						delete pEdge;
					}
				}
			}
			pos=selected_elements.GetHeadPosition(); // iterate again
		}
		// delete element
		invalid_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=invalid_rect;
		delete pElement;
	}
	selected_elements.RemoveAll();
	return invalid_rect;
}

CRect CNetDesign::RecalculateEdges(const CPNElement* pDragged_element/*=NULL*/)
{	// recalculate intersections with elements and edges
	CRect invalid_rect,edge_rect;
	invalid_rect.SetRectEmpty();
	CRect tmp_rect,oldrect;
	tmp_rect.SetRectEmpty();
	POSITION pos=m_element_list.GetTailPosition();
	CPNEdge* pEdge;
	while(pos!=NULL)
	{
		pEdge=dynamic_cast<CPNEdge*>(m_element_list.GetPrev(pos));
		// recalculate only edges that needs recalculation == they are connected with pDragged_element
		if(pEdge)
		{
			edge_rect=pEdge->GetRect();
			if(pEdge->Recalculate(pDragged_element))
			{
				invalid_rect.UnionRect(tmp_rect,edge_rect);
				tmp_rect=invalid_rect;
				invalid_rect.UnionRect(tmp_rect,pEdge->GetRect());
				tmp_rect=invalid_rect;
			}
		}
	}
	return invalid_rect;
}

CPNElement* CNetDesign::ElementAt(const CPoint& point,bool events/*=false*/,bool conditions/*=false*/)
{	// return top element on point
	bool is_event;
	bool is_condition;
	CPNElement* pElement;
	POSITION pos=m_element_list.GetTailPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetPrev(pos);
		if(pElement->HitTest(point))
		{
			is_event=false;
			is_condition=false;
			if(events) // uprednostnuje udalosti v simulaci a u kresleni hran
				is_event=(dynamic_cast<CPNEvent*>(pElement)!=NULL);
			if(conditions) // uprednostnuje podminky u FLIP_TOKEN
				is_condition=(dynamic_cast<CPNCondition*>(pElement)!=NULL);
			if((events && is_event) || (conditions && is_condition) || (!events && !conditions))return pElement;
		}
	}
	return NULL;
}

CRect CNetDesign::SelectElements(CRect& rect,bool simulation_mode,CPNElementList& selected_elements)
{	// select elements with in rect
	CRect invalid_rect;
	CRect tmp_rect;
	tmp_rect.SetRectEmpty();
	CRect rect_element;
	bool select;
	CPNElement* pElement;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		rect_element=pElement->GetRect();
		if( (rect_element.left>rect.left) && (rect_element.top>rect.top) &&
			(rect_element.right<rect.right) && (rect_element.bottom<rect.bottom))
		{
			select=false;
			if(simulation_mode)
			{
				CPNEvent* pEvent;
				pEvent=dynamic_cast<CPNEvent*>(pElement);
				if(pEvent)if(pEvent->IsRealizable())select=true;
			}
			else
				select=true;
			if(select)
			{
				pElement->Select();
				selected_elements.AddTail(pElement);
				invalid_rect.UnionRect(tmp_rect,rect_element);
				tmp_rect=invalid_rect;
			}
		}
	}
	return invalid_rect;
}

CSetConditions CNetDesign::ComputeElementsSets()
{	// compute presetes and postsets
	int edges=0;
	int events=0;
	int conditions=0;
	CPNEvent* pEvent;
	CPNElement* pElement;
	CPNCondition* pCondition;
	// reset presets and postsets and compute initial case
	CSetConditions	initial_case;
//	m_net_conditions.Reset();
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)
		{
			events++;
			pEvent->GetPostSet().Reset();
			pEvent->GetPreSet().Reset();
		}
		pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pCondition)
		{
			conditions++;
			pCondition->GetPostSet().Reset();
			pCondition->GetPreSet().Reset();
			if(pCondition->GetToken())initial_case.Add(pCondition);
//			m_net_conditions.Add(pCondition);
		}
	}
	// compute presets and postsets of events and conditions
	CPNEdge* pEdge;
	pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pEdge=dynamic_cast<CPNEdge*>(m_element_list.GetNext(pos));
		if(pEdge)
		{
			edges++;
			pEdge->UpdateCorrespondingSets();
			pEdge->ComputeTokenPath();
		}
	}
	// output simulation informations
	CString out_str;
	m_nEdges=edges;
	m_nEvents=events;
	m_nConditions=conditions;
	out_str.Format("Net elements: %d conditions, %d events, %d edges",conditions,events,edges);
	_AddOutputListString(OUTPUT_STANDARD,out_str);
	return initial_case;
}

void CNetDesign::UpdateTokensPath()
{	// compute presetes and postsets
	CPNEdge* pEdge;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pEdge=dynamic_cast<CPNEdge*>(m_element_list.GetNext(pos));
		if(pEdge)pEdge->ComputeTokenPath();
	}
}

void CNetDesign::ReflectCase(CSetConditions& case_set)
{	// sets marking from case to the net
	CPNCondition* pCondition;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{	// for every condition
		pCondition=dynamic_cast<CPNCondition*>(m_element_list.GetNext(pos));
		if(pCondition)pCondition->SetToken(case_set.Includes(pCondition));
	}
}

void CNetDesign::UpdateEvents(CSetConditions& case_set)
{
	CPNEvent* pEvent;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{	// for every event
		pEvent=dynamic_cast<CPNEvent*>(m_element_list.GetNext(pos));
		if(pEvent)pEvent->UpdateRealizable(case_set);
	}
}

void CNetDesign::ClearNet()
{	// clear net from simulation
	CPNEvent* pEvent;
	CPNElement* pElement;
	CPNCondition* pCondition;
	POSITION pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)
		{
			pEvent->ClearRealizable();
			pEvent->SetRealizing(false);
			continue;
		}
		pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pCondition)pCondition->SetHalfFullToken(false);
	}
}

int CNetDesign::DoNetComplement(CDC* pDC,bool onlycompute/*=false*/) 
{	// 
	CString out_str;
	int nComplements=0;
	CPNCondition* pCondition;
	CPNCondition* pFindCondition;
	POSITION oldpos=NULL;
	POSITION pos=m_element_list.GetHeadPosition();
	// aby potom nehledal i v nove pridanych podminkach
	POSITION endpos=m_element_list.GetTailPosition();
	while((pos!=NULL) && (oldpos!=endpos))
	{	// for every condition
		oldpos=pos;
		pCondition=dynamic_cast<CPNCondition*>(m_element_list.GetNext(pos));
		if(pCondition)
		{
			bool find=false;
			// hleda komplement, zacina hledat pro urychleni od posledni pozice
			POSITION findpos=m_element_list.GetHeadPosition();
			while(findpos!=NULL)
			{	// for every condition
				pFindCondition=dynamic_cast<CPNCondition*>(m_element_list.GetNext(findpos));
				if(pFindCondition)
				{
					if(pFindCondition!=pCondition)
					{
						if((pCondition->GetPreSet()==pFindCondition->GetPostSet()) && (pCondition->GetPostSet()==pFindCondition->GetPreSet()))
						{
							find=true;
							break;
						}
					}
				}
			}
			if(!find)
			{	// create complement
				nComplements++;
				if(onlycompute)continue;
				CPNLabel* pLabel;
				pLabel=pCondition->GetLabel();
				CString labelstr;
				if(pLabel)labelstr=pLabel->GetText();
				out_str="Adding complement { not ";
				out_str+=labelstr;
				out_str+=" } to condition { ";
				out_str+=labelstr;
				out_str+=" }";
				_AddOutputListString(OUTPUT_ANALYSIS,out_str);
				labelstr="not "+labelstr;
				CPNLabel* pNewLabel;
				pNewLabel=new CPNLabel();
				pNewLabel->SetText(labelstr,pDC);
				CPoint point;
				point=pCondition->GetPosition();
				if(point.y<80)
					point.y+=60;
				else
					point.y-=60;
				// add new condition
				CPNCondition* pNewCondition=new CPNCondition(point,pNewLabel);
				pNewCondition->SetToken(!pCondition->GetToken());
				AddElement(pNewCondition);
				AddElement(pNewLabel);
				pNewLabel->SetOwner(pNewCondition);
				// prida hrany
				CPNEvent* pEvent;
				CSetEvents events;
				events=pCondition->GetPreSet();
				events.InitIteration();
				pEvent=events.NextIteration();
				while(pEvent)
				{	// 
					CPNEdge* pEdge=new CPNEdge(pNewCondition,pEvent);
					AddElement(pEdge);
					pEvent=events.NextIteration();
				}
				events=pCondition->GetPostSet();
				events.InitIteration();
				pEvent=events.NextIteration();
				while(pEvent)
				{	// 
					CPNEdge* pEdge=new CPNEdge(pEvent,pNewCondition);
					AddElement(pEdge);
					pEvent=events.NextIteration();
				}
			}
		}
	}
	return nComplements;
}

template<>
static UINT AFXAPI HashKey<CString>(CString& str)
{	
	LPCTSTR key=(LPCTSTR)str;
	UINT nHash=0;
	while(*key)nHash=(nHash<<5)+nHash+*key++;
	return nHash;
}

int CNetDesign::TestElementsNaming()
{
	_AddOutputListString(OUTPUT_STANDARD,"  Correct naming...");
	// test na unikatnost a neprazdnost jmena podminky
	// test na neprazdnost jmena udalosti
	int nSameName=0;
	int nBlankCondition=0;
	int nBlankEvent=0;
	POSITION pos;
	int nErrors=0;
	CPNEvent* pEvent;
	CPNLabel* pLabel;
	CString label_str;
	CPNElement* pElement;
	CPNCondition* pCondition;
	CMap<CString,CString&,DWORD,DWORD> name_map;
	pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pCondition)
		{	// 
			pLabel=pCondition->GetLabel();
			if(pLabel)
			{
				label_str=pLabel->GetText();
				if(label_str.IsEmpty())
				{
					nBlankCondition++;
				}
				else
				{
					DWORD count=0;
					name_map.Lookup(label_str,count);
					count++;
					name_map.SetAt(label_str,count);
				}
			}
		}
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)
		{	// 
			pLabel=pEvent->GetLabel();
			if(pLabel)
				if(pLabel->GetText().IsEmpty())nBlankEvent++;
		}
	}
	bool bSameName=false;
	CString same_name_str;
	DWORD count=0;
	pos=name_map.GetStartPosition();
	while(pos!=NULL)
	{
		name_map.GetNextAssoc(pos,label_str,count);
		if(count>1)
		{
			if(bSameName)same_name_str+=",";
			same_name_str+=" ";
			same_name_str+=label_str;
			bSameName=true;
			nSameName+=count;
		}
	}
	name_map.RemoveAll();
	CString out_str;
	if(bSameName)
	{
		nErrors++;
		out_str.Format("found %d conditions with same name: ",nSameName);
		out_str+=same_name_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	if(nBlankCondition>0)
	{
		nErrors++;
		out_str.Format("found %d condition(s) with blank name",nBlankCondition);
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	if(nBlankEvent>0)
	{
		nErrors++;
		out_str.Format("found %d event(s) with blank name",nBlankEvent);
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	return nErrors;
}

int CNetDesign::TestIsolatedElements()
{
	_AddOutputListString(OUTPUT_STANDARD,"  Isolated conditions...");
	POSITION pos;
	int nErrors=0;
	CString out_str;
	CPNLabel* pLabel;
	CPNElement* pElement;
	CPNCondition* pCondition;
	int nIsolated_conditions=0;
	bool bIsolated_conditions=false;
	CString isolated_conditions_str;
	pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pCondition)
		{	// test for isolated conditions
			if((pCondition->GetPreSet().IsEmpty()) && (pCondition->GetPostSet().IsEmpty()))
			{
				pLabel=pCondition->GetLabel();
				if(pLabel)
				{
					if(bIsolated_conditions)isolated_conditions_str+=",";
					isolated_conditions_str+=" ";
					isolated_conditions_str+=pLabel->GetText();
				}
				bIsolated_conditions=true;
				nIsolated_conditions++;
			}
		}
	}
	if(bIsolated_conditions)
	{
		nErrors++;
		out_str.Format("found %d isolated condition(s): ",nIsolated_conditions);
		out_str+=isolated_conditions_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	_AddOutputListString(OUTPUT_STANDARD,"  Isolated events...");
	CPNEvent* pEvent;
	int nIsolated_events=0;
	bool bIsolated_events=false;
	CString isolated_events_str;
	pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)
		{	// test for isolated events
			if((pEvent->GetPreSet().IsEmpty()) && (pEvent->GetPostSet().IsEmpty()))
			{
				pLabel=pEvent->GetLabel();
				if(pLabel)
				{
					if(bIsolated_events)isolated_events_str+=",";
					isolated_events_str+=" ";
					isolated_events_str+=pLabel->GetText();
				}
				bIsolated_events=true;
				nIsolated_events++;
			}
		}
	}
	if(bIsolated_events)
	{
		nErrors++;
		out_str.Format("found %d isolated event(s): ",nIsolated_events);
		out_str+=isolated_events_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	return nErrors;
}

int CNetDesign::TestEventsConcession(CTypedPtrArray<CObArray,CPNCase*>& case_graph_cases,int nCases)
{
	_AddOutputListString(OUTPUT_STANDARD,"  Events concession...");
	POSITION pos;
	CString out_str;
	CPNCase* pCase;
	CPNLabel* pLabel;
	CPNEvent* pEvent;
	CPNElement* pElement;
	int nNonRealizable_events=0;
	bool bNonRealizable_events=false;
	CString nonrealizable_events_str;
	pos=m_element_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_element_list.GetNext(pos);
		pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pEvent)
		{	// for all events
			bool realizable=false;
			for(int index=0;index<nCases;index++)
			{	// test all cases
				pCase=case_graph_cases.GetAt(index);
				pEvent->UpdateRealizable(pCase->GetCase());
				if(pEvent->IsRealizable())
				{
					realizable=true;
					break;
				}
			}
			if(!realizable)
			{	// 
				pLabel=pEvent->GetLabel();
				if(pLabel)
				{
					if(bNonRealizable_events)nonrealizable_events_str+=",";
					nonrealizable_events_str+=" ";
					nonrealizable_events_str+=pLabel->GetText();
				}
				bNonRealizable_events=true;
				nNonRealizable_events++;
			}
		}
	}
	if(bNonRealizable_events)
	{
		out_str.Format("found %d event(s) without concession: ",nNonRealizable_events);
		out_str+=nonrealizable_events_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	return bNonRealizable_events? 1 : 0;
}

int CNetDesign::TestDistinguishability()
{
	_AddOutputListString(OUTPUT_STANDARD,"  Indistinguishable conditions...");
	POSITION outer_pos,inner_pos;
	int nErrors=0;
	CString out_str;
	CPNLabel* pLabel;
	CPNElement* pElement;
	CPNCondition* pOuterCondition;
	CPNCondition* pInnerCondition;
	int nIndistinguishable_conditions=0;
	bool bIndistinguishable_conditions=false;
	CString indistinguishable_conditions_str;
	outer_pos=m_element_list.GetHeadPosition();
	while(outer_pos!=NULL)
	{	// outer loop
		pElement=m_element_list.GetNext(outer_pos);
		pOuterCondition=dynamic_cast<CPNCondition*>(pElement);
		if(pOuterCondition)
		{	
			if((!pOuterCondition->GetPreSet().IsEmpty()) || (!pOuterCondition->GetPostSet().IsEmpty()))
			{
				inner_pos=outer_pos;
//				inner_pos=m_element_list.GetHeadPosition();
				while(inner_pos!=NULL)
				{	// inner loop
					pElement=m_element_list.GetNext(inner_pos);
					pInnerCondition=dynamic_cast<CPNCondition*>(pElement);
					if(pInnerCondition)
					{	
//						if(pInnerCondition!=pOuterCondition)
						{
							if((pOuterCondition->GetPreSet()==pInnerCondition->GetPreSet()) && (pOuterCondition->GetPostSet()==pInnerCondition->GetPostSet()))
							{	// indistinguishable conditions
								pLabel=pOuterCondition->GetLabel();
								if(pLabel)
								{
									if(bIndistinguishable_conditions)indistinguishable_conditions_str+=",";
									indistinguishable_conditions_str+=" ";
									indistinguishable_conditions_str+=pLabel->GetText();
								}
								pLabel=pInnerCondition->GetLabel();
								if(pLabel)
								{
									indistinguishable_conditions_str+=" and ";
									indistinguishable_conditions_str+=pLabel->GetText();
								}
								bIndistinguishable_conditions=true;
								nIndistinguishable_conditions+=2;
							}
						}
					}
				}
			}
		}
	}
	if(bIndistinguishable_conditions)
	{
		nErrors++;
		out_str.Format("found %d indistinguishable conditions: ",nIndistinguishable_conditions);
		out_str+=indistinguishable_conditions_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	_AddOutputListString(OUTPUT_STANDARD,"  indistinguishable events...");
	CPNEvent* pOuterEvent;
	CPNEvent* pInnerEvent;
	int nIndistinguishable_events=0;
	bool bIndistinguishable_events=false;
	CString indistinguishable_events_str;
	outer_pos=m_element_list.GetHeadPosition();
	while(outer_pos!=NULL)
	{	// outer loop
		pElement=m_element_list.GetNext(outer_pos);
		pOuterEvent=dynamic_cast<CPNEvent*>(pElement);
		if(pOuterEvent)
		{	
			if((!pOuterEvent->GetPreSet().IsEmpty()) || (!pOuterEvent->GetPostSet().IsEmpty()))
			{
				inner_pos=outer_pos;
//				inner_pos=m_element_list.GetHeadPosition();
				while(inner_pos!=NULL)
				{	// inner loop
					pElement=m_element_list.GetNext(inner_pos);
					pInnerEvent=dynamic_cast<CPNEvent*>(pElement);
					if(pInnerEvent)
					{	
//						if(pInnerEvent!=pOuterEvent)
						{
							if((pOuterEvent->GetPreSet()==pInnerEvent->GetPreSet()) && (pOuterEvent->GetPostSet()==pInnerEvent->GetPostSet()))
							{	// indistinguishable events
								pLabel=pOuterEvent->GetLabel();
								if(pLabel)
								{
									if(bIndistinguishable_events)indistinguishable_events_str+=",";
									indistinguishable_events_str+=" ";
									indistinguishable_events_str+=pLabel->GetText();
								}
								pLabel=pInnerEvent->GetLabel();
								if(pLabel)
								{
									indistinguishable_events_str+=" and ";
									indistinguishable_events_str+=pLabel->GetText();
								}
								bIndistinguishable_events=true;
								nIndistinguishable_events+=2;
							}
						}
					}
				}
			}
		}
	}
	if(bIndistinguishable_events)
	{
		nErrors++;
		out_str.Format("found %d indistinguishable events: ",nIndistinguishable_events);
		out_str+=indistinguishable_events_str;
		_WriteOutError(OUTPUT_STANDARD,out_str);
	}
	return nErrors;
}

int CNetDesign::TestEmptyNet()
{
	if((m_nEvents==0) && (m_nConditions==0))
	{	// empty net
		_WriteOutError(OUTPUT_STANDARD,"Empty net");
		return 1;
	}
	return 0;
}

void CNetDesign::ExportToXml(CString& filename)
{
	CStdioFile file;
	if(file.Open(filename,CFile::modeCreate|CFile::modeWrite|CFile::typeText))	
	{
		POSITION pos;
		CPNEdge* pEdge;
		CPNEvent* pEvent;
		CPNLabel* pLabel;
		CPNElement* pElement;
		CPNCondition* pCondition;
		file.WriteString("<?xml version=\"1.0\" encoding=\"windows-1250\"?>\n");
		file.WriteString("<netdesign>\n");
		pos=m_element_list.GetHeadPosition();
		while(pos!=NULL)
		{
			pElement=m_element_list.GetNext(pos);
			pCondition=dynamic_cast<CPNCondition*>(pElement);
			if(pCondition)
			{
				file.WriteString("  <condition token=\"");
				if(pCondition->GetToken())
					file.WriteString("1");
				else
					file.WriteString("0");
				file.WriteString("\">");
				pLabel=pCondition->GetLabel();
				if(pLabel)file.WriteString(pLabel->GetText());
				file.WriteString("</condition>\n");
			}
		}
		pos=m_element_list.GetHeadPosition();
		while(pos!=NULL)
		{
			pElement=m_element_list.GetNext(pos);
			pEvent=dynamic_cast<CPNEvent*>(pElement);
			if(pEvent)
			{
				file.WriteString("  <event>");
				pLabel=pEvent->GetLabel();
				if(pLabel)file.WriteString(pLabel->GetText());
				file.WriteString("</event>\n");
			}
		}
		pos=m_element_list.GetHeadPosition();
		while(pos!=NULL)
		{
			pElement=m_element_list.GetNext(pos);
			pEdge=dynamic_cast<CPNEdge*>(pElement);
			if(pEdge)
			{
				file.WriteString("  <edge>\n");
				file.WriteString("    <from>");
				pLabel=GetSafeLabel(pEdge->GetFrom());
				if(pLabel)file.WriteString(pLabel->GetText());
				file.WriteString("</from>\n");
				file.WriteString("    <to>");
				pLabel=GetSafeLabel(pEdge->GetTo());
				if(pLabel)file.WriteString(pLabel->GetText());
				file.WriteString("</to>\n");
				file.WriteString("  </edge>\n");
			}
		}
		file.WriteString("</netdesign>\n");
		file.Close();
	}
}

CPNLabel* CNetDesign::GetSafeLabel(CPNElement* pElement)
{
	CPNEvent* pEvent;
	CPNCondition* pCondition;
	pEvent=dynamic_cast<CPNEvent*>(pElement);
	if(pEvent)return pEvent->GetLabel();
	pCondition=dynamic_cast<CPNCondition*>(pElement);
	if(pCondition)return pCondition->GetLabel();
	return NULL;
}
