// GenAlg.cpp - implementation for CGenAlg, CChromozome, CEdgeInfo
//

#include "stdafx.h"
#include "GenAlg.h"
#include <time.h>
#include <math.h>

////////////////////////////////////////////////////////////////////////////
// CChromozome
CChromozome::CChromozome()
{
	m_nGenes=0;
	m_grid_size=0;
	m_grid_span=GRID_SPAN;
	m_modified=true;
	m_nFitness=0;
	m_genes=NULL;
	m_occupation=NULL;
}

CChromozome::CChromozome(CChromozome& chromozome)
{	// copy constructor
	m_nFitness=chromozome.m_nFitness;
	m_modified=chromozome.m_modified;
	m_nGenes=chromozome.m_nGenes;
	m_grid_size=chromozome.m_grid_size;
	m_grid_span=chromozome.m_grid_span;
	m_grid_diagonal=chromozome.m_grid_diagonal;
	m_genes=new DWORD[m_nGenes*2];
	m_occupation=new BYTE[m_grid_size*m_grid_size];
	::CopyMemory(m_genes,chromozome.m_genes,m_nGenes*2*sizeof(DWORD));
	::CopyMemory(m_occupation,chromozome.m_occupation,m_grid_size*m_grid_size);
}

CChromozome::~CChromozome()
{	// free memory
	if(m_genes!=NULL)delete[] m_genes;
	m_genes=NULL;
	if(m_occupation!=NULL)delete[] m_occupation;
	m_occupation=NULL;
}

void CChromozome::Init(int nGenes,int grid_size)
{	// initialization of chromosome
	m_modified=true;
	m_nFitness=0;
	m_nGenes=nGenes;
	m_grid_size=grid_size;
	if(m_genes!=NULL)delete[] m_genes;
	m_genes=new DWORD[m_nGenes*2];
	if(m_occupation!=NULL)delete[] m_occupation;
	m_occupation=new BYTE[m_grid_size*m_grid_size];
	::ZeroMemory(m_occupation,m_grid_size*m_grid_size);
	m_grid_diagonal=int(sqrt(2*m_grid_size*m_grid_size*GRID_SPAN*GRID_SPAN));
}

void CChromozome::Generate()
{	// ramdom place nodes
	m_modified=true;
	int side_x,side_y;
	::ZeroMemory(m_occupation,m_grid_size*m_grid_size);
	for(int gene_index=0;gene_index<m_nGenes;gene_index++)
	{
		do
		{	// find free place
			side_x=rand()%m_grid_size;
			side_y=rand()%m_grid_size;
		}while(m_occupation[side_y*m_grid_size+side_x]!=0);
		m_occupation[side_y*m_grid_size+side_x]=1;
		m_genes[gene_index*2]=(side_x+1)*m_grid_span;
		m_genes[gene_index*2+1]=(side_y+1)*m_grid_span;
	}
}

/*
void CChromozome::Apply(const CTypedPtrArray<CObArray,CPNCase*>& cases,int nCases)
{	// pouze pro DEBUG
	CPoint point;
	CPNCase* pCase;
	::ZeroMemory(m_occupation,m_grid_size*m_grid_size);
	for(int case_index=0;case_index<nCases;case_index++)
	{
		pCase=cases.GetAt(case_index);
		point=pCase->GetPosition();
		point.x=point.x/m_grid_span;
		if(point.x!=0)point.x-=1;
		point.y=point.y/m_grid_span;
		if(point.y!=0)point.y-=1;
		m_occupation[point.y*m_grid_size+point.x]=1;
		m_genes[case_index*2]=(point.x+1)*m_grid_span;
		m_genes[case_index*2+1]=(point.y+1)*m_grid_span;
	}
	m_modified=true;
}
*/

void CChromozome::Mutate_Swap()
{	// switch two nodes
	m_modified=true;
	int pos1=rand()%m_nGenes;
	int pos2=rand()%m_nGenes;
	DWORD tmp_x=m_genes[pos2*2];
	DWORD tmp_y=m_genes[pos2*2+1];
	m_genes[pos2*2]=m_genes[pos1*2];
	m_genes[pos2*2+1]=m_genes[pos1*2+1];
	m_genes[pos1*2]=tmp_x;
	m_genes[pos1*2+1]=tmp_y;
}

void CChromozome::Mutate_Move()
{	// move random node to random place
	m_modified=true;
	int pos=rand()%m_nGenes;
	int side_x,side_y;
	do
	{	// find free place
		side_x=rand()%m_grid_size;
		side_y=rand()%m_grid_size;
	}while(m_occupation[side_y*m_grid_size+side_x]!=0);
	// move to new position
	m_occupation[side_y*m_grid_size+side_x]=1;
	// free old position
	DWORD tmp_x=m_genes[pos*2];
	DWORD tmp_y=m_genes[pos*2+1];
	tmp_x=tmp_x/m_grid_span;
	tmp_x-=1;
	tmp_y=tmp_y/m_grid_span;
	tmp_y-=1;
	m_occupation[tmp_y*m_grid_size+tmp_x]=0;
	// set new coordinates
	m_genes[pos*2]=(side_x+1)*m_grid_span;
	m_genes[pos*2+1]=(side_y+1)*m_grid_span;
}

void CChromozome::Mutate_SmallMove()
{	// move random node by one
	int pos=rand()%m_nGenes;
	DWORD pos_x=m_genes[pos*2];
	DWORD pos_y=m_genes[pos*2+1];
	pos_x=pos_x/m_grid_span;
	pos_x-=1;
	pos_y=pos_y/m_grid_span;
	pos_y-=1;
	m_modified=true;
	// generate move
	int new_pos_x=pos_x+(rand()%3)-1;
	int new_pos_y=pos_y+(rand()%3)-1;
	// correct side points
	if(new_pos_x<0)new_pos_x=0;
	if(new_pos_y<0)new_pos_y=0;
	if(new_pos_x>=m_grid_size)new_pos_x=m_grid_size-1;
	if(new_pos_y>=m_grid_size)new_pos_y=m_grid_size-1;
	if(m_occupation[new_pos_y*m_grid_size+new_pos_x]==0)
	{	// new position is free
		m_occupation[pos_y*m_grid_size+pos_x]=0;
		m_occupation[new_pos_y*m_grid_size+new_pos_x]=1;
		m_genes[pos*2]=(new_pos_x+1)*m_grid_span;
		m_genes[pos*2+1]=(new_pos_y+1)*m_grid_span;
	}
}

void CChromozome::Cross(CChromozome& parent_2,CChromozome& child_1,CChromozome& child_2)
{	// *this=parent_1
	// cross by order
	int pos_1=rand()%(m_nGenes+1);
	int pos_2=rand()%(m_nGenes+1);
	if(pos_1>pos_2)
	{	// normalization
		int tmp=pos_1;
		pos_1=pos_2;
		pos_2=tmp;
	}
	int index;
	// create child_1
	if(child_1.m_genes==NULL)child_1.Init(m_nGenes,m_grid_size);
	// inherit directly part between cross points
	::CopyMemory(child_1.m_genes+pos_1*2,m_genes+pos_1*2,(pos_2-pos_1)*2*sizeof(DWORD));
	::ZeroMemory(child_1.m_occupation,m_grid_size*m_grid_size);
	for(index=pos_1;index<pos_2;index++)
	{	// reconstruct occupation
		DWORD tmp_x=child_1.m_genes[index*2];
		DWORD tmp_y=child_1.m_genes[index*2+1];
		tmp_x=(tmp_x/m_grid_span)-1;
		tmp_y=(tmp_y/m_grid_span)-1;
		child_1.m_occupation[tmp_y*m_grid_size+tmp_x]=1;
	}
	DWORD parent_x;
	DWORD parent_y;
	int parent_pos=pos_2;
	if(parent_pos>=m_nGenes)parent_pos=0;
	int child_pos=pos_2;
	if(child_pos>=m_nGenes)child_pos=0;
	for(index=0;index<(m_nGenes-(pos_2-pos_1));index++)
	{	// copy rest of gens from other parent
		do
		{
			parent_x=parent_2.m_genes[parent_pos*2];
			parent_y=parent_2.m_genes[parent_pos*2+1];
			parent_x=(parent_x/m_grid_span)-1;
			parent_y=(parent_y/m_grid_span)-1;
			parent_pos++;
			if(parent_pos>=m_nGenes)parent_pos=0;
		}while(child_1.m_occupation[parent_y*m_grid_size+parent_x]==1);
		child_1.m_genes[child_pos*2]=(parent_x+1)*m_grid_span;
		child_1.m_genes[child_pos*2+1]=(parent_y+1)*m_grid_span;
		child_1.m_occupation[parent_y*m_grid_size+parent_x]=1;
		child_pos++;
		if(child_pos>=m_nGenes)child_pos=0;
	}
	child_1.m_modified=true;
	// create child_2
	if(child_2.m_genes==NULL)child_2.Init(m_nGenes,m_grid_size);
	// inherit directly part between cross points
	::CopyMemory(child_2.m_genes+pos_1*2,parent_2.m_genes+pos_1*2,(pos_2-pos_1)*2*sizeof(DWORD));
	::ZeroMemory(child_2.m_occupation,m_grid_size*m_grid_size);
	for(index=pos_1;index<pos_2;index++)
	{	// reconstruct occupation
		DWORD tmp_x=child_2.m_genes[index*2];
		DWORD tmp_y=child_2.m_genes[index*2+1];
		tmp_x=(tmp_x/m_grid_span)-1;
		tmp_y=(tmp_y/m_grid_span)-1;
		child_2.m_occupation[tmp_y*m_grid_size+tmp_x]=1;
	}
	parent_pos=pos_2;
	if(parent_pos>=m_nGenes)parent_pos=0;
	child_pos=pos_2;
	if(child_pos>=m_nGenes)child_pos=0;
	for(index=0;index<(m_nGenes-(pos_2-pos_1));index++)
	{	// copy rest of gens from other parent
		do
		{
			parent_x=m_genes[parent_pos*2];
			parent_y=m_genes[parent_pos*2+1];
			parent_x=(parent_x/m_grid_span)-1;
			parent_y=(parent_y/m_grid_span)-1;
			parent_pos++;
			if(parent_pos>=m_nGenes)parent_pos=0;
		}while(child_2.m_occupation[parent_y*m_grid_size+parent_x]==1);
		child_2.m_genes[child_pos*2]=(parent_x+1)*m_grid_span;
		child_2.m_genes[child_pos*2+1]=(parent_y+1)*m_grid_span;
		child_2.m_occupation[parent_y*m_grid_size+parent_x]=1;
		child_pos++;
		if(child_pos>=m_nGenes)child_pos=0;
	}
	child_2.m_modified=true;
}

DWORD CChromozome::ComputeFitness(const CArray<CEdgeInfo,CEdgeInfo&>& edges,int nEdges,int& intersections,int weight_intersections,int weight_edgelength,int weight_objdistance_min,int weight_objdistance_max)
{	// evaluate chromosome
	if(!m_modified)return m_nFitness;
	m_modified=false;
	// compute edges intersections
	int nEdgeIntersections=0;
	if(weight_intersections!=0)
	{
		CEdgeInfo edge_outer;
		for(int index_outer=0;index_outer<nEdges;index_outer++)
		{	// 
			edge_outer=edges.GetAt(index_outer);
			for(int index_inner=index_outer+1;index_inner<nEdges;index_inner++)
			{
				if(edge_outer.Intersect(edges.GetAt(index_inner)))nEdgeIntersections++;
			}
		}
	}
	// normalization
	double intersect_fitness=(nEdges*nEdges-nEdgeIntersections)/(double(nEdges*nEdges));
	intersections=nEdgeIntersections;
	// compute edge length
	int nEdgesLen=0;
	if(weight_edgelength!=0)
	{
		CEdgeInfo edgeinfo;
		for(int index_edge=0;index_edge<nEdges;index_edge++)
		{	// 
			edgeinfo=edges.GetAt(index_edge);
			nEdgesLen+=edgeinfo.GetLength();
		}
	}
	// normalization
	double edges_fitness=(m_grid_diagonal*nEdges-nEdgesLen)/(double(m_grid_diagonal*nEdges));
	// compute distance between nodes
	int nObjectDst_min=0;
	int nObjectDst_max=0;
	if((weight_objdistance_max!=0) || (weight_objdistance_min!=0))
	{
		for(int index_object_outer=0;index_object_outer<m_nGenes;index_object_outer++)
		{	// 
			double max_dst=0;
			double min_dst=m_grid_diagonal;
			DWORD obj_x_outer,obj_y_outer;
			obj_x_outer=m_genes[index_object_outer*2];
			obj_y_outer=m_genes[index_object_outer*2+1];
			for(int index_object_inner=0;index_object_inner<m_nGenes;index_object_inner++)
			{	// 
				if(index_object_inner!=index_object_outer)
				{
					DWORD obj_x_inner,obj_y_inner;
					obj_x_inner=m_genes[index_object_inner*2];
					obj_y_inner=m_genes[index_object_inner*2+1];
					double dst=sqrt((obj_x_inner-obj_x_outer)*(obj_x_inner-obj_x_outer)+(obj_y_inner-obj_y_outer)*(obj_y_inner-obj_y_outer));
					if(dst>max_dst)max_dst=dst;
					if(dst<min_dst)min_dst=dst;
				}
			}
			nObjectDst_min+=int(max_dst); // minimalizace vzdalenosti mezi objekty minimalizaci maximalni vzdalenosti
			nObjectDst_max+=int(min_dst); // maximalizace vzdalenosti mezi objekty maximalizaci minimalni vzdalenosti
		}
	}
	// normalization
	double object_min_fitness=(m_grid_diagonal*m_nGenes-nObjectDst_min)/(double(m_grid_diagonal*m_nGenes));
	double object_max_fitness=(nObjectDst_max)/(double(m_grid_diagonal*m_nGenes));
	m_nFitness=intersect_fitness*1000000*weight_intersections+edges_fitness*1000000*weight_edgelength+object_min_fitness*1000000*weight_objdistance_min+object_max_fitness*1000000*weight_objdistance_max;
	ASSERT(m_nFitness>=0);
	return m_nFitness;
}

CChromozome& CChromozome::operator=(const CChromozome& chromozome)
{	// copy chromosome
	m_modified=chromozome.m_modified;
	m_nGenes=chromozome.m_nGenes;
	m_grid_size=chromozome.m_grid_size;
	m_grid_span=chromozome.m_grid_span;
	m_grid_diagonal=chromozome.m_grid_diagonal;
	if(m_genes==NULL)m_genes=new DWORD[m_nGenes*2];
	if(m_occupation==NULL)m_occupation=new BYTE[m_grid_size*m_grid_size];
	::CopyMemory(m_genes,chromozome.m_genes,m_nGenes*2*sizeof(DWORD));
	::CopyMemory(m_occupation,chromozome.m_occupation,m_grid_size*m_grid_size);
	return *this;
}

////////////////////////////////////////////////////////////////////////////
// CEdgeInfo
int CEdgeInfo::GetLength()
{	// return length of this edge
	int square=(m_x_to-m_x)*(m_x_to-m_x)+(m_y_to-m_y)*(m_y_to-m_y);
	if(square!=0)
		return int(sqrt(square));
	else
		return 0;
}

bool CEdgeInfo::Intersect(CEdgeInfo& edge_info,bool recurse/*=false*/)
{	// test for intersection between this edge and edge in edge_info
	CRect rect;
	if(m_rect.IsRectEmpty() || edge_info.m_rect.IsRectEmpty() || rect.IntersectRect(&m_rect,&edge_info.m_rect))
	{
		double cit=m_u*(edge_info.m_y-m_y)+m_v*(m_x-edge_info.m_x);
		double jm=m_v*edge_info.m_u-m_u*edge_info.m_v;
		if(jm==0.0)
		{	 // lines are collinear
			if(cit==0.0)
				return true; // lines are overlaping
			else
				return false; // intersection in the infinity
		}
		double t=cit/jm;
		if(((t==0.0) || (t==1.0)) && (!recurse))
		{
			CEdgeInfo info;
			info=*this;
			return edge_info.Intersect(info,true);
		}
		return ((t>0.0) && (t<1.0));
	}
	return false;
}

void CEdgeInfo::ApplyChromozome(CChromozome& chromozome)
{	// apply node coordinates to edges
	m_x=chromozome.GetGene_x(m_from);
	m_y=chromozome.GetGene_y(m_from);
	m_x_to=chromozome.GetGene_x(m_to);
	m_y_to=chromozome.GetGene_y(m_to);
	m_u=m_x_to-m_x;
	m_v=m_y_to-m_y;
	m_rect.SetRect(m_x,m_y,m_x_to,m_y_to);
	m_rect.NormalizeRect();
}

////////////////////////////////////////////////////////////////////////////
// CGenAlg
CGenAlg::CGenAlg()
{	// initialization
	m_prob_mutate_move=45;
	m_prob_mutate_swap=35;
	m_prob_mutate_smallmove=25;
	m_prob_cross=10;
	m_nPopulation=3;
	m_weight_intersections=5;
	m_weight_edgelength=1;
	m_weight_objdistance_min=0;
	m_weight_objdistance_max=0;

	m_nEdges=0;
	m_nGrid_size=0;
	m_nChromozome_len=0;
	m_nGenerations=0;
	m_nIntersections=0;
	m_best_chromozome=0;
	m_best_fitness=0;
}

int CGenAlg::FindCaseIndex(const CTypedPtrArray<CObArray,CPNCase*>& cases,int nCases,const CPNElement* pElement)
{	// find index of case in chromosome genes
	for(int case_index=0;case_index<nCases;case_index++)
	{
		if(cases.GetAt(case_index)==pElement)return case_index;
	}
	ASSERT(FALSE);
	return 0;
}

void CGenAlg::SetInfo(int grid_size)
{
	m_nGrid_size=grid_size;
}

void CGenAlg::Init(const CTypedPtrArray<CObArray,CPNStep*>& steps,int nSteps,const CTypedPtrArray<CObArray,CPNCase*>& cases,int nCases)
{	// initialization of chromosome
	srand((unsigned)time(NULL));

	CPNStep* pStep;
	CPNElement* pElement;
	CEdgeInfo edge_info;
	m_nEdges=nSteps;
	m_edges.RemoveAll();
	m_edges.SetSize(m_nEdges,1);
	for(int edge_index=0;edge_index<m_nEdges;edge_index++)
	{	// priprava hran do lepsiho formatu
		pStep=steps.GetAt(edge_index);
		pElement=pStep->GetElementFrom();
		edge_info.m_from=FindCaseIndex(cases,nCases,pElement);
		pElement=pStep->GetElementTo();
		edge_info.m_to=FindCaseIndex(cases,nCases,pElement);
		m_edges.SetAt(edge_index,edge_info);
	}
	m_nChromozome_len=nCases;
}

void CGenAlg::PreparePopulation()
{	// prepare population
	m_population.RemoveAll();
	m_population.SetSize(m_nPopulation*2,1);
	m_nGenerations=0;
	CChromozome chromozome;
	chromozome.Init(m_nChromozome_len,m_nGrid_size);
	for(int index=0;index<m_nPopulation;index++)
	{	// generate random population
		chromozome.Generate();
		m_population.SetAt(index,chromozome);
	}
	m_nIntersections=0;
	m_population_orig=0;
	m_population_new=m_nPopulation;
	EvaluatePopulation();
}

CChromozome CGenAlg::TournamentSelect()
{	// select chromosome from population
	int ch_1=m_population_orig+(rand()%m_nPopulation);
	int ch_2=m_population_orig+(rand()%m_nPopulation);
	int ch_3=m_population_orig+(rand()%m_nPopulation);
	int fit_1=m_population.GetAt(ch_1).m_nFitness;
	int fit_2=m_population.GetAt(ch_2).m_nFitness;
	int fit_3=m_population.GetAt(ch_3).m_nFitness;
	if(fit_1>fit_2)
	{
		if(fit_1>fit_3)
			return m_population.GetAt(ch_1);
		else
			return m_population.GetAt(ch_3);
	}
	else
	{
		if(fit_2>fit_3)
			return m_population.GetAt(ch_2);
		else
			return m_population.GetAt(ch_3);
	}
}

bool CGenAlg::RandomPass(int value)
{	// 
	if(value==0)return false;
	return ((rand()%100)<value);
}

int CGenAlg::StepGeneration()
{	// compute next generation
	CChromozome chromozome_1;
	CChromozome chromozome_2;
	int index;
	for(index=0;index<(m_nPopulation/2);index++)
	{	// 
		chromozome_1=TournamentSelect();
		chromozome_2=TournamentSelect();
		if(RandomPass(m_prob_cross))
		{	// cross two chromosomes
			CChromozome child_1;
			CChromozome child_2;
			chromozome_1.Cross(chromozome_2,child_1,child_2);
			chromozome_1=child_1;
			chromozome_2=child_2;
		}
		// mutate
		if(RandomPass(m_prob_mutate_move))chromozome_1.Mutate_Move();
		if(RandomPass(m_prob_mutate_move))chromozome_2.Mutate_Move();
		if(RandomPass(m_prob_mutate_swap))chromozome_1.Mutate_Swap();
		if(RandomPass(m_prob_mutate_swap))chromozome_2.Mutate_Swap();
		if(RandomPass(m_prob_mutate_smallmove))chromozome_1.Mutate_SmallMove();
		if(RandomPass(m_prob_mutate_smallmove))chromozome_2.Mutate_SmallMove();
		m_population.SetAt(m_population_new+index*2,chromozome_1);
		m_population.SetAt(m_population_new+index*2+1,chromozome_2);
	}
	// elitizm
	chromozome_1=m_population.GetAt(m_best_chromozome);
	m_population.SetAt(m_population_new+index*2,chromozome_1);
	// switch populations
	int temp=m_population_orig;
	m_population_orig=m_population_new;
	m_population_new=temp;
	EvaluatePopulation();
	m_nGenerations++;
	return m_nGenerations;
}

void CGenAlg::EvaluatePopulation()
{	// evaluate all chromosomes in population
	m_best_chromozome=0;
	DWORD fitness=0;
	m_best_fitness=0;
	m_nIntersections=0;
	int nIntersections;
	CEdgeInfo edge_info;
	CChromozome chromozome;
	for(int index_pop=m_population_orig;index_pop<(m_nPopulation+m_population_orig);index_pop++)
	{
		chromozome=m_population.GetAt(index_pop);
		if(chromozome.IsModified())
		{
			for(int index_edge=0;index_edge<m_nEdges;index_edge++)
			{	// apply node coordinates to edges
				edge_info=m_edges.GetAt(index_edge);
				edge_info.ApplyChromozome(chromozome);
				m_edges.SetAt(index_edge,edge_info);
			}
		}
		fitness=chromozome.ComputeFitness(m_edges,m_nEdges,nIntersections,m_weight_intersections,m_weight_edgelength,m_weight_objdistance_min,m_weight_objdistance_max);
		if(fitness>m_best_fitness)
		{	// save best chromosome index
			m_best_chromozome=index_pop;
			m_best_fitness=fitness;
			m_nIntersections=nIntersections;
		}
	}
}

void CGenAlg::ApplyBestChromozome(CTypedPtrArray<CObArray,CPNCase*>& cases,int nCases)
{	// set node coordinates of best chromosome to case graph
	CPoint point;
	CPNCase* pCase;
	CChromozome chromozome;
	chromozome=m_population.GetAt(m_best_chromozome);
	for(int case_index=0;case_index<nCases;case_index++)
	{
		pCase=cases.GetAt(case_index);
		point.x=chromozome.GetGene_x(case_index);
		point.y=chromozome.GetGene_y(case_index);
		pCase->MoveTo(point);
	}
}

void CGenAlg::DeleteContents()
{	// clear and prepare
	m_nEdges=0;
	m_edges.RemoveAll();
	m_population.RemoveAll();
	m_nGenerations=0;
	m_nIntersections=0;
	m_best_chromozome=0;
	m_best_fitness=0;
}
