// CaseGraph.cpp - implementation for CCaseGraph
//

#include "stdafx.h"
#include "CaseGraph.h"
#include "CESim.h"
#include "MainFrm.h"

////////////////////////////////////////////////////////////////////////////
// CCaseGraph
IMPLEMENT_SERIAL(CCaseGraph,CObject,1|VERSIONABLE_SCHEMA)

CCaseGraph::CStepInfo& CCaseGraph::CStepInfo::operator = (CCaseGraph::CStepInfo& stepinfo)
{
	m_step_set=stepinfo.m_step_set;
	m_step_preset=stepinfo.m_step_preset;
	m_step_postset=stepinfo.m_step_postset;
	return *this;
}

CCaseGraph::CFRCasesClass& CCaseGraph::CFRCasesClass::operator = (CCaseGraph::CFRCasesClass& frcases)
{
	m_frcases_set=frcases.m_frcases_set;
	return *this;
}

CCaseGraph::CCaseGraph()
{
	m_graph_placed=FALSE;
	m_smooth_initialized=false;
	m_window_size.cx=100;
	m_window_size.cy=100;
	m_case_browse_index=0;
	m_nOldCases=0;
	m_nCases=0;
	m_nSteps=0;
	m_nLabels=0;
	m_nEvents=0;
	m_grid_size=0;
	m_min_grid_size=0;
	m_nGenSteps=0;
	m_nRlzEvents=0;
	m_max_step_level=0;
	m_case_browse_index=0;
//	m_case_graph_cases.SetSize(150,100);
//	m_case_graph_steps.SetSize(150,100);
}

void CCaseGraph::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CObject::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_nCases;
		ar << m_nSteps;
		ar << m_nLabels;
		ar << m_nEvents;
		ar << m_grid_size;
		ar << m_min_grid_size;
		ar << m_window_size;
		ar << m_graph_placed;
		ar << m_nOldCases;
	}
	else
	{
		ar >> m_nCases;
		ar >> m_nSteps;
		ar >> m_nLabels;
		ar >> m_nEvents;
		ar >> m_grid_size;
		ar >> m_min_grid_size;
		ar >> m_window_size;
		ar >> m_graph_placed;
		ar >> m_nOldCases;
	}
	m_oldcases_positions.Serialize(ar);
	m_case_graph_cases.Serialize(ar);
	m_case_graph_steps.Serialize(ar);
	m_case_graph_labels.Serialize(ar);
	m_net_events.Serialize(ar);
	m_elements_list.Serialize(ar);
}

void CCaseGraph::SetWindowSize(CSize window_size)
{
	CSize min_window_size;
	// compute minimal window size
	CRect window_rect,tmp_rect;
	window_rect.SetRectEmpty();
	tmp_rect.SetRectEmpty();
	CPNElement* pElement;
	POSITION pos=m_elements_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_elements_list.GetNext(pos);
		window_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=window_rect;
	}
	int index;
	for(index=0;index<m_nCases;index++)
	{
		pElement=m_case_graph_cases.GetAt(index);
		window_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=window_rect;
	}
	for(index=0;index<m_nSteps;index++)
	{
		pElement=m_case_graph_steps.GetAt(index);
		window_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=window_rect;
	}
	for(index=0;index<m_nLabels;index++)
	{
		pElement=m_case_graph_labels.GetAt(index);
		window_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=window_rect;
	}
	min_window_size.cx=window_rect.right;
	min_window_size.cy=window_rect.bottom;
	if(min_window_size.cx<100)min_window_size.cx=100;
	if(min_window_size.cy<100)min_window_size.cy=100;
	if(min_window_size.cx<(m_grid_size+1)*CASE_GRID_SPAN)min_window_size.cx=(m_grid_size+1)*CASE_GRID_SPAN;
	if(min_window_size.cy<(m_grid_size+1)*CASE_GRID_SPAN)min_window_size.cy=(m_grid_size+1)*CASE_GRID_SPAN;
	// set window size
	if(window_size.cx<min_window_size.cx)window_size.cx=min_window_size.cx;
	if(window_size.cy<min_window_size.cy)window_size.cy=min_window_size.cy;
	m_window_size=window_size;
}

void CCaseGraph::DeleteContents(bool newdoc/*=true*/)
{
	m_window_size.cx=100;
	m_window_size.cy=100;
	int index;
	if(!newdoc)
	{	// just prepare for new case graph
		if(m_graph_placed)
		{
			m_nOldCases=m_nCases;
			m_oldcases_positions.SetSize(m_nOldCases);
			for(index=0;index<m_nCases;index++)
			{	// save positions of cases
				m_oldcases_positions.SetAt(index,m_case_graph_cases.GetAt(index)->GetPosition());
			}
		}
	}
	else
	{
		m_nOldCases=0;
		m_oldcases_positions.RemoveAll();
	}
	m_graph_placed=FALSE;
	for(index=0;index<m_nCases;index++)delete m_case_graph_cases.GetAt(index);
	for(index=0;index<m_nSteps;index++)delete m_case_graph_steps.GetAt(index);
	for(index=0;index<m_nLabels;index++)delete m_case_graph_labels.GetAt(index);
	m_case_graph_cases.RemoveAll();
	m_case_graph_steps.RemoveAll();
	m_case_graph_labels.RemoveAll();
	m_net_events.RemoveAll();
	m_nEvents=m_nCases=m_nSteps=m_nLabels=0;
	POSITION pos=m_elements_list.GetHeadPosition();
	while(pos!=NULL)delete m_elements_list.GetNext(pos);
	m_elements_list.RemoveAll();
	m_smooth_initialized=false;
	m_dlg_smooth.DeleteContents();
}

void CCaseGraph::Draw(CDC* pDC)
{
	CPNElement* pelement;
	POSITION pos=m_elements_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pelement=m_elements_list.GetNext(pos);
		pelement->Draw(pDC);
	}
	CPNCase* pCase;
	CPNStep* pStep;
	CPNLabel* pLabel;
	for(int i=0;i<m_nCases;i++)
	{
		pCase=m_case_graph_cases.GetAt(i);
		pCase->Draw(pDC);
	}
	for(i=0;i<m_nSteps;i++)
	{
		pStep=m_case_graph_steps.GetAt(i);
		pStep->Draw(pDC);
	}
	for(i=0;i<m_nLabels;i++)
	{
		pLabel=m_case_graph_labels.GetAt(i);
		pLabel->Draw(pDC);
	}
}

void CCaseGraph::PrepareCaseGraph(const CPNElementList& element_list)
{	// prepare array of events for faster computing
	m_nEvents=0;
	CPNEvent* pEvent;
	POSITION pos=element_list.GetHeadPosition();
	while(pos!=NULL)
	{	// count events
		pEvent=dynamic_cast<CPNEvent*>(element_list.GetNext(pos));
		if(pEvent)m_nEvents++;
	}
	m_net_events.SetSize(m_nEvents,10);
	m_nEvents=0;
	pos=element_list.GetHeadPosition();
	while(pos!=NULL)
	{	// fill array
		pEvent=dynamic_cast<CPNEvent*>(element_list.GetNext(pos));
		if(pEvent)
		{
			m_net_events.SetAtGrow(m_nEvents,pEvent);
			m_nEvents++;
		}
	}
}

void CCaseGraph::ComputeCaseGraph(const CSetConditions& initial_case,const CPNElementList& element_list)
{
	PrepareCaseGraph(element_list);

	m_smooth_initialized=false;
	// reset
	m_nSteps=0;
	m_nCases=0;
	m_steps.SetSize(15,10);
	m_realizable_events.SetSize(15,10);

	// insert initial case
	CPNCase* pCase;
	pCase=new CPNCase(initial_case);
	m_case_graph_cases.SetAtGrow(0,pCase);
	m_nCases++;

	// compute case graph
	CPNStep* pStep;
	CPNCase* pFromCase;
	CStepInfo step_info;
	CSetConditions act_case,new_case;
	int case_index=0;
	do
	{	// prochazi existujici pripady
		pFromCase=m_case_graph_cases.GetAt(case_index);
		case_index++;
		act_case=pFromCase->GetCase();
		for(int direction=DIRECTION_FWD;direction<=DIRECTION_BCKWD;direction++)
		{	// pro dany pripad zkusi proveditelnost dopredu i dozadu
			ComputeRealizableEvents(act_case,direction);
			GenerateAllSteps();
			int step_index=0;
			while(GetNextStep(step_index,step_info))
			{	// prochazi vsechny kroky z daneho pripadu
				new_case=act_case;
				RealizeStep(new_case,step_info,direction);
				pCase=FindCase(new_case);
				if(pCase==NULL)
				{	// insert new case
					pCase=new CPNCase(new_case);
					m_case_graph_cases.SetAtGrow(m_nCases,pCase);
					m_nCases++;
				}
				if(!FindStep(pFromCase,pCase,step_info.m_step_set,direction))
				{	// insert new step
					if(direction==DIRECTION_FWD)
						pStep=new CPNStep(step_info.m_step_set,pFromCase,pCase);
					else
						pStep=new CPNStep(step_info.m_step_set,pCase,pFromCase);
					m_case_graph_steps.SetAtGrow(m_nSteps,pStep);
					m_nSteps++;
				}
			}
		}
	}while(case_index<m_nCases);
	m_realizable_events.RemoveAll();
	m_steps.RemoveAll();
}

BOOL CCaseGraph::IsContactFree()
{	// zjisti, zda je system bezkontaktni
	CPNCase* pCase;
	CPNEvent* pEvent;
	for(int event_index=0;event_index<m_nEvents;event_index++)
	{	// projde vsechny udalosti systemu
		pEvent=m_net_events.GetAt(event_index);
		for(int case_index=0;case_index<m_nCases;case_index++)
		{	// u kazde udalosti se v kazdem pripadu zkoumaji podminky kontaktu
			pCase=m_case_graph_cases.GetAt(case_index);
			if(pCase->GetCase().IsContact(pEvent->GetPreSet(),pEvent->GetPostSet()))return FALSE;
		}
	}
	return TRUE;
}

void CCaseGraph::LabelElements(CDC* pDC)
{	// create labels for case graph elements
	CString text;
	CPNCase* pCase;
	CPNStep* pStep;
	CPNLabel* pLabel;
	for(int i=0;i<m_nCases;i++)
	{	// pripady maji popisek v sobe
		pCase=m_case_graph_cases.GetAt(i);
		text.Format("c%d",i+1);
		pCase->SetText(text,pDC);
	}
	m_nLabels=m_nSteps;
	m_case_graph_labels.SetSize(m_nSteps,1);
	for(i=0;i<m_nSteps;i++)
	{	// kroky maji popisek externi label
		pStep=m_case_graph_steps.GetAt(i);
		pLabel=new CPNLabel();
		text.Format("s%d",i+1);
		pLabel->SetText(text,pDC);
		pLabel->SetOwner(pStep);
		m_case_graph_labels.SetAt(i,pLabel);
		pStep->SetLabel(pLabel);
	}
}

CRect CCaseGraph::RecalculateSteps(const CPNElement* pDragged_element/*=NULL*/,bool first/*=false*/)
{	// recalculate intersections with elements and edges
	CPNStep* pStep;
	CPNLabel* pLabel;
	CRect invalid_rect,step_rect,label_rect;
	invalid_rect.SetRectEmpty();
	CRect tmp_rect,oldrect;
	tmp_rect.SetRectEmpty();
	for(int i=0;i<m_nSteps;i++)
	{
		pStep=m_case_graph_steps.GetAt(i);
		if(first)
		{
			pStep->FirstRecalculate();
		}
		else
		{
			pLabel=pStep->GetLabel();
			if(pLabel)label_rect=pLabel->GetRect();
			step_rect=pStep->GetRect();
			if(pStep->Recalculate(pDragged_element))
			{
				invalid_rect.UnionRect(tmp_rect,step_rect);
				tmp_rect=invalid_rect;
				invalid_rect.UnionRect(tmp_rect,pStep->GetRect());
				tmp_rect=invalid_rect;
				if(pLabel)
				{
					invalid_rect.UnionRect(tmp_rect,label_rect);
					tmp_rect=invalid_rect;
					invalid_rect.UnionRect(tmp_rect,pLabel->GetRect());
					tmp_rect=invalid_rect;
				}
			}
		}
	}
	return invalid_rect;
}

void CCaseGraph::ComputeRealizableEvents(CSetConditions& act_case,int direction)
{	// fill array with realizable events in actual case
	m_nRlzEvents=0;
	CPNEvent* pEvent;
	for(int i=0;i<m_nEvents;i++)
	{
		pEvent=m_net_events.GetAt(i);
		if(pEvent->UpdateRealizable(act_case,direction==DIRECTION_FWD))
		{
			m_realizable_events.SetAtGrow(m_nRlzEvents,pEvent);
			m_nRlzEvents++;
		}
	}
}

void CCaseGraph::GenerateAllSteps()
{	// generate steps from list of realizable events
	m_nGenSteps=0;
	if(m_nRlzEvents>0)
	{
		CPNEvent* pEvent;
		CStepInfo step_info;
		if(m_nRlzEvents==1)
		{	// only one realizable event
			pEvent=m_realizable_events.GetAt(0);
			step_info.m_step_set.Add(pEvent);
			step_info.m_step_preset.Union(pEvent->GetPreSet());
			step_info.m_step_postset.Union(pEvent->GetPostSet());
			m_steps.SetAtGrow(0,step_info);
			m_nGenSteps=1;
		}
		else
		{	// m_nRlzEvents>1
			m_max_step_level=m_nRlzEvents-1;
			GenerateStepsCombinations(step_info,0,1);
			// plny krok obsahujici vsechny udalosti je nutne vytvorit explicitne
			bool step=true;
			for(int i=0;i<m_nRlzEvents;i++)
			{
				pEvent=m_realizable_events.GetAt(i);
				step=pEvent->GetPreSet().Intersect(step_info.m_step_preset).IsEmpty();
				step=step && pEvent->GetPostSet().Intersect(step_info.m_step_postset).IsEmpty();
				if(!step)break;
				step_info.m_step_set.Add(pEvent);
				step_info.m_step_preset.Union(pEvent->GetPreSet());
				step_info.m_step_postset.Union(pEvent->GetPostSet());
			}
			if(step)
			{	// mame novy krok
				m_steps.SetAtGrow(m_nGenSteps,step_info);
				m_nGenSteps++;
			}
		}
	}
}

void CCaseGraph::GenerateStepsCombinations(CStepInfo& step_info,int index,int level)
{	// recursively compute combinations of events
	bool step;
	CPNEvent* pEvent;
	CStepInfo new_step_info;
	for(int i=index;i<m_nRlzEvents;i++)
	{
		pEvent=m_realizable_events.GetAt(i);
		step=true;
		if(level!=1)
		{	// test for conflict
			step=pEvent->GetPreSet().Intersect(step_info.m_step_preset).IsEmpty();
			step=step && pEvent->GetPostSet().Intersect(step_info.m_step_postset).IsEmpty();
		}
		if(step)
		{	// mame novy krok
			new_step_info=step_info;
			new_step_info.m_step_set.Add(pEvent);
			new_step_info.m_step_preset.Union(pEvent->GetPreSet());
			new_step_info.m_step_postset.Union(pEvent->GetPostSet());
			m_steps.SetAtGrow(m_nGenSteps,new_step_info);
			m_nGenSteps++;
			if(level<m_max_step_level) // recursively search
				GenerateStepsCombinations(new_step_info,i+1,level+1);
		}
	}
}

bool CCaseGraph::GetNextStep(int& step_index,CStepInfo& step_info)
{	// prochazi pole kroku
	if(step_index<m_nGenSteps)
	{
		step_info=m_steps.GetAt(step_index);
		step_index++;
		return true;
	}
	return false;
}

CSetEvents CCaseGraph::GetStep(CSetConditions& act_case,bool random/*=true*/)
{
	ComputeRealizableEvents(act_case,DIRECTION_FWD);
	GenerateAllSteps();
	if(m_nGenSteps==0)return CSetEvents();
	if(random)
	{	// random step
		int step=rand() % m_nGenSteps;
		return m_steps.GetAt(step).m_step_set;
	}
	else
	{	// biggest step
		int len=0;
		int max_len=0;
		int max_index=0;
		for(int index=0;index<m_nGenSteps;index++)
		{
			len=m_steps.GetAt(index).m_step_set.GetLength();
			if(len>max_len)
			{
				max_len=len;
				max_index=index;
			}
		}
		return m_steps.GetAt(max_index).m_step_set;
	}
}

void CCaseGraph::RealizeStep(CSetConditions& act_case,CStepInfo& step_info,int direction)
{	// realizing step
	if(direction==DIRECTION_FWD)
	{	// new_case=(actual_case\preset_of_step) + postset_of_step
		act_case.Difference(step_info.m_step_preset);
		act_case.Union(step_info.m_step_postset);
	}
	else
	{	// new_case=(actual_case\postset_of_step) + preset_of_step
		act_case.Difference(step_info.m_step_postset);
		act_case.Union(step_info.m_step_preset);
	}
}

CPNCase* CCaseGraph::FindCase(CSetConditions& act_case)
{
	CPNCase* pCase;
	for(int i=0;i<m_nCases;i++)
	{
		pCase=m_case_graph_cases.GetAt(i);
		if(pCase->GetCase()==act_case)return pCase;
	}
	return NULL;
}

bool CCaseGraph::FindStep(CPNCase* from_case,CPNCase* to_case,CSetEvents& step_set,int direction)
{
	CPNStep* pStep;
	for(int i=0;i<m_nSteps;i++)
	{
		pStep=m_case_graph_steps.GetAt(i);
		if(direction==DIRECTION_FWD)
		{
			if(pStep->IsCorresponding(from_case,to_case))
			{
				if(pStep->GetStep()==step_set)return true;
			}
		}
		else
		{
			if(pStep->IsCorresponding(to_case,from_case))
			{
				if(pStep->GetStep()==step_set)return true;
			}
		}
	}
	return false;
}

void CCaseGraph::Writeoutcaseclass() 
{
	_AddOutputListString(OUTPUT_ANALYSIS,"Writing out case class...");
	for(int i=0;i<m_nCases;i++)
		_AddOutputListString(OUTPUT_ANALYSIS,m_case_graph_cases.GetAt(i)->GetCase().Format());
	CString out_str;
	out_str.Format("Total %d case(s)",m_nCases);
	_AddOutputListString(OUTPUT_ANALYSIS,out_str);
}

void CCaseGraph::ExportCaseClass(CString& filename) 
{
	CStdioFile file;
	if(file.Open(filename,CFile::modeCreate|CFile::modeWrite|CFile::typeText))	
	{
		file.WriteString("<?xml version=\"1.0\" encoding=\"windows-1250\"?>\n");
		file.WriteString("<caseclass>\n");
		for(int i=0;i<m_nCases;i++)
		{
			file.WriteString("  <case>");
			file.WriteString(m_case_graph_cases.GetAt(i)->GetCase().Format());
			file.WriteString("</case>\n");
		}
		file.WriteString("</caseclass>\n");
		file.Close();
	}
}

void CCaseGraph::ExportToXml(CString& filename)
{
	CStdioFile file;
	if(file.Open(filename,CFile::modeCreate|CFile::modeWrite|CFile::typeText))	
	{
		file.WriteString("<?xml version=\"1.0\" encoding=\"windows-1250\"?>\n");
		file.WriteString("<casegraph>\n");
		for(int i=0;i<m_nCases;i++)
		{
			file.WriteString("  <case name=\"");
			file.WriteString(m_case_graph_cases.GetAt(i)->GetText());
			file.WriteString("\">");
			file.WriteString(m_case_graph_cases.GetAt(i)->GetCase().Format());
			file.WriteString("</case>\n");
		}
		CPNCase* pCase;
		CPNLabel* pLabel;
		CPNElement* pElement;
		for(i=0;i<m_nSteps;i++)
		{
			pLabel=m_case_graph_steps.GetAt(i)->GetLabel();
			file.WriteString("  <step name=\"");
			if(pLabel)file.WriteString(pLabel->GetText());
			file.WriteString("\">\n");
			file.WriteString(m_case_graph_steps.GetAt(i)->GetStep().Format());
			file.WriteString("    <from name=\"");
			pElement=m_case_graph_steps.GetAt(i)->GetFrom();
			pCase=dynamic_cast<CPNCase*>(pElement);
			if(pCase)file.WriteString(pCase->GetText());
			file.WriteString("\">\n");
			file.WriteString("    <to name=\"");
			pElement=m_case_graph_steps.GetAt(i)->GetTo();
			pCase=dynamic_cast<CPNCase*>(pElement);
			if(pCase)file.WriteString(pCase->GetText());
			file.WriteString("\">\n");
			file.WriteString("  </step>\n");
		}
		file.WriteString("</casegraph>\n");
		file.Close();
	}
}

void CCaseGraph::WriteOutLegend()
{
	CString out_str;
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	((CMainFrame*)AfxGetApp()->m_pMainWnd)->OutputwindowResetcontent(OUTPUT_CASEGRAPH);
	_AddOutputListString(OUTPUT_CASEGRAPH,"Names of cases...");
	CPNCase* pCase;
	CPNLabel* pLabel;
	for(int i=0;i<m_nCases;i++)
	{
		pCase=m_case_graph_cases.GetAt(i);
		out_str=pCase->GetText();
		out_str+=": ";
		out_str+=pCase->GetCase().Format();
		_AddOutputListString(OUTPUT_CASEGRAPH,out_str);
	}
	_AddOutputListString(OUTPUT_CASEGRAPH,"");
	_AddOutputListString(OUTPUT_CASEGRAPH,"Names of steps...");
	CPNStep* pStep;
	for(i=0;i<m_nSteps;i++)
	{
		pStep=m_case_graph_steps.GetAt(i);
		pLabel=pStep->GetLabel();
		if(pLabel)
		{
			out_str=pLabel->GetText();
			out_str+=": ";
			out_str+=pStep->GetStep().Format();
			_AddOutputListString(OUTPUT_CASEGRAPH,out_str);
		}
	}
}

CPNElement* CCaseGraph::ElementAt(const CPoint& point)
{	// return top element on point
	int i;
	CPNElement* pElement;
	for(i=0;i<m_nCases;i++)
	{
		pElement=m_case_graph_cases.GetAt(i);
		if(pElement->HitTest(point))return pElement;
	}
	for(i=0;i<m_nSteps;i++)
	{
		pElement=m_case_graph_steps.GetAt(i);
		if(pElement->HitTest(point))return pElement;
	}
	for(i=0;i<m_nLabels;i++)
	{
		pElement=m_case_graph_labels.GetAt(i);
		if(pElement->HitTest(point))return pElement;
	}
	POSITION pos=m_elements_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_elements_list.GetNext(pos);
		if(pElement->HitTest(point))return pElement;
	}
	return NULL;
}

CRect CCaseGraph::SelectElements(CRect& rect,CPNElementList& selected_elements)
{	// select elements with in rect
	int i;
	CRect rect_element;
	CPNElement* pElement;
	CRect invalid_rect;
	CRect tmp_rect;
	tmp_rect.SetRectEmpty();
	for(i=0;i<m_nCases;i++)
	{
		pElement=m_case_graph_cases.GetAt(i);
		rect_element=pElement->GetRect();
		if( (rect_element.left>rect.left) && (rect_element.top>rect.top) &&
			(rect_element.right<rect.right) && (rect_element.bottom<rect.bottom))
		{
			pElement->Select();
			selected_elements.AddTail(pElement);
			invalid_rect.UnionRect(tmp_rect,rect_element);
			tmp_rect=invalid_rect;
		}
	}
	for(i=0;i<m_nSteps;i++)
	{
		pElement=m_case_graph_steps.GetAt(i);
		rect_element=pElement->GetRect();
		if( (rect_element.left>rect.left) && (rect_element.top>rect.top) &&
			(rect_element.right<rect.right) && (rect_element.bottom<rect.bottom))
		{
			pElement->Select();
			selected_elements.AddTail(pElement);
			invalid_rect.UnionRect(tmp_rect,rect_element);
			tmp_rect=invalid_rect;
		}
	}
	for(i=0;i<m_nLabels;i++)
	{
		pElement=m_case_graph_labels.GetAt(i);
		rect_element=pElement->GetRect();
		if( (rect_element.left>rect.left) && (rect_element.top>rect.top) &&
			(rect_element.right<rect.right) && (rect_element.bottom<rect.bottom))
		{
			pElement->Select();
			selected_elements.AddTail(pElement);
			invalid_rect.UnionRect(tmp_rect,rect_element);
			tmp_rect=invalid_rect;
		}
	}
	POSITION pos=m_elements_list.GetHeadPosition();
	while(pos!=NULL)
	{
		pElement=m_elements_list.GetNext(pos);
		rect_element=pElement->GetRect();
		if( (rect_element.left>rect.left) && (rect_element.top>rect.top) &&
			(rect_element.right<rect.right) && (rect_element.bottom<rect.bottom))
		{
			pElement->Select();
			selected_elements.AddTail(pElement);
			invalid_rect.UnionRect(tmp_rect,rect_element);
			tmp_rect=invalid_rect;
		}
	}
	return invalid_rect;
}

void CCaseGraph::InitCaseBrowse()
{
	m_case_browse_index=0;
}

CPNCase* CCaseGraph::GetNextCase()
{
	CPNCase* pCase=NULL;
	if(m_case_browse_index<m_nCases)
	{
		pCase=m_case_graph_cases.GetAt(m_case_browse_index);
		m_case_browse_index++;
	}
	return pCase;
}

bool CCaseGraph::Smooth(CSize& window_size)
{
	if(!m_smooth_initialized)
	{
		m_smooth_initialized=true;
		m_dlg_smooth.SetInfo(m_grid_size,m_min_grid_size);
		m_dlg_smooth.Init(m_case_graph_steps,m_nSteps,m_case_graph_cases,m_nCases);
	}
	if(m_dlg_smooth.DoModal()==IDOK)
	{
		if(m_dlg_smooth.BestExist())
		{
			m_dlg_smooth.ApplyBestChromozome(m_case_graph_cases,m_nCases);
			RecalculateSteps(NULL,true);
			m_grid_size=m_dlg_smooth.GetGridSize();
			SetWindowSize(CSize(100,100));
			window_size=m_window_size;
			return true;
		}
	}
	return false;
}

CRect CCaseGraph::PlaceSelectedElements(bool place_back,CPNElementList& selected_elements)
{	// place elements to back or front of all selected elements
	CRect invalid_rect,tmp_rect;
	tmp_rect.SetRectEmpty();
	invalid_rect.SetRectEmpty();
	POSITION sel_pos=selected_elements.GetHeadPosition();
	CPNElement* pElement;
	while(sel_pos!=NULL)
	{
		pElement=selected_elements.GetNext(sel_pos);
		// 
		POSITION el_pos=m_elements_list.Find(pElement);
		if(el_pos!=NULL)
		{
			m_elements_list.RemoveAt(el_pos);
			if(place_back)
				m_elements_list.AddHead(pElement);
			else
				m_elements_list.AddTail(pElement);
		}
		invalid_rect.UnionRect(tmp_rect,pElement->GetRect());
		tmp_rect=invalid_rect;
	}
	return invalid_rect;
}

void CCaseGraph::HighLightCase(CSetConditions& case_set,bool highlight/*=true*/)
{
	CPNCase* pCase;
	for(int index=0;index<m_nCases;index++)
	{
		pCase=m_case_graph_cases.GetAt(index);
		pCase->HighLight(highlight && (pCase->GetCase()==case_set));
	}
}

void CCaseGraph::PrintLegend(CDC* pDC,int textheight)
{
	CPNCase* pCase;
	CPNLabel* pLabel;
	CString out_str;
	CPoint point(720,-740);
	pDC->TextOut(point.x,point.y,"Names of cases:");
	point.y-=textheight;
	for(int i=0;i<m_nCases;i++)
	{
		pCase=m_case_graph_cases.GetAt(i);
		out_str=pCase->GetText();
		out_str+=": ";
		out_str+=pCase->GetCase().Format();
		pDC->TextOut(point.x,point.y,out_str);
		point.y-=textheight;
	}
	point.y-=textheight;
	pDC->TextOut(point.x,point.y,"Names of steps:");
	point.y-=textheight;
	CPNStep* pStep;
	for(i=0;i<m_nSteps;i++)
	{
		pStep=m_case_graph_steps.GetAt(i);
		pLabel=pStep->GetLabel();
		if(pLabel)
		{
			out_str=pLabel->GetText();
			out_str+=": ";
			out_str+=pStep->GetStep().Format();
			pDC->TextOut(point.x,point.y,out_str);
			point.y-=textheight;
		}
	}
}
void CCaseGraph::ComputeReach(const CSetConditions& initial_case)
{	// compute forward reachable cases
	CPNCase* pCase=NULL;
	for(int index=0;index<m_nCases;index++)
	{	// find initial case
		pCase=m_case_graph_cases.GetAt(index);
		if(pCase->GetCase()==initial_case)break;
	}
	if(pCase!=NULL)
	{
		pCase->SetReachAbleInit();
		if(pCase->IsReachAble())return; // already computed
		ReachCase(pCase);
	}
}

void CCaseGraph::ReachCase(CPNCase* pCase)
{	// recursive depth first search
	pCase->SetReachAble();
	CPNStep* pStep;
	CPNCase* pNextCase;
	for(int index=0;index<m_nSteps;index++)
	{
		pStep=m_case_graph_steps.GetAt(index);
		if(pStep->GetFrom()==pCase)
		{
			pNextCase=dynamic_cast<CPNCase*>(pStep->GetTo());
			if(pNextCase!=NULL)
				if(!pNextCase->IsReachAble())ReachCase(pNextCase);
		}
	}

}

void CCaseGraph::Computeallreachablecases()
{	// projde vsechny pripady a pro kazdy vypocte jeho dopredne dosazitelnou mnozinu pripadu
	int case_index;
	CPNCase* pCase;
	CSetConditions initial_case;
	for(case_index=0; case_index < m_nCases; case_index++)
	{	// find initial case
		pCase=m_case_graph_cases.GetAt(case_index);
		if(pCase->IsReachAbleInit())
		{
			initial_case=pCase->GetCase();
			break;
		}
	}
	m_frcases_classes_list.RemoveAll();
	for(int initial_case_index=0; initial_case_index < m_nCases; initial_case_index++)
	{	
		for(int case_index=0; case_index < m_nCases; case_index++)
		{	// reset informaci o dosazitelnosti
			pCase=m_case_graph_cases.GetAt(case_index);
			pCase->SetReachAble(FALSE);
			pCase->SetReachAbleInit(FALSE);
		}
		// vypocet nove dosazitelne mnoziny z noveho pocatecniho pripadu
		pCase=m_case_graph_cases.GetAt(initial_case_index);
		ReachCase(pCase);

		CFRCasesClass frcases_class;
		for(case_index=0; case_index < m_nCases; case_index++)
		{	// vytvoreni mnoziny dopredne dosazitelnych pripadu
			pCase=m_case_graph_cases.GetAt(case_index);
			if(pCase->IsReachAble())
			{
				frcases_class.AddCase(pCase);
			}
		}
		bool create_new=true;
		POSITION pos=m_frcases_classes_list.GetHeadPosition();
		while(pos!=NULL)
		{	// zjistit, zda se jedna o novou tridu pripadu
			if(m_frcases_classes_list.GetAt(pos).m_frcases_set==frcases_class.m_frcases_set)
			{	// stejna trida jiz existuje
				m_frcases_classes_list.GetAt(pos).IncCount();
				create_new=false;
				break;
			}
			m_frcases_classes_list.GetNext(pos);
		}
		if(create_new)
		{	// zaklada se nova trida
			m_frcases_classes_list.AddHead(frcases_class);
		}
	}

	// po provedeni vypoctu je highlight reachable cases neplatne
	for(case_index=0; case_index < m_nCases; case_index++)
	{	// reset informaci o dosazitelnosti
		pCase=m_case_graph_cases.GetAt(case_index);
		pCase->SetReachAble(FALSE);
		pCase->SetReachAbleInit(FALSE);
	}
	pCase=NULL;
	for(case_index=0; case_index < m_nCases; case_index++)
	{	// find initial case
		pCase=m_case_graph_cases.GetAt(case_index);
		if(pCase->GetCase()==initial_case)break;
	}
	if(pCase!=NULL)
	{
		pCase->SetReachAbleInit();
		ReachCase(pCase);
	}

	// write out results
	_AddOutputListString(OUTPUT_ANALYSIS,"Writing out forward reachable cases classes...");
	CString out_str;
	out_str.Format("Total %d classes(s):",m_frcases_classes_list.GetCount());
	_AddOutputListString(OUTPUT_ANALYSIS,out_str);
	_AddOutputListString(OUTPUT_ANALYSIS,"");
	int index=1;
	POSITION pos=m_frcases_classes_list.GetHeadPosition();
	while(pos!=NULL)
	{	// 
		out_str.Format("Class %d [%d]:",index,m_frcases_classes_list.GetAt(pos).GetCount());
		_AddOutputListString(OUTPUT_ANALYSIS,out_str);
		out_str=m_frcases_classes_list.GetAt(pos).m_frcases_set.Format(CSetCases::STYLE_CASES);
		if(out_str.GetLength()>1)
		{	// pred zobrazenim pripadoveho grafu nejsou pripady pojmenovane, nelze je vypsat
			if((out_str.GetAt(1)!=',') && (out_str.GetAt(1)!='}')) // detekce prazdneho seznamu
				_AddOutputListString(OUTPUT_ANALYSIS,out_str);
		}
		_AddOutputListString(OUTPUT_ANALYSIS,m_frcases_classes_list.GetAt(pos).m_frcases_set.Format(CSetCases::STYLE_CONDITIONS));
		_AddOutputListString(OUTPUT_ANALYSIS,"");
		m_frcases_classes_list.GetNext(pos);
		index++;
	}
}

void CCaseGraph::HighlightReachableCases(bool highlight)
{	// zapne zobrazovani dopredne dosazitelnych pripadu
	CPNCase* pCase=NULL;
	for(int index=0;index<m_nCases;index++)
	{	// zapnout se musi kazdy pripad zvlast
		pCase=m_case_graph_cases.GetAt(index);
		pCase->ShowReachAble(highlight);
	}
}

void CCaseGraph::SetMinGridSize(int grid_size)
{ 
	m_min_grid_size=grid_size;
	m_grid_size=grid_size;
}

void CCaseGraph::Place()
{	// rozmisti pripady po plose, zachovava predchozi rozmisteni
	m_graph_placed=TRUE;
	CArray<CPoint,CPoint&> case_occupation;
	case_occupation.SetSize(m_nCases);
	srand((unsigned)time(NULL));
	CPNCase* pCase;
	CPoint position;
	for(int index=0;index<m_nCases;index++)
	{
		pCase=m_case_graph_cases.GetAt(index);
		if(index>=m_nOldCases)
		{
			position=GetRandomCasePosition(case_occupation,index);
		}
		else
		{
			position=m_oldcases_positions.GetAt(index);
			if((position.x>(m_window_size.cx-15)) || (position.y>(m_window_size.cy-15)))
				position=GetRandomCasePosition(case_occupation,index);
		}
		case_occupation.SetAt(index,position);
		pCase->MoveTo(position);
	}
	case_occupation.RemoveAll();
}

CPoint CCaseGraph::GetRandomCasePosition(CArray<CPoint,CPoint&>& case_occupation, int nCaseCount)
{	// nalezne volne misto pro case
	bool occupied;
	int side_x,side_y;
	CPoint new_pos,case_pos;
	do
	{
		occupied=false;
		side_x=rand()%m_grid_size;
		side_y=rand()%m_grid_size;
		new_pos.x=(side_x+1)*CASE_GRID_SPAN;
		new_pos.y=(side_y+1)*CASE_GRID_SPAN;
		for(int index=0;index<nCaseCount;index++)
		{
			case_pos=case_occupation.GetAt(index);
			if(new_pos==case_pos)
			{
				occupied=true;
				break;
			}
		}
	}while(occupied);
	return new_pos;
}
