// CESimNetDesignView.cpp : implementation of the CCESimNetDesignView class
//

#include "stdafx.h"
#include "CESim.h"

#include "CESimDoc.h"
#include "CESimNetDesignView.h"
#include "MainFrm.h"
#include "DlgProperties.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView
IMPLEMENT_DYNCREATE(CCESimNetDesignView, CScrollView)

BEGIN_MESSAGE_MAP(CCESimNetDesignView, CScrollView)
	//{{AFX_MSG_MAP(CCESimNetDesignView)
	ON_COMMAND(ID_VIEW_GRID, OnViewGrid)
	ON_UPDATE_COMMAND_UI(ID_VIEW_GRID, OnUpdateViewGrid)
	ON_WM_LBUTTONDOWN()
	ON_WM_CONTEXTMENU()
	ON_WM_SETCURSOR()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONUP()
	ON_COMMAND(ID_EDIT_PLACETOFRONT, OnEditPlacetofront)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PLACETOFRONT, OnUpdateEditPlacetofront)
	ON_COMMAND(ID_EDIT_PLACETOBACK, OnEditPlacetoback)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PLACETOBACK, OnUpdateEditPlacetoback)
	ON_WM_ERASEBKGND()
	ON_COMMAND(ID_EDIT_DELETE, OnEditDelete)
	ON_UPDATE_COMMAND_UI(ID_EDIT_DELETE, OnUpdateEditDelete)
	ON_COMMAND(ID_EDGE_DELETEPOINT, OnEdgeDeletepoint)
	ON_UPDATE_COMMAND_UI(ID_EDGE_DELETEPOINT, OnUpdateEdgeDeletepoint)
	ON_COMMAND(ID_EDGE_DELETEALLPOINTS, OnEdgeDeleteallpoints)
	ON_UPDATE_COMMAND_UI(ID_EDGE_DELETEALLPOINTS, OnUpdateEdgeDeleteallpoints)
	ON_COMMAND(ID_POLYGON_DELETEPOINT, OnPolygonDeletepoint)
	ON_UPDATE_COMMAND_UI(ID_POLYGON_DELETEPOINT, OnUpdatePolygonDeletepoint)
	ON_WM_LBUTTONDBLCLK()
	ON_COMMAND(ID_SIMULATION_REALIZESTEP, OnSimulationRealizeStep)
	ON_UPDATE_COMMAND_UI(ID_SIMULATION_REALIZESTEP, OnUpdateSimulationRealizeStep)
	ON_COMMAND(ID_SIMULATION_RESETSYSTEM, OnSimulationResetsystem)
	ON_UPDATE_COMMAND_UI(ID_SIMULATION_RESETSYSTEM, OnUpdateSimulationResetsystem)
	ON_BN_CLICKED(IDC_BTN_SIM_START, OnBtnSimStart)
	ON_BN_CLICKED(IDC_BTN_SIM_RESET, OnBtnSimReset)
	ON_BN_CLICKED(IDC_BTN_SIM_STEP, OnBtnSimStep)
	ON_BN_CLICKED(IDC_BTN_SIM_STOP, OnBtnSimStop)
	ON_COMMAND(ID_ANALYSIS_DONETCOMPLEMENT, OnAnalysisDonetcomplement)
	ON_UPDATE_COMMAND_UI(ID_ANALYSIS_DONETCOMPLEMENT, OnUpdateAnalysisDonetcomplement)
	ON_COMMAND(ID_VIEW_PROPERTIES_NETDESIGN, OnViewPropertiesNetdesign)
	ON_UPDATE_COMMAND_UI(ID_VIEW_PROPERTIES_NETDESIGN, OnUpdateViewPropertiesNetdesign)
	ON_COMMAND(ID_NETDESIGN_EXPORTTOEMF, OnNetdesignExporttoemf)
	ON_WM_KEYUP()
	ON_COMMAND(ID_VIEW_RULER, OnViewRuler)
	ON_UPDATE_COMMAND_UI(ID_VIEW_RULER, OnUpdateViewRuler)
	ON_WM_MOUSEWHEEL()
	//}}AFX_MSG_MAP
	ON_COMMAND_RANGE(ID_TOOLS_SELECT,ID_TOOLS_TOKEN,OnToolChange)
	ON_UPDATE_COMMAND_UI_RANGE(ID_TOOLS_SELECT,ID_TOOLS_TOKEN,OnUpdateToolsChange)
	ON_MESSAGE(APP_MSG_SIMULATION_SPEED,OnSimulationSpeed)
	ON_MESSAGE(APP_MSG_UPDATE_VIEW,OnUpdateView)
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView construction/destruction
CCESimNetDesignView::CCESimNetDesignView()
{
	m_export=false;
	m_tool_selected=TOOL_SELECT;
	m_mouse_capture=false;
	m_last_drag_pt=CPoint(-1,-1);
	m_capture_type=CT_NONE;
	m_hit_element=NULL;
	m_context_element=NULL;
	m_ignore_move=false;
	m_erase_background=false;
	m_simulation_mode=false;
	m_automatic_simulation=false;
	m_animating_token=false;

	m_ruler_last_pt=false;
	::ZeroMemory(&m_ruler_logfont,sizeof(m_ruler_logfont));
	m_ruler_logfont.lfCharSet=1;
	m_ruler_logfont.lfHeight=-10;
	strcpy(m_ruler_logfont.lfFaceName,"Arial");
	m_ruler_font.CreateFontIndirect(&m_ruler_logfont);
}

CCESimNetDesignView::~CCESimNetDesignView()
{
}

BOOL CCESimNetDesignView::PreCreateWindow(CREATESTRUCT& cs)
{
	return CScrollView::PreCreateWindow(cs);
}

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView drawing
void CCESimNetDesignView::OnDraw(CDC* pDC)
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	if(m_export)
	{
		m_export=false;
		CMetaFileDC metafile;
		metafile.CreateEnhanced(pDC,pDoc->GetTitle(),NULL,"CESim\0CESim\0\0");
		metafile.SetAttribDC(pDC->m_hAttribDC);
		pDoc->NetDesign_Draw(&metafile);
		HENHMETAFILE hmeta=metafile.CloseEnhanced();
		CopyEnhMetaFile(hmeta,m_export_path);
		DeleteEnhMetaFile(hmeta);
	}

	CDC memdc;
	CBitmap bitmap;
	CBitmap* oldbitmap=NULL;
	CRect cliprect;
	pDC->GetClipBox(&cliprect);

	CDC* pDrawDC=pDC;
	if(!pDC->IsPrinting())
	{	// draw to offscreen bitmap for flicker free repaints
		if(memdc.CreateCompatibleDC(pDC))
		{
			if(bitmap.CreateCompatibleBitmap(pDC,cliprect.Width(),cliprect.Height()))
			{
				OnPrepareDC(&memdc,NULL);
				oldbitmap=memdc.SelectObject(&bitmap);
				CRect devicerect=cliprect;
				pDC->LPtoDP(&devicerect);
				memdc.OffsetViewportOrg(-devicerect.left,-devicerect.top);
				memdc.IntersectClipRect(cliprect);
				memdc.FillSolidRect(cliprect,pDC->GetBkColor());
				pDrawDC=&memdc;
			}
		}
		CBrush* oldbrush=(CBrush*)pDrawDC->SelectStockObject(NULL_BRUSH);
		CSize window_size=pDoc->NetDesign_GetWindowSize();
		pDrawDC->Rectangle(0,0,window_size.cx,window_size.cy);
		pDrawDC->SelectObject(oldbrush);
		
		if(pDoc->IsGridVisible())
		{	// draw grid
			for(int x=pDoc->GetGridSpan();x<window_size.cx;x+=pDoc->GetGridSpan())
				for(int y=pDoc->GetGridSpan();y<window_size.cy;y+=pDoc->GetGridSpan())
					pDrawDC->SetPixel(x,y,RGB(128,128,128));
		}
		pDoc->NetDesign_Draw(pDrawDC); // draw net elements

		if(!m_simulation_mode && pDoc->IsRulerVisible())
		{	// draw ruler
			CPoint point;
			point=GetScrollPosition();
			int off_x=point.x;
			int off_y=point.y;
			// horizontal line
			pDrawDC->FillSolidRect(CRect(0,off_y,window_size.cx,off_y+RULER_THICK),RULER_CLR);
			pDrawDC->MoveTo(RULER_THICK,off_y+RULER_THICK);
			pDrawDC->LineTo(window_size.cx,off_y+RULER_THICK);
			// vertical line
			pDrawDC->FillSolidRect(CRect(off_x,0,off_x+RULER_THICK,window_size.cy),RULER_CLR);
			pDrawDC->MoveTo(off_x+RULER_THICK,RULER_THICK);
			pDrawDC->LineTo(off_x+RULER_THICK,window_size.cy);
			// draw mid-lines
			int x,y;
			for(x=RULER_SPAN/4;x<window_size.cx;x+=RULER_SPAN/4)
			{	// horizontal
				if((x % RULER_SPAN)==0)continue;
				if((x % (RULER_SPAN/2))==0)
					pDrawDC->MoveTo(x,off_y+RULER_THICK/2);
				else
					pDrawDC->MoveTo(x,off_y+RULER_THICK-RULER_THICK/3);
				pDrawDC->LineTo(x,off_y+RULER_THICK);
			}
			for(y=RULER_SPAN/4;y<window_size.cy;y+=RULER_SPAN/4)
			{	// vertical
				if((y % RULER_SPAN)==0)continue;
				if((y % (RULER_SPAN/2))==0)
					pDrawDC->MoveTo(off_x+RULER_THICK/2,y);
				else
					pDrawDC->MoveTo(off_x+RULER_THICK-RULER_THICK/3,y);
				pDrawDC->LineTo(off_x+RULER_THICK,y);
			}
			// draw numbers
			COLORREF old_clr=pDrawDC->SetBkColor(RULER_CLR);
			CFont* old_font=pDrawDC->SelectObject(&m_ruler_font);
			CString text;
			CSize text_size;
			for(x=RULER_SPAN;x<window_size.cx;x+=RULER_SPAN)
			{	// horizontal numbers
				text.Format("%d",x/RULER_SPAN);
				text_size=pDrawDC->GetTextExtent(text);
				pDrawDC->TextOut(x-text_size.cx/2,off_y,text);
			}
			for(y=RULER_SPAN;y<window_size.cy;y+=RULER_SPAN)
			{	// vertical numbers
				text.Format("%d",y/RULER_SPAN);
				text_size=pDrawDC->GetTextExtent(text);
				pDrawDC->TextOut(off_x+(RULER_THICK-text_size.cx)/2,y-text_size.cy/2,text);
			}
			// topleft
			pDrawDC->FillSolidRect(CRect(off_x,off_y,off_x+RULER_THICK+1,off_y+RULER_THICK),RULER_CLR);
			pDrawDC->TextOut(5,0,"0");
			pDrawDC->SelectObject(old_font);
			pDrawDC->SetBkColor(old_clr);
		}
	}
	else
		pDoc->NetDesign_Draw(pDrawDC); // draw net elements
	if(pDrawDC!=pDC)
	{	// draw memory DC
		pDC->BitBlt(cliprect.left,cliprect.top,cliprect.Width(),cliprect.Height(),&memdc,cliprect.left,cliprect.top,SRCCOPY);
		memdc.SelectObject(oldbitmap);
	}
	m_erase_background=false;
}

void CCESimNetDesignView::DrawRulerPos(CDC* pDC,CPoint& point,bool bClearOnly/*=false*/)
{	// draw ruler position
	CPoint scrollpos=GetScrollPosition();
	int off_x=scrollpos.x;
	int off_y=scrollpos.y;

	// horizontal
	if(m_ruler_last_pt)
	{
		InvertLine(pDC,CPoint(m_ruler_last_mouse_pt.x,off_y),CPoint(m_ruler_last_mouse_pt.x,off_y+RULER_THICK));
/*			InvertLine(pDC,CPoint(m_ruler_last_mouse_pt.x-RULER_ARROW_HEIGHT/2,off_y+(RULER_THICK-RULER_ARROW_WIDTH)),CPoint(m_ruler_last_mouse_pt.x+RULER_ARROW_HEIGHT/2+1,off_y+(RULER_THICK-RULER_ARROW_WIDTH)));
		InvertLine(pDC,CPoint(m_ruler_last_mouse_pt.x-1,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(m_ruler_last_mouse_pt.x-1,off_y+RULER_THICK-1));
		InvertLine(pDC,CPoint(m_ruler_last_mouse_pt.x,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(m_ruler_last_mouse_pt.x,off_y+RULER_THICK));
		InvertLine(pDC,CPoint(m_ruler_last_mouse_pt.x+1,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(m_ruler_last_mouse_pt.x+1,off_y+RULER_THICK-1));*/
	}
	if(!bClearOnly)
	{
		InvertLine(pDC,CPoint(point.x,off_y),CPoint(point.x,off_y+RULER_THICK));
/*			InvertLine(pDC,CPoint(point.x-RULER_ARROW_HEIGHT/2,off_y+(RULER_THICK-RULER_ARROW_WIDTH)),CPoint(point.x+RULER_ARROW_HEIGHT/2+1,off_y+(RULER_THICK-RULER_ARROW_WIDTH)));
		InvertLine(pDC,CPoint(point.x-1,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(point.x-1,off_y+RULER_THICK-1));
		InvertLine(pDC,CPoint(point.x,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(point.x,off_y+RULER_THICK));
		InvertLine(pDC,CPoint(point.x+1,off_y+(RULER_THICK-(RULER_ARROW_WIDTH-1))),CPoint(point.x+1,off_y+RULER_THICK-1));*/
	}
	// vertical
	if(m_ruler_last_pt)InvertLine(pDC,CPoint(off_x,m_ruler_last_mouse_pt.y),CPoint(off_x+RULER_THICK,m_ruler_last_mouse_pt.y));
	if(!bClearOnly)InvertLine(pDC,CPoint(off_x,point.y),CPoint(off_x+RULER_THICK,point.y));

	m_ruler_last_pt=true;
	m_ruler_last_mouse_pt=point;
}

void CCESimNetDesignView::OnInitialUpdate()
{
	CScrollView::OnInitialUpdate();

	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	m_tool_selected=TOOL_SELECT;
	m_mouse_capture=false;
	m_last_drag_pt=CPoint(-1,-1);
	m_capture_type=CT_NONE;
	m_hit_element=NULL;
	m_context_element=NULL;
	m_erase_background=false;
	m_simulation_mode=false;
	m_automatic_simulation=false;
	m_animating_token=false;

	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//	((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);

	SetScrollSizes(MM_TEXT,pDoc->NetDesign_GetWindowSize());
}

void CCESimNetDesignView::InvalidateRect(CRect rect,CDC* pDC /*=NULL*/)
{
	if(pDC==NULL)
	{
		CClientDC dc(this);
		OnPrepareDC(&dc);
		dc.LPtoDP(&rect);
		CScrollView::InvalidateRect(rect,FALSE);
	}
	else
	{
		pDC->LPtoDP(&rect);
		CScrollView::InvalidateRect(rect,FALSE);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView printing
BOOL CCESimNetDesignView::OnPreparePrinting(CPrintInfo* pInfo)
{
	return DoPreparePrinting(pInfo);
}

void CCESimNetDesignView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	((CMainFrame*)AfxGetApp()->m_pMainWnd)->HideOutputBar();

	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	// calculate scale
	pDC->SetMapMode(MM_ISOTROPIC);
	int cx=pDoc->NetDesign_GetWindowSize().cx;
	int horz=pDC->GetDeviceCaps(HORZRES);
	pDC->SetWindowExt(cx,cx);
	pDC->SetViewportExt(horz,horz);
	int vert=pDC->GetDeviceCaps(VERTRES);
	int cy=pDoc->NetDesign_GetWindowSize().cy;
	CSize size;
	size.cx=0;
	size.cy=cy;
	size.cy+=70;
	pDC->LPtoDP(&size);
	while((size.cy>vert)&&(horz>0))
	{	// horizontal paper position
		horz-=30;
		pDC->SetViewportExt(horz,horz);
		size.cy=cy;
		size.cy+=70;
		pDC->LPtoDP(&size);
	};

	// document information
	CRect rect;
	CFont font;
	LOGFONT lf;
	memset(&lf,0,sizeof(LOGFONT));
	lf.lfHeight=100;
	lf.lfWeight=400;
	::lstrcpy(lf.lfFaceName,"Arial CE");
	font.CreatePointFontIndirect(&lf);
	CFont* oldfont=pDC->SelectObject(&font);
	CString text;
	text=GetDocument()->GetTitle(); 
	rect=pInfo->m_rectDraw;
	pDC->DPtoLP(rect);
	rect.top=rect.bottom-pDC->GetTextExtent(text).cy-10;
	rect.left=10;
	rect.right-=20;
	pDC->FillSolidRect(rect.left,rect.top,rect.right,2,RGB(0,0,0)); // separator
	rect.top+=5;
	rect.left+=5;
	pDC->SetBkMode(TRANSPARENT);
	pDC->DrawText(text,rect,DT_SINGLELINE|DT_LEFT); // file name
	CTime cas=CTime::GetCurrentTime(); // date and time
	text.Format("%d/%d/%d  %.2u:%.2u:%.2u",cas.GetDay(),cas.GetMonth(),cas.GetYear(),cas.GetHour(),cas.GetMinute(),cas.GetSecond());
	rect.left=rect.right-pDC->GetTextExtent(text).cx-10;
	pDC->DrawText(text,rect,DT_SINGLELINE|DT_LEFT);
	pDC->SelectObject(oldfont);

	OnDraw(pDC); // draw view
}

void CCESimNetDesignView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
}

void CCESimNetDesignView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
}

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView diagnostics
#ifdef _DEBUG
void CCESimNetDesignView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CCESimNetDesignView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CCESimDoc* CCESimNetDesignView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CCESimDoc)));
	return (CCESimDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CCESimNetDesignView message handlers
void CCESimNetDesignView::OnToolChange(UINT nID)
{
	m_tool_selected=nID;
}

void CCESimNetDesignView::OnUpdateToolsChange(CCmdUI* pCmdUI)
{
	pCmdUI->SetRadio(static_cast<unsigned int>(m_tool_selected)==pCmdUI->m_nID);
	if(m_simulation_mode)
		pCmdUI->Enable(pCmdUI->m_nID==TOOL_SELECT);
	else
		pCmdUI->Enable();
}

void CCESimNetDesignView::OnViewGrid()
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	pDoc->SetModifiedFlag();
	pDoc->SetGridVisible(!pDoc->IsGridVisible());
	Invalidate(FALSE);
}

void CCESimNetDesignView::OnUpdateViewGrid(CCmdUI* pCmdUI)
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->SetCheck(!m_simulation_mode && pDoc->IsGridVisible());
	pCmdUI->Enable(!m_simulation_mode);
}

void CCESimNetDesignView::OnViewRuler() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	pDoc->SetRulerVisible(!pDoc->IsRulerVisible());
	Invalidate(FALSE);
}

void CCESimNetDesignView::OnUpdateViewRuler(CCmdUI* pCmdUI) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->SetCheck(!m_simulation_mode && pDoc->IsRulerVisible());
	pCmdUI->Enable(!m_simulation_mode);
}

void CCESimNetDesignView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	CClientDC dc(this);
	OnPrepareDC(&dc);
	dc.DPtoLP(&point);
	if(point.x>pDoc->NetDesign_GetWindowSize().cx)return; // out of view
	if(point.y>pDoc->NetDesign_GetWindowSize().cy)return;
	m_last_mouse_pt=point;
	m_ruler_last_pt=false;

	UINT tool_selected=m_tool_selected;
	// to bypass selected tool from drawing edge and polygon directly to dragging point
	if(pDoc->CanDragPoint(point))tool_selected=TOOL_SELECT;
	// multiselect is enabled for SHIFT or CONTROL key down
	bool modif_control=(nFlags & MK_CONTROL) || (nFlags & MK_SHIFT);

	switch(tool_selected)
	{
	case TOOL_SELECT:
		{
			m_capture_type=CT_NONE;
			m_hit_element=pDoc->SelectedDraggedEdgeAt(point); // uprednostneni vyberu pretahovacich bodu oznacenych hran
			if(m_hit_element==NULL)m_hit_element=pDoc->NetDesign_ElementAt(point,m_simulation_mode);
			if(m_automatic_simulation)break; // v automaticke simulaci se nesmi dat nic oznacit
			if(m_simulation_mode && m_hit_element)
			{	// v simulaci lze oznacit pouze realizovatelne udalosti
				CPNEvent* pEvent;
				pEvent=dynamic_cast<CPNEvent*>(m_hit_element);
				if(pEvent)
				{
					if(!pEvent->IsRealizable())m_hit_element=NULL;
				}
				else
					m_hit_element=NULL;
			}
			if(m_hit_element)
			{	// element was selected
				if(pDoc->IsElementSelected(m_hit_element))
				{	
					if(modif_control)
					{	// flip selection
						InvalidateRect(pDoc->DeselectElement(m_hit_element),&dc);
						if((!pDoc->IsElementSelected()) && (!m_simulation_mode))
						{
							ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
							((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//							((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);
						}
						break;
					}
					if(pDoc->IsMultiselect())m_capture_type=CT_DRAGGING_MULTI_SELECT;
				}
				else
				{	// new element was selected
					InvalidateRect(pDoc->SelectElement(m_hit_element,modif_control),&dc);
					if(pDoc->IsMultiselect())
					{	// 
						m_capture_type=CT_DRAGGING_MULTI_SELECT;
						if(!m_simulation_mode)
						{
							ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
							((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//							((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);
						}
					}
					else
					{
						if(!m_simulation_mode)m_hit_element->UpdatePropertiesDlg();
					}
				}
				if(m_capture_type==CT_NONE)
				{	// 
					if(dynamic_cast<CPNCondition*>(m_hit_element))m_capture_type=CT_DRAGGING_COND;
					if(dynamic_cast<CPNEvent*>(m_hit_element))m_capture_type=CT_DRAGGING_EVNT;
					if(dynamic_cast<CPNLabel*>(m_hit_element))m_capture_type=CT_DRAGGING_LBL;
					CPNEdge* pEdge=dynamic_cast<CPNEdge*>(m_hit_element);
					if(pEdge)
					{
						if(pEdge->IsDragging(point))m_capture_type=CT_DRAGGING_EDGE;
					}
					CPNPolygon* pPolygon=dynamic_cast<CPNPolygon*>(m_hit_element);
					if(pPolygon)
					{
						if(pPolygon->IsDragging(point))
							m_capture_type=CT_DRAGGING_POLYGON_POINT;
						else
							m_capture_type=CT_DRAGGING_POLYGON;
					}
				}
			}
			else
			{
				if(!modif_control)
				{	// deselect all elements
					CRect old_sel_rect=pDoc->DeselectAllElements();
					if(!old_sel_rect.IsRectNull())InvalidateRect(old_sel_rect,&dc);

					if(!m_simulation_mode)
					{
						ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
						((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//						((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);
					}
				}
				m_capture_type=CT_DRAWING_SELECT_RECT;
				m_mouse_moved=false;
				m_first_mouse_pt=point;
			}
			if(m_capture_type!=CT_NONE)
			{
				SetCapture();
				m_mouse_capture=true;
				m_ignore_move=true;
				m_last_drag_pt=CPoint(-1,-1);
				if(m_capture_type!=CT_DRAWING_SELECT_RECT)::SetCursor(::LoadCursor(NULL,IDC_SIZEALL));
			}
			break;
		}
	case TOOL_CONDITION:
		{	// create corresponding label
			CPNLabel* pLabel=new CPNLabel();
			pLabel->SetText(pDoc->GetAutomaticLabelText(CCESimDoc::LABEL_CONDITION),&dc);
			// add new condition
			CPNCondition* pCondition=new CPNCondition(SnapOnGrid(point),pLabel);
			pDoc->NetDesign_AddElement(pCondition);
			pDoc->NetDesign_AddElement(pLabel);
			pLabel->SetOwner(pCondition);
			CRect invalid_rect;
			invalid_rect.UnionRect(pLabel->GetRect(),pDoc->SelectElement(pCondition));
			InvalidateRect(invalid_rect,&dc);
			pCondition->UpdatePropertiesDlg();
			pDoc->SetNetModified();
			break;
		}
	case TOOL_EVENT:
		{	// create corresponding label
			CPNLabel* pLabel=new CPNLabel();
			pLabel->SetText(pDoc->GetAutomaticLabelText(CCESimDoc::LABEL_EVENT),&dc);
			// add new event
			CPNEvent* pEvent=new CPNEvent(SnapOnGrid(point),pLabel);
			pDoc->NetDesign_AddElement(pEvent);
			pDoc->NetDesign_AddElement(pLabel);
			pLabel->SetOwner(pEvent);
			CRect invalid_rect;
			invalid_rect.UnionRect(pLabel->GetRect(),pDoc->SelectElement(pEvent));
			InvalidateRect(invalid_rect,&dc);
			pEvent->UpdatePropertiesDlg();
			pDoc->SetNetModified();
			break;
		}
	case TOOL_LABEL:
		{	// create label
			CPNLabel* pLabel=new CPNLabel(SnapOnGrid(point));
			pLabel->SetText(pDoc->GetAutomaticLabelText(CCESimDoc::LABEL_TEXT_NET_DESIGN),&dc);
			pDoc->NetDesign_AddElement(pLabel);
			CRect invalid_rect;
			InvalidateRect(pDoc->SelectElement(pLabel),&dc);
			pLabel->UpdatePropertiesDlg();
			break;
		}
	case TOOL_EDGE:
		{	// draw edge
			m_hit_element=pDoc->NetDesign_ElementAt(point,true,true);
			if((dynamic_cast<CPNCondition*>(m_hit_element)) || (dynamic_cast<CPNEvent*>(m_hit_element)))
			{	// edge must start on condition or event
				m_capture_type=CT_DRAWING_EDGE;
				m_mouse_moved=false;
				SetCapture();
				m_mouse_capture=true;
				m_edge_to=m_edge_from=point;
			}
			break;
		}
	case TOOL_POLYGON:
		{
			CPNPolygon* pPolygon=new CPNPolygon(SnapOnGrid(point));
			pDoc->NetDesign_AddElement(pPolygon);
			CRect invalid_rect;
			InvalidateRect(pDoc->SelectElement(pPolygon),&dc);
			pPolygon->UpdatePropertiesDlg();
			break;
		}
	case TOOL_TOKEN:
		{	// flip initial token
			CPNCondition* pCondition=dynamic_cast<CPNCondition*>(pDoc->NetDesign_ElementAt(point,false,true));
			if(pCondition)
			{
				pCondition->SetToken(!pCondition->GetToken());
				InvalidateRect(pCondition->GetRect(),&dc);
				pDoc->SetModifiedFlag();
				ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
				((CMainFrame*)AfxGetApp()->m_pMainWnd)->UpdatePropertiesDlg();
				pDoc->SetNetModified();
			}
			break;
		}
	default:
		ASSERT(FALSE);
		break;
	}

	CScrollView::OnLButtonDown(nFlags,point);
}

void CCESimNetDesignView::OnMouseMove(UINT nFlags, CPoint point) 
{
	CMainFrame* pFrame=(CMainFrame*)AfxGetApp()->m_pMainWnd;
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	CString status_text;
	CClientDC dc(this);
	OnPrepareDC(&dc);
	dc.DPtoLP(&point);
	CSize window_size=pDoc->NetDesign_GetWindowSize();
	if(point.x<0)point.x=0;
	if(point.y<0)point.y=0;
	if(point.x>window_size.cx)point.x=window_size.cx;
	if(point.y>window_size.cy)point.y=window_size.cy;

	if((m_mouse_capture) && (!m_ignore_move))
	{	// mouse button is down
		switch(m_capture_type)
		{
		case CT_DRAWING_EDGE: // drawing new edge
			m_mouse_moved=true;
			InvertLine(&dc,m_edge_from,m_edge_to);
			InvertLine(&dc,m_edge_from,point);
			m_edge_to=point;
			pFrame->ShowMousePos(point);
			break;
		case CT_DRAWING_SELECT_RECT:
			if(!m_mouse_moved)::SetCursor(::LoadCursor(NULL,IDC_CROSS));
			m_mouse_moved=true;
			InvertRect(&dc,m_first_mouse_pt,m_last_mouse_pt);
			InvertRect(&dc,m_first_mouse_pt,point);
			m_last_mouse_pt=point;
			pFrame->ShowMousePos(point);
			break;
		case CT_DRAGGING_LBL:
		case CT_DRAGGING_COND:
		case CT_DRAGGING_EVNT:
		case CT_DRAGGING_EDGE:
		case CT_DRAGGING_POLYGON:
		case CT_DRAGGING_POLYGON_POINT:
			{
			if(m_simulation_mode)break;
			CPoint move=point-m_last_mouse_pt;
			m_last_mouse_pt=point;
			CPoint position;
			if(m_last_drag_pt==CPoint(-1,-1))m_last_drag_pt=m_hit_element->GetPosition();
			position=m_last_drag_pt;
			position=position+move;
			m_last_drag_pt=position;
			position=SnapOnGrid(position);
			if(position!=m_hit_element->GetPosition())pDoc->SetModifiedFlag();
			if((m_capture_type==CT_DRAGGING_COND) || (m_capture_type==CT_DRAGGING_EVNT))
			{
				CRect invalid_rect;
				invalid_rect.UnionRect(pDoc->NetDesign_RecalculateEdges(m_hit_element),m_hit_element->MoveTo(position));
				InvalidateRect(invalid_rect,&dc);
			}
			else
				InvalidateRect(m_hit_element->MoveTo(position),&dc);
			pFrame->ShowMousePos(position);
			if(!m_simulation_mode && pDoc->IsRulerVisible())DrawRulerPos(&dc,position);
			break;
			}
		case CT_DRAGGING_MULTI_SELECT:
			{
			if(m_simulation_mode)break;
			CPoint move=point-m_last_mouse_pt;
			m_last_mouse_pt=point;
			CRect move_rect,edges_rect;
			move_rect=pDoc->MoveSelectedElementsByVector(move);
			edges_rect=pDoc->NetDesign_RecalculateEdges();
			CRect invalid_rect;
			invalid_rect.UnionRect(move_rect,edges_rect);
			InvalidateRect(invalid_rect,&dc);
			pFrame->ShowMousePos(point);
			break;
			}
		case CT_NONE:
			break;
		default:
			ASSERT(FALSE);
			break;
		}
	}
	else
	{
		pFrame->ShowMousePos(point);
//		if(pDoc->CanDragPoint())
/*		m_guidepoint=point;
		Invalidate(FALSE);*/
	}
	m_ignore_move=false;

/*		m_guidepoint=point;
		Invalidate(FALSE);*/

	CScrollView::OnMouseMove(nFlags, point);
}

void CCESimNetDesignView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if(m_mouse_capture)
	{
		::ReleaseCapture();
	
		CCESimDoc* pDoc=GetDocument();
		ASSERT_VALID(pDoc);
		CClientDC dc(this);
		OnPrepareDC(&dc);
		dc.DPtoLP(&point);
		if(point.x<0)point.x=0;
		if(point.y<0)point.y=0;
		CSize window_size=pDoc->NetDesign_GetWindowSize();
		if(point.x>window_size.cx)point.x=window_size.cx;
		if(point.y>window_size.cy)point.y=window_size.cy;
		
		switch(m_capture_type)
		{
		case CT_DRAWING_EDGE: // drawing edge
			{
			if(!m_mouse_moved)break;
			InvertLine(&dc,m_edge_from,m_edge_to);
			CCESimDoc* pDoc=GetDocument();
			ASSERT_VALID(pDoc);
			CPNElement* pElement=pDoc->NetDesign_ElementAt(point,true,true);
			if( ((dynamic_cast<CPNCondition*>(m_hit_element)) && (dynamic_cast<CPNEvent*>(pElement))) ||
				((dynamic_cast<CPNEvent*>(m_hit_element)) && (dynamic_cast<CPNCondition*>(pElement))) )
			{	// edge is between corresponding elements
				CPNEdge* pEdge=new CPNEdge(m_hit_element,pElement);
				if(pDoc->NetDesign_AddElement(pEdge))
				{
					pEdge->UpdatePropertiesDlg();
					pDoc->SetNetModified();
					InvalidateRect(pDoc->SelectElement(pEdge),&dc);
				}
			}
			break;
			}
		case CT_DRAWING_SELECT_RECT:
			{
			if(!m_mouse_moved)break;
			InvertRect(&dc,m_first_mouse_pt,point);
			CRect rect;
			rect.SetRect(m_first_mouse_pt,point);
			rect.NormalizeRect();
			if(((point.x-m_first_mouse_pt.x)*(point.x-m_first_mouse_pt.x)+(point.y-m_first_mouse_pt.y)*(point.y-m_first_mouse_pt.y))>200)
			{
				if(!m_automatic_simulation)
				{
					InvalidateRect(pDoc->NetDesign_SelectElements(rect),&dc);
					if(!m_simulation_mode)
					{
						ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
						((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//						((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);
					}
				}
			}
			break;
			}
		case CT_DRAGGING_EDGE:
		case CT_DRAGGING_COND:
		case CT_DRAGGING_EVNT:
		case CT_DRAGGING_LBL:
		case CT_DRAGGING_POLYGON:
		case CT_DRAGGING_POLYGON_POINT:
			if(!m_simulation_mode && pDoc->IsRulerVisible())DrawRulerPos(&dc,point,true);
			break;
		case CT_DRAGGING_MULTI_SELECT:
		case CT_NONE:
			break;
		default:
			ASSERT(FALSE);
			break;
		}

		m_capture_type=CT_NONE;
		m_mouse_capture=false;
		m_hit_element=NULL;
	}
	
	CScrollView::OnLButtonUp(nFlags, point);
}

void CCESimNetDesignView::InvertLine(CDC* pDC,const CPoint& from,const CPoint& to)
{	// draw inverted line
	int oldmode=pDC->SetROP2(R2_NOT);
	pDC->MoveTo(from);
	pDC->LineTo(to);
	pDC->SetROP2(oldmode);
}

void CCESimNetDesignView::InvertRect(CDC* pDC,const CPoint& from,const CPoint& to)
{	// draw inverted rect
	pDC->SelectStockObject(NULL_BRUSH);
	int oldmode=pDC->SetROP2(R2_NOT);
	CRect rect;
	rect.SetRect(from,to);
	pDC->Rectangle(&rect);
	pDC->SetROP2(oldmode);
}

void CCESimNetDesignView::OnContextMenu(CWnd* /*pWnd*/, CPoint point) 
{
	CMenu menu;
	VERIFY(menu.LoadMenu(IDR_CONTEXTMENU));
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	CPoint menu_point=point;
	CClientDC dc(this);
	OnPrepareDC(&dc);
	dc.DPtoLP(&point);
	ScreenToClient(&point);
	if(point.x>pDoc->NetDesign_GetWindowSize().cx)return; // out of view
	if(point.y>pDoc->NetDesign_GetWindowSize().cy)return;
	m_last_mouse_pt=point;

	int menu_type=0;
	m_context_element=pDoc->NetDesign_ElementAt(point);
	if(m_context_element)
	{	// 
		bool selected=pDoc->IsElementSelected(m_context_element);
		if(selected && m_simulation_mode)menu_type=1;
		if(selected && m_context_element->IsKindOf(RUNTIME_CLASS(CPNEdge)))menu_type=3;
		if(selected && m_context_element->IsKindOf(RUNTIME_CLASS(CPNPolygon)))menu_type=4;
	}
	if((!m_simulation_mode) || (menu_type!=0))
	{
		CMenu* pContextMenu=menu.GetSubMenu(menu_type);
		ASSERT(pContextMenu!=NULL);
		pContextMenu->TrackPopupMenu(TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_RIGHTBUTTON,menu_point.x,menu_point.y,AfxGetMainWnd());
	}
}

BOOL CCESimNetDesignView::OnSetCursor(CWnd* /*pWnd*/, UINT /*nHitTest*/, UINT /*message*/) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	char* cursor=IDC_ARROW;
	if(m_capture_type==CT_DRAWING_SELECT_RECT)
	{
		cursor=IDC_CROSS;
	}
	else
	{
		if(m_simulation_mode)
		{
			cursor=IDC_ARROW;
		}
		else
		{
			CPoint point;
			GetCursorPos(&point);
			ScreenToClient(&point);
			if(pDoc->CanDragPoint(point))
				cursor=IDC_SIZEALL;
			else
				cursor=(m_tool_selected==TOOL_SELECT)? IDC_ARROW : IDC_CROSS;
		}
	}
	::SetCursor(AfxGetApp()->LoadStandardCursor(cursor));
	return TRUE;
}

CPoint CCESimNetDesignView::SnapOnGrid(CPoint point)
{	// modify point to snap on grid
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	if(pDoc->IsGridVisible())
	{
		point.x=((point.x+(pDoc->GetGridSpan()/2))/pDoc->GetGridSpan())*pDoc->GetGridSpan();
		point.y=((point.y+(pDoc->GetGridSpan()/2))/pDoc->GetGridSpan())*pDoc->GetGridSpan();
	}
	return point;
}

void CCESimNetDesignView::OnEditPlacetofront() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	if(m_simulation_mode || !pDoc->IsElementSelected())return;
	InvalidateRect(pDoc->NetDesign_PlaceSelectedElements(CCESimDoc::PLACE_FRONT));
}

void CCESimNetDesignView::OnEditPlacetoback() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	if(m_simulation_mode || !pDoc->IsElementSelected())return;
	InvalidateRect(pDoc->NetDesign_PlaceSelectedElements(CCESimDoc::PLACE_BACK));
}

void CCESimNetDesignView::OnUpdateEditPlacetoback(CCmdUI* pCmdUI) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->Enable(!m_simulation_mode && pDoc->IsElementSelected());
}

void CCESimNetDesignView::OnUpdateEditPlacetofront(CCmdUI* pCmdUI) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->Enable(!m_simulation_mode && pDoc->IsElementSelected());
}

BOOL CCESimNetDesignView::OnEraseBkgnd(CDC* pDC) 
{
	if(m_erase_background)
		return CScrollView::OnEraseBkgnd(pDC);
	else
		return TRUE;	
}

void CCESimNetDesignView::OnNcPaint()
{
	m_erase_background=true;
    CWnd::OnNcPaint();
}

void CCESimNetDesignView::OnEditDelete() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	if(m_simulation_mode || !pDoc->NetDesign_CanRemoveElements())return;
	InvalidateRect(pDoc->NetDesign_RemoveSelectedElements());
	pDoc->SetNetModified();
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE);
//	((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP);
}

void CCESimNetDesignView::OnUpdateEditDelete(CCmdUI* pCmdUI) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->Enable(!m_simulation_mode && pDoc->NetDesign_CanRemoveElements());
}

void CCESimNetDesignView::OnEdgeDeletepoint() 
{
	bool enable=false;
	CPNEdge* pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)
	{
		enable=(pEdge->IsDragging(m_last_mouse_pt) && pEdge->CanDeletePoint());
	}
	if(!enable)return;

	pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)
	{
		CRect prevrect=pEdge->GetRect();
		pEdge->DeletePoint();
		CRect invalid_rect;
		invalid_rect.UnionRect(prevrect,pEdge->GetRect());
		InvalidateRect(invalid_rect);
		CCESimDoc* pDoc=GetDocument();
		ASSERT_VALID(pDoc);
		pDoc->SetModifiedFlag();
	}
}

void CCESimNetDesignView::OnUpdateEdgeDeletepoint(CCmdUI* pCmdUI) 
{
	bool enable=false;
	CPNEdge* pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)
	{
		enable=(pEdge->IsDragging(m_last_mouse_pt) && pEdge->CanDeletePoint());
	}
	pCmdUI->Enable(enable);
}

void CCESimNetDesignView::OnEdgeDeleteallpoints() 
{
	bool enable=false;
	CPNEdge* pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)enable=pEdge->IsMultipoint();
	if(!enable)return;

	pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)
	{
		CRect prevrect=pEdge->GetRect();
		pEdge->DeleteAllPoints();
		CRect invalid_rect;
		invalid_rect.UnionRect(prevrect,pEdge->GetRect());
		InvalidateRect(invalid_rect);
		CCESimDoc* pDoc=GetDocument();
		ASSERT_VALID(pDoc);
		pDoc->SetModifiedFlag();
	}
}

void CCESimNetDesignView::OnUpdateEdgeDeleteallpoints(CCmdUI* pCmdUI) 
{
	bool enable=false;
	CPNEdge* pEdge=dynamic_cast<CPNEdge*>(m_context_element);
	if(pEdge)enable=pEdge->IsMultipoint();
	pCmdUI->Enable(enable);
}

void CCESimNetDesignView::OnPolygonDeletepoint() 
{
	bool enable=false;
	CPNPolygon* pPolygon=dynamic_cast<CPNPolygon*>(m_context_element);
	if(pPolygon)
	{
		enable=(pPolygon->IsDragging(m_last_mouse_pt) && pPolygon->CanDeletePoint());
	}
	if(!enable)return;

	pPolygon=dynamic_cast<CPNPolygon*>(m_context_element);
	if(pPolygon)
	{
		CRect prevrect=pPolygon->GetRect();
		pPolygon->DeletePoint();
		CRect invalid_rect;
		invalid_rect.UnionRect(prevrect,pPolygon->GetRect());
		InvalidateRect(invalid_rect);
		CCESimDoc* pDoc=GetDocument();
		ASSERT_VALID(pDoc);
		pDoc->SetModifiedFlag();
	}
}

void CCESimNetDesignView::OnUpdatePolygonDeletepoint(CCmdUI* pCmdUI) 
{
	bool enable=false;
	CPNPolygon* pPolygon=dynamic_cast<CPNPolygon*>(m_context_element);
	if(pPolygon)
	{
		enable=(pPolygon->IsDragging(m_last_mouse_pt) && pPolygon->CanDeletePoint());
	}
	pCmdUI->Enable(enable);
}

LRESULT CCESimNetDesignView::OnUpdateView(WPARAM wparam,LPARAM /*lparam*/)
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pDoc->SetModifiedFlag();
	pDoc->SetNetModified();
	CClientDC dc(this);
	OnPrepareDC(&dc);
	CPNElement* pElement;
	pElement=pDoc->GetFirstSelectedElement();
	if(!pElement)return 0;
	switch(wparam)
	{
	case UPDATE_VIEW_ELEMENT:
		InvalidateRect(pElement->GetRect(),&dc);
		break;
	case UPDATE_VIEW_TEXT:
		{
		CRect prevrect=pElement->GetRect();
		CPNLabel* pLabel;
		pLabel=dynamic_cast<CPNLabel*>(pElement);
		if(pLabel)pLabel->UpdateText(&dc);
		CRect invalid_rect;
		invalid_rect.UnionRect(prevrect,pElement->GetRect());
		InvalidateRect(invalid_rect,&dc);
		break;
		}
	case UPDATE_VIEW_LABEL_EVNT:
		{
		CPNEvent* m_pEvent;
		m_pEvent=dynamic_cast<CPNEvent*>(pElement);
		if(m_pEvent)
		{
			CPNLabel* pLabel;
			pLabel=m_pEvent->GetLabel();
			if(pLabel)
			{
				CRect prevrect=pLabel->GetRect();
				pLabel->UpdateText(&dc);
				CRect invalid_rect;
				invalid_rect.UnionRect(prevrect,pLabel->GetRect());
				InvalidateRect(invalid_rect,&dc);
			}
		}
		break;
		}
	case UPDATE_VIEW_LABEL_COND:
		{
		CPNCondition* m_pCondition;
		m_pCondition=dynamic_cast<CPNCondition*>(pElement);
		if(m_pCondition)
		{
			CPNLabel* pLabel;
			pLabel=m_pCondition->GetLabel();
			if(pLabel)
			{
				CRect prevrect=pLabel->GetRect();
				pLabel->UpdateText(&dc);
				CRect invalid_rect;
				invalid_rect.UnionRect(prevrect,pLabel->GetRect());
				InvalidateRect(invalid_rect,&dc);
			}
		}
		break;
		}
	default:
		ASSERT(FALSE);
		break;
	}
	return 0;
}

void CCESimNetDesignView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	if(!SimulationRealizeStep())return;

	CScrollView::OnLButtonDblClk(nFlags, point);
}

int CCESimNetDesignView::AnimateStep()
{	// vraci 0 pri OK, 1 pri ukonceni simulace, 2 pri ukonceni programu
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	m_animating_token=true;
	DWORD token_delay=pDoc->GetTokenAnimationDelay();
	pDoc->PrepareTokenAnimation(CCESimDoc::PRE_SET);
	pDoc->RealizeStep(CCESimDoc::RLZ_DIFF_CASE);
	DWORD start_time=::GetTickCount();
	while(pDoc->StepTokenAnimation())
	{	// tokens from preset
		while((::GetTickCount()-start_time)<token_delay)
		{
			Sleep(SLEEP_DELAY);
			MSG msg;
			while(PeekMessage(&msg,NULL,0,0,PM_REMOVE))
			{
				if(msg.message==WM_QUIT)
				{
					PostQuitMessage(0);
					return 2;
				}
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			if(!m_simulation_mode)return 1;
		}
		Invalidate(FALSE); // aby se dalo plynule regulovat rychlost tokenu i behem jeho pohybu
		token_delay=pDoc->GetTokenAnimationDelay();
		start_time=::GetTickCount();
	}
	pDoc->RealizeStep(CCESimDoc::RLZ_SWITCH_CONDITON);
	pDoc->PrepareTokenAnimation(CCESimDoc::POST_SET);
	start_time=::GetTickCount();
	while(pDoc->StepTokenAnimation())
	{	// token from postset
		while((::GetTickCount()-start_time)<token_delay)
		{
			Sleep(SLEEP_DELAY);
			MSG msg;
			while(PeekMessage(&msg,NULL,0,0,PM_REMOVE))
			{
				if(msg.message==WM_QUIT)
				{
					PostQuitMessage(0);
					return 2;
				}
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			if(!m_simulation_mode)return 1;
		}
		Invalidate(FALSE);
		token_delay=pDoc->GetTokenAnimationDelay();
		start_time=::GetTickCount();
	}
	pDoc->RealizeStep(CCESimDoc::RLZ_UNION_CASE);
	pDoc->DeselectAllElements();
	m_animating_token=false;
	Invalidate(FALSE);
	return 0;
}

void CCESimNetDesignView::OnSimulationRealizeStep() 
{
	SimulationRealizeStep();
}

bool CCESimNetDesignView::SimulationRealizeStep() 
{	// vrati true pri OK, a false pro ukonceni programu
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	if(!m_simulation_mode || m_automatic_simulation || !pDoc->IsElementSelected())return true;
	if(pDoc->TestRealizeStep())
	{
		ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
		CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;
		frame->EnableSimulationButtons(false,false,false,false);
		int ret=AnimateStep();
		if(ret==2)return false; // ukonceni programu
		frame->EnableSimulationButtons(true,true,false,true);
	}
	return true;
}

void CCESimNetDesignView::OnUpdateSimulationRealizeStep(CCmdUI* pCmdUI) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pCmdUI->Enable(m_simulation_mode && !m_automatic_simulation && pDoc->IsElementSelected());
}

void CCESimNetDesignView::OnSimulationResetsystem() 
{
	if(!m_simulation_mode || m_automatic_simulation || m_animating_token)return;
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pDoc->ResetSystem();
	Invalidate(FALSE);
}

void CCESimNetDesignView::OnUpdateSimulationResetsystem(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_simulation_mode && !m_animating_token && !m_automatic_simulation);
}

void CCESimNetDesignView::OnBtnSimStart() 
{	
	if(!m_automatic_simulation)
	{	// start automatic simulation
		AutomaticSimulation();
	}
}

void CCESimNetDesignView::AutomaticSimulation()
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pDoc->DeselectAllElements();
	_AddOutputListString(OUTPUT_SIMULATION,"Starting automatic simulation...");
	m_automatic_simulation=true;
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;
	frame->EnableSimulationButtons(false,false,true,false);
	bool select=true;
	while(m_automatic_simulation && select)
	{
		select=pDoc->AutoSelectStep();
		if(select) // jestli lze pokracovat z daneho pripadu v simulaci
		{
			int ret=AnimateStep();
			if(ret==1)m_automatic_simulation=false; // ukonceni simulace
			if(ret==2)return; // ukonceni programu
		}
		if(pDoc->TestBreakpoint())break;
	}
	CString out_str;
	out_str="Automatic simulation stopped";
	if(!select)out_str+=", no executable event found";
	_AddOutputListString(OUTPUT_SIMULATION,out_str);
	m_automatic_simulation=false;
	frame->EnableSimulationButtons(true,true,false,true);
}

LRESULT CCESimNetDesignView::OnSimulationSpeed(WPARAM wparam,LPARAM /*lparam*/)
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	pDoc->SetTokenAnimationDelay(wparam);
	return 0;
}

void CCESimNetDesignView::OnBtnSimReset() 
{
	OnSimulationResetsystem();
}

void CCESimNetDesignView::OnBtnSimStep() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	pDoc->AutoSelectStep(false);
	OnSimulationRealizeStep();
}

void CCESimNetDesignView::OnBtnSimStop() 
{	// stop automatic simulation
	m_automatic_simulation=false;
}

void CCESimNetDesignView::OnUpdate(CView* /*pSender*/,LPARAM /*lHint*/,CObject* /*pHint*/)
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

	m_simulation_mode=pDoc->IsSimulationMode();
	m_automatic_simulation=false;
	m_animating_token=false;
	if(m_simulation_mode)
		m_tool_selected=TOOL_SELECT;
	Invalidate(FALSE);
}

void CCESimNetDesignView::OnAnalysisDonetcomplement() 
{	// 
	if(!m_simulation_mode)return;
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);

/*	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowOutputBar();*/

	CClientDC dc(this);
	OnPrepareDC(&dc);
	pDoc->NetDesign_DoNetComplement(&dc);
	pDoc->SetNetModified();
}

void CCESimNetDesignView::OnUpdateAnalysisDonetcomplement(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_simulation_mode);
}

void CCESimNetDesignView::OnViewPropertiesNetdesign() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	CDlgProperties dlg;
	dlg.SetWindowSize(pDoc->NetDesign_GetWindowSize());
	if(dlg.DoModal()==IDOK)
	{
		pDoc->NetDesign_SetWindowSize(dlg.GetWindowSize());
		SetScrollSizes(MM_TEXT,pDoc->NetDesign_GetWindowSize());
		pDoc->SetModifiedFlag();
		Invalidate(FALSE);
	}
}

void CCESimNetDesignView::OnUpdateViewPropertiesNetdesign(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!m_simulation_mode);
}

void CCESimNetDesignView::SwitchedTo()
{
	if(m_simulation_mode)
	{
		CCESimDoc* pDoc=GetDocument();
		ASSERT_VALID(pDoc);
		pDoc->UpdateNet();
		ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
		((CMainFrame*)AfxGetApp()->m_pMainWnd)->ShowPropertiesDlg(CMainFrame::PROPDLG_SIMULATION);
	}
}

void CCESimNetDesignView::OnNetdesignExporttoemf() 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	char szFilter[]="Net Design (*.emf)|*.emf|All files (*.*)|*.*||";
	CFileDialog filedlg(FALSE,"",pDoc->GetTitle(),OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT|OFN_PATHMUSTEXIST,szFilter,NULL);
	if(filedlg.DoModal()==IDOK)
	{
		m_export=true;
		m_export_path=filedlg.GetPathName();
		Invalidate(FALSE);
	}
}

void CCESimNetDesignView::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if(nChar==VK_DELETE)OnEditDelete();
	
	CScrollView::OnKeyUp(nChar, nRepCnt, nFlags);
}

BOOL CCESimNetDesignView::OnScroll(UINT nScrollCode, UINT nPos, BOOL bDoScroll) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	if(!m_simulation_mode && pDoc->IsRulerVisible())Invalidate(FALSE);

	return CScrollView::OnScroll(nScrollCode, nPos, bDoScroll);
}

BOOL CCESimNetDesignView::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
	CCESimDoc* pDoc=GetDocument();
	ASSERT_VALID(pDoc);
	if(!m_simulation_mode && pDoc->IsRulerVisible())Invalidate(FALSE);
	
	return CScrollView::OnMouseWheel(nFlags, zDelta, pt);
}
