function [thr, H0, H1] = calibrate(detector)
% CALIBDETECTOR calculates rejection thresholds in each detector stage.
%
%   thr = CALIBDETECTOR(H0, H1, alpha);
%
% Detectors trained with WALDBOOST_LBP_TRAIN have constant rejection
% thresholds (typically set to -1). Calibration of these detectors is done
% by shifting responses by a constant amount (e.g. 0.05) after each stage.
% Such approach is not optimal and may lead to bad detection performance
% (lower recall and high false positive rate). This calibration method uses
% WaldBoost approach -- it selects such thresholds that leads to a defined
% false negative rate (alpha) and discards as many false positives as
% possible. After this, no other calibation is required.
%
% Input:
%  H0, H1   [MxT] and [NxT] matrices with cumulative responses of negative
%           and positive training samples.
%  alpha    [0.01] Target false negative rate
%
% Output:
%  thr      [1xT] single with rejection thresholds.
%
% Example:
%  thr = CALIBDETECTOR(H0, H1, 0.05);
%  detector.clf.thr = thr;
%  

detector.opts.nPos = 5000;
detector.opts.nNeg = 15000;

%detector = lbpModify(detector, 'cascThr',-1, 'cascCal',0.000);

Is0 = sampleWins( detector, 1, 0 );
X0 = chnsCompute1( Is0, detector.opts ); clear Is0;
X0 = lbpdetector.lbp(X0, detector.clf.ftr) + 1;
H0 = cell(1, size(X0,2));
for i = 1:size(X0,2), H0{i} = detector.clf.hs(X0(:,i),i); end;
H0 = horzcat(H0{:}); H0 = cumsum(H0,2); clear X0;

Is1 = sampleWins( detector, 1, 1 );
X1 = chnsCompute1( Is1, detector.opts ); clear Is1;
X1 = lbpdetector.lbp(X1, detector.clf.ftr) + 1;
H1 = cell(1, size(X1,2));
for i = 1:size(X1,2), H1{i} = detector.clf.hs(X1(:,i),i); end;
H1 = horzcat(H1{:}); H1 = cumsum(H1,2); clear X1;


if nargin < 3, alpha = 0.01; end;
T = size(H0,2); T1 = size(H1,2); assert(T==T1);
A = 1/alpha; thr = -inf(1,T,'single');
for t = 1:T
    if size(H0,1) <= 1, break; end;
    r0 = min([H0(:,t);H1(:,t)]); r1 = max([H0(:,t);H1(:,t)]);
    b = linspace(r0, r1, 200);
    HT0 = cumsum(histc(H0(:,t),b)); HT1 = cumsum(histc(H1(:,t),b));
    HT0 = HT0./HT0(end); HT1 = HT1./HT1(end );
    R = HT0 ./ HT1;
    ti = find(R > A, 1, 'last');
    if ~isempty(ti)
        thr(t) = b(ti);
        H0(H0(:,t) < thr(t),:) = [];
        H1(H1(:,t) < thr(t),:) = [];
    end
end

end


function [Is,IsOrig] = sampleWins( detector, stage, positive )
% Load or sample windows for training detector.
opts=detector.opts; start=clock;
if( positive ), n=opts.nPos; else n=opts.nNeg; end
if( positive ), crDir=opts.posWinDir; else crDir=opts.negWinDir; end
if( exist(crDir,'dir') && stage==0 )
  % if window directory is specified simply load windows
  fs=bbGt('getFiles',{crDir}); nImg=length(fs); assert(nImg>0);
  if(nImg>n), fs=fs(:,randSample(nImg,n)); else n=nImg; end
  for i=1:n, fs{i}=[{opts.imreadf},fs(i),opts.imreadp]; end
  Is=cell(1,n); parfor i=1:n, Is{i}=feval(fs{i}{:}); end
else
  % sample windows from full images using sampleWins1()
  hasGt=positive||isempty(opts.negImgDir); fs={opts.negImgDir};
  if(hasGt), fs={opts.posImgDir,opts.posGtDir}; end
  fs=bbGt('getFiles',fs); nImg=size(fs,2); assert(nImg>0);
  if(~isinf(n)), fs=fs(:,randperm(nImg)); end; Is=cell(nImg*1000,1);
  diary('off'); tid=ticStatus('Sampling windows',1,30); k=0; i=0; batch=16;
  while( i<nImg && k<n )
    batch=min(batch,nImg-i); Is1=cell(1,batch);
    parfor j=1:batch, ij=i+j;
      I = feval(opts.imreadf,fs{1,ij},opts.imreadp{:}); %#ok<PFBNS>
      [h,w,~] = size(I); if any([h,w]<opts.modelDsPad), continue; end; %%%
      gt=[]; if(hasGt), [~,gt]=bbGt('bbLoad',fs{2,ij},opts.pLoad); end
      Is1{j} = sampleWins1( I, gt, detector, stage, positive );
    end
    Is1=[Is1{:}]; k1=length(Is1); Is(k+1:k+k1)=Is1; k=k+k1;
    if(k>n), Is=Is(randSample(k,n)); k=n; end
    i=i+batch; tocStatus(tid,max(i/nImg,k/n));
  end
  Is=Is(1:k); diary('on');
  fprintf('Sampled %i windows from %i images.\n',k,i);
end
% optionally jitter positive windows
if(length(Is)<2), Is={}; return; end
nd=ndims(Is{1})+1; Is=cat(nd,Is{:}); IsOrig=Is;
if( positive && isstruct(opts.pJitter) )
  opts.pJitter.hasChn=(nd==4); Is=jitterImage(Is,opts.pJitter);
  ds=size(Is); ds(nd)=ds(nd)*ds(nd+1); Is=reshape(Is,ds(1:nd));
end
% make sure dims are divisible by shrink and not smaller than modelDsPad
ds=size(Is); cr=rem(ds(1:2),opts.pPyramid.pChns.shrink); s=floor(cr/2)+1;
e=ceil(cr/2); Is=Is(s(1):end-e(1),s(2):end-e(2),:,:); ds=size(Is);
if(any(ds(1:2)<opts.modelDsPad)), error('Windows too small.'); end
% optionally save windows to disk and update log
nm=[opts.name 'Is' int2str(positive) 'Stage' int2str(stage)];
if( opts.winsSave ), save(nm,'Is','-v7.3'); end
fprintf('Done sampling windows (time=%.0fs).\n',etime(clock,start));
diary('off'); diary('on');
end

function Is = sampleWins1( I, gt, detector, stage, positive )
% Sample windows from I given its ground truth gt.
opts=detector.opts; shrink=opts.pPyramid.pChns.shrink;
modelDs=opts.modelDs; modelDsPad=opts.modelDsPad;
if( positive ), bbs=gt; bbs=bbs(bbs(:,5)==0,:); else
  if( stage==0 )
    % generate candidate bounding boxes in a grid
    [h,w,~]=size(I); h1=modelDs(1); w1=modelDs(2);
    n=opts.nPerNeg; ny=sqrt(n*h/w); nx=n/ny; ny=ceil(ny); nx=ceil(nx);
    [xs,ys]=meshgrid(linspace(1,w-w1,nx),linspace(1,h-h1,ny));
    bbs=[xs(:) ys(:)]; bbs(:,3)=w1; bbs(:,4)=h1; bbs=bbs(1:n,:);
  else
    % run detector to generate candidate bounding boxes
    bbs=lbpDetect(I,detector); [~,ord]=sort(bbs(:,5),'descend');
    bbs=bbs(ord(1:min(end,opts.nPerNeg)),1:4);
  end
  if( ~isempty(gt) )
    % discard any candidate negative bb that matches the gt
    n=size(bbs,1); keep=false(1,n);
    for i=1:n, keep(i)=all(bbGt('compOas',bbs(i,:),gt,gt(:,5))<.1); end
    bbs=bbs(keep,:);
  end
end
% grow bbs to a large padded size and finally crop windows
modelDsBig=max(8*shrink,modelDsPad)+max(2,ceil(64/shrink))*shrink;
r=modelDs(2)/modelDs(1); assert(all(abs(bbs(:,3)./bbs(:,4)-r)<1e-5));
r=modelDsBig./modelDs; bbs=bbApply('resize',bbs,r(1),r(2));
Is=bbApply('crop',I,bbs,'replicate',modelDsBig([2 1]));
end

function chns = chnsCompute1( Is, opts )
% Compute single scale channels of dimensions modelDsPad.
if(isempty(Is)), chns=[]; return; end
fprintf('Extracting features... '); start=clock; fs=opts.filters;
pChns=opts.pPyramid.pChns; smooth=opts.pPyramid.smooth;
dsTar=opts.modelDsPad/pChns.shrink; ds=size(Is); ds(1:end-1)=1;
Is=squeeze(mat2cell2(Is,ds)); n=length(Is); chns=cell(1,n);
parfor i=1:n
  C=chnsCompute(Is{i},pChns); C=convTri(cat(3,C.data{:}),smooth);
  if(~isempty(fs)), C=repmat(C,[1 1 size(fs,4)]);
    for j=1:size(C,3), C(:,:,j)=conv2(C(:,:,j),fs(:,:,j),'same'); end; end
  if(~isempty(fs)), C=imResample(C,.5); shr=2; else shr=1; end
  ds=size(C); cr=ds(1:2)-dsTar/shr; s=floor(cr/2)+1; e=ceil(cr/2);
  C=C(s(1):end-e(1),s(2):end-e(2),:); chns{i}=C;
end; chns=cat(4,chns{:});
fprintf('done (time=%.0fs).\n',etime(clock,start));
end