/*
								+----------------------------------+
								|                                  |
								|  *** Common spline template ***  |
								|                                  |
								|   Copyright  -tHE SWINe- 2008   |
								|                                  |
								|             Spline.h             |
								|                                  |
								+----------------------------------+
*/

#pragma once
#ifndef __GENERIC_SPLINE_TEMPLATE_INCLUDED
#define __GENERIC_SPLINE_TEMPLATE_INCLUDED

/**
 *	@file lml/Spline.h
 *	@date 2008
 *	@author -tHE SWINe-
 *	@brief common spline template
 *
 *	@date 2009-01-15
 *
 *	fixed error in CSpline::ErasePoints() where cached lengths
 *	were erased instead of the points themselves
 *
 *	@date 2009-05-23
 *
 *	removed all instances of std::vector::reserve and replaced them by stl_ut::Reserve_*
 *
 *	@date 2009-10-20
 *
 *	fixed some warnings when compiling under VC 2005, implemented "Security
 *	Enhancements in the CRT" for VC 2008. compare against MyProjects_2009-10-19_
 *
 *	@date 2012-06-19
 *
 *	Moved multiple inclusion guard before file documentation comment.
 *
 */

#include "../NewFix.h"
#include "../CallStack.h"
#include <vector>
#include "../Numerical.h"

/**
 *	@brief simple generic spline storage template
 *
 *	@tparam CPointClass is type of points the spline is defined on
 *	@tparam CSplineImpl is a particular spline implementation
 */
template <class CPointClass, class CSplineImpl>
class CSplineBase : public CSplineImpl {
public:
	typedef CPointClass _TyPoint; /**< @brief type of points the spline is defined on */

protected:
	/**
	 *	@brief tangent magnitude adaptor for integrator (gives arc length)
	 */
	class CTangentMag {
	protected:
		const CSplineBase &m_r_spline;
		size_t m_n_arc;

	public:
		inline CTangentMag(const CSplineBase &r_spline, size_t n_arc)
			:m_r_spline(r_spline), m_n_arc(n_arc)
		{}

		inline float operator ()(float f_t) const
		{
			return m_r_spline.v_Arc_Derivative(f_t, m_n_arc).f_Length();
		}
	};

	std::vector<_TyPoint> m_point_list;

public:
	/**
	 *	@brief default constructor (has no effect)
	 */
	inline CSplineBase()
	{}

	/**
	 *	@brief constructor
	 *	@param[in] r_point_list is list of spline points, it is copied locally
	 *	@note This function throws std::bad_alloc.
	 */
	inline CSplineBase(const std::vector<_TyPoint> &r_point_list) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.end(), r_point_list.begin(), r_point_list.end());
		// copy points
	}

	/**
	 *	@brief constructor
	 *
	 *	@param[in] p_point_begin is iterator pointing to the first spline point
	 *	@param[in] p_point_end is iterator pointing to one past the last spline point
	 *
	 *	@note This function throws std::bad_alloc.
	 */
	template <class CPointIter>
	inline CSplineBase(CPointIter p_point_begin, CPointIter p_point_end) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.end(), p_point_begin, p_point_end);
		// copy points
	}

	/**
	 *	@brief copy-constructor
	 *	@param[in] r_other is spline to copy from
	 *	@return Returns reference to this.
	 *	@note This function throws std::bad_alloc.
	 */
	inline CSplineBase(const CSplineBase &r_other) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.end(),
			r_other.m_point_list.begin(), r_other.m_point_list.end());
		// copy points
	}

	/**
	 *	@brief copy-operator
	 *	@param[in] r_other is spline to copy from
	 *	@return Returns reference to this.
	 *	@note This function throws std::bad_alloc.
	 */
	inline CSplineBase &operator =(const CSplineBase &r_other) // throw(std::bad_alloc)
	{
		m_point_list.clear();
		m_point_list.insert(m_point_list.end(),
			r_other.m_point_list.begin(), r_other.m_point_list.end());
		// copy points

		return *this;
	}

	/**
	 *	@brief gets number of spline points
	 *	@return Returns number of spline points.
	 */
	inline size_t n_Point_Num() const
	{
		return m_point_list.size();
	}

	/**
	 *	@brief gets a spline point
	 *	@param[in] n_index is zero-based index of the spline point
	 *	@return Returns const reference to the selected spline point
	 */
	inline const _TyPoint &r_Point(size_t n_index) const
	{
		return m_point_list[n_index];
	}

	/**
	 *	@brief gets a spline point
	 *	@param[in] n_index is zero-based index of the spline point
	 *	@return Returns reference to the selected spline point
	 *	@note This resets cached spline segment lengths.
	 */
	inline _TyPoint &r_Point(size_t n_index)
	{
		return m_point_list[n_index];
	}

	/**
	 *	@brief erases points inside a range
	 *
	 *	@param[in] n_begin is zero-based index of the first point to be erased
	 *	@param[in] n_end is zero-based index of one past the last point
	 *		to be erased, or -1 as all points until the end
	 */
	inline void ErasePoints(size_t n_begin = 0, size_t n_end = size_t(-1))
	{
		if(n_end != size_t(-1)) {
			m_point_list.erase(m_point_list.begin() + n_begin,
				m_point_list.begin() + n_end);
		} else {
			m_point_list.erase(m_point_list.begin() + n_begin,
				m_point_list.end());
		}
		// erase points
	}

	/**
	 *	@brief inserts an array of points and places them at a specified location
	 *
	 *	@param[in] n_insert_before is zero-based index of spline point where to insert new points
	 *	@param[in] p_point_begin is iterator pointing to the first spline point
	 *	@param[in] p_point_end is iterator pointing to one past the last spline point
	 *
	 *	@note This function throws std::bad_alloc.
	 */
	template <class CPointIter>
	void Insert(size_t n_insert_before, CPointIter p_point_begin, CPointIter p_point_end) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.begin() + n_insert_before, p_point_begin, p_point_end);
		// add new points
	}

	/**
	 *	@brief inserts an array of points and places them at the end
	 *
	 *	@param[in] p_point_begin is iterator pointing to the first spline point
	 *	@param[in] p_point_end is iterator pointing to one past the last spline point
	 *
	 *	@note This function throws std::bad_alloc.
	 */
	template <class CPointIter>
	void Insert_Back(CPointIter p_point_begin, CPointIter p_point_end) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.end(), p_point_begin, p_point_end);
		// add new points
	}

	/**
	 *	@brief inserts a single point at a specified location
	 *
	 *	@param[in] n_insert_before is zero-based index of spline point where to insert new points
	 *	@param[in] t_point is value of the point to be inserted 
	 *
	 *	@note This function throws std::bad_alloc.
	 */
	inline bool Insert(size_t n_insert_before, _TyPoint t_point) // throw(std::bad_alloc)
	{
		m_point_list.insert(m_point_list.begin() + n_insert_before, t_point);
	}

	/**
	 *	@brief inserts a single point at the end
	 *	@param[in] t_point is value of the point to be inserted 
	 *	@note This function throws std::bad_alloc.
	 */
	inline bool PushBack(_TyPoint t_point) // throw(std::bad_alloc)
	{
		m_point_list.push_back(t_point);
	}

	/**
	 *	@brief calculates spline length
	 *	@param[in] f_max_error is maximum relative error
	 *	@return Returns spline length with the specified maximal error.
	 *	@note This calls f_Arc_Length() in a loop, not particularly efficient.
	 */
	float f_Length(float f_max_error = 1e-3f) const
	{
		size_t n_arc_num;
		if(!(n_arc_num = n_Arc_Num()))
			return 0;

		f_max_error /= n_arc_num; // this makes it more precise, not less
		// error multiplies with the number of arcs

		float f_length = 0;
		for(size_t i = 0, n = n_arc_num; i < n; ++ i)
			f_length += f_Arc_Length(i, f_max_error);
		// sum up arc lengths, don't cache them

		return f_length;
	}

	/**
	 *	@brief calculates spline length and returns arc lengths
	 *
	 *	@param[out] r_arc_length is filled with arc lengths upon return
	 *	@param[in] f_max_error is maximum relative error
	 *
	 *	@return Returns spline length with the specified maximal error.
	 *
	 *	@note This calls f_Arc_Length() in a loop, not particularly efficient.
	 */
	float f_Length(std::vector<float> &r_arc_length, float f_max_error = 1e-3f) const //throw(std::bad_alloc)
	{
		size_t n_arc_num;
		if(!(n_arc_num = n_Arc_Num()))
			return 0;

		f_max_error /= n_arc_num; // this makes it more precise, not less
		// error multiplies with the number of arcs

		float f_length = 0;
		for(size_t i = 0, n = n_arc_num; i < n; ++ i) {
			float f_arc = f_Arc_Length(i, f_max_error);
			r_arc_length.push_back(f_arc);
			f_length += f_arc;
		}
		// sum up arc lengths, don't cache them

		return f_length;
	}

	/**
	 *	@brief calculates spline arc length
	 *
	 *	@param[in] n_arc is zero-based index of spline arc
	 *	@param[in] f_max_error is maximum relative error
	 *
	 *	@return Returns length of the selected spline arc with the specified maximal error.
	 */
	float f_Arc_Length(size_t n_arc, float f_max_error = 1e-3f) const
	{
		_ASSERTE(n_arc < n_Arc_Num());
		return f_Integrate(CTangentMag(*this, n_arc), 0, 1, f_max_error);
	}

	/**
	 *	@brief calculates index of arc and a local position given position on spline
	 *
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@param[out] r_f_local_t is local position along the particular arc (range [0, 1])
	 *	@param[in] f_spline_length is total spline length
	 *	@param[in] r_arc_length is a list of lengths of all the spline arcs
	 *
	 *	@return Returns (zero-based) index of arc at the given position along the spline.
	 */
	size_t n_Arc(float f_t, float &r_f_local_t, float f_spline_length,
		const std::vector<float> &r_arc_length) const
	{
		size_t n_arc_num;
		if(!(n_arc_num = n_Arc_Num()))
			return size_t(-1);
		_ASSERTE(r_arc_length.size() == n_arc_num);

		if(f_t <= 0) {
			r_f_local_t = 0;
			return 0;
		} else if(f_t >= 1) {
			r_f_local_t = 1;
			return n_arc_num - 1;
		}
		// clamp t to range [0, 1]

		float f_dist = f_t * f_spline_length/*f_Length(f_max_error)*/;
		for(size_t i = 0, n = n_arc_num; i < n; ++ i) {
			float f_arc_len = r_arc_length[i]/*f_Arc_Length(i, f_max_error)*/;
			if(f_dist < f_arc_len) {
				r_f_local_t = f_dist / f_arc_len;
				return i;
			}
			f_dist -= f_arc_len;
		}
		// find arc by spline length

		r_f_local_t = 1;
		return n_arc_num - 1;
		// last arc was hit (should happen quite seldom)
	}

	/**
	 *	@brief gets number of spline arcs
	 *	@return Returns number of spline arcs.
	 *	@note this function is supposed to be implemented in the final spline class
	 */
	inline size_t n_Arc_Num() const
	{
		return CSplineImpl::n_Arc_Num(); // these do not have to be virtual anymore
	}

	/**
	 *	@brief calculates position on a spline arc
	 *
	 *	@param[in] f_t is position on a spline arc in range [0, 1]
	 *	@param[in] n_arc is zero-based index of spline arc
	 *
	 *	@return Returns point at the given position along the selected arc.
	 *
	 *	@note This function is supposed to be implemented in the final spline class.
	 */
	virtual _TyPoint v_Arc_Position(float f_t, size_t n_arc) const
	{
		return CSplineImpl::v_Arc_Position(f_t, n_arc); // these do not have to be virtual anymore
	}

	/**
	 *	@brief calculates tangent vector on a spline arc
	 *
	 *	@param[in] f_t is position on a spline arc in range [0, 1]
	 *	@param[in] n_arc is zero-based index of spline arc
	 *
	 *	@return Returns tangent at the given position along the selected arc.
	 *
	 *	@note This function is supposed to be implemented in the final spline class.
	 */
	virtual _TyPoint v_Arc_Derivative(float f_t, size_t n_arc) const
	{
		return CSplineImpl::v_Arc_Derivative(f_t, n_arc); // these do not have to be virtual anymore
	}
};

/**
 *	@brief simple generic spline storage template
 *
 *	@tparam CPointClass is type of points the spline is defined on
 *	@tparam CSplineImpl is a particular spline implementation
 */
template <class CPointClass, class CSplineImpl>
class _CSpline : public CSplineBase<CPointClass, CSplineImpl> {
public:
	typedef CPointClass _TyPoint; /**< @brief type of points the spline is defined on */
	typedef CSplineImpl _TySplineImpl; /**< @brief spline implementation name */

public:
	/**
	 *	@copydoc CSplineBase::CSplineBase()
	 */
	inline _CSpline()
		:CSplineBase<CPointClass, CSplineImpl>()
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(const std::vector<_TyPoint>&)
	 */
	inline _CSpline(const std::vector<_TyPoint> &r_point_list) // throw(std::bad_alloc)
		:CSplineBase<CPointClass, CSplineImpl>(r_point_list)
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(CPointIter,CPointIter)
	 */
	template <class CPointIter>
	inline _CSpline(CPointIter p_point_begin, CPointIter p_point_end) // throw(std::bad_alloc)
		:CSplineBase<CPointClass, CSplineImpl>(p_point_begin, p_point_end)
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(const CSplineBase&)
	 */
	inline _CSpline(const _CSpline &r_other) // throw(std::bad_alloc)
		:CSplineBase<CPointClass, CSplineImpl>(r_other)
	{}

	/**
	 *	@copydoc CSplineBase::operator=()
	 */
	inline _CSpline &operator =(const _CSpline &r_other) // throw(std::bad_alloc)
	{
		*(CSplineBase<CPointClass, CSplineImpl>*)this = r_other;
		return *this;
	}

	/**
	 *	@brief calculates point at a given position along the spline (uniform subdivision)
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns point at the specified position along the spline.
	 *	@note This doesn't use any kind or approximation of constant step
	 *		subdivision, each spline segment is considered unit length
	 */
	_TyPoint v_Position_Uniform(float f_t) const
	{
		size_t n_arc_num = CSplineBase<CPointClass, CSplineImpl>::n_Arc_Num();
		size_t n_arc = (f_t > 0)? ((f_t < 1)? size_t(f_t * n_arc_num) : n_arc_num - 1) : 0;
		float f_arc_t = f_t * n_arc_num - n_arc;
		return CSplineBase<CPointClass, CSplineImpl>::v_Arc_Position(f_arc_t, n_arc);
	}

	/**
	 *	@brief calculates tangent vector at a given position along the spline (uniform subdivision)
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns tangent at the specified position along the spline.
	 *	@note This doesn't use any kind or approximation of constant step
	 *		subdivision, each spline segment is considered unit length
	 */
	_TyPoint v_Derivative_Uniform(float f_t) const
	{
		size_t n_arc_num = CSplineBase<CPointClass, CSplineImpl>::n_Arc_Num();
		size_t n_arc = (f_t > 0)? ((f_t < 1)? size_t(f_t * n_arc_num) : n_arc_num - 1) : 0;
		float f_arc_t = f_t * n_arc_num - n_arc;
		return CSplineBase<CPointClass, CSplineImpl>::v_Arc_Derivative(f_arc_t, n_arc);
	}

	/**
	 *	@brief calculates point at a given position along the spline (semi-constant step subdivision)
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns point at the specified position along the spline.
	 *	@note Semi-constant step means arc lengths are used to determine spline arc
	 *		on position f_t, but the arc itself is subdivided with constant step.
	 */
	_TyPoint v_Position_SCS(float f_t, float f_spline_length,
		const std::vector<float> &r_arc_length) const
	{
		float f_arc_t;
		size_t n_arc = CSplineBase<CPointClass, CSplineImpl>::n_Arc(f_t, f_arc_t, f_spline_length, r_arc_length);
		return CSplineBase<CPointClass, CSplineImpl>::v_Arc_Position(f_arc_t, n_arc);
	}

	/**
	 *	@brief calculates tangent vector at a given position along the spline (semi-constant step subdivision)
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns tangent at the specified position along the spline.
	 *	@note Semi-constant step means arc lengths are used to determine spline arc
	 *		on position f_t, but the arc itself is subdivided with constant step.
	 */
	_TyPoint v_Derivative_SCS(float f_t, float f_spline_length,
		const std::vector<float> &r_arc_length) const
	{
		float f_arc_t;
		size_t n_arc = CSplineBase<CPointClass, CSplineImpl>::n_Arc(f_t, f_arc_t, f_spline_length, r_arc_length);
		return CSplineBase<CPointClass, CSplineImpl>::v_Arc_Derivative(f_arc_t, n_arc);
	}

	// todo - add also a new f_Arc_Position() which will calculates position on the arc for constant step
	// todo - add a function that will train some polynomial function (least squares, wow) for constant step correction
	// todo - think about caching spline props, it is not very well done here (maybe split the spline to several classes, base for storage and containing interpolation interface, then more of those to support caching - or maybe even an external structure that would be filled and explicitly passed to the functions (problems with usability in other code, virtual functions and such
	// *** maybe create a simple spline interpolator / sampler interface and then template everything else away - flexible while also fast in scenarios where the spline type is known at compile-time))
};

/**
 *	@brief simple spline interpolator interface
 *	@tparam CPointClass is type of points the spline is defined on
 */
template <class CPointClass>
class CSplineSampler {
public:
	typedef CPointClass _TyPoint; /**< @brief type of points the spline is defined on */

public:
	/**
	 *	@brief calculates point at a given position along the spline
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns point at the specified position along the spline.
	 *	@note This interface does not specify subdivision step,
	 *		which is entirely dependent on the implementation.
	 */
	virtual _TyPoint v_Position(float f_t) const = 0;

	/**
	 *	@brief calculates tangent vector at a given position along the spline
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns tangent at the specified position along the spline.
	 *	@note This interface does not specify subdivision step,
	 *		which is entirely dependent on the implementation.
	 */
	virtual _TyPoint v_Derivative(float f_t) const = 0;

	// should there be more? spline length? something else?
};

/**
 *	@brief simple spline interpolator interface
 *	@tparam CPointClass is type of points the spline is defined on
 */
template <class CSplineClass>
class CSplineSampler_UniformStep {
public:
	typedef CSplineSampler_UniformStep<CSplineClass> _TySampler; /**< @brief this type */
	typedef CSplineClass _TySpline; /**< @brief type of the spline */
	typedef typename CSplineClass::_TyPoint _TyPoint; /**< @brief type of points the spline is defined on */

protected:
	const _TySpline *m_p_spline; /**< @brief reference to the sampled spline */

public:
	/**
	 *	@brief default constructor; creates sampler on a spline
	 *	@param[in] r_spline is a spline to be sampled
	 */
	inline CSplineSampler_UniformStep(const _TySpline &r_spline)
		:m_p_spline(&r_spline)
	{}

	/**
	 *	@brief copy-constructor (copies spline reference only, spline data is not modified)
	 *	@param[in] r_other is sampler to copy from
	 */
	inline CSplineSampler_UniformStep(const _TySampler &r_other)
		:m_p_spline(r_other.m_p_spline)
	{}

	/**
	 *	@brief copy-operator (copies spline reference only, spline data is not modified)
	 *	@param[in] r_other is sampler to copy from
	 *	@return Returns reference to this.
	 */
	inline CSplineSampler_UniformStep &operator =(const _TySampler &r_other)
	{
		m_p_spline = r_other.m_p_spline;
		return *this;
	}

	/**
	 *	@brief calculates point at a given position along the spline
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns point at the specified position along the spline.
	 *	@note This interface does not specify subdivision step,
	 *		which is entirely dependent on the implementation.
	 */
	virtual _TyPoint v_Position(float f_t) const
	{
		return m_p_spline->v_Position_Uniform(f_t);
	}

	/**
	 *	@brief calculates tangent vector at a given position along the spline
	 *	@param[in] f_t is position on the spline in range [0, 1]
	 *	@return Returns tangent at the specified position along the spline.
	 *	@note This interface does not specify subdivision step,
	 *		which is entirely dependent on the implementation.
	 */
	virtual _TyPoint v_Derivative(float f_t) const
	{
		return m_p_spline->v_Derivative_Uniform(f_t);
	}
};

#endif // __GENERIC_SPLINE_TEMPLATE_INCLUDED
