/*
								+----------------------------------+
								|                                  |
								|  ***   Catmull-Rom spline   ***  |
								|                                  |
								|   Copyright  -tHE SWINe- 2008   |
								|                                  |
								|           CatmullRom.h           |
								|                                  |
								+----------------------------------+
*/

#pragma once
#ifndef __CATMULL_ROM_SPLINE_INCLUDED
#define __CATMULL_ROM_SPLINE_INCLUDED

/**
 *	@file lml/Catmull-Rom spline
 *	@date 2008
 *	@author -tHE SWINe-
 *	@brief CatmullRom.h
 *
 *	@date 2009-01-21
 *
 *	CCatmullRomSpline is now template parametrized by point type, default is Vector3f
 *
 *	@date 2012-06-19
 *
 *	Moved multiple inclusion guard before file documentation comment.
 *
 */

#include "Spline.h"

/**
 *	@brief simple Ctamull - Rom spline implementation
 */
template <class CPointClass = Vector3f>
class CCatmullRomSplineImpl {
public:
	/**
	 *	@brief gets number of spline arcs
	 *	@return Returns number of spline arcs.
	 */
	inline size_t n_Arc_Num() const
	{
		return (m_point_list.size() > 3)? m_point_list.size() - 3 : 0;
	}

	/**
	 *	@brief calculates position on a spline arc
	 *
	 *	@param[in] f_t is position on a spline arc in range [0, 1]
	 *	@param[in] n_arc is zero-based index of spline arc
	 *
	 *	@return Returns point at the given position along the selected arc.
	 *
	 *	@note This function is supposed to be implemented in the final spline class.
	 */
	CPointClass v_Arc_Position(float f_t, size_t n_arc) const
	{
		_ASSERTE(n_arc < n_Arc_Num());

		CPointClass v_pt0 = m_point_list[n_arc];
		CPointClass v_pt1 = m_point_list[n_arc + 1];
		CPointClass v_pt2 = m_point_list[n_arc + 2];
		CPointClass v_pt3 = m_point_list[n_arc + 3];
		// get arc points + neighbor points

		float f_t2 = f_t * f_t;
		float f_t3 = f_t2 * f_t;
		// calculate t^2, t^3

		float f_weight0 = -f_t3 + 2 * f_t2 - f_t;
		float f_weight1 = 3 * f_t3 - 5 * f_t2 + 2;
		float f_weight2 = -3 * f_t3 + 4 * f_t2 + f_t;
		float f_weight3 = f_t3 - f_t2;
		// calculate polynoms

		return (v_pt0 * f_weight0 + v_pt1 * f_weight1 +
			v_pt2 * f_weight2 + v_pt3 * f_weight3) * .5f;
		// interpolate
	}

	/**
	 *	@brief calculates tangent vector on a spline arc
	 *
	 *	@param[in] f_t is position on a spline arc in range [0, 1]
	 *	@param[in] n_arc is zero-based index of spline arc
	 *
	 *	@return Returns tangent at the given position along the selected arc.
	 */
	CPointClass v_Arc_Derivative(float f_t, size_t n_arc) const
	{
		_ASSERTE(n_arc < n_Arc_Num());

		CPointClass v_pt0 = m_point_list[n_arc];
		CPointClass v_pt1 = m_point_list[n_arc + 1];
		CPointClass v_pt2 = m_point_list[n_arc + 2];
		CPointClass v_pt3 = m_point_list[n_arc + 3];
		// get arc points + neighbor points

		float f_t2 = f_t * f_t;
		float f_t3 = f_t2 * f_t;
		// calculate t^2, t^3

		float f_weight0 = -3 * f_t2 + 2 * 2 * f_t - 1;
		float f_weight1 = 3 * 3 * f_t2 - 5 * 2 * f_t + 0;
		float f_weight2 = -3 * 3 * f_t2 + 4 * 2 * f_t + 1;
		float f_weight3 = 3 * f_t2 - 2 * f_t;
		// calculate polynom derivatives

		return (v_pt0 * f_weight0 + v_pt1 * f_weight1 +
			v_pt2 * f_weight2 + v_pt3 * f_weight3) * .5f;
		// interpolate
	}
};

/**
 *	@brief simple Ctamull - Rom spline template, parametrized by point type
 */
template <class CPointClass>
class CCatmullRomSpline : public _CSpline<CPointClass, CCatmullRomSpline> {
public:
	/**
	 *	@copydoc CSplineBase::CSplineBase()
	 */
	inline CCatmullRomSpline()
		:_CSpline()
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(const std::vector<_TyPoint>&)
	 */
	inline CCatmullRomSpline(const std::vector<_TyPoint> &r_point_list) // throw(std::bad_alloc)
		:_CSpline(r_point_list)
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(CPointIter,CPointIter)
	 */
	template <class CPointIter>
	inline CCatmullRomSpline(CPointIter p_point_begin, CPointIter p_point_end) // throw(std::bad_alloc)
		:_CSpline(p_point_begin, p_point_end)
	{}

	/**
	 *	@copydoc CSplineBase::CSplineBase(const CSplineBase&)
	 */
	inline CCatmullRomSpline(const CCatmullRomSpline &r_other) // throw(std::bad_alloc)
		:_CSpline(r_other)
	{}

	/**
	 *	@copydoc CSplineBase::operator=()
	 */
	inline CCatmullRomSpline &operator =(const CCatmullRomSpline &r_other) // throw(std::bad_alloc)
	{
		*(_CSpline*)this = r_other;
		return *this;
	}
};

#endif // __CATMULL_ROM_SPLINE_INCLUDED
