/* Copyright (c) 2006-2008, Georgios Portokalidis
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of the Vrije Universiteit nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include "argos-config.h"
#include "argos-common.h"
#include "cpu.h"
#include "psql-config.h"
#include "argos-csi.h"
#include "argos-psql.h"
#include "argos-check.h"
#include "argos-memmap.h"
#include "exec-all.h"
// From postgresql-devel package
#include <libpq-fe.h>

#ifndef CONFIG_USER_ONLY

#define LOG_FL_TEMPLATE "argos.csi.%d"
#define LOG_MSG_TEMPLATE "[ARGOS] Log generated <%s>\n"

#define ARGOS_LOG_VERSION    2
#define ARGOS_MBLOCK_VERSION 1
#define ARGOS_NT_MASK        128 //!< Net tracker version mask
#define ARGOS_BE_MASK        64	 //!< Non-arch data are in big-endian

#define ARGOS_ARCH_I386   0
#define ARGOS_ARCH_X86_64 1 

#define PHYS_ADDR_MASK 0xfffff000

static uint64_t as_start_kernel[][2] = { 
	{ 0xc0000000U, 0xffff810000000000ULL }, // Linux kernel
	{ 0x80000000U, 0x0000000080000000ULL },  // Windows 2k kernel
	{ 0x80000000U, 0x0000000080000000ULL },  // Windows XP kernel
};

static uint64_t as_stop_kernel[][2] = { 
	{ 0xffffffffU, 0xffffffffffffffffULL }, // Linux kernel
	{ 0xffffffffU, 0x00000000ffffffffULL },  // Windows 2k kernel
	{ 0xffffffffU, 0x00000000ffffffffULL },  // Windows XP kernel
};

/*************************** CARGOS DEFINED CONSTANTS ***************************/

static const char * cargos_type_strings[] = { "JMP", "P_JMP", "TSS", "CALL", "RET", 
	"CI", "R_IRET", "SYSEXIT", "SYSRET", "R_JMP", "P_CALL", "R_CALL",
	"P_RET" };
static const char cargos_archs_string[][7] = { "i386", "x86_64" };

/********************************************************************************/

#ifdef TARGET_X86_64
#define FIRST_KERNEL_ADDR(os) (as_start_kernel[(os)][1])
#define LAST_KERNEL_ADDR(os) (as_stop_kernel[(os)][1] - TARGET_PAGE_SIZE + 1)
#define FIRST_USER_ADDR(os) (0x00000000)
#define LAST_USER_ADDR(os) (as_start_kernel[(os)][1] - TARGET_PAGE_SIZE + 1)

#define INCIDENT_MODEL          "INSERT INTO argos.incidents (version, arch, type, timestamp, eflags, faultyEIP) VALUES "\
                                "('0x%02x', '%s', '%s', '%s', '0x%16x', '0x%16llx');"
#else
#define FIRST_KERNEL_ADDR(os) (as_start_kernel[(os)][0])
#define LAST_KERNEL_ADDR(os) (as_stop_kernel[(os)][0] - TARGET_PAGE_SIZE + 1)
#define FIRST_USER_ADDR(os) (0x00000000)
#define LAST_USER_ADDR(os) (as_start_kernel[(os)][0] - TARGET_PAGE_SIZE + 1)

#define INCIDENT_MODEL          "INSERT INTO argos.incidents (version, arch, type, timestamp, eflags, faultyEIP) VALUES "\
                                "('0x%02x', '%s', '%s', '%s', '0x%08x', '0x%08x');"
#endif

#define MEMBLOCKS_MODEL         "INSERT INTO argos.memoryblocks (id_incident, version, tainted, size, memphysical, memvirtual, data) VALUES "\
                                "(%d, '0x%02x', %s, %u, '0x%08x', '0x%08x', '%s');"
#define REGISTERS_MODEL         "INSERT INTO argos.registers (id_incident, id_register, value, memoryfromaddr, netidx, tainted) VALUES "\
                                "(%d, %d, '0x%08x', '0x%08x', %d, %s);"
#define EXPLOIT_MEMBLOCKS_MODEL "INSERT INTO argos.exploitmemoryblocks (id_incident, memblocksize, memblockdata) VALUES "\
                                "(%d, %d, '%s');"
// Forensics shellcode

// RID: 62
#define LINUXSC_RID_OFF 62
// Length: 97
#define LINUXSC_LENGTH 97

// RID: 257
#define WIN32SC_RID_OFF 257
// Length: 293
#define WIN32SC_LENGTH 293

#define PAGE_OFF_MASK (TARGET_PAGE_SIZE - 1)

// Log headers

struct argos_mblock_hdr_struct {
	uint8_t format;
	uint8_t tainted;
	uint16_t size;
	target_ulong paddr;
	target_ulong vaddr;
} __attribute__((packed));

typedef struct argos_mblock_hdr_struct argos_mblock_hdr_t;


struct argos_log_hdr_struct {
	uint8_t format;
	uint8_t arch;
	uint16_t type;
	uint32_t ts;
	target_ulong reg[CPU_NB_REGS];
	target_ulong rorigin[CPU_NB_REGS];
#ifdef ARGOS_NET_TRACKER
	uint32_t netidx[CPU_NB_REGS];
#endif
	target_ulong eip,
		     eiporigin;
#ifdef ARGOS_NET_TRACKER
uint32_t eipnetidx;
#endif
	target_ulong old_eip;
	target_ulong eflags;
} __attribute__((packed));

typedef struct argos_log_hdr_struct argos_log_hdr_t;


static inline int argos_psql_header_write(PGconn *conn, CPUX86State *env, 
		target_ulong new_pc, target_ulong old_pc, int code, 
		argos_rtag_t *eiptag)
{
	int i;
        char * query;
        char * str_date;
        PGresult * res;

	argos_log_hdr_t hdr;

	hdr.format = ARGOS_LOG_VERSION;
#ifdef ARGOS_NET_TRACKER
	hdr.format |= ARGOS_NT_MASK;
#endif
#ifdef WORDS_BIGENDIAN
	hdr.format |= ARGOS_BE_MASK;
#endif
#ifdef TARGET_X86_64
	hdr.arch = ARGOS_ARCH_X86_64;
#else
	hdr.arch = ARGOS_ARCH_I386;
#endif
	hdr.type = code;
	hdr.ts = time(NULL);
	for (i = 0; i < CPU_NB_REGS; i++) {
		hdr.reg[i] = env->regs[i];
		hdr.rorigin[i] = argos_tag_origin(env->regtags + i);
#ifdef ARGOS_NET_TRACKER
		hdr.netidx[i] = argos_tag_netidx(env->regtags + i);
#endif
	}
	hdr.eip = new_pc;
	hdr.eiporigin = (eiptag)? argos_tag_origin(eiptag) : 0;
	hdr.old_eip = old_pc;
#ifdef ARGOS_NET_TRACKER
	hdr.eipnetidx = (eiptag)? argos_tag_netidx(eiptag) : 0;
#endif
	// TODO: Do i need to incorporate hflags?
	hdr.eflags = env->eflags;



        /* Here comes the DB code */
        int id_incident = 0;
        /* Into str_date insert current timestamp in format "YYYY/MM/DD  HH:MM:SS"*/
        time_t tim = (time_t) hdr.ts;
        struct tm * timestamp = localtime(&tim);
        if ((str_date = (char *) malloc(20 * sizeof(char)) ) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        snprintf(str_date, 20, "%d/%02d/%02d %02d:%02d:%02d", timestamp->tm_year+1900, timestamp->tm_mon+1, timestamp->tm_mday, 
                timestamp->tm_hour, timestamp->tm_min, timestamp->tm_sec);

        /* Into query insert SQL script defined by INCIDENT_MODEL string */
        int size = snprintf(NULL, 0, INCIDENT_MODEL, ARGOS_LOG_VERSION, cargos_archs_string[hdr.arch], cargos_type_strings[hdr.type], 
                str_date, hdr.eflags, hdr.eiporigin);
        if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        (void) snprintf(query, size, INCIDENT_MODEL, ARGOS_LOG_VERSION, cargos_archs_string[hdr.arch], cargos_type_strings[hdr.type], 
                str_date, hdr.eflags, hdr.eiporigin);

        /* Execute SQL string */
        res = PQexec(conn, query);
 
        /* Free part */
        free(str_date);
        free(query);

        /* Check SQL output */
        if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
            perror("INSERT INCIDENT command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }

        PQclear(res);
        /* We need ID of this incident, select the ID from last inserted row from the command above */
        res = PQexec(conn, "SELECT id FROM argos.incidents ORDER BY id DESC LIMIT 1;");
        if ((!res) || (PQresultStatus(res) != PGRES_TUPLES_OK)) {
            perror("SELECT ID of INCIDENT command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }
        id_incident = (int) atoi(PQgetvalue(res, 0, 0));

        /* Into query insert SQL script defined by REGISTERS_MODEL string */
	for (i = 0; i < CPU_NB_REGS; i++) {
		hdr.reg[i] = env->regs[i];
		hdr.rorigin[i] = argos_tag_origin(env->regtags + i);
#ifdef ARGOS_NET_TRACKER
		hdr.netidx[i] = argos_tag_netidx(env->regtags + i);
                //printf("%d 0x%08X\t0x%08X\t%X \n", i, hdr.reg[i], hdr.rorigin[i], hdr.netidx[i]);
#endif
                size = snprintf(NULL, 0, REGISTERS_MODEL, id_incident, i, hdr.reg[i], hdr.rorigin[i], hdr.netidx[i],
                        (hdr.rorigin[i] != 0) ? "TRUE" : "FALSE");
                if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
                {
                    perror("Can't allocate memory");
                    return -1;
                }
                (void) snprintf(query, size, REGISTERS_MODEL, id_incident, i, hdr.reg[i], hdr.rorigin[i], hdr.netidx[i],
                        (hdr.rorigin[i] != 0) ? "TRUE" : "FALSE");
                /* Execute SQL string */
                res = PQexec(conn, query);
        
                /* Free part */
                free(query);
                /* Check SQL output */
                if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
                    perror("REGISTERS INSERT command failed");
                    perror(PQerrorMessage(conn));
                    PQclear(res);
                    return -1;
                }
                PQclear(res);
	}
        /* EIP Register */
        size = snprintf(NULL, 0, REGISTERS_MODEL, id_incident, i, hdr.eip, hdr.eiporigin, hdr.eipnetidx,
                (hdr.eiporigin != 0) ? "TRUE" : "FALSE");
        if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        (void) snprintf(query, size, REGISTERS_MODEL, id_incident, i, hdr.eip, hdr.eiporigin, hdr.eipnetidx,
                (hdr.eiporigin != 0) ? "TRUE" : "FALSE");
        /* Execute SQL string */
        res = PQexec(conn, query);

        /* Free part */
        free(query);
        /* Check SQL output */
        if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
            perror("REGISTERS(EIP) INSERT command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }
        PQclear(res);

	return id_incident;
}

static inline int
argos_mblock_header_init(argos_mblock_hdr_t *hdr, target_ulong vaddr,
		target_ulong paddr, int tainted)
{
	int j, t;
	unsigned int remainder;

	memset(hdr, 0, sizeof(argos_mblock_hdr_t));
	hdr->format = ARGOS_MBLOCK_VERSION;
#ifdef ARGOS_NET_TRACKER
	if (tainted) hdr->format |= ARGOS_NT_MASK;
#endif
/*
#ifdef WORDS_BIGENDIAN
	hdr->format |= ARGOS_BE_MASK;
#endif
*/
	hdr->vaddr = vaddr;
	hdr->paddr = paddr;
	hdr->tainted = (tainted)? 0x01 : 0x00;

	remainder = TARGET_PAGE_SIZE - (paddr & PAGE_OFF_MASK);
	for (j = 0; j < remainder; j++, paddr++)
	{
		t = argos_memmap_istainted(paddr);
		if ((tainted != 0 && t == 0) || (tainted == 0 && t != 0))
			break;
	}
	hdr->size = j;
	return j;
}

static inline int
argos_mblock_write(PGconn * conn, argos_mblock_hdr_t *hdr, int id_incident, argos_rtag_t *eiptag)
{
        PGresult * res;
        target_ulong eiporigin;
	eiporigin = (eiptag)? argos_tag_origin(eiptag) : 0;

	//printf("0x%02x\t%s\t", ARGOS_MBLOCK_VERSION, (hdr->tainted)? "YES" : "NO");
        //printf("%4hu\t0x%08x\t0x%08x\t", hdr->size, hdr->paddr, hdr->vaddr);
        char * data;
        char * mblocks = NULL;
        char * query;
        //argos_netidx_t * nt = argos_memmap_ntdata(hdr->paddr);
        if ((mblocks = malloc( hdr->size * 3 * sizeof(char) + 1)) == NULL) // 4 bytes of INT in string "0xFFFFFFFF, "
        {
            perror("Can't allocate memory");
            return -1;
        }
        if ((data = (char *) malloc(hdr->size+1)) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        int i;
        /* For each bytes in memory, copy it to char (byte) array data */
        for (i=0; i<hdr->size; i++) {
            memcpy(data+i, (phys_ram_base + hdr->paddr + i), 1);
            snprintf(mblocks+(i*3), 4, "%02X ", (unsigned char)data[i]);
        }
        /* Remove last ", " */
        memset(mblocks+(hdr->size * 3 * sizeof(char) - 1), '\0', 2);
        //printf("%s\n", mblocks);
#ifdef ARGOS_NET_TRACKER
	if (hdr->tainted)
	{
            //argos_netidx_t *nt = argos_memmap_ntdata(hdr->paddr);
            /*if (fwrite(nt, 4, hdr->size, fp) != hdr->size)
                    goto error;*/
        }
#endif

        if (hdr->paddr <= eiporigin && eiporigin < hdr->paddr + hdr->size) {

            /* Into query insert SQL script defined by EXPLOIT_MEMBLOCKS_MODEL string */
            int size = snprintf(NULL, 0, EXPLOIT_MEMBLOCKS_MODEL, id_incident, hdr->size, mblocks);
            if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
            {
                perror("Can't allocate memory");
                return -1;
            }
            (void)  snprintf(query, size, EXPLOIT_MEMBLOCKS_MODEL, id_incident, hdr->size, mblocks);
            
            /* Execute SQL string */
            res = PQexec(conn, query);

            /* Free part */
            free(query);
            /* Check SQL output */
            if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
                perror("PACKET MBLOCKS: INSERT command failed");
                perror(PQerrorMessage(conn));
                PQclear(res);
                return -1;
            }
            PQclear(res);
        }

        /* Into query insert SQL script defined by MEMBLOCKS_MODEL string */
        int size = snprintf(NULL, 0, MEMBLOCKS_MODEL, id_incident, ARGOS_MBLOCK_VERSION, (hdr->tainted)? "TRUE" : "FALSE",
                hdr->size, hdr->paddr, hdr->vaddr, mblocks);
        if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        (void) snprintf(query, size, MEMBLOCKS_MODEL, id_incident, ARGOS_MBLOCK_VERSION, (hdr->tainted)? "TRUE" : "FALSE",
                hdr->size, hdr->paddr, hdr->vaddr, mblocks);
        
        /* Execute SQL string */
        res = PQexec(conn, query);

        /* Free part */
        free(query);
        free(data);
        free(mblocks);
        /* Check SQL output */
        if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
            perror("MBLOCKS: INSERT command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }
        PQclear(res);

	return 0;
}

static int argos_page_write(PGconn * conn, CPUX86State *env, target_ulong pc,
		target_ulong vaddr, target_ulong paddr, int id_incident, argos_rtag_t *eiptag)
{
	int i = 0, force = 0, tainted;
	argos_mblock_hdr_t hdr;

	if ((pc & TARGET_PAGE_MASK) == vaddr)
		force = 1;
	while (i < TARGET_PAGE_SIZE) {
		tainted = argos_memmap_istainted(paddr + i);
                
		if (tainted || force) {
			i += argos_mblock_header_init(&hdr, vaddr + i, 
					paddr + i, tainted);
			if (argos_mblock_write(conn, &hdr, id_incident, eiptag) != 0)
				return -1;
		}
		else 
			++i;
	}
	return 0;
}

static int 
argos_psql_process_proc(PGconn *conn, CPUX86State *env, target_ulong pc, 
		int log, int clean, int id_incident, argos_rtag_t *eiptag)
{
	target_ulong page_i, page_max, paddr;
	PhysPageDesc *pdesc;


	// Code privilege level 0 (kernel)
	if ((env->hflags & HF_CPL_MASK) == 0) {
		page_i = FIRST_KERNEL_ADDR(argos_os_hint);
		page_max = LAST_KERNEL_ADDR(argos_os_hint);
	} else { // User
		page_i = FIRST_USER_ADDR(argos_os_hint);
		page_max = LAST_USER_ADDR(argos_os_hint);
	}

	do {
		paddr = cpu_get_phys_page_debug(env, page_i);
		if (paddr == -1)
			goto next;
		if (!(pdesc = phys_page_find(paddr >> TARGET_PAGE_BITS)))
			goto next;
		if (log && argos_page_write(conn, env, pc, page_i, 
				pdesc->phys_offset & TARGET_PAGE_MASK, id_incident, eiptag) != 0)
			return -1;
		if (clean)
			argos_memmap_clear(pdesc->phys_offset & 
					TARGET_PAGE_MASK, TARGET_PAGE_SIZE);
next:
		page_i += (target_ulong)TARGET_PAGE_SIZE;
	} while (page_i <= page_max && page_i > 0);

	if (clean) {
	        int i;

		argos_tag_clear(&env->t0tag);
		argos_tag_clear(&env->t1tag);
		argos_tag_clear(&env->t2tag);

		for (i = 0; i < CPU_NB_REGS; i++)
			argos_tag_clear(env->regtags + i);

		argos_bytemap_reset(env->envmap, ENVMAP_SIZE);
	}
	
	return 0;
}

#endif

#define ARGOS_FSC_FAILED "[ARGOS] Injecting forensics code failed."\
	" No appropriate location found\n"
#define ARGOS_FSC_SUCCESS "[ARGOS] Injecting forensics shellcode at "\
				"0x%08x[0x%08x]\n"
static inline int
argos_packet_write(PGconn * conn, int id_incident, argos_rtag_t *eiptag)
{
        int i, length, count, id;
        char * data = NULL;
        char * query;
        PGresult * res;
        target_ulong eipnetidx;
        eipnetidx = (eiptag)? argos_tag_netidx(eiptag) : 0;

        count = i = length = id = 0;
        res = PQexec(conn, NETTRACKER_GET_MODEL);
        while (i < PQntuples(res)) {
            length = (int) atoi(PQgetvalue(res, i, 1));
            count += length;
            if (count >= eipnetidx) {
                /* Get the data value of this row */
                data = PQgetvalue(res, i, 2) != NULL ? strdup(PQgetvalue(res, i, 2)) : NULL;
                //id = (int) atoi(PQgetvalue(res, i, 0));
                break;
            }
            i++;
        }
        
        PQclear(res);
        //if ((data != NULL) && (length != 0)) printf("%d, %d, %d\n", eipnetidx, length, id);
        /*res = PQexec(conn, NETTRACKER_DEL_MODEL);
        if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
            perror("PACKET: DELETE command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }*/

        /* Into query insert SQL script defined by MEMBLOCKS_MODEL string */
        int size = snprintf(NULL, 0, EXPLOIT_PACKET_MODEL, id_incident, length, data);
        if ((query = (char *) malloc((size+1) * sizeof(char))) == NULL)
        {
            perror("Can't allocate memory");
            return -1;
        }
        (void) snprintf(query, size, EXPLOIT_PACKET_MODEL, id_incident, length, data);

        /* Execute SQL string */
        res = PQexec(conn, query);

        /* Free part */
        free(query);
        free(data);
        /* Check SQL output */
        if ((!res) || (PQresultStatus(res) != PGRES_COMMAND_OK)) {
            perror("PACKET: INSERT command failed");
            perror(PQerrorMessage(conn));
            PQclear(res);
            return -1;
        }
        PQclear(res);

	return 0;
}


// Log header format
// Version | Arch | Attack type | Timestamp | Registers
// Version =  8 bits, [Net tracker bit][Big endianess bit][Version]
// Arch = ARGOS_ARCH_I386 or ARGOS_ARCH_X86_64
// Attack type = (look in argos_check.h)
// Timestamp = 32 bit timestamp
// Registers = Register contents in the architecture's endianess
int 
argos_psql(CPUX86State *env, target_ulong new_pc, argos_rtag_t *eiptag, 
		target_ulong old_pc, int code)
{
#ifdef CONFIG_USER_ONLY
	qemu_fprintf(stderr, "forensics currently not supported\n");
	return 0;
#else
	int rid;
        PGconn * conn;

        /* Initialize DB */
        conn = PQsetdbLogin(APSQL_HOST, APSQL_PORT, NULL, NULL, APSQL_DB, APSQL_LOGIN, APSQL_PASSWD);

        if (PQstatus(conn) == CONNECTION_BAD)
        {
	    perror("Connection to database failed.");
            perror(PQerrorMessage(conn));
	    return -1;
        }

	int id_incident = argos_psql_header_write(conn, env, new_pc, old_pc, code, eiptag);
	if (id_incident <= 0)
            goto cleanup;

	if (argos_psql_process_proc(conn, env, new_pc, 1, 0, id_incident, eiptag) != 0)
		goto cleanup;

        if (argos_packet_write(conn, id_incident, eiptag) != 0)
            goto cleanup;

        PQfinish(conn);
	//rid = rand();
	rid = argos_instance_id;
	return rid;

cleanup:
        PQfinish(conn);
	return -1;
#endif
}
