//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/10/25                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Module description:
 * - Module reads a file specified in -file argument
 * - Result is sent to connected MDSTk channel
 */

#include "LoadFile.h"

#include <OMToolkit\IO\OMIO.h>
#include <OMToolkit\OMTypes.h>

///////////////////////////////////////////////////////////////////////////////////////////////////
// Module constants
///////////////////////////////////////////////////////////////////////////////////////////////////

// Module description
const std::string MODULE_DESCRIPTION    = "Module that loads mesh data";

// Additional command line arguments (input file)
const std::string MODULE_ARGUMENTS      = "file";

// Additional arguments (input file)
const std::string MODULE_ARG_FILE		= "file";

// Type of accepted mesh
typedef OMToolkit::Types::ModuleMeshd			MeshT;

///////////////////////////////////////////////////////////////////////////////////////////////////
// Constructor
///////////////////////////////////////////////////////////////////////////////////////////////////
OMLoadFile::OMLoadFile(const std::string& sDescription) : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Destructor
///////////////////////////////////////////////////////////////////////////////////////////////////
OMLoadFile::~OMLoadFile()
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Do on startup
///////////////////////////////////////////////////////////////////////////////////////////////////
bool OMLoadFile::startup()
{
	// Disable all OpenMesh errorlogs (for not mix MDSTk log)
	omlog().disable();
	omerr().disable();
	omout().disable();
    
	// Note
    MDS_LOG_NOTE("Module startup");

    // Test of existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

	// Test existence of -file argument
	if (!m_Arguments.exists(MODULE_ARG_FILE))
	{
		MDS_CERR('<' << m_sFilename << "> Bad parameter value (input file missing): type -h for help" << std::endl);
        printUsage();
        return false;
	}

	m_Arguments.value(MODULE_ARG_FILE, m_file);

    // O.K.
    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Main module loop
///////////////////////////////////////////////////////////////////////////////////////////////////
bool OMLoadFile::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pOChannel = getOutput(0);

	// Create mesh and set read options
	MeshT mesh;
	OMToolkit::IO::Options opt = OMToolkit::IO::Options::Default;
	
	// Read it and write into output channel
	if (OMToolkit::IO::readMesh(mesh, m_file, opt))
	{
		if (!OMToolkit::IO::writeMesh(mesh, *pOChannel))
		{
			MDS_CERR('<' << m_sFilename << "> Failed to write output data" << std::endl);
			return false;
		}
	}
	// error reading file
	else 
	{
		MDS_CERR('<' << m_sFilename << "> Failed to read input file: " <<  m_file << std::endl);
		return false;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// On module shutdown
///////////////////////////////////////////////////////////////////////////////////////////////////
void OMLoadFile::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Writes extended use of this module
///////////////////////////////////////////////////////////////////////////////////////////////////
void OMLoadFile::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR("Necessary arguments: [-file filename]" << std::endl);
    MDS_CERR("Options:" << std::endl);
	MDS_CERR("  -file Specifies input file." << std::endl);
	MDS_CERR("	  -Format is determined from a file (allowed are STL, OFF, OBJ, PLY, OM)." << std::endl);
    MDS_CERR(std::endl);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Main - executing a module
///////////////////////////////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
    // Creation of a module using smart pointer
    OMLoadFilePtr spModule(new OMLoadFile(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

