//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/11/23                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Description:
 * - Class implements a serializable vector concept to be used in mesh property
 */

#ifndef _OM_SERIALIZABLE_VECTOR_H_
#define _OM_SERIALIZABLE_VECTOR_H_

#include <OpenMesh\Core\Utils\BaseProperty.hh>
#include <OMToolkit\IO\OMStoreRestore.h>

namespace OMToolkit {
namespace Types {

	/**
	 * Class is supposed to be used as mesh property - can be serialized
	 */
	template <typename T>
	class OMSerializableVector : public std::vector<T>
	{
	}; // class OMSerializableVector

} // namespace Types
} // namespace OMToolkit

namespace OpenMesh{
namespace IO{

	/**
	 * Binary serializer for OMSerializableVector
	 */
	template <class T> struct binary<OMToolkit::Types::OMSerializableVector<T>>
    {
		/**
		 * Definition of value type
		 */
		typedef OMToolkit::Types::OMSerializableVector<T> value_type;

		/**
		 * Must be defined for writer/reader
		 */
		static const bool is_streamable = true;

		/**
		 * Static sizeof method - we have unknown size
		 */

		static size_t size_of(void) 
		{ 
			return UnknownSize;
		}

		/**
		 * Dynamic sizeof method - computes size of serialized structure
		 */
		static size_t size_of(const value_type& _v) 
		{ 
			return _v.size() + sizeof(unsigned int);
		}
    
		/**
		 * Store into output stream
		 * Stores int as size of vector and then sequence of values
		 */
		static size_t store(std::ostream& _os, const value_type& _v, bool _swap=false)
		{ 
			size_t bytes;
			unsigned int size = _v.size();
			bytes = IO::store( _os, size, _swap);
			for (unsigned int i = 0; i < size; ++i)
				bytes += IO::store( _os, _v[i], _swap );
			return _os.good() ? bytes : 0;
		}

		/**
		 * Restore from intput stream
		 * Restores size of vector, reallocates _v and restores values
		 */
		static size_t restore( std::istream& _is, value_type& _v, bool _swap=false)
		{ 
			size_t bytes;
			unsigned int size;
			bytes = IO::restore( _is, size, _swap );
			_v.resize(size);
			for (unsigned int i = 0; i < size && _is.good(); ++i)
				bytes += IO::restore( _is, _v[i], _swap );
			return _is.good() ? bytes : 0;
		}
    };
} // namespace IO
} // namespace OpenMesh

#endif //_OM_SERIALIZABLE_VECTOR_H_