/** @file main.cpp
* hlavni zdrojovy soubor programu
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <csignal>
#include <fstream>
#include <sstream>
#include <list>
#include <regex.h>
#include <libnet.h>
#include <pcap.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/if_ether.h>
#include <netinet/ip6.h>
#include <netinet/tcp.h>

#include <pthread.h>

#include "noise.h"
#include "packet.h"
#include "errors.h"
#include "config.h"
#include "net.h"
#include "translator.h"
#include "deceive.h"
#include "tcp.h"
#include "main.h"

using namespace std;
TConfig Config;
unsigned long int pktNum = 0;

/**
 * Callback funkce, ktera je volana, kdyz je zachycen paket
 * @param userData uzivatelska data
 * @param packet hlavicka paketu
 * @param packet zachyceny paket
 */
void packetRcvd(u_char *userData, const struct pcap_pkthdr* packetHeader, const u_char* packet)
{
  struct ether_header *ether = (struct ether_header *) packet; // ukazatel na strukturu Ethernetovou hlavicku
  Packet *pkt;

  // Overime, zda se jedna o IPv6 datagram
  if (ntohs(ether->ether_type) != ETHERTYPE_IPV6) {
    return; // nejedna se o IPv6 datagram => konec zpracovavani
  }

  // Overime, zda nejde o nami vyslany odchozi paket
  libnet_ether_addr ifMac = getInterfaceMac(Config.pkt_ifName, Config);
  if (compareMac(*(libnet_ether_addr *)ether->ether_shost, ifMac)) {
    return; // jde o nami odeslany paket, ktery nas dale nezajima
  }

  /* ********** Analyza paketu ********** */
  u_int eth_frameLength = packetHeader->len;                           // delka celeho prijateho ramce
  u_int ipv6_datagramLength = 0;                                       // delka IPv6 datagramu
  uint8_t *ipv6_datagram = (uint8_t *)(packet + sizeof(struct ether_header));
  ipv6_datagramLength = eth_frameLength - sizeof(struct ether_header); // delka IPv6 datagramu:  delka ramce - delka eth. hlavicky

  // zjiteni smeru paketu
  TPacketDirection direction;
  if (Config.pkt_ifType == IN_IF) {direction = TO_RECEIVER;} else {direction = TO_SENDER;}

  // Inkrementujeme citac paketu
  pktNum++;

  // vytvoreni instance tridy Packet
  try {
    pkt = new Packet(ipv6_datagram, ipv6_datagramLength, direction, pktNum);
    if (pkt == NULL || pkt->ipv6_len() == 0) {
      printError(EMEMORY);
      return;
    }
  } catch (std::bad_alloc &b) {
    printError(EMEMORY);
    return;
  }
  pkt->setCapturedInterface(Config.pkt_ifName);    // nastavime rozhrani, na kterem byl zachycen paket

  // Overime, zda neni cilova adresa typu link local
  if (isLinkLocal(pkt->ipv6_dstIP())) {
    return; // nezajimaji nas - nebude provaden preklad
  }

  // Overime, zda obe adresy (zdrojova i cilova) maji platne IID
  if ((!validateInterfaceID(pkt->ipv6_srcIP())) || (!validateInterfaceID(pkt->ipv6_dstIP()))) {
    sendIcmpMessage(pkt->ipv6_srcIP(), Config.pkt_ifName, ICMPV6_PARAMETER, ICMPV6_HOP_LIMIT, 0, 0, Config); //pokud ne, posleme ICMPv6 zpravu param. problem error
  }


  //  Podle typu rozhrani (vnitni/vnejsi) rozhodneme co s paketem dale
  list<TRule>::iterator r;
  bool rule_found = false;
  in6_addr inPrefix, exPrefix;
  in6_addr source_ip_final, dest_ip_final;
  int inPrefixLen, exPrefixLen;
  string targetInterface = ""; // cilove rozhrani, kam bude zpracovany paket preposlan

  if (Config.pkt_ifType == IN_IF) {
    /* *********************************** */
    /*  Paket prisel z VNITRNIHO ROZHRANI  */
    /* *********************************** */
    // overime, zda zdrojova IP odpovida internimu prefixu nektereho z pravidel
    for(r=Config.rules_list.begin(); r != Config.rules_list.end(); ++r) {
      if (r->inIf == Config.pkt_ifName && comparePrefix(pkt->ipv6_srcIP(), r->inPrefix, r->inPrefixLength)) {
        rule_found = true;
        inPrefix = r->inPrefix;
        exPrefix = r->exPrefix;
        inPrefixLen = r->inPrefixLength;
        exPrefixLen = r->exPrefixLength;
      }
    }
    if (!rule_found) {
      // nebylo nalezeno zadne odpovidajici pravidlo
      // => posleme ICMPv6 zpravu "destination unreachable"
      sendIcmpMessage(pkt->ipv6_srcIP(), Config.pkt_ifName, ICMPV6_UNREACH, ICMPV6_HOP_LIMIT, 0, 0, Config);
      return;
    }
    translateAddress(pkt->ipv6_srcIP(), source_ip_final, inPrefix, inPrefixLen, exPrefix, exPrefixLen);
    dest_ip_final = pkt->ipv6_dstIP();

  } else if (Config.pkt_ifType == OUT_IF) {
    /* ********************************** */
    /*  Paket prisel z VNEJSIHO ROZHRANI  */
    /* ********************************** */
    // overime, zda nejde o ICMPv6 zpravu urcenou nam
    if (processIcmpEchoReply(pkt, Config)) {
      return;
    }
    // overime, zda cilova IP odpovida externimu prefixu nektereho z pravidel
    for(r=Config.rules_list.begin(); r != Config.rules_list.end(); ++r) {
      if (comparePrefix(pkt->ipv6_dstIP(), r->exPrefix, r->exPrefixLength)) {
        targetInterface = r->inIf;
        rule_found = true;
        inPrefix = r->inPrefix;
        exPrefix = r->exPrefix;
        inPrefixLen = r->inPrefixLength;
        exPrefixLen = r->exPrefixLength;
      }
    }

    if (!rule_found) {
      // nebylo nalezeno zadne odpovidajici pravidlo
      // => posleme ICMPv6 zpravu "destination unreachable"
      sendIcmpMessage(pkt->ipv6_srcIP(), Config.pkt_ifName, ICMPV6_UNREACH, ICMPV6_HOP_LIMIT, 0, 0, Config);
      return;
    }
    source_ip_final = pkt->ipv6_srcIP();
    translateAddress(pkt->ipv6_dstIP(), dest_ip_final, exPrefix, exPrefixLen, inPrefix, inPrefixLen);
  } else {
    return;
  }

  // nahradime adresy IPv6 datagramu za pozadovane (source nebo dest byla prelozena)
  pkt->set_translated_ipv6_srcIP(source_ip_final);
  pkt->set_translated_ipv6_dstIP(dest_ip_final);
  pkt->setTargetInterface(targetInterface);        // nastavime vystupni rozhrani

  /* ************************************** */

  if (deceiveConditions(pkt, Config)) {
    // Pridame paket do seznamu, ktery zpracovava druhe vlakno
    pthread_mutex_lock(&Config.pkt_queue_mutex);

    try {Config.pkt_queue.push_back(pkt);} catch (...) {printError(EMEMORY);}
    Config.newPacket = true;
    pthread_cond_broadcast(&Config.deception_cond);
    pthread_mutex_unlock(&Config.pkt_queue_mutex);
  } else {
  // => pokud ne, paket bude odeslan standardnim zpusobem
    /* *********************************************************************** */
    /*  STANDARDNI ODESLANI PAKETU S PRELOZENOU ADRESOU NA PRISLUSNE ROZHRANI  */
    /* *********************************************************************** */
    sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), targetInterface, Config);
    delete pkt;
    return;
  }
  if (*userData == NULL) {} // eliminace warningu pri kompilaci, parametr jinak neni pouzivan
}

/**
 * Je aktivni rezim verbose ("ukecany")?
 * @return ano, ci ne
 */
bool verbose() {
  return Config.verbose;
}


/**
 * Handler funkce signalu pro ukonceni programu
 * @param s cislo signalu
 */
void endProgram(int s) {
  if (s < 0) {
    return; // neplatny signal
  }

  cout << "Zachycen signal pro ukonceni programu.." << endl;

  Config.sig = true;
  Config.main_loop = false; // konec smycky - i v druhem vlakne

  // pokud vlakno deceptionThread ceka na paket, zajistime, aby pokracovalo
  pthread_mutex_lock(&Config.pkt_queue_mutex);
  pthread_cond_broadcast(&Config.deception_cond);
  pthread_mutex_unlock(&Config.pkt_queue_mutex);

  // pockame na ukonceni vlakna deceptionThread;
  if (pthread_join ( Config.deception_thread, NULL )) {
    pthread_exit(NULL);
  }

  pthread_cond_destroy(&Config.deception_cond);
  pthread_cancel(Config.deception_thread);

  pthread_mutex_unlock(&Config.pkt_queue_mutex);
  pthread_mutex_destroy(&Config.pkt_queue_mutex);
  pthread_mutex_destroy(&Config.libnet_mutex);
  pthread_mutex_destroy(&Config.reply_mutex);

  Config.pkt_queue.clear();

  return;
}


/**
 * Inicializace konfigurace programu
 * @param konfiguracni struktura programu
 */
int initConfig(TConfig &cfg) {
  cfg.main_loop = true;
  cfg.err = false;
  cfg.sig = false;
  cfg.reload = false;
  cfg.config_file = "";
  cfg.max_fd = -1;
  cfg.pkt_ifName = "";
  cfg.pkt_ifType = UNKNOWN_IF;
  for (unsigned int i = 0; i < 16; i++) {
    cfg.senderPrefix.s6_addr[i] = 0;
  }
  for (unsigned int i = 0; i < 16; i++) {
    cfg.receiverPrefix.s6_addr[i] = 0;
  }
  cfg.senderPrefixLen = -1;         // ANY
  cfg.receiverPrefixLen = -1;       // ANY
  cfg.senderPort = 0;               // ANY
  cfg.receiverPort = 0;             // ANY

  cfg.noiseAmount = 0;
  cfg.segLen = 1;
  cfg.fakeMsgFile1 = "";
  cfg.fakeMsgFile2 = "";
  cfg.pktNanoInterval = 0.0;

  cout.setf(ios::fixed,ios::floatfield);
  cout.precision(3);

  if (pthread_mutex_init(&cfg.pkt_queue_mutex, NULL) != 0) {
    return EINITMUTEX;
  }
  if (pthread_mutex_init(&cfg.libnet_mutex, NULL) != 0) {
    return EINITMUTEX;
  }
  if (pthread_mutex_init(&cfg.reply_mutex, NULL) != 0) {
    return EINITMUTEX;
  }
  if (pthread_cond_init(&cfg.deception_cond, NULL) != 0) {
    return EINITCOND;
  }

  //inicializace generatoru nahodnych cisel
  srand((unsigned int) time(NULL));

  return OK;
}


/**
 * Hlavni funkce programu
 * @param argc pocet zadanych parametru
 * @param argv vycet zadanych parametru
 * @return navratovy kod programu
 */
int main (int argc, char *argv[]) {
  int errNum = OK;
  errNum = initConfig(Config);

  if (errNum != OK) {
    printError(errNum);
    return EXIT_FAILURE;
  }

  /*
    Nejprve overime zadane vstupy a nacteme seznam pravidel pro preklad
    a take seznam vystupnich rozhrani.
  */

  // Kontrola zadanych parametru;
  Config.verbose = false;
  if (argc == 3) {
    if (string(argv[1]) == "-v") {
      Config.verbose = true;
      Config.config_file = string(argv[2]);
    } else if (string(argv[2]) == "-v") {
      Config.verbose = true;
      Config.config_file = string(argv[1]);
    }
  } else if (argc == 2 ) {
    if (string(argv[1]) == "-h") {
      printHelp();
      return EXIT_SUCCESS;
    } else {
      Config.config_file = string(argv[1]);
    }
  } else {
    printError(EPARAMS);
    printHelp();
    return EXIT_FAILURE;
  }


  // Nacteme konfiguraci ze souboru
  errNum = loadConfig(Config);
  if (errNum != OK) {
    printError(errNum);
    return EXIT_FAILURE;
  }



  if (verbose()) {
    cout << "=== LIS Deception Proxy ===" << endl;
    if (Config.fakeMsgFile1 == "") {cout << "Podvrzena zprava c.1 nebyla nastavena." << endl;}
    else {cout << "Podvrzena zprava c.1 bude pouzita." << endl;}
    if (Config.fakeMsgFile1 == "") {cout << "Podvrzena zprava c.2 nebyla nastavena." << endl;}
    else {cout << "Podvrzena zprava c.2 bude pouzita." << endl;}
    cout << "Uroven sumu: " << Config.noiseAmount << endl;
    cout << "Interval mezi segmentovanymi pakety: " << Config.pktNanoInterval << " ns" << endl;
    cout << "Delka segmentu: " << Config.segLen << endl;
  }

  ///////////////////////////////////////////////////////////////////////
  /*
    V tuto chvili mame nactena pravidla pro preklad a vime, na kterych
    zarizenich budeme naslouchat.
  */

  // Nacteme rozhrani na kterych budeme naslouchat
  if (openInterfaces(Config.interface_list, Config.read_set, Config.max_fd)) {
    Config.interfaces_open = true;
  } else {
    /*
      O vypis chyb se v tomto pripade postara primo funkce openInterfaces(),
      ktera vypise, ktera konkretni rozhrani se nepodari otevrit pro naslouchani.
    */
    return EXIT_FAILURE;
  }

  ///////////////////////////////////////////////////////////////////////
  /*
    Pozadovana zarizeni jsou otevrena pro naslouchani a muzeme zacit
    zachytavat pakety, ktere budou dale zpracovavany.
  */
  if (verbose()) {cout << "Zahajuji naslouchani na rozhranich..." << endl;}

  // Vytvoreni druheho vlakna, ktere zajistuje utajeni pozadovane komunikace
  if (pthread_create(&Config.deception_thread, NULL, deceptionThread, &Config) ) {
    printError(ETHREAD);
    return EXIT_FAILURE;
  }

  ///////////////////////////////////////////////////////////////////////
  // hlavni smycka programu - zachytavani paketu
  list<TInterface>::iterator i;
  while (Config.main_loop) {
    errNum = select(Config.max_fd+1, &(Config.read_set), NULL, NULL, NULL);
    /*
      Bud byl zachycen paket nebo nastala chyba, to zjistime dale.

      (Pokud je promenna Sig nastavena na true, znamena to, ze
      byl prijat signal a provadi se znovunacteni konfigurace
      nebo ukonceni programu.)
    */
    if (Config.sig) {Config.sig = false; continue;}
    if (errNum == -1 && !Config.sig) {
      printError(ELISTEN);
      Config.err = true;
      break;
    } else if (errNum == 0 && !Config.sig) {
      // timeout - neni uvazovan, nemusime resit
    } else if (!Config.sig) {
      // Zjistime, z jakeho rozhrani paket prisel
      for(i=Config.interface_list.begin(); (i != Config.interface_list.end()) && !Config.sig; ++i) {
        if (FD_ISSET(i->fd, &Config.read_set) && !Config.sig) {
          /*
            Na vstupnim/vystupnim rozhrani byl zachycen paket, ktery by nas
            potencionalne mohl zajimat. Zavola se handler funkce packetRcvd,
            ve ktere bude pomoci fork() vytvoren potomek procesu, ktery
            dany paket zpracuje.
          */
          Config.pkt_ifName = i->name;  // potomek bude vedet, na jakem rozhrani byl paket zachycen
          Config.pkt_ifType = i->type;  // + typ rozhrani vnitrni/vnejsi
          // Zpracovani paketu - zavola se handler-funkce packetRcvd
          errNum = pcap_dispatch((pcap_t*)i->descr, 1, packetRcvd, NULL);
        }
        if (errNum == -1) {
          // Chyba pri zachytavani paketu
          printError(ECAPTURE);
          Config.err = true;
          break;
        }
      }
    }
    /*
      Pred dalsi iteraci hlavni smycky musime mnozinu filedescriptoru
      opet naplnit filedescriptory rozhrani, na kterych naslouchame.
    */
    FD_ZERO(&(Config.read_set));

    for(i=Config.interface_list.begin(); (i != Config.interface_list.end()) && !Config.sig; ++i) {
      FD_SET(i->fd, &(Config.read_set)); // znovu naplnime mnozinu file deskriptoru
    }
  }
  // konec hlavni smycky programu
  ///////////////////////////////////////////////////////////////////////

  // naslouchani jiz dale neprobiha, zahajime ukoncovani programu

  // uzavreni otevrenych rozhrani
  if (Config.interfaces_open) {
    closeInterfaces(Config);
  }

  if (Config.err) {  // v prubehu programu byla detekovana chyba
    return EXIT_FAILURE;
  } else {    // vse probehlo v poradku
    return EXIT_SUCCESS;
  }
}
