/** @file deceive.cpp
* zdrojovy soubor obsahujici implementaci oklamani odposlechu
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <fstream>
#include <vector>
#include <ctime>
#include <list>
#include <libnet.h>
#include <netinet/if_ether.h>
#include <netinet/ip6.h>
#include <netinet/tcp.h>
#include <time.h>

#include "main.h"
#include "deceive.h"
#include "tcp.h"
#include "net.h"
#include "errors.h"
#include "noise.h"
#include "translator.h"

list<TAck> acksToIgnore;
list<Packet*> pkt_deceive;

/**
 * Je paket soucasti utajovaneho spojeni
 * @param paket
 * @param informace o paketu
 * @param konfigurace programu
 * @return (true = jde o utajovane spojeni, false = jde o bezne spojeni)
 */
bool deceiveConditions(Packet *pkt, TConfig &cfg) {
  if (!pkt->is_tcp()) {
    return false; // nejde o TCP segment
  }

  TConnection con;
  con.sndIP = pkt->ipv6_sndIP();
  con.rcvIP = pkt->ipv6_rcvIP();
  con.sndPort = pkt->tcp_sndPort();
  con.rcvPort = pkt->tcp_rcvPort();

  if (secretConnection(con, cfg)) {
    return true;  // ano, v ramci tohoto spojeni budou utajovana data
  } else {
    return false; // ne, jde o bezne spojeni
  }
}


/**
 * Vlakno programu slouzici ke zpracovani paketu v ramci spojeni, urcenych k utajeni
 * @param konfigurace programu
 */
void * deceptionThread(void * arg) {
  TConfig * cfg;
  cfg = (TConfig *)arg;

  while (cfg->main_loop) {
    // ================================ smycka vlakna ================================
    // provedeme praci s pakety, ktere mame
    if (!pkt_deceive.empty()) {
      list<Packet*>::iterator i;
      for(i=pkt_deceive.begin(); i != pkt_deceive.end(); ++i) {
        if (dec_processPacket(*i, *cfg)) {
          delete *i;
          *i = NULL;
          i = pkt_deceive.erase(i);
        }
      }
    }

    // zruseni vsech neaktivnich spojeni
    list<TConnection*>::iterator i;
    for(i=cfg->con_list.begin(); i !=cfg->con_list.end(); ++i) {
      if (*i != NULL && (*i)->active == false &&
        pthread_kill((*i)->hc_thread, 0) != 0)
      {
        try {
          pthread_mutex_destroy(&((*i)->con_mutex));
          delete(*i);
          *i = NULL;
          i = cfg->con_list.erase(i);
        } catch (...) {
          break;
        }
      }
    }

    // pokud existuji, vezmeme nove pakety z fronty
    pthread_mutex_lock(&(cfg->pkt_queue_mutex));
    if (pkt_deceive.empty()) {   // pokud jsme zpracovali vsechny nase pakety
      while (!cfg->newPacket && cfg->main_loop) { // budeme cekat na nove
        pthread_cond_wait(&(cfg->deception_cond), &(cfg->pkt_queue_mutex));
      }
    }
    while (!cfg->pkt_queue.empty()) {
      try {pkt_deceive.push_back(cfg->pkt_queue.front());} catch (...) {printError(EMEMORY);}
      cfg->pkt_queue.pop_front();
    }
    cfg->newPacket = false;
    pthread_mutex_unlock(&(cfg->pkt_queue_mutex));
    // ===============================================================================
  }

  // konec programu => uvolneni alokovane pameti po paketech, pokud nejake zbyly
  if (!pkt_deceive.empty()) {
    cout << "koncim" << endl;
    list<Packet*>::iterator i;
    for(i=pkt_deceive.begin(); i != pkt_deceive.end(); ++i) {
      if (*i != NULL) {
        delete *i;
        *i = NULL;
        i = pkt_deceive.erase(i);
      }
    }
  }

  return NULL;
}


/**
 * Zpracovani paketu ze seznamu
 * @param ulozeny paket
 * @return uspech (paket muze byt odstranen ze seznamu)
 */
bool dec_processPacket(Packet *pkt, TConfig &cfg) {
  TConnection *con = getConnectionPointer(pkt->ipv6_sndIP(), pkt->ipv6_rcvIP(), pkt->tcp_sndPort(), pkt->tcp_rcvPort(), cfg);
  if (con == NULL) {
    // spojeni zatim neexistuje
    if (pkt->tcp_syn()) {
      // prisel SYN paket => vytvorime spojeni
      con = addConnection(pkt->ipv6_sndIP(), pkt->ipv6_rcvIP(), pkt->tcp_sndPort(), pkt->tcp_rcvPort(), cfg);
      if (con == NULL) {
        // chyba pri vytvareni spojeni
        printError(ECREATECON);
        return true;
      }
      if (verbose()) {
        cout << "+ Nove spojeni: [" << ipv6ToStr(pkt->ipv6_sndIP()) << "]:" << pkt->tcp_sndPort();
        cout << " --- [" << ipv6ToStr(pkt->ipv6_rcvIP()) << "]:" << pkt->tcp_rcvPort() << endl;
      }
      // spojeni uspesne vytvoreno
    } else {
      // spojeni neexistuje a nejde ani o SYN paket
      sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
      return true;
    }
  }

  // ... spojeni existuje ...

  // jde o FIN nebo RST paket?
  if (pkt->tcp_fin() || pkt->tcp_rst()) {
    /*
    Jde-li o FIN nebo RST paket od odesilatele, spojeni existuje
    a ve fronte cekaji jeste nejake pakety (od odesilatele), ktere byly odeslany drive
    FIN/RST se pozdrzi (byl pravdepodobne poslan pozdeji nez cekajici pakety)
    */
    if (pkt->packetDirection() == TO_RECEIVER) {
      bool hold = false;
      if (!pkt_deceive.empty()) {
      list<Packet*>::iterator pi;
      for(pi=pkt_deceive.begin(); pi != pkt_deceive.end(); ++pi) {
        if ((*pi)->packetDirection() == TO_RECEIVER           // od odesilatele
           && (*pi)->get_pktNum() < pkt->get_pktNum()           // byl poslan drive
           && compareAddresses((*pi)->ipv6_sndIP(), con->sndIP) // odpovida IP odesilatele
           && compareAddresses((*pi)->ipv6_rcvIP(), con->rcvIP) // odpovida IP prijemce
           && (*pi)->tcp_sndPort() == con->sndPort              // odpovida port podesilatele
           && (*pi)->tcp_rcvPort() == con->rcvPort              // odpovida port prijemce
          ) {
            hold = true;
            break;
          }
        }
        if (hold) {
          return false; // paket se zatim nebude mazat
        }
      }
    }

    // => spojeni bude ukonceno
    clearIgnoreList(con);
    removeConnection(con, cfg);
    con = NULL;
    if (verbose()) {
      cout << "- Ukonceno spojeni: [" << ipv6ToStr(pkt->ipv6_sndIP()) << "]:" << pkt->tcp_sndPort();
      cout << " --- [" << ipv6ToStr(pkt->ipv6_rcvIP()) << "]:" << pkt->tcp_rcvPort() << endl;
    }

    sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
    return true;
  }

  // ... spojeni existuje a neni potreba jej ukoncit ...

  if (pkt->packetDirection() == TO_SENDER) {
  // ###################################################################################################
  // ###################################### <= SMER K ODESILATELI ######################################
  // ###################################################################################################

    // =========== zname uz alespon predpokladany pocet skoku a cilove rozhrani? =============
    pthread_mutex_lock(&con->con_mutex); // ++ LOCK ++
    if (!con->estimated_hc_set || con->outIf == "") {
      con->outIf = pkt->capturedInterface(); // nastavime vnejsi rozhrani
      uint8_t pktHlim = pkt->ipv6_hlim();
      if (pktHlim > 128) {
        con->hopCount = 255 - pktHlim;
      } else if (pktHlim > 64) {
        con->hopCount = WINDOWS_DEFAULT_HLIM - pktHlim;
      } else {
        con->hopCount = LINUX_DEFAULT_HLIM - pktHlim;
      }
      con->estimated_hc_set = true; // predpokladany pocet skoku je nastaven
    }
    pthread_mutex_unlock(&con->con_mutex); // -- UNLOCK --
    // ========================================================================================

    // nejde nahodou o nezadouci ACK packet, ktery mame ignorovat?
    if (pkt->tcp_ack() && ackToIgnore(pkt->tcp_ackSeq(), con)) {
      return true;
    }

    // paket odesleme standardnim zpusobem
    sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
    return true;
  } else {
  // ###################################################################################################
  // ######################################## SMER K PRIJEMCI => #######################################
  // ###################################################################################################
  // ============ zname uz vystupni rozhrani? ==============
  if (pkt->tcp_dataLen() == 0) {
    // paket neobsahuje zadna TCP data
    sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
    return true;
  } else {
    // paket obsahuje TCP data, ktera je nutne odeslat po bytech se sumem
    bool pktSent = false;
    pthread_mutex_lock(&con->con_mutex); // ++ LOCK ++
    if (con->hc_failure) {
      // nastal problem pri zjistovani poctu skoku => posleme normalne
      sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
      pktSent = true;
    } else if (con->final_hc_set) { // byl overen hop count?
      if (con->hopCount == 0) {
        // 0 skoku - nemuzeme provadet oklamani
        // !!!!!!!!!!!!!!!!!!!!! UPRAVA - TODO - ZDE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        sendNormally(pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
        //sendWithNoise(con, pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
        // !!!!!!!!!!!!!!!!!!!!! UPRAVA - TODO - ZDE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        pktSent = true;
      } else {
        // realizujeme oklamani odposlechu
        sendWithNoise(con, pkt->ipv6Ptr(), pkt->ipv6_len(), pkt->targetInterface(), cfg);
        pktSent = true;
      }
    }
    pthread_mutex_unlock(&con->con_mutex); // -- UNLOCK --
    if (pktSent) {
      return true;  // paket byl odeslan
    } else {
      return false; // paket bude odeslan az bude overen hop count
    }
  }
  // ###################################################################################################
  // ###################################################################################################
  // ###################################################################################################
  }
}


/**
 * Odeslani zpravy po bytech spolu se sumem
 * @param IPv6 header
 * @param delka IPv6 datagramu
 * @param konfigurace programu
 */
void sendWithNoise(TConnection *con, uint8_t *ipv6_datagram, unsigned int ipv6_datagramLen, string targetInterface, TConfig &cfg) {
  Packet * decPkt;

  // vytvoreni instance tridy Packet
  try {
    decPkt = new Packet(ipv6_datagram, ipv6_datagramLen, TO_RECEIVER, 0);
    if (decPkt == NULL || decPkt->ipv6_len() == 0) {
      printError(EMEMORY);
      return;
    }
  } catch (std::bad_alloc &b) {
    printError(EMEMORY);
    return;
  }

  if (!decPkt->is_tcp() || decPkt->tcp_dataLen() == 0 || decPkt->ipv6Ptr() == NULL || decPkt->tcpDataPtr() == NULL) {
    return; // nastala chyba pri praci s paketem
  }

  uint32_t tcp_startSeqNum = decPkt->tcp_seq();
  uint32_t tcp_seqNum = tcp_startSeqNum;
  uint32_t tcp_finalSeqNum = tcp_startSeqNum + decPkt->tcp_dataLen() - 1;


  std::vector<unsigned char> secretMessageChars;
  for (unsigned int i = 0; i < decPkt->tcp_dataLen(); i++) {
    unsigned char msgChar;
    msgChar = *(decPkt->tcpDataPtr() + i);
    try {secretMessageChars.push_back(msgChar);} catch (...) {printError(EMEMORY); return;}
  }

  // minimalni Hop Limit k dosazeni cile
  uint8_t noise_maxHlim = con->hopCount - 1;
  uint8_t real_minHlim = con->hopCount;
  uint8_t real_maxHlim = real_minHlim + noise_maxHlim / 2;

  // otevreni souboru s podvrzenymi zpravami
  ifstream fakeMsg1(cfg.fakeMsgFile1.c_str(), ios::in);
  ifstream fakeMsg2(cfg.fakeMsgFile2.c_str(), ios::in);

  // Postupne odeslani jednotlivych znaku
  vector<unsigned char>::iterator msgIterator;

  // vytvorime novy paket
  TBytePkt bytePkt;
  bytePkt.ipv6_header = *(decPkt->ipv6_headerPtr());
  //bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_plen = htons(sizeof(tcphdr) + 1);
  bytePkt.tcp_header = *(decPkt->tcp_headerPtr());
  bytePkt.tcp_header.doff = 5;
  for (unsigned int i = 0; i < TCP_MAX_DATALEN; i++) {
    bytePkt.msg[i] = 0;
  }

  if (verbose()) {
    cout << "Fragmentuji TCP segment s daty o velikosti: " << secretMessageChars.size() << " B pro [";
    cout << ipv6ToStr(bytePkt.ipv6_header.ip6_dst) << "]" << endl;
  }

  //pocet nahodne generovanych znaku pred a po bytu skutecne zpravy
  unsigned int noiseBytes1 = 0;
  unsigned int noiseBytes2 = 0;
  if (cfg.noiseAmount > 0) {
    noiseBytes1 = rand() % cfg.noiseAmount;
    noiseBytes2 = cfg.noiseAmount - noiseBytes1;
  }

  unsigned int pktSent = 0;
  unsigned int byteSent = 0;

  msgIterator=secretMessageChars.begin();



  for (unsigned int offset = 0; offset < decPkt->tcp_dataLen();) {
    // ======== ZACATEK SEKVENCE ======== ---------------------------------------------------------------------

      unsigned int partLen;
      if (cfg.segLen == 0) {
        // Nebudeme zpravu segmentovat
        partLen = decPkt->tcp_dataLen();
      } else {
        // Budeme zpravu segmentovat
        if (offset + cfg.segLen >= decPkt->tcp_dataLen()) {
          partLen = decPkt->tcp_dataLen() - offset;
        } else {
          partLen = cfg.segLen;
        }
      }

      unsigned int pktSize = sizeof(ip6_hdr) + sizeof(tcphdr) + partLen;

      // nastavime sekvencni cislo
      bytePkt.tcp_header.seq = htonl(tcp_seqNum);

      uint8_t hlim = 0;

      // Nastavime velikost paketu
      bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_plen = htons(sizeof(tcphdr) + partLen);

      // Cast podvrzene zpravy c.1 ****************************************************************
      waitNano(cfg.pktNanoInterval);
      if (fakeMsg1.is_open() && !fakeMsg1.eof()) {
        for (unsigned int i = 0; i < partLen; i++) {
          // Naplnime segment daty
          unsigned char c = 0;
          c = fakeMsg1.get();
          bytePkt.msg[i] = (uint8_t)c;
        }
        hlim = genNoiseHlim(noise_maxHlim);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + partLen);
        sendNormally ((uint8_t *)(&bytePkt), pktSize, targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

      // Cast sumu pred casti skutecne zpravy *****************************************************
      waitNano(cfg.pktNanoInterval);
      for (unsigned int n = 0; n < noiseBytes1; n++) {
        for (unsigned int i = 0; i < partLen; i++) {
          // Naplnime segment daty
          unsigned char c = 0;
          c = randomChar();
          bytePkt.msg[i] = (uint8_t)c;
        }
        hlim = genNoiseHlim(noise_maxHlim);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + partLen);
        sendNormally ((uint8_t *)(&bytePkt), pktSize, targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

      // Cast skutecne zpravy *********************************************************************
      waitNano(cfg.pktNanoInterval);
      for (unsigned int i = 0; i < partLen; i++) {
        // Naplnime segment daty
        unsigned char c = 0;
        c = *msgIterator;
        bytePkt.msg[i] = (uint8_t)c;
        msgIterator++;
      }
      hlim = randomUint(real_minHlim, real_maxHlim);
      bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
      tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + partLen);
      sendNormally ((uint8_t *)(&bytePkt), pktSize, targetInterface, cfg);
      pktSent++; byteSent += pktSize;

      // Cast sumu po casti skutecne zpravy ********************************************************
      waitNano(cfg.pktNanoInterval);
      for (unsigned int n = 0; n < noiseBytes2; n++) {
        for (unsigned int i = 0; i < partLen; i++) {
          // Naplnime segment daty
          unsigned char c = 0;
          c = randomChar();
          bytePkt.msg[i] = (uint8_t)c;
        }
        hlim = genNoiseHlim(noise_maxHlim);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + partLen);
        sendNormally ((uint8_t *)(&bytePkt), pktSize, targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

      // Cast podvrzene zpravy c.2 ****************************************************************
      waitNano(cfg.pktNanoInterval);
      if (fakeMsg2.is_open() && !fakeMsg2.eof()) {
        for (unsigned int i = 0; i < partLen; i++) {
          // Naplnime segment daty
          unsigned char c = 0;
          c = fakeMsg2.get();
          bytePkt.msg[i] = (uint8_t)c;
        }
        hlim = genNoiseHlim(noise_maxHlim);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + partLen);
        sendNormally ((uint8_t *)(&bytePkt), pktSize, targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }













      if (tcp_seqNum < tcp_finalSeqNum) { // pokud nejde o posledni segment
        addToIgnoreList(tcp_seqNum + 1, con);
      }
      tcp_seqNum += partLen;

      offset += partLen;

      // ======== KONEC SEKVENCE ======== ---------------------------------------------------------------------
    }


  //unsigned int pktSize = sizeof(ip6_hdr) + sizeof(tcphdr) + dataSize;
  //unsigned int pktSize = sizeof(TBytePkt) + ETH_OVERHEAD;
/*
  if (!secretMessageChars.empty()) {
    for(msgIterator=secretMessageChars.begin(); msgIterator != secretMessageChars.end(); ++msgIterator) {
      // ======== ZACATEK SEKVENCE ========



      // --------- odeslani pozadovanych bytu ----------

      //byte podvrzene zpravy c. 1
      if (fakeMsg1.is_open() && !fakeMsg1.eof()) {
        hlim = genNoiseHlim(noise_maxHlim);
        waitNano(cfg.pktNanoInterval);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        realSegLen =
        for (unsigned int i = 0; i < cfg.segLen; i++) {
          if (secretMessageChars.size() < cfg.segLen) {
            realSegLen
          }
          c = fakeMsg1.get();
          bytePkt.msg_byte = (uint8_t)c;
        }


        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + 1);
        sendNormally ((uint8_t *)(&bytePkt), sizeof(TBytePkt), targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }


      // sum pred bytem skutecne zpravy
      for (unsigned int i = 0; i < noiseBytes1; i++) {
        c = randomChar();
        hlim = genNoiseHlim(noise_maxHlim);
        waitNano(cfg.pktNanoInterval);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        bytePkt.msg_byte = (uint8_t)c;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + 1);
        sendNormally ((uint8_t *)(&bytePkt), sizeof(TBytePkt), targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

      // byte skutecne zpravy
      c = *msgIterator;
      hlim = randomUint(real_minHlim, real_maxHlim);
      waitNano(cfg.pktNanoInterval);
      bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
      bytePkt.msg_byte = (uint8_t)c;
      tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + 1);
      sendNormally ((uint8_t *)(&bytePkt), sizeof(TBytePkt), targetInterface, cfg);
      pktSent++; byteSent += pktSize;

      // sum po bytu skutecne zpravy
      for (unsigned int i = 0; i < noiseBytes2; i++) {
        c = randomChar();
        hlim = genNoiseHlim(noise_maxHlim);
        waitNano(cfg.pktNanoInterval);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        bytePkt.msg_byte = (uint8_t)c;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + 1);
        sendNormally ((uint8_t *)(&bytePkt), sizeof(TBytePkt), targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

      // byte podvrzene zpravy c. 2
      if (fakeMsg2.is_open() && !fakeMsg2.eof()) {
        c = fakeMsg2.get();
        hlim = genNoiseHlim(noise_maxHlim);
        waitNano(cfg.pktNanoInterval);
        bytePkt.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hlim;
        bytePkt.msg_byte = (uint8_t)c;
        tcpChecksum_for_ipv6(&(bytePkt.ipv6_header), &(bytePkt.tcp_header), sizeof(tcphdr) + 1);
        sendNormally ((uint8_t *)(&bytePkt), sizeof(TBytePkt), targetInterface, cfg);
        pktSent++; byteSent += pktSize;
      }

    }

  }
  */

  if (verbose()) {
    cout << "Odeslano " << pktSent << " paketu o celkove velikosti ";
    if (byteSent > 1048576) {
      cout << (double)byteSent / 1048576.0 << " MB";
    } else if (byteSent > 1024) {
      cout << (double)byteSent / 1024.0 << " kB";
    } else {
      cout << byteSent << " B";
    }
    cout << " cili [" << ipv6ToStr(bytePkt.ipv6_header.ip6_dst) << "]" << endl;
  }

  secretMessageChars.clear();
  delete decPkt;

  return;
}


/**
 * Cekani po dobu urcenou v nanosekundach
 * @param pocet nanosekund
 */
void waitNano(double nanoSec) {
  if (nanoSec == 0) {
    return;
  }
  clock_t endwait;
  endwait = clock() + CLOCKS_PER_SEC * 0.000000001 * nanoSec;
  while (clock() < endwait);
}


/**
 * Standardni odeslani zpravy
 * @param IPv6 header
 * @param delka IPv6 datagramu
 * @param konfigurace programu
 */
void sendNormally(uint8_t *ipv6_datagram, unsigned int ipv6_datagramLen, string ifName, TConfig &cfg) {
  pthread_mutex_lock(&cfg.libnet_mutex);

  libnet_t *l; // kontext libnetu
  char errbuf[LIBNET_ERRBUF_SIZE];

  if (ifName == "") {
    l = libnet_init (LIBNET_RAW6, NULL, errbuf);
  } else {
    char *devname = const_cast<char*>(ifName.c_str());
    l = libnet_init (LIBNET_RAW6, devname, errbuf);
  }

  if (l == NULL)
  {
    printError(ECONTEXT);
    pthread_mutex_unlock(&cfg.libnet_mutex);
    return;
  }

  libnet_write_raw_ipv6 (l, ipv6_datagram, ipv6_datagramLen);
  libnet_destroy (l);

  pthread_mutex_unlock(&cfg.libnet_mutex);
  return;
}

/**
 * Vlakno slouzici k detekci poctu skoku pro dane spojeni
 * @param spojeni
 */
void * hcThread(void * arg) {
  TConnection * con;
  con = (TConnection *)arg;

  TConfig *cfg = (TConfig *)con->cfgPtr;

  in6_addr rcvAddr;
  string outInterface = "";

  bool loop = true;
  bool firstTry = true;

  uint8_t est_hc = 0;
  uint8_t final_hc = 0;

  uint8_t this_hc = 0;
  bool this_succ = false;
  uint8_t last_hc = 0;
  bool last_succ = false;
  bool reply = false;
  clock_t endwait;

  bool canWeStart = false;
  bool failure = false;

  while (!canWeStart) {
    pthread_mutex_lock(&con->con_mutex);
    // byl jiz stanoven predpokladany pocet skoku a zname vystupni rozhrani?
    //out << ".. zkousim .." << endl;
    if (con->estimated_hc_set && con->outIf != "") {
      outInterface = con->outIf;
      rcvAddr = con->rcvIP;
      est_hc = con->hopCount;
      canWeStart = true;         // pokud ano, muzeme zacit s overovanim
    }
    pthread_mutex_unlock(&con->con_mutex);
    if (!con->active) {
      break;
    }
  }

  //////////////////

  if (verbose()) {cout << "Zahajuji detekci poctu skoku k " << ipv6ToStr(con->rcvIP) << endl;}

  if (!con->active) {
    if (verbose()) {cout << "Detekce poctu skoku k [" << ipv6ToStr(con->rcvIP) << "] byla prerusena." << endl;}
    return NULL;
  }

  while (loop) {
    // === 1. co budeme delat? ===========================
    if (firstTry) {
      // jde o prvni pokus
      this_hc = est_hc;
    } else {
      // nejde o prvni pokus
      if (last_succ) {
        if (this_hc == 0) {  // cil je prilis blizko!
          failure = true;
          break;
        }
        this_hc = last_hc - 1;
      } else {
        if (this_hc >= LINUX_DEFAULT_HLIM) {  // cil neni mozne dosahnout!
          failure = true;
          break;
        }
        this_hc = last_hc + 1;
      }
    }


    // === 2. poslani zpravy ===========================
    sendIcmpMessage(rcvAddr, outInterface, ICMPV6_ECHO, this_hc, (uint16_t)this_hc, 1, *cfg);
    if (verbose()) {cout << "odesilam ICMPv6, HL: " << (unsigned int)this_hc << ", cil: " << ipv6ToStr(con->rcvIP) << endl;}


    // === 3. cekani ===========================
    reply = false;
    endwait = clock() + ECHO_WAIT_LIMIT_SEC * CLOCKS_PER_SEC;
    while (clock() < endwait) {
      if (icmpReplyExists(rcvAddr, (uint16_t)this_hc, 1, *cfg)) {
        reply = true;
        break;
      }
    }
    // === 4. prisla odpoved? ===========================
    if (reply) {
      this_succ = true;
      if (this_hc == 0 && est_hc == 0) {
        final_hc = 0;
        loop = false;
        break;
      }
      if (verbose()) {cout << "ICMPv6 odpoved (HL: " << (unsigned int)this_hc << ") od cile [" << ipv6ToStr(con->rcvIP) << "] prisla." << endl;}
    } else {
      this_succ = false;
      if (verbose()) {cout << "ICMPv6 odpoved (HL: " << (unsigned int)this_hc << ") od cile [" << ipv6ToStr(con->rcvIP) << "] neprisla." << endl;}
    }

    if (firstTry) { // pokud jde o prvni pokus, nic nekontrolujeme
      firstTry = false;
      last_succ = this_succ;
      last_hc = this_hc;  // a jdeme dal
      continue;
    }

    if (this_succ) {
      if (last_succ) {
        last_succ = this_succ;
        last_hc = this_hc;
      } else {
        loop = false;
        final_hc = this_hc;
      }
    } else {
      if (last_succ) {
        loop = false;
        final_hc = last_hc;
      } else {
        last_succ = this_succ;
        last_hc = this_hc;
      }
    }
    if (!con->active) {
      break;
    }
  }

  if (!con->active) {
    if (verbose()) {cout << "Detekce poctu skoku k [" << ipv6ToStr(con->rcvIP) << "] byla prerusena." << endl;}
    return NULL;
  }

  pthread_mutex_lock(&con->con_mutex);
  con->final_hc_set = true;
  if (failure) {
    con->hopCount = 0;
    con->hc_failure = true;
    if (verbose()) {cout << "Detekce poctu skoku k [" << ipv6ToStr(con->rcvIP) << "] selhala" << endl;}
  } else {
    if (verbose()) {cout << "Detekce poctu skoku k [" << ipv6ToStr(con->rcvIP) << "] dokoncena: " << (unsigned int)final_hc << endl;}
    con->hopCount = final_hc;
  }
  pthread_mutex_unlock(&con->con_mutex);
  return NULL;
}

/**
 * Je v ramci tohoto spojeni vyzadovano oklamani odposlechu?
 * @param spojeni
 * @param konfigurace programu
 * @return (true = oklamani je pozadovano, false = oklamani neni pozadovano)
 */
bool secretConnection(TConnection &con, TConfig &cfg) {
  bool secret = true;
  if (cfg.senderPrefixLen != -1) { // pokud SENDER_PREFIX neni ANY
    if (!comparePrefix(con.sndIP, cfg.senderPrefix, cfg.senderPrefixLen)) {
      secret = false;
    }
  }
  if (cfg.receiverPrefixLen != -1) { // pokud RECEIVER_PREFIX neni ANY
    if (!comparePrefix(con.rcvIP, cfg.receiverPrefix, cfg.receiverPrefixLen)) {
      secret = false;
    }
  }
  if (cfg.senderPort != 0) { // pokud SENDER_PORT neni ANY
    if (con.sndPort != cfg.senderPort) {
      secret = false;
    }
  }
  if (cfg.receiverPort != 0) { // pokud RECEIVER_PORT neni ANY
    if (con.rcvPort != cfg.receiverPort) {
      secret = false;
    }
  }
  return secret;
}


/**
 * Vrati ukazatel na spojeni nebo NULL
 * @param spojeni
 * @param konfigurace programu
 * @return ukazatel na spojeni
 */
TConnection * getConnectionPointer(in6_addr sndIP, in6_addr rcvIP, unsigned short int sndPort, unsigned short int rcvPort, TConfig &cfg) {
  TConnection *pointer;
  pointer = NULL;
  if (cfg.con_list.empty()) {
    return NULL;
  }
  list<TConnection*>::iterator i;
  for(i=cfg.con_list.begin(); i !=cfg.con_list.end(); ++i) {
    if (*i != NULL) {
      if (compareAddresses(sndIP, (*i)->sndIP)
        && compareAddresses(rcvIP, (*i)->rcvIP)
        && sndPort == (*i)->sndPort
        && rcvPort == (*i)->rcvPort
        && (*i)->active) {
        pointer = *i;
        break;
      }
    }
  }
  return pointer;
}


/**
 * Vytvor spojeni
 * @param spojeni
 * @param konfigurace programu
 */
TConnection * addConnection(in6_addr sndIP, in6_addr rcvIP, unsigned short int sndPort, unsigned short int rcvPort, TConfig &cfg) {
  if (getConnectionPointer(sndIP, rcvIP, sndPort, rcvPort, cfg) != NULL) {
    return getConnectionPointer(sndIP, rcvIP, sndPort, rcvPort, cfg); // ochrana proti dvojimu pridani stejneho spojeni
  }
  TConnection *con = new TConnection;
  con->active = true;
  con->sndIP = sndIP;
  con->rcvIP = rcvIP;
  con->sndPort = sndPort;
  con->rcvPort = rcvPort;

  con->hopCount = 0;
  con->estimated_hc_set = false;
  con->final_hc_set = false;
  con->hc_failure = false;
  con->outIf = "";

  con->cfgPtr = (void *)(&cfg);

  if (pthread_mutex_init(&con->con_mutex, NULL) != 0) {
    printError(EINITMUTEX);
    return NULL;
  }
  pthread_mutex_lock(&con->con_mutex);
  if (pthread_create(&con->hc_thread, NULL, hcThread, con)) {
    printError(ETHREAD);
    pthread_mutex_unlock(&con->con_mutex);
    return NULL;
  }
  try {cfg.con_list.push_back(con);} catch (...) {printError(EMEMORY);}
  pthread_mutex_unlock(&con->con_mutex);

  return con;
}


/**
 * Odstran spojeni
 * @param spojeni
 * @param konfigurace programu
 */
void removeConnection(TConnection *con, TConfig &cfg) {
  if (con == NULL) {
    return;
  }

  // pockame, az vlakno hcThread nebude pracovat se zadnym sdilenym zdrojem
  pthread_mutex_lock(&cfg.libnet_mutex);
  pthread_mutex_lock(&cfg.reply_mutex);
  pthread_mutex_lock(&con->con_mutex);

  con->active = false; // spojeni uz nebude aktivni

  // semafory opet odemkneme
  pthread_mutex_unlock(&con->con_mutex);
  pthread_mutex_unlock(&cfg.reply_mutex);
  pthread_mutex_unlock(&cfg.libnet_mutex);
  return;
}


/**
 * Porovnani dvou IPv6 adres
 * @param IPv6 adresa
 * @param IPv6 adresa
 * @return shoda
 */
bool compareAddresses(in6_addr address1, in6_addr address2) {
  bool match = true;
  for (unsigned int i = 0; i < 16; i++) {
    if (address1.s6_addr[i] != address2.s6_addr[i]) {
      match = false;
    }
  }
  return match;
}


/**
 * Budeme tento ACK segment ignorovat?
 * @param ACK cislo
 * @param spojeni
 * @return ignorovat nebo ne
 */
bool ackToIgnore(unsigned long int ackNum, TConnection *con) {
  if (con != NULL) {
    if (acksToIgnore.empty()) {
      return false;
    }
    bool ignore = false;
    list<TAck>::iterator i;
    for(i=acksToIgnore.begin(); i !=acksToIgnore.end(); ++i) {
      if (i->ackNum == ackNum
        && compareAddresses(i->sndIP, con->sndIP)
        && compareAddresses(i->rcvIP, con->rcvIP)
        && i->sndPort == con->sndPort
        && i->rcvPort == con->rcvPort
      ) {
        ignore = true;
        break;
      }
    }
    return ignore;
  } else {
    return false;
  }
}

/**
 * Pridani ACK potvrzeni na seznam ignorovanych ACK potvrzeni
 * @param cislo ACK potvrzeni
 * @param spojeni
 */
void addToIgnoreList(unsigned long int ackNum, TConnection *con) {
  if (con != NULL) {
    TAck ack;
    ack.ackNum = ackNum;
    ack.sndIP = con->sndIP;
    ack.rcvIP = con->rcvIP;
    ack.sndPort = con->sndPort;
    ack.rcvPort = con->rcvPort;
    try {
      acksToIgnore.push_back(ack);
    } catch  (...) {
      return;
    }
  }
}


/**
 * Smazani sezamu ACK potvrzeni pro dane spojeni
 * @param cislo ACK potvrzeni
 * @param spojeni
 */
void clearIgnoreList(TConnection *con) {
  if (con != NULL) {
    if (acksToIgnore.empty()) {
      return;
    }
    list<TAck>::iterator i;
    for(i=acksToIgnore.begin(); i !=acksToIgnore.end(); ++i) {
      compareAddresses(i->sndIP, i->sndIP);
      compareAddresses(con->sndIP, con->sndIP);
      if (compareAddresses(i->sndIP, con->sndIP)
        && compareAddresses(i->rcvIP, con->rcvIP)
        && i->sndPort == con->sndPort
        && i->rcvPort == con->rcvPort
      ) {
        i = acksToIgnore.erase(i);
      }
    }
    return;
  }
}


/**
 * Identifikace a zpracovani odpovedi na ICMPv6 echo
 * @param paket
 * @param konfigurace programu
 * @return jedna se o odpoved na ICMPv6 echo nebo ne
 */
bool processIcmpEchoReply(Packet *Packet, TConfig &cfg) {
  if (Packet->is_icmpv6()) {
    in6_addr cptIfAddr;
    // ziskame adresu rozhrani
    cptIfAddr = getInterfaceIP(Packet->capturedInterface(), cfg);

    // overime, zda jde o odpoved pro nas
    if (Packet->is_icmpv6_echoReply() && compareAddresses(Packet->ipv6_dstIP(), cptIfAddr)) {
      TIcmpReply reply;
      reply.srcAddr = Packet->ipv6_srcIP();
      reply.id = Packet->icmpv6_echoReply_id();
      reply.seq = Packet->icmpv6_echoReply_seq();

      // pridame odpoved do seznamu
      pthread_mutex_lock(&cfg.reply_mutex);
      cfg.reply_list.push_back(reply);
      pthread_mutex_unlock(&cfg.reply_mutex);
      return true;
    }
  }
  return false;
}


/**
 * Existuje odpoved na danou ICMPv6 echo zpravu?
 * @param adresa vzdaleneho cile (zdrojova adresa odpovedi)
 * @param ID odpovedi
 * @param hodnota Sequence odpovedi
 * @param konfigurace programu
 * @return ano, ci ne
 */
bool icmpReplyExists(in6_addr srcAddr, uint16_t id, uint16_t seq, TConfig &cfg) {
  bool exists = false;

  // zamkneme semafor seznamu ICMPv6 odpovedi
  pthread_mutex_lock(&cfg.reply_mutex);

  if (cfg.reply_list.empty()) {
    pthread_mutex_unlock(&cfg.reply_mutex);
    return false;
  }

  list<TIcmpReply>::iterator i;
  for(i=cfg.reply_list.begin(); i !=cfg.reply_list.end(); ++i) {
    if (compareAddresses(srcAddr, i->srcAddr)
      && id == i->id
      && seq == i->seq) {
      exists = true;
      break;
    }
  }

  if (exists) {
    i = cfg.reply_list.erase(i);
  }

  // odemkneme semafor seznamu ICMPv6 odpovedi
  pthread_mutex_unlock(&cfg.reply_mutex);

  if (exists) {
    return true;
  } else {
    return false;
  }
}
